/*
 * ZAP! Z39.50 Apache Module
 * 
 * Copyright (C) 1997-2006, Index Data ApS
 * See the file ZAP_LICENSE for details.
 *
 * $Id: mod_zap.c,v 1.164 2008-10-06 13:55:35 adam Exp $
 */

#include <time.h>

#include "zap.h"

#include <yaz/charneg.h>
#include <yaz/log.h>
#include <yaz/oid_db.h>
#include <sys/time.h>
#include <time.h>

#define YAZ_LOG 0

#define ZAP_MAGIC_TYPE "application/x-httpd-zap"
#define ZAP_SCRIPT_TYPE "zap-script"

static int zap_request_sub(ZapRequest *old_req,
			    const char *fname, const char *args, int len);

static void zap_init(ZapRequest *req)
{
    req->grs1_vars_rec_rel = NULL;
    req->grs1_vars_rec_abs = NULL;
    req->marc_vars_buf = NULL;
    req->zap_debug = 0;
#if USE_TCL
    req->tcl_interp = 0;
#endif
    req->cur_pa = 0;
    req->curTemplate = 0;
    req->curVars = 0;
    req->output_buf = xmalloc(req->output_max = 2048);
    req->output_len = 0;
    req->cookie_buf = 0;
    req->cookie_zap = 0;
    req->pending_targets = 0;
    req->cookies_sent = 0;
    req->result = 0;
}


/* --- Logging, etc-------------------------------------------- */

#define MAX_ARGS_LEN 16000

#if USE_APACHE


#else
static int hex_digit(int ch)
{
    if (ch >= '0' && ch <= '9')
	return ch - '0';
    else if (ch >= 'a' && ch <= 'f')
	return ch - 'a'+10;
    else if (ch >= 'A' && ch <= 'F')
	return ch - 'A'+10;
    return 0;
}

static void unescape_any(char *url, char ch)
{
    char *dst = url;
    char *src = url;
    while (*src)
    {
	if (*src == ch)
	{
	    *dst++ = hex_digit(src[1])*16 + hex_digit(src[2]);
	    src += 3;
	}
	else
	    *dst++ = *src++;
    }
    *dst = '\0';
}

#endif

static void escape_any(char ch, const char *src, char *dst0, int max)
{
    char *dst = dst0;
    while (*src && (dst-dst0) < (max-4))
    {
        if (isalnum(*src))
            *dst++ = *src;
        else
        {
            sprintf(dst, "%c%2X", ch, *src);
	    dst += 3;
        }
        src++;
    }
    *dst = '\0';
}

#if USE_APACHE
#else
static void ap_unescape_url(char *url)
{
    unescape_any(url, '%');
}
static char *get_time(void)
{
    static char tformat[64];
    char *cp;
    time_t timeptr;

    time(&timeptr);
    strcpy(tformat, ctime(&timeptr));
    if ((cp = strchr(tformat, '\n')))
	*cp = '\0';
    return tformat;
}

#endif

#define zlog_debug(req, m1, m2) 

#if USE_APACHE
static void apdu_log(ZapRequest *req, ODR odr, Z_APDU *apdu)
{
    FILE *f;
    if ((f=req->conf->apdu_file))
    {
	odr_setprint(odr, f);
	z_APDU(odr, &apdu, 0, 0);
	fflush(f);
	odr_setprint(odr, 0);
    }
}

static void zlog_open(zap_server_conf *conf, apr_pool_t *p, server_rec *s)
{
    char *fname;

    if (!strcmp(conf->log_name, "none"))
        conf->log_file = 0;
    else
    {
	fname = ap_server_root_relative(p,(char *) conf->log_name);
	
	if (!conf->log_file)
	{
	    ZAP_FILE_OPEN_WD(fname, p, conf->log_file);
	}
    }
    if (conf->apdu_name && conf->apdu_file == 0)
    {
	fname =	ap_server_root_relative(p, (char *) conf->apdu_name);
	conf->apdu_file = fopen(fname, "a");
    }
}


void zlog(ZapRequest *req, const char *m1, const char *m2)
{
    char buf[2100];
    struct timeval tv;
    time_t ti;
    struct tm *tim;
     
    if (req->conf->log_file == 0)
        return;
    gettimeofday(&tv, 0);

    ti = tv.tv_sec;
    tim = localtime(&ti);
    strftime(buf, 50, "%d/%m-%H:%M:%S", tim);

    sprintf(buf + strlen(buf), ".%06ld %.1023s%.1023s\n",
	     (long) tv.tv_usec, m1, (m2 ? m2 : ""));
    ZAP_FILE_WRITE(buf, strlen(buf), req->conf->log_file);
    ZAP_FILE_FLUSH(req->conf->log_file);
}
#else
static void apdu_log(ZapRequest *req, ODR odr, Z_APDU *apdu)
{
}

void zlog(ZapRequest *req, const char *m1, const char *m2)
{
    FILE *f;
    
    f = fopen(ZAP_LOG, "a");
    if (!f)
	f = fopen("/tmp/zap_log", "a");
	if (!f)
		return;
    fprintf(f, "[%s] %s%s\n", get_time(), m1, (m2 ? m2 : ""));
    fclose(f);
}
#endif

static void zprintf(ZapRequest *req, const char *fmt, ...)
{
    va_list ap;
    char buf[1024];

    va_start(ap, fmt);
#if HAVE_VSNPRINTF
    vsnprintf(buf, 1023, fmt, ap);
#else
    vsprintf(buf, fmt, ap);
#endif
    zlog(req, buf, 0);
}

#if 0
static void zap_assert_fail(ZapRequest *req, const char *file, int line)
{
    zprintf(req, "ZAP version %s date %s", ZAP_VERSION, ZAP_DATE);
    zprintf(req, "assert fail file=%s line=%d");
    exit(1);
}
#endif


#if USE_APACHE
static void raw_write(ZapRequest *req, const char *m, int n)
{
    ap_rwrite(m, n, req->request);
}
#else
static void raw_write(ZapRequest *req, const char *m, int n)
{
    fwrite(m, 1, (n < 0 ? strlen(m) : n), stdout);
}
#endif

static void raw_puts(ZapRequest *req, const char *m)
{
    raw_write(req, m, -1);
    raw_write(req, "\n", -1);
}

static void html_write(ZapRequest *req, const char *m, int n)
{
    if (req->html_buffer)
    {
	if (n + req->html_len >= req->html_max)
	{
	    req->html_max += n + 2048;
	    req->html_buffer = xrealloc(req->html_buffer, req->html_max);
	}
	memcpy(req->html_buffer + req->html_len, m, n);
	req->html_len += n;
    }
    else
    {
	if (req->output_buf)
	{
	    if (n + req->output_len >= req->output_max)
	    {
		req->output_max += n + 4096;
		req->output_buf = xrealloc(req->output_buf, req->output_max);
	    }
	    memcpy(req->output_buf + req->output_len, m, n);
	    req->output_len += n;
	}
	else
	    raw_write(req, m, n);
    }
}

static void html_puts(ZapRequest *req, const char *m)
{
    html_write(req, m, strlen(m));
}

static void html_write_encoded(ZapRequest *req, const char *cp, int n)
{
    char buf[6];
    int i;
    
    buf[0] = '%';
    for (i = 0; i<n; i++, cp++)
    {
	unsigned char ch = *cp;
	if (ch < ' ' || ch >= 127 || !isalnum(ch))
	{
	    sprintf(buf+1, "%02X", ch);
	    html_write(req, buf, 3);
	}
	else if (ch == ' ')
	    html_write(req, "+", 1);
	else
	    html_write(req, (char*) &ch, 1);
    }
}

/* --- Symbol Table-------------------------------------------- */

Symtab symtabMk(void)
{
    Symtab p = xmalloc(sizeof(*p));
    p->list = NULL;
    p->last = &p->list;
    return p;
}

void symtabDestroy(Symtab *tabp)
{
    Symbol p = (*tabp)->list;
    while (p)
    {
	Symbol p1 = p;
	xfree(p->name);
	if (p->value)
	    xfree(p->value);
	p = p->next;
	xfree(p1);
    }
    xfree(*tabp);
    *tabp = NULL;
}

void symbolAddN(ZapRequest *req, Symtab tab, const char *name,
		 const char *value, int n)
{
    Symbol *p = tab->last;
    *p = xmalloc(sizeof(**p));
    (*p)->next = NULL;
    (*p)->name = xmalloc(strlen(name)+1);
    strcpy((*p)->name, name);
    if (!value)
	(*p)->value = NULL;
    else
    {
	while (n && *value == ' ')
	{
	    value++;
	    n--;
	}
	(*p)->value = xmalloc(n+1);
	if (n)
	    memcpy((*p)->value, value, n);
	(*p)->value[n] = '\0';
#if USE_TCL
        Tcl_SetVar(req->tcl_interp,(*p)->name, (*p)->value,
		   TCL_GLOBAL_ONLY);
#endif
    }
    tab->last = &(*p)->next;
}

void symbolAdd(ZapRequest *req,
		Symtab tab, const char *name, const char *value)
{
    zlog_debug(req, "symbol add name=", name);
    zlog_debug(req, "          value=", value);
    symbolAddN(req, tab, name, value, value ? strlen(value) : 0);
}

Symbol symbolLookup(Symtab tab, const char *name)
{
    struct symbolEntry *p = tab->list;
    const char *cp;
    
    if (!name)
	return p;
    cp = strchr(name, '*');
    for (p = tab->list; p; p = p->next)
	if (cp)
	{
	    if (!strncmp(name, p->name, cp-name))
		return p;
	}
	else
	{
	    if (!strcmp(name, p->name))
		return p;
	}
    return NULL;
}

void symbolSetN(ZapRequest *req, Symtab tab, const char *name,
		 const char *value, int n)
{
    Symbol p = symbolLookup(tab, name);
    
    if (!p)
	symbolAddN(req, tab, name, value, n);
    else
    {
	if (p->value)
	    xfree(p->value);
	if (!value)
	    p->value = NULL;
	else
	{
	    p->value = xmalloc(n+1);
	    if (n)
		memcpy(p->value, value, n);
	    p->value[n] = '\0';
#if USE_TCL
	    Tcl_SetVar(req->tcl_interp, p->name, p->value,
	       TCL_GLOBAL_ONLY);
#endif
	}
    }
}

void symbolSet(ZapRequest *req, Symtab tab, const char *name,
		const char *value)
{
    symbolSetN(req, tab, name, value, value ? strlen(value) : 0);
}

Symbol symbolNext(Symbol p, const char *name)
{
    const char *cp = strchr(name, '*');
    while ((p = p->next))
	if (cp)
	{
	    if (!strncmp(name, p->name, cp-name))
		return p;
	}
	else
	{
	    if (!strcmp(name, p->name))
		return p;
	}
    return NULL;
}


static void symtabDump(ZapRequest *req, Symtab tab)
{
    Symbol p;
    for (p = tab->list; p; p = p->next)
    {
	html_puts(req, "<b>");
	html_puts(req, p->name);
	html_puts(req, "</b>=");
	html_puts(req, p->value ? p->value : "<NULL>");
	html_puts(req, "<br>");
    }
}

void decodeURL(char *buf)
{
    char *cp;
    for (cp = buf; *cp; cp++)
	if (*cp == '+')
	    *cp = ' ';
    ap_unescape_url(buf);
}

static void symbolsURL(ZapRequest *req, Symtab tab, const char *in_buf)
{
    char buf[MAX_ARGS_LEN+1];
    char *cp = buf;

    if (!in_buf)
	return;
    strncpy(buf, in_buf, MAX_ARGS_LEN);
    buf[MAX_ARGS_LEN] = '\0';

    while (*cp)
    {
	char *cp_sep;
	char *cp_name;

	cp_sep = strchr(cp, '=');
	if (!cp_sep)
	{
	    break;
	}
	*cp_sep++ = '\0'; 
	while (*cp == '&')
	    cp++;
	decodeURL(cp);
	cp_name = cp;

	if (!cp_sep)
	{
	    symbolAdd(req, tab, cp_name, NULL);
	    break;
	}
	else
	{
	    cp = cp_sep;
	    cp_sep = strchr(cp, '&');
	    if (cp_sep)
		*cp_sep++ = '\0';
	    decodeURL(cp);
	    symbolAdd(req, tab, cp_name, cp);
	    if (cp_sep)
		cp = cp_sep;
	    else
	    {
		break;
	    }
	}
    }
}

#if USE_APACHE
static Symtab symtabMkArgs(ZapRequest *req)
{
    Symtab tab = symtabMk();
    
    /* get POST data */
    if (ap_should_client_block(req->request))
    {
	int to_read = MAX_ARGS_LEN;
	int len = 0;
	int len_read;
	char tbuf[MAX_ARGS_LEN+1];

	while (to_read > 0 &&
	       (len_read =
		ap_get_client_block(req->request, tbuf + len, to_read)) > 0)
	{
	    len += len_read;
	    to_read -= len_read;
	}
	tbuf[len] = '\0';
	symbolsURL(req, tab, tbuf);
    }
    /* get GET data */
    symbolsURL(req, tab, req->request->args);
    return tab;
}
#else
static Symtab symtabMkArgs(ZapRequest *req)
{
    char tbuf[MAX_ARGS_LEN+1];
    int to_read;
    char *t;
    Symtab tab = symtabMk();
    
    /* get POST data */
    if ((t = getenv("CONTENT_LENGTH")) && (to_read = atoi(t)))
    {
	int len = 0;
	if (to_read > MAX_ARGS_LEN)
	    to_read = MAX_ARGS_LEN;
	while (len < to_read)
	{
	    int j;
	    j = read(0, tbuf + len, to_read - len);
	    if (j <= 0)
		break;
	    len += j;
	}
	tbuf[len] = '\0';
	symbolsURL(req, tab, tbuf);
    }
    /* get GET data */
    if ((t = getenv("QUERY_STRING")))
    {
	strcpy(tbuf, t);
	symbolsURL(req, tab, tbuf);
    }
    return tab;
}
#endif
/* --- Path/Form lookup---------------------------------------- */


static void session(ZapRequest *req);
static void targetsMk(ZapRequest *req, int persistent);
static void targetsDestroy(ZapRequest *req, int cache_flag);
static void targetsEncodeCookie(ZapRequest *req);
static int targetDecodeCookie(ZapRequest *req,
			       const char *name, char **id, const char *proxy);


static Symbol symbolLookupForm0(PA *pa, const char *name)
{
    Symbol s;
    if ((s=symbolLookup(pa->override, name)))
	return s;
    if ((s=symbolLookup(pa->args, name)))
	return s;
    if ((s=symbolLookup(pa->def, name)))
	return s;
    return NULL;
}

static Symbol symbolLookupForm(ZapRequest *req, const char *name)
{
    return symbolLookupForm0(req->cur_pa, name);
}

const char *symbolLookupFormStrTarget(ZapRequest *req, Target t,
				       const char *name, const char *def)
{
    const char *res;
    char *sname = xmalloc(strlen(t->name) + strlen(name) + 3);
    sprintf(sname, "%s(%s)", name, t->name);
    res = symbolLookupFormStr(req, sname, 0);
    xfree(sname);
    if (!res)
	res = symbolLookupFormStr(req, name, def);
    return res;
}

const char *symbolLookupFormStr(ZapRequest *req,
				 const char *name, const char *def)
{
    Symbol s;

    s = symbolLookupForm(req, name);
    if (s)
	return s->value;
    return def;
}

/* --- Templates ---------------------------------------------- */

#define MAX_TEMPLATE_LEN 4096

struct TemplateEntry {
    char *buf;
    int head;
    struct TemplateEntry *next;
};

static int patternMatch(const char *pattern, const char *str)
{
    const char *p = pattern;
    const char *s = str;

    while (*p && *s)
    {
        while (*p == '-')
            p++;    
        while (*s == '-')
            s++;
	if (tolower(*p) != tolower(*s))
	    return 0;
	p++;
	s++;
    }
    if (*p == '\0' && (*s == ' ' || *s == '\0'))
	return 1;
    return 0;
}

static struct TemplateEntry *templateFindEntry(ZapRequest *req,
						Template template, 
						const char *head)
{
    struct TemplateEntry *te = template;
    
    for (; te; te = te->next)
    {
	if (!te->head)
	    continue;
	if (patternMatch(te->buf+2, head))
	{
	    if (req->zap_debug)
	    {
		html_puts(req, "<font size=\"-1\"><pre>rule ");
		html_puts(req, te->buf+2);
		if (strcmp(head, te->buf+2))
		{
		    html_puts(req, " (");
		    html_puts(req, head);
		    html_puts(req, ")");
		}
		html_puts(req, "</pre></font>\n");
	    }
	    break;
	}
    }
    return te;
}

const char *templateSymbolLookup(ZapRequest *req, Symtab vars,
				  const char *varname, 
				  const char *(*dhandler)(ZapRequest *req,
							  const char *cpp))
{
    Symbol sym = symbolLookup(vars, varname);
    const char *cp;
    if (sym && sym->value && *sym->value)
	return sym->value;
    if (dhandler) 
    {
	cp = (*dhandler)(req, varname);
        if (cp && *cp)
            return cp;
    }
    return NULL;
}

static void templateWrite(ZapRequest *req, const char *m, int n,
			   int encodeFlag)
{
    if (!encodeFlag)
	html_write(req, m, n);
    else
	html_write_encoded(req, m, n);
}

static void templateDumpRule(ZapRequest *req, Symtab vars, const char **cpp,
			      const char *stopChr, int execFlag,
			      int encodeFlag,
			      const char *(*dhandler)(
				  ZapRequest *req, const char *cpp))
{
    const char *cp = *cpp;
    const char *cp0 = *cpp;
    WRBUF wrbuf = wrbuf_alloc();

    while (*cp)
    {
	char varName[128];
	const char *varValue;
	int i;

	if (cp[0] == '%' && cp[1] == '{')
	{
#if USE_TCL

	    if (cp > cp0 && execFlag)
		templateWrite(req, cp0, cp - cp0, encodeFlag);
	    cp++;
	    cp0 = ++cp;
	    for (; *cp && (cp[0] != '%' || cp[1] != '}'); cp++)
		;
	    if (!*cp)
		break;

	    if (execFlag)
	    {
                wrbuf_rewind(wrbuf);

                wrbuf_write(wrbuf, cp0, cp - cp0);
		if (Tcl_Eval(req->tcl_interp, wrbuf_cstr(wrbuf)) != TCL_OK)
		{
		    const char *errorInfo =
                        Tcl_GetVar(req->tcl_interp, "errorInfo", 0);
		    char buf[80];

		    sprintf(buf, "<br><b>Tcl error in line %d:</b> ",
                             req->tcl_interp->errorLine);
                    html_puts(req, buf);
                    if (req->tcl_interp->result)
                        html_puts(req, req->tcl_interp->result);
                    html_puts (req, "<br><hr>\n<pre>\n");
                    if (errorInfo)
		        html_puts(req, errorInfo);
		    html_puts(req, "</pre><hr><br>");
		}
	    }
	    
	    cp++;
	    cp0 = ++cp;
#else
	    zprintf(req, "Scripting not enabled (server malconfiguration)");
#endif
	}
	if (*cp == '\\' && cp[1])
	{
	    if (cp > cp0 && execFlag)
		templateWrite(req, cp0, cp - cp0, encodeFlag);
	    cp++;
	    cp0 = cp++;
	    continue;
	}
	if (stopChr && strchr(stopChr, *cp))
	    break;
	if (*cp != '$')
	{
	    cp++;
	    continue;
	}
	if (cp > cp0 && execFlag)
	    templateWrite(req, cp0, cp - cp0, encodeFlag);
	cp++;
	if (*cp == '<' && req->html_buffer == 0)
	{
	    char *buf; 
	    int len;

	    cp++;
	    for (i = 0; i < 127 && *cp != '?' && *cp != '>' && *cp; i++)
		varName[i] = *cp++;
	    varName[i] = '\0';
	    if (*cp == '?')
		cp++;
	    
	    req->html_buffer = xmalloc(req->html_max = 100);
	    templateDumpRule(req, vars, &cp, "}>", 1, encodeFlag,
			      dhandler);

	    buf = req->html_buffer;
	    len = req->html_len;

	    req->html_buffer = 0;
	    req->html_len = req->html_max = 0;
	    
	    if (*cp)
		cp++;
	    zap_request_sub(req, varName, buf, len);
	    xfree(buf);
	}
	else if (*cp == '{')
	{
	    cp++;
	    if (!execFlag)
	    {
		templateDumpRule(req, vars, &cp, "}", 0, encodeFlag,
				  dhandler);
		if (*cp)
		    cp++;
		continue;
	    }
	    for (i = 0; i < 127 && *cp != '?' && *cp != '}' && *cp != '='
		     && *cp != ':' && *cp; i++)
		varName[i] = *cp++;
	    varName[i] = '\0';
	    if (*cp == '=')
	    {
		cp++;
		if (!i)
		    templateDumpRule(req, vars, &cp, "}", 1, 1,
				      dhandler);
		else
		{
		    Symbol sym;
		    int no = 0;
		    
		    for (sym = symbolLookupForm(req, varName); sym;
			 sym = symbolNext(sym, varName))
		    {
			if (no++)
			    html_puts(req, "&");
			html_write(req, sym->name,
					    strlen(sym->name));
			html_puts(req, "=");
			html_write_encoded(req, sym->value,
					    strlen(sym->value));
		    }
		}
	    }
	    else
	    {
		varValue = templateSymbolLookup(req, vars, varName, dhandler);
		if (*cp == '?')
		{
		    cp++;
		    if (varValue)
		    {
			templateDumpRule(req, vars, &cp, ":}", 1,
					  encodeFlag, dhandler);
			if (*cp == ':')
			{
			    cp++;
			    templateDumpRule(req, vars, &cp, "}", 0,
					      encodeFlag, dhandler);
			}
		    } 
		    else
		    {
			templateDumpRule(req, vars, &cp, ":}", 0,
					  encodeFlag, dhandler);
			if (*cp == ':')
			{
			    cp++;
			    templateDumpRule(req, vars, &cp, "}", 1,
					      encodeFlag, dhandler);
			}
		    }
		} 
		else
		{
		    if (!varValue)
		    {
			Symbol s = symbolLookupForm(req, varName);
			if (s)
			    varValue = s->value;
		    }
		    if (varValue)
			templateWrite(req, varValue, strlen(varValue),
				       encodeFlag);
		}
	    }
	    if (*cp)
		cp++;      /* skip } */
	}
	else
	{
	    for (i = 0; i < 127 && isalnum(*cp); i++)
		varName[i] = *cp++;
	    varName[i] = '\0';
	    if (execFlag)
	    {
		varValue = templateSymbolLookup(req, vars, varName, dhandler);
		if (!varValue)
		{
                    Symbol s = symbolLookupForm(req, varName);
                    if (s)
			varValue = s->value;
                }
		if (varValue)
		    templateWrite(req, varValue,
				   strlen(varValue), encodeFlag);
	    }
	}
	cp0 = cp;
    }
    if (cp > cp0 && execFlag)
	templateWrite(req, cp0, cp - cp0, encodeFlag);
    *cpp = cp;
    wrbuf_destroy(wrbuf);
}

static void templateDump(ZapRequest *req, Template template,
			  Symtab vars, const char *head)
{
    struct TemplateEntry *te = templateFindEntry(req, template, head);

    if (!te)
	return ;
    while ((te = te->next))
    {
	const char *cp = te->buf;
	if (te->head)
	    break;
	templateDumpRule(req, vars, &cp, NULL, 1, 0, NULL);
    }
}

static void templateDestroy(Template *tp)
{
    struct TemplateEntry *te = *tp;

    while (te)
    {
	struct TemplateEntry *te1 = te;

	xfree(te->buf);
	te = te->next;
	xfree(te1);
    }
    *tp = NULL;
}

static Template *templateReadFile(ZapRequest *req,
				   Template *tp,
                                   const char *input_fname, int level)
{
    char *cp = 0, *org_fname = 0;

    ZAP_FILE_FD f;
    char buf[MAX_TEMPLATE_LEN]; 
    char fname[256];
    WRBUF wrbuf;
    int org_len;

    *tp = 0;
#if USE_APACHE
    org_fname = req->request->filename;
    org_len = strlen(req->request->filename);
    if ((cp = strrchr(org_fname, '/')))
    {
        cp++;
        org_len = cp - org_fname;
    }
#else
    org_fname = req->script_name;
    org_len = 0;
    if ((cp = strrchr(org_fname, '/')))
    {
        cp++;
        org_len = cp - org_fname;
    }
#endif
    strncpy(fname, input_fname, sizeof(fname)-1);
    fname[sizeof(fname)-1] = '\0';
    if (*fname != '/' && org_fname && org_len)
    {
	memcpy(fname, org_fname, org_len);
	strcpy(fname + org_len, input_fname);
    }
    ZAP_FILE_OPEN_RD(fname, req->request->pool, f);
    if (!f)
    {
	zprintf(req, "Open %s failed (%s)", fname, strerror(errno));
	return 0;
    }
    zlog(req, "Reading ", fname);
    wrbuf = wrbuf_alloc();
    while (ZAP_FILE_GETS(buf, sizeof(buf)-1, f))
    {
	int head = 0;

	if (buf[0] == '%' && buf[1] == '%')
	{
	    char cmdstr[32], sub_fname[64];
	    if (sscanf(buf+2, "%31s %63s", cmdstr, sub_fname) == 2 && 
		!strcmp(cmdstr, "include"))
	    {
		if (level > 4)
		    zprintf(req,
			     "template %s not read - too many levels (%d)",
			     fname, level);
		else
		{
		    Template *tp_new;
		    tp_new = templateReadFile (req, tp, sub_fname, level+1);
		    if (tp_new)
			tp = tp_new;
		}
		continue;
	    }
	    head = 1;
	}
	*tp = xmalloc(sizeof(**tp));
	if (!*tp)
	{
	    zprintf(req, "out of memory");
	    exit(0);
	}
	(*tp)->head = head;

        wrbuf_rewind(wrbuf);
#if USE_TCL
	if (buf[0] == '%' && buf[1] == '{')
        {
	    char *p = buf+2;
	    for (;;)
	    {
                /* look for %} sequence */
                while ((p = strchr(p, '%')))
		    if (*++p == '}')
                        break;
                if (p)
                    break;
                wrbuf_puts(wrbuf, buf);
		if (!ZAP_FILE_GETS(buf, sizeof(buf)-1, f))
		    break;
                p = buf;
	    }
	}
#endif
	for (;;)
	{
	    char *cp = buf + strlen(buf);
	    while (cp != buf && strchr(" \t\r\n", cp[-1]))
	        --cp;
            if (cp == buf || (cp > buf && cp[-1] != '\\'))
            {
                wrbuf_write(wrbuf, buf, cp-buf);
                break;
            }
            wrbuf_write(wrbuf, buf, cp-buf-1);
	    ZAP_FILE_GETS(buf, sizeof(buf)-1, f);
	}
	if (!(*tp)->head)
            wrbuf_puts(wrbuf, "\n");
	if (!((*tp)->buf = xstrdup(wrbuf_cstr(wrbuf))))
	{
	    zprintf(req, "out of memory");
	    exit(0);
	}
	tp = &(*tp)->next;
    }
    *tp = NULL;
    ZAP_FILE_CLOSE(req->request->pool, f);
    wrbuf_destroy(wrbuf);
    return tp;
}

static void templateRead(ZapRequest *req,
			  Template *tp, const char *fname)
{
    templateReadFile(req, tp, fname, 0);
}

static void templateDef(ZapRequest *req, PA *pa,
			 Template t, const char *head, Symtab tab)
{
    struct TemplateEntry *te = templateFindEntry(req, t, head);
    char outbuf[4096];
    WRBUF wrbuf = wrbuf_alloc();
    
    if (!te)
	return ;
    while ((te = te->next))
    {
	char *cp;
	if (te->head)
	    break;
	cp = te->buf;
	if (cp[0] == '%' && cp[1] == '{' && cp[2])
	{
	    if ((cp = strrchr(te->buf+2, '%')))
	    {
		wrbuf_rewind(wrbuf);
		wrbuf_write(wrbuf, te->buf+2, cp - te->buf - 2);
#if USE_TCL
		Tcl_Eval(req->tcl_interp, wrbuf_cstr(wrbuf));
#endif
	    }
	    continue;
	}
	cp = strchr(te->buf, '=');
	if (cp && cp[1])
	{
	    int i;
	    *cp++ = '\0';
	    i = strlen(cp) - 1;
	    while (i > 0 &&
		   (cp[i] == '\n' || cp[i] == ' ' || 
		    cp[i] == '\t' ||  cp[i] == '\r'))
		--i;
	    cp[i+1] = '\0';
            i = 0;
            while (*cp)
            {
		if (*cp == '$' && cp[1] == '{')
		{
		    char name[128];
                    Symbol s;
                    int j = 0;
                    cp++;
		    cp++;
                    while (j<127 && *cp && *cp != '}')
                        name[j++] = *cp++;
		    cp++;
                    name[j] = '\0';
                    s = symbolLookupForm (req, name);
                    if (s)
                    {
                        j = strlen(s->value);
                        if (j + i < sizeof(outbuf))
                        {
                            memcpy (outbuf+i, s->value, j);
                            i += j;
                        }
                    }
		}
		else if (*cp == '$')
                {
		    char name[128];
                    Symbol s;
                    int j = 0;
                    cp++;
                    while (j<127 && isalnum(*cp))
                        name[j++] = *cp++;
                    name[j] = '\0';
                    s = symbolLookupForm (req, name);
                    if (s)
                    {
                        j = strlen(s->value);
                        if (j + i < sizeof(outbuf))
                        {
                            memcpy (outbuf+i, s->value, j);
                            i += j;
                        }
                    }
                }
                else
                {
		    if (i < sizeof(outbuf)-1)
                         outbuf[i++] = *cp;
	            cp++;	
                }
            }
            outbuf[i] = '\0';
	    symbolAdd (req, tab, te->buf, outbuf);
	}
    }
    wrbuf_destroy(wrbuf);
}


void html_var (ZapRequest *req, const char *name, const char *value)
{
    symbolSet (req, req->curVars, name, value);
}

void html_var_num (ZapRequest *req, const char *name, int val)
{
    char str[64];
    sprintf (str, "%d", val);
    html_var (req, name, str);
}

void html_var_n (ZapRequest *req, const char *name,
			const char *buf, int n)
{
    symbolSetN (req, req->curVars, name, buf, n);
}

void html_dump (ZapRequest *req, const char *head)
{
    templateDump (req, req->curTemplate, req->curVars, head);
}

/* --- Query String ------------------------------------------- */

int queryVarCCL (ZapRequest *req)
{
    WRBUF wrbuf_query = wrbuf_alloc();
    int i = 1;
    char fname[32];
    int noTerms = 0;

    wrbuf_puts (wrbuf_query, "");
    while (1)
    {
	const char *entry;

	sprintf (fname, "term%d", i);
	entry = symbolLookupFormStr (req, fname, NULL);
	if (!entry)
	    break;
	if (*entry)
	{
	    if (wrbuf_len(wrbuf_query))
		wrbuf_puts (wrbuf_query, " ");
	    noTerms++;
	    wrbuf_puts (wrbuf_query, entry);
	}
	i++;
    }
    html_var (req, "query", wrbuf_cstr(wrbuf_query));
    wrbuf_destroy(wrbuf_query);
    if (!noTerms)
    {
	html_dump (req, "query-empty");
	return 0;
    }
    return noTerms;
}

int queryVarRPN (ZapRequest *req)
{
    WRBUF wrbuf_query = wrbuf_alloc();
    WRBUF wrbuf_term = wrbuf_alloc();
    int i = 1;
    char fname[32];
    const char *op;
    int noTerms = 0;
    int method = 1; /* 1 = left, 2 = right */

    op = symbolLookupFormStr (req, "op", "left");
    if (op && !strcmp(op, "right"))
        method = 2;
    else 
        method = 1;
    op = 0;

    wrbuf_puts(wrbuf_query, "");
    while (1)
    {
	const char *attrf;

	Symbol sym;
        int extended_term = 0;
	int ccl_term = 0;

        wrbuf_rewind (wrbuf_term);
	sprintf (fname, "term%d", i);
	sym = symbolLookupForm (req, fname);
	if (!sym)
	{
	    sprintf (fname, "entry%d", i);
	    sym = symbolLookupForm (req, fname);
	}
        if (!sym)
        {
	    sprintf (fname, "rawterm%d", i);
	    sym = symbolLookupForm (req, fname);
            extended_term = 1;
        }
        if (!sym)
        {
	    sprintf (fname, "cclterm%d", i);
	    sym = symbolLookupForm (req, fname);
            ccl_term = 1;
        }
        if (!sym)
	{
	    sprintf (fname, "op%d", i);
	    /* break only if both op(i) and term(i) doesn't exist */
	    if (!symbolLookupForm (req, fname))
		break;
	}
        else
        {
            while (sym)
            {   
                Symbol sym_next = symbolNext(sym, fname);
                if (sym->value && *sym->value)
                {
                    wrbuf_puts (wrbuf_term, sym->value);
                    if (sym_next && sym_next->value && *sym_next->value)
                        wrbuf_puts (wrbuf_term, ", ");
		    if (ccl_term)
		    {
			struct ccl_rpn_node *rpn;
			int ccl_error, ccl_pos;

			rpn = ccl_find_str(req->ccl_bibset, sym->value,
					   &ccl_error, &ccl_pos);
			if (rpn)
			    ccl_rpn_delete (rpn);
			if (ccl_error)
			{
			    char str[80];
			    sprintf (str, "%d", ccl_error);
			    html_var (req, "errorcode", str);
			    html_var (req, "errorstring",
				      ccl_err_msg(ccl_error));
			    sprintf (str, "ccl-error %d",
				     ccl_error);
			    html_dump (req, str);
			    return 0;
			}
		    }
                }
                sym = sym_next;
            }
        }
	if (wrbuf_len(wrbuf_term) == 0)
	{
            if (op && method == 2)
            {
	        sprintf (fname, "op%d", i);
	        op = symbolLookupFormStr (req, fname, "and");
            }
	    i++;
	    continue;
	}
	if (op)
	{
	    char mapstr[64];
            const char *op_lang = 0;
	
            sprintf (mapstr, "opdisplay(%.50s)", op);
	    op_lang = symbolLookupFormStr (req, mapstr, op);
            
	    wrbuf_puts(wrbuf_query, " ");
	    if (*op_lang == '@')
                op_lang++;
            wrbuf_puts(wrbuf_query, op_lang);
	    wrbuf_puts(wrbuf_query, " ");
	}
	sprintf (fname, "field%d", i);
	attrf = symbolLookupFormStr (req, fname, NULL);
	if (attrf)
	{
	    char mapstr[64];

	    sprintf (mapstr, "map(%.50s)", attrf);
	    if (symbolLookupForm (req, mapstr))
	    {
                wrbuf_puts(wrbuf_query, attrf);
                wrbuf_puts(wrbuf_query, "=");
	    }
	}
	wrbuf_puts (wrbuf_query, "<b>");
	wrbuf_puts (wrbuf_query, wrbuf_cstr(wrbuf_term));
	wrbuf_puts (wrbuf_query, "</b>");
	noTerms++;

	sprintf (fname, "op%d", i);
	op = symbolLookupFormStr (req, fname, "and");
	i++;
    }
    html_var (req, "query", wrbuf_cstr(wrbuf_query));
    wrbuf_destroy(wrbuf_query);
    wrbuf_destroy(wrbuf_term);
    if (!noTerms)
    {
	html_dump (req, "query-empty");
	return 0;
    }
    return noTerms;
}

int queryVarRPNDirect (ZapRequest *req)
{
    WRBUF wrbuf_query = wrbuf_alloc();
    const char *entry;
    int noTerms = 0;
    
    wrbuf_puts (wrbuf_query, "");
    entry = symbolLookupFormStr (req, "rpnquery", NULL);

    if (entry) {
        wrbuf_puts (wrbuf_query, entry);
        noTerms++;
    }
    
    html_var (req, "query", wrbuf_cstr(wrbuf_query));
    wrbuf_destroy(wrbuf_query);
    if (!noTerms)
    {
        html_dump (req, "query-empty");
        return 0;
    }
    return noTerms;
}

int queryVar (ZapRequest *req)
{
    const char *entry = symbolLookupFormStr (req, "querytype", "rpn");
    
    if (!strcmp (entry, "rpn-direct"))
	return queryVarRPNDirect (req);
    if (!strcmp (entry, "rpn"))
	return queryVarRPN (req);
    if (!strcmp (entry, "ccl"))
	return queryVarCCL (req );
    zlog (req, "unrecognized value for querytype: ", entry);
    return 0;
}

static void html_head (ZapRequest *req)
{
    const char *content_type =
	symbolLookupFormStr(req, "content-type", "text/html");
#if USE_APACHE
    const char *expire_after;
    char *content_type_tmp;
#endif
    if (!req->output_buf)
	return;
    targetsEncodeCookie (req);
#if USE_APACHE
    expire_after = symbolLookupFormStr(req, "expire", 0);
    if (expire_after)
    {
	time_t mod_time;
	time(&mod_time);
	mod_time += atoi(expire_after);
#if APACHE2
	if (mod_time)
	{
	    char dstr[APR_RFC822_DATE_LEN+1];
	    ap_recent_rfc822_date(dstr, mod_time);
	    ap_table_add (req->request->headers_out, "Expires", dstr);
	}
#else
	ap_table_add (req->request->headers_out, "Expires",
            ap_gm_timestr_822(req->request->pool, mod_time));
#endif
    }
#endif
#if USE_APACHE
    content_type_tmp = ap_pcalloc(req->request->pool, strlen(content_type)+1);
    strcpy(content_type_tmp, content_type);
    req->request->content_type = content_type_tmp;
#if APACHE2
#else
    ap_send_http_header(req->request);
#endif
#else
    raw_write (req, "Content-type: ", -1);
    raw_puts (req, content_type);
    raw_puts (req, "");
#endif
    if (req->output_len)
	raw_write (req, req->output_buf, req->output_len);
    else
	raw_puts (req, "");
    xfree (req->output_buf);
    req->output_buf = 0;
}

/* --- Main Handler ------------------------------------------- */

#if USE_TCL

static int get_grs_r (Tcl_Interp *interp, Z_GenericRecord *grs_record,
                             int argc, char **argv, int argno)
{
    static char tmpbuf[32];
    int i;

    if (!grs_record)
	return TCL_OK;
    for (i = 0; i<grs_record->num_elements; i++)
    {
	Z_TaggedElement *e = grs_record->elements[i];
        int yes = 0;

        if (argno >= argc)
            yes = 1;
        else
        {
            const char *cp0 = argv[argno];
            const char *cp1 = strchr (cp0, ',');

            if (!cp1 || cp1-cp0 < 1)
                yes = 1;
            else
            {
                if (*cp0 == '(')
                    cp0++;
                if (e->tagType && atoi(cp0) == *e->tagType) 
                {
                    if (e->tagValue->which == Z_StringOrNumeric_numeric)
                    {
                        if (atoi (cp1+1) == *e->tagValue->u.numeric)
                            yes = 1;
                    }
                    else
                    {
                        int len = strlen(cp1+1);
                        if (cp1[len] == ')')
                            len--;
                        if (len && strlen(e->tagValue->u.string) == len &&
                            !memcmp (cp1+1, e->tagValue->u.string, len))
                            yes = 1;
                    }
                }
            }
        }
        if (!yes)
            continue;
        Tcl_AppendResult (interp, "{ ", NULL);
        sprintf (tmpbuf, "%d", e->tagType ? *e->tagType : 0);
        Tcl_AppendElement (interp, tmpbuf);

        if (e->tagValue->which == Z_StringOrNumeric_numeric)
        {
            Tcl_AppendResult (interp, " numeric ", NULL);
            sprintf (tmpbuf, "%d", *e->tagValue->u.numeric);
            Tcl_AppendElement (interp, tmpbuf);
        }
        else
        {
            Tcl_AppendResult (interp, " string ", NULL);
            Tcl_AppendElement (interp, e->tagValue->u.string);
        }
        switch (e->content->which)
        {
        case Z_ElementData_octets:
            Tcl_AppendResult (interp, " octets {} ", NULL);
            break;
        case Z_ElementData_numeric:
            Tcl_AppendResult (interp, " numeric ", NULL);
            sprintf (tmpbuf, "%d", *e->content->u.numeric);
            Tcl_AppendElement (interp, tmpbuf);
            break;
        case Z_ElementData_date:
            Tcl_AppendResult (interp, " date {} ", NULL);
            break;
        case Z_ElementData_ext:
            Tcl_AppendResult (interp, " ext {} ", NULL);
            break;
        case Z_ElementData_string:
            Tcl_AppendResult (interp, " string ", NULL);
            Tcl_AppendElement (interp, e->content->u.string);
            break;
        case Z_ElementData_trueOrFalse:
            Tcl_AppendResult (interp, " bool ",
                            *e->content->u.trueOrFalse ? "1" : "0", " ", NULL);
            break;
        case Z_ElementData_oid: {
	    char num[10];
	    Odr_oid *ii;

            Tcl_AppendResult (interp, " oid ", NULL);
	    for (ii = e->content->u.oid; ii && *ii >= 0; ii++)
	    {
		sprintf(num, "%d%s", *ii, (*(ii + 1) >= 0) ? "." : " ");
		Tcl_AppendResult (interp, num, NULL);
	    }
            break;
	}
        case Z_ElementData_intUnit:
            Tcl_AppendResult (interp, " intUnit {} ", NULL);
            break;
        case Z_ElementData_elementNotThere:
            Tcl_AppendResult (interp, " notThere {} ", NULL);
            break;
        case Z_ElementData_elementEmpty:
            Tcl_AppendResult (interp, " empty {} ", NULL);
            break;
        case Z_ElementData_noDataRequested:
            Tcl_AppendResult (interp, " notRequested {} ", NULL);
            break;
        case Z_ElementData_diagnostic:
            Tcl_AppendResult (interp, " diagnostic {} ", NULL);
            break;
        case Z_ElementData_subtree:
            Tcl_AppendResult (interp, " subtree { ", NULL);
            get_grs_r (interp, e->content->u.subtree, argc, argv,
		argno+1);
            Tcl_AppendResult (interp, " } ", NULL);
            break;
        }
        Tcl_AppendResult (interp, " } ", NULL);
    }
    return TCL_OK;
}

static int cmd_getGrs(ClientData cd, Tcl_Interp *interp, int argc, char **argv)
{
    int arg = 1;
    ZapRequest *req = cd;
    Z_GenericRecord *record = req->grs1_vars_rec_rel;

    if (argc >= 2 && !strcmp(argv[1], "-r"))
    {
	record = req->grs1_vars_rec_abs;
	arg++;
    }
    return get_grs_r (interp, record, argc, argv, arg);
}

static int cmd_html(ClientData cd, Tcl_Interp *interp, int argc, char **argv)
{
    int i;
    ZapRequest *req = cd;
    if (argc < 2)
	return TCL_ERROR;
    for (i = 1; i<argc; i++)
    {
#if TCL_MAJOR_VERSION > 8 || (TCL_MAJOR_VERSION == 8 && TCL_MINOR_VERSION > 0)
	Tcl_DString ds;
	char *native = Tcl_UtfToExternalDString(0, argv[i], -1, &ds);
	html_puts (req, native);
	Tcl_DStringFree (&ds);
#else
	html_puts (req, argv[i]);
#endif
    }
    return TCL_OK;
}

static int cmd_urlenc (ClientData clientData, Tcl_Interp *interp,
		       int argc, char **argv)
{
    int i;
    char buf1[6];
    char buf2[2];
    
    buf1[0] = '%';
    buf2[1] = '\0';
    for (i = 1; i<argc; i++)
    {
        char *cp = argv[i];
#if TCL_MAJOR_VERSION > 8 || (TCL_MAJOR_VERSION == 8 && TCL_MINOR_VERSION > 0)
	Tcl_DString ds;
	cp = Tcl_UtfToExternalDString(0, cp, -1, &ds);
#endif
        while (*cp)
        { 
            if (*cp < ' ' || *cp >= 127 || *cp == '&' || *cp == '?'
                || *cp == '%' || *cp == '+' || *cp == '"' || *cp == '=')
            {
                sprintf (buf1+1, "%02X", *cp & 0xff);
                Tcl_AppendResult (interp, buf1, NULL);
            }
            else if (*cp == ' ')
            {
                Tcl_AppendResult (interp, "+", NULL);
            }
            else
            {
                buf2[0] = *cp;
                Tcl_AppendResult (interp, buf2, NULL);
            }
            cp++;
        }
#if TCL_MAJOR_VERSION > 8 || (TCL_MAJOR_VERSION == 8 && TCL_MINOR_VERSION > 0)
	Tcl_DStringFree (&ds);
#endif
    }
    return TCL_OK;
}

static int cmd_urldec (ClientData clientData, Tcl_Interp *interp,
		       int argc, char **argv)
{
    int i;
    unsigned val;
    char buf[2];
    
    buf[1] = '\0';
    for (i = 1; i<argc; i++)
    {
        const char *cp = argv[i];
        while (*cp)
        {
            if (*cp == '%' && cp[1] && cp[2])
            {
                if (cp[1] >= 'A')
                    val = cp[1] - 'A'+10;
                else
                    val = cp[1] - '0';

                
                if (cp[2] >= 'A')
                    val = val*16 + (cp[2] - 'A'+10);
                else
                    val = val*16 + (cp[2] - '0');
                buf[0] = val;
                cp += 3;
            }
            else if (*cp == '+')
	    {
		buf[0] = ' ';
		cp++;
	    }
	    else
                buf[0] = *cp++;
            Tcl_AppendResult (interp, buf, NULL);
        }
    }
    return TCL_OK;
}

static int cmd_setz(ClientData cd, Tcl_Interp *interp, int argc, char **argv)
{
    char *value;
    ZapRequest *req = cd;
 
    if (argc == 2)
    {
        value = Tcl_GetVar2(interp, argv[1], NULL, TCL_LEAVE_ERR_MSG);
        if (value == NULL)
            return TCL_ERROR;
        Tcl_SetResult(interp, value, 0);
        return TCL_OK;
    }
    else if (argc == 3)
    {
        value = Tcl_SetVar2(interp, argv[1], NULL, argv[2], TCL_LEAVE_ERR_MSG);
	if (value)
	{	
#if TCL_MAJOR_VERSION > 8 || (TCL_MAJOR_VERSION == 8 && TCL_MINOR_VERSION > 0)
	    Tcl_DString ds;
	    char *native = Tcl_UtfToExternalDString(0, argv[2], -1, &ds);
	    symbolSet (req, req->curVars ? req->curVars:req->cur_pa->override,
		   argv[1], native);
	    Tcl_DStringFree (&ds);
#else
	    symbolSet (req, req->curVars ? req->curVars:req->cur_pa->override,
		   argv[1], argv[2]);
#endif
            return TCL_OK;
	}
	else
            return TCL_ERROR;
    }
    else
    {
        Tcl_SetResult(interp, "setz varName ?newValue?", 0);
        return TCL_ERROR;
    }
}

static int cmd_addz(ClientData cd, Tcl_Interp *interp, int argc, char **argv)
{
    char *value;
    ZapRequest *req = cd;
 
    if (argc == 2)
    {
        value = Tcl_GetVar2(interp, argv[1], NULL, TCL_LEAVE_ERR_MSG);
        if (value == NULL)
            return TCL_ERROR;
        Tcl_SetResult(interp, value, 0);
        return TCL_OK;
    }
    else if (argc == 3)
    {
        value = Tcl_SetVar2(interp, argv[1], NULL, argv[2], TCL_LEAVE_ERR_MSG);
        if (value == NULL)
            return TCL_ERROR;
        Tcl_SetResult(interp, value, 0);
	symbolAdd (req, req->curVars ? req->curVars : req->cur_pa->override,
		   argv[1], argv[2]);
        return TCL_OK;
    }
    else
    {
        Tcl_SetResult(interp, "addz varName ?newValue?", 0);
        return TCL_ERROR;
    }
}

static int cmd_callZap(ClientData cd, Tcl_Interp *interp, int argc, char **argv)
{
    ZapRequest *req = (ZapRequest *) cd;
    char *args;
#if TCL_MAJOR_VERSION > 8 || (TCL_MAJOR_VERSION == 8 && TCL_MINOR_VERSION > 0)
    Tcl_DString ds;
#endif

    if (argc < 2)
    	return TCL_ERROR;
    if (argc >= 3)
    	args = argv[2];
    else
    	args = "";

#if TCL_MAJOR_VERSION > 8 || (TCL_MAJOR_VERSION == 8 && TCL_MINOR_VERSION > 0)
    args = Tcl_UtfToExternalDString(0, args, -1, &ds);
#endif
    if (zap_request_sub(req, argv[1], args, strlen(args)))
    	return TCL_ERROR;

    if (req->result)
	Tcl_AppendResult(interp, req->result, 0);
#if TCL_MAJOR_VERSION > 8 || (TCL_MAJOR_VERSION == 8 && TCL_MINOR_VERSION > 0)
    Tcl_DStringFree (&ds);
#endif
    return TCL_OK;    
}

#define ISO2709_RS 035
#define ISO2709_FS 036
#define ISO2709_IDFS 037

static int marc_compare (const char *f, const char *p)
{
    int ch;

    if (*p == '*')
        return 0;
    if (!f)
        return -*p;
    for (; (ch = *p) && *f; f++, p++)
        switch (*p)
        {
        case '*':
            return 0;
        case '?':
            ch = *f;
            break;
        case '[':
            while (1)
                if (!*++p)
                    break;
                else if (*p == ']')
                {
                    p++;
                    break;
                }
                else if (*p == *f)
                    ch = *p;
            if (ch != *p)
                return *f - ch;
            break;
        default:
            if (ch != *f)
                return *f - ch;
        }
    return *f - ch;
}

int ir_tcl_get_marc (Tcl_Interp *interp, const char *buf, 
                     int argc, char **argv)
{
    int entry_p;
    int record_length;
    int indicator_length;
    int identifier_length;
    int base_address;
    int length_data_entry;
    int length_starting;
    int length_implementation;
    char ptag[4];
    int mode = 0;

    if (!strcmp (argv[1], "field"))
        mode = 'f';
    else if (!strcmp (argv[1], "line"))
        mode = 'l';
    else
    {
        Tcl_AppendResult (interp, "Unknown MARC extract mode", NULL);
	return TCL_ERROR;
    }
    if (!buf)
    {
        Tcl_AppendResult (interp, "Not a MARC record", NULL);
        return TCL_ERROR;
    }
    record_length = atoi_n (buf, 5);
    if (record_length < 25)
    {
        Tcl_AppendResult (interp, "Not a MARC record", NULL);
        return TCL_ERROR;
    }
    indicator_length = atoi_n (buf+10, 1);
    identifier_length = atoi_n (buf+11, 1);
    base_address = atoi_n (buf+12, 5);

    length_data_entry = atoi_n (buf+20, 1);
    length_starting = atoi_n (buf+21, 1);
    length_implementation = atoi_n (buf+22, 1);

    for (entry_p = 24; buf[entry_p] != ISO2709_FS; )
        entry_p += 3+length_data_entry+length_starting;
    base_address = entry_p+1;
    for (entry_p = 24; buf[entry_p] != ISO2709_FS; )
    {
        int data_length;
	int data_offset;
	int end_offset;
	int i, j;
	char tag[4];
	char indicator[128];
	char identifier[128];
	int identifier_flag = 1;

        *ptag = '\0';
        memcpy (tag, buf+entry_p, 3);
	entry_p += 3;
        tag[3] = '\0';
	data_length = atoi_n (buf+entry_p, length_data_entry);
	entry_p += length_data_entry;
	data_offset = atoi_n (buf+entry_p, length_starting);
	entry_p += length_starting;
	i = data_offset + base_address;
	end_offset = i+data_length-1;
	*indicator = '\0';
	if (indicator_length == 2)
	{
	    if (buf[i + indicator_length] != ISO2709_IDFS)
		identifier_flag = 0;
	}
	else if (!memcmp (tag, "00", 2))
	    identifier_flag = 0;
	    
        if (identifier_flag && indicator_length)
	{
            for (j = 0; j<indicator_length; j++)
	        indicator[j] = buf[i++];
	    indicator[j] = '\0';
	}
	if (marc_compare (tag, argv[2]) || marc_compare (indicator, argv[3]))
	    continue;
	while (buf[i] != ISO2709_RS && buf[i] != ISO2709_FS && i < end_offset)
	{
            int i0;

            if (identifier_flag && identifier_length)
	    {
	        i++;
                for (j = 1; j<identifier_length; j++)
		    identifier[j-1] = buf[i++];
		identifier[j-1] = '\0';
	        for (i0 = i; buf[i] != ISO2709_RS && 
		             buf[i] != ISO2709_IDFS &&
	                     buf[i] != ISO2709_FS && i < end_offset; 
			     i++)
		    ;
	    }
	    else
	    {
	        for (i0 = i; buf[i] != ISO2709_RS && 
	                     buf[i] != ISO2709_FS && i < end_offset; 
			     i++)
		    ;
                *identifier = '\0';
	    }
            if (marc_compare (identifier, argv[4])==0)
            {
                char *data = xmalloc (i-i0+1);
             
                memcpy (data, buf+i0, i-i0);
                data[i-i0] = '\0';
                if (mode == 'l')
                {
                    if (strcmp (tag, ptag))
                    {
                        if (*ptag)
                            Tcl_AppendResult (interp, "}} ", NULL);
                        if (!*indicator)
                            Tcl_AppendResult (interp, "{", tag, " {} {", NULL);
                        else
                            Tcl_AppendResult (interp, "{", tag, " {",
                                              indicator, "} {", NULL);
                        strcpy (ptag, tag);
                    }
                    if (!*identifier)
                        Tcl_AppendResult (interp, "{{}", NULL);
                    else
                        Tcl_AppendResult (interp, "{", identifier, NULL);
                    Tcl_AppendElement (interp, data);
                    Tcl_AppendResult (interp, "} ", NULL);
                }
                else
                    Tcl_AppendElement (interp, data);
                xfree (data);
            }
	}
        if (mode == 'l' && *ptag)
            Tcl_AppendResult (interp, "}} ", NULL);
#if 0
	if (i < end_offset)
            yaz_log (LOG_WARN, "MARC: separator but not at end of field");
	if (buf[i] != ISO2709_RS && buf[i] != ISO2709_FS)
            yaz_log (LOG_WARN, "MARC: no separator at end of field");
#endif
    }
    return TCL_OK;
}

int cmd_getMarc (ClientData cd, Tcl_Interp *interp, int argc, char **argv)
{
    ZapRequest *req = cd;
    if (argc == 5)
	return ir_tcl_get_marc (interp, req->marc_vars_buf, argc, argv);
    else
	return TCL_OK;
}

#endif


static int zap_request_core (ZapRequest *req, PA *pa, int persistent);

#if USE_APACHE
#if USE_TCL
static int cmd_virtual (ClientData clientData, Tcl_Interp *interp,
			int argc, char **argv)
{
    request_rec *rr = 0;
    ZapRequest *req = clientData;

    html_head (req);

    if (argc != 2)
	return TCL_ERROR;
#if APACHE2
    if (!(rr = ap_sub_req_lookup_uri (argv[1], req->request, 0)))
    {
	Tcl_AppendResult (interp, "bad URI for virtual", 0);
	return TCL_ERROR;
    }
#else
    if (!(rr = ap_sub_req_lookup_uri (argv[1], req->request)))
    {
	Tcl_AppendResult (interp, "bad URI for virtual", 0);
	return TCL_ERROR;
    }
#endif
    if (rr->status != 200)
    {
	char status[32];
	sprintf (status, "%d", rr->status);
	Tcl_AppendResult (interp, "virtual returned status ", status, 0);
	ap_destroy_sub_req (rr);
	return TCL_ERROR;
    }
    if (ap_run_sub_req (rr))
    {
	Tcl_AppendResult (interp, "virtual sub request failed", 0);
	ap_destroy_sub_req (rr);
	return TCL_ERROR;
    }
    ap_destroy_sub_req (rr);
    return TCL_OK;
}
#endif
#endif

#if USE_TCL

int Tcl_ZapInit (ZapRequest *req)
{
    req->tcl_interp = Tcl_CreateInterp();

    if (Tcl_Init(req->tcl_interp) == TCL_ERROR)
        return TCL_ERROR;
    Tcl_CreateCommand(req->tcl_interp, "html", cmd_html, req, 0);
    Tcl_CreateCommand(req->tcl_interp, "getGrs", cmd_getGrs, req, 0);
    Tcl_CreateCommand(req->tcl_interp, "getMarc", cmd_getMarc, req, 0);
    Tcl_CreateCommand(req->tcl_interp, "callZap", cmd_callZap, req, 0);
    Tcl_CreateCommand(req->tcl_interp, "urlenc", cmd_urlenc, req, 0);
    Tcl_CreateCommand(req->tcl_interp, "urldec", cmd_urldec, req, 0);
    Tcl_CreateCommand(req->tcl_interp, "setz", cmd_setz, req, 0);
    Tcl_CreateCommand(req->tcl_interp, "addz", cmd_addz, req, 0);
    return TCL_OK;
}

#endif


static int zap_request (ZapRequest *req)
{
    PA *cur_pa;
    int r;
#if USE_APACHE
    char *user = 0;
#endif

#if USE_TCL
    Tcl_FindExecutable("");
    Tcl_ZapInit(req);

    Tcl_UnsetVar2(req->tcl_interp, "env", "REMOTE_USER", TCL_GLOBAL_ONLY);
    Tcl_UnsetVar2(req->tcl_interp, "env", "REMOTE_IP", TCL_GLOBAL_ONLY);
#endif

#if USE_TCL
#if USE_APACHE
    if (1)
    {
        const char *h =ap_table_get (req->request->headers_in, "Host");
        if (h && strlen(h) < 80)
        {
            char name[20];
            char host[80];
            strcpy (host, h);
            strcpy (name, "HTTP_HOST");
            Tcl_SetVar(req->tcl_interp, name, host, TCL_GLOBAL_ONLY);
        }

    }      
#endif
#endif

#if USE_APACHE
#if APACHE2
    if (req->request->user)
	user = req->request->user;
#else
    if (req->request->connection && req->request->connection->user)
	user = req->request->connection->user;
#endif

    if (user)
    {
#if USE_TCL
        char remote_user[20];
        strcpy (remote_user, "env(REMOTE_USER)");
        Tcl_SetVar(req->tcl_interp, remote_user, user, TCL_GLOBAL_ONLY);
#endif
	zlog (req, "USER ", user);
    }
    if (req->request->connection && req->request->connection->remote_ip)
    {
#if USE_TCL
        char remote_user[20];
        strcpy (remote_user, "env(REMOTE_IP)");
        Tcl_SetVar(req->tcl_interp, remote_user,
                   req->request->connection->remote_ip, TCL_GLOBAL_ONLY);
#endif
	zlog (req, "IP ", req->request->connection->remote_ip);
    }
#if USE_TCL
    Tcl_CreateCommand(req->tcl_interp, "virtual", cmd_virtual, req, 0);
#endif
/* APACHE */
#endif
    cur_pa = xmalloc (sizeof(*cur_pa));
    cur_pa->args = symtabMkArgs (req);
    cur_pa->def = symtabMk ();
    cur_pa->override = symtabMk ();
    r = zap_request_core (req, cur_pa, 0);
    xfree (cur_pa);
    xfree (req->cookie_buf);
    req->cookie_buf = 0;
    xfree (req->cookie_zap);
    req->cookie_zap = 0;
    xfree (req->result);
    req->result = 0;
#if YAZ_LOG
    xmalloc_trav("exit");
#endif
    return r;
}

static int zap_request_sub (ZapRequest *old_req,
			    const char *fname, const char *args, int len)
{
    int r;
    PA *cur_pa;
    ZapRequest new_req;
    char *argsz;
#if USE_APACHE
    char *user = 0;  /* user if authenticated */
#endif

    zlog_debug (old_req, "zap_request_sub", "");
    new_req.cookie_buf = old_req->cookie_buf;
    new_req.cookie_zap = old_req->cookie_zap;
#if USE_APACHE
    new_req.request = old_req->request;
    new_req.conf = old_req->conf;
#else
    new_req.script_name = old_req->script_name;
#endif
    new_req.output_buf = old_req->output_buf;
    new_req.output_max = old_req->output_max;
    new_req.output_len = old_req->output_len;
    new_req.zap_debug = 0;
    new_req.result = 0;

#if USE_TCL
    Tcl_ZapInit(&new_req);
#if USE_APACHE
#if APACHE2
    if (new_req.request->user)
	user = new_req.request->user;
#else
    if (new_req.request->connection && new_req.request->connection->user)
	user = new_req.request->connection->user;
#endif
    if (user)
    {
        char remote_user[20];
        strcpy (remote_user, "env(REMOTE_USER)");
        Tcl_SetVar(new_req.tcl_interp, remote_user, user, TCL_GLOBAL_ONLY);
    }
#endif
#endif
    templateRead (&new_req, &new_req.curTemplate, fname);
    if (!new_req.curTemplate)
	return 0;
    cur_pa = xmalloc (sizeof(*cur_pa));
    cur_pa->args = symtabMk ();
    cur_pa->def = symtabMk ();
    cur_pa->override = symtabMk ();
    argsz = xmalloc (len+1);
    memcpy (argsz, args, len);
    argsz[len] = '\0';
    symbolsURL (&new_req, cur_pa->args, argsz);
    xfree (argsz);
    r = zap_request_core (&new_req, cur_pa, 1);

    old_req->result = new_req.result;
    old_req->output_buf = new_req.output_buf;
    old_req->output_max = new_req.output_max;
    old_req->output_len = new_req.output_len;
    old_req->cookie_zap = new_req.cookie_zap;
    xfree (cur_pa);
    return r;
}

static void destroy_ccl(ZapRequest *req)
{
    ccl_qual_rm(&req->ccl_bibset);
}

static CCL_bibset initialize_ccl_bibset (ZapRequest *req, const char *name)
{
    FILE *f = 0;
    char *org_fname, *cp;
    char fname[256];
    int org_len;
    CCL_bibset bibset = ccl_qual_mk ();
    const char *setting;

#if USE_APACHE
    org_fname = req->request->filename;
#else
    org_fname = req->script_name;
#endif
    org_len = strlen(org_fname);
    cp = 0;
    if (!cp)
        cp = strrchr (org_fname, '/');
    if (!cp)
        cp = strrchr (org_fname, '\\');
    if (cp)
    {
        cp++;
        org_len = cp - org_fname;
    }
    strncpy (fname, name, sizeof(fname)-1);
    fname[sizeof(fname)-1] = '\0';
    if (*fname != '/' && *fname != '\\' && org_fname && org_len)
    {
	memcpy (fname, org_fname, org_len);
	strcpy (fname + org_len, name);
    }
    f = fopen(fname, "r");
    if (!f)
    {
	ccl_qual_rm(&bibset);
	return 0;
    }
    ccl_qual_file(bibset, f);
    fclose(f);
    if ((setting = symbolLookupFormStr (req, "cclopand", 0)))
	ccl_qual_fitem(bibset, setting, "@and");

    if ((setting = symbolLookupFormStr (req, "cclopor", 0)))
	ccl_qual_fitem(bibset, setting, "@or");

    if ((setting = symbolLookupFormStr (req, "cclopnot", 0)))
	ccl_qual_fitem(bibset, setting, "@not");

    if ((setting = symbolLookupFormStr (req, "cclopset", 0)))
	ccl_qual_fitem(bibset, setting, "@set");

    if ((setting = symbolLookupFormStr (req, "cclcase", 0)))
	ccl_qual_fitem(bibset, setting, "@case");

    return bibset;
}

static void initialize_ccl(ZapRequest *req)
{
    req->ccl_bibset = initialize_ccl_bibset (req, "cclfields.zap");
}

static void targetsLeave (ZapRequest *req);

static int zap_request_core (ZapRequest *req, PA *pa, int persistent)
{
#if USE_TCL
    const char *result;
#endif
    int found_zap_cookie = 0;
    void (*handler)() = NULL;

    req->html_buffer = 0;
    req->html_len = req->html_max = 0;

    req->cur_pa = pa;

    req->curVars = 0;
    req->zap_debug = atoi (symbolLookupFormStr (req, "debug", "0"));
    
    if (req->cookie_buf)
    {
	const char *cp = req->cookie_buf;
	while (*cp)
	{
	    const char *name = cp;
	    const char *val = 0;
	    char *namez = 0;
	    while (*cp && *cp != '=')
		cp++;
	    if (cp > name)
	    {
		namez = xmalloc (cp - name + 20);
		sprintf (namez, "cookie(%.*s)", cp - name, name);
	    }
	    else
		break;
	    if (*cp != '=')
		break;
	    cp++;
	    if (*cp == '"')
	    {
		cp++;
		val = cp;
		while (*cp && *cp != '"')
		    cp++;
	    }
	    else
	    {
		val = cp;
		while (*cp && !strchr (",; ", *cp))
		    cp++;
	    }
	    if (!strcmp (namez, "cookie(ZAP)"))
	    {
		xfree (req->cookie_zap);
		req->cookie_zap = xmalloc (cp - val + 1);
		memcpy (req->cookie_zap, val, cp - val);
		req->cookie_zap[cp - val] = '\0';
		found_zap_cookie = 1;
	    }
	    symbolSetN (req, req->cur_pa->args, namez, val, cp - val);
	    while (*cp && strchr ("\",; ", *cp))
		cp++;
	}
    }
    if (req->cookie_zap)
	symbolSet (req, req->cur_pa->override, "cookie(ZAP)", req->cookie_zap);
    templateDef (req, req->cur_pa, req->curTemplate, "def",
		 req->cur_pa->def);
    templateDef (req, req->cur_pa, req->curTemplate, "override",
		 req->cur_pa->override);

    req->curVars = symtabMk ();
    
    req->zap_debug = atoi (symbolLookupFormStr (req, "debug", "0"));
    req->zap_timeout = atoi (symbolLookupFormStr (req, "timeout", "25"));

    html_dump (req, "begin");

    initialize_ccl(req);
	
    if (req->zap_debug)
    {
	html_puts (req, "ZAP ");
	html_puts (req, ZAP_VERSION);
	html_puts (req, "<br>");
	html_puts (req, "Date ");
	html_puts (req, ZAP_DATE);
	html_puts (req, "<br>");
	html_puts (req, "<dl><dt>Override<dd>");
	symtabDump (req, req->cur_pa->override);
	html_puts (req, "<dt>Args<dd>");
	symtabDump (req, req->cur_pa->args);
	html_puts (req, "<dt>Def<dd>");
	symtabDump (req, req->cur_pa->def);
	html_puts (req, "</dl><hr>");
    }

#ifndef WIN32
    handler = signal (SIGPIPE, SIG_IGN);
#endif
    targetsMk (req, persistent);
    session (req);

    if (!persistent)
	targetsDestroy (req, atoi (symbolLookupFormStr (req, "cache", "0")));
    else
	targetsLeave (req);
#ifndef WIN32
    signal (SIGPIPE, handler);
#endif

    html_dump (req, "end");

    destroy_ccl(req);
    xfree (req->result);
    req->result = 0;
#if USE_TCL
    result = Tcl_GetVar(req->tcl_interp, "result", TCL_LEAVE_ERR_MSG);
    req->result = xstrdup(result ? result : "");
    Tcl_DeleteInterp(req->tcl_interp);
#endif
    
    symtabDestroy (&req->cur_pa->args);
    symtabDestroy (&req->cur_pa->def);
    symtabDestroy (&req->cur_pa->override);
    symtabDestroy (&req->curVars);
    templateDestroy (&req->curTemplate);
    return 0;
}

#if USE_APACHE

static int zap_handler (request_rec *rptr)
{
    const char *cookie_in = 0;
    int retval;
    char pidstr[32];
    ZapRequest req;
    void *sconf = rptr->server->module_config;
    zap_server_conf *conf =
    (zap_server_conf *) ap_get_module_config(sconf, &zap_module);

    zap_init (&req);
    req.request = rptr;
    req.conf = conf;
 
    sprintf (pidstr, "%u", getpid());
    zlog (&req, "zap begin pid=", pidstr);
#if YAZ_LOG
    yaz_log_init(LOG_ALL, pidstr, "/home/adam/proj/apache/yaz_log");
#endif
    zlog (&req, "script=", rptr->filename); 
    zlog (&req, "uri=", rptr->uri);
    zlog (&req, "path_info=", rptr->path_info);
    if (conf->apdu_name)
	zprintf (&req, "apdulog=%s", conf->apdu_name);

    cookie_in = ap_table_get (rptr->headers_in, "Cookie");
    if (cookie_in)
    {
        req.cookie_buf = xstrdup (cookie_in);
        zlog (&req, "Cookie buf = ", req.cookie_buf);
    }
    req.cookie_zap = xmalloc (40);
    if (req.cookie_zap)
    {
	static long seqno = 0;
	static long t = 0;
	if (seqno == 0)
	    t = time(0);
	sprintf (req.cookie_zap, "%lX,%lX,%lX", t, (long) getpid(), seqno);
	seqno++;
    }
#if APACHE2
#else
    if (rptr->finfo.st_mode == 0)
	return NOT_FOUND;
#endif
    templateRead (&req, &req.curTemplate, rptr->filename);
    if (!req.curTemplate)
	return HTTP_FORBIDDEN;
 
    rptr->content_type = "text/html";
#if APACHE2
#else
    ap_soft_timeout ("send status info", rptr);
#endif
    if ((retval = ap_setup_client_block(rptr, REQUEST_CHUNKED_ERROR)))
        return retval;
    retval = zap_request (&req);
    zlog (&req, "zap end pid=", pidstr);
    return retval;
}

static void *create_dir_config(apr_pool_t *p, char *d)
{
    zap_dir_conf *conf = ap_pcalloc(p, sizeof(*conf));
    conf->auth_type = 0;
    return conf;
}

static void *create_config (apr_pool_t *p, server_rec *s)
{
    zap_server_conf *conf = ap_pcalloc (p, sizeof(*conf));

    conf->log_name = ZAP_LOG;
    conf->log_file = 0;

    conf->apdu_name = 0;
    conf->apdu_file = 0;
    return conf;
}

static const char *set_log_name(cmd_parms *cmd, void *dummy, const char *arg)
{
    server_rec *s = cmd->server;
    zap_server_conf *conf = (zap_server_conf *)
	ap_get_module_config(s->module_config, &zap_module);

    conf->log_name = arg;
    return NULL;
}

static const char *set_apdu_name(cmd_parms *cmd, void *dummy, const char *arg)
{
    server_rec *s = cmd->server;
    zap_server_conf *conf = (zap_server_conf *)
	ap_get_module_config(s->module_config, &zap_module);

    conf->apdu_name = arg;
    return NULL;
}

static const command_rec zap_cmds[] =
{
#if defined(AP_HAVE_DESIGNATED_INITIALIZER)
    AP_INIT_TAKE1("ZapLog", set_log_name, NULL, RSRC_CONF, "the name of the zap log file")
    ,
    AP_INIT_TAKE1("APDULog", set_apdu_name, NULL, RSRC_CONF, "the name of the APDU log file")
    ,
#else
    {"ZapLog", set_log_name, NULL, RSRC_CONF, TAKE1,
     "the name of zap log file"},
    {"APDULog", set_apdu_name, NULL, RSRC_CONF, TAKE1,
     "the name of APDU log file"},
#endif
    {NULL}
};

static void init_module(server_rec *s, apr_pool_t *p)
{
    for (; s; s = s->next) {
	zap_server_conf *conf = (zap_server_conf *)
	    ap_get_module_config(s->module_config, &zap_module);
	zlog_open (conf, p, s);
    }
#if YAZ_LOG
    log_init(LOG_ALL, "zap", "/home/adam/proj/apache/yaz_log");
    yaz_log (LOG_LOG, "yaz log started");
#endif
}

#if APACHE2
/* Apache 2 handlers */
static int zap_hook_init(apr_pool_t *p, apr_pool_t *ptemp,
			 apr_pool_t *plog, server_rec *s) {
    ap_add_version_component(p, "zap/" ZAP_VERSION);
    init_module(s, p);
    return OK;
} 

static int zap_handler2 (request_rec *r)
{
    int e;
    if (strcmp(r->handler, ZAP_MAGIC_TYPE) && 
	strcmp(r->handler, ZAP_SCRIPT_TYPE)) {
	return DECLINED;
    }
    return zap_handler(r);
}

static void register_zap_hooks(apr_pool_t *p) {
    ap_hook_post_config(zap_hook_init, NULL, NULL, APR_HOOK_MIDDLE);
    ap_hook_handler(zap_handler2, NULL, NULL, APR_HOOK_MIDDLE);
}
 
module AP_MODULE_DECLARE_DATA zap_module = {
    STANDARD20_MODULE_STUFF,
    create_dir_config,          /* dir config creater */
    NULL,                       /* dir merger --- default is to override */
    create_config,              /* server config */
    NULL,                       /* merge server config */
    zap_cmds,                   /* command table */
    register_zap_hooks          /* register hooks */
};
 
#else
/* Apache 1 handlers */
handler_rec zap_handlers[] =
{
    { ZAP_MAGIC_TYPE, zap_handler },
    { ZAP_SCRIPT_TYPE, zap_handler },
    { NULL }
};

module zap_module =
{
    STANDARD_MODULE_STUFF,
    init_module,                 /* initializer */
    create_dir_config,           /* dir config creater */
    NULL,                        /* dir merger --- default is to override */
    create_config,               /* server config */
    NULL,                        /* merge server config */
    zap_cmds,                    /* command table */
    zap_handlers,                /* handlers */
    NULL,                        /* filename translation */
    NULL,                        /* check_user_id */
    NULL,                        /* check auth */
    NULL,                        /* check access */
    NULL,                        /* type_checker */
    NULL,                        /* fixups */
    NULL                         /* logger */
};
#endif
#else
/* CGI */

int main (int argc, char **argv)
{
    int retval = 0;
    char pidstr[60];
    char fname[256];
    char *fname_part;
    const char *cookie_in;
    char *path_info;

    ZapRequest req;

    zap_init (&req);
#ifdef WIN32
    sprintf (pidstr, "%d,%u", time(0), GetCurrentProcessId());
#else
    sprintf (pidstr, "%ld,%lu", (long) time(0), (long) getpid());
#endif
    zlog (&req, "begin ", pidstr);
    cookie_in = getenv("HTTP_COOKIE");
    if (cookie_in)
	req.cookie_buf = xstrdup (cookie_in);
   
    req.cookie_zap = xstrdup (pidstr);
    fname_part = getenv("PATH_TRANSLATED");
    if (fname_part)
    {
        strncpy (fname, fname_part, sizeof(fname)-1);
        fname[sizeof(fname)-1] = '\0';
    }
    else
    {
        strcpy (fname, ZAP_CONF);
        strcat (fname, "/");
        fname_part = fname + strlen(fname);
        strcpy (fname_part, "default.zap");

        path_info = getenv ("PATH_INFO");
        zlog (&req, "path_info=", path_info);
        if (path_info && *path_info)
        {
	    char *cp = path_info+1;
            while (*cp && *cp != '?')
	        cp++;
	    if (cp && cp - path_info > 1)
	    {
	        int len = cp - path_info - 1;
	        if (len > 80)
		    len = 80;
	        memcpy (fname_part, path_info+1, len);
	        fname_part[len] = '\0';
	    }
        }
    }
    zlog (&req, "script=", fname);
    req.script_name = fname;

    templateRead (&req, &req.curTemplate, fname);
    if (!req.curTemplate)
    {
	raw_puts (&req, "Content-type: text/html\n");
	raw_puts (&req,"<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML 2.0//EN\">");
	raw_puts (&req, "<HTML><HEAD><TITLE>Not Found</TITLE>");
	raw_puts (&req, "</HEAD><BODY>");
	raw_puts (&req, "<H1>Not Found</H1>");
	raw_puts (&req, "ZAP Script");
	raw_puts (&req, fname);
	raw_puts (&req, "was not found on this server.<P>");
	raw_puts (&req, "</BODY></HTML>");
    }
    else
    {
	retval = zap_request (&req);
    }
    zlog (&req, "end ", pidstr);
    exit (0);
}
#endif

/* --- XML-->GRS-1 Mapping ------------------------------------ */

static Z_GenericRecord *text2grs1(char **text, int *len, NMEM o, int level,
				  Z_TaggedElement *wellknown);

/* Read a tagged element - the opening "<" has already been read */
static Z_TaggedElement *text2taggedelement(char **text,
					   int *len, NMEM o, int level)
{
    char tag[32], *p;
    int i = 0;
    Z_TaggedElement *r = nmem_malloc(o, sizeof(*r));
    Z_StringOrNumeric *tagValue = nmem_malloc(o, sizeof(*tagValue));
    Z_ElementData *content = nmem_malloc(o, sizeof(*content));
    static int three = 3;

    r->tagType = &three;
    r->tagOccurrence = 0;
    r->metaData = 0;
    r->appliedVariant = 0;
    r->tagValue = tagValue;
    r->content = content;

    while (*len && isspace(**text)) { (*text)++; (*len)--; }
    while (*len && i < 30 && **text != '>' && !isspace(**text))
    {
	tag[i] = **text;
	i++; (*text)++; (*len)--;
    }
    if (!i || !*len)
	return 0;
    tag[i] = '\0';
    while (*len && **text != '>')
    {
	(*text)++;
	(*len)--;
    }
    if (*len == 0 || **text != '>')
	return 0;
    (*text)++;
    (*len)--;
    tagValue->which = Z_StringOrNumeric_string;
    tagValue->u.string = nmem_strdup(o, tag);

    while (*len && isspace(**text)) { (*text)++; (*len)--; }
    if (!*len)
	return 0;
    if (**text == '<')  /* Composite element */
    {
	r->content->which = Z_ElementData_subtree;
	r->content->u.subtree = text2grs1(text, len, o, level + 1, 0);
    }
    else /* Simple element */
    {
	p = *text;
	i = 0;
	while (*len && **text != '<')
	{
	    (*text)++;
	    (*len)--;
	    i++;
	}
	if (*len < 2)
            return 0;

        if (*(*text + 1) != '/')
	{   /* well known .. */
	    static int one = 1;
            Z_TaggedElement *subr = nmem_malloc(o, sizeof(*subr));
            subr->tagType = &one;

	    subr->tagValue = nmem_malloc(o, sizeof(*subr->tagValue));
            subr->tagValue->which = Z_StringOrNumeric_numeric;
            subr->tagValue->u.numeric =
                nmem_malloc (o, sizeof(*subr->tagValue->u.numeric));
            *subr->tagValue->u.numeric = 19;

            subr->content = nmem_malloc(o, sizeof(*subr->content));
	
	    subr->content->which = Z_ElementData_string;
	    subr->content->u.string = nmem_malloc(o, i + 1);
	    memcpy(subr->content->u.string, p, i);
	    subr->content->u.string[i] = '\0';

    	    subr->tagOccurrence = 0;
	    subr->metaData = 0;
	    subr->appliedVariant = 0;

	    r->content->which = Z_ElementData_subtree;
	    r->content->u.subtree = text2grs1(text, len, o, level + 1, subr);
        }
        else
        {
	    r->content->which = Z_ElementData_string;
	    r->content->u.string = nmem_malloc(o, i + 1);
	    memcpy(r->content->u.string, p, i);
	    r->content->u.string[i] = '\0';

	    while (*len && **text != '>')
	    {
	        (*text)++;
	        (*len)--;
	    }
            if (*len)
            {
		(*text)++;
		(*len)++;
            }
        }
    }
    return r;
}

/* Convert a text buffer containing XML into GRS-1 */
static Z_GenericRecord *text2grs1(char **text, int *len, NMEM o, int level,
				  Z_TaggedElement *wellknown)
{
    Z_GenericRecord *r = nmem_malloc(o, sizeof(*r));
    int elem_alloc = 50;

    r->num_elements = 0;
    r->elements = nmem_malloc(o, sizeof(*r->elements) * elem_alloc);

    if (wellknown)
        r->elements[r->num_elements++] = wellknown;
    while (*len)
    {
    	while (*len && isspace(**text)) { (*text)++; (*len)--; }
	if (**text != '<' || *len < 2)
	    return r;
	(*text)++; (*len)--;
	if (**text == '/') /* end tag */
	{
	    while (*len && **text != '>') { (*text)++; (*len)--; }
            if (*len)
            {
	        (*len)--;
	        (*text)++;
            }
	    if (level == 1)
		level--;
	    else
		return r;
        }
	else if (**text == '!') /* comment */
	{
	    while (*len && **text != '>') { (*text)++; (*len)--; }
	    (*text)++;
	    (*len)--;
	}
	else if (level == 0) /* root tag */
	{
	    /* this is the outer tag; not part of the record contents.
	     * Technically, we should note the name, but right now we don't */
	    while (*len && **text != '>') { (*text)++; (*len)--; }
	    if (*len <= 1) return 0;
	    (*text)++; (*len)--;
	    level = 1;
        }
	else /* We have a regular tag opening marker */
	{
	    if ((r->num_elements + 1) >= elem_alloc)
	    {
	        Z_TaggedElement **tmp = r->elements;

	    	/* double array space, throw away old buffer (nibble memory) */
	    	r->elements = nmem_malloc(o, sizeof(*r->elements) *
		    (elem_alloc *= 2));
		memcpy(r->elements, tmp, r->num_elements * sizeof(*tmp));
	    }
	    r->elements[r->num_elements] =
                text2taggedelement(text, len, o, level);
	    r->num_elements++;
    	}
    }
    return r;
}
		

/* --- MARC Formatting ---------------------------------------- */

static Z_GenericRecord *marc_to_grs1(ZapRequest *req, const char *buf, NMEM o,
				     Odr_oid *oid, int debug)
{
    int entry_p;
    int record_length;
    int indicator_length;
    int identifier_length;
    int base_address;
    int length_data_entry;
    int length_starting;
    int length_implementation;
    int max_elements = 256;
    Z_GenericRecord *r = nmem_malloc (o, sizeof(*r));
    r->elements = nmem_malloc (o, sizeof(*r->elements) * max_elements);
    r->num_elements = 0;

    record_length = atoi_n (buf, 5);
    if (record_length < 25)
        return 0;

    if (isdigit(buf[10]))
        indicator_length = atoi_n (buf+10, 1);
    else
        indicator_length = 2;
    if (isdigit(buf[11]))
	identifier_length = atoi_n (buf+11, 1);
    else
        identifier_length = 2;

    base_address = atoi_n (buf+12, 5);

    length_data_entry = atoi_n (buf+20, 1);
    length_starting = atoi_n (buf+21, 1);
    length_implementation = atoi_n (buf+22, 1);

    if (debug)
    {
	zprintf (req,  "Record length         %5d\n", record_length);
	zprintf (req,  "Indicator length      %5d\n", indicator_length);
	zprintf (req,  "Identifier length     %5d\n", identifier_length);
	zprintf (req,  "Base address          %5d\n", base_address);
	zprintf (req,  "Length data entry     %5d\n", length_data_entry);
	zprintf (req,  "Length starting       %5d\n", length_starting);
	zprintf (req,  "Length implementation %5d\n", length_implementation);
    }
    for (entry_p = 24; buf[entry_p] != ISO2709_FS; )
    {
        entry_p += 3+length_data_entry+length_starting;
        if (entry_p >= record_length)
            return 0;
    }
    base_address = entry_p+1;
    for (entry_p = 24; buf[entry_p] != ISO2709_FS; )
    {
	Z_TaggedElement *tag;
        int data_length;
	int data_offset;
	int end_offset;
	int i;
	char tag_str[4];
	int identifier_flag = 1;

        memcpy (tag_str, buf+entry_p, 3);
	entry_p += 3;
        tag_str[3] = '\0';

        if ((r->num_elements + 1) >= max_elements)
	{
	    Z_TaggedElement **tmp = r->elements;

	    /* double array space, throw away old buffer (nibble memory) */
	    r->elements = nmem_malloc(o, sizeof(*r->elements) *
		    (max_elements *= 2));
	    memcpy(r->elements, tmp, r->num_elements * sizeof(*tmp));
	}
	tag = r->elements[r->num_elements++] = nmem_malloc (o, sizeof(*tag));
	tag->tagType = nmem_malloc(o, sizeof(*tag->tagType));
	*tag->tagType = 3;
	tag->tagOccurrence = 0;
	tag->metaData = 0;
	tag->appliedVariant = 0;
	tag->tagValue = nmem_malloc (o, sizeof(*tag->tagValue));
	tag->tagValue->which = Z_StringOrNumeric_string;
	tag->tagValue->u.string = nmem_strdup(o, tag_str);

	tag->content = nmem_malloc(o, sizeof(*tag->content));
	tag->content->which = Z_ElementData_subtree;

	tag->content->u.subtree =
	    nmem_malloc (o, sizeof(*tag->content->u.subtree));
	tag->content->u.subtree->elements = nmem_malloc (o, sizeof(*r->elements));
	tag->content->u.subtree->num_elements = 1;

	tag = tag->content->u.subtree->elements[0] =
	    nmem_malloc (o, sizeof(**tag->content->u.subtree->elements));

	tag->tagType = nmem_malloc(o, sizeof(*tag->tagType));
	*tag->tagType = 3;
	tag->tagOccurrence = 0;
	tag->metaData = 0;
	tag->appliedVariant = 0;
	tag->tagValue = nmem_malloc (o, sizeof(*tag->tagValue));
	tag->tagValue->which = Z_StringOrNumeric_string;
	tag->content = nmem_malloc(o, sizeof(*tag->content));

	data_length = atoi_n (buf+entry_p, length_data_entry);
	entry_p += length_data_entry;
	data_offset = atoi_n (buf+entry_p, length_starting);
	entry_p += length_starting;
	i = data_offset + base_address;
	end_offset = i+data_length-1;

	if (indicator_length == 2)
	{
	    if (buf[i + indicator_length] != ISO2709_IDFS)
		identifier_flag = 0;
	}
	else if (!memcmp (tag, "00", 2))
	    identifier_flag = 0;
	
        if (identifier_flag && indicator_length)
	{
	    /* indicator */
	    tag->tagValue->u.string = nmem_malloc(o, indicator_length+1);
	    memcpy (tag->tagValue->u.string, buf + i, indicator_length);
	    tag->tagValue->u.string[indicator_length] = '\0';
	    i += indicator_length;

	    tag->content->which = Z_ElementData_subtree;

	    tag->content->u.subtree =
		nmem_malloc (o, sizeof(*tag->content->u.subtree));
	    tag->content->u.subtree->elements =
		nmem_malloc (o, 256 * sizeof(*r->elements));
	    tag->content->u.subtree->num_elements = 0;
	    
	    while (buf[i] != ISO2709_RS && buf[i] != ISO2709_FS
		   && i < end_offset)
	    {
		int i0;
		/* prepare tag */
                Z_TaggedElement *parent_tag = tag;
		Z_TaggedElement *tag = nmem_malloc (o, sizeof(*tag));

                if (parent_tag->content->u.subtree->num_elements < 256)
                    parent_tag->content->u.subtree->elements[
                         parent_tag->content->u.subtree->num_elements++] = tag;

		tag->tagType = nmem_malloc(o, sizeof(*tag->tagType));
		*tag->tagType = 3;
		tag->tagOccurrence = 0;
		tag->metaData = 0;
		tag->appliedVariant = 0;
		tag->tagValue = nmem_malloc (o, sizeof(*tag->tagValue));
		tag->tagValue->which = Z_StringOrNumeric_string;
		
		/* sub field */
		tag->tagValue->u.string = nmem_malloc (o, identifier_length);
		memcpy (tag->tagValue->u.string, buf+i+1, identifier_length-1);
		tag->tagValue->u.string[identifier_length-1] = '\0';
		i += identifier_length;

		/* data ... */
		tag->content = nmem_malloc(o, sizeof(*tag->content));
		tag->content->which = Z_ElementData_string;

		i0 = i;
	        while (buf[i] != ISO2709_RS && buf[i] != ISO2709_IDFS &&
	               buf[i] != ISO2709_FS && i < end_offset)
		    i++;

		tag->content->u.string = nmem_malloc (o, i - i0 + 1);
		memcpy (tag->content->u.string, buf + i0, i - i0);
		tag->content->u.string[i - i0] = '\0';
	    }
	}
	else
	{
	    int i0 = i;

	    tag->tagValue->u.string = "@";
	    tag->content->which = Z_ElementData_string;
	    
	    while (buf[i] != ISO2709_RS && buf[i] != ISO2709_FS &&
		   i < end_offset)
		i++;
	    tag->content->u.string = nmem_malloc (o, i - i0 +1);
	    memcpy (tag->content->u.string, buf+i0, i - i0);
	    tag->content->u.string[i-i0] = '\0';
	}
	if (i < end_offset)
	    zprintf (req,  "-- separator but not at end of field\n");
	if (buf[i] != ISO2709_RS && buf[i] != ISO2709_FS)
	    zprintf (req,  "-- no separator at end of field\n");
    }
    return r;
}

/* --- GRS-1 Formatting --------------------------------------- */

typedef struct {
    int set;
    int which;
    union {
	int numeric;
	char string[32];
    } value;
    int sequenceStart;
    int sequenceNo;
} Element;

#define Element_Numeric  0
#define Element_String   1
#define Element_Any      2
#define Element_Unknown  3

void elementTagStr (char *dst, const Element *e)
{
    if (e->which == Element_Numeric)
	sprintf (dst, "%d", e->value.numeric);
    else if (e->which == Element_String)
	strcpy (dst, e->value.string);
    else if (e->which == Element_Any)
	strcpy (dst, "*");
    else
	strcpy (dst, "?");
}

void elementStr (char *dst, const Element *e)
{
    if (e->set >= 0)
	sprintf (dst, "(%d,", e->set);
    else
	sprintf (dst, "(*,");
    elementTagStr (strlen(dst)+dst, e);
    strcat (dst, ")");
}

int elementMatch (const Element *a, const Element *b)
{
    char buf[128];

    elementStr (buf, a);
    strcat (buf, " ");
    elementStr (buf+strlen(buf), b);

    if (a->set >= 0 && b->set >= 0 && a->set != b->set)
	return 0;
    if (a->which == Element_Any || b->which == Element_Any)
	return 1;
    if (a->which != b->which)
	return 0;
    if (a->which == Element_Numeric)
    {
	if (a->value.numeric != b->value.numeric)
	    return 0;
    }
    else if (a->which == Element_String)
    {
	if (strcmp (a->value.string, b->value.string))
	    return 0;
    }
    return 1;
}

void elementRecord (const Z_TaggedElement *p, Element *e)
{
    if (p->tagType)
	e->set = *p->tagType;
    else
	e->set = 3;
    if (p->tagValue->which == Z_StringOrNumeric_string)
    {
	int len = strlen(p->tagValue->u.string);
	if (len > sizeof(e->value.string)-1)
	    len = sizeof(e->value.string)-1;
	e->which = Element_String;
	memcpy (e->value.string, p->tagValue->u.string, len);
	e->value.string[len] = '\0';
    }
    else if (p->tagValue->which == Z_StringOrNumeric_numeric)
    {
	e->which = Element_Numeric;
	e->value.numeric = *p->tagValue->u.numeric;
    }
    else
	e->which = Element_Unknown;
}

int elementRead (const char **cpp, Element *e)
{
    const char *cp = *cpp;
    const char *elem_sep= " \t\n/:";

    e->sequenceStart = 1;
    e->sequenceNo = 100000;
    if (*cp == '*')
    {
	e->set = -1;
	e->which = Element_Any;
	cp++;
    }
    else if (*cp == '(')
    {
	cp++;
	if (*cp == '?')
	{
	    e->set = -1;
	    cp++;
	}
	else
	{
	    e->set = atoi(cp);
	    while (*cp && isdigit(*cp))
		cp++;
	}
	if (*cp != ',')
	    return -1;
	cp++;
	if (*cp == '*')
	{
	    e->which = Element_Any;
	    cp++;
	}
	else if (isdigit (*cp))
	{
	    e->which = Element_Numeric;
	    e->value.numeric = atoi (cp);
	    while (*cp && *cp != ')')
		cp++;
	}
	else
	{
	    const char *cp0 = cp;
	    int len;
	    
	    while (*cp && *cp != ')')
		cp++;
	    len = cp - cp0;
	    if (len > sizeof(e->value.string) -1)
		len = sizeof(e->value.string) -1;
	    e->which = Element_String;
	    memcpy (e->value.string, cp0, len);
	    e->value.string[len] = '\0';
	}
	if (*cp == ')')
	    cp++;
    }
    else if (*cp && !strchr(elem_sep, *cp))
    {
    	int i=0;

	e->set = 3;
	e->which = Element_String;
	do
	{
	    e->value.string[i] = *cp;    
	    cp++;
	    i++;
	} while (i < 31 && *cp && !strchr(elem_sep, *cp));
	e->value.string[i] = '\0';
    }
    else
	return 0;

    if (*cp == ':')
    {
	cp++;
	if (cp[0] == 'l' && cp[1] == 'a' && cp[2] == 's' && cp[3] == 't')
	{
	    cp += 4;
	    e->sequenceStart = -1;
	}
	if (isdigit(*cp))
	{
	    e->sequenceStart = atoi(cp);
	    while (isdigit(*cp))
		cp++;
	    if (!*cp != '+')
		e->sequenceNo = 1;
	    else
	    {
		cp++;
		if (isdigit(*cp))
		{
		    e->sequenceNo = atoi(cp);
		    while (isdigit(*cp))
			cp++;
		}
	    }
	}
    }
    if (*cp == '/')
	    cp++;
    *cpp = cp;
    return 1;
}

const char *grs1_vars_handler (ZapRequest *req, const char *cp)
{
    Element path[10];
    int noLevels = 0;
    int level = 0;
    int i = 0;
    Z_GenericRecord *r = req->grs1_vars_rec_rel;

    if (*cp == '/')   
    {
        r = req->grs1_vars_rec_abs;
        cp++;
    }
    while (elementRead (&cp, &path[noLevels]) == 1)
	noLevels++;
    if (!noLevels)
	return NULL;
    while (i < r->num_elements)
    {
	Element elementR;
	int leaf = (level == noLevels-1);

	elementRecord (r->elements[i], &elementR);
	if (elementMatch (&path[level], &elementR))
	{
	    Z_ElementData *content = r->elements[i]->content;
	    
	    switch (content->which)
	    {
	    case Z_ElementData_string:
		if (leaf)
		    return content->u.string;
		break;
            case Z_ElementData_numeric:
		if (leaf)
		{
		    static char numStr[32];
		    sprintf (numStr, "%d", *content->u.numeric);
		    return numStr;
		}
		break;
	    case Z_ElementData_subtree:
		if (!leaf)
		{
		    r = content->u.subtree;
		    i = 0;
		    level++;
		    continue;
		}
	    }
	}
	i++;
    }
    return NULL;
}

void grs1_dump_rule (ZapRequest *req, const char *cp,
		     int ch, int def, Z_GenericRecord *rec)
{
    req->grs1_vars_rec_rel = rec;
    while (*cp)
    {
	int yes = 0;
	const char *stopChr;
	if (cp[1] == ':' && (cp[2] == '\"' || cp[2] == '{'))
	{
	    if (*cp == ch || ch == 'a')
		yes = 1;
	    cp+=2;
	}
	else if (*cp == '\"' || *cp == '{')
	{
	    if (def)
		yes = 1;
	    else
		yes = 0;
	}
	else
	{
	    cp++;
	    continue;
	}
	if (*cp == '\"')
	    stopChr = "\"";
	else
	    stopChr = "}";
	++cp;
	templateDumpRule (req, req->curVars, &cp, stopChr,
			  yes, 0, grs1_vars_handler);
	if (*cp)
	    cp++;
    }
}


void grs1_dump_r (ZapRequest *req, struct TemplateEntry **tep,
		  Z_GenericRecord *r,
		  int level, Element *path)
{
    int i;
    
    while (*tep && !(*tep)->head)
    {
	struct TemplateEntry *te_next = (*tep)->next;
	const char *cp = (*tep)->buf;
	Element thisElement;     /* last element for rule (leaf) */
	char pathstr[128], *pathcp = pathstr;
	int sequenceCur = 0;
	int leaf = 1;

	*pathcp = '\0';
	
	/* read tag (rule) path */
	for (i = 0; i<level; i++)
	{
	    Element element;
	    if (elementRead (&cp, &element) != 1)
		return;
	    /* see if it matches the previous rule */
	    if (!elementMatch (&element, &path[i]))
		return;
	    elementStr (pathcp, &element);
	    strcat (pathcp, "/");
	    pathcp += strlen(pathcp);
	}
	*tep = te_next;
	if (*cp == '!')
	{
	    cp++;
	    while (*cp == ' ' || *cp == '\t')
		cp++;
	    grs1_dump_rule (req, cp, 'd', 1, r);
	    continue;
	}
	if (elementRead (&cp, &thisElement) != 1)
	    continue;
	elementStr (pathcp, &thisElement);

	if (elementRead (&cp, &path[level+1]) == 1)
	{
	    leaf = 0;
	}
	while (*cp == ' ' || *cp == '\t')
	    cp++;
	for (i = 0; i<r->num_elements; i++)
	{
	    Element elementR;     /* element for GRS-1 record */
	    
	    elementRecord (r->elements[i], &elementR);
	    if (elementMatch (&thisElement, &elementR))
	    {
		Z_ElementData *content = r->elements[i]->content;
		char varStr[128];

		*tep = te_next;
		if (leaf)
		{
		    sequenceCur++;
		    if (sequenceCur < thisElement.sequenceStart ||
		    sequenceCur >=
			thisElement.sequenceStart + thisElement.sequenceNo)
			continue;
		    html_var (req,"tagpath", pathstr);
		    elementTagStr (varStr, &elementR);
		    html_var (req, "tagvalue", varStr);
		    sprintf (varStr, "%d", elementR.set);
		    html_var (req, "tagset", varStr);
		    sprintf (varStr, "%d", sequenceCur);
		    html_var (req, "sequence", varStr);
		}
		switch (content->which)
		{
		case Z_ElementData_string:
		    if (leaf)
		    {
			html_var (req, "data", content->u.string);
			grs1_dump_rule (req, cp, 'd', 1, r);
		    }
		    break;
                case Z_ElementData_numeric:
		    if (leaf)
		    {
                        sprintf (varStr, "%d", *content->u.numeric);
			html_var (req, "data", varStr);
			grs1_dump_rule (req, cp, 'd', 1, r);
		    }
                    break;
		case Z_ElementData_subtree:
		    memcpy (&path[level], &thisElement, sizeof(thisElement));
		    if (leaf)
		    {
			grs1_dump_rule (req, cp, 'b', 0, content->u.subtree);
		    }
		    grs1_dump_r (req, tep, content->u.subtree, level+1, path);
		    if (leaf)
		    {
			grs1_dump_rule (req, cp, 'e', 0, content->u.subtree);
		    }
		    break;
		}
	    }
	}
    }
}

void grs1_dump (ZapRequest *req, Z_GenericRecord *r, char *dumpstr)
{
    struct TemplateEntry *te =
	templateFindEntry (req, req->curTemplate, dumpstr);
    struct TemplateEntry **tep = &te;
    Element path[10];
    req->grs1_vars_rec_abs = r;

    if (!r || !te)
	return ;
    te = te->next;
    grs1_dump_r (req, tep, r, 0, path);
}


void sutrs_dump (ZapRequest *req, NMEM o, Z_SUTRS *r)
{
    Z_GenericRecord *rr;
    struct TemplateEntry *te =
	templateFindEntry (req, req->curTemplate, "format sutrs");
    struct TemplateEntry **tep = &te;
    Element path[10];
    int len;
    char *text;

    if (!te)
	return ;

    text = (char *) r->buf;
    len = r->len;
    rr = text2grs1(&text, &len, o, 0, 0);
    req->grs1_vars_rec_abs = rr;
    if (!rr)
	return;

    te = te->next;
    grs1_dump_r (req, tep, rr, 0, path);
}

/* --- Z39.50 Searching --------------------------------------- */

static Target targetList = NULL;

#define TARGET_SELECT_READ 1
#define TARGET_SELECT_WRITE 2

#define TARGET_STATE_UNCONNECTED  0
#define TARGET_STATE_CONNECTING   1
#define TARGET_STATE_CONNECTED    2


static int targetDecodeCookie (ZapRequest *req,
			       const char *name, char **id, const char *proxy)
{
    *id = 0;
    if (!req->cookie_zap || !*req->cookie_zap || !proxy || !*proxy)
	return 0;

    req->cookies_sent++;
    *id = xstrdup (req->cookie_zap);
    return 1;
}

static Target targetAdd (ZapRequest *req, const char *name,
			 const char *rawName,
                         const char *fullName,
			 const char *cookie,
			 const char *proxy)
{
    Target t;

    t = xmalloc (sizeof(*t));

    t->name = xstrdup(name);
    zlog (req, t->name, " make target");

    t->rawName = xstrdup (rawName);
    zlog (req, " real name ", t->rawName);

    t->fullName = xstrdup(fullName);
    zlog (req, " full name ", t->fullName);

    t->proxy = 0;
    if (proxy && *proxy)
    {
	t->proxy = xstrdup(proxy);
	zlog (req, " proxy name ", t->proxy);
    }

    t->order = 0;
    t->reconnectFlag = 0;
    t->cs = NULL;
    t->state = TARGET_STATE_UNCONNECTED;

    t->buf_out = 0;
    t->len_out = 0;

    t->buf_in = 0;
    t->len_in = 0;

    t->odr_in = odr_createmem (ODR_DECODE);
    t->odr_out = odr_createmem (ODR_ENCODE);
    t->odr_print = odr_createmem (ODR_PRINT);
    odr_setprint(t->odr_print, 0);

    t->queryString = 0;
    t->queryType = -1;

    t->cookie = 0;
    if (cookie)
    {
	t->cookie = xstrdup(cookie);
	zlog (req, " cookie id ", t->cookie);
    }

    t->search_nmem = nmem_create ();
    t->presentStart = 1;
    t->presentPos = 1;
    t->presentNumber= 10;
    t->elementSetNames = 0;
    t->schema = 0;
    t->recordList = 0;
    t->sortList = 0;
    t->resultCount = 0;

    t->preferredRecordSyntax = 0;

    t->next = targetList;
    targetList = t;
    return t;
}

static void targetDisconnect (ZapRequest *req, Target t);

static void targetDestroy (ZapRequest *req, Target t)
{
    targetDisconnect (req, t);
    odr_destroy (t->odr_in);
    odr_destroy (t->odr_out);
    odr_destroy (t->odr_print);
    nmem_destroy (t->search_nmem);
    if (t->name)
	zlog (req, t->name, " destroy");
    xfree (t->name);
    xfree (t->rawName);
    xfree (t->fullName);
    xfree (t->proxy);
    xfree (t->cookie);
    xfree (t);

}

int queryMkCCL (ZapRequest *req, WRBUF wrbuf_query, Target t)
{
    int i = 1;
    char fname[32];
    int noTerms = 0;

    wrbuf_puts (wrbuf_query, "");
    while (1)
    {
	const char *entry;

	sprintf (fname, "term%d", i);
	entry = symbolLookupFormStr (req, fname, NULL);
	if (!entry)
	    break;
	if (*entry)
	{
	    if (wrbuf_len(wrbuf_query))
		wrbuf_puts (wrbuf_query, " ");
	    noTerms++;
	    wrbuf_puts (wrbuf_query, entry);
	}
	i++;
    }
    return noTerms;
}

static int queryMkRPN (ZapRequest *req, WRBUF wrbuf_query, Target t)
{
    int i = 1;
    char fname[64];
    const char *op;
    char *cp;
    int noTerms = 0;
    int method = 1; /* 1 = left, 2 = right */
    WRBUF wrbuf_term = wrbuf_alloc();
    WRBUF wrbuf_tmp = wrbuf_alloc();

    op = symbolLookupFormStr (req, "op", "left");
    if (op && !strcmp(op, "right"))
        method = 2;
    else 
        method = 1;

    op = 0;
    /* 
       method1: termI opI ...   termN -> opI   termI termN
       method2: termI ... opN-1 termN -> opN-1 termI termN
     */
    wrbuf_rewind (wrbuf_query);
    wrbuf_puts (wrbuf_query, "");
    while (1)
    {
	Symbol sym;
        int extended_term = 0, ccl_term = 0;

        wrbuf_rewind (wrbuf_term);
	sprintf (fname, "term%d", i);
	sym = symbolLookupForm (req, fname);
	if (!sym)
	{
	    sprintf (fname, "entry%d", i);
	    sym = symbolLookupForm (req, fname);
	}
        if (!sym)
        {
	    sprintf (fname, "rawterm%d", i);
	    sym = symbolLookupForm (req, fname);
            extended_term = 1;
        }
	if (!sym)
	{
	    sprintf (fname, "cclterm%d", i);
	    sym = symbolLookupForm (req, fname);
	    ccl_term = 1;
	}
        if (!sym)
        {
	    sprintf (fname, "op%d", i);
	    if (!symbolLookupFormStr (req, fname, NULL))
                break;
        }
        else
        {
            while (sym)
            {   
                Symbol sym_next = symbolNext(sym, fname);
                if (sym->value && *sym->value)
                {
                    if (sym_next && sym_next->value && *sym_next->value)
                        wrbuf_puts (wrbuf_term, "@or ");
                    if (!extended_term)
                        wrbuf_puts (wrbuf_term, "{");
                    if (ccl_term)
		    {
			char cclfname[64];
			struct ccl_rpn_node *rpn;
			Symbol cclsym;
			int ccl_error, ccl_pos;
			CCL_bibset use_bibset = req->ccl_bibset;
			CCL_bibset local_bibset = 0;
			
			sprintf (cclfname, "cclfields.%.30s.zap", t->name);
			cp = cclfname;
			while ((cp = strchr (cp, '/')))
			    *cp++ = '_';
			local_bibset = initialize_ccl_bibset (req, cclfname);
			if (local_bibset)
			    use_bibset = local_bibset;

			wrbuf_rewind(wrbuf_tmp);
			wrbuf_puts(wrbuf_tmp, "");
			sprintf (cclfname, "cclfield%d", i);
			for (cclsym = symbolLookupForm (req, cclfname); cclsym;
			     cclsym = symbolNext(cclsym, cclfname))
			{
			    const char *fld = cclsym->value;
			    if (*wrbuf_cstr(wrbuf_tmp))
				wrbuf_puts(wrbuf_tmp, ",");
			    wrbuf_puts(wrbuf_tmp, fld);
			} 
			if (*wrbuf_cstr(wrbuf_tmp))
			{
			    wrbuf_puts(wrbuf_tmp, "=(");
			    wrbuf_puts(wrbuf_tmp, sym->value);
			    wrbuf_puts(wrbuf_tmp, ")");
			}
			else
			    wrbuf_puts(wrbuf_tmp, sym->value);

			rpn = ccl_find_str(use_bibset,
					   wrbuf_cstr(wrbuf_tmp),
					   &ccl_error, &ccl_pos);
			ccl_qual_rm(&local_bibset);

			if (ccl_error)
			{
			    zlog(req, "CCL Error for query: ",
				 wrbuf_cstr(wrbuf_tmp));
			    zlog(req, "Error: ", ccl_err_msg(ccl_error));
			    wrbuf_destroy(wrbuf_term);
			    wrbuf_destroy(wrbuf_tmp);
			    return 0;
			}
			ccl_pquery(wrbuf_term, rpn);
			ccl_rpn_delete(rpn);
		    }
		    else
			wrbuf_puts (wrbuf_term, sym->value);
                    if (!extended_term)
                        wrbuf_puts (wrbuf_term, "}");
                    wrbuf_puts (wrbuf_term, " ");
                }
                sym = sym_next;
            }
        }
	if (wrbuf_len(wrbuf_term) == 0)
	{
            if (op && method == 2)
            {
	        sprintf (fname, "op%d", i);
	        op = symbolLookupFormStr (req, fname, "and");
            }
	    i++;
	    continue;
	}
        wrbuf_rewind(wrbuf_tmp);
	if (op)
	{
	    if (*op != '@')
                wrbuf_puts(wrbuf_tmp, "@");
            wrbuf_puts(wrbuf_tmp, op);
            wrbuf_puts(wrbuf_tmp, " ");
	    wrbuf_write(wrbuf_tmp, wrbuf_buf(wrbuf_query),
			wrbuf_len(wrbuf_query));
            wrbuf_puts(wrbuf_tmp, " ");
	}
	sprintf (fname, "field%d", i);
	for (sym = symbolLookupForm (req, fname); sym;
	     sym = symbolNext(sym, fname))
	{
	    const char *attrf = sym->value;
	    char mapstr[200];
	    Symbol sym;

	    sprintf (mapstr, "map(%.60s,%.120s)", attrf, t->name);
	    sym = symbolLookupForm (req, mapstr);
	    if (!sym)
	    {
	        sprintf (mapstr, "map(%.60s)", attrf);
	        sym = symbolLookupForm (req, mapstr);
	    }
	    if (sym)
		attrf = sym->value;
	    if (attrf)
	    {
                wrbuf_puts(wrbuf_tmp, " ");
                wrbuf_puts(wrbuf_tmp, attrf);
                wrbuf_puts(wrbuf_tmp, " ");
	    }
	}

	noTerms++;

        wrbuf_rewind(wrbuf_query);
        wrbuf_write(wrbuf_query, wrbuf_buf(wrbuf_tmp), wrbuf_len(wrbuf_tmp));
        wrbuf_write(wrbuf_query, wrbuf_buf(wrbuf_term), wrbuf_len(wrbuf_term));

	sprintf (fname, "op%d", i);
	op = symbolLookupFormStr (req, fname, "and");
	i++;
    }
    wrbuf_destroy(wrbuf_term);
    wrbuf_destroy(wrbuf_tmp);

    return noTerms;
}

static int queryMkRPNDirect (ZapRequest *req, WRBUF wrbuf_query, Target t)
{
    const char *entry;
    int noTerms = 0;
    
    wrbuf_puts (wrbuf_query, "");
    entry = symbolLookupFormStr (req, "rpnquery", NULL);
    
    if (entry) {
        wrbuf_puts (wrbuf_query, entry);
        noTerms++;
    }
    return noTerms;
}

static int queryMk (ZapRequest *req, WRBUF wrbuf_query, Target t)
{
    const char *entry = symbolLookupFormStr (req, "querytype", "rpn");
    
    zlog(req, "query type: ", entry);
    if (!strcmp (entry, "rpn-direct") && queryMkRPNDirect (req, wrbuf_query, t))
	return Z_Query_type_1;
    if (!strcmp (entry, "rpn") && queryMkRPN (req, wrbuf_query, t))
	return Z_Query_type_1;
    if (!strcmp (entry, "ccl") && queryMkCCL (req, wrbuf_query, t))
	return Z_Query_type_2;
    return -1;
}

static void resultSetPrepare (ZapRequest *req, Target t)
{
    WRBUF wrbuf_query = wrbuf_alloc();

    t->recordList = 0;
    t->queryType = queryMk (req, wrbuf_query, t);
  
    t->queryString = nmem_malloc (t->search_nmem, wrbuf_len(wrbuf_query) + 1);
    memcpy (t->queryString, wrbuf_buf(wrbuf_query), wrbuf_len(wrbuf_query));
    t->queryString[wrbuf_len(wrbuf_query)] = '\0';
    

    t->elementSetNames =
	nmem_strdup (t->search_nmem,
		     symbolLookupFormStrTarget(req, t, "element", ""));
    t->schema = 
	nmem_strdup (t->search_nmem,
		     symbolLookupFormStrTarget(req, t, "schema", ""));
    t->presentStart = atoi (symbolLookupFormStrTarget (req, t, "sortnext", "-1"));
    if (t->presentStart == -1)
	t->presentStart = atoi (symbolLookupFormStrTarget (req, t, "start", "1"));
    t->presentPos = t->presentStart;
    t->presentNumber = atoi (symbolLookupFormStrTarget (req, t,
							"number", "10"));
    t->preferredRecordSyntax =
	nmem_strdup (t->search_nmem,
		     symbolLookupFormStrTarget(req, t, "syntax", ""));
    wrbuf_destroy(wrbuf_query);
}

static int sendSearch (ZapRequest *req, Target t);
static int sendScan (ZapRequest *req, Target t);
static int sendNop (ZapRequest *req, Target t);

static void targetsMk (ZapRequest *req, int persistent)
{
    WRBUF wrbuf;
    Symbol sym;
    Target t;
    int order = 1;
    const char *actionStr;
    const char *proxy_default = symbolLookupFormStr(req, "proxy", 0);
    int (*actionFunc)(ZapRequest *req, Target t) = sendSearch;

#ifndef WIN32
    gettimeofday (&req->start_time, 0);
#endif
    for (t = targetList; t; t = t->next)
    {
	if (persistent)
	    t->parent_order = t->order;
	else
        {
	    t->parent_order = 0;
            nmem_reset (t->search_nmem);
	    t->recordList = 0;
        }
	t->order = 0;
    }

    actionStr = symbolLookupFormStr (req, "action", "search");
    if (!strcmp (actionStr, "search"))
    {
	if (!queryVar (req))
	    return;
	html_dump (req, "query-ok");
	actionFunc = sendSearch;
    }
    else if (!strcmp (actionStr, "scan"))
    {
	if (!queryVar (req))
	    return;
	html_dump (req, "scan-ok");
	actionFunc = sendScan;
    }
#if USE_ES
    else if (!strcmp (actionStr, "es"))
    {
	actionFunc = sendES;
    }
#endif
    else if (!strcmp (actionStr, "init"))
    {
	actionFunc = sendNop;
    }
    for (sym = symbolLookupForm (req, "target*"); sym;
	 sym = symbolNext(sym, "target*"))
    {
	char *cookie_id;
	char s1[256], s2[256], s3[256];
        const char *rawName, *fullName, *proxy;
	int no = 0;

	if (!*sym->value)
	    continue;

	sprintf (s1, "host(%.230s)", sym->value);
	sprintf (s2, "name(%.230s)", sym->value);
	sprintf (s3, "proxy(%.230s)", sym->value);
      
        rawName = symbolLookupFormStr (req, s1, sym->value),
	fullName = symbolLookupFormStr (req, s2, sym->value),
	proxy = symbolLookupFormStr (req, s3, proxy_default);

	targetDecodeCookie(req, sym->value, &cookie_id, proxy);

	/* see if we already have a live target association ... */
        for (t = targetList; t; t = t->next, no++)
	    if (!cookie_id && (persistent || t->order == 0)
                         && !strcmp(t->name, sym->value) &&
		!strcmp(t->rawName, rawName))
	    {
		/* make sure that proxy is the same (or none) */
		if (proxy && *proxy)
		{
		    if (t->proxy && !strcmp (t->proxy, proxy))
			break;
		}
		else
		{
		    if (t->proxy == 0)
			break;
		}
	    }
	if (!t)
	{
	    t = targetAdd (req, sym->value, rawName, fullName, cookie_id,
			   proxy);
	    t->parent_order = 0;
	}
	else
	{
	    zlog (req, t->name, " reuse target");
	    xfree (t->cookie);
	    t->cookie = 0;
	    if (cookie_id)
	    {
		t->cookie = xstrdup (cookie_id);
		zlog (req, " cookie id ", t->cookie);
	    }
	}
	t->action = actionFunc;
	t->order = order++;
	xfree (cookie_id);
    }
    wrbuf = wrbuf_alloc();
    for (t = targetList; t; t = t->next)
    {
        if (t->order)
        {
            char tname[1024];
            wrbuf_puts (wrbuf, "&target=");
            escape_any('%', t->name, tname, sizeof(tname)-1);
            wrbuf_puts (wrbuf, tname);
        }
    }
    symbolAdd(req, req->cur_pa->override, "alltargets", wrbuf_cstr(wrbuf));
    wrbuf_destroy(wrbuf);

    wrbuf = wrbuf_alloc();
    if ((sym = symbolLookupForm(req, "ptarget*"))) {
        for (; sym; sym = symbolNext(sym, "ptarget*")) {
            char tname[1024];
            
            if (!*sym->value)
                continue;
            
            wrbuf_puts (wrbuf, "&ptarget=");
            escape_any('%', sym->value, tname, sizeof(tname)-1);
            wrbuf_puts (wrbuf, tname);
        }
    } else {
        for (t = targetList; t; t = t->next) {
            if (t->order) {
                char tname[1024];
                wrbuf_puts (wrbuf, "&ptarget=");
                escape_any('%', t->name, tname, sizeof(tname)-1);
                wrbuf_puts (wrbuf, tname);
            }
        }
    }
    html_var(req, "palltargets", wrbuf_cstr(wrbuf));
    wrbuf_destroy(wrbuf);
}

static void targetsEncodeCookie (ZapRequest *req)
{
    if (req->cookies_sent)
    {
	Symbol sym = symbolLookup (req->cur_pa->override, "cookie(*");
	for (;sym; sym = symbolNext (sym, "cookie(*"))
	{
	    char cookie_buf[300];
	    const char *cp0, *cp1;
	    int len;
	    cp0 = strchr (sym->name, '(');
	    cp1 = strchr (sym->name, ')');
	    if (!cp0 || !cp1)
		continue;
	    cp0++;
	    len = cp1 - cp0;
	    if (len <= 0 || len > 127)
		continue;

	    sprintf (cookie_buf,
		     "%.*s=\"%.127s\"; Max-Age=\"0\"; Version=\"1\"",
		     len, cp0, sym->value);
#if USE_APACHE
	    ap_table_add(req->request->headers_out, "Set-Cookie",cookie_buf);
#else
	    raw_write (req, "Set-Cookie: ", -1);
	    raw_puts (req, cookie_buf);
#endif
	}
    }
}

static void targetsLeave (ZapRequest *req)
{
    Target t;
    for (t = targetList; t; t = t->next)
    {
	if (t->order)
	    t->order = 0;
	else
	    t->order = t->parent_order;
    }
}

static void targetsDestroy (ZapRequest *req, int cache_flag)
{
#if 1
    Target *tp = &targetList;
    while (*tp)
    {
	Target t = *tp;
	if (!cache_flag || t->cookie)
	{
	    *tp = t->next;
	    zprintf (req, "targetDestroy! order=%d parent_order=%d",
			t->order, t->parent_order);
	    targetDestroy (req, t);
	}
	else
	    tp = &(*tp)->next;
    }
#else
    while (targetList)
    {
	targetDestroy (req, targetList);
	targetList = targetList->next;
#endif
}

static void targetDisconnect (ZapRequest *req, Target t)
{
    if (t->cs)
	cs_close (t->cs);
    xfree (t->buf_in);
    t->buf_in = 0;
    t->len_in = 0;
    t->cs = 0;
    odr_reset (t->odr_in);
    odr_reset (t->odr_out);
    t->state = TARGET_STATE_UNCONNECTED;
    t->mask_select = 0; 
    zprintf (req, "%s disconnect cookie = %s", t->name,
	     t->cookie ? t->cookie : "null");
    if (!t->reconnectFlag)
    {
        req->pending_targets--;
	zlog (req, t->name, " disconnect");
        if (!req->pending_targets)
            html_head(req);
    }
}

static int targetConnect (ZapRequest *req, Target t)
{
    int r;
    void *addr;
    const char *zaddr;

    zlog (req, t->name, " connect");
    t->reconnectFlag = 0;
    t->mask_select = 0;
    t->cs = cs_create (tcpip_type, 0, PROTO_Z3950);
    if (!t-> cs)
    {
        zlog (req, t->name, " cs_create failed");
	return -2;
    }
    zaddr = t->proxy ? t->proxy : t->rawName;
    addr = cs_straddr(t->cs, zaddr);
    if (!addr)
    {
        zprintf (req, "%s cs_straddr failed of %s", t->name, zaddr);
	return -2;
    }
    if ((r=cs_connect (t->cs, addr)) < 0)
    {
	zlog (req, t->name, " connect failed");
	html_var (req, "target", t->name);
        html_var (req, "name", t->fullName);
        html_var (req, "host", t->rawName);
	html_dump (req, "server-error connection");
	targetDisconnect (req, t);
	return -3;
    }
    if (r == 1)
    {
	zlog (req, t->name, " connect pending");
	t->mask_select = TARGET_SELECT_WRITE;
	t->state = TARGET_STATE_CONNECTING;
    }
    else
    {
	zlog (req, t->name, " connect ok");
	t->state = TARGET_STATE_CONNECTED;
	(*t->connectResponse)(req, t);
	t->mask_select = TARGET_SELECT_READ;
    }
    return 0;
}

static void targetHandleWrite (ZapRequest *req, Target t);

int targetSendAPDU (ZapRequest *req, Target t, Z_APDU *a)
{
    if (t->cookie)
    {
	Z_OtherInformation **oi;
	zlog (req, t->name, " encoding cookie");
	yaz_oi_APDU(a, &oi);
	yaz_oi_set_string_oid(oi, t->odr_out, yaz_oid_userinfo_cookie,
			      1, t->cookie);
    }
#if USE_APACHE
#if 0
    if (req->request->connection)
    {
	Z_OtherInformation **oi;
	zlog (req, t->name, " encoding client_ip");
	yaz_oi_APDU(a, &oi);
	yaz_oi_set_string_oidval(oi, t->odr_out, VAL_CLIENT_IP, 1,
				 req->request->connection->remote_ip);
    }
#endif
#endif
    if (!z_APDU(t->odr_out, &a, 0, 0))
    {
	const char *e = odr_getelement(t->odr_out);
	zlog (req, t->name, " APDU encoding failed");
	if (e)
	    zlog(req, "element: ", e);
	html_var (req, "target", t->name);
	html_var (req, "name", t->fullName);
        html_var (req, "host", t->rawName);
	html_dump (req, "server-error protocol");
	targetDisconnect (req, t);
	return -1;
    }
    apdu_log(req, t->odr_print, a);
    t->buf_out = odr_getbuf(t->odr_out, &t->len_out, 0);
    odr_reset(t->odr_out);
    targetHandleWrite (req, t);
    return 0;
}

static void targetHandleConnect (ZapRequest *req, Target t)
{
    zlog (req, t->name, " connect ok");
    t->state = TARGET_STATE_CONNECTED;
    (*t->connectResponse)(req, t);
}

static void targetHandleRead (ZapRequest *req, Target t)
{
    int r;
    Z_APDU *apdu;

    r = cs_get (t->cs, &t->buf_in, &t->len_in);
    if (r <= 0 && errno == EAGAIN)
        return;
    
    if (r == 1)
    {
	zlog (req, t->name, " read partial");
	return;
    }
    if (r <= 0)
    {
	zlog (req, t->name, " connection lost");
	targetDisconnect (req, t);
        zprintf (req, "%s cookie = %s", t->name, t->cookie ? t->cookie : "null");
	if (t->reconnectFlag)
	{
	    zlog (req, t->name, " will reconnect");
	    targetConnect (req, t);
	}
	else
	{
	    html_var (req, "target", t->name);
	    html_var (req, "name", t->fullName);
	    html_var (req, "host", t->rawName);
	    html_dump (req, "server-error connection");
	}
	return;
    }
    odr_reset (t->odr_in);
    odr_setbuf (t->odr_in, t->buf_in, r, 0);
    if (!z_APDU (t->odr_in, &apdu, 0, 0))
    {
	html_var (req, "target", t->name);
	html_var (req, "name", t->fullName);
        html_var (req, "host", t->rawName);
	html_dump (req, "server-error protocol");
	targetDisconnect (req, t);
    }
    else
    {
	apdu_log(req, t->odr_print, apdu);
	(*t->apduResponse)(req, t, apdu);
    }
}

static void targetHandleWrite (ZapRequest *req, Target t)
{
    int r;

    if ((r=cs_put (t->cs, t->buf_out, t->len_out)) < 0)
    {
	zlog (req, t->name, " write failed");
	targetDisconnect (req, t);
	if (t->reconnectFlag)
	{
	    zlog (req, t->name, " will reconnect");
	    targetConnect (req, t);
	}
	else
	{
	    html_var (req, "target", t->name);
	    html_var (req, "name", t->fullName);
	    html_var (req, "host", t->rawName);
	    html_dump (req, "server-error connection");
	}
	return;
    }
    else if (r == 1)
    {
	t->mask_select = TARGET_SELECT_READ|TARGET_SELECT_WRITE;
    }
    else
    {
	t->mask_select = TARGET_SELECT_READ;
    }
}

static void targetIdle (ZapRequest *req, Target t)
{
    t->mask_select = 0;
}

static int targetNextEvent (ZapRequest *req)
{
    Target t;
    fd_set input, output;
    int no = 0;
    int max_fd = 0;
    struct timeval tv;

    zprintf (req, "waiting.. timeout = %d", req->zap_timeout);
    tv.tv_sec = req->zap_timeout;
    tv.tv_usec = 0;

    FD_ZERO (&input);
    FD_ZERO (&output);
    for (t = targetList; t; t = t->next)
    {
	int fd;
	if (!t->order || !t->cs)
	    continue;
	fd = cs_fileno (t->cs);
	if (max_fd < fd)
	    max_fd = fd;
	if (t->mask_select & TARGET_SELECT_READ)
	{
	    FD_SET (fd, &input);
	    no++;
	}
	if (t->mask_select & TARGET_SELECT_WRITE)
	{
	    FD_SET (fd, &output);
	    no++;
	}
    }
    if (!no)
	return 0;
    no = select (max_fd+1, &input, &output, 0, &tv);
    zprintf (req, "select returned no=%d", no);
    for (t = targetList; t; t = t->next)
    {
	int fd;
	if (!t->order || !t->cs)
	{
            zprintf (req, "continue t=%s", t->name);
	    continue;
        }
	fd =cs_fileno(t->cs);
        zprintf (req, "non-continue t=%s fd=%d", t->name, fd);
	if (no <= 0)
	{
	    if (t->mask_select)
	    {
		zlog (req, t->name, " timeout");
		html_var (req, "target", t->name);
		html_var (req, "name", t->fullName);
		html_var (req, "host", t->rawName);
		if (t->state == TARGET_STATE_CONNECTED)
			html_dump (req, "server-error timeout");
		else
			html_dump (req, "server-error connection");
		targetDisconnect (req, t);
	    }
	}
	else if (t->state == TARGET_STATE_CONNECTING)
	{
	    if (FD_ISSET (fd,&input) || FD_ISSET (fd,&output))
		targetHandleConnect (req, t);
	}
	else
	{
	    if (FD_ISSET (fd,&input))
		targetHandleRead (req, t);
	    if (t->cs && FD_ISSET (fd,&output))
		targetHandleWrite (req, t);
	}
    }
    return no;
}

static int sendNop (ZapRequest *req, Target t)
{
    return 1;
}

static void sendInit (ZapRequest *req, Target t)
{
    int i = 0;
    char *version_string;
    Z_APDU *apdu = zget_APDU(t->odr_out, Z_APDU_initRequest);
    Z_InitRequest *ireq = apdu->u.initRequest;
    Z_IdPass *pass = odr_malloc(t->odr_out, sizeof(*pass));
    Z_IdAuthentication *auth = odr_malloc(t->odr_out, sizeof(*auth));
    const char *auth_open =
	symbolLookupFormStrTarget(req, t, "authOpen", 0);
    const char *auth_groupId =
	symbolLookupFormStrTarget(req, t, "authGroupId", 0);
    const char *auth_userId =
	symbolLookupFormStrTarget(req, t, "authUserId", 0);
    const char *auth_password =
	symbolLookupFormStrTarget(req, t, "authPassword", 0);
    const char *charset =
	symbolLookupFormStrTarget(req, t, "charset", 0);
    const char *lang =
	symbolLookupFormStrTarget(req, t, "lang", 0);
   
    ODR_MASK_SET(ireq->options, Z_Options_search);
    ODR_MASK_SET(ireq->options, Z_Options_present);
    ODR_MASK_SET(ireq->options, Z_Options_namedResultSets);
    ODR_MASK_SET(ireq->options, Z_Options_scan);

    ODR_MASK_SET(ireq->protocolVersion, Z_ProtocolVersion_1);
    ODR_MASK_SET(ireq->protocolVersion, Z_ProtocolVersion_2);
    ODR_MASK_SET(ireq->protocolVersion, Z_ProtocolVersion_3);

    ireq->implementationName = "ZAP/YAZ";

    version_string = odr_malloc (t->odr_out, 
        strlen(ireq->implementationVersion) + 5 + strlen(ZAP_VERSION));
    strcpy (version_string, ZAP_VERSION);
    strcat (version_string, "/");
    strcat (version_string, ireq->implementationVersion);
    ireq->implementationVersion = version_string;

    *ireq->maximumRecordSize = 1024*1024;
    *ireq->preferredMessageSize = 1024*1024;

    if (auth_open && *auth_open)
    {
	auth->which = Z_IdAuthentication_open;
	auth->u.open = odr_strdup(t->odr_out, auth_open);
	ireq->idAuthentication = auth;
    }
    pass->groupId = 0;
    if (auth_groupId && *auth_groupId)
    {
        pass->groupId = odr_strdup(t->odr_out, auth_groupId);
	i++;
    }
    pass->userId = 0;
    if (auth_userId && *auth_userId)
    {
	pass->userId = odr_strdup(t->odr_out, auth_userId);
	strcpy(pass->userId, auth_userId);
	i++;
    }
    pass->password = 0;
    if (auth_password && *auth_password)
    {
	pass->password = odr_strdup(t->odr_out, auth_password);
	i++;
    }
    if(i)
    {
	auth->which = Z_IdAuthentication_idPass;
	auth->u.idPass = pass;
	ireq->idAuthentication = auth;
    }
    if (t->proxy)
	yaz_oi_set_string_oid(&ireq->otherInfo, t->odr_out,
			      yaz_oid_userinfo_proxy, 1, t->rawName);

    if (charset || lang)
    {
        Z_OtherInformation **p;
        Z_OtherInformationUnit *p0;
        
        yaz_oi_APDU(apdu, &p);
        
        if ((p0=yaz_oi_update(p, t->odr_out, NULL, 0, 0))) {
            ODR_MASK_SET(ireq->options, Z_Options_negotiationModel);
            
            p0->which = Z_OtherInfo_externallyDefinedInfo;
            p0->information.externallyDefinedInfo =
                yaz_set_proposal_charneg(
                    t->odr_out,

                    (const char**)&charset, charset?1:0,




                    (const char**)&lang, 


                    lang?1:0,


                    1);
        }
    }

    zlog (req, t->name, " init");
    targetSendAPDU (req, t, apdu);
}

static void connectResponse (ZapRequest *req, Target t)
{
    sendInit (req, t);
}

static char **setDatabaseNames (Target t, int *num)
{
    char **databaseNames;
    char *c, *cp = strchr (t->rawName, '/');
    int no = 2;

    if (cp)
    {
        c = cp;
	while ((c = strchr(c, '+')))
	{
	    c++;
	    no++;
        }
    }
    else
	cp = "/Default";
    databaseNames = odr_malloc (t->odr_out, no * sizeof(*databaseNames));
    no = 0;
    while (*cp)
    {
        c = ++cp;
        c = strchr (c, '+');
        if (!c)
            c = cp + strlen(cp);
        else if (c == cp)
            continue;
        /* cp ptr to first char of db name, c is char following db name */
        databaseNames[no] = odr_malloc (t->odr_out, 1+c-cp);
        memcpy (databaseNames[no], cp, c-cp);
        databaseNames[no][c-cp] = '\0';
        no++;
        cp = c;
    }
    databaseNames[no] = NULL;
    *num = no;
    return databaseNames;
}

static void searchHits (ZapRequest *req, Target t, int pass);

static int sendPresent (ZapRequest *zreq, Target t)
{
    Z_APDU *apdu = zget_APDU(t->odr_out, Z_APDU_presentRequest);
    Z_PresentRequest *req = apdu->u.presentRequest;
    int i = 0;

    if (!t->recordList)         /* no records to retrieve at all .. */
	return 0;

    while (1)
    {
	if (i >= t->recordList->num_records) 
	{                       /* got all records ... */
	    searchHits (zreq, t, 0);
	    return 0;
	}
	if (!t->recordList->records[i])
	    break;
	i++;
    }
    /* got record(s) to retrieve */
    
    req->resultSetStartPoint = odr_malloc (t->odr_out, sizeof(int));
    *req->resultSetStartPoint = t->presentStart + i;

    req->numberOfRecordsRequested = odr_malloc (t->odr_out, sizeof(int));
    *req->numberOfRecordsRequested = t->recordList->num_records - i;

    if (*t->preferredRecordSyntax)
    {
        req->preferredRecordSyntax =
	    yaz_string_to_oid_odr(yaz_oid_std(),
				  CLASS_RECSYN, t->preferredRecordSyntax,
				  t->odr_out);
    }
    
    if (*t->schema)
    {
	Z_RecordComposition *compo = odr_malloc (t->odr_out, sizeof(*compo));

        req->recordComposition = compo;
        compo->which = Z_RecordComp_complex;
        compo->u.complex = (Z_CompSpec *)
            odr_malloc(t->odr_out, sizeof(*compo->u.complex));
        compo->u.complex->selectAlternativeSyntax = (bool_t *) 
            odr_malloc(t->odr_out, sizeof(bool_t));
        *compo->u.complex->selectAlternativeSyntax = 0;

        compo->u.complex->generic = (Z_Specification *)
            odr_malloc(t->odr_out, sizeof(*compo->u.complex->generic));

        compo->u.complex->generic->which = Z_Schema_oid;
        compo->u.complex->generic->schema.oid = (Odr_oid *)
	    yaz_string_to_oid_odr(yaz_oid_std(),
				  CLASS_SCHEMA, t->schema, t->odr_out);

        if (! *t->elementSetNames)
            compo->u.complex->generic->elementSpec = 0;
        else
        {
            compo->u.complex->generic->elementSpec = (Z_ElementSpec *)
                odr_malloc(t->odr_out, sizeof(Z_ElementSpec));
            compo->u.complex->generic->elementSpec->which =
                Z_ElementSpec_elementSetName;
            compo->u.complex->generic->elementSpec->u.elementSetName =
                t->elementSetNames;
        }
        compo->u.complex->num_dbSpecific = 0;
        compo->u.complex->dbSpecific = 0;
        compo->u.complex->num_recordSyntax = 0;
        compo->u.complex->recordSyntax = 0;
    }
    else if (*t->elementSetNames)
    {
	Z_ElementSetNames *esn = odr_malloc (t->odr_out, sizeof(*esn));
	Z_RecordComposition *compo = odr_malloc (t->odr_out, sizeof(*compo));
	
	esn->which = Z_ElementSetNames_generic;
	esn->u.generic = t->elementSetNames;
        compo->which = Z_RecordComp_simple;
        compo->u.simple = esn;
	req->recordComposition = compo;
    }
    targetSendAPDU (zreq, t, apdu);
    return 1;
}

static int sendScan (ZapRequest *zreq, Target t)
{
    WRBUF wrbuf_query = wrbuf_alloc();
    Z_APDU *apdu = zget_APDU(t->odr_out, Z_APDU_scanRequest);
    Z_ScanRequest *req = apdu->u.scanRequest;
    
    queryMkRPN (zreq, wrbuf_query, t);
   
    wrbuf_puts(wrbuf_query, "");
    if (!(req->termListAndStartPoint =
	  p_query_scan(t->odr_out, PROTO_Z3950, &req->attributeSet,
                       wrbuf_cstr(wrbuf_query))))
    {
        wrbuf_destroy(wrbuf_query);
	return 0;
    }
    req->databaseNames = setDatabaseNames (t, &req->num_databaseNames);
    req->numberOfTermsRequested = odr_malloc (t->odr_out, sizeof(int));
    *req->numberOfTermsRequested =
	atoi (symbolLookupFormStr (zreq, "scannumber", "10"));
    html_var_num (zreq, "scannumber", *req->numberOfTermsRequested);
    req->preferredPositionInResponse = odr_malloc (t->odr_out, sizeof(int));
    *req->preferredPositionInResponse =
	atoi (symbolLookupFormStr (zreq, "scanposition", "5"));
    html_var_num (zreq, "scanposition", *req->preferredPositionInResponse);

    zlog (zreq, t->name, " scan request");
    zprintf (zreq, "number = %d position = %d", *req->numberOfTermsRequested,
	     *req->preferredPositionInResponse);
    zlog (zreq, " term ", wrbuf_cstr(wrbuf_query));
    targetSendAPDU (zreq, t, apdu);
    wrbuf_destroy(wrbuf_query);
    return 1;
}

static int sendSearch (ZapRequest *req, Target t)
{
    Z_APDU *apdu = zget_APDU(t->odr_out, Z_APDU_searchRequest);
    Z_SearchRequest *sreq = apdu->u.searchRequest;
    Z_Query query;
    Odr_oct *ccl_query = odr_malloc (t->odr_out, sizeof(*ccl_query));
    
    resultSetPrepare (req, t);
    query.which = t->queryType;
    switch (query.which)
    {
    case Z_Query_type_1:
	query.u.type_1 = p_query_rpn(t->odr_out, t->queryString);
	if (!query.u.type_1)
	{
	    zlog(req, "p_query_rpn failed", 0);
	    zlog(req, "RPN: ", t->queryString);
	}
	break;
    case Z_Query_type_2:
	query.u.type_2 = ccl_query;
	ccl_query->buf = (unsigned char*) t->queryString;
	ccl_query->len = strlen(t->queryString);
	break;
    default:
	zlog(req, "Bad query type in sendSearch", 0);
	break;
    }
    if (t->presentStart == 1 && !*t->schema &&
        atoi(symbolLookupFormStrTarget (req, t, "piggyback", "1")))
    {
	sreq->largeSetLowerBound = odr_malloc (t->odr_out, sizeof(int));
	*sreq->largeSetLowerBound = 999999;
	sreq->smallSetUpperBound = &t->presentNumber;
	sreq->mediumSetPresentNumber = &t->presentNumber;
	if (*t->elementSetNames)
	{
	    Z_ElementSetNames *esn = odr_malloc (t->odr_out, sizeof(*esn));
	    
	    esn->which = Z_ElementSetNames_generic;
	    esn->u.generic = t->elementSetNames;
	    sreq->mediumSetElementSetNames = esn;
	    sreq->smallSetElementSetNames = esn;
	}
    }
    else
    {
	sreq->smallSetUpperBound = odr_malloc (t->odr_out, sizeof(int));
	*sreq->smallSetUpperBound = 0;
	sreq->largeSetLowerBound = odr_malloc (t->odr_out, sizeof(int));
	*sreq->largeSetLowerBound = 1;
	sreq->mediumSetPresentNumber = odr_malloc (t->odr_out, sizeof(int));
	*sreq->mediumSetPresentNumber = 0;
    }
    sreq->query = &query;
    if (*t->preferredRecordSyntax)
    {
        sreq->preferredRecordSyntax =
	    yaz_string_to_oid_odr(yaz_oid_std(),
				  CLASS_RECSYN, t->preferredRecordSyntax,
				  t->odr_out);
    }
    sreq->databaseNames = setDatabaseNames (t, &sreq->num_databaseNames);

    zlog (req, t->name, " search request");
    zlog (req, " RPN query ", t->queryString);
    if (*(t->queryString) == '\0')
    {
        zlog (req, t->name, " query empty");
        html_dump (req, "query-empty");
        return 0;
    }
    targetSendAPDU (req, t, apdu);
    return 1;
}

int responseDiag (ZapRequest *req, Target t, Z_DiagRec *p)
{
    Z_DefaultDiagFormat *r;
    char *addinfo = 0;

    if (p->which != Z_DiagRec_defaultFormat)
    {
	return -1;
    }
    r = p->u.defaultFormat;
    html_var_num (req, "errorcode", *r->condition);
    html_var (req, "errorstring", diagbib1_str(*r->condition));


    switch (r->which)
    {
    case Z_DefaultDiagFormat_v2Addinfo:
	addinfo = r->u.v2Addinfo;
	break;
    case Z_DefaultDiagFormat_v3Addinfo:
	addinfo = r->u.v3Addinfo;
	break;
    }
    if (addinfo && *addinfo)
	html_var (req, "addinfo", addinfo);
    else
	html_var (req, "addinfo", "");
    return *r->condition;
}

void responseDB (ZapRequest *req, Target t, Z_DatabaseRecord *p,
		 char **sort_criteria)
{
    char recordstr[OID_STR_MAX+20];
    char formatstr[OID_STR_MAX+20];
    char *str_prefix = "";
    Z_External *r = (Z_External *) p;
    oid_class oclass;
    /* oident *ent = oid_getentbyoid(r->direct_reference); */
    char oid_name[OID_STR_MAX];
    const char *name = yaz_oid_to_string_buf(r->direct_reference, &oclass,
					     oid_name);
    
    if (sort_criteria)
	str_prefix = "sort-";

    req->grs1_vars_rec_abs = 0;
    req->grs1_vars_rec_rel = 0;
    req->marc_vars_buf = 0;

    sprintf (recordstr, "%srecord %s", str_prefix, name);
    sprintf (formatstr, "%sformat %s", str_prefix, name);
    if (r->which == Z_External_sutrs)
    {
	html_var_n (req, "record", (char *) (r->u.sutrs->buf),
	            r->u.sutrs->len);
	html_dump (req, recordstr);
	sutrs_dump(req, t->search_nmem, r->u.sutrs);
    }
    else if (r->which == Z_External_grs1)
    {
	req->grs1_vars_rec_abs = r->u.grs1;
	html_dump (req, recordstr);
	grs1_dump (req, r->u.grs1, formatstr);
    }
    else if (r->which == Z_External_octet)
    {
	char *buf = (char *) (r->u.octet_aligned->buf);
	int len = r->u.octet_aligned->len;
	html_var_n (req, "record", buf, len);

	if (yaz_oid_is_iso2709(r->direct_reference))
	{
	    req->grs1_vars_rec_abs =
		marc_to_grs1 (req, buf, t->search_nmem,
                              r->direct_reference, 0);
	    req->marc_vars_buf = buf;
	}
	else
	{
	    req->grs1_vars_rec_abs = text2grs1 (&buf, &len, t->search_nmem,
                                                0, 0);
	}
        html_dump (req, recordstr);
	grs1_dump (req, req->grs1_vars_rec_abs, formatstr);
    }
    if (sort_criteria)
    {
#if USE_TCL
	const char *s = Tcl_GetVar(req->tcl_interp, "sort", 0);
#else
        const char *s = 0;
#endif
	if (s)
	    *sort_criteria = nmem_strdup(t->search_nmem, s);
	else
	    *sort_criteria = "";
    }
}

static void responseDBOSD (ZapRequest *req, Target t, Z_NamePlusRecordList *p,
                           int resultCount, char **sortList)
{
    int i;
    if (!sortList)
	html_dump (req, "records begin");
    for (i = 0; p && i < p->num_records && i < resultCount; i++)
    {
        if (!p->records[i])
	{
	    p->num_records = i;
	    break;
        }
	if (p->records[i]->databaseName)
	    html_var (req, "database", p->records[i]->databaseName);
	else
	    html_var (req, "database", "");
	html_var_num (req, "no", i + t->presentStart);
	if (p->records[i]->which == Z_NamePlusRecord_surrogateDiagnostic)
	{
	    int code;

	    code = responseDiag (req, t, p->records[i]->u.surrogateDiagnostic);
	    if (!sortList)
		html_dump (req, "record sd");
	}
	else if (p->records[i]->which == Z_NamePlusRecord_databaseRecord)
	    responseDB (req, t, p->records[i]->u.databaseRecord, 
			(sortList ? sortList + i : 0));
    }
    if (!sortList)
	html_dump (req, "records end");
}

static void searchHits (ZapRequest *req, Target t, int pass)
{
    char buf[64];
    int start;
    char server_total = *symbolLookupFormStrTarget (req, t, "servertotal", "0");
    int presentNumber = atoi (symbolLookupFormStr (req, "number", "10"));

    if (!t->cs || !t->recordList)
        return ;
    html_var (req, "target", t->name);
    html_var (req, "name", t->fullName);
    html_var (req, "host", t->rawName);
    html_var (req, "startprevious", 0);
    html_var (req, "startnext", 0);
    if (t->presentStart > 1)
    {
	start = t->presentStart - presentNumber;
	if (start < 1)
	    start = 1;
	sprintf (buf, "%d", start);
	html_var (req, "startprevious", buf);
    }
    if ((start = t->presentStart + presentNumber) <= t->resultCount)
    {
	sprintf (buf, "%d", start);
	html_var (req, "startnext", buf);
    }
    sprintf (buf, "server-hits %d", t->resultCount);
    html_var (req, "hits", buf+12);
    zprintf (req, "%s hits %s", t->name, buf+12);
    if (pass == 0)
        html_dump (req, buf);

    if (pass == 1 && server_total == 's')
    {
	t->sortList =
	    nmem_malloc (t->search_nmem,
			 t->presentNumber * sizeof(*t->sortList));
        responseDBOSD (req, t, t->recordList, t->resultCount, t->sortList);
    }
    else
    {
	t->sortList = 0;
	if ((pass == 0 && server_total == '0') ||
	    (pass == 1 && server_total != '0'))
	    responseDBOSD (req, t, t->recordList, t->resultCount, 0);
    }
}

void handleRecords (ZapRequest *req, Target t, Z_Records *sr,
		    int presentphase)
{
    if (sr && sr->which == Z_Records_NSD)
    {
	int code;
	char buf[64];

	Z_DiagRec dr, *dr_p = &dr;
	dr.which = Z_DiagRec_defaultFormat;
	dr.u.defaultFormat = sr->u.nonSurrogateDiagnostic;

	code = responseDiag (req, t, dr_p);
	
	sprintf (buf, "server-error %d", code);
	html_dump (req, buf);
	t->recordList = 0;
    }
    else if (sr && sr->which == Z_Records_multipleNSD)
    {
        int code;
	char buf[64];
	if (sr->u.multipleNonSurDiagnostics->num_diagRecs >= 1)
	{
            code = responseDiag(req,  t,
				sr->u.multipleNonSurDiagnostics->diagRecs[0]);
	    sprintf (buf, "server-error %d", code);
	    html_dump (req, buf);
        }
        else
        {
	    sprintf (buf, "server-error s");
	    html_dump (req, buf);
        }
	t->recordList = 0;
    }
    else 
    {
	if (t->presentNumber > 0 && !t->recordList)
	{
	    int i;
	    
	    t->recordList =
		nmem_malloc (t->search_nmem, sizeof(*t->recordList));
	    t->recordList->records =
		nmem_malloc (t->search_nmem, t->presentNumber
			     * sizeof(*t->recordList->records));
	    for (i = 0; i<t->presentNumber; i++)
		t->recordList->records[i] = 0;
	    if (t->presentNumber+t->presentStart-1 > t->resultCount)
		t->presentNumber = t->resultCount- t->presentStart + 1;
	    t->recordList->num_records = t->presentNumber;
	}
	if (sr && sr->which == Z_Records_DBOSD)
	{
	    NMEM nmem = odr_extract_mem (t->odr_in);
	    int j, i;
	    Z_NamePlusRecordList *p =
		sr->u.databaseOrSurDiagnostics;
	    for (j = 0; j < t->recordList->num_records; j++)
		if (!t->recordList->records[j])
		    break;
	    for (i = 0; i<p->num_records; i++)
		t->recordList->records[i+j] = p->records[i];
	    /* transfer our response to search_nmem .. we need it later */
	    nmem_transfer (t->search_nmem, nmem);
	    nmem_destroy (nmem);
	    if (presentphase && p->num_records == 0)
	    {
		html_dump (req, "server s");
		t->recordList = 0;
	    }
	}
	else if (presentphase)
	{
	    html_dump (req, "server s");
	    t->recordList = 0;
	}
    }
}

static void searchResponse (ZapRequest *req, Target t, Z_SearchResponse *sr)
{
    t->resultCount = *sr->resultCount;
    handleRecords (req, t, sr->records, 0);
}

static void presentResponse (ZapRequest *req, Target t, Z_PresentResponse *pr)
{
    handleRecords (req, t, pr->records, 1);
}

static void scanResponse (ZapRequest *req, Target t, Z_ScanResponse *res)
{
    int i;
    Z_Entry **entries = NULL;
    int num_entries = 0;
    char str[40];
    
    if (res->entries)
    {
	if ((entries = res->entries->entries))
	    num_entries = res->entries->num_entries;
	if (num_entries > 0)
	{
	    i = 0;
	    if (entries[i]->which == Z_Entry_termInfo)
	    {
		Z_TermInfo *t = entries[i]->u.termInfo;
		if (t->term->which == Z_Term_general)
		{
		    html_var_n (req, "firstterm", 
				(const char *) t->term->u.general->buf,
				t->term->u.general->len);
		}
	    }
	    i = num_entries-1;
	    if (entries[i]->which == Z_Entry_termInfo)
	    {
		Z_TermInfo *t = entries[i]->u.termInfo;
		if (t->term->which == Z_Term_general)
		{
		    html_var_n (req, "lastterm", 
				(const char *) t->term->u.general->buf,
				t->term->u.general->len);
		}
	    }
	}
    }
    zprintf (req, "scan-status = %d", *res->scanStatus);
    html_var_num (req, "scanstatus", *res->scanStatus);
    html_dump (req, "scan-begin");
    sprintf (str, "scan-status %d", *res->scanStatus);
    html_dump (req, str);
    if (res->entries && res->entries->nonsurrogateDiagnostics)
    {
	int i;
	for (i = 0; i<res->entries->num_nonsurrogateDiagnostics; i++)
	{
	    if (!i)
		html_dump (req, "scan-error-begin");
	    responseDiag (req, t, res->entries->nonsurrogateDiagnostics[i]);
	    html_dump (req, "scan-error");
	}
	if (i)
	    html_dump (req, "scan-error-end");
    }
    for (i = 0; i < num_entries; i++)
	if (entries[i]->which == Z_Entry_termInfo)
	{
	    Z_TermInfo *t = entries[i]->u.termInfo;
	    
	    html_var_num (req, "no", i+1);
	    html_var (req, "term", 0);
	    html_var (req, "hits", 0);
	    html_var (req, "termisfirst", (i==0 ? "1" : 0));
	    html_var (req, "termislast", (i==num_entries-1 ? "1" : 0));
	    html_var (req, "termpos",
		      (i+1 == *res->positionOfTerm ? "1": 0));
	    if (t->term->which == Z_Term_general)
	    {
		if (t->globalOccurrences)
		    html_var_num (req, "hits", *t->globalOccurrences);
		html_var_n (req, "term", 
			    (const char *) t->term->u.general->buf,
			    t->term->u.general->len);
		if (t->displayTerm)
		    html_var (req, "displayterm", t->displayTerm);
		else
		    html_var_n (req, "displayterm", 
				(const char *) t->term->u.general->buf,
				t->term->u.general->len);
	    }
	    html_dump (req, "scan-term-normal");
	}
	else
	{
	    responseDiag (req, t, entries[i]->u.surrogateDiagnostic);
	    html_dump (req, "scan-term-error");
	}
    html_dump (req, "scan-end");
}

char *get_cookie(Z_OtherInformation **otherInfo)
{
    return yaz_oi_get_string_oid(otherInfo, yaz_oid_userinfo_cookie, 1, 0);
}

static void logTime (ZapRequest *req, Target t, const char *event)
{
#ifdef WIN32
    zprintf (req, "%s %s response", t->name, event);
#else
    struct timeval tv;
    long sdiff, udiff;
    gettimeofday (&tv, 0);
    sdiff = tv.tv_sec - req->start_time.tv_sec;
    udiff = tv.tv_usec - req->start_time.tv_usec;
    if (udiff < 0)
    {
	udiff += 1000000;
	sdiff --;
    }
    zprintf (req, "%s %s response %ld.%06ld", t->name, event, sdiff, udiff);
#endif
}

void apduResponse (ZapRequest *req, Target t, Z_APDU *apdu)
{
    ODR odr_in = t->odr_in;
    Z_InitResponse *initrs;
    html_var (req, "target", t->name);
    html_var (req, "rpnquery", t->queryString);
    html_var (req, "name", t->fullName);
    html_var (req, "host", t->rawName);
    if (t->preferredRecordSyntax)
        html_var (req, "syntax", t->preferredRecordSyntax);
    else
        html_var (req, "syntax", "");

    targetIdle (req, t);

    switch(apdu->which)
    {
    case Z_APDU_initResponse:
	logTime(req, t, "init");
	initrs = apdu->u.initResponse;
	if (!*initrs->result)
	{
	    zlog (req, t->name, " init rejected");
	    html_dump (req, "server-error init");
            req->pending_targets--;
            if (!req->pending_targets)
                html_head(req);
	}
	else
	{
	    char *cookie = get_cookie (&apdu->u.initResponse->otherInfo);
	    zprintf(req, "%s cookie=%s", t->name, cookie ? cookie : "null");
	    xfree (t->cookie);
	    t->cookie = 0;
	    if (cookie)
	    {
		xfree(t->cookie);
		t->cookie = xstrdup(cookie);
	    }
            req->pending_targets--;
            if (!req->pending_targets)
                html_head(req);
	    (*t->action) (req, t);
	}
	break;
    case Z_APDU_searchResponse:
	logTime(req, t, "search");
	searchResponse (req, t, apdu->u.searchResponse);
	sendPresent (req, t);
	break;
    case Z_APDU_presentResponse:
	logTime(req, t, "present");
	presentResponse (req, t, apdu->u.presentResponse);
	sendPresent (req, t);
	break;
    case Z_APDU_scanResponse:
	logTime(req, t, "scan");
	/* create new decoding stream and keep the current in odr_in */
	t->odr_in = odr_createmem(ODR_DECODE);
	scanResponse (req, t, apdu->u.scanResponse);
	/* destroy the current one. t->odr_in is new! */
	odr_destroy (odr_in);
	break;
#if USE_ES
    case Z_APDU_extendedServicesResponse:
	logTime(req, t, "es");
	esResponse (req, t, apdu->u.extendedServicesResponse);
	break;
#endif
    case Z_APDU_close:
	logTime(req, t, "close");
	if (t->reconnectFlag)
	{
	    zlog (req, t->name, " will reconnect");
	    targetConnect (req, t);
	}
        else
        {
	    html_dump (req, "server-error connection");
	    targetDisconnect (req, t);
        }
	break;
    default:
        html_dump (req, "server-error protocol");
        targetDisconnect (req, t);
	break;
    }
}

struct record_sort_entry {
    Target target;
    Z_NamePlusRecord *record;
    char *criteria;
    int offset;
};

static int record_comp(const void *avp, const void *bvp)
{
    struct record_sort_entry *a = (struct record_sort_entry*) avp;
    struct record_sort_entry *b = (struct record_sort_entry*) bvp;
    return strcmp(a->criteria, b->criteria);
}

static void record_sort(ZapRequest *req)
{
    int show_offset = atoi (symbolLookupFormStr (req, "show_offset", "0"));
    int show_number = atoi (symbolLookupFormStr (req, "show_number", "-1"));
    struct record_sort_entry *records;
    int num = 0, i;
    Target t;
    
    if (show_number == -1)
	show_number = atoi(symbolLookupFormStr(req, "number", "10"));

    for (t = targetList; t; t = t->next)
    {
	if (t->order && t->cs && t->recordList && t->sortList)
	    num += t->recordList->num_records;
    }
    if (!num)
	return;
    records = xmalloc(sizeof(*records) * num);

    for (i = 1; i<20; i++)
    {
	char str_name[100];
	char str_val[100];

	if ((i-1)*show_number > num)
	    break;
	sprintf (str_name, "sortoffset%d", i);
	sprintf (str_val, "show_offset=%d&show_number=%d&number=%d",
		 (i-1)*show_number,
		 show_number,
		 i*show_number);
	symbolAdd(req, req->cur_pa->override, str_name, str_val);
    }

    i = 0;
    for (t = targetList; t; t = t->next)
    {
	if (t->order && t->cs && t->recordList && t->sortList)
	{
	    int j;
	    for (j = 0; j<t->recordList->num_records; j++)
	    {
		if (t->recordList->records[j]->which == Z_NamePlusRecord_databaseRecord)
		{
		    records[i].record = t->recordList->records[j];
		    records[i].target = t;
		    records[i].criteria = t->sortList[j];
		    records[i].offset = j;
		    i++;
		}
	    }
	}
    }
    num = i;

    qsort(records, num, sizeof(*records), record_comp);

    html_dump (req, "records begin");
    for (i = show_offset; i < show_offset+show_number && i < num; i++)
    {
	t = records[i].target;

	html_var_num (req, "no", i+1);

	html_var (req, "target", t->name);
	html_var (req, "name", t->fullName);
	html_var (req, "host", t->rawName);
	
	if (records[i].record->databaseName)
	    html_var (req, "database", records[i].record->databaseName);
	else
	    html_var (req, "database", "");
	if (records[i].record->which == Z_NamePlusRecord_databaseRecord)
	    responseDB (req, records[i].target,
			records[i].record->u.databaseRecord, 0);
    }
    html_dump (req, "records end");

    for (t = targetList; t; t = t->next)
    {
	if (t->order && t->cs && t->sortList)
	    t->sortList = 0;
    }
    xfree (records);
}

static void session (ZapRequest *req)
{
    Target t;
    int i;
    
    for (t = targetList; t; t = t->next)
    {
	if (!t->order)
	    continue;
	t->connectResponse = connectResponse;
	t->apduResponse = apduResponse;
	t->mask_select = 0;
	if (t->state == TARGET_STATE_UNCONNECTED)
        {
            req->pending_targets++;
	    targetConnect (req, t);
        }
	else if (t->state == TARGET_STATE_CONNECTED)
	{
	    t->reconnectFlag = 1;
	    (*t->action)(req, t);
	}
    }
    if (!req->pending_targets)
        html_head(req);
    for (i = 0; i<300 && targetNextEvent(req) > 0; i++)
	;
    if (i == 300)
        zprintf (req, "LOOP : 300 iterations exceeded");
    if (!req->pending_targets)
        html_head(req);

    for (t = targetList; t; t = t->next)
    {
	if (t->order)
	    searchHits (req, t, 1);
    }
    record_sort(req);
}
