/*
 * ZAP! Z39.50 Apache Module
 * 
 * Copyright (C) 1999-2003, Index Data
 * See the file ZAP_LICENSE for details.
 *
 * $Id: zap_es.c,v 1.11 2007-05-07 11:15:33 adam Exp $
 */

#include "zap.h"

#include <yaz/ill.h>
#include <yaz/oid_db.h>

static const char *get_ill_element (void *clientData, const char *element)
{
    ZapRequest *zreq = (ZapRequest *) clientData;

    return symbolLookupFormStr (zreq, element, 0);
}

static Z_External *create_external_illRequest(ZapRequest *zreq, Target t)
{
    ODR out = t->odr_out;
    ILL_Request *req;
    Z_External *r = 0;
    struct ill_get_ctl ctl;

    ctl.odr = t->odr_out;
    ctl.clientData = zreq;
    ctl.f = get_ill_element;

    req = ill_get_ILLRequest(&ctl, "ill", 0);

    if (!ill_Request (out, &req, 0, 0))
    {
	int ill_request_size;
	char *ill_request_buf = odr_getbuf (out, &ill_request_size, 0);
	if (ill_request_buf)
	    odr_setbuf (out, ill_request_buf, ill_request_size, 1);
	zlog (zreq, t->name, " couldn't encode ILL-Request packet");
	return 0;
    }
    else
    {
	int illRequest_size = 0;
	char *illRequest_buf = odr_getbuf (out, &illRequest_size, 0);
	
	r = (Z_External *) odr_malloc (out, sizeof(*r));
	r->direct_reference = odr_oiddup(out, yaz_oid_general_isoill_1);
	r->indirect_reference = 0;
	r->descriptor = 0;
	r->which = Z_External_single;
	
	r->u.single_ASN1_type = (Odr_oct *)
	    odr_malloc (out, sizeof(*r->u.single_ASN1_type));
	r->u.single_ASN1_type->buf = odr_malloc (out, illRequest_size);
	r->u.single_ASN1_type->len = illRequest_size;
	r->u.single_ASN1_type->size = illRequest_size;
	memcpy (r->u.single_ASN1_type->buf, illRequest_buf, illRequest_size);
	printf ("len = %d\n", illRequest_size);
    }
    return r;
}


static Z_ItemOrder *sendItemOrder (ZapRequest *zreq, Target t)
{
    Z_ItemOrder *req = odr_malloc (t->odr_out, sizeof(*req));
    const char *str;

    req->which=Z_IOItemOrder_esRequest;
    req->u.esRequest = (Z_IORequest *) 
	odr_malloc(t->odr_out,sizeof(Z_IORequest));

    zlog (zreq, t->name, " sendItemOrder");
    /* to keep part ... */
    req->u.esRequest->toKeep = (Z_IOOriginPartToKeep *)
	odr_malloc(t->odr_out,sizeof(Z_IOOriginPartToKeep));
    req->u.esRequest->toKeep->supplDescription = 0;
    req->u.esRequest->toKeep->contact =
	odr_malloc (t->odr_out, sizeof(*req->u.esRequest->toKeep->contact));

    str = symbolLookupFormStr (zreq, "es-itemorder-contact-name", 0);
    req->u.esRequest->toKeep->contact->name = str ?
	nmem_strdup (t->odr_out->mem, str) : 0;

    str = symbolLookupFormStr (zreq, "es-itemorder-contact-phone", 0);
    req->u.esRequest->toKeep->contact->phone = str ?
	nmem_strdup (t->odr_out->mem, str) : 0;

    str = symbolLookupFormStr (zreq, "es-itemorder-contact-email", 0);
    req->u.esRequest->toKeep->contact->email = str ?
	nmem_strdup (t->odr_out->mem, str) : 0;

    req->u.esRequest->toKeep->addlBilling = 0;

    /* not to keep part ... */
    req->u.esRequest->notToKeep = (Z_IOOriginPartNotToKeep *)
	odr_malloc(t->odr_out,sizeof(Z_IOOriginPartNotToKeep));

    req->u.esRequest->notToKeep->resultSetItem = (Z_IOResultSetItem *)
	odr_malloc(t->odr_out, sizeof(Z_IOResultSetItem));
    req->u.esRequest->notToKeep->resultSetItem->resultSetId = "default";
    req->u.esRequest->notToKeep->resultSetItem->item =
	(int *) odr_malloc(t->odr_out, sizeof(int));
    *req->u.esRequest->notToKeep->resultSetItem->item =
	atoi (symbolLookupFormStr (zreq, "es-itemorder-item", "1"));

    req->u.esRequest->notToKeep->itemRequest = 
	create_external_illRequest(zreq, t);
    
    return req;
}

int sendES (ZapRequest *zreq, Target t)
{
    Z_APDU *apdu = zget_APDU(t->odr_out, Z_APDU_extendedServicesRequest);
    Z_ExtendedServicesRequest *req = apdu->u.extendedServicesRequest;
    const char *str;

    zlog (zreq, t->name, " sendES");
    str = symbolLookupFormStr (zreq, "es-function", "create");
    if (!strcmp (str, "create"))
	*req->function = Z_ExtendedServicesRequest_create;
    else if (!strcmp (str, "modify"))
	*req->function = Z_ExtendedServicesRequest_modify;
    else if (!strcmp (str, "delete"))
	*req->function = Z_ExtendedServicesRequest_delete;
    else
    {
	zlog (zreq, "unrecognised value for es-function: ", str);
	return 0;
    }
    str = symbolLookupFormStr (zreq, "es-package-type", "nop");
    if (!strcmp (str, "itemorder"))
    {
	Z_External *r = odr_malloc (t->odr_out, sizeof(*r));

	req->taskSpecificParameters = r;
	r->direct_reference =
	    odr_oiddup(t->odr_out, yaz_oid_extserv_item_order);
	r->indirect_reference = 0;
	r->descriptor = 0;
	r->which = Z_External_itemOrder;
	r->u.itemOrder = sendItemOrder (zreq, t);
    }
    else
    {
	zlog (zreq, "unrecognised value for es-package-type: ", str);
	return 0;
    }
    req->packageType = odr_oiddup(t->odr_out, yaz_oid_extserv_item_order);

    str = symbolLookupFormStr (zreq, "es-package-name", 0);
    if (str)
        req->packageName = nmem_strdup (t->odr_out->mem, str);


    str = symbolLookupFormStr (zreq, "es-user-id", 0);
    if (str)
	req->userId = nmem_strdup (t->odr_out->mem, str);
    
    zlog (zreq, t->name, " ES request");
    targetSendAPDU (zreq, t, apdu);
    return 0;
}

void esResponse (ZapRequest *req, Target t,
		 Z_ExtendedServicesResponse *res)
{
    Odr_oct *id = res->referenceId;
 
    if (id)
	html_var_n (req, "refid", (const char *) id->buf, id->len);
    else
	html_var (req, "refid", "");

    html_var (req, "targetreference", "");
    if (res->taskPackage && 
	res->taskPackage->which == Z_External_extendedService)
    {
	Z_TaskPackage *taskPackage = res->taskPackage->u.extendedService;
	Odr_oct *id = taskPackage->targetReference;
	
	if (id)
	    html_var_n(req, "targetreference", (const char *)id->buf, id->len);
    }
    switch (*res->operationStatus)
    {
    case Z_ExtendedServicesResponse_done:
	html_dump (req, "es-response done");
	break;
    case Z_ExtendedServicesResponse_accepted:
	html_dump (req, "es-response accepted");
	break;
    case Z_ExtendedServicesResponse_failure:
	html_dump (req, "es-response failure");
	break;
    default:
	html_dump (req, "es-response unknown");
	break;
    }
    if (res->diagnostics)
    {
	int i;
	html_dump (req, "es-diagnostics-begin");
	for (i = 0; i < res->num_diagnostics; i++)
	{
	    responseDiag (req, t, res->diagnostics[i]);
	    html_dump (req, "es-diagnostic");
	}
	html_dump (req, "es-diagnostics-end");
    }
}
