/* 
 * Copyright (C) 1995-2013 Index Data
 * 
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <dirent.h>

#include <iostream>
#include <fstream>

#include <yaz/xmalloc.h>
#include <yaz/tpath.h>
#include <yaz/test.h>
#include <yaz/log.h>
#include <yaz/srw.h>

#include <metaproxy/filter.hpp>
#include <metaproxy/package.hpp>
#include <metaproxy/router_chain.hpp>
#include <metaproxy/router_xml.hpp>
#include <metaproxy/util.hpp>

// #include "filter_http_rewrite.hpp"

#include "filter_cproxy.hpp"

namespace mp = metaproxy_1;
namespace yf = mp::filter;

// Global variables
mp::odr odr;

// Do not delete dump dirs while running
// Comment out to clean up things properly
#define KEEP_DUMP_DIRS 1  

//////////////////////////
// Helpers to check things

// Check that the HTTP response code matches
static bool check_response_code( mp::Package *pack, int code )
{
    Z_GDU *gdu_res = pack->response().get();
    if ( ! gdu_res ) {
        return false;
    } 
    Z_HTTP_Response *hres = gdu_res->u.HTTP_Response;
    if ( code != hres->code ) 
    {
        yaz_log(YLOG_LOG,"HTTP response check failed, got %d, expected %d",
                hres->code, code);
        return false;
    }
    return true;        
} // check_response_code

// Check that the HTTP response contains a given string
static bool check_response_content( mp::Package *pack, const char *fragment )
{
    Z_GDU *gdu_res = pack->response().get();
    if ( ! gdu_res ) 
    {
        return false;
    } 
    Z_HTTP_Response *hres = gdu_res->u.HTTP_Response;
    if ( hres->content_len <= 0 )
    {
        yaz_log(YLOG_LOG,"HTTP response without content, can not find '%s' in it",
                fragment);
        return false;
        // This should not happen, but I have seen the following fail with
        // bad content, only a few random times. This is a desperate attempt
        // to catch the situation
    }
    if ( strstr( hres->content_buf, fragment ) == 0 )
    {
        yaz_log(YLOG_LOG,"HTTP response check failed. Did not find '%s' in \n'%s'"
            "  (content_len=%d content_buf=%p)",
                fragment, hres->content_buf, hres->content_len, hres->content_buf );
        return false;
    }
    return true;
} // check_response_content


// Check that a header exists in headers, and that it contains the fragment
// Can be used for request and response headers (see below)
static bool check_header (Z_HTTP_Header *headers, 
                const char *header, const char *fragment,
                const char *section)
{
    if ( !headers) {
        yaz_log(YLOG_LOG,"No headers at all in %s", section);
        return false;
    }
    const char *hdrval = z_HTTP_header_lookup(headers, header);
    if ( !hdrval )
    {
        yaz_log(YLOG_LOG,"No %s header '%s' at all", section, header);
        return false;
    }

    if ( strstr( hdrval, fragment ) == 0 )
    {
        yaz_log(YLOG_LOG,"HTTP %s header check failed. Did not find '%s' in '%s'",
                section, fragment, hdrval );
        return false;
    }
    return true;
}

// Check a that response header exists, and contains the fragment
static bool check_response_header( mp::Package *pack,
                const char *header, const char *fragment )
{
    Z_GDU *gdu_req = pack->response().get();
    if ( ! gdu_req )
    {
        return false;
    }
    Z_HTTP_Response *hresp = gdu_req->u.HTTP_Response;
    if ( ! check_header( hresp->headers, header, fragment, "response") )
    {
        return false;
    }
    return true;
} // check_request_header

// Check a that request header exists, and contains the fragment
static bool check_request_header( mp::Package *pack,
                const char *header, const char *fragment )
{
    Z_GDU *gdu_req = pack->request().get();
    if ( ! gdu_req )
    {
        return false;
    }
    Z_HTTP_Request *hreq = gdu_req->u.HTTP_Request;
    if ( ! check_header( hreq->headers, header, fragment, "request") )
    {
        return false;
    }
    return true;
} // check_request_header

// Check that we do *not* have a request header (like an empty cookie line)
static bool check_no_request_header( mp::Package *pack, const char *header)
{
    Z_GDU *gdu_req = pack->request().get();
    if ( ! gdu_req )
    {
        return false;
    }
    Z_HTTP_Request *hreq = gdu_req->u.HTTP_Request;
    const char *hdrval = z_HTTP_header_lookup(hreq->headers, header);
    if ( hdrval )
    { // oops, got a header balie
        yaz_log(YLOG_LOG,"Unwanted request header '%s' '%s' ", header, hdrval);
        return false;
    }
    return true;
} // check_no_request_header

// Check that we do *not* have a response header (like an empty cookie line)
static bool check_no_response_header( mp::Package *pack, const char *header)
{
    Z_GDU *gdu_resp = pack->response().get();
    if ( ! gdu_resp )
    {
        return false;
    }
    Z_HTTP_Response *hresp = gdu_resp->u.HTTP_Response;
    const char *hdrval = z_HTTP_header_lookup(hresp->headers, header);
    if ( hdrval )
    { // oops, got a header balie
        yaz_log(YLOG_LOG,"Unwanted response header '%s' '%s' ", header, hdrval);
        return false;
    }
    return true;
} // check_no_response_header


// Check that at least one of the headers with the given name match the string
// We can have multiple 'Link' headers, for example.
static bool check_any_header (Z_HTTP_Header *hp,
                const char *header, const char *fragment,
                const char *section)
{
    if ( !hp) {
        yaz_log(YLOG_LOG,"No headers at all in %s", section);
        return false;
    }
    for (; hp; hp = hp->next)
    {
        //yaz_log(YLOG_LOG,"any: looking at header '%s' '%s' (%s)",
        //        hp->name, hp->value, header);
        if (yaz_strcasecmp(hp->name, header) == 0)
        {
            // yaz_log(YLOG_LOG,"any:   Right kind of header '%s' '%s'",
            //        hp->name, hp->value);
            if ( strstr( hp->value, fragment ) != 0 )
            {
                //yaz_log(YLOG_LOG,"any:     Found the fragment '%s' in '%s'",
                //       fragment, hp->value);
                return true;
            }

        }
    }
    yaz_log(YLOG_LOG,"HTTP %s header check failed. Did not find '%s' "
            "in any %s header",
            section, fragment, header );
    return false;
}

static bool check_any_request_header(mp::Package *pack,
                               const char *header, const char *fragment)
{
    Z_GDU *gdu_req = pack->request().get();
    if ( ! gdu_req )
    {
        return false;
    }
    Z_HTTP_Request *hreq = gdu_req->u.HTTP_Request;
    return check_any_header( hreq->headers, header, fragment, "response");
}

static bool check_any_response_header(mp::Package *pack,
                               const char *header, const char *fragment)
{
    Z_GDU *gdu_req = pack->response().get();
    if ( ! gdu_req )
    {
        return false;
    }
    Z_HTTP_Response *hres = gdu_req->u.HTTP_Response;
    return check_any_header( hres->headers, header, fragment, "response");
}

/////////////////////////
// Helpers to make things

// Make an instance of the filter_cproxy, and configure it
mp::filter::CProxy *make_filter( const char *xmlconf) 
{
    mp::filter::CProxy *cpx = new mp::filter::CProxy;
    assert(cpx);
    xmlDocPtr doc = xmlParseMemory(xmlconf, strlen(xmlconf));
    assert(doc);
    xmlNode *root_element = xmlDocGetRootElement(doc);
    assert(root_element);
    cpx->configure(root_element, true, "");
    xmlFreeDoc(doc);
    return cpx;
} // make_filter

// Create a package with a request to the given URL
mp::Package *make_package(const char *url, int use_full_host, 
                          std::string method = "GET",
                          std::string content = "")
{
    mp::Package *pack = new mp::Package;
    Z_GDU *gdu_req = z_get_HTTP_Request_uri(odr, url, 0, use_full_host);
    Z_HTTP_Request *hreq = gdu_req->u.HTTP_Request;
    hreq->method = odr_strdup(odr,method.c_str());
    if ( !content.empty() )
    {
        hreq->content_buf = odr_strdup(odr, content.c_str() );
        hreq->content_len = strlen(hreq->content_buf);        
        
    }
    pack->request() = gdu_req;
    return pack;
}


// Create a package with the URL, and let the filter process it
mp::Package *process_url(mp::filter::CProxy *cpx,
                        const char *url)
{
    mp::Package *pack = make_package(url,0);  // default to no full host
    cpx->process(*pack);
    return pack;
}

// Add a header line to the request
void add_request_header(mp::Package *pack,
                        std::string header,
                        std::string value )
{
    Z_GDU *gdu_req = pack->request().get();
    assert(gdu_req);
    Z_HTTP_Request *hreq = gdu_req->u.HTTP_Request;
    z_HTTP_header_set(odr, &hreq->headers,
                          header.c_str(), value.c_str() );
}

// url-encode a parameter, typically a name=value string. Will also encode
// the separating '&', so don't include that here
std::string url_enc( std::string comp )
{
    int complen = comp.length();
    char ubuf[3*complen+128]; // each char can turn into %XX, plus some extra
    yaz_encode_uri_component(ubuf, comp.c_str());
    return std::string(ubuf);
}

/////////////////////////
// Helpers to make files
// We keep a (global) list of files we have created, so we can delete them
std::vector<std::string> tempfilelist;

void make_cf_config( std::string name,
                     std::string proxyhostname,
                     std::string sesdir,
                     std::string cfengine )
{
    std::ofstream f(name.c_str());
    f << "# cproxy config file\n\n";
    f << "# Created by the unit test for fileter_cproxy\n";
    f << "# Should have been deleted at the end of the tests\n\n";
    f << "proxyhostname: " << proxyhostname << "\n";
    f << "sessiondir: " << sesdir << "\n";
    f << "cfengine: " << cfengine << "\n";
    f.close();
    tempfilelist.push_back(name);
} // make_cf_config

void make_session_file( std::string sesdir,
                        std::string sesno,
                        std::string sessiondata)
{
    std::string name = sesdir + "/cf." + sesno;  // "./cf.17"
    std::ofstream f(name.c_str());
    f << "CfSession " << sesno << "\n";
    f << "# File created by filter_cproxy unit test \n";
    f << "# Should have been deleted at the end of the tests\n";
    f << sessiondata;
    f.close();
    tempfilelist.push_back(name);
} // make_session_file

void make_param_file( std::string sesdir,
                      std::string sesno,
                      std::string connector,
                      std::string auth,
                      std::string proxy,
                      std::string realm )
{
    std::string name = sesdir + "/cf." + sesno + ".p" ;  // "./cf.17.p"
    std::ofstream f(name.c_str());
    f << "#content_proxy\n";
    f << "# File created by filter_cproxy unit test\n";
    f << "# Should have been deleted at the end of tests\n";
    f << "connector: " << connector << "\n";
    f << "auth: " << auth << "\n";
    f << "proxy: " << proxy << "\n";
    f << "realm: " << realm << "\n";
    f.close();
    tempfilelist.push_back(name);    
} // make_param_file 

// ;ake a html file for test2 to fetch
// Does not actually have to be html
// filename should include the whole path 
// (most likely ./ )
void make_html_file( std::string filename,
                     std::string content )
{
    std::ofstream f(filename.c_str());
    f << content;
    f.close();
    tempfilelist.push_back(filename);    
} // make_html_file

// Remove files created earlier
void remove_temp_files()
{
    std::vector<std::string>::iterator it;
    for ( it = tempfilelist.begin(); it != tempfilelist.end(); ++it)
    {
        remove ( it->c_str() );
    }    
} // remove_temp_files

// Remove a dump dir created by a test
void remove_dump_dir( std::string dumpdir, bool force=false )
{
#ifdef KEEP_DUMP_DIRS
    if ( !force)
        return;
#endif
    DIR *dir;
    struct dirent *entry;
    if ((dir = opendir(dumpdir.c_str() )) == NULL) {
        return; // ignore the errors here
    }
    while ((entry = readdir(dir)) != NULL)
    {
        //yaz_log(YLOG_LOG, "  %s", entry->d_name);
        if ( strcmp(entry->d_name, ".") == 0 ||
             strcmp(entry->d_name, "..") == 0 )
        {
            continue; // can't remove those
        }
        std::string path = dumpdir + "/" + std::string(entry->d_name) ;
        //yaz_log(YLOG_LOG,"  Removing '%s' ", path.c_str() );
        remove ( path.c_str() );
    }
    closedir( dir );
    rmdir( dumpdir.c_str() );
} // remove_dump_dir

/////////////////////////
// test 1
// Things that can be tested directly against a filter_cproxy,
// without being in a filter chain.

void test1(void)
{
    const char * conf =
        "<?xml version='1.0'?>\n"
        "<filter type='cproxy'>\n"
        "  <debug>1</debug>\n"    // gets enabled via the URL when needed
        "  <cfconfig>./cf-config.unittest.cfg</cfconfig>\n"
        "  <sessionmaxage>1</sessionmaxage>\n"
        "</filter>\n";
    make_cf_config("./cf-config.unittest.cfg", // config file to write
                   "localhost:9036/XXX/node102",  // url prefix
                   ".", // session directory
                   "" );  // cf-zserver for content connectors, "localhost:9001"
    mp::filter::CProxy *cpx = make_filter( conf );
    mp::Package *pack;

    // No proxy host or prefix in the URL. Debug not enabled
    pack = process_url(cpx, "http://bad.request/no.session/at.all");
    YAZ_CHECK( check_response_code(pack, 400) ); 
    YAZ_CHECK( check_response_content(pack,
                                "No cproxy session found in the URL" ) );
    delete(pack);

    // No proxy host or prefix in the URL. Enabling debug. 
    pack = process_url(cpx,
                   "http://bad.request/cproxydebug-3/no.session/at.all");
    YAZ_CHECK( check_response_code(pack, 400) );
    YAZ_CHECK( check_response_content(pack, "Using debug level 3" ) );
    YAZ_CHECK( check_response_content(pack,
                                  "Url after removing: /no.session/at.all" ) );
    delete(pack);

    // Debug with a named flag
    pack = process_url(cpx,
                  "http://bad.request/cproxydebug-verbose/no.session/at.all");
    YAZ_CHECK( check_response_code(pack, 400) );
    YAZ_CHECK( check_response_content(pack, "Using debug level 3" ) );
    YAZ_CHECK( check_response_content(pack,
                                 "Url after removing: /no.session/at.all" ) );
    delete(pack);

    // Debug with multiple named flags
    pack = process_url(cpx,  "http://bad.request/"
       "cproxydebug-keepcontent-nomove-dump-verbose/no.session/at.all");
    YAZ_CHECK( check_response_code(pack, 400) );
    YAZ_CHECK( check_response_content(pack, "Using debug level 31" ) );
    YAZ_CHECK( check_response_content(pack, "Url after removing: /no.session/at.all" ) );
    delete(pack);

    // No session number in the URL
    pack = process_url(cpx, "http://localhost:9036/XXX/cproxydebug-3/node102/nosession.com/error.txt");
    YAZ_CHECK( check_response_code(pack, 400) );
    YAZ_CHECK( check_response_content(pack, "No session found in the URL" ) );
    YAZ_CHECK( check_response_content(pack, "No referer header at all" ) );
    delete(pack);

    // Non-existing session
    pack = process_url(cpx, "http://localhost:9036/XXX/cproxydebug-3/node102/"
                            "99/badsession.com/error.txt");
    YAZ_CHECK( check_response_code(pack, 400) );
    YAZ_CHECK( check_response_content(pack, "Got session '99' "
                                 "host 'badsession.com' path 'error.txt'" ) );
    YAZ_CHECK( check_response_content(pack, "got sessionfilename './cf.99'" ) );
    YAZ_CHECK( check_response_content(pack,
             "Creating ./cf.99.dump/badsession.com_error.txt failed with 2" ) );
    YAZ_CHECK( check_response_content(pack, "Could not open session file ./cf.99" ) );
    YAZ_CHECK( check_response_content(pack, "Error 400 No session" ) );
    YAZ_CHECK( check_response_content(pack,
             "No cfengine specified, not even looking at content connectors" ) );
    delete(pack);

    // Existing session file, but pretty much emtpy. No cookies in request
    // nor in the session.
    make_session_file( "." , "100",
              "#pretty empty session file, huh\n");
    pack = process_url(cpx, "http://localhost:9036/XXX/cproxydebug-7/node102/"
                      "100/oksession.com/page.txt");
    YAZ_CHECK( check_response_code(pack, 200) );
    YAZ_CHECK( check_response_content(pack, "Got session '100'" ) );
    YAZ_CHECK( check_response_content(pack, "got sessionfilename './cf.100'" ) );
    YAZ_CHECK( check_response_content(pack, "Original request cookie line ''" ) );
    YAZ_CHECK( check_response_content(pack, "Final cookie line: ''" ) );
    YAZ_CHECK( check_no_request_header(pack, "Cookie") );
    delete(pack);

    // Existing session file, but pretty much emtpy.
    // URL contains another part that could be misunderstood for a
    // sessionid. Never been a problem here, but the rewrite filter got
    // this detail wrong once, so we might as well test for it.
    pack = process_url(cpx, "http://localhost:9036/XXX/cproxydebug-7/node102/"
                      "100/oksession.com/2013/page.txt");
    YAZ_CHECK( check_response_code(pack, 200) );
    YAZ_CHECK( check_response_content(pack, "Got session '100'" ) );
    YAZ_CHECK( check_response_content(pack, "got sessionfilename './cf.100'" ) );
    delete(pack);
    
    // Full URL in the GET line
    pack = make_package("http://localhost:9036/XXX/cproxydebug-7/node102/"
           "100/oksession.com/page.txt",1); // the 1 here makes it full URL
    cpx->process(*pack);
    YAZ_CHECK( check_response_code(pack, 200) );
    delete(pack);

    // Url that contains a lowercase prefix
    // We have seen javascript that explictily lowercases whole URLs CPXY-64
    // Would be good if could enforce prefixes to be lowercase to begin with,
    // but there may be uppercase profixes in use in the wild.
    pack = process_url(cpx, "http://localhost:9036/xxx/cproxydebug-7/node102/"
        "100/oksession.com/2013/page.txt");
    YAZ_CHECK( check_response_code(pack, 200) );
    delete(pack);
    
    // Url that contains a bad prefix 
    pack = process_url(cpx, "http://localhost:9036/XXX/cproxydebug-7/node102/"
    "/badprefix/100/oksession.com/2013/page.txt");
    YAZ_CHECK( check_response_code(pack, 400) );
    delete(pack);
    
    // Url that contains a bad prefix in the beginning
    pack = process_url(cpx, "http://localhost:9036/badprefix/XXX/cproxydebug-7/node102/"
    "100/oksession.com/2013/page.txt");
    YAZ_CHECK( check_response_code(pack, 400) );
    delete(pack);
    
    
    // Create a dump dir
    pack = process_url(cpx, "http://localhost:9036/XXX/node102/"
                      "cproxydebug-verbose-dump-nomove/"
                      "100/oksession.com/page.txt");
    YAZ_CHECK( check_response_code(pack, 200) );
    YAZ_CHECK( check_response_content(pack,
             "Created a dump directory ./cf.100.dump/" ) );
    delete(pack);
    
    // URL without a path, with and without trailing slash
    pack = process_url(cpx, "http://localhost:9036/XXX/cproxydebug-7/node102/"
                      "100/oksession.com");
    YAZ_CHECK( check_response_code(pack, 200) );
    delete(pack);
    pack = process_url(cpx, "http://localhost:9036/XXX/cproxydebug-7/node102/"
                      "100/oksession.com/");
    YAZ_CHECK( check_response_code(pack, 200) );
    delete(pack);
    
    // Simple cookie header, no cookies in the session
    // Also, attempt to create the dump dir again
    pack = make_package("http://localhost:9036/XXX/cproxydebug-23/node102/"
                        "100/oksession.com/page.txt",0);
    add_request_header(pack, "x-foobar","Testing a dummy header");
    add_request_header(pack, "Cookie","name1=value1;name2=value2 ; name3=value3");
    cpx->process(*pack);
    
    YAZ_CHECK( check_response_code(pack, 200) );
    YAZ_CHECK( check_response_content(pack, "x-foobar: Testing a dummy header" ) );
    YAZ_CHECK( check_response_content(pack,
                    "Cookie: name1=value1;name2=value2 ; name3=value3" ) );
    YAZ_CHECK( check_response_content(pack,
                           "Dump dir ./cf.100.dump/ exists already") );
    delete(pack);

    // simple referer header
    pack = make_package("http://localhost:9036/"
        "cproxydebug-trace-verbose-keepcontent-nomove/"
        "page.txt",0);
    // Must have the content-bit (8) in debug, or we fake a regular OK response
    add_request_header(pack, "Referer",
         "http://localhost:9036/XXX/node102/100/ref.host/ref.path");
    cpx->process(*pack);

    YAZ_CHECK( check_response_code(pack, 302) );
    YAZ_CHECK( check_response_header(pack, "Location",
                   "http://localhost:9036/XXX/node102/100/ref.host/page.txt") );
    delete(pack);
    
    // simple referer header with just a host name, no trailing slash
    pack = make_package("http://localhost:9036/cproxydebug-15/page.txt",0);
    // Must have the content-bit (8) in debug, or we fake a regular OK response
    add_request_header(pack, "Referer",
                       "http://localhost:9036/XXX/node102/100/ref.host");
    cpx->process(*pack);
    
    YAZ_CHECK( check_response_code(pack, 302) );
    YAZ_CHECK( check_response_header(pack, "Location",
                  "http://localhost:9036/XXX/node102/100/ref.host/page.txt") );
    delete(pack);
    
    // Referer header that ends in a sequence of digits, that can be mistaken
    // for a session number
    // Real-life example from CP-3094 SAFARI_BOOKSONLINE
    pack = make_package("http://ebsco2-cfproxy.indexdata.com:8888/cproxydebug-15/static/201306-7302-proquestcombo/images/6.0/logo.png",0);
    add_request_header(pack, "Referer",
                       "http://localhost:9036/XXX/node102/100/proquestcombo.safaribooksonline.com/book/programming/game-programming/9780321670311");
    cpx->process(*pack);

    YAZ_CHECK( check_response_code(pack, 302) );
    YAZ_CHECK( check_response_header(pack, "Location",
                    "http://localhost:9036/XXX/node102/100/proquestcombo.safaribooksonline.com/static/201306-7302-proquestcombo/images/6.0/logo.png") );    
    delete(pack);

    // simple referer header with just a host name, and a trailing slash
    pack = make_package("http://localhost:9036/cproxydebug-15/page.txt",0);
    // Must have the content-bit (8) in debug, or we fake a regular OK response
    add_request_header(pack, "Referer",
                       "http://localhost:9036/XXX/node102/100/ref.host/");
    cpx->process(*pack);
    
    YAZ_CHECK( check_response_code(pack, 302) );
    YAZ_CHECK( check_response_header(pack, "Location",
                                     "http://localhost:9036/XXX/node102/100/ref.host/page.txt") );
    
    delete(pack);
    
    // Referer header with a longer path, and params
    pack = make_package("http://localhost:9036/cproxydebug-15/"
               "path/to/page.txt?param=value",0);
    // Must have the content-bit (8) in debug, or we fake a regular OK response
    add_request_header(pack, "Referer",
         "http://localhost:9036/XXX/node102/100/"
         "ref.host/ref.path/ref.to/ref.page.html");
    cpx->process(*pack);

    YAZ_CHECK( check_response_code(pack, 302) );
    YAZ_CHECK( check_response_header(pack, "Location",
          "http://localhost:9036/XXX/node102/100/"
          "ref.host/path/to/page.txt?param=value") );
    delete(pack);

    // Referer header, using a full host in the path
    pack = make_package("http://localhost:9036/cproxydebug-15/"
               "path/to/page.txt?param=value",1);
    // Must have the content-bit (8) in debug, or we fake a regular OK response
    add_request_header(pack, "Referer",
         "http://localhost:9036/XXX/node102/100/"
         "ref.host/ref.path/ref.to/ref.page.html");
    cpx->process(*pack);

    YAZ_CHECK( check_response_code(pack, 302) );
    YAZ_CHECK( check_response_header(pack, "Location",
          "http://localhost:9036/XXX/node102/100/"
          "ref.host/path/to/page.txt?param=value") );
    delete(pack);
    
    // No referer header, fall back to a cproxy session cookie
    pack = make_package("http://localhost:9036/cproxydebug-15/"
               "path/to/page.txt",1);
    // Must have the content-bit (8) in debug, or we fake a regular OK response
    add_request_header(pack, "Cookie",
                "name1=value1;cproxysession=fakehost.com/100 ; name3=value3");
    cpx->process(*pack);

    YAZ_CHECK( check_response_code(pack, 302) );
    YAZ_CHECK( check_response_header(pack, "Location",
          "http://localhost:9036/XXX/node102/100/"
          "fakehost.com/path/to/page.txt") );
    delete(pack);

    // Post request with a bad url, but with a referer header
    // CPXY-81. Should process it directly, and not redirect.
    std::string content = url_enc("foo=bar") + "&";
    pack = make_package("http://localhost:9036/cproxydebug-nomove-verbose/"
               "path/to/page.txt?param=value",1,"POST",content);
    add_request_header(pack, "Referer",
         "http://localhost:9036/XXX/node102/100/"
         "ref.host/ref.path/ref.to/ref.page.html");
    cpx->process(*pack);
    YAZ_CHECK( !check_response_code(pack, 302) );   // not a redirect!
    
    remove_dump_dir("./cf.100.dump");

    // More complex session
    make_session_file( "." , "101",
              "Referer http://www.indexdata.com/software/ \n"
              "Username someuser\n"
              "Password secretpassword\n"
              "Proxyip 1.2.3.4\n"
              "Cookie www.indexdata.com; one=ONE\n"
              "Cookie .indexdata.com; two=TWO\n"
              "Cookie other.indexdata.com; three=THREE\n"
              "Cookie WwW.InDeXdAtA.cOm; mixedcase=true\n" );
    // The cookie called mixedcase is to check that the domain matching is
    // indeed case-insensitive (CP-3482)
    pack = process_url(cpx, "http://localhost:9036/XXX/cproxydebug-7/node102/"
                      "101/www.indexdata.com/staff/");
    YAZ_CHECK( check_response_code(pack, 200) );
    YAZ_CHECK( check_response_content(pack, "Got session '101'" ) );
    YAZ_CHECK( check_response_content(pack, "got sessionfilename './cf.101'" ) );
    YAZ_CHECK( check_response_content(pack, "Original request cookie line ''" ) );
    YAZ_CHECK( check_response_content(pack,
                      "Final cookie line: 'mixedcase=true;two=TWO;one=ONE'" ) );
    YAZ_CHECK( check_request_header(pack, "Cookie","two=TWO;one=ONE") );
    YAZ_CHECK( check_request_header(pack, "Referer","http://www.indexdata.com/software/") );
    YAZ_CHECK( check_request_header(pack, "X-Metaproxy-Proxy", "1.2.3.4") );
    
    delete(pack);

    // Cookie merging. Cookies one and two apply to the url, but one is already
    // set in the request, so will not be overwritten.
    pack = make_package("http://localhost:9036/XXX/cproxydebug-7/node102/"
                        "101/www.indexdata.com/staff/",0);
    add_request_header(pack, "Cookie","one=first;four=fourth");
    cpx->process(*pack);

    YAZ_CHECK( check_response_code(pack, 200) );
    YAZ_CHECK( check_request_header(pack, "Cookie",
                            "one=first;four=fourth;mixedcase=true;two=TWO") );
    delete(pack);

    // New-form cookie lines
    make_session_file( "." , "102",
              "Cookie www.indexdata.com; one=ONE\n"
              "Set-Cookie newcookie=newvalue;\n"
              "Set-Cookie newcookie2=v2;Domain=www.indexdata.com;Path=/;\n"
              "Set-Cookie newcookie3=v3;Domain=www.indexdata.com;Path=/staff;\n"
              "Set-Cookie newcookie4=v4;Domain=www.indexdata.com;Path=/about;\n"
                     );
    pack = make_package("http://localhost:9036/XXX/cproxydebug-verbose-nomove-dump/node102/"
                        "102/www.indexdata.com/staff/",0);
    add_request_header(pack, "Cookie","one=first;four=fourth");
    cpx->process(*pack);

    YAZ_CHECK( check_response_code(pack, 200) );
    YAZ_CHECK( check_response_content(pack, "Got session '102'" ) );
    YAZ_CHECK( check_response_content(pack, "New cookie line name='newcookie' value='newvalue'" ) );

    YAZ_CHECK( check_request_header(pack, "Cookie", // no newcookie4, wrong path
            "one=first;four=fourth;newcookie3=v3;newcookie2=v2;newcookie=newvalue") );
    delete(pack);
    remove_dump_dir("./cf.102.dump");


    
    // Creating a session with the help of a parameter file
    // This one fails, since there is no connector mentioned in the p-file
    make_cf_config("./cf-config.unittest.cfg", // config file to write
                   "localhost:9036/XXX/node102",  // url prefix
                   ".", // session directory
                   "localhost:9001" );  // cf-zserver
    delete(cpx);
    cpx = make_filter( conf );   // read the new config with a cf-zserver
    make_param_file(".", "103",
                    "",
                    "user/passwd",
                    "1.2.3.4:80",
                    "testrealm" );

    pack = process_url(cpx, "http://localhost:9036/XXX/cproxydebug-7/node102/"
                    "103/www.indexdata.com/staff/");
    YAZ_CHECK( check_response_code(pack, 400 ) );
    YAZ_CHECK( check_response_content(pack,
                    "No content connector in the param file" ) );
    delete(pack);

    // Creating a session with the help of a param file
    // This will fail too, since we don't have a cf-engine in this test suite.
    // But it should build a proper SRU request.
    make_param_file(".", "103",
                    "fake_content_connector",
                    "user/passwd",
                    "1.2.3.4:80",
                    "testrealm" );

    pack = process_url(cpx, "http://localhost:9036/XXX/cproxydebug-7/node102/"
                        "103/www.indexdata.com/staff/");
    YAZ_CHECK( check_response_code(pack, 400 ) );
    YAZ_CHECK( check_response_content(pack, "Making SRU request "
        "http://localhost:9001/fake_content_connector"
            ",cproxysession=103"
            "&realm=testrealm"
            "&user=user"
            "&password=passwd"
            "&proxy=1.2.3.4:80"
          "?version=1.2"
          "&operation=searchRetrieve"
          "&x-pquery=dummy"
          "&maximumRecords=0" ) );
    delete(pack);

    // Same again, with only username, no password, no realm
    // Using a connector that exists in the cf, so you can verify that
    // it can be used. This needs to be a manual test:
    //   - Edit your /etc/cf-proxy/cproxy.cfg
    //     - Point the sessiondir to the directory you run this test in
    //   - Start a cf-zserver: cd .../cf/engine/src; ./cf-zserver @:9001
    //   - make unittest. This reports FAILURE
    //   - Look at the output, you should see a searchRetrieveResponse
    //     with zero hits.
    //   - After that, there should be the request headers, which should
    //     include this:
    //       Referer: file:///page.html
    // The last point proves that it is actually using a session file
    // created by the cf-zserver.
    // Stop your cf-zserver, and the tests will pass again
    make_param_file(".", "104",
                    "cf-zserver-regression/proxy-url-in-init",
                    "user",
                    "1.2.3.4:80",
                    "" );

    pack = process_url(cpx, "http://localhost:9036/XXX/cproxydebug-7/node102/"
                        "104/www.indexdata.com/staff/");
    YAZ_CHECK( check_response_code(pack, 400 ) );
    YAZ_CHECK( check_response_content(pack, "Making SRU request "
        "http://localhost:9001/cf-zserver-regression/proxy-url-in-init"
            ",cproxysession=104"
            "&user=user"
            "&proxy=1.2.3.4:80"
          "?version=1.2"
          "&operation=searchRetrieve"
          "&x-pquery=dummy"
          "&maximumRecords=0" ) );
    delete(pack);

    // A cf-config file with a bad proxy prefix (all numerical component)
    // Those will get confused with the session number, so they are disallowed
    // (actually, filter_cproxy will handle the just fine, but later a
    // filter_rewrite may run into problems. The old cproxy disallowed this,
    // so we do too)
    // There is a piece of code to detect them, this is to test that piece.
    make_cf_config("./cf-config.unittest.cfg", // config file to write
                   "localhost:9036/XXX/2013",  // Illegal url prefix
                   ".", // session directory
                   "localhost:9001" );  // cf-zserver
    try {
        mp::filter::CProxy *bad_cpx = make_filter( conf );
        YAZ_CHECK (!"Did not throw an exception for a bad proxyprefix!");
    }
    catch (std::exception & e) {
        YAZ_CHECK ("got an exception as expected" );
    }

    delete(cpx);
}


///////////////
// Test 2
// Build a chain of filters, to see everything works together
//  - filter_cproxy to process session files etc
//  - filter_rewrite_html to deproxify the request and proxify resulting html
//  - filter_file to serve static file(s)
//  - Build static test files for filter_file to return

void test2 (void)
{
    mp::Package *pack;

    make_cf_config("./cf-config.unittest2.cfg", // config file to write
            "localhost:9036/YYY/ZZZ",  // url prefix
            ".", // session directory
            "" );  // cf-zserver for content connectors, "localhost:9001"

// Comment out the following to log the packets in every between            
#define LOG_ALL 1
            
    std::string conf =
    "<?xml version='1.0'?>\n"
    "<metaproxy xmlns='http://indexdata.com/metaproxy' version='1.0'>\n"
        "<dlpath>../../metaproxy/src</dlpath>\n"
        "<start route='start'/>\n"
        "<routes>\n"
            "<route id='start'>\n"
                "<filter type='cproxy'>\n"
                    "<debug>0</debug>\n"    // will be enabled as needed
                    "<cfconfig>./cf-config.unittest2.cfg</cfconfig>\n"
                    "<sessionmaxage>1</sessionmaxage>\n"
                "</filter>\n"

#ifdef LOG_ALL
                "<filter type='log'>\n"
                    "<message>L1: between cproxy and rewrite</message> \n"
                    "<category apdu='true'/> \n"
                "</filter>\n"
#endif
                
                // Include the standard cproxy rewrite config example, so we get
                // to test that one, since that is what everybody is supposed to
                // be using (and I'd hate to repeat the whole config here)
                "<include src='cproxyrewrite.xml' /> \n"

#ifdef LOG_ALL
                "<filter type='log'>\n"
                    "<message>L2: between rewrite and http_file</message> \n"
                    "<category apdu='true'/> \n"
                "</filter>\n"
#endif
                
                // Note that the rewrite above always makes requests with
                // absolute paths, so we need an area with such. I guess it
                // should not do that.
                // The whole configuration of the rewrite will be changed soon,
                // when the new model gets implemented, so I leave this as is
                // for now. TODO
                "<filter type='http_file'>\n"
                    "<mimetypes>/etc/mime.types</mimetypes>\n"
                    "<area>\n"
                        "<documentroot>.</documentroot>\n"
                        "<prefix>/testfile</prefix>\n"
                    "</area>\n"
                    "<area>\n"
                        "<documentroot>.</documentroot>\n"
                        "<prefix>http://dummyhost.com/testfile</prefix>\n"
                    "</area>\n"
                "</filter>\n"
                // Second filter to produce raw HTTP responses
                "<filter type='http_file'>\n"
                    "<mimetypes>/etc/mime.types</mimetypes>\n"
                    "<area>\n"
                        "<documentroot>.</documentroot>\n"
                        "<prefix>/rawtestfile</prefix>\n"
                        "<raw>true</raw>\n"
                    "</area>\n"
                    "<area>\n"
                        "<documentroot>.</documentroot>\n"
                        "<prefix>http://dummyhost.com/rawtestfile</prefix>\n"
                        "<raw>true</raw>\n"
                    "</area>\n"
                    // Another area for a more complex hostname, so we can see
                    // cookie domain magic happening
                    "<area>\n"
                        "<documentroot>.</documentroot>\n"
                        "<prefix>http://sub.dummydomain.com/rawtestfile</prefix>\n"
                        "<raw>true</raw>\n"
                    "</area>\n"
                "</filter>\n"

#ifdef LOG_ALL
                "<filter type='log'>\n"
                    "<message>L3: between http_file and bouce</message> \n"
                    "<category apdu='true'/> \n"
                "</filter>\n"
#endif
                
                // A bounce filter to catch those requests that miss the 
                // http_file. With the echo flag, we can use this for testing
                // mysterious requests
                "<filter type='bounce'>\n"
                    "<echo>true</echo>\n"
                "</filter>\n"                    
                
            "</route>\n"
        "</routes>\n"
    "</metaproxy>\n";

    xmlDocPtr doc = xmlParseMemory(conf.c_str(), conf.size());
    mp::RouterXML router(doc,false, "../etc/" );
    YAZ_CHECK("configured the router chain all right");
    
    std::string page1 = 
      "<html><body>\n"
      "<h1>test page</h1>\n"
      "absolute links: <br/>\n"
      "<a href='http://www.a1.com'>a1.com</a><br/>\n"
      "<a href='http://www.a2.com/'>a2.com/</a><br/>\n"
      "<a href='http://www.a3.com/staff'>a3.com/staff</a><br/>\n"
      "<a href='http://www.a4.com/staff/'>a4.com/staff/</a><br/>\n"
      "<a href='http://www.a5.com/staff/page.html'>"
         "a5.com/staff/page.html</a><br/>\n"
      "<a href='//doubleslash.com/a6'>doubleslash.com/a6</a><br/>\n"
      // See EUC-2437, Morningstar has stuff like href="//im.morningstar.com/..."
      "<A href='http://www.a7.com/uppercasetag'>a7.com/uppercasetag</A><br/>\n"
      "<a href='ftp://ftp.indexdata.com/pub/a8/file.gz'>ftp a8/file.gz</a><br/>\n"
      "<a href='https://secure.indexdata.com/a9/'>https a9</a><br/>\n"
      "<a href='http://www.a10.com/../path/page.html'>dot-dot path</a>\n"
      // See CPXY-61, BIP (Books in Print) had abs links with a path that 
      // starts with a double-dot component. Won't matter in a browser, will
      // break in the middle of the proxified path
      
      "host-relative links: <br/>\n"
      "<a href='/'>/</a><br/>\n"
      "<a href='/h1'>/h1</a><br/>\n"
      "<a href='/h2/page'>h2/page</a><br/>\n"
      "<a href='/h3/page/'>h3/page/</a><br/>\n"
      "<a href='/h4/page.html'>h4/page.html</a><br/>\n"

      "relative links: <br/>\n"
      "<a href='r1'>r1</a><br/>\n"
      "<a href='r2/'>r2/</a><br/>\n"
      "<a href='r3/page'>r3/page</a><br/>\n"
      "<a href='r4/page/'>r4/page/</a><br/>\n"
      "<a href='r5/page.html'>r5/page.html</a><br/>\n"

      "plain text: <br/>\n"
      "http://www.p1.com <br/>\n"
      "http://www.p2.com/staff/page.html <br/>\n"

      "script<br/>\n"
      "<a href=\"http://www.preparing.com/some/funny/path/\">\n"
      "<script type=\"text/javascript\">\n"
      "  var url1=\"http://url1.in.script\"; \n"
      "  var url2=\"http://url2.in.script/\"; \n"
      "  var url3=\"http://url3.in.script/page\"; \n"
      "  var url4=\"htt\" + \"p://url4\" + \".in.script\"; \n"
      "  var hostrel1=\"/\"; \n"
      "  var hostrel2=\"/hr2\"; \n"
      "  var hostrel3=\"/hr3/page\"; \n"
      "  var hostrel4=\"/hr4/page/\"; \n"
      "  var hostrel5=\"/hr5/page.html\"; \n"
      "  quotes: http://unquoted.com/  \n"
      "  quotes: 'http://singlequoted.com/'  \n"
      "  quotes: \"http://doublequoted.com/\"  \n"
      "  quotes: 'http://mixedquoted1.com/\"  \n"
      "  quotes: \"http://mixedquoted2.com/'  \n"
      "  // 'http://outcommented.com/'  \n"
      "  document.write(\"</script\" + \">\") \n"   // These should not 
      "  document.write(\"<\\/script>\") \n"        // end the script tag
      "  if(a<b)\n" // MP-486
      "  doc.write(\"<a href='http://write.com'>Click</a>\"); \n"
      "  if ( url.beginswith(\"http://\") ) ... \n"
      "  <a href=/unquoted >unquoted</a> \n"   // This should not be proxified inside a script
                                 // but may happen if the fake closing tags above are taken
                                 // for the real thing.
      "</script>\n"
      "<a href=\"#\" onclick=\"window.location=\'http://first.com\';\" /> \n"
      "<a href=\"#\" onclick=\'window.location=\"http://second.com\";\' /> \n"
      "<p/>\n"
      "<div>This line should be attacked by the custom replace trick </div>\n"
      "<div>Another line that should be attacked by the another trick in ANOTHER way</div>\n"
      "<script>x<y;</script>\n"
      "<span/\n"
      "<script>x=y;//comment\n"
              "c=d; \n"
              "a=b;</script>\n"
      "</body></html>\n";
    
    make_html_file( "./file1.html", page1 );
    
    make_session_file( "." , "200",
                "#Session for the combined test\n"
                "Set-Cookie sessioncookie=sesval;path=/\n"
                "Custom-Pattern attacked Custom-Replacement repaired\n"
                "Custom-Pattern line should Custom-Replacement line might Custom-Options none\n"
                "Custom-Pattern (cu[^ ]+om) (replace) (trick) "
                     "Custom-Replacement ---$1---$2-$2-$2-$3 "
                     "Custom-Options none\n"
                "Custom-Pattern Another "
                     "Custom-Replacement (One More) "
                     "Custom-Options i "
                     "Custom-Content-Type html \n"
                "Custom-Pattern The "
                     "Custom-Replacement Should-not-have-been-replaced "
                     "Custom-Options i "
                     "Custom-Content-Type text/plain \n"
                     );
    
    // Simple test with nomove, only invokes cproxy filter
    // Tests mostly that the other filters load all right
    pack = make_package("http://localhost:9036/"
                        "cproxydebug-verbose-nomove/"
                        "YYY/ZZZ/200/"
                        "www.indexdata.com/staff/",0);
    pack->router(router).move();
    YAZ_CHECK( check_response_code(pack, 200) );
    YAZ_CHECK( check_response_content(pack, "got sessionfilename './cf.200'" ) );
    delete(pack);

    // simple request going through the whole filter chain
    pack = make_package("http://localhost:9036/"
            "cproxydebug-verbose-dump/"
            "YYY/ZZZ/200/"
            "dummyhost.com/testfile/file1.html",1);  // note the 1 for full URL
    pack->router(router).move();
    YAZ_CHECK( check_response_code(pack, 200) );
    YAZ_CHECK( check_response_content(pack, "got sessionfilename './cf.200'" ) );
    YAZ_CHECK( check_response_header( pack, "Content-Type", "text/plain" ) );
       // since we are debugging
    YAZ_CHECK( check_response_content(pack, "<h1>test page</h1>" ) );
    YAZ_CHECK( check_response_content(pack,
      "http://localhost:9036/YYY/ZZZ/200/www.a5.com/staff/page.html" ) );
    // the cproxysession cookie has the target host name, and the session number
    YAZ_CHECK( check_response_header(pack,"Set-Cookie",
                            "cproxysession=dummyhost.com/200; Path=/" ) );
    delete(pack);

    // Same as above, but without any debug info
    pack = make_package("http://localhost:9036/"
            "YYY/ZZZ/200/"
            "dummyhost.com/testfile/file1.html",1);
    pack->router(router).move();
    YAZ_CHECK( check_response_code(pack, 200) );
    YAZ_CHECK( check_response_header( pack, "Content-Type", "text/html" ) );
    
    YAZ_CHECK( check_response_content(pack, "<h1>test page</h1>" ) );

    YAZ_CHECK( check_response_content(pack, 
                 "\'http://localhost:9036/YYY/ZZZ/200/www.a1.com\'" ) );
    YAZ_CHECK( check_response_content(pack, 
                 "\'http://localhost:9036/YYY/ZZZ/200/www.a2.com/\'" ) );
    
    YAZ_CHECK( check_response_content(pack, 
                 "'http://localhost:9036/YYY/ZZZ/200/www.a3.com/staff'" ) );
    YAZ_CHECK( check_response_content(pack, 
                 "http://localhost:9036/YYY/ZZZ/200/www.a4.com/staff/" ) );
    YAZ_CHECK( check_response_content(pack, 
                 "http://localhost:9036/YYY/ZZZ/200/www.a5.com/staff/page.html" ) );
    YAZ_CHECK( check_response_content(pack, 
                 "http://localhost:9036/YYY/ZZZ/200/doubleslash.com/a6" ) );

    YAZ_CHECK( check_response_content(pack, 
                 "http://localhost:9036/YYY/ZZZ/200/www.a7.com/uppercasetag" ) );
    YAZ_CHECK( check_response_content(pack, 
                 "ftp://ftp.indexdata.com/pub/a8/file.gz" ) );
    YAZ_CHECK( check_response_content(pack, "https://secure.indexdata.com/a9/" ) );
        
    YAZ_CHECK( check_response_content(pack, 
                 "http://localhost:9036/YYY/ZZZ/200/www.a10.com/path/page.html" ) );
    
    // host-relative links
    YAZ_CHECK( check_response_content(pack, "\'http://localhost:9036/YYY/ZZZ/200/dummyhost.com/\'" ) );
    YAZ_CHECK( check_response_content(pack, "\'http://localhost:9036/YYY/ZZZ/200/dummyhost.com/h1\'" ) );
    YAZ_CHECK( check_response_content(pack, "\'http://localhost:9036/YYY/ZZZ/200/dummyhost.com/h2/page\'" ) );
    YAZ_CHECK( check_response_content(pack, "\'http://localhost:9036/YYY/ZZZ/200/dummyhost.com/h3/page/\'" ) );
    YAZ_CHECK( check_response_content(pack, "\'http://localhost:9036/YYY/ZZZ/200/dummyhost.com/h4/page.html\'" ) );

    // Check that the links in the text content are not changed
    YAZ_CHECK( check_response_content(pack, "http://www.p1.com " ) );
    YAZ_CHECK( check_response_content(pack, "http://www.p2.com/staff/page.html" ) );

    YAZ_CHECK( check_response_content(pack,
           "url1=\"http://localhost:9036/YYY/ZZZ/200/url1.in.script\"" ) );

    YAZ_CHECK( check_response_content(pack,
           "url3=\"http://localhost:9036/YYY/ZZZ/200/url3.in.script/page\"" ) );

    // We can not proxify host-relative links, any string that starts with a '/'
    // would look like one, and could be something totally different, like an XPath
    YAZ_CHECK( check_response_content(pack, "hostrel1=\"/\"" ) );
    YAZ_CHECK( check_response_content(pack, "hostrel2=\"/hr2\"" ) );
    YAZ_CHECK( check_response_content(pack, "hostrel3=\"/hr3/page\"" ) );
    YAZ_CHECK( check_response_content(pack, "hostrel4=\"/hr4/page/\"" ) );
    YAZ_CHECK( check_response_content(pack, "hostrel5=\"/hr5/page.html\"" ) );
    
    YAZ_CHECK( check_response_content(pack, "http://unquoted.com" ) );
    YAZ_CHECK( check_response_content(pack, "http://localhost:9036/YYY/ZZZ/200/singlequoted.com" ) );
    YAZ_CHECK( check_response_content(pack, "http://localhost:9036/YYY/ZZZ/200/doublequoted.com" ) );
    YAZ_CHECK( check_response_content(pack, "http://localhost:9036/YYY/ZZZ/200/mixedquoted1.com" ) );
    YAZ_CHECK( check_response_content(pack, "http://localhost:9036/YYY/ZZZ/200/mixedquoted2.com" ) );
    YAZ_CHECK( check_response_content(pack, "http://outcommented.com" ) );
    YAZ_CHECK( check_response_content(pack, "http://localhost:9036/YYY/ZZZ/200/write.com" ) );
      // See MP-486
    YAZ_CHECK( check_response_content(pack, "if(a<b)" ) );
    YAZ_CHECK( check_response_content(pack, "document.write(\"</script\" + \">\")" ) );
    YAZ_CHECK( check_response_content(pack, "document.write(\"<\\/script>\")" ) );

    // The script thing goes wrong at the '<', and thereafter messes up whitespace:
    YAZ_CHECK( !check_response_content(pack, "if(a<b) doc.write" ) );
      // Make sure we stay in script mode all the way to the end
    YAZ_CHECK( check_response_content(pack, "<a href=/unquoted >unquoted</a>" ) );

    YAZ_CHECK( check_response_content(pack,
        "onclick=\"window.location='http://localhost:9036/YYY/ZZZ/200/first.com';" ) );
    YAZ_CHECK( check_response_content(pack,
        "onclick=\'window.location=\"http://localhost:9036/YYY/ZZZ/200/second.com\";" ) );

    // test for CPXY-47, require at least something in the host portion
    // Have seen such construct in real life.
    YAZ_CHECK( check_response_content(pack,
        "if ( url.beginswith(\"http://\") ) " ) );
    
    // Simple custom replacement with a space in both strings
    YAZ_CHECK( check_response_content(pack,
        "This line might be repaired by the") );
    // Back references
    YAZ_CHECK( check_response_content(pack,
        "---custom---replace-replace-replace-trick") );
    // multiple replacements on one line, ignore case
    YAZ_CHECK( check_response_content(pack,
        "(One More) line") );
    YAZ_CHECK( check_response_content(pack,
        "(One More) way") );
    YAZ_CHECK( !check_response_content(pack,
        "Should-not-have-been-replaced") );

    // Test for closing script tag, when preceded by '<x' CPXY-68
    YAZ_CHECK( check_response_content(pack,
        "x<y;</script>") );
    YAZ_CHECK( check_response_content(pack,
        "a=b;</script>") );
    
    delete(pack);

    // Url that looks like it might have a session number further down
    // Tests mostly the configuration above, the request from regex must have
    // a non-greedy .+? in the pxpath!
    pack = make_package("http://localhost:9036/"
        "cproxydebug-verbose/"
        "YYY/ZZZ/200/"
        "dummyhost.com/testfile/file1.html?badurl=/foo/299/something",0);
    pack->router(router).move();
    YAZ_CHECK( check_response_code(pack, 200) );
    delete(pack);

    
    // Page not found
    pack = make_package("http://localhost:9036/"
        "cproxydebug-verbose/"
        "YYY/ZZZ/200/"
        "dummyhost.com/testfile/nosuchfile.html",1);
    pack->router(router).move();
    YAZ_CHECK( check_response_code(pack, 404) );
    YAZ_CHECK( check_response_content(pack, "<TITLE>YAZ" ) );
    YAZ_CHECK( check_response_content(pack, "Error: 404" ) );
    // At the moment the rewrite blindly replaces everything
    YAZ_CHECK( check_response_content(pack, "http://www.w3.org/TR/html4/strict.dtd" ) );
    delete(pack);
    
    // Page not found, without debug info
    pack = make_package("http://localhost:9036/"
        "YYY/ZZZ/200/"
        "dummyhost.com/testfile/nosuchfile.html",1);
    pack->router(router).move();
    YAZ_CHECK( check_response_code(pack, 404) );
    YAZ_CHECK( check_response_content(pack, "<TITLE>YAZ" ) );
    YAZ_CHECK( check_response_content(pack, "Error: 404" ) );
    delete(pack);
    
    // Post request with a bad url, but with a referer header
    // CPXY-81. Should process it directly, and not redirect.
    // Also, makes an entry in the _cookietrace, which we analyze later
    // TODO: Not coded that way yet (29-Aug-2013)
    std::string contentparams = url_enc("foo=bar") + "&";
    pack = make_package("http://localhost:9036/"
               "cproxydebug-keepcontent-nomove-dump-verbose/"
               "testfile/file1.html",
               1,"POST",contentparams);
    add_request_header(pack, "Cookie",
                "n1=v1;cproxysession=dummyhost.com/200 ; n2=v2");
    pack->router(router).move();
    YAZ_CHECK( check_response_code(pack, 200) );  // most of all, not a 302


    // Test with custom response headers
    std::string rawpage =
      "HTTP/1.1 200 OK\n"
     // "Content-Type: text/html\n"
      "Content-Type: text/html;charset=UTF-8\n"
        // The regexp inthe rules should match no matter what charset we have!
      "X-foo: raw\n"
      "Link: <http://see.also.com/absolute/>; rel=\"alternate\" \n"
      "Link: <relative/path/>; rel=\"alternate\" \n"
      "Link: </host/relative/>; rel=\"alternate\" \n"
      "X-Do-Not-Look: http://do.not.look.com/\n"
      "Set-Cookie: n1=v\n"
      "Set-Cookie: n2=v; path=/foo2\n"
      "Set-Cookie: n3=v; path=/\n"
      "Set-Cookie: n4=v; domain=dummydomain.com\n"
      "Set-Cookie: n5=v; domain=.dummydomain.com\n"
      "Set-Cookie: n6=v; path=/rawtestfile; domain=dummydomain.com\n"
      "Set-Cookie: n7=v; path=/rawtestfile; domain=.dummydomain.com\n"
      "Set-Cookie: n8=v; pAtH=/raWtestFILE; dOmAiN=DummyDOMAIN.com; "
           "Expires: 31 Dec 2013 12:34:56; Secure \n"
      "Set-Cookie: n9=v; domain=wrong.com\n"
      "Set-Cookie: e=\n"   // empty value, see CPXY-75
        "\n"
      "<html><head>"
      "</head>\n"
      "<body>\n"
      "<h1>test page</h1>\n"
      "</body></html>\n";

    make_html_file( "./rawfile.html", rawpage );
    
    pack = make_package("http://localhost:9036/"
            "cproxydebug-verbose-dump-keepcontent/"
            "YYY/ZZZ/200/"
            "sub.dummydomain.com/rawtestfile/rawfile.html",0);
    pack->router(router).move();
    YAZ_CHECK( check_response_code(pack, 200) );
    YAZ_CHECK( check_response_header(pack,"Content-Type", "text/html") );
    YAZ_CHECK( check_response_header(pack,"X-foo", "raw") );
    // Link headers are special, they have <>'s around the URL, and thus need
    // their own rules.
    YAZ_CHECK( check_response_header(pack,"Link",
                 "<http://localhost:9036/YYY/ZZZ/200/see.also.com/absolute/>;") );
    YAZ_CHECK( check_any_response_header(pack,"Link", "<relative/path/>;"));
    YAZ_CHECK( check_any_response_header(pack,"Link",
      "<http://localhost:9036/YYY/ZZZ/200/sub.dummydomain.com/host/relative/>;") );
      // The Link header needs a special regexp, as it has a funny format, with
      // the URL inside angle brackets. Need to write that. TODO!
    YAZ_CHECK( check_response_header(pack,"X-Do-Not-Look",
                                    "http://do.not.look.com/") );
    // Check the cookie rewriting
    // Set-Cookie: n1=v
    YAZ_CHECK( check_any_response_header(pack,"Set-Cookie",
                  "n1=v" ) );
       // correctly proxified path, no domain
    YAZ_CHECK( check_any_response_header(pack,"Set-Cookie",
            "n1=v; "
            "path=/YYY/ZZZ/200/sub.dummydomain.com/rawtestfile/rawfile.html" ) );
       
    // Set-Cookie: n2=v; path=/foo2
    YAZ_CHECK( check_any_response_header(pack,"Set-Cookie",
                "n2=v; "
                "path=/YYY/ZZZ/200/sub.dummydomain.com/foo2" ) );
    // Set-Cookie: n3=v; path=/
    YAZ_CHECK( check_any_response_header(pack,"Set-Cookie",
                "n3=v; "
                "path=/YYY/ZZZ/200/sub.dummydomain.com/" ) );
    //Set-Cookie: n4=v; domain=dummydomain.com
    YAZ_CHECK( check_any_response_header(pack,"Set-Cookie",
                //"n4=v; domain=localhost:9036; "
                "n4=v; domain=localhost; "
                "path=/YYY/ZZZ/200/dummydomain.com/" ) );
    // Set-Cookie: n5=v; domain=.dummydomain.com     - wildcard domain !!
    YAZ_CHECK( check_any_response_header(pack,"Set-Cookie",
                "n5=v; domain=localhost; "
                "path=/YYY/ZZZ/200/dummydomain.com/rawtestfile/rawfile.html" ) );
    YAZ_CHECK( check_any_response_header(pack,"Set-Cookie",
                "n5=v; domain=localhost; "
                "path=/YYY/ZZZ/200/" ) );
    // Set-Cookie: n6=v; path=/rawtestfile; domain=dummydomain.com
    YAZ_CHECK( check_any_response_header(pack,"Set-Cookie",
                "n6=v; "
                "path=/YYY/ZZZ/200/dummydomain.com/rawtestfile; "
                "domain=localhost" ) );
    // Set-Cookie: n7=v; path=/rawtestfile; domain=.dummydomain.com  -- wild !!
    YAZ_CHECK( check_any_response_header(pack,"Set-Cookie",
                "n7=v; "
                "path=/YYY/ZZZ/200/; "
                "domain=localhost" ) );
    YAZ_CHECK( check_any_response_header(pack,"Set-Cookie",
                "n7=v; "
                "path=/YYY/ZZZ/200/dummydomain.com/rawtestfile; "
                "domain=localhost" ) );
    // A check with mixed-case attr names, extra attrs etc
    // n8=v; pAtH=/raWtestFILE; dOmAiN=DummyDOMAIN.com; Expires: 31 Dec 2013 12:34:56; Secure"
    YAZ_CHECK( check_any_response_header(pack,"Set-Cookie",
                "n8=v; "
                "pAtH=/YYY/ZZZ/200/DummyDOMAIN.com/raWtestFILE; "
                "dOmAiN=localhost; "
                "Expires: 31 Dec 2013 12:34:56; Secure " ) );
    
    delete(pack);

    
    // This fetches the big test page again.
    // Since it set some cookies last time, those cookies are expected to 
    // be in the request. They are not, and the cookie analysis (below) is 
    // supposed to notice what is missing.
    pack = make_package("http://localhost:9036/"
        "cproxydebug-verbose-dump-keepcontent/"
        "YYY/ZZZ/200/"
        "sub.dummydomain.com/rawtestfile/rawfile.html?second=time",0);
    add_request_header(pack, "Cookie",
                "n4=xxx;n6=v;n8=v; x9=xxx;sessioncookie=sesval; e=");
    pack->router(router).move();
    YAZ_CHECK( check_response_code(pack, 200) );
    delete(pack);
    
    // Test the cookie analysis mode
    // Makes use of the cookies set up in an earlier tests and logged
    pack = make_package("http://localhost:9036/"
        "cproxydebug-verbose-cookie/"
        "YYY/ZZZ/200/"
        "sub.dummydomain.com/rawtestfile/rawfile.html?cookie=analysis",0);
    pack->router(router).move();
    YAZ_CHECK( check_response_content(pack,
                   "Analyzing cookies from ./cf.200.dump/_cookietrace" ) );
    YAZ_CHECK( check_response_content(pack,
                   "Missing cookie! Expected name='n1' value='v'" ) );
    YAZ_CHECK( check_response_content(pack,
                   "Missing cookie! Expected name='n3' value='v'" ) );
    YAZ_CHECK( check_response_content(pack,
                   "Missing cookie! Expected name='n5' value='v'" ) );
    YAZ_CHECK( check_response_content(pack,
                   "Missing cookie! Expected name='n7' value='v'" ) );
    YAZ_CHECK( check_response_content(pack,
                   "Bad Value. Expected: name='n4' value='v'" ) );
    // empty cookie
    YAZ_CHECK( check_response_content(pack,
                   "OK: name='e' value=''" ) );
    // Post request not redirected (CPXY-81)
    YAZ_CHECK( !check_response_content(pack,
               "Directing to http://localhost:9036/YYY/ZZZ/200/"
               "dummyhost.com/testfile/file1.html" ) );
    //YAZ_CHECK( check_response_code(pack, 200) );
    delete(pack);
    
    std::string movedpage =
      "HTTP/1.1 301 Moved permanently\r\n"
      "Location: http://www.newlocation.com/\r\n"
      "X-foo: bar\r\n"
      "\r\n"
      "Some content";
        // we have no content-type, but may still have content
        // did bite us once...

    make_html_file( "./movedfile", movedpage );

    pack = make_package("http://localhost:9036/"
          //  "cproxydebug-verbose-dump/"
            "YYY/ZZZ/200/"
            "dummyhost.com/rawtestfile/movedfile",0);
    pack->router(router).move();
    YAZ_CHECK( check_response_code(pack, 301) );
    YAZ_CHECK( check_no_response_header(pack,"Content-Type") );
    YAZ_CHECK( check_response_header(pack,"Location", 
                   "http://localhost:9036/YYY/ZZZ/200/www.newlocation.com/") );
    delete(pack);

    std::string movedpage2 =
      "HTTP/1.1 302 Moved\r\n"
      "Refresh: http://refresh1.com/page\r\n"
      "Location: /host/relative/link?with=params\r\n"
      "Refresh: http://refresh2.com/page\r\n"
      "Refresh: /refresh3.com/page\r\n"
      "\r\n";

    make_html_file( "./movedfile2", movedpage2 );

    pack = make_package("http://localhost:9036/"
          //  "cproxydebug-verbose-dump/"
            "YYY/ZZZ/200/"
            "dummyhost.com/rawtestfile/movedfile2",0);
    pack->router(router).move();
    YAZ_CHECK( check_response_code(pack, 302) );
    YAZ_CHECK( check_no_response_header(pack,"Content-Type") );
    YAZ_CHECK( check_response_header(pack,"Location",
                "http://localhost:9036/YYY/ZZZ/200/"
                "dummyhost.com/host/relative/link?with=params") );
    delete(pack);

    
    // Test that the package can get to the bounce filter, and that the request
    // is properly deproxified at that point.
    pack = make_package("http://localhost:9036/"
        //  "cproxydebug-verbose-dump/"
        "YYY/ZZZ/200/"
        "dummyhost.com/wrongpath/somefile",0);
    pack->router(router).move();
    YAZ_CHECK( check_response_code(pack, 400) );
    YAZ_CHECK( check_response_content(pack, 
                            "GET http://dummyhost.com/wrongpath/somefile" ) );
    delete(pack);
    
    // simple POST request
    std::string content = url_enc("foo=bar") + "&" +
      url_enc("url1=http://google.com?q=water") + "&" +
      url_enc("url2=http://localhost:9036/YYY/ZZZ/200/dummyhost.com/testfile/file1.html") ;
    pack = make_package("http://localhost:9036/"
        //  "cproxydebug-verbose-dump/"
        "YYY/ZZZ/200/"
        "dummyhost.com/wrongpath/somefile",0, "POST", content);
    pack->router(router).move();
    YAZ_CHECK( check_response_code(pack, 400) );
    YAZ_CHECK( check_response_content(pack, 
                  "POST http://dummyhost.com/wrongpath/somefile" ) );
    YAZ_CHECK( check_response_content(pack, 
           "foo%3Dbar&url1%3Dhttp%3A%2F%2Fgoogle.com%3Fq%3Dwater&" ) );
    YAZ_CHECK( check_response_content(pack, 
      "url2%3Dhttp%3A%2F%2Flocalhost%3A9036%2FYYY%2FZZZ%2F200%2Fdummyhost.com%2Ftestfile%2Ffile1.html" ) );
    delete(pack);

#if 1
    // MP-482
    pack = make_package("http:/localhost:9036/"
          //  "cproxydebug-verbose-dump/"
         //   "YYY/ZZZ/200/"
            "web.gideononline.com/web/images_n/sub_vaccines.gif",0);
    add_request_header(pack, "Referer",
         "http://localhost:9036/XXX/node102/100/ref.host/ref.path");
    pack->router(router).move();
    YAZ_CHECK( check_response_code(pack, 400) );
    delete(pack);
#endif

    remove_dump_dir( "./cf.200.dump" );  
    
} // test2

////////////////////
// main 
int main (int argc, char **argv)
{
    YAZ_CHECK_INIT(argc, argv);
    YAZ_CHECK_LOG();
    // Make sure we don't have old test dirs around, they will screw up
    // some tests that verify the creation of such.
    remove_dump_dir( "./cf.100.dump", true );
    remove_dump_dir( "./cf.102.dump", true );
    remove_dump_dir( "./cf.200.dump", true );
    try {
        test1();
        test2();
    }
    
    catch (std::exception & e) {
        yaz_log(YLOG_LOG,"Some test threw an uncaught exception!");
        yaz_log(YLOG_LOG,"  %s" , e.what() );
        YAZ_CHECK (false);
    }
    remove_temp_files();
    yaz_log(YLOG_LOG,"All tests done");
    YAZ_CHECK_TERM;
}


/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */

