var EXPORTED_SYMBOLS = ["Step"];

Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/objHelper.js');
Components.utils.import('resource://indexdata/util/textHelper.js');
Components.utils.import('resource://indexdata/util/jsonPathHelper.js');

/*
 * Step superclass - all steps need extend this class to conform to the Step
 * interface and implement common functionality (e.g configuration parsing)
 * Most interface methods are "virtual" and must be redefined while common
 * functionality methods may be overriddden to change the default bahaviour.
 *
 * @author jakub
 */

var Step = function () {
  // JS does not support real class inheritance, so we fake it by using the
  // prototype of the super class as the base for the child class. Unfortunately
  // the protype for the child is instantiated only once when the child class
  // is defined. That means that all the members and methods are shared across
  // instances of the same child class (pseudo-static with access to all child
  // memmbers) - carefull with that.
  this.conf = null;  //configuration
  this.task = null;  //parent task
  this.comments = [];
  this.isAlt = false;
};

// interface and base methods (required implementation)

Step.prototype.constructor = function (task) {
  throw new Error("Constructor: method not implemened.");
};

Step.prototype.init = function (task) {
  throw new Error("Init: method not implemened.");
};

Step.prototype.draw = function (surface) {
  throw new Error("Draw: method not implemened.");
};

Step.prototype.run = function (task) {
  throw new Error("Run: method not implemened.");
};

Step.prototype.getComments = function () {
  return this.comments;
};

// read-only attributes

Step.prototype.getClassName = function () {
  return "Step";
};

Step.prototype.getDisplayName = function () {
  return "No name specified.";
};

Step.prototype.getDescription = function () {
  return "No description specified.";
};

Step.prototype.getVersion = function () {
  return "0.0";
};

// methods with default implementation, (overriding optional)

Step.prototype.getIndentRange = function () {
  //[from, to] as indexed from the current position
  //altness is intented on top of it
  return [0,0];
};

Step.prototype.renderArgs = function () {
  // Trivial default implementation: individual step classes should
  // override this to provide a brief string representing the step
  // arguments.
  return "";
};

Step.prototype.toString = function () {
  var name = this.getDisplayName();
  var args = this.renderArgs();
  if (args) name += " (" + args + ")";
  return name;
}


// DEPRECATION

// return true if the step has been deprecated and should no longer be used
// for new connectors
Step.prototype.isDeprecated = function () {
  return false;
};

// reason for deprecation and/or what steps should be used instead
Step.prototype.getDeprecationInfo = function () {
  return "";
};

Step.prototype.saveConf = function (stepNode) {
  stepNode.setAttribute("name", textHelper.ccToUS(this.getClassName()));
  stepNode.setAttribute("version", this.getVersion());
  if (this.isAlt) stepNode.setAttribute("alt", "yes");
  if (this.isDisabled) stepNode.setAttribute("disabled", "yes");
  if (typeof this.processConf === "function") this.processConf();
  if (this.conf === null) {
    dump(this.getClassName() + ": Configuration is null, nothing to save.\n");
    return;
  }
  this.writeConf(this.conf, stepNode);
};

Step.prototype.reqVer = function () {
    return this.conf.reqVer;
}

Step.prototype.loadConf = function (stepNode) {
  var reqVer = stepNode.getAttribute("version");
  var curVer = this.getVersion();
  this.conf = {};
  this.conf["reqVer"] = reqVer;

  this.readConf(this.conf, stepNode);

  // attempt to promote conf to current version
  if (reqVer !== curVer &&
    !this.upgrade(Number(reqVer), Number(curVer), this.conf)) {
    // major.minor version checking
    // major updates break compatibility, minor are only backward compatible
    if (!this.checkVersion(reqVer, curVer)) {
      var name = stepNode.getAttribute("name");
      throw new Error("This connector requires step '" + name
          + "' version " + reqVer
          + " while the available version is " + curVer);
    }
  }
};

// configuration parsing handlers
// override to change default behaviour

Step.prototype.writeConf = function (confObj, stepNode) {
  objHelper.objectToXml("stepConf", confObj, stepNode);
};

Step.prototype.readConf = function (confObj, stepNode) {
  var confNode = stepNode.getElementsByTagName("stepConf")[0];
  if (confNode === undefined) return;
  var parsedConf = objHelper.objectFromXml(confNode);
  // copy the contents
  for (var key in parsedConf)
    confObj[key] = parsedConf[key];
};

// helper attributes

Step.prototype.getPageWindow = function () {
  return this.task.connector.getPageWindow();
};

Step.prototype.getPageDoc = function () {
  return this.task.connector.getPageDoc();
};

// version checking

// check if version A (required) is compatible with version B (available)
// ie. version from connector's step conf (A), version of step in current engine (B)
Step.prototype.checkVersion = function (verA, verB) {
  if (typeof verA != "string") throw new Error("Required version not specified");
  if (typeof verB != "string") throw new Error("Available version not specified");
  var verAparts = verA.split(".");
  var verBparts = verB.split(".");

  var verAmajor = Number(verAparts[0]);
  var verBmajor = Number(verBparts[0]);

  if (verAmajor > verBmajor) return false;

// Minor version increments are backwards compatible.

//   var verAminor = Number(verAparts[1]);
//   var verBminor = Number(verBparts[1]);
//
//   if (verAminor > verBminor) return false;

  return true;
}

// return true if conf has successfully been modified for
// compatibility with current version
Step.prototype.upgrade = function (confVer, curVer, conf) {
  // can't upgrade if the connector is newer than the step
  if (confVer > curVer)
    return false;
  // this is something you can keep adding to if you do tests as ranges ie.:

  // if (confVer < 2)
  //   ...do updates to 3
  // if (confVer < 3)
  //   ...update to 4

  // when 5 comes out you can just just add another if for that
  // (possibly just returning true if you've done a major version
  // increment that doesn't break backwards compatibility) and
  // you'll have upgrades all the way from version 2.
  return true;
}

// Used arguments do not need to be unique as they are aggregated in the Task
// TODO - Get from conf, to make it more automatic
Step.prototype.getUsedArgs = function () {
  return [];
};


// Check if the capability flag should default to something.
// Return true, false, or null for don't-care
// Goes through the conf, locates what looks like jsonPaths, and
// guesses decent index- and result- defaults. It tries to be clever, and
// avoid flagging input variables if the conf setting is named 'out', and
// output flags for 'in'. This is totally unreliable, but happens to match
// the way things are done in transform, split, join, and other steps that
// use both input and output.
// if this logic is not suitable for a step, it should override the function
Step.prototype.capabilityFlagDefault = function ( flag ) {
  if ( !this.conf )
    return null;
  for ( var c in this.conf ) {
    if ( this.conf[c] &&
         this.conf[c].path != undefined && // looks like a jsonPath
         this.conf[c].key != undefined ) {
      /*dump("Capdef: '" + flag + "' " +
        "p='" + this.conf[c].path +"' " +
        "k='" + this.conf[c].key +"' " +
        "\n"); */
      if ( c != 'out'  &&
           jsonPathHelper.capabilityFlagDefault(
             this.conf[c], flag, "$.input", "index") )
        return true;
      if ( c != 'in' &&
           jsonPathHelper.capabilityFlagDefault(
            this.conf[c], flag, "$.output", "result") )
        return true;
    }
  }
  return null;
}

//step specific cloning
Step.prototype.clone = function () {
  var clone = new this.constructor();
  var conf = JSON.parse(JSON.stringify(this.conf));
  clone.conf = conf;
  //be picky abot what is being cloned -- we don't want references or internal
  //step state, only 'persistent' attributes should survive cloning
  clone.isAlt = this.isAlt;
  clone.isDisabled = this.isDisabled;
  return clone;
}
