var EXPORTED_SYMBOLS = ["Task"];
Components.utils.import('resource://indexdata/runtime/TaskUnitTest.js');
Components.utils.import('resource://indexdata/runtime/Comment.js');
Components.utils.import('resource://indexdata/util/textHelper.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/logging.js');
Components.utils.import('resource://indexdata/util/EventList.js');

// use 'execution' logger to log all messages, this is IMPORTANT
// so messages are properly passed on.
var logger = logging.getLogger('execution');

var Task = function (connector, name) {
  this.connector = connector;
  this.id = textHelper.randString(32);
  this.name = name;
  this.polymorphic = false;
  this.steps = [];
  this.isInterrupted = false;
  this.interruptTimer = Components.classes["@mozilla.org/timer;1"]
    .createInstance(Components.interfaces.nsITimer);
  this.nextToRun = null;
  this.stopRunningAt = null;
  this.errorTimeout = 30000 // Library search pages can be slow!
  this.isError = false;
  this.tests = [];
  this.lastRunData = null;

  //used events: onTaskStart, onTaskFinish, onStepFinish, onMessageLog
  this._eventList = new EventList();
  var context = this;
  this.addHandler("onTaskStart", function () {context.initRunData();});
  this.clearData();
};

Task.prototype.clearData = function () {
  //simply rebuild with empty objects, point back to the original session
  this.rebuildData({input: {}, output: {}, temp: {}, 
      session: this.connector.data.session});
};

Task.prototype.rebuildData = function (obj) {
  //refer to other, we want this to break if props are missing
  this.data = {input: obj.input, output: obj.output, temp: obj.temp};
  //the new session space becomes principal
  this.connector.data.session = obj.session;
  //refer to connector-level containers
  for (let key in this.connector.data) {
    this.data[key] = this.connector.data[key];
  }
  //be backwards compatible
  this.input = this.data.input;
  this.output = this.data.output;
};

Task.prototype.getName = function (doc) {
  return this.name;
};

Task.prototype.addHandler = function (name, callback, userData) {
  this._eventList.attachEvent(name, callback, userData);
};

Task.prototype.removeHandler = function (name, callback) {
  this._eventList.detachEvent(name, callback);
};

// conf in/out

Task.prototype.loadConf = function (taskNode) {
  this.clearItems();
  for (var j=0; j<taskNode.childNodes.length; j++) {
    let node = taskNode.childNodes[j];
    switch (node.nodeType) {
    // element node
    case 1:
      switch (node.localName) {
      // TODO: we ignore test nodes here but later should give them a separate
      // container from the execution list
      case "step":
        let step = this.loadStep(node.getAttribute("name"));
        if (node.getAttribute("alt") === "yes") step.isAlt = true;
        if (node.getAttribute("disabled") === "yes") step.isDisabled = true;
        step.loadConf(node);
        this.attachStep(step);
        this.steps.push(step);
        break;
      case "block":
        break;
      }
      break;
    // text node
    case 3:
      break;
    // comment node
    case 8:
      this.addComment(node.nodeValue);
      break;
    }
  }

  //test
  var testNodes = taskNode.getElementsByTagName("test");
  this.loadTestsFromConf(testNodes);
  var poly = taskNode.getAttribute("polymorphic");
  if (poly && poly == "false" )
    this.polymorphic = false;
  else
    this.polymorphic = true;
  // this way around, loading old connectors that do not have the
  // flag set, we default to polymorphic, which was the old way.
};

Task.prototype.loadTestsFromConf = function (testNodes) {
  for (var i=0; i<testNodes.length; i++) {
    var testName = testNodes[i].getAttribute("name");
    var test = this.createTest(testName);
    test.loadConf(testNodes[i]);
  }
  return this.tests;
}

Task.prototype.saveConf = function (taskNode) {
  // Copy the polymorphic flag from the template into the connector
  // This is necessary, because we do not have the template available
  // in the engine.
  var templ = this.getTemplate();
  if ( templ ) {
    var poly = templ.properties.polymorphic;
    if ( poly == undefined )
      poly = false;
    taskNode.setAttribute("polymorphic", poly?"true":"false");
  }

  //steps
  for (var i=0; i<this.steps.length; i++) {
    let step = this.steps[i];
    if (step.getClassName() === "Comment") {
      let commentNode = taskNode.ownerDocument.createComment(step.body);
      taskNode.appendChild(commentNode);
    } else {
      let stepNode = xmlHelper.appendNode(taskNode, "step");
      this.steps[i].saveConf(stepNode);
    }
  }
  //tests
  for (var i=0; i<this.tests.length; i++) {
    var testNode = xmlHelper.appendNode(taskNode, "test", null,
        {"name" : this.tests[i].getName()});
    this.tests[i].saveConf(testNode);
  }
};

// delete item
Task.prototype.deleteItem = function (idx) {
  return this.steps.splice(idx, 1)[0];
};

// delete all items
Task.prototype.clearItems = function () {
  this.steps = [];
};

// delete given an array of indices
Task.prototype.deleteItems = function (indices) {
  // mark for deletion
  for (var i=0; i<indices.length; i++) {
    this.steps[indices[i]].mark = true;
  }
  i = 0;
  while (i<this.steps.length) {
    if (this.steps[i].mark) {
      var step = this.steps.splice(i, 1);
    } else {
      i++;
    }
  }
}

Task.prototype.itemUp = function(s) {
  if (typeof(s) != "number")
    throw new TypeError("array index is not a number");
  else if (s < this.steps.length && s > 0)
    this.steps.splice(s-1, 2, this.steps[s], this.steps[s-1]);
  else
    throw new RangeError("cannot shift array element out of bounds");
};

Task.prototype.itemDown = function(s) {
  if (typeof(s) != "number")
    throw new TypeError("array index is not a number");
  else if (s < this.steps.length-1 && s >= 0)
    this.steps.splice(s, 2, this.steps[s+1], this.steps[s]);
  else
    throw new RangeError("cannot shift array element out of bounds");
};

// insert item at index, or at the end if index is
// invalid or missing. return position of inserted step.
Task.prototype.insertItem = function (item, index) {
  if (index === undefined || index < 0 || index > this.steps.length) {
    return this.steps.push(item) - 1;
  } else {
    this.steps.splice(index + 1, 0, item);
    return index + 1;
  }
};

Task.prototype.addComment = function (text, index) {
  let comment = new Comment(text || "");
  return this.insertItem(comment, index)
};

//loads new step instance, attaches it to this task
//and returns it's position in the task
Task.prototype.addStep = function (step_name, index) {
  var step = this.loadStep(step_name);
  this.attachStep(step);
  return this.insertItem(step, index);
};

// attaches step instance to the task
Task.prototype.attachStep = function (step) {
  if (typeof step.isAlt == "undefined") {
    step.isAlt = false;
  }
  step.task = this;
  step.init(this);
};

// loads step instance given the file name
Task.prototype.loadStep = function (step_name) {
  var step;
  try {
    Components.utils.import('resource://indexdata/steps/'
      + step_name + '/' + step_name + '.js');
    step = eval("new " + textHelper.usToCC(step_name) + "()");
  } catch (e) {
    if ( e.name == "NS_ERROR_FILE_NOT_FOUND" )
        throw new Error ( "Required step " +
        "'" + step_name + "' missing." );
    // Could be something like SyntaxError etc
    throw new Error ( "Problem loading step " +
        "'" + step_name + "':  " +
        e.name + ": " + e.message +" " +
        e.fileName + ":" + e.lineNumber );
  }

  return step;
};

Task.prototype.seekRelative = function (steps) {
  if (typeof(steps)!=="number")
     this._error("must provide a number of steps to seek.");
  if (steps!==parseInt(steps))
     this._error("seek distance must be an integer.");
  if (this.nextToRun != null && (this.nextToRun + steps) >= 0 && (this.nextToRun + steps) <= this.stopRunningAt )
    this.nextToRun+=steps;
  else
    this._error("attemped to seek to a step out of range.");
};

Task.prototype.seekEnd = function () {
  this.nextToRun = this.steps.length;
};

// interrupt running task, optional callback that gets executed before
// the task is resumed, the callback return value controls whether
// the taks is resumed (true) or not (false)
Task.prototype.interrupt = function (delay, cb) {
  if (this.isInterrupted)
    this.warn("Interrupting an already interrupted task.");
  this.isInterrupted = true;
  var context = this;
  //timer that get's called when we should wake up
  var timerCb = { notify: function (timer) {
    var cbreturn = true;
    if (typeof cb == "function" ) {
      try {
        cbreturn = cb();
      } catch (e) {
        context.handleError(e);
      }
    }
    if (cbreturn !== false)
      context.resume();
  }};
  this.interruptTimer.initWithCallback(timerCb, delay,
      Components.interfaces.nsITimer.TYPE_ONE_SHOT);
  return this.interruptTimer;
};


// run task
Task.prototype.run = function (/*0-indexed*/ from, /*exclusive*/to,
    saveOutput) {
    from = typeof from == "number" ? parseInt(from) : 0;
    to = typeof to == "number" ? parseInt(to) : this.steps.length;
    this.isError = false;
    this.nextToRun = from;
    this.stopRunningAt = to;
    if (this.stopRunningAt == 0) {
      this.warn("No steps to run.");
      this._eventList.dispatchEvent("onTaskFinish", true, this.output);
      return;
    }
    if (this.nextToRun == 0 && this.steps[this.nextToRun].isAlt) {
      var errMsg = "Task cannot be started from an alternative step.";
      this._error(errMsg);
      this._eventList.dispatchEvent("onTaskFinish", false, errMsg, 0);
      return;
    }

    // Run pre task calls
    this._eventList.dispatchEvent("onTaskStart", true);

    this.info("Task started.");
    this.resume();
};


Task.prototype.findStepIndex = function (step) {
  for (let i=0; i<this.steps.length; i++) {
    //compare references
    if (step === this.steps[i]) return i;
  }
  return -1;
};

// Log error, tell engine, etc.
// Throws the same error again, unless alt step next
Task.prototype.handleError = function (e) {
  var errMsg = "Error in step '"
      + this.steps[this.nextToRun-1] +"' "
      + "#" + (this.nextToRun-1) + ": ";
  // Defensive coding, we have seen 'e is undefined' !!??
  if ( e && e.message) 
      errMsg +=  e.message ;
  if ( e && e.fileName ) {
      errMsg += "\nin " + e.fileName + ":" + e.lineNumber ;
  }
  // errors
  if (this.nextToRun < this.stopRunningAt
      && this.steps[this.nextToRun].isAlt) {
    this.warn(errMsg);
    this.warn("Step failed, trying alternatives...");
    this.isError = true;
  } else {
    this._error(errMsg);
    this._error("Task failed.");
    this._eventList.dispatchEvent("onTaskFinish", false, errMsg, 
        this.nextToRun-1);
    throw e;
  }
}

Task.prototype.resume = function () {
  this.isInterrupted = false;
  if (this.interruptTimer)
    this.interruptTimer.cancel();
  // run consecutive steps unless callback registered
  while (this.nextToRun < this.stopRunningAt && !this.isInterrupted) {
    //increase before running - in case the callback was quicker
    this.nextToRun += 1;
    let currentStep = this.steps[this.nextToRun-1];
    let nextStep = this.steps[this.nextToRun]
    // skip disabled steps
    if (currentStep.isDisabled) {
      this.info("Step #" + this.nextToRun+" is disabled, ignoring");
      continue;
    }
    // skip comments and other non-runnable objects
    if (!currentStep.run) {
      this.debug("Skipping non-runnable entry at steplist index #" + this.nextToRun);
      continue;
    }
    //sweep through alternatives if no errors
    if (!this.isError && currentStep.isAlt) continue;
        // prepare for errors
    try {
      currentStep.run(this);
      this.isError = false;
      this._eventList.dispatchEvent("onStepFinish",
          this.nextToRun-1, this.stopRunningAt);
    } catch (e) {
      this.handleError(e);  // does not return, throws it again
    }
    // if callback return immediately, could do it once but this is cleaner
    if (this.isInterrupted) return;
  }
  // finally
  if (this.nextToRun == this.stopRunningAt) {
    this.info("Task finished with no errors.");
    this._eventList.dispatchEvent("onTaskFinish", true, this.output);
  }
};


Task.prototype.complete = function () {
  this.nextToRun = this.stopRunningAt;
};

// pretty logging

Task.prototype.log = function (level, message, step) {
  var prefix =  "Task [" + this.name + "] - ";
  var stepInfo = " - ";
  if (step != undefined && "getClassName" in step) {
    stepInfo = "- '" + step.getClassName() + "': ";
  }
  var msg = stepInfo + message;
  logger.log(level, msg, prefix);
  var confLevel = logger.getLevel();
  if (level < confLevel) return;
  this._eventList.dispatchEvent("onMessageLog",
      Level.getName(level),
      prefix + Level.getName(level) + msg);
};
Task.prototype.info = function (message, step) {
  this.log(Level.INFO, message, step);
};
Task.prototype.warn = function (message, step) {
  this.log(Level.WARN, message, step);
};
Task.prototype.debug = function (message, step) {
  this.log(Level.DEBUG, message, step);
};

//this method should not be called directly by a step!
//instead throw a StepError!!
Task.prototype._error = function (message, step) {
  this.log(Level.ERROR, message, step);
};

// Run data
Task.prototype.initRunData = function () {
  this.lastRunData = {};
  this.lastRunData.urls = {};
};

Task.prototype.postprocessRunData = function () {
// perhaps have it calculate some stats here?
};

Task.prototype.addRunDatumUrlEvent = function (url, event) {
  var datum = {
    "event": event,
    "timestamp": new Date().getTime(),
  }
  if (!this.lastRunData.urls[url]) {
    this.lastRunData.urls[url] = {};
    this.lastRunData.urls[url].events = [];
  }
  this.lastRunData.urls[url].events.push(datum);
};

Task.prototype.getRunData = function () {
  return this.lastRunData;
};



// arguments

Task.prototype.getArgNames = function () {
  return this.getTemplate().getArgNames();
};

Task.prototype.getUsedArgs = function () {
  var usedArgs = {};
  var argNames = [];
  for (var i=0; i<this.steps.length; i++) {
    var step = this.steps[i];
    if ((step.getClassName() !== "Comment")) {
      var args = step.getUsedArgs();
      if (args) {
        for (var j=0; j<args.length; j++) {
          usedArgs[args[j]] = true;
        }
      }
    }
  }
  return Object.keys(usedArgs);
};

Task.prototype.setArgValue = function (argName, argValue) {
  this.input[argName] = argValue;
};

Task.prototype.getArgValue = function (argName) {
  return typeof this.input[argName] != "undefined" ? this.input[argName] : "";
};

Task.prototype.isArgUsed = function (argName) {
  return this.getUsedArgs().indexOf(argName)!=-1
};

Task.prototype.detach = function () {
  this.connector.detachTask(this);
};

Task.prototype.compareTo = function (task) {
  //sort first by order loaded, if template
  //available, by name otherwise
  var templateA = this.getTemplate();
  var templateB = task.getTemplate();
  if (templateA && templateB) {
    if (templateA.rank > templateB.rank) return 1;
    if (templateA.rank < templateB.rank) return -1;
  } else {
    if (this.name > task.name) return 1;
    if (this.name < task.name) return -1;
  }

  //then by arg list
  var argCountA = this.getUsedArgs().length;
  var argCountB = task.getUsedArgs().length;
  if (argCountA < argCountB)
    return -1;
  if (argCountA > argCountB)
    return 1;
  // argCountA must be equal to argCountB
  return 0;
};

Task.prototype.createTest = function (name) {
  var test = new TaskUnitTest(this, name);
  this.tests.push(test);
  return test;
};

Task.prototype.createDefaultTests = function () {
  var testNodes = this.getTemplate().testNodes;
  this.loadTestsFromConf(testNodes);
};

Task.prototype.removeTestByIdx = function (idx) {
  return this.tests.splice(idx,1);
};

Task.prototype.getTests = function () {
  return this.tests;
};

Task.prototype.getTemplate = function () {
  var connectorTemplate = this.connector.getTemplate();
  if (connectorTemplate) {
    var taskTemplate = connectorTemplate.getTaskTemplate(this.name);
    if (typeof taskTemplate == 'undefined')
      this._error("Connector template does not contain task '"
          +this.name+"' definition");
    return taskTemplate;
  } else {
    this.warn("Missing connector template.");
    return null;
  }
};
