var EXPORTED_SYMBOLS = ["Extract"];
Components.utils.import('resource://indexdata/runtime/Step.js');
Components.utils.import('resource://indexdata/runtime/StepError.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/xulHelper.js');
Components.utils.import('resource://indexdata/util/jsonPathHelper.js');

if (typeof DEBUG == "undefined") const DEBUG = true;

var Extract = function () {
  this.className = "Extract";
  this.truncate = 80;
  this.conf = {};
  this.conf["node"] = "";
  this.conf["sourceAttribute"] = "";
  this.conf["attributes"] = [];
  this.conf["jsonPath"] = "";
};
Extract.prototype = new Step();
Extract.prototype.constructor = Extract;

Extract.prototype.init = function() {
};

Extract.prototype.drawRadios = function(attributes, values, div, checked) {
  let htmlns = "http://www.w3.org/1999/xhtml";
  let doc = div.ownerDocument;
  xmlHelper.emptyChildren(div);

  let a = {};
  a.type = "radio";
  a.name = "attr";
  a.value = "textContent";
  if (a.value === checked) a.checked = true;
  let text = doc.createTextNode(" text" + (values ? values[0] : ""));
  xmlHelper.appendNode(div, "input", null, a, htmlns);
  div.appendChild(text);
  xmlHelper.appendNode(div, "br", null, null, htmlns);

  for (let i = 0; i < attributes.length; i++) {
    let a = {};
    a.type = "radio";
    a.name = "attr";
    a.value = attributes[i];
    if (a.value === checked) a.checked = true;
    let text = doc.createTextNode(" " + attributes[i] + (values ? values[i+1] : ""));
    xmlHelper.appendNode(div, "input", null, a, htmlns);
    div.appendChild(text);
    xmlHelper.appendNode(div, "br", null, null, htmlns);
  }
}

Extract.prototype.draw = function(surface) {
  var context = this;
  var htmlns = "http://www.w3.org/1999/xhtml";
  var attrRadioDiv = {};

  // New node, we can populate attrRadioDiv with attributes and values
  var onNodeSelect = function(node) {
    if (!node) return; // This can happen if the user edits the XPath by hand
    if ( node && node.cf_datanode ) node = node.cf_datanode;
    // Default to text content
    context.conf["sourceAttribute"] = "textContent";
    var attributes = [];
    var values = [" = " + node.textContent.substring(0, context.truncate)];
    for (i = 0; i < node.attributes.length; i++) {
      let attribute = node.attributes.item(i);
      attributes.push(attribute.name);
      values.push(" = " + node.attributes.item(i).value.substring(0, context.truncate));
    }
    // TODO: Add an option to get the inner html/xml of the node
    context.conf.attributes = attributes;
    context.drawRadios(attributes, values, attrRadioDiv, "textContent");
  };

  // XPath field
  let hbox = xmlHelper.appendNode(surface, "hbox", null, {align: "center"}, null);
  xmlHelper.appendNode(hbox, "caption", "Source element: ", null, null);
  var input = xulHelper.singleNodeField(hbox, this, "node", onNodeSelect);

  xulHelper.checkbox(surface, this, "extractAll",
      "Process everything: Extract target attribute from every matching node; complete XPath functions.");

  // JSONPath field
  xulHelper.jsonPathField(surface, this, this.conf, "jsonPath", "Container: ",
    {path:"$.output", key:"hits", append:jsonPathHelper.REPLACE});

  // Attributes
  var attrGroup = xmlHelper.appendNode(surface, "groupbox", null, {flex:"2", pack:"top"}, null);
  xmlHelper.appendNode(attrGroup, "caption", null, {label:"Attributes"}, null);
  var form = xmlHelper.appendNode(surface, "form", null, {name:"cf_extra_radios"}, htmlns);
  var attrRadioDiv = xmlHelper.appendNode(form, "div", null, {style:"-moz-column-width: 20em;"}, htmlns);


  // Stored node, values are likely different now and who knows what
  // page it came from, but attribute names were saved for posterity.
  if (this.conf.node && this.conf.attributes) {
    this.drawRadios(this.conf.attributes, null, attrRadioDiv, this.conf.sourceAttribute);
  }

  attrRadioDiv.addEventListener("change", function(e) {
    context.conf["sourceAttribute"] = e.target.value;
  }, false);


  input.addEventListener("change", function(e) {
    // Refresh display of attributes when the XPath is edited by hand
    // provided there is no inline replacement going on
    if (input.value.indexOf("{$") > -1) {
      onNodeSelect(xmlHelper.getElementByXpath(context.getPageDoc(), input.value));
    }
  }, false);

  xmlHelper.appendNode(surface, "spacer", null, {flex: 1}, null);
};

Extract.prototype.run = function (task) {
  let sourceAttr = this.conf.sourceAttribute
  if (!sourceAttr) sourceAttr = "textContent";

  let doc = this.getPageDoc();
  let nodeSpec = jsonPathHelper.inlineReplaceNodeSpec(this.conf.node, task.data);

  // If extractAll is set, process whole document instead of short-circuiting
  // Since the XPathResult object isn't exposed in the extension context, we'll
  // hardcode the result type values and provide comments with the constant's name.
  if (this.conf.extractAll) {
    xmlHelper.checkNodeSpec(nodeSpec);
    let result = xmlHelper.evaluateNodeSpec(doc, nodeSpec, 0); // 0 -> ANY_TYPE

    switch (result.resultType) {
    case 1: // NUMBER_TYPE
      var extracted = [""+result.numberValue];  // force to a string
      break;
    case 2: // STRING_TYPE
      var extracted = [result.stringValue];
      break;
    case 3: // BOOLEAN_TYPE
      var extracted = [""+result.booleanValue]; // force to a string
      break;
    case 4: // UNORDERED_NODE_ITERATOR_TYPE
      var extracted = [];
      let cur = result.iterateNext();
      while (cur) {
        if (sourceAttr === "textContent") {
          var value = cur.textContent;
        } else {
          var value = cur.getAttribute(sourceAttr);
        }
        if ((value !== "") && (value !== null)) {
          extracted.push(value);
        }
        cur = result.iterateNext();
      }
      break;
    default:
      throw new StepError("unexpected result extracting all from " +
         JSON.stringify(nodeSpec));
    }

    if ((typeof(extracted) === "object") && (extracted !== [])) {
      jsonPathHelper.set(this.conf.jsonPath, extracted, task.data);
    }
  } else { // regular one-node thing
    let node = xmlHelper.getElementByNodeSpec(doc, nodeSpec);
    if (typeof(node) !== "object") {
      throw new StepError("cannot retrieve node from  " +
          JSON.stringify(nodeSpec));
    }

    if (sourceAttr === "textContent") {
      var value = node.textContent;
    } else {
      var value = node.getAttribute(sourceAttr);
    }

    jsonPathHelper.set(this.conf.jsonPath, [value], task.data);

    // If in builder, highlight match
    let xulAppInfo = Components.classes["@mozilla.org/xre/app-info;1"].getService(Components.interfaces.nsIXULAppInfo);
    if (xulAppInfo.name !=="ConnectorFramework") {
      xulHelper.unhighlightAll(doc,true);
      xulHelper.highlightNode(node,"yellow",true);
    }
  }
};

Extract.prototype.getClassName = function () {
  return "Extract";
};

Extract.prototype.getDisplayName = function () {
  return "Extract value";
};

Extract.prototype.getDescription = function () {
  return "Populates a result with text from an element text or attribute.";
};

Extract.prototype.getVersion = function () {
  return "3.0";
};

Extract.prototype.upgrade = function (confVer, curVer, conf) {
  // can't upgrade if the connector is newer than the step
  if (confVer > curVer)
    return false;
  if (confVer < 0.3) {
    conf.jsonPath = jsonPathHelper.specFromArray(["output", conf.result]);
    delete conf.result;
  }
  return true;
};

Extract.prototype.renderArgs = function () {
  if (!this.conf.node.xpath) return "";
  return this.conf.node.xpath;
}
