/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Mozilla Corporation.
 * Portions created by the Initial Developer are Copyright (C) 20007
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Pelle Johnsen <pjohnsen@mozilla.com>
 *   Dave Camp <dcamp@mozilla.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "cf_config.h"
#include "xpcom-config.h"
#include "mozilla-config.h"
#ifdef XPCOM_GLUE
#include "nsXPCOMGlue.h"
#endif

#include "EmbeddingSetup.h"

// CRT headers
#include <iostream>
#include <string>
using namespace std;

#include "nsXULAppAPI.h"

#include "nsCOMPtr.h"
#include "nsStringAPI.h"
#include "nsILocalFile.h"

#include "nsDirectoryServiceDefs.h"
#include "nsAppDirectoryServiceDefs.h"
#include "nsProfileDirServiceProvider.h"

#ifdef WIN32
  //TODO: make this file fully X platform
#  include <windows.h>
#  undef MAX_PATH
#  define MAX_PATH _MAX_PATH
#else
#  include <limits.h>
#  include <unistd.h>
#  include <string.h>
#  define MAX_PATH PATH_MAX
#endif

static int gInitCount = 0;

#include "nsIDirectoryService.h"
#include "nsAppDirectoryServiceDefs.h"
static nsIDirectoryServiceProvider *sAppFileLocProvider = nsnull;
static nsCOMPtr<CF_nsINativeFile> sProfileDir = nsnull;
static nsCOMPtr<CF_nsINativeFile> xuldir = nsnull;
static nsCOMPtr<CF_nsINativeFile> appdir = nsnull;
static CF_Logger *static_logger = 0;

class MozEmbedDirectoryProvider : public nsIDirectoryServiceProvider2
{
public:
  NS_DECL_ISUPPORTS_INHERITED
  NS_DECL_NSIDIRECTORYSERVICEPROVIDER
  NS_DECL_NSIDIRECTORYSERVICEPROVIDER2
};

static MozEmbedDirectoryProvider *kDirectoryProvider = nsnull;

NS_IMPL_QUERY_INTERFACE2(MozEmbedDirectoryProvider,
                         nsIDirectoryServiceProvider,
                         nsIDirectoryServiceProvider2)

NS_IMETHODIMP_(nsrefcnt)
MozEmbedDirectoryProvider::AddRef()
{
    return 1;
}

NS_IMETHODIMP_(nsrefcnt)
MozEmbedDirectoryProvider::Release()
{
    return 1;
}

NS_IMETHODIMP
MozEmbedDirectoryProvider::GetFile(const char *aKey, CF_BOOL *aPersist,
                                   nsIFile* *aResult)
{
    if (sAppFileLocProvider)
    {
        nsresult rv = sAppFileLocProvider->GetFile(aKey, aPersist,
                                                   aResult);
        if (NS_SUCCEEDED(rv))
            return rv;
    }
    if (xuldir && !strcmp(aKey, NS_GRE_DIR))
    {
        *aPersist = PR_TRUE;
        return xuldir->Clone(aResult);
    }

    if (sProfileDir && !strcmp(aKey, NS_APP_USER_PROFILE_50_DIR))
    {
        *aPersist = PR_TRUE;
        return sProfileDir->Clone(aResult);
    }

    if (sProfileDir && !strcmp(aKey, NS_APP_PROFILE_DIR_STARTUP))
    {
        *aPersist = PR_TRUE;
        return sProfileDir->Clone(aResult);
    }

    if (sProfileDir && !strcmp(aKey, NS_APP_CACHE_PARENT_DIR))
    {
        *aPersist = PR_TRUE;
        return sProfileDir->Clone(aResult);
    }
    return NS_ERROR_FAILURE;
}

NS_IMETHODIMP
MozEmbedDirectoryProvider::GetFiles(const char *aKey,
                                    nsISimpleEnumerator* *aResult)
{
    nsCOMPtr<nsIDirectoryServiceProvider2>
        dp2(do_QueryInterface(sAppFileLocProvider));

    if (!dp2)
        return NS_ERROR_FAILURE;

    return dp2->GetFiles(aKey, aResult);
}


/* Initialize Embedding.

Links:
https://developer.mozilla.org/en/xpcom_glue
https://developer.mozilla.org/en/GRE
*/

#ifndef XPCOM_GLUE
// if XPCOM_GLUE is disabled, you can choose to use NS_InitXPCOM2
#define USE_INITXPCOM2 0
#endif

#ifdef XPCOM_GLUE
XRE_InitEmbedding2Type XRE_InitEmbedding2 = 0;
XRE_TermEmbeddingType XRE_TermEmbedding = 0;
#endif

nsresult InitEmbedding(const char *aAppPath, const char *aProfileDir,
                       CF_Logger *logger)
{
    nsresult rv;

    ++gInitCount;
    if (gInitCount > 1)
        return NS_OK;

    static_logger = logger;
// From https://developer.mozilla.org/en/GRE
//  "Support for locating a standalone glue was removed in
//   Gecko 6.0 (Firefox 6.0 / Thunderbird 6.0 / SeaMonkey 2.3)."
//
//       This means that GRE_GetGREPathWithProperties is no longer defined.
// unfortunately we still need to pass xuldir to XRE_InitEmbedding2 below.
    const char *gre_home = getenv("GRE_HOME");
#ifdef DEFAULT_GRE_HOME
    if (!gre_home)
        gre_home = DEFAULT_GRE_HOME;
#endif
    if (!gre_home)
    {
        logger->printf("engine", "You must set GRE_HOME to "
                       "the location of libxul.so");
        return 1;
    }
    std::string xpcomDir(gre_home);
    logger->printf("engine", "xpcom dir: %s", xpcomDir.c_str());
    logger->printf("engine", "profile dir: %s", aProfileDir ? aProfileDir : "null");


#ifdef XPCOM_GLUE
    std::string xpcom_path(gre_home);
    xpcom_path += "/libxpcom.so";
    rv = XPCOMGlueStartup(xpcom_path.c_str());
    if (NS_FAILED(rv))
    {
        logger->printf("engine", "Failed to starting XPCOM glue (%s)",
                       xpcom_path.c_str());
        return 4;
    }
    const struct nsDynamicFunctionLoad kXULFuncs[] = {
        {"XRE_InitEmbedding2", (NSFuncPtr*) &XRE_InitEmbedding2},
        {"XRE_TermEmbedding", (NSFuncPtr*) &XRE_TermEmbedding},
        {nsnull, nsnull },
    };
    rv = XPCOMGlueLoadXULFunctions(kXULFuncs);
    if (NS_FAILED(rv))
    {
        logger->printf("engine", "Failed to locate XPCOM initialization functions");
        return 5;
    }
#endif

    rv = NS_NewNativeLocalFile(nsCString(xpcomDir.c_str()), PR_FALSE,
                               getter_AddRefs(xuldir));
    if (NS_FAILED(rv))
    {
        logger->printf("engine", "Unable to create nsILocalFile for xuldir ");
        return 6;
    }

    char self[MAX_PATH];
    if (!aAppPath)
    {
        // if app path is not given, use current working directory
        aAppPath = getcwd(self, sizeof(self)-1);
    }
    string selfPath;
    if (aAppPath)
        selfPath = aAppPath;

    rv = NS_NewNativeLocalFile(nsCString(selfPath.c_str()), PR_FALSE,
                               getter_AddRefs(appdir));
    if (NS_FAILED(rv))
    {
        logger->printf("engine", "Unable to create nsILocalFile for appdir");
        return 8;
    }

    // setup profile dir
    if (aProfileDir && *aProfileDir)
    {
        rv = NS_NewNativeLocalFile(nsCString(aProfileDir), PR_FALSE,
                                   getter_AddRefs(sProfileDir));
        NS_ENSURE_SUCCESS(rv, rv);
    }

    if (!kDirectoryProvider)
        kDirectoryProvider = new MozEmbedDirectoryProvider;

#if USE_INITXPCOM2
    rv = NS_InitXPCOM2(nsnull, appdir, kDirectoryProvider);
    if (NS_FAILED(rv))
    {
        logger->printf("engine", "XRE_InitXPCOM2 failed rv=%08x", rv);
        return 9;
    }
#else
    rv = XRE_InitEmbedding2(xuldir, appdir, kDirectoryProvider);
    if (NS_FAILED(rv))
    {
        logger->printf("engine", "XRE_InitEmbedding failed rv=%08x", rv);
        return 9;
    }
#ifndef XPCOM_GLUE
    XRE_NotifyProfile();
#endif
#endif
    return NS_OK;
}

nsresult TermEmbedding(CF_Logger *logger)
{
    --gInitCount;
    if (gInitCount > 0)
        return NS_OK;

#if USE_INITXPCOM2
    NS_ShutdownXPCOM(nsnull);
#else
#ifdef XPCOM_GLUE
    if (XRE_TermEmbedding)
        XRE_TermEmbedding();
#else
    XRE_TermEmbedding();
#endif
#endif
    if (sProfileDir)
    {
        sProfileDir->Remove(PR_TRUE);

        // make sure this is freed before shutting down xpcom
        sProfileDir = nsnull;
    }

    logger->printf("engine", "TermEmbedding completed");
    return NS_OK;
}

#if MOZ_MAJOR <= 12
#include "jemalloc.h"

extern "C" {

    extern size_t je_malloc_usable_size_in_advance(size_t size)
        NS_VISIBILITY_DEFAULT;
    size_t je_malloc_usable_size_in_advance(size_t size)
    {
        if (size) {
            return ((size - 1) / 8 + 1) * 8;
        }
        return 8;
    }
    extern void jemalloc_stats(jemalloc_stats_t *stats)
        NS_VISIBILITY_DEFAULT;
    void jemalloc_stats(jemalloc_stats_t *stats)
    {
        if (stats) {
            memset(stats, 0, sizeof(stats));
        }
    }
}
#endif

/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */
