/* This file is part of the Connector Framework
 * Copyright (C) 2008-2012 Index Data
 * See the file LICENSE for details.
 */

/** \file
 * \brief JSON to XML conversion
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <yaz/xmalloc.h>
#include <yaz/yaz-iconv.h>

#include "json.h"

static void open_tag(WRBUF wrb, const char *name)
{
    wrbuf_puts(wrb, "<");
    wrbuf_puts(wrb, name);
    wrbuf_puts(wrb, ">");
}

static void close_tag(WRBUF wrb, const char *name)
{
    wrbuf_puts(wrb, "</");
    wrbuf_puts(wrb, name);
    wrbuf_puts(wrb, ">");
}

void make_xml_al(struct json_node *n, WRBUF wrb, struct json_node *parent)
{
    if (!n)
        return;
    if (n->type != json_node_list && n->type != json_node_array)
    {
        if (parent)
            open_tag(wrb, parent->u.string);
    }
    switch (n->type)
    {
    case json_node_object:
        make_xml_al(n->u.link[0], wrb, 0);
        break;
    case json_node_array:
        make_xml_al(n->u.link[0], wrb, parent);
        break;
    case json_node_list:
        for (; n; n = n->u.link[1])
            make_xml_al(n->u.link[0], wrb, parent);
        break;
    case json_node_pair:
        make_xml_al(n->u.link[1], wrb, n->u.link[0]);
        break;
    case json_node_string:
        wrbuf_xmlputs(wrb, n->u.string);
        break;
    case json_node_number:
        wrbuf_printf(wrb, "%lg", n->u.number);
        break;
    case json_node_true:
        wrbuf_puts(wrb, "true");
        break;
    case json_node_false:
        wrbuf_puts(wrb, "false");
        break;
    case json_node_null:
        wrbuf_puts(wrb, "null");
        break;
    }
    if (n && n->type != json_node_list && n->type != json_node_array)
    {
        if (parent)
            close_tag(wrb, parent->u.string);
    }
}

/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */
