var EXPORTED_SYMBOLS = ["ParseXpattern"];

// parse a page using the XPattern

/* TODO:

- Why-not button: Point to a node, and modify the pattern so that
  it matches somewhere too.
- Patterns that contain ()'s and or-operators? Better editor for or-bags!
- Negations in the designer

- Low-intensity highlights for other hits on the screen, while editing
  and/or designing.

- UI improvements
  - Make each tab have its own scroll bars, if needed, instead of one
    for them all - edit/reformat button falls off the screen if too much
    design or history...
*/

Components.utils.import('resource://indexdata/runtime/Step.js');
Components.utils.import('resource://indexdata/runtime/StepError.js');
Components.utils.import('resource://indexdata/runtime/Task.js');
Components.utils.import('resource://indexdata/util/xpattern.js');
Components.utils.import('resource://indexdata/util/xpatternText.js');
Components.utils.import('resource://indexdata/util/xpatternMaker.js');
Components.utils.import('resource://indexdata/util/xulHelper.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/textHelper.js');
Components.utils.import('resource://indexdata/util/inspector.js');
Components.utils.import('resource://indexdata/util/jsonPathHelper.js');
Components.utils.import('resource://indexdata/ui/app.js');
Components.utils.import('resource://indexdata/util/logging.js');

if (typeof DEBUG == "undefined") const DEBUG = true;
var logger = logging.getLogger();

var ParseXpattern = function () {
  this.className = "ParseXpattern";
  this.conf = {};
  this.conf["xpattern"] = "";
  this.conf["hitarea"] = xmlHelper.nodeSpecFromString("html/body");
  this.conf["jsonPath"] = { path: "$.output", key:"results" };
  this.conf["highlighthits"] = true;
  this.conf["autogenerate"] = true; // generate patter at every click
  this.conf["hitnumbercheck"] = false; // check the hitnumbers
  this.conf["failhitnumber"] = true; // fail step if missing hitnumbers
  this.conf["removehitnumber"] = true; // do not pass them on to the system
  this.conf["keephitnumber"] = true; // remember them from page to page
  this.conf["hitnumbervar"] = { path: "$.session", key:"hitnumber" };
  //this.cpatt, this.epatt - the finput fields for xpattern
};
ParseXpattern.prototype = new Step();
ParseXpattern.prototype.constructor = ParseXpattern;

ParseXpattern.prototype.init = function() {
};


// highlight colors - just something that is clearly visible
// will try to pick different colors every time, as far as possible
var highlightcolors = [ "Yellow", "Aquamarine ", "DarkSalmon",
  "GreenYellow", "LightSkyBlue", "YellowGreen", "HotPink", "gold" ];
var nexthighlightcolor = 0;

var gethighlightcolor = function() {
  var col = highlightcolors[nexthighlightcolor];
  nexthighlightcolor = (nexthighlightcolor +1 ) % highlightcolors.length;
  return col;
};

ParseXpattern.prototype.draw = function(surface, win) {
  var context = this;
  this.inBuilder = true; // pass a signal to run() to do highlights

  this.tabs = xulHelper.tabBox( surface,
      [ "Design", "Edit","History", "Options", "Hitnumber check" ],
      { flex: 1 });

  this.cpatt = this.drawCreateTab( this.tabs[0], win, this);
  this.epatt = this.drawEditTab(   this.tabs[1], this);
  this.htab = this.drawHistoryTab( this.tabs[2], this);
  this.otab = this.drawOptionTab(  this.tabs[3], this);
  this.checktab = this.drawCheckTab(  this.tabs[4], this);
  
  // Keep the inputs in sync
  // (if value not changed, don't assign, to prevent event cascades)
  this.cpatt.addEventListener("change", function(e) {
    logger.debug("cpatt change: " + context.cpatt.value);
    context.setPattern(context.cpatt.value);
  }, false);
  this.epatt.addEventListener("command", function(e) {
    logger.debug("epatt command: " + context.epatt.value);
    context.setPattern(context.epatt.value);
  }, false);
  this.epatt.addEventListener("change", function(e) {
    logger.debug("epatt change: " + context.epatt.value );
    context.setPattern(context.epatt.value);
  }, false);
}; // draw

// Set the XPattern. Updates all places it is displayed, as
// well as the xpattern in conf, and the history.
// (carefully not reassigning identical values, to avoid
// event cascades)
ParseXpattern.prototype.setPattern = function(patt) {
  //logger.debug("setPattern: p=" + patt + " x=" + this.conf.xpattern +
  //  " cl.tc=" + this.cpatt.textContent + " e= " + this.epatt.value );
  if ( this.conf.xpattern != patt ) {
    logger.debug("Updating xpattern in conf: " + patt );
    this.conf.xpattern = patt;
  }
  if ( this.cpatt && this.cpatt.textContent != patt ) {
    logger.debug("Updating cpatt label: " + patt );
    this.cpatt.textContent = patt;
  }
  if ( this.epatt && this.epatt.value != patt ) {
    logger.debug("Updating epatt display: " + patt );
    this.epatt.value = patt;
  }
  var trimpat = function(p) { // trim patterns to ignore small changes
    //logger.debug("trim: before: '" + p + "'" );
    p = p.replace(/^\s+/,""); // leadign whitespace
    p = p.replace(/\s+$/,""); // trailing whitespace
    p = p.replace(/\s+/g," "); // any whitespace
    //logger.debug("trim: after:  '" + p + "'" );
    return p;
  }
  var oldhist = this.conf.xpatternhistory;
  if (!oldhist) // can happen with a brand new connector
    oldhist = []; // defensive coding, anyway.
  var hist = [ patt ];
  for ( var i = 0; (i < oldhist.length) && (hist.length<20); i++) {
    //logger.debug("Looking at oldhist " + i + ": " + oldhist[i] );
    var good = true;
    for ( var j = 0; j <hist.length; j++ )
      if ( trimpat(hist[j]) == trimpat(oldhist[i]) ) {
        good = false;
        break;
      }
    if ( good ) {
      hist.push (oldhist[i]);
      //logger.debug("hist : " + oldhist[i] );
    }
  }
  this.conf.xpatternhistory = hist;
  this.drawHistoryTab( this.tabs[2], this);
}; // setPattern


// Draw the options tab
ParseXpattern.prototype.drawOptionTab = function(surface, context) {
  //dump("DrawOptionTab Start \n");
  var vbox = xmlHelper.appendNode(surface, "vbox", null, {flex:1}, null);
  var inpbox = xmlHelper.appendNode(vbox, "hbox", null, null, null);
  var hitarea = xulHelper.xpathField( inpbox, this, "hitarea",
          "Area where hits are to be found" );
  var hlbox = xmlHelper.appendNode(vbox, "hbox", null, null, null);
  var highlightcheck = xulHelper.checkbox( hlbox, this,
          "highlighthits", "Highlight hits on the page" );
  highlightcheck.addEventListener("command", function(e) {
    var hitarea = context.conf['hitarea'];
    var pageDoc = context.getPageDoc();
    var node = xmlHelper.getElementByNodeSpec(pageDoc, hitarea);
    if (node != null) {
        xulHelper.unhighlightAll(node,false);
    }  // Ok to unhighlight both on check and uncheck
  }, false);
  var agbox = xmlHelper.appendNode(vbox, "hbox", null, null, null);
  var autogencheck = xulHelper.checkbox( agbox, this,
          "autogenerate", "Autogenerate the pattern when " +
          "changing anything in the designer");
  var hbox = xmlHelper.appendNode(vbox, "hbox", null, null, null);
  xulHelper.jsonPathField(hbox, this, this.conf, "jsonPath",
      "Where to put the results: ",
      {path:"$.output", key:"results", append:jsonPathHelper.REPLACE});
}; // drawOptionTab

// Tab to control the hitnumber sanity check
ParseXpattern.prototype.drawCheckTab = function(surface, context) {
  var vbox = xmlHelper.appendNode(surface, "vbox", null, {flex:1}, null);
  xulHelper.checkbox( vbox, this, "hitnumbercheck",
             "Check that hitnumbers are continuous. " +
             "(Your pattern must extract hitnumbers from the page");
  xulHelper.checkbox( vbox, this, "failhitnumber",
                      "Fail the step if detecting a missing hitnumber." +
                      "(otherwise just logs an error)" );
  xulHelper.checkbox( vbox, this, "removehitnumber",
             "Remove the hitnumber from the results after checking");
  xulHelper.checkbox( vbox, this, "keephitnumber",
             "Keep the hit number for the next page");
  xulHelper.jsonPathField(vbox, this, this.conf, "hitnumbervar",
                          "Where to keep the hit number ",
                          {path:"$.session", key:"hitnumber"});
}; // drawCheckTab 
  
ParseXpattern.prototype.drawEditTab = function(surface, context) {
  //dump("drawEdittab start \n");
  var vbox = xmlHelper.appendNode(surface, "vbox", null,
      {"flex":1,"align":"stretch"}, null);
  var hbox = xmlHelper.appendNode(vbox, "hbox", null,
      {"flex":1,"align":"stretch"}, null);
  var xpatt = xulHelper.inputField( hbox, this, "xpattern", "XPattern", 120,
            { "multiline": true, "rows": 30,
              "flex": 1, "width" : 580, "align":"stretch" } );
  var hbox2 = xmlHelper.appendNode(vbox, "hbox", null, null, null);
  var formatButton = xmlHelper.appendNode(hbox2, "button",
        null, {"label": "Reformat", "width" : 120 }, null);
  var errbox = xmlHelper.appendNode(hbox2, "hbox", null, { "flex":1}, null);

  // callback function to check the xpattern syntax
  var checksyntax = function (ev) {
    var patt = xpatt.value;
    xmlHelper.emptyChildren(errbox);
    try {
      if (patt) {
        var par = new XpatternTextParser(patt);
        var xp = par.parse();
        xpatt.removeAttribute("class");
        formatButton.setAttribute("disabled",false);
      }
    } catch (e) {
      var lasterrpos = par.getErrorPos();
      var line = 1;
      var col = 1;
      var p = 0;
      while ( ++p < lasterrpos ) {
          if ( patt.substring( p, p+1) == "\n" ) {
              line ++;
              col = 1;
          } else
              col ++;
      }
      //xulHelper.captionField(errbox,"" + e );
      xulHelper.labelField(errbox, "" + line + ":" + col +":" );
      var errcapt = xulHelper.captionField(errbox,  par.getErrorMsg());
      errbox.addEventListener("click", function(e){
          xpatt.selectionStart = lasterrpos;  // positions the caret
          xpatt.selectionEnd = lasterrpos;
      }, false );
      errcapt.setAttribute("tooltipText",
            "Click to position the cursor at the error" );
      xpatt.setAttribute("class", "incomplete");
      formatButton.setAttribute("disabled",true);
    }
  }; // checksyntax

  xpatt.addEventListener("input", checksyntax, false);
  xpatt.addEventListener("change", checksyntax, false);
  checksyntax();
  
  // Reformat the xpattern
  // or the selected part of it
  function reformat(e) {
    try {
      var patt = xpatt.value;
      var offsetcorrection = 0;
      var before = "";
      var after = "";
      //dump("reformat: sel= " + xpatt.selectionStart + " - " + xpatt.selectionEnd + "\n" );
      var selstart = xpatt.selectionStart;
      var selend = xpatt.selectionEnd;
      var indent = 0;
      if ( selstart != selend ) {
        offsetcorrection = selstart;
        before = patt.substring(0, xpatt.selectionStart);
        after = patt.substring(xpatt.selectionEnd);
        patt = patt.substring(xpatt.selectionStart, xpatt.selectionEnd);
        // check current indentation
        var p = selstart-1;
        while ( (p >= 0) && xpatt.value.charAt(p) == " ") {
          p--;
          indent ++;
          //dump("counting indent " + indent + " at " + p +
          //     "=" + xpatt.value.charAt(p) + "\n");
        }
        indent = Math.floor(indent / 2);
        //dump("sel: indent = " + indent + "\n" +
        //     "B='" + before + "'\n" +
        //     "P='" + patt + "'\n" +
        //     "A='" + after + "'\n");
      }
      var oldpatt = patt;
      var par = new XpatternTextParser(patt);
      var xp = par.parse();
      //dump("got offset " + xp.getStringoffset() + " for root \n");
      var s;
      if ( patt.indexOf("\n") == -1 ) {
        s = xp.dumpString(indent); // structured dump
        s = s.replace(/^ +/,""); // remove trailing space
      } else {
        s = xp.dumpString(-1); // one-liner
      }
      if ( s.charAt(s.length-1) == "\n" )
        s = s.slice(0,-1); // remove trailing newline
      xpatt.value = before + s + after;
      var newend = selstart + s.length;
      if ( selstart != selend ) {
        // Reset the selection
        xpatt.selectionStart = selstart;
        xpatt.selectionEnd = selstart + s.length;
      } else { // no selection
        // whole pattern reformatted, place cursor to the beginning
        xpatt.selectionStart = 0;
        xpatt.selectionEnd = 0;
      }
      //dump("After reformat, setting cpatt. old= " +
      //    context.cpatt.value + " to " + xpatt.value + "\n");
      // Pass value to create tab too, and history
      context.setPattern(context.epatt.value);
      xpatt.focus(); // focus back to the input, to show selection
    } catch (e) {
      dump("Caught an exception " + e + "\n");
    }
  };  // reformat
  formatButton.addEventListener("command", reformat, false );

  // Helper to highlight the DOM node on the page that matches
  // the xpattern node that corresponds to the given position
  // on the xpattern string.
  function highlightDomNode( pattern, position ) {
    try {
      var hitarea = context.conf['hitarea'];
      var pageDoc = context.getPageDoc();
      var node = xmlHelper.getElementByNodeSpec(pageDoc, hitarea);
      if (node == null)
          return; // Can't do nothing
      xulHelper.unhighlightAll(node,false);
      var par = new XpatternTextParser(pattern);
      par.setOffsetCorrection(0); // tell the parser we want offsets!
      var xp = par.parse();
      xp.askForNodes();
      var keynode = null;
      xp.foreachnode( function(n) {
        n.setVariable(""); // Clear all variables
        if ( n.getStringoffset() == position ) {
          keynode = n; // and remember the *last* matching node
          // Has to be last, because or-groups produce hidden
          // pattern nodes at the same offset as their first
          // alternative, and we don't want to highligh the whole
          // or-group.
        }
      } );
      if ( keynode ) {
        keynode.setVariable("$$$");  //impossible name
        var hitsarray = xp.match(node);        
        if ( context.conf['highlighthits'] )
          context.highlightHits(hitsarray);
      }
    } catch (e) {
      logger.debug("Oops, highlightDomNode caught an exception " + e);
      // should not happen. If it does, we don't get a highlight
      // which is kind of acceptable.
    }
  }; // highlightDomNode

  // Select whole nodes with dblclick,
  // including children
  function dblclick(e) {
    var patt = xpatt.value;
    var p = xpatt.selectionStart;
    //logger.debug("dblclick:  sel: " +
    //   xpatt.selectionStart + "-" + xpatt.selectionEnd );
    if ( xpatt.selectionStart == xpatt.selectionEnd )
      return; // nothing selected, do not expand
    var firstchar = xpatt.value.charAt(xpatt.selectionStart);
    //logger.debug("dblclick: first = '" + firstchar + "'\n");
    if ( !firstchar.match(/[A-Za-z0-9\{\[\(]/) )
      return; // not at a tag
    var terminators = ":{}()|" ; // TODO Check groups?
    var open = "{[(";
    var close = ")]}";
    // scan to the end of the current node
    while ( terminators.indexOf(patt.charAt(p)) < 0 ) {
      p++;
    }
    //logger.debug("dblclick: node itself ends at " + p );
    // scan its children
    if ( open.indexOf(patt.charAt(p)) >= 0  ) {
      var depth = 0;
      do {
        if (open.indexOf(patt.charAt(p)) >= 0 )
          depth ++;
        if (close.indexOf(patt.charAt(p)) >= 0 )
          depth --;
        p++;
        //logger.debug("dblclick: children at " + p + " depth=" + depth );
      } while (depth > 0 );
    }
    // and trailing whitespace
    while ( patt.charAt(p) == " ")
      p++;
    //logger.debug("dblclick: hl ends at " + p );
    xpatt.selectionEnd = p;
    highlightDomNode( patt, xpatt.selectionStart );
  };  //dblclick

  xpatt.addEventListener("dblclick", dblclick, false );
  return xpatt;
}; // drawEditTab

// Draw the first tab, with the xpatternMaker stuff
// Second version, starting from the pattern
ParseXpattern.prototype.drawCreateTab = function(surface, win, context) {
  // watch out, this is a complex function, with event handlers
  // inside event handlers, and dynamically adjusted displays
  // on (at least) two levels!
  var mainwin = win;
  xmlHelper.emptyChildren(surface);
  //dump("drawCreateTab start\n");
  var groupid = 1; // enumerates all groups we may meet


  var vbox = xmlHelper.appendNode(surface, "vbox", null, {flex:1}, null);

  var hbox = xmlHelper.appendNode(surface, "hbox", null, null, null);

  var xpatt = xmlHelper.appendNode(vbox, "label",this.conf.xpattern ,
            { width:"100%" ,style:"font-weight: bold;" } );
  var alertmsg =  xmlHelper.appendNode(vbox, "label",
         "Warning: Pattern too complex for the designer",
         { width:"100%" ,style:"font-weight: bold; color:red;", hidden:true } );
  var nodebox = xmlHelper.appendNode(vbox, "vbox", null, null, null);
  nodebox.cf_opengroup = null ; // not inside a group
  nodebox.cf_currentgroup = 0;
  xmlHelper.appendNode(vbox, "separator", null,
      {"class": "groove-thin"}, null);
  var buttonbox = xmlHelper.appendNode(vbox, "hbox", null, null, null);
  var addNodeButton = xmlHelper.appendNode(buttonbox, "button",
      null, {"label": "Design XPattern"}, null);
  var addGroupButton = xmlHelper.appendNode(buttonbox, "button",
      null, {"label": "Add a group ()", "hidden": true }, null);
  var endGroupButton = xmlHelper.appendNode(buttonbox, "button",
      null, {"label": "End group", "hidden": true }, null);
  xulHelper.captionField(buttonbox,"", { "width":20 } );
  var generateButton = xmlHelper.appendNode(buttonbox, "button",
      null, {"label": "generate a pattern", "hidden": true}, null);


  // Adds a field to the nodebox. That is, one field, and possibly
  // its attributes too.
  // If domNode == null and patternNode == null, asks the user to
  // point to the field.
  // If domNode != null and patternNode == null, creates a group
  // Otherwise creates a node from  the arguments
  // (within ParseXpattern.prototype.draw)
  var addField = function ( domNode, patternNode ) {
    var gbox;
    gbox = xmlHelper.appendNode(nodebox, "vbox", null, null, null);
    //xmlHelper.appendNode(gbox, "separator", null,
    //    {"class": "groove-thin"}, null);
    var gline = xmlHelper.appendNode(gbox, "hbox", null, null, null);
    gbox.cf_domNode = null;
    gbox.cf_isgroup = false; // overwrite later if needed
    gbox.cf_parentgroup = nodebox.cf_opengroup; // can be null
    gbox.cf_groupid = nodebox.cf_currentgroup;
    gbox.cf_grouptype = "("; // default to normal group, if a group at all
    gline.setAttribute("hidden", true);
    // Calculate indentation (depth of nested groups)
    var indentlevel = 3;
    var parentgroup = gbox.cf_parentgroup;
    while ( parentgroup != null ) {
      indentlevel += 15;
      parentgroup = parentgroup.cf_parentgroup;
    }
    var filler = xulHelper.captionField(gline,"");
    filler.setAttribute("width", indentlevel );
    var txtcapt = xulHelper.captionField(gline,"");
    txtcapt.setAttribute("width", 220 - indentlevel );
    // set up a group type pulldown, although it is almost always hidden and
    // unused.
    var gtypelist = [ "(Group)", "Or-list" ];
    var gtypeinp = xulHelper.arraySelectField( gline, context, "", gtypelist );
    gtypeinp.setAttribute("width", 220 - indentlevel );
    gtypeinp.setAttribute("hidden", true );

    if ( gbox.cf_parentgroup ) {
        gbox.cf_jsonpref = gbox.cf_parentgroup.cf_jsonpref;
        if ( gbox.cf_parentgroup.cf_varname &&
                    gbox.cf_parentgroup.cf_varname.value ) {
            gbox.cf_jsonpref+= "[0]." + gbox.cf_parentgroup.cf_varname.value;
        }
    } else {
        gbox.cf_jsonpref = context.conf["jsonPath"].path + "." +
            context.conf["jsonPath"].key;
    }

    var getgroups = ( patternNode != null && patternNode.getType &&
      patternNode.getType() == "("  );
      //(patternNode.getType() == "(" || patternNode.getType() == "|" ) );
    // building an existing group
    getgroups |= ( patternNode != null && domNode == null ); // group click
    var varnameinput = xulHelper.singleJsonPathField(gline, context,
          "", "", gbox.cf_jsonpref, getgroups, [ "none" ]);
    gbox.cf_varname = varnameinput;
    //dump("addfield: varname: '" + gbox.cf_varname.value + "'\n");

    var cardinalitylist = [ "obligatory ", "optional ?",
              "repeating +", "opt. repeat *" ];
              // note - the last char of these must be the char to set!
    var cardinalityinput = xulHelper.arraySelectField( gline, context,
           "", cardinalitylist );
    //cardinalityinput.setAttribute("align","left");  // ???
    gbox.cf_cardinality = cardinalityinput;

    var attrButton = xmlHelper.appendNode(gline, "button",
        null, {"label": "Attributes"}, null);

    var rmButton = xmlHelper.appendNode(gline, "button",
        null, {"label": "Remove"}, null);

    if ( domNode == null && patternNode == null ) { // ask the user to point
      var pleasecapt =  xulHelper.captionField(nodebox,
        "Please click on some part of a good hit");
      app.newHl(context.getPageDoc(), function() {
          xmlHelper.removeNode(pleasecapt);
          gline.setAttribute("hidden", false);
          var node = this;
          if ( node.cf_datanode ) {
            node = node.cf_datanode;
          }

          gbox.cf_domNode = node;
          var dnn = node.localName || node.nodeName;
          var txt = dnn+ ":" + node.textContent.substr(0,20) ;
          txtcapt.label = txt;
          app.highlighter.destroy();
          if ( context.conf['highlighthits'] ) {  // highlighthits
            var col = gethighlightcolor();
            xulHelper.highlightNode(node, col, true );
            xulHelper.highlightNode(txtcapt, col, true );
            gbox.cf_highlight = col;
            //logger.debug("Added selected node " + gbox.cf_domNode );
            generatePattern(true);
          }
          return false;
      } ); // highlighter
    } else if ( domNode == null ) {
      makeGroupBox();  // new empty group
    } else { // have both pattern and dom node. Take data from there.
      gline.setAttribute("hidden", false);
      if ( patternNode.nodeType == "(" ) { // group node
        logger.debug("Making group");
        makeGroupBox();
      } else { // regular node
        var dnn = domNode.localName || domNode.nodeName;
        logger.debug("Making node " + dnn + " " + domNode +
          " t= " + typeof(domNode.getAttribute));
        var txt = dnn + ":" + domNode.textContent.substr(0,20) ;
        txtcapt.label = txt;
        gbox.cf_domNode = domNode;
        var col = domNode.getAttribute("cf_user_highlight");
        //dump("node: txt='" + txt + "' dom= " + domNode + " hi=" + col +"\n");
        xulHelper.highlightNode(txtcapt, col, true );
        gbox.cf_highlight = col;
      }
      xulHelper.selectMenulistItemWithValue(varnameinput,
            patternNode.getVariable(),true);
      var card = patternNode.getCardinality();
      if ( card == "?" )
        cardinalityinput.selectedIndex = 1;
      else if ( card == "+" )
        cardinalityinput.selectedIndex = 2;
      else if ( card == "*" )
        cardinalityinput.selectedIndex = 3;
      var an = patternNode.attributes;
      while ( an ) {
        // dump("Found attr " + an.dumpString(-2) + "\n" );
        addAttrLine(an, indentlevel);
        an = an.nextSibling;
      }
    } // AddField

    // Make a group box with indentation.
    // Keep track of the group stack
    // Can be a regular group, or an or-bag
    // (still within addField within ParseXpattern.prototype.draw)
    function makeGroupBox() {
      txtcapt.label = "(  ) ";
      txtcapt.hidden = true;
      gtypeinp.hidden = false;
      gline.setAttribute("hidden", false);
      endGroupButton.setAttribute("hidden", false);
      gbox.cf_parentgroup = nodebox.cf_opengroup; // remember parent
          // can be null on top-level groups, no problem
      nodebox.cf_opengroup = gbox;
      gbox.cf_isgroup = true;
      gbox.cf_groupid = groupid;
      nodebox.cf_currentgroup = groupid;
      groupid ++;
      //dump("Opening a new group \n");
      attrButton.setAttribute("disabled", true); // groups have no attrs

      gtypeinp.addEventListener("command", function(e) {
        if (gtypeinp.value == "(Group)" ){
          varnameinput.disabled = false;
          gbox.cf_grouptype = "("; // normal group
        } else {
          cardinalityinput.selectedIndex = 2;  // default to +
          varnameinput.disabled = true; // or-bags don't have variables (?)
          gbox.cf_grouptype = "|"; // or-bag
        }
        generatePattern(true);
      },false);
    };

    function endGroupBox() {
      if ( nodebox.cf_opengroup == null ) {
        logger.debug("Attempting to close a group when none open!");
        return; // should never happen, the button should be disabled
      }
      //dump("EndGroup: opengroup = " + nodebox.cf_opengroup + " into " +
      //    nodebox.cf_opengroup.cf_parentgroup + "\n");
      var gbox = xmlHelper.appendNode(nodebox.cf_opengroup,
                                      "vbox", null, null, null);
      gbox.cf_domNode = null;
      nodebox.cf_opengroup = nodebox.cf_opengroup.cf_parentgroup;
      // pop it off the stack. Can well be null
      if ( nodebox.cf_opengroup != null ) {
        nodebox.cf_currentgroup = nodebox.cf_opengroup.cf_groupid;
      } else {
        nodebox.cf_currentgroup = 0;
      }
      if ( nodebox.cf_opengroup == null ) {
        endGroupButton.setAttribute("hidden", true );
      }
    }; // endGroupBox

    // Add an attribute line to a field
    function addAttrLine ( attrNode, indentlevel ) {
      if ( gbox.cf_domNode == null ) {
        return; // happens in a group node
        // TODO - a better way to distinguish??
      }
      var abox = xmlHelper.appendNode(gbox, "vbox", null, null, null);
      var aline = xmlHelper.appendNode(abox, "hbox", null, null, null);
      var aline2 = xmlHelper.appendNode(abox, "hbox", null, null, null);
      var filler =  xulHelper.captionField(aline,"");
      indentlevel += 15;
      filler.setAttribute("width", indentlevel);
      var filler2 =  xulHelper.captionField(aline2,"");
      filler2.setAttribute("width", indentlevel);
      var capt =  xulHelper.captionField(aline, "Attribute");

      var attrs = gbox.cf_domNode.attributes ;
      var attrlist = [];
      for ( var a = 0 ; a < attrs.length; a++ ) {
        var aname = attrs[a].nodeName ;
        if (aname.substr(0,3) != "cf_" ) { // skip our own highlights!
          attrlist.push( aname );
        }
      }
      var attrselect = xulHelper.arraySelectField( aline,
          context, "", attrlist);
      abox.cf_attrselect = attrselect;
      var cont = xulHelper.captionField(aline2,"");
      cont.setAttribute("width", 220);
      abox.cf_attrvalue="";

      var conditions = [ "must be present",
                          "must equal this value" ];
      var cond = xulHelper.arraySelectField( aline,
                          context, "", conditions );
      abox.cf_attrcond=cond;

      xulHelper.captionField(aline," going into ");
      var varnameinput = xulHelper.singleJsonPathField(aline, context,
          "", "", gbox.cf_jsonpref, false, [ "none" ]);

      abox.cf_attrvariable = varnameinput;

      if ( attrNode != null ) {
        var aname = attrNode.nodeType;
        aname = aname.replace("@","" );
        var selection=-1;
        for (var i=0; i<attrlist.length; i++) {
          if (attrlist[i] == aname ) {
            attrselect.selectedIndex = i;
          }
        }
        if ( attrNode.attrValue != "" ) {
          cond.selectedIndex = 1;
        }
        var vn = attrNode.getVariable();
        if ( vn != "" ) {
          //dump("Setting varname (" + varnameinput.value + ") " +
          //    "to '" + attrNode.getVariable() + "' \n");
          varnameinput.value = attrNode.getVariable();
        }
      }

      var updateAttrTxt = function() {
        var attrname = attrselect.selectedItem.value;
        var aval = gbox.cf_domNode.getAttribute(attrname);
        //dump("attr '" + attrname + "' '" + aval + "'\n");
        abox.cf_attrvalue = aval;
        if (aval.length > 80) {
          aval=aval.substr(0,80)+"..."
        }
        cont.label = aval;
      }; // updateAttrTxt

      updateAttrTxt();
      abox.setAttribute("cf_attrline", true);

      attrselect.addEventListener("command", function(e) {
        updateAttrTxt();
        generatePattern(true);
        }, false);

      varnameinput.addEventListener("command", function(e) {
        generatePattern(true);
      },false);

      cond.addEventListener("command", function(e) {
        generatePattern(true);
      },false);

      var rmAttrButton = xmlHelper.appendNode(aline, "button",
          null, {"label": "Remove"}, null);

      rmAttrButton.addEventListener("click", function(e) {
        xmlHelper.removeNode(abox);
        generatePattern(true);
        }, false );

    } // addAttrLine

    varnameinput.addEventListener("command", function(e) {
      //logger.debug("varnameinput.command: " + varnameinput.value );
      generatePattern(true);
    },false);

    cardinalityinput.addEventListener("command", function(e) {
      //logger.debug("varnameinput.command: " + varnameinput.value );
      generatePattern(true);
    },false);

    attrButton.addEventListener("click", function(e) {
      addAttrLine(null, indentlevel);
      generatePattern(true);
    }, false );   // attrbutton.click

    rmButton.addEventListener("click", function(e) {
      if ( gbox.cf_domNode != null ) {
          xulHelper.unhighlightNode(gbox.cf_domNode, true);
      }
      xmlHelper.removeNode(gline);
      xmlHelper.removeNode(gbox);
      generatePattern(true);
    }, false );
  }; // addField


  // Helper to populate the designer
  function designerNode(xp, besthit, depth, len) {
    var prefix = "desN: " + depth + "." + len + " [" + xp.dumpString(-2)+ "] " ;
    logger.debug(prefix + "starting xp.nodeType=" + xp.nodeType );
    if ( xp.nodeType == "(" ) { // group node
      var grp = addField(null,this ); // fake a dom node to signal a group
      if ( xp.firstChild ) {
        if ( xp.firstChild.nodeType == "|" ) { // or-bag
          logger.debug(prefix+"Making OR-bag");
          var alt = xp.firstChild.firstChild;
          var altlen = 0;
          while(alt) {
            designerNode(alt, besthit, depth+1, altlen);
            altlen++;
            alt = alt.nextAlternative;
          }
        } else { // regular group
          logger.debug(prefix+"Making a (regular) group");
          designerNode(xp.firstChild, besthit, depth+1, 0 );
        }
      }
      endGroupBox();

    } else { // not a group, must be a real thing.
      // see if we have the xp node in the hits
      var hitidx = -1;
      for ( var i=0; i<besthit.hits.length; i++) {
        var h = besthit.hits[i];
        logger.debug(prefix + "hit " + i + " " + h.name + ": " + h.value + "  " + h.pattern.dumpString(-2) );
        if ( h.pattern == xp ) {
          if ( hitidx == -1 || h.value.length > 0) {
            hitidx = i;
            logger.debug(prefix + "found at hit " + i + ": '" + h.name + "': '" + h.value +"'");
            if ( hitidx == -1 || h.value.length > 0)
              break; // be satisfied with the first that has real text
          }
        }
      }
      logger.debug(prefix + "Decided on hit " + hitidx  );
      if ( hitidx != -1) {
        var h = besthit.hits[hitidx];
        logger.debug(prefix + "dom: " + h.dom.nodeName + " p=" + h.dom.parentNode.nodeName );
        var domnode = h.dom;
        if ( domnode.nodeName == "#text" )
          domnode = domnode.parentNode;
        addField(domnode, h.pattern ); // regular node
      }
      // Recurse through the pattern
      if ( xp.firstChild )
        designerNode(xp.firstChild, besthit, depth+1, 0 );

    } // real node
    if ( xp.nextSibling )
      designerNode(xp.nextSibling, besthit, depth, len+1 );

  }; // designerNode

  // Populate the designer from the pattern;
  function populateDesigner() {
    var task = new Task(context.task.connector, "dummy");
    try { // run the step with a fake task parameter to get the outputs
    context.run(task);
    } catch (e) {
      var where = "";
      if ( e.fileName )
        where = "\n" + e.fileName + "." + e.lineNumber
        win.alert("OOPS! " + e + where);
      logger.error("XPattern addnode: running the task failed with " +
      e + where );
      return;
    }
    // Choose the best hit, the one with most different hits
    // (it is the most complex)
    // TODO - we could measure actual complexity better, this
    // fails when one field repeats many times, instead of many different
    // fields...
    var besthit = task.xpatternHits[0] ;
    for ( var i=0; i<task.xpatternHits.length; i++) {
      if ( task.xpatternHits[i].hits.length > besthit.hits.length ) {
        besthit = task.xpatternHits[i];
        logger.debug("Hit " + i + " is the best so far with " +
        besthit.hits.length + " lines");
      }
    }
    logger.debug("Looking at " + besthit.hits.length + " hits");
      designerNode(task.parsedXpattern, besthit, 0,0 );

    // Try to generate, and see if the result differs
    // In that case, display a warning
    var oldpatt = context.conf.xpattern;
    generatePattern(false);
    var oldpatt2 = oldpatt.replace(/\s+/g,""); // ignore whitespace
    var newpatt = context.conf.xpattern.replace(/\s+/g,"");
    if ( oldpatt2 != newpatt ) {
      context.setPattern(oldpatt);
      logger.debug("Too complex. Have '" + oldpatt2 +
      "' would get '" + newpatt + "'" );
      alertmsg.hidden = false;
    }
  }; // populateDesigner

  // Add a node to the designer, or if not in design mode yet,
  // add the whole pattern.
  // (still within ParseXpattern.prototype.draw)
  addNodeButton.addEventListener("click", function(e) {
      if ( context.conf['xpattern'] != "" &&
          addNodeButton.label != "Add another node" ) {
        // Generate the inputs from the pattern and page
        populateDesigner();
      } else {
        addField(null,null ); // ask user for field etc
      }
      addNodeButton.label="Add another node";
        // instead of "start creating pattern"

      if ( generateButton.getAttribute("hidden") == true ) {
        var pageDoc = context.getPageDoc();
        var hitarea = context.conf['hitarea'];
        var startnode = xmlHelper.getElementByXpath(pageDoc,hitarea);
        xulHelper.unhighlightAll(startnode, true);
      }
      generateButton.setAttribute("hidden", false );
      addGroupButton.setAttribute("hidden", false );
      return false;
    }, false); // addNodeButton.click

  addGroupButton.addEventListener("click", function(e) {
    addField(null,this );  // just anything non-null to signal group
  }, false); // generateButton.click


  endGroupButton.addEventListener("click", function(e) {
    endGroupBox();
  }, false);

  function endGroupBox() {
    if ( nodebox.cf_opengroup == null ) {
      logger.debug("Attempting to close a group when none open!");
      return; // should never happen, the button should be disabled
    }
    //dump("EndGroup: opengroup = " + nodebox.cf_opengroup + " into " +
    //    nodebox.cf_opengroup.cf_parentgroup + "\n");
    var gbox = xmlHelper.appendNode(nodebox.cf_opengroup,
                                    "vbox", null, null, null);
    gbox.cf_domNode = null;
    nodebox.cf_opengroup = nodebox.cf_opengroup.cf_parentgroup;
    // pop it off the stack. Can well be null
    if ( nodebox.cf_opengroup != null ) {
      nodebox.cf_currentgroup = nodebox.cf_opengroup.cf_groupid;
    } else {
      nodebox.cf_currentgroup = 0;
    }
    if ( nodebox.cf_opengroup == null ) {
      endGroupButton.setAttribute("hidden", true );
    }
  }; // endGroupBox


  // Generate the xpattern out of the input fields and the page
  // (still within ParseXpattern.prototype.draw)
  generateButton.addEventListener("click", function(e) {
    generatePattern(false);
  },false );  // generateButton

  // Generate the pattern from the selections
  // autogen indicates that this is an automatic generation (and not a result
  // of the user clicking the generate button). That means that it can be
  // disabled by a config option, and error handling may be different.
  // (still within ParseXpattern.prototype.draw)
  function generatePattern(autogen) {
      if ( autogen && ! context.conf["autogenerate"] ) {
        logger.debug("no autogeneration");
        return;
      }
      logger.debug("About to generate pattern");
      var pageDoc = context.getPageDoc();
      var hitarea = context.conf['hitarea'];
      // dump("About to get startnode from '" + hitarea + "'\n");
      var startnode = null;
      try {
        //dump("Starting with hitarea " + hitarea + "\n");
        startnode = xmlHelper.getElementByNodeSpec(pageDoc,hitarea);
      } catch (e) {
        logger.debug("Did not get the hitarea: " + e + "\n");
        if (!autogen) {
          win.alert("The xpath for the hit area is bad. \n'"+hitarea+"' \n" );
        }
        return;
      }
      if (startnode == null) {
        if (!autogen) {
          win.alert("The hit area is bad. Can not create pattern!");
        }
        return;
      }
      var maker = new XpatternMaker( startnode );
      var n = nodebox.firstChild;
      //dump("generate: nodebox: \n"); xmlHelper.dumpxml( nodebox );
      logger.debug("maker: first child of the box: " + n );
      if ( n == null ) { // nothing to build on, clear the pattern
        logger.debug("Nothing to create a pattern from, clearing it");
        context.setPattern(""); // sets the history and display
      }
      while ( n != null ) {
        fn = n.cf_domNode;
        logger.debug("maker: domnode: " + fn );
        if ( fn != undefined || n.cf_isgroup ) {
          var varname = null;
          if ( n.cf_varname != undefined ) {
            varname = n.cf_varname.value;
          }
          logger.debug("maker: var=" + varname + " " + fn  );
          var card = null;
          if ( n.cf_cardinality != undefined ) {
            card = n.cf_cardinality.value;
            card = card.charAt( card.length-1 ); // the last one
          }
          var an = n.firstChild;
          var attrstr="";
          while (an != null ) {
            if ( an.getAttribute("cf_attrline")) {
              //dump("Found attrline!\n"); xmlHelper.dumpxml(an);
              var attrname = "";
              var attrvalue= "";
              var attrvariable="";
              if ( an.cf_attrselect &&
                    an.cf_attrcond && an.cf_attrvariable ) {
                attrname = "@" + an.cf_attrselect.value;
                if (an.cf_attrcond.selectedIndex == 1) {
                  // must equal
                  attrvalue = " =\""+an.cf_attrvalue+"\"";
                }
                if ( an.cf_attrvariable.value != "none" ) {
                  attrvariable = " $" + an.cf_attrvariable.value;
                }
                attrclause = attrname + " " + attrvalue + attrvariable;
                if (attrstr) attrstr += " : ";
                attrstr += attrclause;
              }
            }
            an = an.nextSibling;
          }
          logger.debug("Got attributes for '" + varname + "': " + attrstr  );
          var groupid = n.cf_groupid ;
          if ( n.cf_isgroup ) {
            var gtype = n.cf_grouptype;
            logger.debug("maker: Found a '" + gtype + "'-GROUP: '" + groupid +"' ");
            var parentgroup =0;
            if ( n.cf_parentgroup ) {
              parentgroup = n.cf_parentgroup.cf_groupid;
            }
            maker.addGroup(fn, varname, card, attrstr,
              groupid, gtype, parentgroup );
          } else {
            maker.addNode(fn, varname, card, attrstr, groupid );
          }
        }
        n = n.nextSibling;
      } // while node loop
      var p = maker.getPattern();
      logger.debug("generate: Old pattern: " + context.conf['xpattern'] );
      if ( p != null ) {
        ps = p.dumpString(-1);
        logger.debug("generated pattern '" + ps + "'");
        context.setPattern(ps); // sets the history and display
      } else {
        logger.debug("Failed to generate pattern");
        if (!autogen) {
          win.alert("Could not create a pattern.\n" +
               "Are you sure all nodes are inside the hit area?");
        }
      }
      alertmsg.hidden = true;

  } // generatePattern
  logger.debug("Draw design tab, returning " + xpatt );
  return xpatt;
}; // This is the end of ParseXpattern.prototype.drawCreateTab. Finally!


// History tab
// Always redraws the whole tab from the conf history, and from its cache
var historyhits={}; // global cache for redisplaying hit counts

ParseXpattern.prototype.drawHistoryTab = function (surface, context) {
  xmlHelper.emptyChildren(surface);
  //logger.debug("drawHistoryTab start");
  var vbox = xmlHelper.appendNode(surface, "vbox", null, null, null);
  //xulHelper.captionField(vbox,"Xpattern History");
  var historybox = xmlHelper.appendNode(vbox, "vbox", null, {flex:1}, null);
  if ( !context.conf.xpatternhistory ||
       !context.conf.xpatternhistory.length ) {
    logger.debug("No history, nothing to draw");
    return;
    }

  for ( var i = 0; i < context.conf.xpatternhistory.length; i++) {
    var patt = context.conf.xpatternhistory[i];
    var pb = xmlHelper.appendNode(historybox, "hbox" );
    xmlHelper.appendNode(pb,"vbox", null, { height:7}  );
    //logger.debug("h " + i + " hist = " + historyhits[patt] );
    if ( typeof(historyhits[patt]) == "undefined" )
      historyhits[patt] = "Try";
    var trybut = xmlHelper.appendNode(pb, "button", null,
           {label: historyhits[patt], width:50 } );
    trybut.cf_patt = patt;
    var revertbut = xmlHelper.appendNode(pb, "button", null,
           {label: "Revert", width:50, disabled:(i==0) } );
    revertbut.cf_patt = patt;
    
    var l = 0;  // find the part to highlight
    var r = patt.length-1;
    if ( i == context.conf.xpatternhistory.length -1 ) {
      // last one, no next to compare
      r = 0;
    } else { // find the difference from previous
             // actually, find length of matching beginning and end,
             // the rest must be the different part. Ignore whitespace
      var prev = context.conf.xpatternhistory[i+1];
      var pr = prev.length -1;
      var pl = 0;
      while ( l<patt.length && pl < prev.length) {
        if ( patt[l] == prev[pl] ) {
          l++;
          pl++;
        } else if ( patt[l].match(/\s/) ) {
          l++;
        } else if ( prev[pl].match(/\s/) ) {
          pl++;
        } else break; // found a true difference
      }
      while ( l > 0 && patt[l] && patt[l].match(/[a-zA-Z0-9$]/) )
        l--; // back to beginning of word
      while ( r > l && pr > pl  ) {
        if ( patt[r] == prev[pr] ) {
          r--;
          pr--;
        } else if ( patt[r].match(/\s/) ) {
          r--;
        } else if ( prev[pr].match(/\s/) ) {
          pr--
        } else break;
      }
      while ( r<patt.length && patt[r] && patt[r].match(/[a-zA-Z0-9]/) )
        r++; // skip to end of word
    }
    xmlHelper.appendNode(pb, "label", null, 
                { value: patt.substr(0,l),
                  style:"margin-right: 0px; " } );
    xmlHelper.appendNode(pb, "label", null, 
                { value: patt.substr(l, r-l+1), 
                         style:"margin-left: 0px; margin-right: 0px;font-weight:bold;"} );
    xmlHelper.appendNode(pb, "label", null, 
                         { value: patt.substr(r+1),
                         style:"margin-left: 0px; " } );
                         
    revertbut.addEventListener("click", function(e) {
      logger.debug("Revert button " + this.cf_patt );
      context.setPattern(this.cf_patt);
    }, false);
          
    trybut.addEventListener("click", function(e) {
      logger.debug("Try button " + this.cf_patt );
      var hits = "??";
      var tooltip = "";
      var task = new Task(context.task.connector, "dummy");
      var origpatt = context.conf.xpattern;
      try { // run the step with a fake task parameter to get the outputs
        context.conf.xpattern = this.cf_patt;
        context.run(task);
        hits = "" + task.xpatternHits.length + " hit";
        if ( task.xpatternHits.length != 1)
          hits += "s";
        if (task.xpatternwarnings) {
          hits += " " + task.xpatternwarnings + "w";
        } else { // count number of pieces in the hits
            var pcs = 0;
            for (var i=0; i<task.xpatternHits.length; i++) {
              var h = task.xpatternHits[i];
              for ( var j=0; j<h.hits.length; j++ ) {
                if ( h.hits[j].name && h.hits[j].value )
                  pcs++;
              }
            }
            hits += " " + pcs + "p";
            if ( task.xpatternHits.length + pcs >= 100)
              hits = hits.replace ( / hits?/, "h" );  // shorten text to fit a button
        }
      } catch (e) {
        logger.debug("Trying history pattern " + this.cf_patt + " failed " + 
            "with " + e  );
        if ( e.fileName )
          logger.debug(" at " + e.fileName + "." + e.lineNumber );
        if ( e.message == "No hits found" )
          hits = "No hits";
        else {
          hits = "Error"; // the user can revert and see it in the editor
          tooltip = e.message;
        }
      }
      this.label = hits;
      this.setAttribute("tooltiptext",tooltip);
      historyhits[this.cf_patt] = hits;
      context.conf.xpattern = origpatt;
      }, false );
    
  } // history loop
}; // history tab


// Highlight all hits
ParseXpattern.prototype.highlightHits = function (hitsarray) {
  // find the highlights from the dom tree
  for (var i=0; i<hitsarray.length; i++) {
    var h = hitsarray[i];
    for (var j=0; j<h.hits.length; j++) {
      var d = h.hits[j]["dom"];
      var p = h.hits[j]["pattern"];
      if ( d && p ) {
        if (d.nodeName == "#text" )
          d = d.parentNode;
        if ( d.nodeName != "#comment" && d.nodeName != "#cdata-section" ) {
          var hi = d.getAttribute("cf_user_highlight");
          if (hi) {
              p.highlight = hi;
              //dump("found highlight " + hi +
              //     " at " + h.hits[j]["value"] + "\n");
          }
          if ( ! p.highlight ) { // set highlight if not yet set
            var col = gethighlightcolor();
            p.highlight = col;
            xulHelper.highlightNode(d, col, true );
            /*
            dump("Assigned highlight N " + col +
                " to pattern " + p.dumpString(-2) +
                " and dom node " + xmlHelper.getElementXpath(d) + "\n");
            */
          }
        }
      }
    }
  }
  // and distribute them to all hits
  for (var i=0; i<hitsarray.length; i++) {
    var h = hitsarray[i];
    for (var j=0; j<h.hits.length; j++) {
      var d = h.hits[j]["dom"];
      var p = h.hits[j]["pattern"];
      if ( d && p ) {
        if (p.highlight) {
          if (d.nodeName == "#text" ) {
            d = d.parentNode;
          }
          if ( d.nodename != "#comment" ) {
            xulHelper.highlightNode(d, p.highlight, false);
            //dump("setting highlight " + hi +
            //     " at " + h.hits[j]["value"] + "\n");
          }
        }
      }
    }
  }
}; // highlightHits

// Extract one hit into a structure of javascript objects that will
// later end up as suitable json, then xml
// Note that we need to append consequtive elements in the hits array,
// and keep track of nested groups
//  h is the hits array from XPattern match
//  j is the start index into it, in case of groups, recursion, etc
//  hitchecker contains the hitnumber to expect (and a pointer to the
//  task, in order to get to its logger...)

ParseXpattern.prototype._onehit = function (h,j, hitchecker) {
  var idx = {}; // current index for each tag (append to the fifth title)
  var res = {}; // the resulting record field->array of values
  while ( j < h.hits.length && h.hits[j]["name"] != "/" ) {
    var fullname = h.hits[j]["name"];
    var n = fullname.replace ( /\/.+$/, "" );  // remove the modifiers
    var skipthis = false; // signals to skip the hitnumber
    if ( n == "hitnumber" ) {
      if ( this.conf.removehitnumber )
        skipthis = true; // don't collect that variable at all!
      if (  this.conf.hitnumbercheck ) {
        var hn = parseInt(h.hits[j]["value"]);
        if ( hn ) { // got the actual variable, not a start marker
          logger.debug("Looking at hit number " + hn + " expecting " + hitchecker.hitnumber );
          if ( hn && hitchecker.hitnumber && hn != hitchecker.hitnumber ) {
            hitchecker.warnings++;
            if (this.conf.failhitnumber) {
              throw new StepError ("Hitnumber mismatch. " +
                  "Expected " + hitchecker.hitnumber + " got " + hn );
            } else {
              hitchecker.task.warn("Hitnumber mismatch. " +
                "Expected " + hitchecker.hitnumber + " got " + hn );
            }
          }      
          hitchecker.hitnumber = hn +1; // expect one higher next time
        }
      } // want to check 
    } // we have a hitnumber
    if ( ! skipthis ) {
      var lastchar = n.charAt(n.length-1,1);
      if ( lastchar == "/" ) {
        n = n.slice(0,-1);
      }
      var v = h.hits[j]["value"];
      //dump("parse: hit ." + j + " n='" + n + "'" + " v='" + v + "' " + "\n");
      if ( v == "" && n == "" ) { // no name, no value, must be a builder marker
        // Ignore the whole thing.
      } else if ( v == "" ) { // signals a new field
        if ( idx[n] == undefined ) { // first time we see it
          idx[n]=0;
          //dump("First time we see '" + n + "'\n");
          res[n]=[""];
        } else {
          idx[n]++;
          //dump("Incremented index '" + n + "' to " + idx[n] + "\n");
          res[n][idx[n]]="";
        }
        if ( lastchar == "/" ) { // nested group
          //dump("Recursion \n");
          var rr = this._onehit(h, j+1, hitchecker);
          //dump("Recursion done\n");
          j=rr["j"];
          res[n][idx[n]]=rr["res"];
        }
      } else { // real data
        if ( idx[n] == undefined ) {
          // May happen in groups, with unvariabled fields as in
          // ( TD { A $author : B $title } ) $item
          // <TD><A>au</A><B>ti</B>extra text</TD>
          //dump("Skipping unnamed subfield\n");
        }
        else {
          var k = idx[n];
          if (res[n][k] == undefined) {
              res[n][k] = "";
          }
          if (res[n][k] != "" && fullname.indexOf("/whitespace") == -1) {
              res[n][k] += " ";
          }
          res[n][k] += v;
          //dump("Appended to res['" + n + "']["+k+"]: '" + res[n][k] + "'\n");
        }
      }
    } // ! skipthis
    j++;
  }
  var r = {};
  r["res"] = res;
  r["j"] = j;
  return r;
} // _onehit


ParseXpattern.prototype.run = function (task) {
  var hitarea = this.conf['hitarea'];
  var pageDoc = this.getPageDoc();
  var node = null;
  try {
    node = xmlHelper.getElementByNodeSpec(pageDoc, hitarea);
  } catch (e) {
    var ns = xmlHelper.nodeSpecToString(hitarea);
    task.debug("Bad XPath for hitarea" + ns );
    task.debug(e.message);
    if ( e.fileName  )  // catch syntax errors in code, etc
          logger.debug("  in " + e.fileName + "." + e.lineNumber  );
     throw new StepError("Hit area '" + ns + "' not legal xpath");
  }
  if (node == null) {
    var ns = xmlHelper.nodeSpecToString(hitarea);
    throw new StepError("Hit area '" + ns + "' not found in the document");
  }
  //dump("hit area node: \n"); xmlHelper.dumpxml(node);
  //dump("Parsing mode: " + pageDoc.compatMode + "\n");
  var patt = this.conf['xpattern'];
  try {
      //dump("Parsing '" + patt + "'\n");
      var par = new XpatternTextParser(patt);
      var xp = par.parse();
  } catch (e) {
      dump("Parse " + e + "\n'" + patt + "'\n" );
      throw new StepError("Bad xpattern '" + patt + "'" );
      return;
  }
  //dump("Xpattern: \n" + xp.dumpString(-1) + "\n" );

  if (this.inBuilder) {
    // dump("in builder, setting hightlights on \n");
    xulHelper.unhighlightAll(node,false);
    xp.askForNodes();
  }

  var hitsarray = xp.match(node);
  //dump("match returned " + hitsarray.length + " hits\n");
  if (this.inBuilder ) {
      if ( this.conf['highlighthits'] ) {
        this.highlightHits(hitsarray);
      }
      task.xpatternHits = hitsarray; // pass to the builder
      task.parsedXpattern = xp; // remember the actual pattern tree
  }
  var resarr = [];
  var hitnumber = undefined; 
  if (this.conf.keephitnumber) {
    hitnumber = parseInt(jsonPathHelper.get(this.conf["hitnumbervar"], task.data));
  }
  logger.debug("Expecting to start with hitnumber " + hitnumber );
  var hitchecker = { hitnumber: hitnumber, warnings: 0, task: task };
  for (var i=0; i<hitsarray.length; i++) {
      var r = this._onehit(hitsarray[i],0, hitchecker);
      resarr.push(r["res"]);
  }
  if (this.conf.keephitnumber) {
    jsonPathHelper.set(this.conf["hitnumbervar"], hitchecker.hitnumber, task.data);
  }
  if (this.inBuilder ) {
    task.xpatternwarnings = hitchecker.warnings;
  }
  if (resarr.length == 0 ) {
    throw new StepError("No hits found");
  } else {
    jsonPathHelper.set(this.conf["jsonPath"], resarr, task.data);
  }
}; // run

ParseXpattern.prototype.getClassName = function () {
    return "ParseXpattern";
};

ParseXpattern.prototype.getDisplayName = function () {
  return "Parse by XPattern";
};

ParseXpattern.prototype.getDescription = function () {
  return "Parses results using XPattern.";
};

ParseXpattern.prototype.getVersion = function () {
  // 0.2 introduced the negations
  // 0.3 supports the new data model
  // 1.0 data-model related global version bump
  // 1.1 autogenerate flag in config
  // 2.0 hitnumber check
  // 3.0 xpattern modifiers
  return "3.0";
};

ParseXpattern.prototype.upgrade = function (confVer, curVer, conf) {
  // can't upgrade if the connector is newer than the step
  if (confVer > curVer)
    return false;
  if ( typeof(conf['highlighthits']) == 'undefined' ){
    conf['highlighthits'] = true;
  }
  if (confVer < 0.3) {
    conf['jsonPath'] = { path: "$.output", key:"results" };
  }
  if (confVer < 1.1) {
    logger.debug("Upgrading parse_xpattern step to 1.1" );
    conf.autogenerate = true;
  }
  if (confVer < 2.0) {
    conf.hitnumbercheck = false;
    conf.removehitnumber = true;
    conf.keephitnumber = true; 
    conf.failhitnumber = true; 
    conf.hitnumbervar = { path: "$.session", key:"hitnumber" };
  }
  return true;
};

// There is really no point in providing a renderArgs() method
// for the XPattern parser, as the XPattern itself is too long to fit
// nicely and too complex to be grokkable at a glance.
// Therefore, no renderArgs function.


// TODO - Wouldn't this be easier from the string-format XPattern?
// Just find all $something strings...
ParseXpattern.prototype.capabilityFlagDefault = function (flag) {
  if (flag.substring(0, 7) != "result-")
    return null;

  var xp_returns_field = function (level, field, xp) {
    //dump("in xp_returns_field(" + level + ", " + field + ", " + xp + ") -- xp.variable=" + xp.variable +"\n");
    if (xp.variable && xp.variable == field)
      return true;
    if (xp.attributes && xp_returns_field(level+1, field, xp.attributes) )
      return true;
    if (xp.firstChild && xp_returns_field(level+1, field, xp.firstChild) )
      return true;
    if (xp.nextAlternative && xp_returns_field(level+1, field, xp.nextAlternative) )
      return true;
    if (xp.nextSibling && xp_returns_field(level+1, field, xp.nextSibling) )
      return true;
    return false;
  }

  // TODO We should cache the parsed pattern, we get here often
  if ( this.conf.xpattern ) { // maybe we don't have any defined yet
    var par = new XpatternTextParser(this.conf['xpattern']);
    var xp = par.parse();
    if (xp_returns_field(0, flag.substring(7), xp)) {
      //dump("xp_returns_field(" + flag + ") was true!\n");
      return true;
    } else {
      //dump("xp_returns_field(" + flag + ") was false!\n");
      return null;
    }
  }
  return null;
}


