var EXPORTED_SYMBOLS = ["Transform"];
Components.utils.import('resource://indexdata/runtime/Step.js');
Components.utils.import('resource://indexdata/runtime/StepError.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/xulHelper.js');
Components.utils.import("resource://indexdata/util/logging.js");
Components.utils.import('resource://indexdata/ui/RegexEditor.js');
Components.utils.import('resource://indexdata/util/jsonPathHelper.js');

var Transform = function () {
  this.conf = {};
  this.conf['usedArgs'] = [];
  this.conf['container'] = '';  // Name of container value to operate in
  this.conf['in'] = '';         // Name of value to read from
  this.conf['regex'] = '';      // Regular expression to apply
  this.conf['outputs'] = [{'replacement':"", 'jp':{append:jsonPathHelper.REPLACE, 'unchanged':true}}];
  this.conf.global = this.conf.icase = false;
  this.conf.cleanWhitespace = true;        // Toggle trimming whitespace
  this.conf.noMatchFail = true;
  this.conf.noMatchNoop = false;
};
Transform.prototype = new Step();
Transform.prototype.constructor = Transform;

Transform.prototype.init = function() {};

Transform.recipes = [
  [ "label",          "regex",          "replace", "global" ],
  // Row 0 (above) gives the names of the configuration elements
  // Subsequent rows (below) give sets of values for these elements
  [ "Remove commas",   ",",              "",        true ],
  [ "First number",    "^\\D*([0-9][0-9,.]*).*",  "$1",        false ],
  [ "Last number",     ".*?([0-9][0-9,.]*)\\D*$", "$1",        false ],
  [ "Remove HTML",     "<.*?>",          "",        true ],
];


Transform.prototype.draw = function(surface) {
  Components.utils.import('resource://indexdata/ui/taskPane.js');
  var context = this;

  // Source
  var sourceField = xulHelper.jsonPathField(surface, this, this.conf, "in", "Source: ");

  // Regex
  var hbox = xmlHelper.appendNode(surface, "hbox", null, { align: "center" });
  var regexInput = xulHelper.inputField(hbox, this, "regex", "Regular expression:", null, { flex:1 });
  var regexEditorButton = xmlHelper.appendNode(hbox, "button", null, {"label": "Editor"}, null);
  regexEditorButton.addEventListener("command", function(e) {
    var flags = "";
    if (gFlag.checked) flags += 'g';
    if (iFlag.checked) flags += 'i';
    new RegexEditor(regexInput.value, "Editable target text.", flags, function (regex, flags) {
      regexInput.value = regex;
      gFlag.checked = flags.indexOf('g') !== -1;
      iFlag.checked = flags.indexOf('i') !== -1;
    });
  }, false);

  // Flags
  var hbox = xmlHelper.appendNode(surface, "hbox", null, { align: "center" });
  var gFlag = xulHelper.checkbox(hbox, this, "global", "Global?");
  xmlHelper.appendNode(hbox, "spacer", null, { flex: "1" }, null);
  var iFlag = xulHelper.checkbox(hbox, this, "icase", "Ignore case?");

  // Options
  xmlHelper.appendNode(hbox, "spacer", null, { flex: "1" }, null);
  xulHelper.checkbox(hbox, this, "cleanWhitespace", "Clean whitespace?");
  xmlHelper.appendNode(hbox, "spacer", null, { flex: "1" }, null);
  xulHelper.checkbox(hbox, this, "noMatchFail", "Fail if no match?");
  xmlHelper.appendNode(hbox, "spacer", null, { flex: "1" }, null);
  xulHelper.checkbox(hbox, this, "noMatchNoop", "Leave unchanged if no match?");
  xmlHelper.appendNode(hbox, "spacer", null, { flex: "1" }, null);
  xulHelper.checkbox(hbox, this, "rmFromSource", "Remove match from source?");

  // Outputs
  for (let i = 0; i < this.conf.outputs.length; i++) {
    var output = this.conf.outputs[i];
    xmlHelper.appendNode(surface, "separator", null, { class: "groove" });
    var targetField = xulHelper.jsonPathField(surface, this, output, "jp", "Target: ");
    var fieldBox = xmlHelper.appendNode(surface, "hbox", null, {align: "center"});
    xmlHelper.appendNode(fieldBox, "caption", "Replace with: ");

    let replacementTextbox = xmlHelper.appendNode(fieldBox, "textbox", null,
      {flex:"1", value: output.replacement} , null);
    replacementTextbox.addEventListener("input", Transform.makeReplacementListener(output), false);

    let rm = xmlHelper.appendNode(fieldBox, "image", null,
      {src: "chrome://cfbuilder/content/icons/window-close.png"}, null);
    rm.addEventListener("click", Transform.makeRemoveListener(context, surface, i), false);
  }
  let add = xmlHelper.appendNode(fieldBox, "image", null, {src: "chrome://cfbuilder/content/icons/list-add.png"}, null);
  add.addEventListener("click", function (e) {
    context.conf.outputs.push({'replacement':"", 'jp':{append:jsonPathHelper.REPLACE}});
    xmlHelper.emptyChildren(surface);
    context.draw(surface);
  }, false);

  // Update inital output (if unchanged) to match input value
  var updateOut = function (e) {
    if (e.target.localName !== 'button'
        && context.conf.outputs[0]
        && context.conf.outputs[0].jp.unchanged) {
      context.conf.outputs[0].jp.key = context.conf.in.key;
      context.conf.outputs[0].jp.path = context.conf.in.path;
      xmlHelper.emptyChildren(taskPane.stepBox);
      context.draw(taskPane.stepBox);
    }
  }
  sourceField.addEventListener("command", updateOut, false);
  sourceField.addEventListener("change", updateOut, false);

  // Remove "unchanged" property when output changes
  var nixUnchanged = function (e) {
    if (context.conf.outputs[0] && context.conf.outputs[0].jp.unchanged) {
      delete context.conf.outputs[0].jp.unchanged;
    }
  }
  targetField.addEventListener("command", nixUnchanged, false);
  targetField.addEventListener("change", nixUnchanged, false);

  // Recipes
  xmlHelper.appendNode(surface, "separator", null, { class: "groove" });
  var hbox = xmlHelper.appendNode(surface, "hbox", null, { align: "center" });
  xmlHelper.appendNode(hbox, "caption", "Common transformations:");
  for (var i = 1; i < Transform.recipes.length; i++) {
    var recipe = Transform.recipes[i];
    var label = recipe[0];
    var button = xmlHelper.appendNode(hbox, "button", null, { label: label });
    button.addEventListener("command", Transform.makeRecipeListener(context, surface, i), false);
  }
};

// An extra layer of indirection is required to make the closures
// work correctly, since if the anonymous function is given inline,
// all instances of it will share the same (final) values.
//
Transform.makeRecipeListener = function (context, surface, i) {
  return function (e) {
    var names = Transform.recipes[0];
    var recipe = Transform.recipes[i];
    var j = 1;
    while (1) {
      var val = recipe[j];
      if (val === undefined) break;
      context.conf[names[j]] = val;
      j++;
    }
    xmlHelper.emptyChildren(surface);
    context.draw(surface);
  };
};

Transform.makeReplacementListener = function (output) {
  return function (e) {
    output.replacement = e.target.value;
  };
};

Transform.makeRemoveListener = function (context, surface, i) {
  return function (e) {
    if (context.conf.outputs.length > 1) {
      context.conf.outputs.splice(i, 1);
      xmlHelper.emptyChildren(surface);
      context.draw(surface);
    }
  };
};

Transform.prototype.processConf = function () {
  // Gather used arguments from inline JSONpaths
  let usedArgs = [];
  let usedArgRegex = /\{\$\.input\.(.+?)\}/g;
  let match;
  for (let i = 0; i < this.conf.outputs.length; i++) {
    while ((match = usedArgRegex.exec(this.conf.outputs[i].replacement)) !== null) {
      usedArgs.push(match[1]);
    }
  }

  // If we're reading (conf.in) from $.input, that key is also a used arg.
  // Not, however, if we're writing to it.
  if (this.conf.in.path === "$.input") {
    usedArgs.push(this.conf.in.key);
  }
  this.conf.usedArgs = usedArgs;

  // Precompute inline replacements
  for (let i = 0; i < this.conf.outputs.length; i++) {
    let output = this.conf.outputs[i];
    if (typeof(output.replacement) === "string")
      output.inline = jsonPathHelper.processInline(output.replacement);
    else
      throw new StepError("Missing replacement string for output " + i);
  }
};

Transform.prototype.normaliseWhitespace = function (value) {
  if (value) {
    // trim whitespace
    let result = value.replace(/^\s*(.*)\s*$/, "$1");
    // normalise whitespace
    result = result.replace(/\s+/g, " ");
    return result;
  }
};

Transform.prototype.run = function (task) {
  var context = this;
  let jpIn = this.conf.in;
  let flags = "";
  if (this.conf.global) flags += "g";
  if (this.conf.icase) flags += "i";
  var re = new RegExp(this.conf.regex, flags);

  if (!jpIn)
    throw new StepError("Data to transform not specified");
  if (!re)
    throw new StepError("Regular expression not specified or invalid");

  for (let i = 0; i < this.conf.outputs.length; i++) {
    var output = this.conf.outputs[i];
    jsonPathHelper.mapElements(jpIn, output.jp, function (value) {
      // preprocess whitespace
      if (context.conf.cleanWhitespace) value = context.normaliseWhitespace(value);

      // non-existent source treated as empty string for ^$ and other fun
      if (typeof(value) === "undefined") value = "";

      // determine if we should fail due to lack of match
      if (!re.test(value)) {
        if (context.conf.noMatchFail === true) {
          throw new StepError("Value '" + value + "' does not match regexp /"
                               + context.conf["regex"] + "/");
        } else {
          // leave target unchanged if instructed
          if (context.conf.noMatchNoop === true) return "##NOOP##";
          // otherwise pass the unmodified string
          else return value;
        }
      }

      // handle inline value replacement
      let replacement = jsonPathHelper.applyInline(output.inline, context.task.data);
      let newvalue = value.replace(re, replacement);

      // postprocess whitespace
      if (context.conf.cleanWhitespace)  newvalue = context.normaliseWhitespace(newvalue);

      // Transforming into nothing, return undefined to leave no trace.
      if (newvalue === "") return undefined;
      else return newvalue;
    }, task.data);
  }

  // Remove from source.
  if (this.conf.rmFromSource) {
    jsonPathHelper.mapElements(jpIn, jpIn, function(value) {
      // Don't create a source just to remove from
      if (typeof value === 'undefined') return undefined;
      let result = value.replace(re, "");
      if (context.conf.cleanWhitespace) result = context.normaliseWhitespace(result);
      if (result === "") return undefined;
      return result;
    }, task.data);
  }
};

Transform.prototype.getClassName = function () {
  return "Transform";
};

Transform.prototype.getDisplayName = function () {
  return "Transform";
};

Transform.prototype.getDescription = function () {
  return "Transforms a value (for example one extracted from a document or parsed out) by applying a regular expression to yield a modified version.";
};

Transform.prototype.getVersion = function () {
  return "3.1";
};

Transform.prototype.getUsedArgs = function () {
  return this.conf.usedArgs;
};

Transform.prototype.renderArgs = function () {
  if (!this.conf.in.key) return "";
  let flags = "";
  let replacement = "";
  let target = "";
  if (this.conf.icase) flags += "i";
  if (this.conf.global) flags += "g";
  if (this.conf.outputs.length === 1 && this.conf.outputs[0].replacement && this.conf.outputs[0].jp) {
    replacement = this.conf.outputs[0].replacement;
    target = this.conf.outputs[0].jp.key;
  } else if (this.conf.outputs.length > 1) {
    replacement = "<multiple>";
    target = this.conf.outputs.map(function (o) { return o.jp.key }).join(" ,");
  } else {
    return "";
  }
  return "s/" + this.conf.regex + "/" + replacement + "/" + flags + " (" + this.conf.in.key + "->" + target + ")";
};

Transform.prototype.upgrade = function (confVer, curVer, conf) {
  // can't upgrade if the connector is newer than the step
  if (confVer > curVer)
    return false;

  if (confVer < 0.2) {
    if (typeof conf['noMatchFail'] === "undefined") {
      if (conf['global']) {
        conf['noMatchFail'] = false;
      } else {
        conf['noMatchFail'] = true;
      }
    }
  }

  if (confVer < 0.3) {
    let container = this.conf.container;

    jsonPathHelper.upgradePostProc(this.conf);

    // Process inline args and rename replace to something less confusing
    this.conf.replacement = this.conf.replace.replace(/\$\{(.+?)\/(.+?)\}/g, "{$.$1.$2}");
    this.conf.replacement = this.conf.replacement.replace(/\$\{(.+?)\}/g, "{$.input.$1}");
    delete(this.conf.replace);

    // Result records had erroneously been handled somewhat differently
    if (typeof container === "string" && container !== "INPUT" && container !== "NONE") {
      this.conf.noMatchFail = false;
      if (this.conf.in.key !== this.conf.out.key)
        this.conf.out.append = jsonPathHelper.APPEND;
      else
        this.conf.out.append = jsonPathHelper.REPLACE;
    } else {
        this.conf.out.append = jsonPathHelper.REPLACE;
    }
  }

  if (confVer < 3.0) {
    this.conf.outputs = [{'replacement':this.conf.replacement, 'inline':this.conf.inline, 'jp':this.conf.out}];
    delete this.conf.replacement;
    delete this.conf.inline;
    delete this.conf.jpOut;
  }

  if (confVer < 3.1) this.conf.noMatchNoop = true;

  // Rebuild generated conf items
  this.processConf();
  return true;
};

// Capability flags: The (new) step default is good enough!
