/* This file is part of the Connector Framework
 * Copyright (C) 2008-2012 Index Data
 * See the file LICENSE for details.
 */
/**
 * \file cf_factory.cpp
 * \brief Engine factory (reads environment / defaults)
 */
#include <yaz/srw.h>
#include <yaz/wrbuf.h>
#include <yaz/comstack.h>
#include "cf_factory.h"
#include <stdlib.h>
#include <sys/stat.h>
#include <pwd.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <yaz/log.h>
#include <yaz/url.h>

CF_Factory::~CF_Factory()
{
    wrbuf_destroy(cf_module_path);
    wrbuf_destroy(cf_app_path);
    wrbuf_destroy(cf_profile_path);
    wrbuf_destroy(cf_connector_path);
    wrbuf_destroy(cf_repo_fetch_url);
    wrbuf_destroy(cf_repo_auth_url);
    wrbuf_destroy(tmp_dir);
}

CF_Factory::CF_Factory(const xmlNode *n)
{
    const char *base_path = 0;
    const char *module_path = 0;
    const char *connector_path = 0;
    const char *app_path = 0;
    const char *profile_path = 0;
    const char *repo_fetch_url = 0;
    const char *repo_auth_url = 0;
    const char *repo_proxy = 0;
    const char *tmp_dir = 0;

    base_path = getenv("CF_BASE_PATH");
    module_path = getenv("CF_MODULE_PATH");
    connector_path = getenv("CF_CONNECTOR_PATH");
    app_path = getenv("CF_APP_PATH");
    profile_path = getenv("CF_PROFILE_PATH");
    repo_fetch_url = getenv("CF_REPO_FETCH_URL");
    repo_auth_url = getenv("CF_REPO_AUTH_URL");
    repo_proxy = getenv("CF_REPO_PROXY");
    tmp_dir = getenv("CF_TMP_DIR");

    if (n)
    {
        const char *display = 0;
        const struct _xmlAttr *attr;
        const char *display_lock = 0;
        const char *display_cmd = 0;
        for (attr = n->properties; attr; attr = attr->next)
        {
            if (attr->children && attr->children->type == XML_TEXT_NODE)
            {
                if (!strcmp((const char *) attr->name, "base_path"))
                    base_path = (const char *) attr->children->content;
                else if (!strcmp((const char *) attr->name, "module_path"))
                    module_path = (const char *) attr->children->content;
                else if (!strcmp((const char *) attr->name, "connector_path"))
                    connector_path = (const char *) attr->children->content;
                else if (!strcmp((const char *) attr->name, "app_path"))
                    app_path = (const char *) attr->children->content;
                else if (!strcmp((const char *) attr->name, "profile_path"))
                    profile_path = (const char *) attr->children->content;
                else if (!strcmp((const char *) attr->name, "repo_fetch_url"))
                    repo_fetch_url = (const char *) attr->children->content;
                else if (!strcmp((const char *) attr->name, "repo_auth_url"))
                    repo_auth_url = (const char *) attr->children->content;
                else if (!strcmp((const char *) attr->name, "repo_proxy"))
                    repo_proxy = (const char *) attr->children->content;
                else if (!strcmp((const char *) attr->name, "display"))
                    display = (const char *) attr->children->content;
                else if (!strcmp((const char *) attr->name, "tmp_dir"))
                    tmp_dir = (const char *) attr->children->content;
                else if (!strcmp((const char *) attr->name, "display_lock"))
                    display_lock = (const char *) attr->children->content;
                else if (!strcmp((const char *) attr->name, "display_cmd"))
                    display_cmd = (const char *) attr->children->content;
                else
                    yaz_log(YLOG_WARN, "Unrecognized CF attribute %s",
                            (const char *) attr->name);
            }
        }
        if (display_lock)
            setenv("CF_DISPLAY_LOCK", display_lock, 1);
        if (display_cmd)
            setenv("CF_DISPLAY_CMD", display_cmd, 1);
        if (display)
            setenv("DISPLAY", display, 1);
    }

    if (repo_auth_url)
    {
        this->cf_repo_auth_url = wrbuf_alloc();
        wrbuf_puts(this->cf_repo_auth_url, repo_auth_url);
    }
    else
        this->cf_repo_auth_url = 0;

    if (repo_proxy)
    {
        this->cf_repo_proxy = wrbuf_alloc();
        wrbuf_puts(this->cf_repo_proxy, repo_proxy);
    }
    else
        this->cf_repo_proxy = 0;

    this->tmp_dir = wrbuf_alloc();
    wrbuf_puts(this->tmp_dir, tmp_dir ? tmp_dir : "/tmp/cf-engine");

    char cwd_buf[1024];
    if (getcwd(cwd_buf, sizeof(cwd_buf)-1) == 0)
    {
        yaz_log(YLOG_FATAL|YLOG_ERRNO, "getcwd");
        exit(1);
    }

    WRBUF base_path_wr = 0;
    if (!base_path && !module_path)
    {
        const char *suffix = "/engine/src";
        int off = strlen(cwd_buf) - strlen(suffix);
        if (off > 0 && strcmp(cwd_buf + off, suffix) == 0)
        {
            base_path_wr = wrbuf_alloc();
            wrbuf_write(base_path_wr, cwd_buf, off);
            base_path = wrbuf_cstr(base_path_wr);
        }
    }
    this->cf_module_path = wrbuf_alloc();
    if (module_path)
        wrbuf_puts(this->cf_module_path, module_path);
    else if (base_path)
        wrbuf_printf(this->cf_module_path,
                     "%s/builder/modules", base_path);
    else
    {
        yaz_log(YLOG_FATAL, "Missing module path.");
        yaz_log(YLOG_LOG, "CF_BASE_PATH or CF_MODULE_PATH must be set");
        exit(1);
    }

    if (repo_fetch_url)
    {
        this->cf_connector_path = 0;
        this->cf_repo_fetch_url = wrbuf_alloc();
        wrbuf_puts(this->cf_repo_fetch_url, repo_fetch_url);
    }
    else
    {
        this->cf_repo_fetch_url = 0;
        this->cf_connector_path = wrbuf_alloc();
        if (connector_path)
            wrbuf_puts(this->cf_connector_path, connector_path);
        else if (base_path)
            wrbuf_printf(this->cf_connector_path, "%s/connectors", base_path);
        else
            wrbuf_puts(this->cf_connector_path, cwd_buf);
    }
    if (app_path)
    {
        this->cf_app_path = wrbuf_alloc();
        wrbuf_puts(this->cf_app_path, app_path);
        if (!getenv("HOME"))
            setenv("HOME", app_path, 1);
    }
    else
        this->cf_app_path = 0;
    if (profile_path)
    {
        this->cf_profile_path = wrbuf_alloc();
        wrbuf_puts(this->cf_profile_path, profile_path);
    }
    else
        this->cf_profile_path = 0;
    wrbuf_destroy(base_path_wr);
}

static WRBUF get_url(const char *uri, WRBUF username, WRBUF password,
                     WRBUF proxy, int *code)
{
    WRBUF result = 0;
    yaz_url_t p = yaz_url_create();
    ODR odr = odr_createmem(ODR_ENCODE);
    Z_HTTP_Header *http_headers = 0;

    if (proxy)
        yaz_url_set_proxy(p, wrbuf_cstr(proxy));

    if (username && password)
    {
        z_HTTP_header_add_basic_auth(odr, &http_headers,
                                     wrbuf_cstr(username),
                                     wrbuf_cstr(password));
    }

    Z_HTTP_Response *http_response =
        yaz_url_exec(p, uri, "GET", http_headers, 0, 0);

    if (http_response)
    {
        *code = http_response->code;
        result = wrbuf_alloc();
        wrbuf_write(result,
                    http_response->content_buf, http_response->content_len);
    }
    odr_destroy(odr);
    yaz_url_destroy(p);
    return result;
}

static void wrbuf_subst(WRBUF v, const char *pattern, const char *replacement)
{
    WRBUF tmp = wrbuf_alloc();
    wrbuf_puts(tmp, wrbuf_cstr(v));

    const char *start_str = wrbuf_cstr(tmp);
    const char *subst_str = strstr(start_str, pattern);
    if (subst_str)
    {
        wrbuf_rewind(v);

        wrbuf_write(v, start_str, subst_str - start_str);
        wrbuf_puts(v, replacement);
        wrbuf_puts(v, subst_str + strlen(pattern));
    }
    wrbuf_destroy(tmp);
}

int CF_Factory::get_cf_tmp_dir(WRBUF w)
{
    wrbuf_puts(w, wrbuf_cstr(tmp_dir));

    uid_t euid = geteuid();
    char buf[2048];
    struct passwd pw1, *pw = 0;
    getpwuid_r(euid, &pw1, buf, sizeof(buf), &pw);
    if (!pw)
    {
        yaz_log(YLOG_FATAL|YLOG_ERRNO, "this getpwuid");
        return -1;
    }
    wrbuf_puts(w, "-");
    wrbuf_puts(w, pw->pw_name);

    if (mkdir(wrbuf_cstr(w), 0700) == -1 && errno != EEXIST)
    {
        yaz_log(YLOG_FATAL|YLOG_ERRNO, "mkdir %s", wrbuf_cstr(w));
        return -1;
    }
    return 0;
}

WRBUF CF_Factory::get_cf_via_url(WRBUF username,
                                 WRBUF password, const char *db, int *http_status)
{
    int success = 0;
    WRBUF cf_fname = wrbuf_alloc();

    if (get_cf_tmp_dir(cf_fname))
    {
        wrbuf_destroy(cf_fname);
        return 0;
    }
    if (username)
    {
        wrbuf_puts(cf_fname, "/");
        wrbuf_puts(cf_fname, wrbuf_cstr(username));
    }
    if (mkdir(wrbuf_cstr(cf_fname), 0700) == -1 &&
        errno != EEXIST)
    {
        yaz_log(YLOG_WARN|YLOG_ERRNO, "mkdir %s",
                wrbuf_cstr(cf_fname));
        wrbuf_destroy(cf_fname);
        return 0;
    }
    wrbuf_puts(cf_fname, "/");
    wrbuf_puts(cf_fname, db);
    wrbuf_puts(cf_fname, ".cf");

    struct stat statbuf;
    if (stat(wrbuf_cstr(cf_fname), &statbuf) == 0)
    {
        time_t now;

        time(&now);

        if (now >= statbuf.st_mtime && (now - statbuf.st_mtime) < 60)
            return cf_fname;
    }

    WRBUF uri = wrbuf_alloc();
    wrbuf_puts(uri, wrbuf_cstr(cf_repo_fetch_url));

    wrbuf_subst(uri, "%s", db);
    wrbuf_subst(uri, "%u", username ? wrbuf_cstr(username) : "");

    WRBUF cf_content =
        get_url(wrbuf_cstr(uri), username, password, cf_repo_proxy, http_status);
    yaz_log(YLOG_LOG, "GET %s %d", wrbuf_cstr(uri), *http_status);
    if (cf_content && *http_status == 200)
    {
        FILE *out = fopen(wrbuf_cstr(cf_fname), "wb");
        if (out)
        {
            size_t w = fwrite(wrbuf_buf(cf_content),
                              1, wrbuf_len(cf_content), out);
            if (w != wrbuf_len(cf_content))
                yaz_log(YLOG_WARN|YLOG_ERRNO, "fwrite %s",
                        wrbuf_cstr(cf_fname));
            if (fclose(out))
                yaz_log(YLOG_WARN|YLOG_ERRNO, "fclose %s",
                        wrbuf_cstr(cf_fname));
            else
                success = 1;
        }
    }
    wrbuf_destroy(cf_content);
    wrbuf_destroy(uri);
    if (success)
        return cf_fname;
    wrbuf_destroy(cf_fname);
    return 0;
}

WRBUF CF_Factory::get_cf_file(WRBUF username, WRBUF password, const char *db, int *http_status)
{
    if (strstr(db, "../"))
    {
        *http_status = 404;
        return 0;
    }
    if (cf_connector_path)
    {
        WRBUF cf_file = wrbuf_alloc();
        wrbuf_puts(cf_file, wrbuf_cstr(cf_connector_path));
        wrbuf_puts(cf_file, "/");
        wrbuf_puts(cf_file, db);
        wrbuf_puts(cf_file, ".cf");
        *http_status = 200; /* fake OK */
        return cf_file;
    }
    else if (cf_repo_fetch_url)
    {
        return get_cf_via_url(username, password, db, http_status);
    }
    else
        return 0;
}

CF_Engine *CF_Factory::create(enum CF_Engine::Flags flags, const char *proxy,
                              CF_Logger *logger, int fd)
{
    return new CF_Engine(
        flags,
        wrbuf_cstr(cf_module_path),
        (cf_app_path ? wrbuf_cstr(cf_app_path) : 0 ),
        (cf_profile_path ? wrbuf_cstr(cf_profile_path) : 0 ),
        proxy,
        logger,
        fd);
}

void CF_Factory::auth(WRBUF username, WRBUF password, int *http_status)
{
    if (!cf_repo_auth_url)
        *http_status = 200;
    else
    {
        WRBUF user_dir = wrbuf_alloc();

        *http_status = 500;
        if (get_cf_tmp_dir(user_dir) == 0)
        {
            // build the filename that holds password
            if (username)
            {
                wrbuf_puts(user_dir, "/");
                wrbuf_puts(user_dir, wrbuf_cstr(username));
            }
            if (mkdir(wrbuf_cstr(user_dir), 0700) == -1 &&
                errno != EEXIST)
            {
                yaz_log(YLOG_WARN|YLOG_ERRNO, "mkdir %s",
                        wrbuf_cstr(user_dir));
                wrbuf_destroy(user_dir);
                return;
            }
            wrbuf_puts(user_dir, "/password");
        }
        if (wrbuf_len(user_dir))
        {
            // cache user+password for a minute.
            struct stat statbuf;
            if (stat(wrbuf_cstr(user_dir), &statbuf) == 0)
            {
                time_t now;

                time(&now);

                if (now >= statbuf.st_mtime && (now - statbuf.st_mtime) < 60)
                {
                    FILE *f = fopen(wrbuf_cstr(user_dir), "r");
                    if (f)
                    {
                        char buf[1024];
                        size_t r = fread(buf, 1, sizeof(buf)-1, f);
                        if ((password && wrbuf_len(password) == r &&
                             memcmp(wrbuf_buf(password), buf, r) == 0)
                            || (!password && 0 == r))
                            *http_status = 200; // cached password
                        fclose(f);
                    }
                }
            }
        }
        if (*http_status != 200)
        {
            WRBUF content = get_url(wrbuf_cstr(cf_repo_auth_url),
                                    username, password,
                                    cf_repo_proxy, http_status);
            yaz_log(YLOG_LOG, "GET %s %d", wrbuf_cstr(cf_repo_auth_url),
                    *http_status);
            wrbuf_destroy(content);

            if (wrbuf_len(user_dir) && *http_status == 200)
            {
                FILE *f = fopen(wrbuf_cstr(user_dir), "w");
                if (!f)
                    yaz_log(YLOG_WARN|YLOG_ERRNO, "Could not create %s",
                            wrbuf_cstr(user_dir));
                else
                {
                    size_t wr =
                        fwrite(wrbuf_buf(password), 1, wrbuf_len(password), f);
                    if (wr != wrbuf_len(password))
                    {
                        fclose(f);
                        f = 0;
                    }
                    if (fclose(f))
                        f = 0;
                    if (!f)
                        unlink(wrbuf_cstr(user_dir));
                }
            }
        }
        wrbuf_destroy(user_dir);
    }
}

/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */

