/* This file is part of the Connector Framework
 * Copyright (C) 2008-2012 Index Data
 * See the file LICENSE for details.
 */

/** \file
 * \brief Z39.50 server talking to the CF
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include <yaz/log.h>
#include <yaz/backend.h>
#include <yaz/diagbib1.h>
#include <yaz/yaz-util.h>

#include <unistd.h>
#include <signal.h>
#include <yaz/snprintf.h>
#include <yaz/wrbuf.h>

#include "cf_factory.h"
#include "cf_metadata.h"
#include "cf_engine.h"
#include "cf_logger.h"

#include "json.h"
#include "rpn_to_json.h"
#include "type7_sort.h"
#include "database_args.h"
#include "cf_assert.h"

/** Z39.50 session info. */
struct session : CF_Logger
{
    CF_Engine *cf_engine; /**< Engine for session */
    char *database;       /**< Full database URI for current cf_engine */
    json_node *records;   /**< Records for last search */
    Odr_int hits;         /**< Number of hits for last search */
    char *subdatabase;    /**< from searchreq, dbname after '/' */
    WRBUF repo_username;
    WRBUF repo_password;
    WRBUF db_username;
    WRBUF db_password;
    WRBUF tmp_dir;
    FILE *engine_log;
    int log_fd;
    int no_searches;
    CF_Metadata metadata;
    void printf(const char *level, const char *fmt, ...);
};

/** Z39.50 session ptr. */
typedef struct session *sessionhandle;

/** Handle for the whole server. */
struct serverinfo
{
    char *cf_zauth;
    CF_Factory *cf_factory;
};


void session::printf(const char *level, const char *fmt, ...)
{
    va_list ap;
    char buf[4096];

    va_start(ap, fmt);
    yaz_vsnprintf(buf, sizeof(buf)-1, fmt, ap);

    const char *lead = "[JavaScript Warning: ";
    if (strncmp(lead, buf, strlen(lead)))
    {
        if (engine_log)
        {
            fputs(buf, engine_log);
            fputc('\n', engine_log);
            fflush(engine_log);
        }
        else
            yaz_log(YLOG_LOG, "%s %s", level, buf);
    }

    va_end(ap);
}


/** global server info */
struct serverinfo server_handle; /* read-only excent bend_{start,stop} */

static void connector_close(sessionhandle ses)
{
    if (ses->cf_engine)
    {
        try
        {
            const char *outp = 0;
            ses->cf_engine->run_task_opt("exit", "{}", &outp);
        }
        catch (CF_Engine_Error &e)
        {
            yaz_log(YLOG_WARN, "exit task returned error: %s", e.what());
        }
    }
    xfree(ses->database);
    ses->database = 0;
    delete ses->cf_engine;
    ses->cf_engine = 0;
    xfree(ses->subdatabase);
    ses->subdatabase = 0;
    ses->no_searches = 0;
}

static void connector_open(sessionhandle ses, bend_search_rr *rr)
{
    if (rr->num_bases != 1)
    {   // only one database may be given
        rr->errcode = YAZ_BIB1_COMBI_OF_SPECIFIED_DATABASES_UNSUPP;
        return;
    }   // see if we are still using the same database
    if (ses->cf_engine && ses->database
        && !strcmp(rr->basenames[0], ses->database)
        && ses->no_searches < 20) // reuse only up to a certain number
    {
        try
        {
            const char *outp = 0;
            ses->cf_engine->run_task_opt("exit", "{}", &outp);
        }
        catch (CF_Engine_Error &e)
        {
            yaz_log(YLOG_WARN, "exit task returned error: %s", e.what());
            connector_close(ses);
        }
        if (ses->cf_engine)
            return;
    }

    connector_close(ses);

    ses->database = xstrdup(rr->basenames[0]);

    // parse database
    // database?p1=v2&p2=v2&..
    const char *sep = strchr(ses->database, '?');
    if (!sep)
        sep = strchr(ses->database, ',');

    // must create a new cf_engine because database is not the same

    WRBUF cf_db = wrbuf_alloc();
    if (!sep)
        wrbuf_puts(cf_db, ses->database);
    else
        wrbuf_write(cf_db, ses->database, sep - ses->database);
    int http_status = 500;
    WRBUF cf_file = server_handle.cf_factory->get_cf_file(
        ses->repo_username, ses->repo_password, wrbuf_cstr(cf_db),
        &http_status);
    wrbuf_destroy(cf_db);

    if (!cf_file)
    {
        rr->errstring = odr_strdup(rr->stream, rr->basenames[0]);
        switch (http_status)
        {
        case 404:
            rr->errcode = YAZ_BIB1_DATABASE_UNAVAILABLE;
            break;
        case 401:
            rr->errcode = YAZ_BIB1_ACCESS_TO_SPECIFIED_DATABASE_DENIED;
            break;
        default:
            rr->errcode = YAZ_BIB1_TEMPORARY_SYSTEM_ERROR;
        }
        return;
    }
    WRBUF json = wrbuf_alloc();
    const char *proxy = 0;
    int r = cf_database_args(
        sep, rr->stream, &ses->subdatabase, &proxy, json,
        ses->db_username ? wrbuf_cstr(ses->db_username) : 0,
        ses->db_password ? wrbuf_cstr(ses->db_password) : 0);
    if (r)
    {
        rr->errcode = YAZ_BIB1_TEMPORARY_SYSTEM_ERROR;
        rr->errstring = (char *) odr_malloc(rr->stream, 40 + wrbuf_len(json));
        sprintf(rr->errstring, "Bad database argument: %s", wrbuf_cstr(json));
    }
    if (rr->errcode == 0)
    {
        // construct engine
        try
        {
            ses->cf_engine = server_handle.cf_factory->create(CF_Engine::FORK,
                                                              proxy,
                                                              ses,
                                                              ses->log_fd);
        }
        catch (CF_Engine_Error &e) {
            rr->errcode = YAZ_BIB1_TEMPORARY_SYSTEM_ERROR;
            rr->errstring = odr_strdup(rr->stream, e.what());
        }
    }
    if (rr->errcode == 0)
    {
        // load connector file and run init
        try
        {
            ses->cf_engine->load_cf(wrbuf_cstr(cf_file));
            const char *outp = 0;
            ses->cf_engine->run_task_opt("init", wrbuf_cstr(json), &outp);
        }
        catch (CF_Engine_Error_System &e)
        {
            rr->errcode = YAZ_BIB1_TEMPORARY_SYSTEM_ERROR;
            const char *what = e.what();
            if (what)
                rr->errstring = odr_strdup(rr->stream, what);
        }
        catch (CF_Engine_Error_Task &e)
        {
            const char *addinfo = 0;
            rr->errcode = YAZ_BIB1_UNSPECIFIED_ERROR;
            if (e.what())
            {
                yaz_log(YLOG_LOG, "init returned: %s", e.what());
            }
            cf_interpret_assertions(&rr->errcode, e.what(), &addinfo);
            if (addinfo)
                rr->errstring = odr_strdup(rr->stream, addinfo);
        }
    }
    ses->metadata.parse_cf(wrbuf_cstr(cf_file));
    wrbuf_destroy(json);
    wrbuf_destroy(cf_file);
}


/** The actual search function. */
static int my_search(void *handle, bend_search_rr *rr)
{
    if (strcmp(rr->setname, "default"))
    {
        rr->errcode = YAZ_BIB1_RESULT_SET_NAMING_UNSUPP;
        return 0;
    }
    sessionhandle ses = sessionhandle(handle);

    json_remove_node(ses->records);
    ses->records = 0;
    rr->hits = 0;
    ses->hits = 0;

    connector_open(ses, rr);
    if (rr->errcode)
        return 0;

    if (!ses->cf_engine)
    {
        yaz_log(YLOG_WARN, "No error and no engine");
        rr->errcode = YAZ_BIB1_TEMPORARY_SYSTEM_ERROR;
        return 0;
    }

    ses->no_searches++;

    /* The error handling is tricky:
       - parse full. Should not fail, so we can exit if it does.
       - parse regular. Save error code for later
       - Loop through jsons, try to search each
       - if a search fails, save the error for later (override parse errors
         except if error is 'no matching args').
       - If a search succeeds, return immediately
       - If nothing more to search, return saved error, or a new one.
    */
    CF_RPN_to_JSON rpn_to_json;
    rpn_to_json.set_capability_flags(ses->metadata.get("flags"));
    int errcode = 0;
    char *errmsg = 0; /* last error message, allocated on rr->stream */
    if (rr->query->which == Z_Query_type_1
        || rr->query->which == Z_Query_type_101)
    {
        NMEM nmem = ((ODR) rr->stream)->mem;
        struct sort_elem *sort_list = 0;
        type7_sort(rr->query->u.type_1, nmem, &sort_list);
        if ( !rpn_to_json.parse_full( rr->query->u.type_1,
                ses->subdatabase, sort_list )){
            // Should succeed for al reasonable queries. If not, we can
            // declare the whole search a failure.
            const char *addinfo;
            rr->errcode = rpn_to_json.get_diagnostic(&addinfo);
            if (addinfo)
                rr->errstring = odr_strdup(rr->stream, addinfo);
            yaz_log(YLOG_DEBUG,"fullquery parse failed with %d '%s",
                    rr->errcode, addinfo );
            return 0;
        }

        if ( ! rpn_to_json.parse(rr->query->u.type_1,
                ses->subdatabase, sort_list))
        {   // This can reasonably fail for queries that produced a
            //valid fullquery
            // Remember the error code for later
            const char *addinfo;
            errcode = rpn_to_json.get_diagnostic(&addinfo);
            errmsg = odr_strdup_null(rr->stream, addinfo);
            yaz_log(YLOG_DEBUG,"regular parse failed with %d '%s",
                    errcode, addinfo );
        }
    }
    else
    {
        rr->errcode = YAZ_BIB1_QUERY_TYPE_UNSUPP;
        rr->errstring = (char *)"Only type_1 and type_101 supported";
        return 0;
    }
    const char *outp = 0;
    int query_no = -1; // pass over multiple JSON alternatives
    for ( ; ; query_no++)
    {
        const char *json_res;
        if ( query_no == -1 )
            json_res = rpn_to_json.get_full();
        else
            json_res = rpn_to_json.get_result(query_no);
        if (!json_res) // nothing has succeeded
        {
            if (errcode)
            { // we have seen an error earlier
                rr->errcode = errcode;
                rr->errstring = errmsg;
            } else {
                rr->errcode = YAZ_BIB1_UNSUPP_USE_ATTRIBUTE;
                rr->errstring = 0;
            }
            return 0;
        }
        try
        {
            ses->cf_engine->run_task("search", json_res, &outp);
            break; // search succeeded, stop trying
        }
        catch (CF_Engine_Error_System &e)
        {
            const char *what = e.what();
            rr->errcode = YAZ_BIB1_TEMPORARY_SYSTEM_ERROR;
            if (what)
                rr->errstring = odr_strdup(rr->stream, what);
            return 0; // too serious error to keep trying
        }
        catch (CF_Engine_Error_Task &e)
        {  // Remember the error, but keep trying
           // Do not overwrite an earlier error (typically parse error),
           // if the error is "no matching args" (typically, unsupp fullq)
            const char *what = e.what();
            //json_res = rpn_to_json.get_result(1);
            yaz_log(YLOG_DEBUG,"Search (%d) failed: e='%s' old err=%d",
                    query_no, what, errcode);
            const char *lead = "Task: no implementation of task";
            if (strncmp(what, lead, strlen(lead)))
            {
                const char *addinfo = 0;
                errcode = YAZ_BIB1_TEMPORARY_SYSTEM_ERROR;
                cf_interpret_assertions(&errcode, what, &addinfo);
                errmsg = odr_strdup_null(rr->stream, addinfo);
            }
        }
    }
    if (rr->errcode)  // should not happen, we have returned at all errors
        return 0;

    const char *json_errmsg = 0;
    struct json_node *res = json_parse(outp, &json_errmsg);
    if (!res)
    {
        rr->errcode = YAZ_BIB1_TEMPORARY_SYSTEM_ERROR;
        rr->errstring = odr_strdup_null(rr->stream, json_errmsg);
        return 0;
    }
    struct json_node *hit = json_get_object(res, "hits");
    if (hit && hit->type == json_node_array)
        hit = json_get_elem(hit, 0);
    if (hit)
    {
        if (hit->type == json_node_number)
            ses->hits = (Odr_int) hit->u.number;
        else if (hit->type == json_node_string)
            ses->hits = odr_atoi(hit->u.string);
    }
    else
    {
        ses->hits = 100;
        rr->estimated_hit_count = 1;
    }
    json_remove_node(res);
    rr->hits = ses->hits;

    if (ses->hits == 0)
    {
        yaz_log(YLOG_LOG, "No hits");
        return 0; /* no need to get and parse them! */
    }
    try
    {
        ses->cf_engine->run_task("parse", "", &outp);
    }
    catch (CF_Engine_Error &e)
    {
        const char *what = e.what();
        rr->errcode = YAZ_BIB1_TEMPORARY_SYSTEM_ERROR;
        if (what)
            rr->errstring = odr_strdup(rr->stream, what);
    }

    res = json_parse(outp, 0);
    struct json_node *rp = json_detach_object(res, "results");
    if (rp)
    {
        json_parser_t parser = json_parser_create();

        json_parser_subst(parser, 0, rp);

        ses->records = json_parser_parse(parser,
                                         "{\"records\":%0}");
        json_parser_destroy(parser);
    }
    json_remove_node(res);
    return 0;
}

static bool get_next(sessionhandle ses)
{
    const char *outp = 0;
    try
    {
        ses->cf_engine->run_task("next", "", &outp);
        outp = 0;
        ses->cf_engine->run_task("parse", "", &outp);
    }
    catch (CF_Engine_Error &e)
    {
        ;
    }

    struct json_node *n_out = json_parse(outp, 0);
    struct json_node *n_results = json_detach_object(n_out, "results");

    if (n_results && n_results->type == json_node_array)
    {
        json_append_array(json_get_object(ses->records, "records"), n_results);
        json_remove_node(n_out);
        return true;
    }
    json_remove_node(n_out);
    json_remove_node(n_results);
    return false;
}

static int json_append_object(struct json_node *dst, struct json_node *src)
{
 if (dst && src &&
        dst->type == json_node_object && src->type == json_node_object)
    {
        struct json_node **np = &dst->u.link[0];
        while (*np)
            np = &(*np)->u.link[1];
        *np = src->u.link[0];
        src->u.link[0] = 0;
        json_remove_node(src);
        return 0;
    }
    return -1;
}

static struct json_node *get_detailed_record(sessionhandle ses, int n,
                                             int *errcode,
                                             struct json_node *record_r)
{
    WRBUF wrb = wrbuf_alloc();
    struct json_node *detail_x;
    struct json_node *detail_rec = 0;

    wrbuf_printf(wrb, "details%d", n);
    detail_x = json_get_object(ses->records, wrbuf_cstr(wrb));
    if (detail_x)
        detail_rec = json_get_elem(detail_x, 0);
    else
    {
        wrbuf_rewind(wrb);
        wrbuf_puts(wrb, "{\"record\":");
        json_write_wrbuf(record_r, wrb);
        wrbuf_puts(wrb, "}");

        const char *outp = 0;
        try
        {
            ses->cf_engine->run_task_opt("details", wrbuf_cstr(wrb), &outp);
        }
        catch (CF_Engine_Error &e)
        {
            const char *what = e.what();
            *errcode = YAZ_BIB1_SYSTEM_ERROR_IN_PRESENTING_RECORDS;
            yaz_log(YLOG_WARN, "details task failed: %s",
                    what ? what : "null");
            record_r = 0;
        }
        if (!outp || !strcmp(outp, "{}"))
            return record_r;
        struct json_node *n_out = json_parse(outp, 0);
        struct json_node *n_results = json_detach_object(n_out, "results");

        json_remove_node(n_out);

        if (n_results && n_results->type == json_node_array)
        {
            json_parser_t parser = json_parser_create();

            wrbuf_rewind(wrb);
            wrbuf_printf(wrb, "{\"details%d\":", n);
            wrbuf_puts(wrb, "%0}");

            json_parser_subst(parser, 0, n_results);

            detail_x = json_parser_parse(parser, wrbuf_cstr(wrb));

            json_parser_destroy(parser);

            if (detail_x)
                json_append_object(ses->records, detail_x);
            detail_rec = json_get_elem(n_results, 0);
        }
    }
    wrbuf_destroy(wrb);
    return detail_rec;
}


static struct json_node *get_record(sessionhandle ses, int n,
                                    int *errcode, int details)
{
    struct json_node *r;
    if (!ses->records)
    {
        *errcode = YAZ_BIB1_PRESENT_REQUEST_OUT_OF_RANGE;
        yaz_log(YLOG_WARN, "get_record: no records (1)");
        return 0;
    }
    struct json_node *recs = json_get_object(ses->records, "records");
    if (!recs)
    {
        *errcode = YAZ_BIB1_SYSTEM_ERROR_IN_PRESENTING_RECORDS;
        yaz_log(YLOG_WARN, "get_record: no records (2)");
        return 0;
    }
    if (n > ses->hits)
        return 0; // don't even try to get more than hitcount
    int rc = json_count_children(recs);
    while (n > rc)
    {
        if (!get_next(ses))
        {
            yaz_log(YLOG_WARN, "get_record: position=%d, max=%d", n, rc);
            ses->hits = n;  // no need to try this range again, bug 4262
            return 0;
        }
        int lastrc = rc;
        rc = json_count_children(recs);
        if (rc == lastrc)
        {
            yaz_log(YLOG_WARN, "get_record: no records although next was ok");
            return 0;
        }
    }
    r = json_get_elem(recs, n - 1);
    if (r && details)
        return get_detailed_record(ses, n, errcode, r);
    return r;
}

static bool get_record_xml(sessionhandle ses, int n, WRBUF wrb, int *errcode,
                           int details)
{
    struct json_node *r = get_record(ses, n, errcode, details);
    if (!r)
        return false;
    wrbuf_puts(wrb, "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n");
    wrbuf_puts(wrb, "<record>\n");
    make_xml_al(r, wrb, 0);
    wrbuf_puts(wrb, "</record>\n");
    return true;
}

/** retrieval of a single record (present, and piggy back search) */
static int my_fetch(void *handle, bend_fetch_rr *r)
{
    const Odr_oid *oid = r->request_format;
    sessionhandle ses= sessionhandle(handle);
    int details = 0;
    Z_RecordComposition *comp = r->comp;
    const char *element_set_name = 0;

    r->last_in_set = 0;
    r->basename = ses->database;

    if (comp && comp->which == Z_RecordComp_simple &&
        comp->u.simple->which == Z_ElementSetNames_generic)
        element_set_name = comp->u.simple->u.generic;
    else if (comp && comp->which == Z_RecordComp_complex &&
        comp->u.complex->generic->which == Z_Schema_uri)
        element_set_name = comp->u.complex->generic->schema.uri;
    if (element_set_name && *element_set_name == 'F')
        details = 1;

    /* if no record syntax was given assume XML */
    if (!oid || !oid_oidcmp(oid, yaz_oid_recsyn_xml))
    {
        WRBUF wrb = wrbuf_alloc();
        if (get_record_xml(ses, r->number, wrb, &r->errcode, details))
        {
            r->record = odr_strdup(r->stream, wrbuf_cstr(wrb));
            r->len = strlen(r->record);
        }
        wrbuf_destroy(wrb);
        r->output_format = odr_oiddup(r->stream, yaz_oid_recsyn_xml);
    }
    else if (!oid_oidcmp(oid, yaz_oid_recsyn_json))
    {
        struct json_node *json = get_record(ses, r->number, &r->errcode, details);
        if (json)
        {
            WRBUF wrb = wrbuf_alloc();
            json_write_wrbuf(json, wrb);
            r->record = odr_strdup(r->stream, wrbuf_cstr(wrb));
            r->len = strlen(r->record);
            r->output_format = odr_oiddup(r->stream, yaz_oid_recsyn_json);
            wrbuf_destroy(wrb);
        }
    }
    else
    { /* only XML and JSON syntax supported . Return diagnostic */
        char buf[OID_STR_MAX];
        r->errcode = YAZ_BIB1_RECORD_SYNTAX_UNSUPP;
        r->errstring = odr_strdup(r->stream, oid_oid_to_dotstring(oid, buf));
    }
    return 0;
}

static void read_zauth(sessionhandle ses, const Z_IdAuthentication *auth)
{
    WRBUF z_username = 0;
    WRBUF z_password = 0;
    if (auth)
    {
        switch (auth->which)
        {
        case Z_IdAuthentication_open:
            if (auth->u.open)
            {
                const char *sep = strchr(auth->u.open, '/');
                if (sep)
                {
                    z_username = wrbuf_alloc();
                    z_password = wrbuf_alloc();
                    wrbuf_write(z_username,
                                auth->u.open, sep - auth->u.open);
                    wrbuf_puts(z_password, sep+1);
                }
            }
            break;
        case Z_IdAuthentication_idPass:
            if (auth->u.idPass)
            {
                Z_IdPass *idPass = auth->u.idPass;
                // we are not using idPass->groupId
                if (idPass->userId)
                {
                    z_username = wrbuf_alloc();
                    wrbuf_puts(z_username, idPass->userId);
                }
                if (idPass->password)
                {
                    z_password = wrbuf_alloc();
                    wrbuf_puts(z_password, idPass->password);
                }
            }
        }
    }
    if (server_handle.cf_zauth)
    {
        const char *s = server_handle.cf_zauth;
        const char *sep = strchr(s, '/');
        if (sep)
        {
            ses->repo_username = wrbuf_alloc();
            ses->repo_password = wrbuf_alloc();
            wrbuf_write(ses->repo_username, s, sep - s);
            wrbuf_puts(ses->repo_password, sep+1);
        }
        ses->db_username = z_username;
        ses->db_password = z_password;
    }
    else
    {
        ses->repo_username = z_username;
        ses->repo_password = z_password;
    }
}

static bend_initresult *my_init(bend_initrequest *q)
{
    bend_initresult *r = (bend_initresult *)
        odr_malloc(q->stream, sizeof (*r));

    r->errcode = 0;
    r->errstring = 0;
    q->bend_search = my_search;
    q->bend_fetch = my_fetch;
    q->query_charset = (char *) "UTF-8";
    q->records_in_same_charset = 1;
    q->implementation_name = odr_strdup(q->stream, "CF:" CF_VERSION);
    q->named_result_sets = 0;  // we do not support named result sets

    sessionhandle ses = new session;
    ses->cf_engine = 0;
    ses->database = 0;
    ses->records = 0;
    ses->hits = 0;
    ses->subdatabase = 0;
    ses->repo_username = 0;
    ses->repo_password = 0;
    ses->db_username = 0;
    ses->db_password = 0;
    ses->engine_log = 0;
    ses->log_fd = -1;
    ses->no_searches = 0;
    ses->tmp_dir = wrbuf_alloc();
    server_handle.cf_factory->get_cf_tmp_dir(ses->tmp_dir);

    {
        const char *cf_log_str = getenv("CF_LOG");

        if (cf_log_str && *cf_log_str == '1')
        {
            char fname[100];
            sprintf(fname, "%s/cf.%lld.log", wrbuf_cstr(ses->tmp_dir),
                    (long long) getpid());
            yaz_log(YLOG_LOG, "create %s", fname);
            ses->engine_log = fopen(fname, "w");
            if (ses->engine_log)
                ses->log_fd = fileno(ses->engine_log);
        }
        else
        {
            FILE *yf = yaz_log_file();
            if (yf && fileno(yf) != 2)
                ses->log_fd = fileno(yf);
        }
    }

    read_zauth(ses, q->auth);
    r->handle = ses;

    int http_status = 500;
    server_handle.cf_factory->auth(
        ses->repo_username, ses->repo_password, &http_status);
    switch (http_status)
    {
    case 200: /* OK */
        break;
    case 401:
        r->errcode = YAZ_BIB1_INIT_AC_BAD_USERID_AND_OR_PASSWORD;
        break;
    default:
        r->errcode = YAZ_BIB1_INIT_AC_AUTHENTICATION_SYSTEM_ERROR;
    }
    return r;
}

/** \brief close handler */
static void my_close(void *handle)
{
    sessionhandle ses = sessionhandle(handle);

    connector_close(ses);

    if (ses->engine_log)
    {
        fclose(ses->engine_log);
        ses->engine_log = 0;
    }

    wrbuf_destroy(ses->tmp_dir);
    json_remove_node(ses->records);
    wrbuf_destroy(ses->repo_username);
    wrbuf_destroy(ses->repo_password);
    wrbuf_destroy(ses->db_username);
    wrbuf_destroy(ses->db_password);
    delete ses;
    return;
}

static void bend_start(struct statserv_options_block *sob)
{
    const char *cp;
    yaz_log(YLOG_LOG, "cf-zserver start %s", CF_VERSION);
    server_handle.cf_factory = new CF_Factory(0);
    cp = getenv("CF_ZAUTH");
    server_handle.cf_zauth = cp ? xstrdup(cp) : 0;

    if (getenv("CF_XVFB_LOCKDIR"))
    {
        yaz_log(YLOG_FATAL, "CF_XVFB_LOCKDIR is no longer supported");
        yaz_log(YLOG_LOG, "Use CF_DISPLAY_LOCK instead");
        exit(1);
    }
}

static void bend_stop(struct statserv_options_block *sob)
{
    yaz_log(YLOG_LOG, "cf-zserver stop");
    delete server_handle.cf_factory;
    xfree(server_handle.cf_zauth);
}

static void sig_usr1_handler(int s)
{
    yaz_log(YLOG_LOG, "cf-zserver received SIGUSR1");
    yaz_log(YLOG_LOG, "cf-zserver stop");
    _exit(0);
}

static void sig_term_handler(int s)
{
    yaz_log(YLOG_LOG, "cf-zserver received SIGTERM");
    yaz_log(YLOG_LOG, "cf-zserver stop");
    _exit(0);
}

/** \brief main as usual */
int main(int argc, char **argv)
{
    struct statserv_options_block *sob = statserv_getcontrol();

    signal(SIGTERM, sig_term_handler);
    signal(SIGUSR1, sig_usr1_handler);

    sob->bend_start = bend_start;
    sob->bend_stop = bend_stop;
    /* Save arguments for future reference */
    statserv_setcontrol(sob);
    return statserv_main(argc, argv, my_init, my_close);
}
/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */

