/* This file is part of the Connector Framework
 * Copyright (C) 2008-2012 Index Data
 * See the file LICENSE for details.
 */

/** \file
 * \brief Parse database arguments (for init)
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <yaz/json.h>
#include <yaz/xmalloc.h>
#include <yaz/srw.h>
#include "database_args.h"

static void mksep(int &sep, WRBUF w)
{
    if (sep)
        wrbuf_puts(w, ",");
    sep = 1;
}
int cf_database_args(const char *url_arg,
                     ODR odr,
                     char **subdatabase,
                     const char **proxy,
                     WRBUF result,
                     const char *default_user,
                     const char *default_password)
{
    const char *param_user = default_user;
    const char *param_group = 0;
    const char *param_password = default_password;
    const char *param_proxy = 0;
    const char *param_session = 0;
    bool nocproxy = false;
    int sep = 0;

    wrbuf_puts(result, "{");
    if (url_arg)
    {
        // parse parameters
        char **names;
        char **values;
        int no_parms = yaz_uri_to_array(url_arg + 1, odr, &names, &values);
        int i;
        for (i = 0; i < no_parms; i++)
        {
            const char *name = names[i];
            const char *value = values[i];
            if (!strcmp(name, "user"))
                param_user = value;
            else if (!strcmp(name, "group"))
                param_group = value;
            else if (!strcmp(name, "password"))
                param_password = value;
            else if (!strcmp(name, "proxy"))
            {
                param_proxy = value;
                *proxy = value;
            }
            else if (!strcmp(name, "cproxysession"))
                param_session = value;
            else if (!strcmp(name, "nocproxy"))
                nocproxy = true;
            else if (!strcmp(name, "subdatabase"))
            {
                mksep(sep, result);
                wrbuf_puts(result, "\"database\":\"");
                wrbuf_json_puts(result, value);
                wrbuf_puts(result, "\"");
                *subdatabase = xstrdup(value);
            }
            else if (name[0] == 'x' && name[1] == '-')
            {
                mksep(sep, result);
                wrbuf_puts(result, "\"");
                wrbuf_puts(result, name);
                wrbuf_puts(result, "\":\"");
                wrbuf_json_puts(result, value);
                wrbuf_puts(result, "\"");
            }
            else
            {
                wrbuf_rewind(result);
                wrbuf_puts(result, name);
                return -1;
            }
        }
    }
    if (param_user)
    {
        mksep(sep, result);
        wrbuf_puts(result, "\"username\":\"");
        wrbuf_json_puts(result, param_user);
        wrbuf_puts(result, "\"");
    }
    if (param_group)
    {
        mksep(sep, result);
        wrbuf_puts(result, "\"group\":\"");
        wrbuf_json_puts(result, param_group);
        wrbuf_puts(result, "\"");
    }
    if (param_password)
    {
        mksep(sep, result);
        wrbuf_puts(result, "\"password\":\"");
        wrbuf_json_puts(result, param_password);
        wrbuf_puts(result, "\"");
    }
    if (nocproxy)
    {
        mksep(sep, result);
        wrbuf_puts(result, "\"nocproxy\":\"1\"");
    }
    if (param_session)
    {
        mksep(sep, result);
        wrbuf_puts(result, "\"cproxysession\":\"");
        wrbuf_json_puts(result, param_session);
        wrbuf_puts(result, "\"");
    }
    if (param_proxy)
    {
        mksep(sep, result);
        wrbuf_puts(result, "\"proxyip\":\"");
        wrbuf_json_puts(result, param_proxy);
        wrbuf_puts(result, "\"");
    }
    wrbuf_puts(result, "}");
    return 0;
}

/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */
