/* This file is part of the Connector Framework
 * Copyright (C) 2008-2012 Index Data
 * See the file LICENSE for details.
 */

/** \file
 * \brief Web Service and Z39.50 server
 */

#include <metaproxy/package.hpp>
#include <metaproxy/util.hpp>


#include <stdexcept>
#include <list>
#include <map>
#include <iostream>
#include <string.h>

#include <boost/thread/mutex.hpp>
#include <boost/thread/condition.hpp>
#include <boost/thread/thread.hpp>
#include <boost/thread/xtime.hpp>
#include <boost/scoped_ptr.hpp>

#include <yaz/zgdu.h>
#include <yaz/srw.h>
#include <yaz/log.h>
#include <yaz/otherinfo.h>
#include <yaz/diagbib1.h>
#include <yaz/oid_db.h>
#include <yaz/snprintf.h>

#include "cf_assert.h"
#include "cf_factory.h"
#include "cf_logger.h"
#include "cf_metadata.h"
#include "json.h"
#include "rpn_to_json.h"
#include "type7_sort.h"
#include "database_args.h"

namespace mp = metaproxy_1;
namespace yf = mp::filter;
using namespace mp;

namespace metaproxy_1 {
    namespace filter {
        class CF : public Base {
            class Rep;
            class Connector;
            class Session;
            class ZSession;
            class Worker;
        public:
            ~CF();
            CF();
            void process(metaproxy_1::Package & package) const;
            void configure(const xmlNode * ptr, bool test_only,
                           const char *path);
            void start() const;
        private:
            boost::scoped_ptr<Rep> m_p;
        };
        class CF::Session : CF_Logger {
            friend class CF;
            friend class Rep;
            CF_Engine *engine;
            int age;
            std::string session_name;
            bool in_use;
            bool destroyed;
            FILE *log_file;
            mp::wrbuf file_area;
            Session(std::string session_name);
            ~Session();
            void log(int level, const char *fmt, ...);
            CF_Metadata metadata;
        public:
            void printf(const char *level, const char *fmt, ...);
        };

        class CF::ZSession {
            friend class Rep;
            bool in_use;

            char *database;
            json_node *records;
            Odr_int hits;
            char *subdatabase;
            WRBUF username;
            WRBUF password;
            CF::Session *cf_session;
            int no_searches;
            bool get_next();
            struct json_node *get_record(int n, int *errcode);
            bool get_record_xml(int n, WRBUF wrb, int *errcode);

            ZSession(const Z_IdAuthentication *auth);
            ~ZSession();
            void close();
        };

        class CF::Worker {
        public:
            Worker(CF::Rep *rep);
            void operator() (void);
        private:
            CF::Rep *m_p;
        };
        class CF::Rep {
            friend class CF;

            void process_connector(metaproxy_1::Package & package,
                                   const char *suffix_path);


            Z_Records *fetch_z3950(
                CF::ZSession *zsession,
                mp::odr &odr, Odr_oid *preferredRecordSyntax,
                Z_ElementSetNames *esn,
                int start, int number, int &error_code, std::string &addinfo,
                int *number_returned, int *next_position);

            void process_z3950(metaproxy_1::Package & package,
                               CF::ZSession *zsession);

            Z_APDU *open_cf_z3950(Z_APDU *req,  CF::ZSession *zsession,
                                  mp::odr &odr, unsigned long session_id);
            Z_APDU *search_z3950(Z_APDU *apdu_req, CF::ZSession *zsession,
                                 mp::odr &odr, unsigned long session_id);
            Z_APDU *present_z3950(Z_APDU *apdu_req, CF::ZSession *zsession,
                                  mp::odr &odr);
            void post_connector(Package &package, Z_HTTP_Request *req,
                                const char *suffix_path);

            CF_Engine *create_connector(CF_Engine::Flags engine_flag,
                                        const char *proxy, CF::Session *ses,
                                        std::string &addinfo);

            bool load_connector(Package &package, Z_HTTP_Request *req,
                                CF::Session *cf_session);

            void task_connector(Package &package, Z_HTTP_Request *req,
                                CF::Session *cf_session,
                                const char *args);
            void expire(void);

            CF::ZSession *get_zsession(Package &package);
            void release_zsession(Package &package);

            std::map<int,Session *> m_connectors;

            std::map<mp::Session,ZSession *> m_clients;

            std::string http_prefix;

            bool enable_z3950;

            CF_Factory *cf_factory;

            int max_age;
            int m_connector_next_id;

            boost::condition m_cond_session_ready;
            boost::mutex m_mutex; // protects Rep (m_connectors, id)
            boost::thread_group m_thrds;
        public:
            Rep();
            ~Rep();
        };
    }
}

static void cf_http_msg(Z_GDU *gdu, ODR o, const char *msg, int code)
{
    Z_HTTP_Response *hres = gdu->u.HTTP_Response;

    hres->content_buf = odr_strdup(o, msg);
    hres->content_len = strlen(hres->content_buf);
    hres->code = code;

    z_HTTP_header_add(o, &hres->headers, "Content-Type", "text/plain");
}

yf::CF::ZSession::ZSession(const Z_IdAuthentication *auth)
{
    in_use = true;

    database = 0;
    records = 0;
    hits = 0;
    subdatabase = 0;
    username = 0;
    password = 0;
    cf_session = 0;
    no_searches = 0;

    if (auth)
    {
        switch (auth->which)
        {
        case Z_IdAuthentication_open:
            if (auth->u.open)
            {
                const char *sep = strchr(auth->u.open, '/');
                if (sep)
                {
                    username = wrbuf_alloc();
                    password = wrbuf_alloc();
                    wrbuf_write(username,
                                auth->u.open, sep - auth->u.open);
                    wrbuf_puts(password, sep+1);
                }
            }
            break;
        case Z_IdAuthentication_idPass:
            if (auth->u.idPass)
            {
                Z_IdPass *idPass = auth->u.idPass;
                // we are not using idPass->groupId
                if (idPass->userId)
                {
                    username = wrbuf_alloc();
                    wrbuf_puts(username, idPass->userId);
                }
                if (idPass->password)
                {
                    password = wrbuf_alloc();
                    wrbuf_puts(password, idPass->password);
                }
            }
        }
    }
}

yf::CF::ZSession::~ZSession()
{
    close();
    wrbuf_destroy(username);
    wrbuf_destroy(password);
    json_remove_node(records);
}


void yf::CF::ZSession::close()
{
    if (cf_session && cf_session->engine)
    {
        const char *output_json = 0;
        try {
            cf_session->engine->run_task_opt("exit", "{}", &output_json);
        }
        catch (CF_Engine_Error &e)
        {
            yaz_log(YLOG_WARN, "run_task_opt exit failed: %s", e.what());
        }
    }

    delete cf_session;
    cf_session = 0;
    xfree(database);
    xfree(subdatabase);
    database = 0;
    subdatabase = 0;
    no_searches = 0;
}

bool yf::CF::ZSession::get_next()
{
    const char *outp = 0;
    try
    {
        cf_session->engine->run_task("next", "", &outp);
        outp = 0;
        cf_session->engine->run_task("parse", "", &outp);
    }
    catch (CF_Engine_Error &e)
    {
        ;
    }

    struct json_node *n_out = json_parse(outp, 0);
    struct json_node *n_results = json_detach_object(n_out, "results");

    if (n_results && n_results->type == json_node_array)
    {
        json_append_array(json_get_object(records, "records"), n_results);
        json_remove_node(n_out);
        return true;
    }
    json_remove_node(n_out);
    json_remove_node(n_results);
    return false;
}

struct json_node * yf::CF::ZSession::get_record(int n, int *errcode)
{
    if (!records)
    {
        *errcode = YAZ_BIB1_PRESENT_REQUEST_OUT_OF_RANGE;
        cf_session->log(YLOG_WARN, "get_record: no records (1)");
        return 0;
    }
    struct json_node *recs = json_get_object(records, "records");
    if (!recs)
    {
        *errcode = YAZ_BIB1_SYSTEM_ERROR_IN_PRESENTING_RECORDS;
        cf_session->log(YLOG_WARN, "get_record: no records (2)");
        return 0;
    }
    if (n > hits)
        return 0; // don't even try to get more than hitcount
    int rc = json_count_children(recs);
    while (n > rc)
    {
        if (!get_next())
        {
            cf_session->log(YLOG_WARN, "get_record: position=%d, max=%d",
                            n, rc );
            hits = n; // no need to try this range again, bug 4262
            return 0;
        }
        int lastrc = rc;
        rc = json_count_children(recs);
        if (rc == lastrc)
        {
            cf_session->log(YLOG_WARN, "get_record: no records although "
                            "next was OK");
            return 0;
        }
    }
    return json_get_elem(recs, n - 1);
}

bool yf::CF::ZSession::get_record_xml(int n, WRBUF wrb, int *errcode)
{
    struct json_node *r = get_record(n, errcode);
    if (!r)
        return false;
    wrbuf_puts(wrb, "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n");
    wrbuf_puts(wrb, "<record>\n");
    make_xml_al(r, wrb, 0);
    wrbuf_puts(wrb, "</record>\n");
    return true;
}


void yf::CF::Session::printf(const char *level, const char *fmt, ...)
{
    va_list ap;
    char buf[4096];

    va_start(ap, fmt);
    yaz_vsnprintf(buf, sizeof(buf)-1, fmt, ap);

    const char *lead = "[JavaScript Warning: ";
    if (strncmp(lead, buf, strlen(lead)))
    {
        yaz_log(YLOG_LOG, "%s %s %s", session_name.c_str(), level, buf);
        if (log_file)
        {
            fprintf(log_file, "%s %s\n", level, buf);
            fflush(log_file);
        }
    }
    va_end(ap);
}

void yf::CF::Session::log(int level, const char *fmt, ...)
{
    va_list ap;
    char buf[4096];

    va_start(ap, fmt);
    yaz_vsnprintf(buf, sizeof(buf)-1, fmt, ap);
    yaz_log(level, "%s %s", session_name.c_str(), buf);
    if (log_file)
        fprintf(log_file, "%s\n", buf);

    va_end(ap);
}

yf::CF::Session::Session(std::string a_session_name)
    : session_name(a_session_name)
{
    this->engine = 0;
    this->log_file = 0;
    this->in_use = true;
    this->destroyed = false;
    this->age = 0;
}

yf::CF::Session::~Session()
{
    delete engine;
    if (log_file)
        fclose(log_file);
}

yf::CF::Rep::Rep()
{
    cf_factory = 0;
    http_prefix = "connector";
    max_age = 5;
    enable_z3950 = false;
    m_connector_next_id = 1;
}

yf::CF::Rep::~Rep()
{
    delete cf_factory;
}

yf::CF::CF() : m_p(new CF::Rep)
{

}

yf::CF::~CF()
{

}

yf::CF::Worker::Worker(CF::Rep *rep) : m_p(rep)
{
}

void yf::CF::Worker::operator() (void)
{
    m_p->expire();
}

void yf::CF::Rep::expire(void)
{
    while (true)
    {
        boost::xtime xt;
        boost::xtime_get(&xt, boost::TIME_UTC);
        xt.sec += 60;
        boost::thread::sleep(xt);

        {
            boost::mutex::scoped_lock lock(m_mutex);

            std::map<int,Session *>::iterator map_it;
            map_it = m_connectors.begin();
            while (map_it != m_connectors.end())
            {
                Session *ses = map_it->second;
                if (!ses->in_use)
                {
                    ses->age++;
                    if (ses->age > max_age)
                    {
                        m_connectors.erase(map_it);
                        delete ses;
                        map_it = m_connectors.begin();
                    }
                    else
                        map_it++;
                }
                else
                    map_it++;
            }
        }
    }
}

CF_Engine *yf::CF::Rep::create_connector(CF_Engine::Flags engine_flag,
                                         const char *proxy, CF::Session *ses,
                                         std::string &addinfo)
{
    if (cf_factory->get_cf_tmp_dir(ses->file_area))
    {
        /* failed to get tmpdir */
        ses->log(YLOG_WARN, "Could not get tmpdir");
        addinfo = "tmpdir problem";
        return 0;
    }
    wrbuf_printf(ses->file_area, "/%s", ses->session_name.c_str());

    mp::wrbuf cf_log_fname;
    wrbuf_printf(cf_log_fname, "%s.log", wrbuf_cstr(ses->file_area));
    FILE *log_file = fopen(wrbuf_cstr(cf_log_fname), "w+");
    ses->log_file = log_file;
    if (!log_file)
    {
        addinfo = "could not open ";
        addinfo += wrbuf_cstr(cf_log_fname);
        ses->log(YLOG_WARN|YLOG_ERRNO, "could not open %s",
                 wrbuf_cstr(cf_log_fname));
        return 0;
    }

    CF_Engine *cf = 0;
    try {
        cf = cf_factory->create(engine_flag, proxy, ses, fileno(log_file));
    }
    catch (CF_Engine_Error &e)
    {
        addinfo = e.what();
        ses->log(YLOG_WARN, "Could not construct CF_Engine: %s",
                e.what());
        return 0;
    }
    ses->engine = cf;
    return cf;
}

bool yf::CF::Rep::load_connector(Package &package, Z_HTTP_Request *req,
                                 CF::Session *ses)
{

    if (req->content_len > 0)
    {   // empty file is a NO-OP
        mp::wrbuf cf_fname;
        const char *conn_args = z_HTTP_header_lookup(req->headers, "X-CF-Args");
        mp::odr odr;

        wrbuf_printf(cf_fname, "%s.cf", wrbuf_cstr(ses->file_area));
        FILE *cf_file = fopen(wrbuf_cstr(cf_fname), "wb");
        if (!cf_file)
        {
            Z_GDU *zgdu_res
                = odr.create_HTTP_Response(package.session(), req, 200);
            ses->log(YLOG_WARN|YLOG_ERRNO, "fopen %s", wrbuf_cstr(cf_fname));
            cf_http_msg(zgdu_res, odr, "can not create temporary CF file", 500);
            package.response() = zgdu_res;
            return false;
        }
        fwrite(req->content_buf, 1, req->content_len, cf_file);
        fclose(cf_file);

        try {
            ses->engine->load_cf(wrbuf_cstr(cf_fname), conn_args);
        }
        catch (CF_Engine_Error &e) {
            std::string addinfo;
            addinfo = std::string("load ") + wrbuf_cstr(cf_fname) +
                std::string(" failed: ") + e.what();
            ses->log(YLOG_WARN, "CF_Engine. %s", addinfo.c_str());
            Z_GDU *zgdu_res
                = odr.create_HTTP_Response(package.session(), req, 200);
            cf_http_msg(zgdu_res, odr, addinfo.c_str(), 500);
            package.response() = zgdu_res;
            return false;
        }
    }
    return true;
}

/** \brief POST connector (creates a new connector session) */
void yf::CF::Rep::post_connector(Package &package,
                                 Z_HTTP_Request *req,
                                 const char *suffix_path)
{
    char **names = 0;
    char **values = 0;
    int no_parms = 0;
    CF_Engine::Flags engine_flag = CF_Engine::FORK;
    mp::odr odr;
    const char *args_str = strchr(suffix_path, '?');
    if (args_str)
        no_parms = yaz_uri_to_array(args_str, odr, &names, &values);

    const char *proxy = 0;
    // POST /connector[?proxy=IP]
    bool use_xml = false;
    if (strcmp(req->method, "POST"))
    {
        // only POST allowed
        Z_GDU *zgdu_res
            = odr.create_HTTP_Response(package.session(), req, 200);
        cf_http_msg(zgdu_res, odr, "only POST accepted for path", 405);
        package.response() = zgdu_res;
        return;
    }
    int i;
    for (i = 0; i < no_parms; i++)
    {
        if (!strcmp(names[i], "proxy"))
            proxy = values[i];
        else if (!strcmp(names[i], "thread"))
        {
            if (values[i] && *values[i] == '1')
                engine_flag = CF_Engine::THREAD;
        }
        else
        {
            Z_GDU *zgdu_res
                = odr.create_HTTP_Response(package.session(),  req, 400);
            package.response() = zgdu_res;
            return;
        }
    }

    int id = 0;
    Session *ses = 0;
    {
        boost::mutex::scoped_lock lock(m_mutex);
        id = m_connector_next_id++;
        char session_name_str[20];

        sprintf(session_name_str, "%d", id);
        ses = new Session(session_name_str);

    }

    std::string addinfo;
    CF_Engine *cf = create_connector(engine_flag, proxy, ses, addinfo);
    if (!cf)
    {
        Z_GDU *zgdu_res
            = odr.create_HTTP_Response(package.session(), req, 200);
        cf_http_msg(zgdu_res, odr, addinfo.c_str(), 500);
        package.response() = zgdu_res;
        delete ses;
        return ;
    }

    if (!load_connector(package, req, ses))
    {
        delete ses;
        return;
    }

    {
        boost::mutex::scoped_lock lock(m_mutex);
        m_connectors[id] = ses;
    }
    Z_GDU *zgdu_res
        = odr.create_HTTP_Response(package.session(), req, 200);

    Z_HTTP_Response *response = zgdu_res->u.HTTP_Response;

    z_HTTP_header_add_content_type(odr, &response->headers,
                                   use_xml ?
                                   "application/xml" : "application/json",
                                   "utf-8");
    WRBUF resp_content = wrbuf_alloc();
    if (use_xml)
    {
        wrbuf_printf(resp_content, "<id>%d</id>\n", id);
    }
    else
    {
        wrbuf_printf(resp_content, "{\"id\":%d}", id);
    }

    response->content_buf = wrbuf_buf(resp_content);
    response->content_len = wrbuf_len(resp_content);

    package.response() = zgdu_res;
    wrbuf_destroy(resp_content);
    ses->in_use = false;
}

/** \brief operate on existing connector */
void yf::CF::Rep::task_connector(Package &package, Z_HTTP_Request *req,
                                 CF::Session *cf_session,
                                 const char *args)
{
    mp::odr odr;
    const char *ctype = z_HTTP_header_lookup(req->headers,
                                             "Content-Type");
    if (!strcmp(args, "load_cf"))
    {
        if (!load_connector(package, req, cf_session))
            return;
        Z_GDU *zgdu_res = odr.create_HTTP_Response(package.session(), req, 200);
        package.response() = zgdu_res;
    }
    else if (!strcmp(args, "dom_string"))
    {
        Z_GDU *zgdu_res = 0;
        char *dom_str = 0;
        try {
            cf_session->engine->dom_string(&dom_str);
        }
        catch (...) {
            zgdu_res = odr.create_HTTP_Response(package.session(), req, 500);
            package.response() = zgdu_res;
            return;
        }
        zgdu_res = odr.create_HTTP_Response(package.session(), req, 200);
        Z_HTTP_Response *response = zgdu_res->u.HTTP_Response;
        z_HTTP_header_add_content_type(odr, &response->headers, "text/plain", 0);
        response->content_buf = dom_str;
        response->content_len = strlen(dom_str);
        package.response() = zgdu_res;
    }
    else if (ctype && !yaz_strcmp_del("application/json", ctype, "; "))
    {  /* JSON request */
        const char *output_json = 0;
        WRBUF input_json = wrbuf_alloc();
        wrbuf_write(input_json, req->content_buf, req->content_len);

        if (strncmp(args, "run_task/", 9) == 0)
        {
            try {
                cf_session->engine->run_task(args+9, wrbuf_cstr(input_json), &output_json);
            }
            catch (CF_Engine_Error &e) {
                // run_task failed
                Z_GDU *zgdu_res
                    = odr.create_HTTP_Response(package.session(), req, 200);
                cf_session->log(YLOG_LOG, "run_task failed: %s", e.what());
                cf_http_msg(zgdu_res, odr, e.what(), 500);
                package.response() = zgdu_res;
                return ;
            }
        }
        else if (strncmp(args, "run_task_opt/", 13) == 0)
        {
            try {
                cf_session->engine->run_task_opt(args+13, wrbuf_cstr(input_json), &output_json);
            }
            catch (CF_Engine_Error &e) {
                // run_task failed
                Z_GDU *zgdu_res
                    = odr.create_HTTP_Response(package.session(), req, 200);
                cf_session->log(YLOG_LOG, "run_task_opt failed: %s", e.what());
                cf_http_msg(zgdu_res, odr, e.what(), 500);
                package.response() = zgdu_res;
                return ;
            }
        }
        else if (strncmp(args, "run_tests/", 10) == 0)
        {
            bool result = false;
            try {
                result = cf_session->engine->run_tests(args + 10);
            }
            catch (CF_Engine_Error &e) {
                // run_task failed
                Z_GDU *zgdu_res
                    = odr.create_HTTP_Response(package.session(), req, 200);
                cf_session->log(YLOG_LOG, "run_tests failed: %s", e.what());
                cf_http_msg(zgdu_res, odr, e.what(), 500);
                package.response() = zgdu_res;
                return ;
            }
            if (result)
                output_json = "{\"result\": true}";
            else
                output_json = "{\"result\": false}";
        }
        else
        {
            // wrong args
            Z_GDU *zgdu_res
                = odr.create_HTTP_Response(package.session(), req, 200);
            cf_session->log(YLOG_LOG, "bad WS request: %s", args);
            cf_http_msg(zgdu_res, odr, "invalid WS request", 400);
            package.response() = zgdu_res;
            return ;
        }
        Z_GDU *zgdu_res
            = odr.create_HTTP_Response(package.session(), req, 200);
        Z_HTTP_Response *response = zgdu_res->u.HTTP_Response;

        z_HTTP_header_add_content_type(odr, &response->headers,
                                       "application/json",
                                       "utf-8");
        response->content_buf = (char *) output_json;
        response->content_len = strlen(output_json);
        package.response() = zgdu_res;
    }
    else
    {
        // bad content type
        Z_GDU *zgdu_res
            = odr.create_HTTP_Response(package.session(), req, 200);
        cf_http_msg(zgdu_res, odr, "invalid content-type", 400);
        package.response() = zgdu_res;
    }
}

void yf::CF::Rep::release_zsession(Package &package)
{
    boost::mutex::scoped_lock lock(m_mutex);
    std::map<mp::Session,ZSession *>::iterator it;

    it = m_clients.find(package.session());
    if (it != m_clients.end())
    {
        if (package.session().is_closed())
        {
            delete it->second;
            m_clients.erase(it);
        }
        else
        {
            it->second->in_use = false;
        }
        m_cond_session_ready.notify_all();
    }
}

yf::CF::ZSession *yf::CF::Rep::get_zsession(Package &package)
{
    boost::mutex::scoped_lock lock(m_mutex);

    std::map<mp::Session,yf::CF::ZSession *>::iterator it;

    while (true)
    {
        it = m_clients.find(package.session());
        if (it == m_clients.end())
            break;

        if (!it->second->in_use)
        {
            it->second->in_use = true;
            return it->second;
        }
        m_cond_session_ready.wait(lock);
    }
    return 0;
}

Z_APDU *yf::CF::Rep::open_cf_z3950(Z_APDU *req,  CF::ZSession *zsession,
                                   mp::odr &odr, unsigned long session_id)
{
    Z_SearchRequest *sr = req->u.searchRequest;

    if (sr->num_databaseNames != 1)
    {   // only one database may be given
        return odr.create_searchResponse(
            req, YAZ_BIB1_COMBI_OF_SPECIFIED_DATABASES_UNSUPP, 0);
    }   // see if we are still using the same database
    if (zsession->cf_session && zsession->database &&
        !strcmp(sr->databaseNames[0], zsession->database)
        && zsession->no_searches < 20)
    {
        try
        {
            const char *outp = 0;
            zsession->cf_session->engine->run_task_opt("exit", "{}", &outp);
        }
        catch (CF_Engine_Error &e)
        {
            yaz_log(YLOG_WARN, "exit task returned error: %s", e.what());
            zsession->close(); // zsession->database becomes 0
        }
        if (zsession->database)
            return 0; // return if exit went OK
    }

    zsession->close();
    zsession->database = xstrdup(sr->databaseNames[0]);

    // parse database
    // database?p1=v2&p2=v2&..
    const char *db_sep = strchr(zsession->database, '?');
    if (!db_sep)
        db_sep = strchr(zsession->database, ',');

    // must create a new cf_engine because database is not the same
    WRBUF cf_db = wrbuf_alloc();
    if (!db_sep)
        wrbuf_puts(cf_db, zsession->database);
    else
        wrbuf_write(cf_db, zsession->database, db_sep - zsession->database);
    int http_status = 500;
    WRBUF cf_file = cf_factory->get_cf_file(zsession->username,
                                            zsession->password,
                                            wrbuf_cstr(cf_db),
                                            &http_status);
    wrbuf_destroy(cf_db);

    if (!cf_file)
    {
        int errcode = 0;
        switch (http_status)
        {
        case 404:
            errcode = YAZ_BIB1_DATABASE_UNAVAILABLE;
            break;
        case 401:
            errcode = YAZ_BIB1_ACCESS_TO_SPECIFIED_DATABASE_DENIED;
            break;
        default:
            errcode = YAZ_BIB1_TEMPORARY_SYSTEM_ERROR;
        }
        return odr.create_searchResponse(req, errcode, sr->databaseNames[0]);
    }
    WRBUF json = wrbuf_alloc();
    const char *proxy = 0;
    int r = cf_database_args(
        db_sep, odr, &zsession->subdatabase, &proxy, json, 0, 0);
    if (r)
    {
        char *addinfo = (char *) odr_malloc(odr, 40 + wrbuf_len(json));
        sprintf(addinfo, "Bad database argument: %s", wrbuf_cstr(json));

        Z_APDU *a = odr.create_searchResponse(req,
                                              YAZ_BIB1_TEMPORARY_SYSTEM_ERROR,
                                              addinfo);
        wrbuf_destroy(json);
        return a;
    }

    Session *ses = 0;
    {
        int id = 0;
        boost::mutex::scoped_lock lock(m_mutex);
        id = m_connector_next_id++;
        char session_name_str[40];

        sprintf(session_name_str, "%lu.%d", session_id, id);
        ses = new Session(session_name_str);
    }

    std::string addinfo;
    CF_Engine *cf = create_connector(CF_Engine::FORK, proxy, ses, addinfo);
    if (!cf)
    {
        delete ses;
        wrbuf_destroy(json);
        return odr.create_searchResponse(req,
                                         YAZ_BIB1_TEMPORARY_SYSTEM_ERROR,
                                         addinfo.c_str());
    }
    zsession->cf_session = ses;

    // load connector file and run init
    try
    {
        zsession->cf_session->engine->load_cf(wrbuf_cstr(cf_file));

        const char *outp = 0;
        zsession->cf_session->engine->run_task_opt("init",
                                                   wrbuf_cstr(json), &outp);
    }
    catch (CF_Engine_Error_System &e)
    {
        int errcode = YAZ_BIB1_TEMPORARY_SYSTEM_ERROR;

        wrbuf_destroy(json);
        wrbuf_destroy(cf_file);

        return odr.create_searchResponse(req, errcode, e.what());
    }
    catch (CF_Engine_Error_Task &e)
    {
        const char *addinfo = 0;
        int errcode = YAZ_BIB1_UNSPECIFIED_ERROR;

        wrbuf_destroy(json);
        wrbuf_destroy(cf_file);

        cf_interpret_assertions(&errcode, e.what(), &addinfo);
        return odr.create_searchResponse(req, errcode, addinfo);
    }
    ses->metadata.parse_cf(wrbuf_cstr(cf_file));
    wrbuf_destroy(json);
    wrbuf_destroy(cf_file);
    return 0;
}

Z_Records *yf::CF::Rep::fetch_z3950(
    CF::ZSession *zsession,
    mp::odr &odr, Odr_oid *preferredRecordSyntax,
    Z_ElementSetNames *esn,
    int start, int number, int &error_code, std::string &addinfo,
    int *number_returned, int *next_position)
{
    Z_Records *rec = (Z_Records *) odr_malloc(odr, sizeof(Z_Records));
    rec->which = Z_Records_DBOSD;
    rec->u.databaseOrSurDiagnostics = (Z_NamePlusRecordList *)
        odr_malloc(odr, sizeof(Z_NamePlusRecordList));
    rec->u.databaseOrSurDiagnostics->num_records = number;
    rec->u.databaseOrSurDiagnostics->records = (Z_NamePlusRecord **)
        odr_malloc(odr, sizeof(Z_NamePlusRecord *) * number);
    int i;
    WRBUF wrb = wrbuf_alloc();
    for (i = 0; i < number; i++)
    {
        const Odr_oid *oid = preferredRecordSyntax;

        rec->u.databaseOrSurDiagnostics->records[i] = (Z_NamePlusRecord *)
            odr_malloc(odr, sizeof(Z_NamePlusRecord));
        Z_NamePlusRecord *npr = rec->u.databaseOrSurDiagnostics->records[i];
        npr->databaseName = zsession->database;
        npr->which = Z_NamePlusRecord_databaseRecord;

        wrbuf_rewind(wrb);

        /* if no record syntax was given assume XML */
        if (!oid || !oid_oidcmp(oid, yaz_oid_recsyn_xml))
        {
            if (zsession->get_record_xml(start + i, wrb, &error_code))
            {
                npr->u.databaseRecord =
                    z_ext_record_xml(odr, wrbuf_buf(wrb), wrbuf_len(wrb));
            }
            else
                break;
        }
        else if (!oid_oidcmp(oid, yaz_oid_recsyn_json))
        {
            struct json_node *json =
                zsession->get_record(start + i, &error_code);
            if (json)
            {
                json_write_wrbuf(json, wrb);

                npr->u.databaseRecord =
                    z_ext_record_oid(odr, yaz_oid_recsyn_json,
                                     wrbuf_buf(wrb), wrbuf_len(wrb));
            }
            else
                break;
        }
        else
        { /* unsupported syntax. Return diagnostic */
            char buf[OID_STR_MAX];
            error_code = YAZ_BIB1_RECORD_SYNTAX_UNSUPP;
            addinfo = odr_strdup(odr, oid_oid_to_dotstring(oid, buf));
            break;
        }
    }
    wrbuf_destroy(wrb);
    *next_position = start + i;
    *number_returned = i;
    rec->u.databaseOrSurDiagnostics->num_records = i;
    return rec;
}

Z_APDU *yf::CF::Rep::search_z3950(Z_APDU *req, CF::ZSession *zsession,
                                  mp::odr &odr, unsigned long session_id)
{
    Z_SearchRequest *sr = req->u.searchRequest;
    if (strcmp(sr->resultSetName, "default"))
    {
        return odr.create_searchResponse(
            req, YAZ_BIB1_RESULT_SET_NAMING_UNSUPP, 0);
    }

    json_remove_node(zsession->records);
    zsession->records = 0;
    zsession->hits = 0;

    Z_APDU *apdu = open_cf_z3950(req, zsession, odr, session_id);
    if (apdu)
        return apdu;

    zsession->no_searches++;

    CF_RPN_to_JSON rpn_to_json;
    rpn_to_json.set_capability_flags(
        zsession->cf_session->metadata.get("flags"));
    if (sr->query->which == Z_Query_type_1
        || sr->query->which == Z_Query_type_101)
    {
        NMEM nmem = ((ODR) odr)->mem;
        struct sort_elem *sort_list = 0;
        type7_sort(sr->query->u.type_1, nmem, &sort_list);
        if (!rpn_to_json.parse(sr->query->u.type_1,
                zsession->subdatabase,sort_list))
        {
            const char *addinfo;
            int errcode = rpn_to_json.get_diagnostic(&addinfo);
            return odr.create_searchResponse(req, errcode, addinfo);
        }
    }
    else
    {
        return odr.create_searchResponse(req, YAZ_BIB1_QUERY_TYPE_UNSUPP, 0);
    }

    const char *outp = 0;
    int query_no; // pass over multiple JSON alternatives
    for (query_no = 0; ; query_no++)
    {
        const char *json_res = rpn_to_json.get_result(query_no);
        if (!json_res)
        {
            return odr.create_searchResponse(
                req, YAZ_BIB1_UNSUPP_USE_ATTRIBUTE, 0);
        }
        try
        {
            zsession->cf_session->engine->run_task("search", json_res, &outp);
        }
        catch (CF_Engine_Error_System &e)
        {
            const char *what = e.what();
            return odr.create_searchResponse(req, YAZ_BIB1_TEMPORARY_SYSTEM_ERROR,
                                             what);
        }
        catch (CF_Engine_Error_Task &e)
        {
            const char *lead = "Task: no implementation of task";
            const char *what = e.what();
            json_res = rpn_to_json.get_result(1);

            if (what && !strncmp(what, lead, strlen(lead)))
            {
                continue; // try next query_no ..
            }
            else
            {
                return odr.create_searchResponse(
                    req, YAZ_BIB1_TEMPORARY_SYSTEM_ERROR, what);
            }
        }
        break;
    }

    const char *errmsg = 0;
    struct json_node *res = json_parse(outp, &errmsg);
    if (!res)
    {
        return odr.create_searchResponse(req, YAZ_BIB1_TEMPORARY_SYSTEM_ERROR,
                                         errmsg);
    }
    struct json_node *hit = json_get_object(res, "hits");
    if (hit && hit->type == json_node_array)
        hit = json_get_elem(hit, 0);
    if (hit)
    {
        if (hit->type == json_node_number)
            zsession->hits = (Odr_int) hit->u.number;
        else if (hit->type == json_node_string)
            zsession->hits = odr_atoi(hit->u.string);
    }
    json_remove_node(res);

    if (zsession->hits == 0)
    {
        return odr.create_searchResponse(req, 0, 0);
    }
    try
    {
        zsession->cf_session->engine->run_task("parse", "", &outp);
    }
    catch (CF_Engine_Error &e)
    {
        const char *what = e.what();
        return odr.create_searchResponse(
            req, YAZ_BIB1_TEMPORARY_SYSTEM_ERROR, what);
    }

    res = json_parse(outp, 0);
    struct json_node *rp = json_detach_object(res, "results");
    if (rp)
    {
        json_parser_t parser = json_parser_create();

        json_parser_subst(parser, 0, rp);

        zsession->records = json_parser_parse(parser, "{\"records\":%0}");

        json_parser_destroy(parser);
    }
    json_remove_node(res);


    int number = 0;
    mp::util::piggyback(*sr->smallSetUpperBound,
                        *sr->largeSetLowerBound,
                        *sr->mediumSetPresentNumber,
                        zsession->hits,
                        number);

    int number_returned = 0;
    int next_position = 0;
    Z_Records *records = 0;
    if (number)
    {
        Z_ElementSetNames *esn;
        if (number > *sr->smallSetUpperBound)
            esn = sr->mediumSetElementSetNames;
        else
            esn = sr->smallSetElementSetNames;

        std::string addinfo;
        int error_code = 0;
        records = fetch_z3950(
            zsession,
            odr, sr->preferredRecordSyntax, esn,
            1, number,
            error_code, addinfo,
            &number_returned,
            &next_position);
    }
    Z_APDU *apdu_res = odr.create_searchResponse(req, 0, 0);
    *apdu_res->u.searchResponse->resultCount = zsession->hits;
    apdu_res->u.searchResponse->records = records;

    *apdu_res->u.searchResponse->numberOfRecordsReturned = number_returned;
    *apdu_res->u.searchResponse->nextResultSetPosition = next_position;

    return apdu_res;
}

Z_APDU *yf::CF::Rep::present_z3950(Z_APDU *req, CF::ZSession *zsession,
                                   mp::odr &odr)
{
    Z_PresentRequest *pr = req->u.presentRequest;
    int number_returned = 0;
    int next_position = 0;
    int error_code = 0;
    std::string addinfo;
    Z_ElementSetNames *esn = 0;

    if (pr->recordComposition)
    {
        if (pr->recordComposition->which == Z_RecordComp_simple)
            esn = pr->recordComposition->u.simple;
        else
            return
                odr.create_presentResponse(
                    req, YAZ_BIB1_ONLY_A_SINGLE_ELEMENT_SET_NAME_SUPPORTED, 0);
    }

    if (strcmp(pr->resultSetId, "default") || !zsession->cf_session)
    {
        return odr.create_presentResponse(
            req, YAZ_BIB1_SPECIFIED_RESULT_SET_DOES_NOT_EXIST, pr->resultSetId);
    }

    Z_Records *records = fetch_z3950(
        zsession, odr, pr->preferredRecordSyntax, esn,
        *pr->resultSetStartPoint, *pr->numberOfRecordsRequested,
        error_code, addinfo,
        &number_returned,
        &next_position);

    if (error_code)
    {
        return odr.create_presentResponse(req, error_code,
                                          addinfo.c_str());
    }
    Z_APDU *apdu_res = odr.create_presentResponse(req, 0, 0);
    Z_PresentResponse *resp = apdu_res->u.presentResponse;
    resp->records = records;
    *resp->numberOfRecordsReturned = number_returned;
    *resp->nextResultSetPosition = next_position;
    return apdu_res;
}

void yf::CF::Rep::process_z3950(Package &package, CF::ZSession *zsession)
{
    Z_GDU *gdu = package.request().get();
    Z_APDU *apdu_req = gdu->u.z3950;
    Z_APDU *apdu_res = 0;
    mp::odr odr;

    if (apdu_req->which == Z_APDU_initRequest)
    {
        int errcode = 0;
        const char *addinfo = 0;
        Z_InitRequest *req = apdu_req->u.initRequest;

        if (zsession) // double init !
            errcode = YAZ_BIB1_PERMANENT_SYSTEM_ERROR;

        if (!errcode)
        {
            zsession = new ZSession(req->idAuthentication);

            int http_status = 500;
            cf_factory->auth(zsession->username, zsession->password,
                             &http_status);
            switch (http_status)
            {
            case 200: /* OK */
                break;
            case 401:
                errcode = YAZ_BIB1_INIT_AC_BAD_USERID_AND_OR_PASSWORD;
                break;
            default:
                errcode = YAZ_BIB1_INIT_AC_AUTHENTICATION_SYSTEM_ERROR;
            }
        }

        apdu_res = odr.create_initResponse(apdu_req, errcode, addinfo);
        Z_InitResponse *resp = apdu_res->u.initResponse;

        resp->implementationName = odr_strdup(odr, "CF:" CF_VERSION);

        int i;
        static const int masks[] = {
            Z_Options_search, Z_Options_present, -1
        };
        for (i = 0; masks[i] != -1; i++)
            if (ODR_MASK_GET(req->options, masks[i]))
                ODR_MASK_SET(resp->options, masks[i]);
        static const int versions[] = {
            Z_ProtocolVersion_1,
            Z_ProtocolVersion_2,
            Z_ProtocolVersion_3,
            -1
        };
        for (i = 0; versions[i] != -1; i++)
            if (ODR_MASK_GET(req->protocolVersion, versions[i]))
                ODR_MASK_SET(resp->protocolVersion, versions[i]);
            else
                break;

        Odr_int s = 5 * 1024 * 1024;
        if (*req->maximumRecordSize < s)
            s = *req->maximumRecordSize;
        *resp->maximumRecordSize = s;

        s = 5 * 1024 * 1024;
        if (*req->preferredMessageSize < s)
            s = *req->preferredMessageSize;
        *resp->preferredMessageSize = s;

        if (zsession)
        {
            if (errcode)
                delete zsession;
            else
            {
                boost::mutex::scoped_lock lock(m_mutex);
                m_clients[package.session()] = zsession;
            }
        }
        if (errcode)
            package.session().close();
    }
    else
    {
        if (!zsession)
        {
            apdu_res = odr.create_close(apdu_req,
                                        Z_Close_protocolError,
                                        "no init");
            package.session().close();
        }
        else
        {
            switch (apdu_req->which)
            {
            case Z_APDU_searchRequest:
                apdu_res = search_z3950(apdu_req, zsession, odr,
                                        package.session().id());
                break;
            case Z_APDU_presentRequest:
                apdu_res = present_z3950(apdu_req, zsession, odr);
                break;
            case Z_APDU_close:
                apdu_res = odr.create_close(apdu_req,
                                            Z_Close_finished,
                                            0);
            }
        }
    }
    if (apdu_res)
        package.response() = apdu_res;
}

void yf::CF::Rep::process_connector(Package &package, const char *suffix_path)
{
    Z_GDU *zgdu = package.request().get();
    Z_HTTP_Request *req = zgdu->u.HTTP_Request;

    if (*suffix_path != '/')
    {
        if (!strcmp(req->method, "POST"))
            post_connector(package, req, suffix_path);
        else if (!strcmp(req->method, "GET"))
        {
            mp::odr odr;
            Z_GDU *zgdu_res
                = odr.create_HTTP_Response(package.session(), req, 200);
            cf_http_msg(zgdu_res, odr, CF_VERSION, 200);
            package.response() = zgdu_res;
        }
        else
        {
            // only GET/POST allowed
            mp::odr odr;
            Z_GDU *zgdu_res
                = odr.create_HTTP_Response(package.session(), req, 200);
            cf_http_msg(zgdu_res, odr, "only POST allowed for without", 405);
            package.response() = zgdu_res;
        }
    }
    else
    {    // POST /connector/id/args or DELETE /connector/id
        mp::odr odr;
        char args[60];
        int id = 0;

        *args = '\0';
        int no_args = sscanf(suffix_path, "/%d/%59s", &id, args);

        if (no_args < 1)
        {
            Z_GDU *zgdu_res
                = odr.create_HTTP_Response(package.session(), req, 200);
            cf_http_msg(zgdu_res, odr,
                        "missing arguments for connector POST", 400);
            package.response() = zgdu_res;
            return ;
        }
        if (!strcmp(req->method, "POST"))
        {
            Session *cf_session = 0;
            {
                boost::mutex::scoped_lock lock(m_mutex);

                std::map<int,Session *>::const_iterator map_it;

                map_it = m_connectors.find(id);
                if (map_it == m_connectors.end())
                {
                    Z_GDU *zgdu_res
                        = odr.create_HTTP_Response(package.session(), req, 200);
                    cf_http_msg(zgdu_res, odr, "no such session", 400);

                    package.response() = zgdu_res;
                    return ;
                }
                cf_session = map_it->second;

                if (!strcmp(args, "log"))
                {
                    Z_GDU *zgdu_res
                        = odr.create_HTTP_Response(package.session(), req, 200);
                    Z_HTTP_Response *response = zgdu_res->u.HTTP_Response;

                    z_HTTP_header_add_content_type(odr, &response->headers,
                                                   "text/plain", 0);
                    if (cf_session->log_file)
                    {
                        long sz = ftell(cf_session->log_file);
                        if (sz > 0)
                        {
                            int fd = fileno(cf_session->log_file);
                            char *buf = (char*) odr_malloc(odr, sz);
                            pread(fd, buf, sz, 0);
                            response->content_buf = buf;
                            response->content_len = sz;
                        }
                    }
                    package.response() = zgdu_res;
                    return;
                }
                else if (!strcmp(args, "screen_shot"))
                {
                    Z_GDU *zgdu_res = 0;
                    WRBUF fname = wrbuf_alloc();

                    wrbuf_printf(fname, "%s.png", wrbuf_cstr(cf_session->file_area));
                    try {
                        cf_session->engine->screen_shot(wrbuf_cstr(fname));
                    }
                    catch (CF_Engine_Error &e) {
                        zgdu_res = odr.create_HTTP_Response(package.session(), req, 500);
                        package.response() = zgdu_res;
                        wrbuf_destroy(fname);
                        return ;
                    }
                    FILE *file = fopen(wrbuf_cstr(fname), "rb");
                    if (!file)
                    {
                        zgdu_res = odr.create_HTTP_Response(package.session(), req, 500);
                        package.response() = zgdu_res;
                    }
                    else
                    {
                        zgdu_res  = odr.create_HTTP_Response(package.session(), req, 200);
                        Z_HTTP_Response *response = zgdu_res->u.HTTP_Response;

                        z_HTTP_header_add_content_type(odr, &response->headers,
                                                       "image/png", 0);

                        fseek(file, 0L, SEEK_END);

                        long sz = ftell(file);
                        rewind(file);

                        char *buf = (char*) odr_malloc(odr, sz);
                        fread(buf, 1, sz, file);
                        fseek(file, sz, SEEK_SET);

                        response->content_buf = buf;
                        response->content_len = sz;

                        fclose(file);

                        package.response() = zgdu_res;
                    }
                    wrbuf_destroy(fname);
                    return;
                }
                // only one user at a time!
                if (cf_session->in_use)
                {
                    Z_GDU *zgdu_res
                        = odr.create_HTTP_Response(package.session(), req, 200);
                    cf_http_msg(zgdu_res, odr, "session already active", 400);
                    package.response() = zgdu_res;
                    return ;
                }
                cf_session->in_use = true;
            }
            // args is one of engine methods: "run_task", "run_task_opt",
            // "run_tests", "load_cf", "dom_string"
            task_connector(package, req, cf_session, args);
            {
                boost::mutex::scoped_lock lock(m_mutex);
                cf_session->in_use = false;
                cf_session->age = 0;
                if (cf_session->destroyed)
                    delete cf_session;
            }
        }
        else if (!strcmp(req->method, "DELETE"))
        {
            {
                std::map<int,Session *>::iterator map_it;

                boost::mutex::scoped_lock lock(m_mutex);
                map_it = m_connectors.find(id);
                if (map_it != m_connectors.end())
                {
                    Session *ses = map_it->second;
                    m_connectors.erase(map_it);
                    if (ses->in_use)
                        ses->destroyed = true;
                    else
                        delete ses;
                }
            }
            Z_GDU *zgdu_res
                = odr.create_HTTP_Response(package.session(), req, 200);
            package.response() = zgdu_res;
        }
        else
        {
            // only DELETE/POST allowed
            Z_GDU *zgdu_res
                = odr.create_HTTP_Response(package.session(), req, 200);
            cf_http_msg(zgdu_res, odr, "only DELETE/POST allowed for path", 405);
            package.response() = zgdu_res;
        }
    }
}

void yf::CF::start() const
{
    Worker w(m_p.get());
    m_p->m_thrds.add_thread(new boost::thread(w));
}

void yf::CF::process(Package &package) const
{
    Z_GDU *gdu = package.request().get();
    CF::ZSession *zsession = m_p->get_zsession(package);

    if (!gdu)
        package.move();
    else if (gdu->which == Z_GDU_HTTP_Request)
    {
        const char *lead = m_p->http_prefix.c_str();
        Z_HTTP_Request *req = gdu->u.HTTP_Request;
        if (req->path[0] == '/' && !strncmp(req->path+1, lead, strlen(lead)))
        {
            m_p->process_connector(package, req->path + 1 + strlen(lead));
        }
        else
            package.move();
    }
    else if (gdu->which == Z_GDU_Z3950)
    {
        if (m_p->enable_z3950)
            m_p->process_z3950(package, zsession);
        else
            package.move();
    }
    m_p->release_zsession(package);
}

void yf::CF::configure(const xmlNode * ptr, bool test_only,
                       const char *path)
{
    for (ptr = ptr->children; ptr; ptr = ptr->next)
    {
        if (ptr->type != XML_ELEMENT_NODE)
            continue;
        if (!strcmp((const char *) ptr->name, "url_prefix"))
        {
            m_p->http_prefix = mp::xml::get_text(ptr);
        }
        else if (!strcmp((const char *) ptr->name, "z39.50"))
        {
            const struct _xmlAttr *attr;
            for (attr = ptr->properties; attr; attr = attr->next)
            {
                if (!strcmp((const char *) attr->name,  "enable"))
                    m_p->enable_z3950 =
                        mp::xml::get_bool(attr->children, false);
                else
                    throw mp::filter::FilterException(
                        "Bad attribute " + std::string((const char *)
                                                       attr->name));
            }
        }
        else if (!strcmp((const char *) ptr->name, "env"))
        {
            m_p->cf_factory = new CF_Factory(ptr);
        }
        else
        {
            throw mp::filter::FilterException("Bad element "
                                               + std::string((const char *)
                                                             ptr->name));
        }
    }
    if (!m_p->cf_factory)
    {
        m_p->cf_factory = new CF_Factory(0);
    }
}

static yf::Base* filter_creator()
{
    return new mp::filter::CF;
}

extern "C" {
    struct metaproxy_1_filter_struct metaproxy_1_filter_cf = {
        0,
        "cf",
        filter_creator
    };
}


/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */

