var EXPORTED_SYMBOLS = ["Conditional"];

Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/jsonPathHelper.js');
Components.utils.import('resource://indexdata/util/logging.js');

var logger = logging.getLogger();


/* The constructor takes a reference to an array of condidionts which will be
 * updated in the draw() method and used to determine the output of
 * eval(). Each have the following properties: 
 *
 * operands: [{usejp: <boolean>, jp: {path: key:}, value: <constant>}, ...]
 * operator: key from the operators object defined below
 * negate: if true, negate the result (default: false) 
 * connective: "AND" or "OR", (default: "AND") 
 *
 * The second parameter is an object for JSONpaths to run against.
 *
 *
 * Possible future features:
 * - test if ANY or ALL of a jp evaluate to true rather than just FIRST
 */
var Conditional = function (conditions, jptarget) {
  this.conditions = conditions;
  this.jptarget = jptarget
};

// static
Conditional.operators = {
  always: {
    label: 'always',
    short: 'always',
    compare: function() {
      return true;
    },
    arity: 0
  },
  empty: {
    label: 'is empty',
    short: 'empty',
    compare: function(v) {
      if (typeof(v) === 'undefined' || typeof(v) === 'null' || v === '') return true;
      else return false;
    },
    arity: 1
  },
  equals: {
    label: 'equals',
    short: '==',
    compare: function(l, r) {
      return l == r;
    },
    arity: 2
  },
  match: {
    label: 'matches regex',
    short: '=~', 
    compare: function(l, r) {
      // TODO: these should probably throw execeptions to be handled appropriately by steps
      // everything matches nothing
      if (typeof(r) === 'undefined' || typeof(r) === 'null' || r === '') return true;
      // regex only works on strings
      if (typeof(l) != "string" ) l = "";
      let re = new RegExp(String(r));
      return re.test(String(l));
    },
    arity: 2
  },
  contains: {
    label: 'contains string',
    short: 'contains', 
    compare: function(l, r) {
      // everything matches contains nothing
      if (typeof(r) === 'undefined' || typeof(r) === 'null' || r === '') return true;
      return String(l).indexOf(String(r)) !== -1;
    },
    arity: 2
  },
  gt: {
    label: 'greater than',
    short: '>', 
    compare: function(l, r) {
      return Number(l) > Number(r);
    },
    arity: 2
  },
  gteq: {
    label: 'greater than or equal',
    short: '>=', 
    compare: function(l, r) {
      return Number(l) >= Number(r);
    },
    arity: 2
  },
  lt: {
    label: 'less than',
    short: '<', 
    compare: function(l, r) {
      return Number(l) < Number(r);
    },
    arity: 2
  },
  lteq: {
    label: 'less than or equal',
    short: '<=', 
    compare: function(l, r) {
      return Number(l) <= Number(r);
    },
    arity: 2
  }
};
Conditional.oneToString = function(cond) { 
  let op = this.operators[cond.operator];
  let str = '';
  if (cond.negate) str += '!';
  str += '(';
  if (op.arity > 0) {
    if (cond.operands[0].usejp === true && cond.operands[0].jp.key) str += cond.operands[0].jp.path + '.' + cond.operands[0].jp.key;
    else str += cond.operands[0].value || '';
  }
  str += ' ' + op.short + ' ';
  if (op.arity > 1) {
    if (cond.operands[1].usejp === true && cond.operands[1].jp.key) str += cond.operands[1].jp.path + '.' + cond.operands[1].jp.key;
    else str += cond.operands[1].value || '';
  }
  str += ')';
  return str;
}
  
// Would be nice to have the option of a class rather than instance unit test
Conditional.prototype.unitTest = function () { return Conditional.unitTest(); };
Conditional.unitTest = function () {
  let record = {"results":[{"author":["billy","bob"], "title":["a book"], "item":[{"location":["a library"]}, {"location":["my house"]}]}, {"author":["jill"], "title":["a paper"], "journaltitle":"unexpected scalars"}], "regex":"^a.*r$"};
  let tests = [
    {
      conditions: [
        {operands: [{usejp: false, value: '3'}, {usejp: false, value: 2}], operator: 'gt'},
        {operands: [{usejp: false, value: ''}, {usejp: false, value: ''}], operator: 'always', connective: 'AND'}
      ], 
      expected: true
    },
    {
      conditions: [{
        operands: [
          {usejp: true, jp: {path: '$.results[0]', key: 'title'}},
          {usejp: false, value: 'book'}
        ],
        operator: 'contains',
        negate: true
      }], 
      expected: false
    }, 
    {
      conditions: [{
        operands: [
          {usejp: true, jp: {path: '$.results[1]', key: 'title'}},
          {usejp: true, jp: {path: '$', key: 'regex'}}
        ],
        operator: 'match',
      }], 
      expected:true 
    }, 
    {
      conditions: [
        {
          operands: [{usejp: true, jp: {path: '$', key: 'nothere'}}],
          operator: 'empty',
          negate: true
        },
        {
          operands: [{usejp: true, jp: {path: '$.results[1]', key: 'title'}}],
          operator: 'empty',
          connective: 'OR',
        }
      ], 
      expected:false 
    },
    {
      conditions: [
        {
          operands: [{usejp: false, value:''}],
          operator: 'empty',
        },
        {
          operands: [{usejp: false, value:42}],
          operator: 'empty',
          negate: true
        }
      ], 
      expected:true 
    } 
  ];
  let result = false;
  for (let i = 0; i < tests.length; i++) {
    let test = tests[i];
    let instance = new Conditional(test.conditions, record);
    try {
      result = instance.eval();
    } catch (e) {
      logger.error(e);
      return false
    }
    if (result === test.expected) {
      logger.info(" * " + String(instance) + " is " + test.expected + ",  OK");
    } else {
      logger.error(" * " + String(instance) + " isn't " + test.expected  + ", FAIL");
      return false;
    }
  }
  return true;
};
// because closures
Conditional.makeRemoveListener = function (context, step, surface, i) {
  return function (e) {
    if (context.conditions.length > 1) {
      context.conditions.splice(i, 1);
      context.draw(surface, step);
    }
  };
};

// instance
Conditional.prototype.draw = function (vbox, step, caption) {
  Components.utils.import('resource://indexdata/util/xulHelper.js');
  var context = this;
  var defaultCondition = {
    operands: [{usejp: true, jp: {path:'', key:''}}, {usejp: false, value:''}],
    operator: 'contains'
  };
  if (this.conditions.length < 1) this.conditions.push(defaultCondition); 
  xmlHelper.emptyChildren(vbox);

  // adds condition.left/condition.right into an hbox
  valuePicker = function(box, obj) {
    if (obj.usejp) {
      if (typeof(obj.jp) === 'undefined') obj.jp = {};
      xulHelper.jsonPathField(box, step, obj, 'jp');
    } else {
      let constantInput = xmlHelper.appendNode(box, "textbox", null,
        {flex:'1', value: obj.value||''}, null);
      constantInput.addEventListener('input', xulHelper.makeValueListener(obj, 'value'), false);
    } 
    let switchButton = xmlHelper.appendNode(box, 'button', null, 
      {'label': obj.usejp ? 'Constant' : 'Path'}, null);
    switchButton.addEventListener('command', function(e) {
      obj.usejp = !obj.usejp;
      context.draw(vbox, step);
    }, false);
  }

  // conditions
  let condition;
  let hbox;
  for (let i = 0; i < this.conditions.length; i++) {
    condition = this.conditions[i];
    let currentBox = xmlHelper.appendNode(vbox, 'vbox');

    // first row
    hbox = xmlHelper.appendNode(vbox, "hbox", null, {align: "center"}); 
    if (caption && i === 0) xmlHelper.appendNode(hbox, "caption", caption);
    if (i > 0) {
      let boolSelect = xmlHelper.appendNode(hbox, "menulist");
      let boolMenu = xmlHelper.appendNode(boolSelect, "menupopup");
      let boolItemAnd = xmlHelper.appendNode(boolMenu, "menuitem", null,
        {value: 'AND', label: 'AND'});
      let boolItemOr = xmlHelper.appendNode(boolMenu, "menuitem", null,
        {value: 'OR', label: 'OR'});
      xulHelper.selectMenulistItemWithValue(boolSelect, condition.connective || 'AND');
      boolSelect.addEventListener('select',
        xulHelper.makeValueListener(condition, 'connective'), false);
    }
    valuePicker(hbox, condition.operands[0]);

    // second row
    hbox = xmlHelper.appendNode(vbox, "hbox", null, {align: "center"}); 
    let negateButton = xmlHelper.appendNode(hbox, "checkbox", null,
      {'label': 'NOT'}, null);
    if (condition.negate === true) negateButton.checked = true;
    negateButton.addEventListener("command",
      xulHelper.makeToggleListener(condition, 'negate'), false);
    let opSelect = xmlHelper.appendNode(hbox, "menulist");
    let opMenu = xmlHelper.appendNode(opSelect, "menupopup");
    let opItems = {};
    for (let key in Conditional.operators) {
      let op = Conditional.operators[key];
      opItems[key] = xmlHelper.appendNode(opMenu, "menuitem", null,
        {value: key, label: op.label});
    }
    xulHelper.selectMenulistItemWithValue(opSelect, condition.operator);
    opSelect.addEventListener("select", xulHelper.makeValueListener(condition, 'operator'), false);
    valuePicker(hbox, condition.operands[1]);
    let rm = xmlHelper.appendNode(hbox, "image", null,
      {src: "chrome://cfbuilder/content/icons/window-close.png"}, null);
    rm.addEventListener("click", Conditional.makeRemoveListener(context, step, vbox, i), false);
  }
  let add = xmlHelper.appendNode(hbox, "image", null, {src: "chrome://cfbuilder/content/icons/list-add.png"}, null);
  add.addEventListener("click", function (e) {
    context.conditions.push(defaultCondition);
    context.draw(vbox, step);
  }, false);
};
Conditional.prototype.eval = function () {
  let combined = this._evalOne(this.conditions[0]);
  for (let i = 1; i < this.conditions.length; i++) {
    let current = this._evalOne(this.conditions[i]);
    if (this.conditions[i].connective === 'OR') combined = combined || current;
    else combined = combined && current;
  }
  return combined;
};
Conditional.prototype.toString = function () {
  if (!Array.isArray(this.conditions) || this.conditions.length < 1) return "";
  let str = Conditional.oneToString(this.conditions[0]);
  for (let i=1; i<this.conditions.length; i++) {
    let cur = this.conditions[i];
    str += cur.connective === 'AND' ? ' && ' : ' || ';  
    str += Conditional.oneToString(cur);
  }
  return str; 
};
Conditional.prototype.getUsedArgs = function () {
  let result = [];
  for (let i = 0; i < this.conditions.length; i++) {
    let cond = this.conditions[i];
    for (let j = 0; j < cond.arity; j++) {
      let op = cond.operands[j]; 
      if (op.path === "$.input") return [op.key];
    }
  }
  return result;
}


// private
Conditional.prototype._getValue = function (cond) {
  if (cond.usejp === true) {
    return jsonPathHelper.getFirst(cond.jp, this.jptarget);
  } else return cond.value;
};
Conditional.prototype._evalOne = function (condition) {
  let op = Conditional.operators[condition.operator];
  let left;
  let right;
  if (op.arity > 0) left = this._getValue(condition.operands[0]);
  if (op.arity > 1) right = this._getValue(condition.operands[1]); 
  if (condition.negate) return !(op.compare(left, right));
  else return op.compare(left, right);
};

