var EXPORTED_SYMBOLS = ["TaskUnitTest"];

Components.utils.import('resource://indexdata/util/logging.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');

var logger = logging.getLogger('test_execution');

var OutputAssert = function (path, regex) {
  //split the path into parts
  this.path = path.split('/');
  while (this.path.length > 0 && this.path[0] === "")
    this.path.shift();
  // compile the regex
  this.regex = new RegExp(regex);
};

OutputAssert.prototype.getPath = function () {
  return this.path.join('/');
}

OutputAssert.prototype.getValue = function () {
  return this.regex.source;
}

OutputAssert.prototype.assert = function (results) {
  this._assert(this.path, results);
}

OutputAssert.prototype._assert = function (assertPath, results) {
  if (typeof results == "undefined" || results === null)
    throw new Error("Cannot assert on null or undefined elements");
  var path = assertPath.slice();
  var data = results;

  // need to traverse
  if (path.length > 0) {
    var atop = path.shift();
    switch (typeof data) {
      case "object" :
        //array
        if (typeof data.length === 'number' &&
            !data.propertyIsEnumerable('length')) {
          var i;
          //check if should go for all or one
          if (atop.length == 1 && atop.charAt(0) == '*') {
            for (i=0; i<data.length; i++)
              this._assert(path, data[i]);
          } else if ((i = this.getInt(atop)) != null
              && i < data.length) {
            this._assert(path, data[i]);
          } else {
            throw new Error("Cannot find element at ..." + path.join(','));
          }
        // hash
        } else {
          if (atop.length == 1 && atop.charAt(0) == '*') {
            for (var key in data) {
              this._assert(path, data[key]);
            }
          } else {
            this._assert(path, data[atop]);
          }
        }
        return;
      default:
        throw new Error("Cannot follow path on non-recursive elements");
    }
  // on the target
  } else {
    switch (typeof data) {
      case "number" :
        data = data.toString();
      case "string" :
        this.execRegex(data);
        logger.info("Assertion (" + this.getPath() + "=" + this.getValue() +
            ") successful on " + data);
        return;
      default:
        throw new Error("Assert target element has to be a string or a number" +
            ", found " + typeof data);
    }
  }
};

OutputAssert.prototype.getInt = function (str) {
  var num = 0;
  for (var i=str.length-1; i>=0; i--) {
    var chr = str.charAt(i);
    if (chr >= '0' && chr <= '9')
      num += parseInt(chr) + 10*i;
    else
      throw new Error("Malformed array index - '" + str + "'");
  }
  return num;
};

OutputAssert.prototype.execRegex = function (data) {
  if (!this.regex.test(data)) throw new Error("Assertion ("
      + this.getPath() + "=" + this.getValue() + ") failed on " + data);;
};

////////////////////////////////////////////////////////////////////////////////

var TaskUnitTest = function (task, name) {
  this._task = task;
  this._name = name;
  this._asserts = [];
  this._args = {};
  this.onTestSuccess = null;
  this.onTestFailure = null;
  logger.debug("Test for task '" + task.name + "' created.");
};

TaskUnitTest.prototype.getName = function () {
  return this._name;
};

TaskUnitTest.prototype.setName = function (name) {
  this._name = name;
};

TaskUnitTest.prototype.getArgs = function () {
  return this._args;
};

TaskUnitTest.prototype.setArg = function (name, value) {
  this._args[name] = value;
  logger.debug("Test arg set: " + name + "=" + value);
};

TaskUnitTest.prototype.getArg = function (name) {
  if ( this._args[name] == undefined )
    return null;  // without plopping a warning in the js console
  return this._args[name];
};

TaskUnitTest.prototype.delArg = function (name) {
  delete this._args[name];
};

TaskUnitTest.prototype.createAssert = function (path, regex) {
  var assert = new OutputAssert(path, regex);
  this.addAssert(assert);
  logger.debug("Assert created: " + path+"="+regex);
  return assert;
};

TaskUnitTest.prototype.addAssert = function (assert) {
  if (!(assert instanceof OutputAssert)) {
      throw new Error("Assert is not a proper type.");
  }
  this._asserts.push(assert);
};

TaskUnitTest.prototype.removeAssert = function (assert) {
  for (var i=0; i<this._asserts.length; i++) {
    if (assert === this._asserts[i]) {
      this._asserts.splice(i,1);
      return;
    }
  }
};

TaskUnitTest.prototype.getAssert = function (idx) {
  if (idx >= 0 && idx <= this._asserts.length)
    return this._asserts[idx];
  else
    return null;
};

TaskUnitTest.prototype.getAsserts = function () {
  return this._asserts;
};

TaskUnitTest.prototype.generateTaskInput = function () {
  // This is used in the builder to get arguments
  // from the current test whenever you run a Task.
  //
  // Only used in the engine with the test command. And so we can't use
  // the template to determine if we need to decode JSON, but we're only
  // working on canned tests so it's not that great of a risk to base
  // the decision on presence of braces.

  let taskIn = {};
  for (var key in this._args) {
    let arg = this._args[key];
    // If brace-encased parse test string as JSON
    // if ((arg.charAt(0) === '{') && (arg.charAt(arg.length-1) === '}')) {
    // Allow whitespace around it, see CP-2999
    if ( arg.match( /^\s*\{.*\}\s*$/ ) ) {
      try {
        taskIn[key] = JSON.parse(arg);
      } catch (e) {
        this._task.warn('Cannot parse input "' + arg + '" as JSON, using as-is');
        taskIn[key] = arg;
      }
    } else {
      taskIn[key] = arg;
    }
  }
  return taskIn;
};

TaskUnitTest.prototype.run = function () {
  logger.info("Test started.");
  logger.debug("Task ARGS: "
      + JSON.stringify(this._args));
  var context = this;
  this._task.addHandler("onTaskFinish", function (outcome, result) {
    context._task.removeHandler("onTaskFinish", arguments.callee);
    // failed
    if (!outcome) {
      logger.info("Test failed - task failed.");
      if (context.onTestFailure !== null) context.onTestFailure();
      return;
    // success
    } else {
      for (var i=0; i<context._asserts.length; i++) {
        try {
          context._asserts[i].assert(result);
        } catch(e) {
          logger.info("Test failed: " + e.message);
          if (context.onTestFailure !== null) context.onTestFailure();
          throw e;
        }
      }
    }
    logger.debug("Task RESULTS: "
      + JSON.stringify(context._task.output));
    logger.info("Test finished.");
    if (context.onTestSuccess !== null) context.onTestSuccess();
  });
  this._task.data.input = this.generateTaskInput();
  this._task.run();
};

TaskUnitTest.prototype.saveConf = function (testNode) {
  //args
  for (var arg in this._args) {
    xmlHelper.appendNode(testNode, "arg", null,
        {"name": arg, "value": this._args[arg]});
  }
  //asserts
  for (var i=0; i<this._asserts.length; i++) {
    xmlHelper.appendNode(testNode, "assert", null,
        {"path": this._asserts[i].getPath(),
         "value": this._asserts[i].getValue()});
  }
};

TaskUnitTest.prototype.loadConf = function (testNode) {
  //args
  var argNodes = testNode.getElementsByTagName("arg");
  for (var i=0; i<argNodes.length; i++) {
    var name = argNodes[i].getAttribute("name");
    var value = argNodes[i].getAttribute("value");
    this.setArg(name, value);
  }
  //asserts
  var assertNodes = testNode.getElementsByTagName("assert");
  for (var i=0; i<assertNodes.length; i++) {
    var path = assertNodes[i].getAttribute("path");
    var regex = assertNodes[i].getAttribute("value");
    this.createAssert(path, regex);
  }
};

