var EXPORTED_SYMBOLS = ["Highlighter"];
Components.utils.import('resource://indexdata/util/htmlHelper.js');
Components.utils.import('resource://indexdata/util/logging.js');
var logger = logging.getLogger();

var Highlighter = function (domDoc, cb) {
  this.doc = domDoc;
  this.cb = cb;
  if ( domDoc.cf_displaydoc ) {
    //dump("Highlighter, using display doc \n");
    this.doc = domDoc.cf_displaydoc;
  }
  this.outlineStyle = '2px dashed #ff7deb';
  this.oldOlStyle = '';
  this.current = null;
  this.destroyed = false;
  var context = this;

  // Notification Box
  var wm = Components.classes["@mozilla.org/appshell/window-mediator;1"]
                  .getService(Components.interfaces.nsIWindowMediator);
  var mainWindow = wm.getMostRecentWindow("navigator:browser");
  this.nb = mainWindow.getNotificationBox(this.doc.defaultView);
  if (!this.nb.getNotificationWithValue('highlight-active')) {
    var nButtons = [{
      label: 'Cancel',
      accessKey: 'C',
      popup: null,
      callback: function() {
        context.destroy();
      }
    }];
    this.nb.appendNotification(
      "Node selector active, choose a node... " +
          "([ESC] cancels, [+] parent, [-] first child, [Enter] selects)",
      'highlight-active',
      'chrome://browser/skin/Info.png',
      this.nb.PRIORITY_WARNING_MEDIUM,
      nButtons);
  }

  mainWindow.addEventListener("keypress", function (e) {
    context.onKeyPress(e);
  }, false);

  // Per-element listeners
  this.eventListeners = {
    'mouseover': function (e) {context.onMouseOver(e);},
    'mouseout': function (e) {context.onMouseOut(e);},
    'click': function (e) {
      e.preventDefault();
      e.stopPropagation();
      context.current.style.outline = context.oldOlStyle;
      cb.apply(context.current, [e]);
    }
  };
  this.attachEventListeners(this.eventListeners);
}

Highlighter.prototype = {
  destroy: function () {
    this.detachEventListeners(this.eventListeners);
    this.removeHighlight();
    // Not quite sure how to reproduce but sometimes this.nb.getNotificationWithValue
    // isn't there. Odd since a Highlighter is created with a notification box. Maybe
    // when you cancel too fast, before it's fully instantiated? Unlikely. Anyhow,
    // hopefully adding a check for that helps smooth things along:
    if (this.nb && this.nb.getNotificationWithValue) {
      let n = this.nb.getNotificationWithValue('highlight-active');
      if (n) n.close();
    }
    this.destroyed = true;
  },
  onMouseOver: function (e) {
    this.highlightNode(e.target);
    e.stopPropagation();
  },
  onMouseOut: function (e) {
    var element = e.target;
    element.style.outline = this.oldOlStyle;
    e.stopPropagation();
    var n = this.nb.getNotificationWithValue('highlight-active');
    if (n) {
      n.label = "Node selector active, choose a node... ([ESC] cancels, [+] parent, [-] first child, [Enter] selects)";
    }
  },
  onKeyPress: function (e) {
    switch(e.keyCode) {
      case 13: // enter
        this.current.style.outline = this.oldOlStyle;
        this.cb.apply(this.current, [e]);
        break;
      case 27: // escape
        this.destroy();
        break;
    }
    if (this.current) {
      switch(String.fromCharCode(e.charCode)) {
        case '+':
          if (this.current.parentNode &&
            this.current.localName != "body") {
            this.highlightNode(this.current.parentNode);
          }
          break;
        case '-':
          if (this.current.childNodes && this.current.childNodes[0]) {
            if (this.current.childNodes[0].localName) {
              this.highlightNode(this.current.childNodes[0]);
            } else if (this.current.childNodes[1] &&
                this.current.childNodes[1].localName) {
              this.highlightNode(this.current.childNodes[1]);
            }
          }
          break;
      }
    }
  },
  attachEventListeners: function (eventNamesHandlers) {
    var attachToElement = function (element) {
      for (var key in eventNamesHandlers)
        element.addEventListener(key, eventNamesHandlers[key], false);
    }
    this.__foreachChildDo(this.doc.body, attachToElement);
    var frames = htmlHelper.collectFrames(this.doc);
    for (var i=0; i<frames.length; i++)
      this.__foreachChildDo(frames[i], attachToElement);
  },
  detachEventListeners: function (eventNamesHandlers) {
    var detachFromElement = function (element) {
      for (var key in eventNamesHandlers)
        element.removeEventListener(key, eventNamesHandlers[key], false);
    }
    this.__foreachChildDo(this.doc.body, detachFromElement);
    var frames = htmlHelper.collectFrames(this.doc);
    for (var i=0; i<frames.length; i++)
      this.__foreachChildDo(frames[i], detachFromElement);

  },
  highlightNode: function (element) {
    this.removeHighlight();
    var n = this.nb.getNotificationWithValue('highlight-active');
    if (n) {
      n.label = "Node selector active, " + element.localName + " highlighted ([ESC] cancels, [+] parent, [-] first child)";
    }
    this.current = element;
    this.oldOlStyle = element.style.outline;
    element.style.outline = this.outlineStyle;
  },
  removeHighlight: function () {
    if (this.current) {
      this.current.style.outline = this.oldOlStyle;
      this.current = null;
    }
  },
  __foreachChildDo: function (element, action) {
    if (element && element.hasChildNodes()) {
      var children = element.childNodes;
      for (var i = 0; i < children.length; i++) {
        this.__foreachChildDo(children[i], action);
        action(children[i]);
      }
    }
  }
}
