var EXPORTED_SYMBOLS = ["Task"];
Components.utils.import('resource://indexdata/runtime/TaskUnitTest.js');
Components.utils.import('resource://indexdata/util/textHelper.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/logging.js');
Components.utils.import('resource://indexdata/util/EventList.js');
Components.utils.import('resource://indexdata/runtime/Block.js');

// use 'execution' logger to log all messages, this is IMPORTANT
// so messages are properly passed on.
var logger = logging.getLogger('execution');

var Task = function (connector, name) {
  this.connector = connector;
  this.block = new Block(this);
  this.id = textHelper.randString(32);
  this.name = name;
  this.polymorphic = false;
  this.isInterrupted = false;
  this.interruptTimer = Components.classes["@mozilla.org/timer;1"]
    .createInstance(Components.interfaces.nsITimer);
  this.errorTimeout = 30000 // Library search pages can be slow!
  this.tests = [];
  this.lastRunData = null;
  this.data = {}; // must be maintained as a reliable reference
  //used events: onTaskStart, onTaskFinish, onStepFinish, onMessageLog
  this._eventList = new EventList();
  var context = this;
  this.addHandler("onTaskStart", function () {context.initRunData();});
  this.clearData();
};
  
Task.dataKeys = ['input', 'output', 'temp'];
Task.connectorDataKeys = ['session', 'system'];

Task.prototype.clearData = function () {
  this.rebuildData({});
};

// called on Task.run(), enables session test values to update after 
// connector is loaded and builder data browser to clear
Task.prototype.updateConnectorDataReferences = function () {
  for (let i = 0; i <  Task.connectorDataKeys.length; i++) {
    let key = Task.connectorDataKeys[i];
    this.data[key] = this.connector.data[key]; 
  };
};

// rebuild data from keys of obj, preserving task.data reference
Task.prototype.rebuildData = function (obj) {
  // preserve this.data object so we don't break references
  for (let i = 0; i < Task.dataKeys.length; i++) {
    let key = Task.dataKeys[i];
    if (typeof this.data[key] !== 'undefined') delete this.data[key]; 
    if (typeof obj[key] !== 'undefined') this.data[key] = obj[key]; 
    else this.data[key] = {};
  };  
  // preserve existing this.connector.data values (system space and such)
  // yet allow overriding via obj
  for (let i = 0; i <  Task.connectorDataKeys.length; i++) {
    let key = Task.connectorDataKeys[i];
    if (typeof obj[key] !== 'undefined') {
      this.connector.data[key] = obj[key]; 
      this.data[key] = this.connector.data[key]; 
    } else this.data[key] = this.connector.data[key];
  };

  // be backwards compatible
  this.input = this.data.input;
  this.output = this.data.output;
};

Task.prototype.getName = function (doc) {
  return this.name;
};

Task.prototype.addHandler = function (name, callback, userData) {
  this._eventList.attachEvent(name, callback, userData);
};

Task.prototype.removeHandler = function (name, callback) {
  this._eventList.detachEvent(name, callback);
};

Task.prototype.loadConf = function (taskNode) {
  this.clearItems();
  //blocks/steps
  this.block = Block.fromConf(this, taskNode);
  //test
  var testNodes = taskNode.getElementsByTagName("test");
  this.loadTestsFromConf(testNodes);
  var poly = taskNode.getAttribute("polymorphic");
  if (poly && poly == "false" )
    this.polymorphic = false;
  else
    this.polymorphic = true;
  // this way around, loading old connectors that do not have the
  // flag set, we default to polymorphic, which was the old way.
};

Task.prototype.loadTestsFromConf = function (testNodes) {
  for (var i=0; i<testNodes.length; i++) {
    var testName = testNodes[i].getAttribute("name");
    var test = this.createTest(testName);
    test.loadConf(testNodes[i]);
  }
  return this.tests;
}

Task.prototype.saveConf = function (taskNode) {
  // Copy the polymorphic flag from the template into the connector
  // This is necessary, because we do not have the template available
  // in the engine.
  var templ = this.getTemplate();
  if ( templ ) {
    var poly = templ.properties.polymorphic;
    if ( poly == undefined )
      poly = false;
    taskNode.setAttribute("polymorphic", poly?"true":"false");
  }

  //block
  this.block.saveConf(taskNode);

  //tests
  for (var i=0; i<this.tests.length; i++) {
    var testNode = xmlHelper.appendNode(taskNode, "test", null,
        {"name" : this.tests[i].getName()});
    this.tests[i].saveConf(testNode);
  }
}

// delete all items
Task.prototype.clearItems = function () {
  return this.block.clearItems();
};


// attaches step instance to the task
Task.prototype.attachStep = function (step, recurse) {
  if (typeof step.isAlt == "undefined") {
    step.isAlt = false;
  }
  step.task = this;
  step.init(this);
  if (recurse) {
    let blocks = step.blocks;
    for (let i=0; i<blocks.length; i++) {
      let b = blocks[i];
      b.task = this;
      for (let j=0; j<b.steps.length; j++) {
        this.attachStep(b.steps[j]);
      }
    }
  }
};

// interrupt running task, optional callback that gets executed before
// the task is resumed, the callback return value controls whether
// the taks is resumed (true) or not (false)
Task.prototype.interrupt = function (delay, cb) {
  if (this.isInterrupted)
    this.warn("Interrupting an already interrupted task.");
  this.isInterrupted = true;
  var context = this;
  //timer that get's called when we should wake up
  var timerCb = { notify: function (timer) {
    logger.debug("Interrupt timer firing off");
    var cbreturn = true;
    if (typeof cb == "function" ) {
      try {
        cbreturn = cb();
      } catch (e) {
        //make sure the callback has not resumed on it's own
        if (context.isInterrupted) {
          logger.debug("Handling error after an interrupt");
          context.block.handleError(e);
        }
      }
    }
    if (context.isInterrupted && cbreturn !== false) {
      logger.debug("Resuming the execution via the interrupt timer");
      context.block.resume();
    }
  }};
  this.interruptTimer.initWithCallback(timerCb, delay,
      Components.interfaces.nsITimer.TYPE_ONE_SHOT);
  return this.interruptTimer;
};


// run task
Task.prototype.run = function (/*0-indexed*/ from, /*exclusive*/to) {
  this.updateConnectorDataReferences();
  this.block.run(null, from, to);
};


Task.prototype.resume = function () {
  this.block.resume();
};

Task.prototype.complete = function () {
  var block = this.block;
  do {
    block.complete();
  } while (block = block.nextBlock); 
};

// pretty logging

Task.prototype.log = function (level, message, step) {
  var prefix =  "Task [" + this.name + "] - ";
  var stepInfo = " - ";
  if (step != undefined && "getClassName" in step) {
    stepInfo = "- '" + step.getClassName() + "': ";
  }
  var msg = stepInfo + message;
  logger.log(level, msg, prefix);
  var confLevel = logger.getLevel();
  if (level < confLevel) return;
  this._eventList.dispatchEvent("onMessageLog",
      Level.getName(level),
      prefix + Level.getName(level) + msg);
};
Task.prototype.info = function (message, step) {
  this.log(Level.INFO, message, step);
};
Task.prototype.warn = function (message, step) {
  this.log(Level.WARN, message, step);
};
Task.prototype.debug = function (message, step) {
  this.log(Level.DEBUG, message, step);
};

//this method should not be called directly by a step!
//instead throw a StepError!!
Task.prototype._error = function (message, step) {
  this.log(Level.ERROR, message, step);
};

// Run data
Task.prototype.initRunData = function () {
  this.lastRunData = {};
  this.lastRunData.urls = {};
};

Task.prototype.postprocessRunData = function () {
// perhaps have it calculate some stats here?
};

Task.prototype.addRunDatumUrlEvent = function (url, event) {
  var datum = {
    "event": event,
    "timestamp": new Date().getTime(),
  }
  if (!this.lastRunData.urls[url]) {
    this.lastRunData.urls[url] = {};
    this.lastRunData.urls[url].events = [];
  }
  this.lastRunData.urls[url].events.push(datum);
};

Task.prototype.getRunData = function () {
  return this.lastRunData;
};



// arguments

Task.prototype.getArgNames = function () {
  return this.getTemplate().getArgNames();
};

Task.prototype.getUsedArgs = function () {
  return this.block.getUsedArgs();
};

Task.prototype.setArgValue = function (argName, argValue) {
  this.input[argName] = argValue;
};

Task.prototype.getArgValue = function (argName) {
  return typeof this.input[argName] != "undefined" ? this.input[argName] : "";
};

Task.prototype.isArgUsed = function (argName) {
  return this.getUsedArgs().indexOf(argName)!=-1
};

Task.prototype.detach = function () {
  this.connector.detachTask(this);
};

Task.prototype.compareTo = function (task) {
  //sort first by order loaded, if template
  //available, by name otherwise
  var templateA = this.getTemplate();
  var templateB = task.getTemplate();
  if (templateA && templateB) {
    if (templateA.rank > templateB.rank) return 1;
    if (templateA.rank < templateB.rank) return -1;
  } else {
    if (this.name > task.name) return 1;
    if (this.name < task.name) return -1;
  }

  //then by arg list
  var argCountA = this.getUsedArgs().length;
  var argCountB = task.getUsedArgs().length;
  if (argCountA < argCountB)
    return -1;
  if (argCountA > argCountB)
    return 1;
  // argCountA must be equal to argCountB
  return 0;
};

Task.prototype.createTest = function (name) {
  var test = new TaskUnitTest(this, name);
  this.tests.push(test);
  return test;
};

Task.prototype.createDefaultTests = function () {
  var testNodes = this.getTemplate().testNodes;
  this.loadTestsFromConf(testNodes);
};

Task.prototype.removeTestByIdx = function (idx) {
  return this.tests.splice(idx,1);
};

Task.prototype.getTests = function () {
  return this.tests;
};

Task.prototype.getTemplate = function () {
  var connectorTemplate = this.connector.getTemplate();
  if (connectorTemplate) {
    var taskTemplate = connectorTemplate.getTaskTemplate(this.name);
    if (typeof taskTemplate == 'undefined')
      this._error("Connector template does not contain task '"
          +this.name+"' definition");
    return taskTemplate;
  } else {
    this.warn("Missing connector template.");
    return null;
  }
};
