var EXPORTED_SYMBOLS = ["ParseXpattern"];

// parse a page using the XPattern

/* TODO:

- Why-not button: Point to a node, and modify the pattern so that
  it matches somewhere too.
- Patterns that contain ()'s and or-operators? Better editor for or-bags!
- Negations in the designer

- Low-intensity highlights for other hits on the screen, while editing
  and/or designing.

- UI improvements
*/

Components.utils.import('resource://indexdata/runtime/Step.js');
Components.utils.import('resource://indexdata/runtime/StepError.js');
Components.utils.import('resource://indexdata/runtime/Task.js');
Components.utils.import('resource://indexdata/util/xpattern.js');
Components.utils.import('resource://indexdata/util/xpatternText.js');
Components.utils.import('resource://indexdata/util/xpatternMaker.js');
Components.utils.import('resource://indexdata/util/xulHelper.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/textHelper.js');
Components.utils.import('resource://indexdata/util/inspector.js');
Components.utils.import('resource://indexdata/util/jsonPathHelper.js');
Components.utils.import('resource://indexdata/thirdparty/jsonPath.js');
Components.utils.import('resource://indexdata/ui/app.js');
Components.utils.import('resource://indexdata/util/logging.js');

if (typeof DEBUG == "undefined") const DEBUG = true;
var logger = logging.getLogger();

var ParseXpattern = function () {
  this.className = "ParseXpattern";
  this.conf = {};
  this.conf["xpattern"] = "";
  this.conf["hitarea"] = xmlHelper.nodeSpecFromString("html/body");
  this.conf["jsonPath"] = { path: "$.output", key:"results" };
  this.conf["highlighthits"] = true;
  this.conf["autogenerate"] = true; // generate pattern at every click
  this.conf["hitnumbercheck"] = false; // check the hitnumbers
  this.conf["failhitnumber"] = true; // fail step if missing hitnumbers
  this.conf["removehitnumber"] = true; // do not pass them on to the system
  this.conf["keephitnumber"] = true; // remember them from page to page
  this.conf["hitnumbervar"] = { path: "$.session", key:"hitnumber" };
  this.conf["xpatternhistory"] = [];
};
ParseXpattern.prototype = new Step();
ParseXpattern.prototype.constructor = ParseXpattern;

ParseXpattern.prototype.init = function() {
};


//////
// Highlight colors

/**
 * Converts an HSV color value to RGB. Conversion formula
 * adapted from http://en.wikipedia.org/wiki/HSV_color_space.
 * Assumes h, s, and v are contained in the set [0, 1] and
 * returns a #RRGGBB string
 */
function hsvToRgb(h, s, v){
    var r, g, b;

    var i = Math.floor(h * 6);
    var f = h * 6 - i;
    var p = v * (1 - s);
    var q = v * (1 - f * s);
    var t = v * (1 - (1 - f) * s);

    switch(i % 6){
        case 0: r = v, g = t, b = p; break;
        case 1: r = q, g = v, b = p; break;
        case 2: r = p, g = v, b = t; break;
        case 3: r = p, g = q, b = v; break;
        case 4: r = t, g = p, b = v; break;
        case 5: r = v, g = p, b = q; break;
    }
    r = Math.floor(r*256);
    b = Math.floor(b*256);
    g = Math.floor(g*256);
    var rgb = r * 256 * 256 + g * 256 + b;
    displaycolor = rgb.toString(16);
    while ( displaycolor.length < 6 )
      displaycolor = "0" + displaycolor;
    displaycolor = "#" + displaycolor;
    return displaycolor;
} // hsvToRgb

var hue = 0;
function gethighlightcolor() {
  hue += .09 + Math.random() * .005; //makes sure we never quite repeat
  hue = hue - Math.floor(hue); // keep it in 0..1
  var col = hsvToRgb(hue, .7, .98 );
  logger.debug("hlcolor: hue=" + hue + " col=" + col);
  return col;
};


function trimpat(p) { // trim patterns to ignore small changes
  //logger.debug("trim: before: '" + p + "'" );
  p = p.replace(/^\s+/,""); // leadign whitespace
  p = p.replace(/\s+$/,""); // trailing whitespace
  p = p.replace(/\s+/g," "); // repeated whitespace
  //logger.debug("trim: after:  '" + p + "'" );
  return p;
}; // trimpat



ParseXpattern.prototype.draw = function(surface, win) {
  var context = this;
  this.inBuilder = true; // pass a signal to run() to do highlights

  this.tabs = xulHelper.tabBox( surface,
      [ "Design", "Edit","History", "Options", "Hitnumber check" ],
      { flex: 1 });

  // Each tab may return a callback to update its content if the pattern
  // gets changed by other tabs. SetPattern will call each of these
  this.designercallback = this.drawCreateTab( this.tabs[0], win, this);
  this.editcallback = this.drawEditTab( this.tabs[1], this);
  this.historycallback = this.drawHistoryTab( this.tabs[2], this);

  this.drawOptionTab(  this.tabs[3], this);
  this.drawCheckTab(  this.tabs[4], this);

}; // draw

// Set the XPattern. Updates all places it is displayed, as
// well as the xpattern in conf, and the history.
// (carefully not reassigning identical values, to avoid
// event cascades)
ParseXpattern.prototype.setPattern = function(patt) {
  //logger.debug("setPattern: p=" + patt + " x=" + this.conf.xpattern +
  //  " cl.tc=" + this.cpatt.textContent + " e= " + this.epatt.value );
  logger.debug("Updating xpattern " + patt );
  if ( this.conf.xpattern != patt ) {
    logger.debug("Updating xpattern in conf"  );
    this.conf.xpattern = patt;
  }
  if ( this.designercallback )
    this.designercallback(patt);
  if ( this.editcallback )
    this.editcallback(patt);
  if ( this.historycallback )
    this.historycallback(patt);
  /*
  if ( this.cpatt && this.cpatt.textContent != patt ) {
    logger.debug("Updating cpatt label" );
    this.cpatt.textContent = patt;
    // textContent seems to make autowrap, value makes one long line.
    // nothing seems to respect newlines that I want to put in.
    // TODO - Refactor the update functions into the tabs, and build 
    // a multi-line display in the create tab.
  }
  if ( this.epatt && this.epatt.value != patt ) {
    logger.debug("Updating epatt display" );
    this.epatt.value = patt;
  }
  */
}; // setPattern

// Helper to collect $variable names
var varnames = []; 
ParseXpattern.prototype.varnamelist = function () {

  function addvarname(v) {
    //logger.debug("Checking variable '" + v + "'" );
    for ( var i=0; i<varnames.length; i++)
      if (varnames[i] == v)
        return; // seen already
    //logger.debug("Adding variable '" + v + "'" );
    varnames.push(v);
  }; // addvarname

  var templ = this.task.getTemplate().properties.data;
  //logger.debug("Got template " + templ );

  var jp = this.conf.jsonPath.path + "." + this.conf.jsonPath.key;
  //logger.debug("Getting jsonpath " + jp );
  var obj = jsonPath(templ, this.conf.jp);
  //logger.debug("Got jsonpath obj " + obj );
  
  if ( !obj ) {
    obj = jsonPath(templ, "$.output.results"); // a good guess
  }
  if ( !obj ) {
    obj = jsonPath(templ, "$.output"); // another good guess
  }
  //obj = obj[0][0]; // it seems always to be a double array ??!!
  // but if there was no such thing in the template, we get nothing much
  // and can't dereference it. Defensive coding below:
  if ( Array.isArray(obj) )
    obj=obj[0];
  if ( Array.isArray(obj) )
    obj=obj[0];
  //dump("got obj: " + obj + "\n");
  //dump(JSON.stringify(obj) + "\n");
  for ( var k in obj ) {
    if ( obj[k].length == 0 ) // plain variable
      addvarname(k);
    else { // contains elements, is a group
      var subobj = obj[k][0];
      for (var i in subobj ) {
        //logger.debug("Subobj " + subobj + " i=" + i + ": " + subobj[i] );
        addvarname(i);
      }
    }
  }
  // Get variables from xpattern history as well
  if ( this.conf.xpatternhistory ) {
    // may not be, when loading a connector
    for ( var i = 0; i < this.conf.xpatternhistory.length; i++) {
      var patt = this.conf.xpatternhistory[i];
      var dollars = patt.match( /\$\w+/g );
      if (dollars) {
        for ( var d = 0; d < dollars.length; d++ )
          addvarname( dollars[d].replace(/\$/,"" ) );
      }
    }
  }

}; // varnamelist

/////////////////////
// Draw the options tab
ParseXpattern.prototype.drawOptionTab = function(surface, context) {
  //dump("DrawOptionTab Start \n");
  var vbox = xmlHelper.appendNode(surface, "vbox", null, {flex:1}, null);
  var inpbox = xmlHelper.appendNode(vbox, "hbox", null, null, null);
  var hitarea = xulHelper.xpathField( inpbox, this, "hitarea",
          "Area where hits are to be found" );
  var hlbox = xmlHelper.appendNode(vbox, "hbox", null, null, null);
  var highlightcheck = xulHelper.checkbox( hlbox, this,
          "highlighthits", "Highlight hits on the page" );
  highlightcheck.addEventListener("command", function(e) {
    var hitarea = context.conf['hitarea'];
    var pageDoc = context.getPageDoc();
    var node = xmlHelper.getElementByNodeSpec(pageDoc, hitarea);
    if (node != null) {
        xulHelper.unhighlightAll(node,false);
    }  // Ok to unhighlight both on check and uncheck
  }, false);
  var agbox = xmlHelper.appendNode(vbox, "hbox", null, null, null);
  var autogencheck = xulHelper.checkbox( agbox, this,
          "autogenerate", "Autogenerate the pattern when " +
          "changing anything in the designer");
  var hbox = xmlHelper.appendNode(vbox, "hbox", null, null, null);
  xulHelper.jsonPathField(hbox, this, this.conf, "jsonPath",
      "Where to put the results: ",
      {path:"$.output", key:"results", append:jsonPathHelper.REPLACE});
}; // drawOptionTab


////////////////
// Tab to control the hitnumber sanity check
ParseXpattern.prototype.drawCheckTab = function(surface, context) {
  var vbox = xmlHelper.appendNode(surface, "vbox", null, {flex:1}, null);
  xulHelper.checkbox( vbox, this, "hitnumbercheck",
             "Check that hitnumbers are continuous. " +
             "(Your pattern must extract hitnumbers from the page)");
  xulHelper.checkbox( vbox, this, "failhitnumber",
                      "Fail the step if detecting a missing hitnumber." +
                      " (Otherwise just logs an error)" );
  xulHelper.checkbox( vbox, this, "removehitnumber",
             "Remove the hitnumber from the results after checking");
  xulHelper.checkbox( vbox, this, "keephitnumber",
             "Keep the hit number for the next page");
  xulHelper.jsonPathField(vbox, this, this.conf, "hitnumbervar",
                          "Where to keep the hit number ",
                          {path:"$.session", key:"hitnumber"});
}; // drawCheckTab 


//////////////////////
ParseXpattern.prototype.drawEditTab = function(surface, context) {
  //dump("drawEdittab start \n");
  var vbox = xmlHelper.appendNode(surface, "vbox", null,
      {"flex":1,"align":"stretch"}, null);
  var hbox = xmlHelper.appendNode(vbox, "hbox", null,
      {"flex":1,"align":"stretch"}, null);
  var xpatt = xulHelper.inputField( hbox, this, "xpattern", "XPattern", 120,
            { "multiline": true, "rows": 30,
            timeout:500,  // ###
              "flex": 1, "width" : 580, "align":"stretch" } );
  var hbox2 = xmlHelper.appendNode(vbox, "hbox", null, null, null);
  var formatButton = xmlHelper.appendNode(hbox2, "button",
        null, {"label": "Reformat", "width" : 120 }, null);
  var errbox = xmlHelper.appendNode(hbox2, "hbox", null, { "flex":1}, null);
  var lastsyntaxcheck = "";


  // callback function to check the xpattern syntax and highlight hits on the page
  var checksyntax = function (ev) {
    var patt = xpatt.value;
    var chkpatt = patt; // ### Clean whitespace, shorten $variable names, etc
    if ( chkpatt == lastsyntaxcheck ) {
      logger.debug("Not checking again, same pattern");
      return;
    }
    lastsyntaxcheck = chkpatt;
    xmlHelper.emptyChildren(errbox);
    var xp = null;
    try {
      if (patt) {
        var par = new XpatternTextParser(patt);
        xp = par.parse();
        xpatt.removeAttribute("class");
        formatButton.setAttribute("disabled",false);
      }
    } catch (e) {
      var lasterrpos = par.getErrorPos();
      var line = 1;
      var col = 1;
      var p = 0;
      while ( ++p < lasterrpos ) {
          if ( patt.substring( p, p+1) == "\n" ) {
              line ++;
              col = 1;
          } else
              col ++;
      }
      //xulHelper.captionField(errbox,"" + e );
      xulHelper.labelField(errbox, "" + line + ":" + col +":" );
      var errcapt = xulHelper.captionField(errbox,  par.getErrorMsg());
      errbox.addEventListener("click", function(e){
          xpatt.selectionStart = lasterrpos;  // positions the caret
          xpatt.selectionEnd = lasterrpos;
      }, false );
      errcapt.setAttribute("tooltipText",
            "Click to position the cursor at the error" );
      xpatt.setAttribute("class", "incomplete");
      formatButton.setAttribute("disabled",true);
      xp = null; // don't try to count hits etc
    }
    if ( xp ) {
      if(1) {
        var task = new Task(context.task.connector, "dummy");
        try { // run the step with a fake task parameter to get the outputs
          context.run(task);
          var hits = "" + task.xpatternHits.length + " hit";
          if ( task.xpatternHits.length != 1)
            hits += "s";
          if (task.xpatternwarnings) {
            hits += " " + task.xpatternwarnings + "w";
          } else { // count number of pieces in the hits
              var pcs = 0;
              for (var i=0; i<task.xpatternHits.length; i++) {
                var h = task.xpatternHits[i];
                for ( var j=0; j<h.hits.length; j++ ) {
                  if ( h.hits[j].name && h.hits[j].value )
                    pcs++;
                }
              }
            hits += " " + pcs + "p";
          }
        xulHelper.labelField(errbox, hits);
        } catch(e) {
          logger.debug("Running generated pattern failed with " + e );
          logger.debug("Trying edit pattern " + this.cf_patt + " failed " +
              "with " + e  );
          if ( e.fileName )
            logger.debug(" at " + e.fileName + "." + e.lineNumber );
          if ( e.message == "No hits found" ) {
            xulHelper.captionField(errbox, "No hits");
          }
        }
      }
    } // count hits
  }; // checksyntax

  // Reformat the xpattern
  // or the selected part of it
  function reformat(e) {
    try {
      var patt = xpatt.value;
      var offsetcorrection = 0;
      var before = "";
      var after = "";
      //dump("reformat: sel= " + xpatt.selectionStart + " - " + xpatt.selectionEnd + "\n" );
      var selstart = xpatt.selectionStart;
      var selend = xpatt.selectionEnd;
      var indent = 0;
      if ( selstart != selend ) {
        offsetcorrection = selstart;
        before = patt.substring(0, xpatt.selectionStart);
        after = patt.substring(xpatt.selectionEnd);
        patt = patt.substring(xpatt.selectionStart, xpatt.selectionEnd);
        // check current indentation
        var p = selstart-1;
        while ( (p >= 0) && xpatt.value.charAt(p) == " ") {
          p--;
          indent ++;
          //dump("counting indent " + indent + " at " + p +
          //     "=" + xpatt.value.charAt(p) + "\n");
        }
        indent = Math.floor(indent / 2);
        //dump("sel: indent = " + indent + "\n" +
        //     "B='" + before + "'\n" +
        //     "P='" + patt + "'\n" +
        //     "A='" + after + "'\n");
      }
      var oldpatt = patt;
      var par = new XpatternTextParser(patt);
      var xp = par.parse();
      //dump("got offset " + xp.getStringoffset() + " for root \n");
      var s;
      if ( patt.indexOf("\n") == -1 ) {
        s = xp.dumpString(indent); // structured dump
        s = s.replace(/^ +/,""); // remove trailing space
      } else {
        s = xp.dumpString(-1); // one-liner
      }
      if ( s.charAt(s.length-1) == "\n" )
        s = s.slice(0,-1); // remove trailing newline
      xpatt.value = before + s + after;
      var newend = selstart + s.length;
      if ( selstart != selend ) {
        // Reset the selection
        xpatt.selectionStart = selstart;
        xpatt.selectionEnd = selstart + s.length;
      } else { // no selection
        // whole pattern reformatted, place cursor to the beginning
        xpatt.selectionStart = 0;
        xpatt.selectionEnd = 0;
      }
      //dump("After reformat, setting cpatt. old= " +
      //    context.cpatt.value + " to " + xpatt.value + "\n");
      // Pass value to create tab too, and history
      context.setPattern(xpatt.value);
      xpatt.focus(); // focus back to the input, to show selection
    } catch (e) {
      dump("Caught an exception " + e + "\n");
    }
  };  // reformat


  formatButton.addEventListener("command", reformat, false );

  // Helper to highlight the DOM node on the page that matches
  // the xpattern node that corresponds to the given position
  // on the xpattern string.
  function highlightDomNode( pattern, position ) {
    try {
      var hitarea = context.conf['hitarea'];
      var pageDoc = context.getPageDoc();
      var node = xmlHelper.getElementByNodeSpec(pageDoc, hitarea);
      if (node == null)
          return; // Can't do nothing
      xulHelper.unhighlightAll(node,false);
      var par = new XpatternTextParser(pattern);
      par.setOffsetCorrection(0); // tell the parser we want offsets!
      var xp = par.parse();
      xp.askForNodes();
      var keynode = null;
      xp.foreachnode( function(n) {
        n.setVariable(""); // Clear all variables
        if ( n.getStringoffset() == position ) {
          keynode = n; // and remember the *last* matching node
          // Has to be last, because or-groups produce hidden
          // pattern nodes at the same offset as their first
          // alternative, and we don't want to highligh the whole
          // or-group.
        }
      } );
      if ( keynode ) {
        keynode.setVariable("$$$");  //impossible name
        var hitsarray = xp.match(node);        
        if ( context.conf['highlighthits'] )
          context.highlightHits(hitsarray);
      }
    } catch (e) {
      logger.debug("Oops, highlightDomNode caught an exception " + e);
      // should not happen. If it does, we don't get a highlight
      // which is kind of acceptable.
    }
  }; // highlightDomNode

  // Select whole nodes with dblclick,
  // including children
  function dblclick(e) {
    var patt = xpatt.value;
    var p = xpatt.selectionStart;
    //logger.debug("dblclick:  sel: " +
    //   xpatt.selectionStart + "-" + xpatt.selectionEnd );
    if ( xpatt.selectionStart == xpatt.selectionEnd )
      return; // nothing selected, do not expand
    var firstchar = xpatt.value.charAt(xpatt.selectionStart);
    //logger.debug("dblclick: first = '" + firstchar + "'\n");
    if ( !firstchar.match(/[A-Za-z0-9\{\[\(]/) )
      return; // not at a tag
    var terminators = ":{}()|" ; // TODO Check groups?
    var open = "{[(";
    var close = ")]}";
    // scan to the end of the current node
    while ( terminators.indexOf(patt.charAt(p)) < 0 ) {
      p++;
    }
    //logger.debug("dblclick: node itself ends at " + p );
    // scan its children
    if ( open.indexOf(patt.charAt(p)) >= 0  ) {
      var depth = 0;
      do {
        if (open.indexOf(patt.charAt(p)) >= 0 )
          depth ++;
        if (close.indexOf(patt.charAt(p)) >= 0 )
          depth --;
        p++;
        //logger.debug("dblclick: children at " + p + " depth=" + depth );
      } while (depth > 0 );
    }
    // and trailing whitespace
    while ( patt.charAt(p) == " ")
      p++;
    //logger.debug("dblclick: hl ends at " + p );
    xpatt.selectionEnd = p;
    highlightDomNode( patt, xpatt.selectionStart );
  };  //dblclick

  var checktimer = Components.classes["@mozilla.org/timer;1"]
          .createInstance(Components.interfaces.nsITimer);

  function refreshtimer() {
    checktimer.cancel();
    checktimer.initWithCallback( {notify: checksyntax} , 500,
        Components.interfaces.nsITimer.TYPE_ONE_SHOT);
  }
  function update(e) {
    //logger.debug("edit:update " + e.type );
    context.setPattern(xpatt.value);
    refreshtimer();
  }
  xpatt.addEventListener("dblclick", dblclick, false );
  xpatt.addEventListener("input", refreshtimer, false);
  // refresh the time on every keystroke, but update conf and tabs only when
  // leaving the edit, not to overfill the history
  xpatt.addEventListener("change", update, false);
  xpatt.addEventListener("command", update, false);
  checksyntax();

  // Callback when other tabs change the pattern
  var updatecallback = function(patt) {
    if ( xpatt.value != patt ) {
      xpatt.value = patt;
    } else {
      logger.debug("not updating edit input");
    }
  }
  
  return updatecallback;
}; // drawEditTab


////////////////////
// Designer tab
// Draw the first tab, with the xpattern designer stuff
// Third version, heavily refactored, with action menu
ParseXpattern.prototype.drawCreateTab = function(surface, win, context) {
  // watch out, this is a complex function, with event handlers
  // inside event handlers, and dynamically adjusted displays
  // on (at least) two levels!
  var mainwin = win;
  xmlHelper.emptyChildren(surface);
  //dump("drawCreateTab start\n");

  var vbox = xmlHelper.appendNode(surface, "vbox", null, {flex:1}, null);

  var hbox = xmlHelper.appendNode(surface, "hbox", null, null, null);

  //var xpatt = xmlHelper.appendNode(vbox, "label",this.conf.xpattern ,
  //                                 { width:"100%" ,style:"font-weight: bold;" } );
  var xpatt = xmlHelper.appendNode(vbox, "description",this.conf.xpattern ,
                                   { width:"100%" ,style:"font-weight: bold;" } );
  // TODO Choose one of label or description! Need to have wrapping text, and
  // newlines honoured in it! Seems not to be possible for now.
  var alertmsg =  xmlHelper.appendNode(vbox, "label",
         "Warning: Pattern too complex for the designer",
         { width:"100%" ,style:"font-weight: bold; color:red;", hidden:true } );
  var pleasemsg =  xmlHelper.appendNode(vbox, "label",
         "Please click on some part of a good hit",
         { width:"100%" ,style:"font-weight: bold; color:red;", hidden:true } );
  var waitmsg =  xmlHelper.appendNode(vbox, "label",
         "Recalculating, please wait",
         { width:"100%" ,style:"font-weight: bold; color:red;", hidden:true } );
  // waitmsg is made visible when someone else sets the pattern. That visibility
  // triggers repopulating the designer when the tab is clicked
  var nodebox = xmlHelper.appendNode(vbox, "vbox", null, null, null);
  this.nodebox = nodebox; // remember for rehighlight
  var nextaltnum = 1; // keep a count on alternative patterns, just for display
  var buttonbox = xmlHelper.appendNode(vbox, "hbox", null, null, null);
  var addbutattr = { "label": "Add a field" };
  if ( this.conf.xpattern )
    addbutattr.hidden = true;
  var addNodeButton = xmlHelper.appendNode(buttonbox, "button",
      null, addbutattr, null);
  xulHelper.captionField(buttonbox,"", { "width":20 } );
  // TODO - Make the button visible (and working) if no autogenerate
  var generateButton = xmlHelper.appendNode(buttonbox, "button",
      null, {"label": "generate a pattern", "hidden": true}, null);
  xulHelper.captionField(buttonbox,"", { "flex":1 } );
  var toolmenu = xulHelper.menu(buttonbox, "Special Actions" );
  addFieldMenu(toolmenu, null, nodebox);
  addGroupMenu(toolmenu, null, nodebox);
  addAltMenu(toolmenu);
  clearMenu(toolmenu);

  populateDesigner();

  addNodeButton.addEventListener("click", function(e) {
      selectAndAddField( nodebox, null );   // null means add to the end
      addNodeButton.hidden = true;
    }, false );   // addNodeButton.click

  // When the user comes back to the tab, check if we need to repopulate the
  // designer lines
  if ( surface.cf_tab ) {  // defensive coding, should always be there
    surface.cf_tab.addEventListener("command", function(e) {
      logger.debug("Designer tab click. fc=" + nodebox.firstChild );
      if ( ! waitmsg.hidden ) {  // designer has been invalidated
        logger.debug("Repopulating the designer");
        xmlHelper.emptyChildren( nodebox );
        populateDesigner(true);
      }
    }, false);
  }

  generateButton.addEventListener("click", function(e) {
    generatePattern(false);
  }, false);

  // Little helper to log the line structure
  // call with just the msg, or nothing, to get started
  function dumplines( msg, line, indent ) {
    if ( !line && !indent ) {
      dumplines( msg, nodebox.firstChild, 1 );
      return;
    }
    msg = msg || "";
    while (line) {
      var str = msg + " ";
      for ( var i=0; i<indent; i++ )
        str += "  ";
      str += line.cf_lineType + " ";
      if ( line.cf_lineType == "field" ) {
        str += line.cf_tagname + " ";
      } else if ( line.cf_lineType == "group" ) {
        str += line.cf_groupType + " ";
      } else if ( line.cf_lineType == "ALT" ) {
        str += "-" + line.cf_altnum;
      }
      str += (line.cf_cardinality || "" ) + " ";
      str += (line.cf_varname || "" );
      logger.debug(str);
      // TODO : Attributes
      if ( line.cf_contentbox )
        dumplines( msg, line.cf_contentbox.firstChild, indent+1 );
      line = line.nextSibling;
    }

  }; // dumplines

  // A helper to clear all lines, and set up display details
  function clearLines() {
    xmlHelper.emptyChildren(nodebox);
    context.setPattern("");
    addNodeButton.hidden = false; // show the initial "add" button
    alertmsg.hidden = true; // hide old "too complex" message
    waitmsg.hidden = true; // hide old "recalculating" message
    var pageDoc = context.getPageDoc(); // unhighlight all nodes
    var hitarea = context.conf['hitarea'];
    var node = xmlHelper.getElementByNodeSpec(pageDoc, hitarea);
    if (node != null) {
        xulHelper.unhighlightAll(node,false);
    }
    nextaltnum = 1;
  }; // clearLines

  // Helpers for producing the pull-down menu structure.
  // Line must be a fieldLine, or compatible - it must have a
  // cf_cardinality, and a redraw function.

  // Simple menu point to clear the whole XPattern - only for the global menu
  function clearMenu( menu ) {
    var clearmenupoint = xulHelper.menuitem( menu, "Clear all");
    clearmenupoint.addEventListener("click", function(e) {
        clearLines();
      }, false);
  }; // clearMenu

  // Cardinality menu. pulldown should be true for pulldown menus,
  // false (or missing) for menu points in deeper menu structures
  function cardinalityMenu ( menu, line, pulldown ) {
    var cardmenu;
    if ( ! pulldown )
      cardmenu = xulHelper.menu( menu, "Cardinality" );
    else
      cardmenu = menu;
    var cardmenupoint = function ( label, cardinality, attrs ) {
      var attrs = { type:"radio", name:"card"};
      if ( line.cf_cardinality == cardinality )
        attrs.checked = true;
      var menupoint = xulHelper.menuitem(cardmenu,label+" "+cardinality, attrs);
        menupoint.addEventListener("click", function(e) {
          line.cf_cardinality = cardinality;
          line.redraw();
          generatePattern(true);
        }, false);
    }; // cardmenupoint
    cardmenupoint("Obligatory", "");
    cardmenupoint("Repeating", "+");
    cardmenupoint("Optional", "?");
    cardmenupoint("Repeating optional", "*");
    cardmenupoint("Repeating non-greedy", "+?");
    cardmenupoint("Repeating optional non-greedy", "*?");
  }; // drawCreateTab.cardinalityMenu

  function variableMenu (menu, line, pulldown ) {
    // Helper to make one menu point
    var varmenu;
    if ( ! pulldown )
      varmenu = xulHelper.menu( menu, "Variable" );
    else {
      varmenu = menu;
    }
    function varmenupoint ( varname, dispname ) {
      if (!dispname) dispname = varname;
      var attrs = { type:"radio", name:"var"};
      if ( line.cf_varname == varname )
        attrs.checked = true;
      var menupoint = xulHelper.menuitem(varmenu,dispname, attrs);
        menupoint.addEventListener("click", function(e) {
          line.cf_varname = varname;
          line.redraw();
          generatePattern(true);
        }, false); // click
    }; // varmenupoint
    context.varnamelist();
    varmenupoint("","(none)");
    for (var i=0; i < varnames.length; i++ )
      varmenupoint( varnames[i] );
  };// drawCreateTab.variableMenu
  
  function attributeMenu (menu, line ) {
    // A helper to make one menu point
    var attrmenu = xulHelper.menu( menu, "Attributes" );
    function attrmenupoint(attrNode) {
      var menupoint = xulHelper.menuitem(attrmenu,attrNode.nodeName);
        menupoint.addEventListener("click", function(e) {
          logger.debug("Attr menu '" + attrNode.nodeName + "' clicked" );
          addAttrLine( attrNode, null, line.cf_attrbox, line.cf_highlight );
          line.redraw();
          generatePattern(true);
        }, false); // click
    }; // attrmenupoint
    var attrs = line.cf_domNode.attributes ;
    if ( !attrs )
      return; // happens with xml documents
    for ( var a = 0 ; a < attrs.length; a++ ) {
      var aname = attrs[a].nodeName ;
      if (aname.substr(0,3) == "cf_" ) { // skip our own highlights!
        logger.debug("Skipping attribute " + aname );
        break;
      }
      attrmenupoint(attrs[a]);
    }
  }; // drawCreateTab.attributeMenu

  function attrOpMenu (menu, line ) {
    // A helper to make one menu point
    function opmenupoint(label, op) {
      var menupoint = xulHelper.menuitem(menu,label + " " + op);
        menupoint.addEventListener("click", function(e) {
          line.cf_attrop = op;
          if ( !line.cf_attrvalue )
            line.cf_attrvalue = line.cf_text; // default to actual value
          line.redraw();
          generatePattern(true);
        }, false); // click
    }; // attrmenupoint
    opmenupoint("must be present","");
    opmenupoint("must equal","=");
    opmenupoint("must match","~");
  }; // drawCreateTab.attrOpMenu

  // Creates a menu point "match value"
  // that opens a line with input box to edit the value in line.cf_attrvalue
  function matchMenu (menu, line, hlColor) {
    var attrs = null;
    if (hlColor)
      attrs = { style: "background-color: " + hlColor + ";"};
    var item = xulHelper.menuitem(menu,"Match ...");
    item.addEventListener("click", function(e) {
      var editline = xmlHelper.appendNode(line.cf_pattbox, "hbox", null, attrs );
      if ( !line.cf_attrop && line.cf_lineType == "attr" )
        line.cf_attrop = "~";
      var capt = "Pattern to match";
      if ( line.cf_attrop && line.cf_attrop == "=" )
        capt = "Value must equal";
      xmlHelper.appendNode(editline, "label", capt);
      var patinp = xmlHelper.appendNode(editline, "textbox", null, 
                      {value:line.cf_attrvalue||"", flex:1}, null);
      patinp.addEventListener("input", function(e) {
          //line.cf_attrvalue = patinp.value;
          line.redraw();
          generatePattern(true);
        }, false);
      var okbut = xmlHelper.appendNode(editline, "button", null,
        {"label": "Ok"});
      okbut.addEventListener("click", function(e) {
          line.cf_attrvalue = patinp.value;
          line.redraw();
          generatePattern(true);
          xmlHelper.removeNode(editline);
        }, false);
      var cancelbut = xmlHelper.appendNode(editline, "button", null,
        {"label": "Cancel"});
      cancelbut.addEventListener("click", function(e) {
          line.redraw();
          generatePattern(true);
          xmlHelper.removeNode(editline);
        }, false);

      line.redraw();
      generatePattern(true);
    }, false); // click
  }; // drawCreateTab.matchMenu

  // Menu for the field modifiers like -html
  function modifierMenu(menu,line) {
    var modsmenu = xulHelper.menu( menu, "Modifiers" );
    function modsmenupoint(modifier, label) {
      var menupoint = xulHelper.menuitem(modsmenu,label);
        menupoint.addEventListener("click", function(e) {
          logger.debug("mods menu '" + modifier + "' clicked" );
          if ( modifier )
            line.cf_modifiers.push(modifier);
          else
            line.cf_modifiers = []; // clear
          line.redraw();
          generatePattern(true);
        }, false); // click
    }; // modsmenupoint
    var modlist = Xpattern.getAllModifiers();
    modsmenupoint( "", "(clear)");
    for ( var m in modlist )
      modsmenupoint( modlist[m], "-"+modlist[m] );
    
  }; // modifierMenu

  // A menu point for collapsing/expanding a node
  // The line must have a cf_collapsed, and collapse()/expand()
  function collapseMenu(menu, line) {
    var item = xulHelper.menuitem(menu,"Collapse ?");
    menu.addEventListener("popupshowing", function(e) {
      logger.debug("In menu.eventhandler: " + e );
        if ( line.cf_collapsed )
          item.label = "Expand";
        else
          item.label = "Collapse";
      return true;
    }, false);
    item.addEventListener("click", function(e) {
        if ( line.cf_collapsed )
          line.expand();
        else
          line.collapse();
        // the collapse/expand redraw as needed
        // no need to generate, nothing has really changed
    }, false);
  }; // drawCreateTab.collapseMenu
  
  // A menu point for ALT lines, to collapse all other ALTs
  function collapseAltMenu(menu, line) {
    var item = xulHelper.menuitem(menu,"Collapse Others");
    item.addEventListener("click", function(e) {
      var topline = nodebox.firstChild;
      if ( topline &&
        topline.cf_lineType == "group" &&
        topline.cf_groupType == "alternatives" ) {
        var altline = topline.cf_contentbox.firstChild;
        
        while (altline) {
          if ( altline != line )
            altline.collapse();
          else
            altline.expand();
          altline = altline.nextSibling;
        }
      }
    }, false);    
  }; // drawCreateTab.collapseAltMenu
    
  function removeMenu(menu, line, name) {
    var item = xulHelper.menuitem(menu,"Remove " + name);
    item.addEventListener("click", function(e) {
        xmlHelper.removeNode(line);
        if ( nodebox.firstChild &&
             nodebox.firstChild.cf_lineType == "group" &&
             nodebox.firstChild.cf_groupType == "alternatives" &&
             nodebox.firstChild.cf_contentbox &&
             ! nodebox.firstChild.cf_contentbox.firstChild ) {
          // We just removed the last alternative, clear the whole thing
          clearLines();
        }
        generatePattern();  // clears the pattern
    }, false);
  }; // drawCreateTab.removeFieldMenu

  // Helper to disable a menu point if we have alternatives
  // Used to disable the add field/group options in the tool menu
  // when we are working with alternative patterns, so that the user
  // won't add loose fields outside the alternative set
  function disableMenuIfAlt(box, menu, item) {
    if ( box == nodebox ) { // we are in the special tool menu
      menu.addEventListener("popupshowing", function(e) {
          if ( nodebox.firstChild &&
              nodebox.firstChild.cf_lineType == "group" &&
              nodebox.firstChild.cf_groupType == "alternatives" )
            item.disabled = true;
          else
            item.disabled = false;
        return true;
      }, false);
    }
  }; //disableMenuIfAlt
  
  function addFieldMenu(menu, line, box) {
    var fielditem = xulHelper.menuitem(menu,"Add field");
    disableMenuIfAlt(box, menu, fielditem);
    fielditem.addEventListener("click", function(e) {
        selectAndAddField(box, line );
    }, false);
  }; // drawCreateTab.addGroupMenu
  
  function addGroupMenu(menu, line, box) {
    var groupitem = xulHelper.menuitem(menu,"Add ()-group");
    groupitem.addEventListener("click", function(e) {
        addGroup( null, box, line, "(" );
    }, false);
    disableMenuIfAlt(box, menu, groupitem);
    var orbagitem = xulHelper.menuitem(menu,"Add OR-bag");
    orbagitem.addEventListener("click", function(e) {
        addGroup( null, box, line, "|" );
    }, false);
    disableMenuIfAlt(box, menu, orbagitem);
  }; // drawCreateTab.addGroupMenu

  // Menu point to add an alternative pattern
  // Three possibilities:
  //  - We have no lines at all. Create first alternative
  //  - We already have alternatives, append a new one
  //  - We have a regular pattern, convert to first alt, and add one more
  function addAltMenu(menu) {
    var altitem = xulHelper.menuitem(menu,"Add alternative pattern");
    altitem.addEventListener("click", function(e) {
      var topline = nodebox.firstChild;
      if ( ! topline ) { // we have nothing, create alt-set
        var altset = addGroup(null, nodebox, null, "alternatives" );
        var alt = addGroup(null, altset.cf_contentbox, null, "ALT", nextaltnum++);
        logger.debug("addAlt: Added the altset, and the first alt");
      } else if ( topline.cf_lineType == "group" && 
        topline.cf_groupType == "alternatives" ) { // append an alt
        var alt =  addGroup(null, topline.cf_contentbox, null, "ALT",nextaltnum++);
        logger.debug("addAlt: Appended an alt to existing set");
      } else {
        logger.debug("addAlt: Have a pattern, need to convert to an alt");
        var lines = [];
        while ( nodebox.firstChild ) {
          logger.debug("addAlt: Removed " + nodebox.firstChild.cf_lineType + " " +
            ( nodebox.firstChild.cf_tagname || nodebox.firstChild.cf_groupType ) );
          lines.push(nodebox.firstChild);
          nodebox.removeChild(nodebox.firstChild);
        }
        var altset = addGroup(null, nodebox, null, "alternatives" );
        nextaltnum = 1;
        var alt = addGroup(null, altset.cf_contentbox, null, "ALT", nextaltnum++);
        var newline;
        logger.debug("addAld: Have " + lines.length + " lines");
        while ( lines.length ) {
          newline = lines.shift();
          logger.debug("addAlt: Inserted " + newline.cf_lineType + " " +
            ( newline.cf_tagname || newline.cf_groupType ) );
          alt.cf_contentbox.appendChild(newline);
        }
        addGroup(null, altset.cf_contentbox, null, "ALT", nextaltnum++);
        logger.debug("addAlt: Moved lines into newly created altset");
      }
      generatePattern(true);
    }, false);
    
  }; // addAltMenu
  
  // Ask the user to point to a part of the hit, and add a line for it.
  function selectAndAddField(box, addAfter ) {
    pleasemsg.hidden = false;
    alertmsg.hidden = true;
    waitmsg.hidden = true;
    app.newHl(context.getPageDoc(), function() {
        pleasemsg.hidden = true;
        var domNode = this;
        if ( domNode.cf_datanode ) {
          domNode = domNode.cf_datanode;
        }
        app.highlighter.destroy();
        var dnn = domNode.localName || domNode.nodeName;
        logger.debug("Click on node " + dnn );
        var col = "";
        if ( context.conf['highlighthits'] ) {
          col = gethighlightcolor();
          xulHelper.highlightNode(domNode, col, true );
        }
        addField ( domNode, null, box, col, addAfter );
        generatePattern(true);
        return false;
      } ); // highlighter
  }; // selectAndAddField

  // Helpers for guessing a $variable

  // Get to the root of the pattern, or at least to the root of the
  // current alternative
  function altroot(fieldbox) {
    while (fieldbox) {
      logger.debug("altroot: " + fieldbox.nodeName + ":" +
        ( fieldbox.cf_tagname || "" ) );
      if ( fieldbox == nodebox )
        return fieldbox; // root of the designer tree
      if ( fieldbox.cf_groupType  && fieldbox.cf_groupType == "ALT" )
        return fieldbox;
      fieldbox = fieldbox.parentNode;
    }
    return null;  // should not happen
  };

  // Check if the given variable is already used (within this alternative)
  // recurses the whole xul domtree from altroot down, this catches groups,
  // attributes, and everything else that might have a cf_varname. 
  function variableAlreadyUsed(altroot, varname) {
    if (!varname)
      return false;
    while ( altroot ) {
      if ( altroot.cf_varname && altroot.cf_varname == varname )
        return true;
      if ( variableAlreadyUsed(altroot.firstChild, varname) )
        return true;
      altroot = altroot.nextSibling;
    }
    return false;
  };

  // Guess a variable for a newly added line
  function guessVar(fieldbox, txt) {
    var root = altroot(fieldbox); // root of the current alternative
    if ( !root )
      return "";  // should not happen
    context.varnamelist();
    logger.debug("guessVar: varnames: " + JSON.stringify(varnames) );
    var vars = {};
    for (var i=0; i<varnames.length; i++) {
      vars[ varnames[i] ] = 0;
    }
    vars[""] = -1; // "none"
    
    var hints = context.task.getTemplate().properties.variablehints;
    if ( hints ) {  // may not be, if not in parse task
      for ( var h = 0; h < hints.length; h++ ) {
        var hint = hints[h];
        logger.debug("guessVar: Looking at a hint for " + hint.var );
        for ( var v in vars ) {
          if ( v.match( hint.var ) ) {
            logger.debug("guessVar: checking '" + v + "' against " + hint.var );
            if ( hint.regexp ) {
              var re = new RegExp(hint.regexp, 'i');  // ignore case in matching text
              if ( txt.match(re) ) {
                vars[v] += parseInt(hint.weight);
                logger.debug("guessVar: Regexp " + hint.regexp + " matches '" +
                 txt + "'.  Adjusted hint for " + v + " by " +
                  hint.weight + ". Now at " + vars[v] );
              } else {
                logger.debug("guessVar: Regexp " + hint.regexp + " did not match " +
                 "'" + txt + "'" );
              }
            } else if ( hint.func ) {
              if ( hint.func == "alreadyused" ) {
                if ( variableAlreadyUsed(root.firstChild, v) ) {
                  vars[v] += parseInt(hint.weight);
                  logger.debug("guessVar: alreadyused. Adjusted " + v + " by " +
                    hint.weight + " now at " + vars[v] );
                }
              }
            } else if ( hint.attr ) {
              if ( hint.attr == fieldbox.cf_tagname ) {
                vars[v] += parseInt(hint.weight);
                logger.debug("guessVar: attr " + hint.attr + " matches '" +
                 txt + "'.  Adjusted hint for " + v + " by " +
                  hint.weight + ". Now at " + vars[v] );                
              }

            } else
              logger.error("Bad variable hint: Need a regexp or a func. " +
                JSON.stringify(hint) );
          }
        }

      }
    }
    var bestvar = varnames[0];  // the first in the list, usually title
    for (var v in vars ) {
      if ( vars[v] > vars[ bestvar ] ) {
        bestvar = v;
      }
      logger.debug("guessVar: " + v + ": " + vars[v] + " best=" + bestvar);
    }
    if ( vars[bestvar] >= 0 ) {
      logger.debug("guessVar: Decided to guess '" + bestvar + "' " +
        " with a score of " + vars[bestvar] );
      var ret = { guess: bestvar, score: vars[bestvar] };
      return ret ;
    }

    return "";  // didn't find anything
  }// guessVar

  // Add a line that displays a one field, and possibly its attributes too
  function addField ( domNode, pattNode, box, hlColor, addAfter ) {
    //logger.debug("addField: dom=" + domNode + " patt=" + pattNode +
    //  " box=" + box + " addAfter=" + addAfter );
    var fieldBox = xmlHelper.appendNode(box, "vbox", null, null, null, false);
    if ( addAfter && addAfter.nextSibling ) 
      box.insertBefore(fieldBox,addAfter.nextSibling);  // Insert in the right place
    else
      box.appendChild(fieldBox); // just append it.
    var attrs = {};
    if ( !hlColor )
      hlColor = xulHelper.getHighlight(domNode);
    if (hlColor)
      attrs = { style: "background-color: " + hlColor + ";"};
    xulHelper.rehighlightNode(domNode); // TODO - check XML
    
    var fieldLine = xmlHelper.appendNode(fieldBox, "hbox", null, attrs, null);
    fieldBox.cf_pattbox = xmlHelper.appendNode(fieldBox, "vbox", null, {flex:1}, null);
    var indentBox = xmlHelper.appendNode(fieldBox, "hbox", null, attrs, null);
    var indent = xmlHelper.appendNode(indentBox, "hbox", null, {width:20}, null);
    fieldBox.cf_attrbox = xmlHelper.appendNode(indentBox, "vbox", null, {flex:1}, null);
    fieldBox.cf_domNode = domNode; // keep a reference to it, for generating
    fieldBox.cf_lineType = "field";
    fieldBox.cf_tagname = domNode.localName || domNode.nodeName;
    fieldBox.cf_text = domNode.textContent || "<" + fieldBox.cf_tagname + ">";
       // IMG tags have no text, so display the tag instead
    fieldBox.cf_cardinality = "";
    fieldBox.cf_varname = "";
    fieldBox.cf_modifiers = [];
    fieldBox.cf_highlight = hlColor;
    fieldBox.cf_collapsed = false;
    var varguessscore = 0;
    if ( pattNode ) {
      fieldBox.cf_cardinality = pattNode.getCardinality();
      fieldBox.cf_varname = pattNode.getPlainVariable();
      fieldBox.cf_modifiers = pattNode.getModifiers();
    } else {
      var g = guessVar(fieldBox, domNode.textContent );
      if ( g ) {
        fieldBox.cf_varname = g.guess;
        varguessscore = g.score;
      }
    }

    // helper do redraw the field elements. Called every time things change.
    fieldBox.redraw = function () {
      var disp = fieldBox.cf_tagname;
      if ( fieldBox.cf_cardinality )
        disp += " " + fieldBox.cf_cardinality;
      xulHelper.changemenulabel( cardmenu, disp);
      if ( fieldBox.cf_varname )
        disp = "$" + fieldBox.cf_varname;
      else
        disp = "$";
      xulHelper.changemenulabel( varmenu, disp);
      disp = "";
      if ( fieldBox.cf_modifiers )
        for ( var m in fieldBox.cf_modifiers )
          disp += " -" + fieldBox.cf_modifiers[m];
      moddisp.value = disp;
      disp = "";
      if ( fieldBox.cf_collapsed ) {
        disp = "[...]";
      } else {
        if ( fieldBox.cf_attrvalue )
          disp = "[ /" + fieldBox.cf_attrvalue+ "/ ]";
      }
      attrdisp.value = disp;
      checkHitArea();
    }; // redraw

    var txtdisp = xmlHelper.appendNode(fieldLine, "label", null, 
                            { flex:1, crop:"end", value:fieldBox.cf_text } );
        // If the value is passed as a text, it gets multilined, and not cropped
        // If passing as a value attribute, cropping takes effect. Wonderful XUL!
    var hitwarning = xmlHelper.appendNode(fieldLine, "label", null,
                    { style:"color:red;font-weight:bold;", hidden:true,
                      value: "(hit area!)" } );
    var cardmenu = xulHelper.popupmenu(fieldLine, "???",
                            {style:"font-weight: bold;"} );
      // ## Wanted it to be underlined as well, but setting
      // text-decoration: underline caused all menu points to be underlined too.
      // Anyway, that would conflict with hotkeys, if we ever set such
    cardinalityMenu(cardmenu, fieldBox, true);

    var varmenu = xulHelper.popupmenu(fieldLine, "???",
                            {style:"font-weight: bold;"} );
    variableMenu(varmenu, fieldBox, true);
    
    var moddisp =  xmlHelper.appendNode(fieldLine, "label", "???",
                           { style:"font-weight: bold;" } );
    var attrdisp =  xmlHelper.appendNode(fieldLine, "label", "???",
                           { style:"font-weight: bold;" } );
    
    fieldBox.redraw();
    
    function checkHitArea() {
      if ( !fieldBox.cf_domNode ) {
        return;
      }
      try { // Check if node within hitarea
        var pageDoc = context.getPageDoc();
        var hitarea = xmlHelper.getElementByNodeSpec(pageDoc, context.conf['hitarea']);
      } catch (e) {
        logger.debug("CheckHitArea: Could not get hit area");
        return; // never mind
      }
      if ( xmlHelper.isAncestor(fieldBox.cf_domNode, hitarea) ) {
        hitwarning.hidden = true;
        logger.debug("checkHitArea: ok");
      } else {
        hitwarning.hidden = false;
        var tooltip = "The node is not inside the hit area! \n"+
                 xmlHelper.nodeSpecToString(context.conf['hitarea']);
        hitwarning.setAttribute("tooltiptext",tooltip);
        logger.debug("checkHitArea: WARN");
      }
    }; // checkHitArea
    
    checkHitArea();
    hitwarning.addEventListener("click",checkHitArea, false);
    

    var toolmenu = xulHelper.menu(fieldLine, "Actions" );
    cardinalityMenu(toolmenu, fieldBox);
    variableMenu(toolmenu, fieldBox);
    attributeMenu(toolmenu, fieldBox);
    matchMenu(toolmenu, fieldBox, hlColor );
    modifierMenu(toolmenu, fieldBox );
    xulHelper.menuseparator(toolmenu);
    collapseMenu(toolmenu, fieldBox);
    xulHelper.menuseparator(toolmenu);
    removeMenu(toolmenu, fieldBox, "field");
    addFieldMenu(toolmenu,fieldBox, box);
    addGroupMenu(toolmenu,fieldBox, box);
    
    // little filler at the end of the line, to make attr menus look indented
    xmlHelper.appendNode(fieldLine, "hbox", null, {width:20}, null);

    if ( !pattNode && varguessscore < 1000 ) { // user-selected field
      varmenu.openPopup(varmenu.cf_label,"after_end"); // and not certain guess
    }

    // collapse/expand the line.
    fieldBox.collapse = function () {
      if ( fieldBox.cf_attrvalue || fieldBox.cf_attrbox.firstChild ) {
        // only collapse if we have something to collapse
        fieldBox.cf_attrbox.hidden = true;
        fieldBox.cf_collapsed = true;
        fieldBox.redraw();
      }
    };
    
    fieldBox.expand = function () {
      fieldBox.cf_attrbox.hidden = false;
      fieldBox.cf_collapsed = false;
      fieldBox.redraw();
    };
    
    fieldBox.addEventListener("dblclick", function(e) {
        if ( fieldBox.cf_collapsed )
          fieldBox.expand();
        else
          fieldBox.collapse();
    }, false);
    return fieldBox;
  }; // drawCreateTab.addField

  function addAttrLine ( domNode, pattNode, box, hlColor) {
    var attrBox = xmlHelper.appendNode(box, "vbox", null, null, null);
    var attrs = {};
    if (hlColor)
      attrs = { style: "background-color: " + hlColor + ";"};
    var attrLine = xmlHelper.appendNode(attrBox, "hbox", null, attrs, null);
    attrBox.cf_pattbox = xmlHelper.appendNode(attrBox, "vbox", null, {flex:1}, null);
    attrBox.cf_domNode = domNode;
    attrBox.cf_lineType = "attr";
    attrBox.cf_tagname = domNode.localName || domNode.nodeName;
    attrBox.cf_text = domNode.textContent;
    attrBox.cf_highlight = hlColor;
    attrBox.cf_varname = "";
    var varguessscore = 0;
    if ( pattNode ) {
      attrBox.cf_attrvalue = pattNode.attrValue || "";
      attrBox.cf_attrop = pattNode.relationValue;
      attrBox.cf_varname = pattNode.getPlainVariable();
    } else {
      var g = guessVar(attrBox, domNode.textContent );
      if ( g ) {
        attrBox.cf_varname = g.guess;
        varguessscore = g.score;
      }
    }

    var txtdisp = xmlHelper.appendNode(attrLine, "label", null,
                         { flex:1, crop:"end", value:attrBox.cf_text } );

    attrBox.redraw = function () {
      var disp = "@" + attrBox.cf_tagname; 
      if ( attrBox.cf_attrvalue ) {
        if ( attrBox.cf_attrop == "=" )
          disp += " =\"" + attrBox.cf_attrvalue + "\"" ;
        else if ( attrBox.cf_attrop == "~" )
          disp += " ~/" + attrBox.cf_attrvalue + "/" ;
      }
      attrBox.cf_attrline = disp; // Save for generating
         // TODO - Don't set real values in display things!
      xulHelper.changemenulabel( attropmenu, disp);
      if ( attrBox.cf_varname )
        disp = "$" + attrBox.cf_varname;
      else
        disp = "$";
      xulHelper.changemenulabel( varmenu, disp);
      
    }; // redraw

    
    xmlHelper.appendNode(attrLine, "label", null,
                           { style:"font-weight: bold;", value:"[" } )
    
    var attropmenu = xulHelper.popupmenu(attrLine, "???",
                            {style:"font-weight: bold;"} );
    attrOpMenu(attropmenu, attrBox, true);
    var varmenu = xulHelper.popupmenu(attrLine, "???",
                            {style:"font-weight: bold;"} );
    variableMenu(varmenu, attrBox, true);
    xmlHelper.appendNode(attrLine, "label", null,
                           { style:"font-weight: bold;", value:"]" } )
    attrBox.redraw();
    
    var toolmenu = xulHelper.menu(attrLine, "Actions" );
    attrOpMenu(toolmenu, attrBox);
    matchMenu(toolmenu, attrBox, hlColor );
    xulHelper.menuseparator(toolmenu);
    variableMenu(toolmenu, attrBox);
    xulHelper.menuseparator(toolmenu);
    removeMenu(toolmenu, attrBox, "attribute");

    /*
     // TODO - This does not work. My theory is that we still have the
     // fieldBox menu open (since add attr comes from there), and the
     // system can not have two menus open at the same time. Yet I seem
     // not to be able to close it... Later...
    if ( !pattNode ) { // user-selected field
      //logger.debug("Closing the field-level menu");
      //box.cf_toolmenu.closePopup(); 
      logger.debug("Popping up the var menu for the attr line" );
      varmenu.openPopup(varmenu.cf_label,"after_end");
    }
    */
    
  }; // drawcreateTab.addAttrLine


  // Add a line for a group
  // grouptype can be "(" for regular group, "|" for or-bag,
  // "alternatives" for the main alternative container, or
  // "alt" for individual alternatives (which will use the altnum for display)
  function addGroup ( pattNode, box, addAfter, grouptype, altnum ) {
    var groupBox = xmlHelper.appendNode(box, "vbox", null, null, null, false);
    if ( addAfter && addAfter.nextSibling ) 
      box.insertBefore(groupBox,addAfter.nextSibling);  // Insert in the right place
    else
      box.appendChild(groupBox); // just append it.
    var attrs = {};
    var groupLine = xmlHelper.appendNode(groupBox, "hbox", null, attrs, null);
    groupBox.cf_pattbox = xmlHelper.appendNode(groupBox, "vbox", null, {flex:1}, null);
    var indentBox = xmlHelper.appendNode(groupBox, "hbox", null, attrs, null);
    var indent = xmlHelper.appendNode(indentBox, "hbox", null, {width:20}, null);
    groupBox.cf_contentbox = xmlHelper.appendNode(indentBox, "vbox", null, {flex:1}, null);
    groupBox.cf_lineType = "group";
    groupBox.cf_varname = "";
    groupBox.cf_collapsed = false;
    groupBox.cf_groupType = grouptype;
    if ( grouptype == "alternatives" ) {
      groupLine.hidden = true; // do not display the line itself
      indent.hidden = true; // and do not indent the following alternatives
    }
    if ( grouptype == "ALT" && altnum )
      groupBox.cf_altnum = altnum;
    if ( pattNode ) {
      groupBox.cf_cardinality = pattNode.getCardinality();
      groupBox.cf_varname = pattNode.getPlainVariable();
    } else {
      if ( grouptype == "|" ) // or-bags default to repeating
        groupBox.cf_cardinality = "+";
    }
    groupBox.dispString = function () {
      var disp;
      if ( groupBox.cf_groupType == "(" ) {
        disp = "Group";
      } else if ( groupBox.cf_groupType == "|" ) {
        disp = "( | ) Or bag";
      } else if ( groupBox.cf_groupType == "ALT" ) {
        disp = "Alt";
        if ( groupBox.cf_altnum)
          disp += "-" + groupBox.cf_altnum;
      } else { // Should not happen!
        disp = "Unknown group type '" + groupBox.cf_groupType + "'";
      }
      if ( groupBox.cf_collapsed ) {
        disp += " (";
        var separator = "";
        var c = groupBox.cf_contentbox.firstChild;
        while (c) {
          //logger.debug("Group disp: " + c.cf_tagname );
          disp += separator;
          separator = " :";
          if ( c.cf_lineType == "field" ) {
            disp += " " + c.cf_tagname;
            if ( c.cf_varname )
              disp += " $" + c.cf_varname;
          } else if ( c.cf_lineType == "group" ) {
            disp += " (...)";
          } else {
            disp += " ??" + c.cf_lineType;
          }
          c = c.nextSibling;
        }
        disp += " )";
      }
      if ( groupBox.cf_cardinality )
        disp += " " + groupBox.cf_cardinality;
      if ( groupBox.cf_varname )
        disp += " $" + groupBox.cf_varname;
      return disp;
    }; // dispString
    var disp = groupBox.dispString();
    var nodedisp = xmlHelper.appendNode(groupLine, "label", null,
           { flex:1, crop:"end", value: disp, style:"font-weight: bold;" } );
    groupBox.redraw = function () {
      nodedisp.value = this.dispString();
    }; // redraw
    var menutitle = "Group Actions";
    var rmtitle = "group"; // what the remove point will say
    if ( groupBox.cf_groupType == "|" ) {
      menutitle = "Or-group Actions";
      rmtitle = "or-group";
    } else if ( groupBox.cf_groupType == "ALT" ) {
      menutitle = "Alt Actions";
      rmtitle = "alternative";
    }
    var toolmenu = xulHelper.menu(groupLine, menutitle );
    if ( groupBox.cf_groupType != "ALT" ) {
      cardinalityMenu(toolmenu, groupBox);
      variableMenu(toolmenu, groupBox);
      xulHelper.menuseparator(toolmenu);
    }
    collapseMenu(toolmenu, groupBox);
    collapseAltMenu(toolmenu, groupBox);
    xulHelper.menuseparator(toolmenu);
    removeMenu(toolmenu, groupBox, rmtitle);
    addFieldMenu(toolmenu,null,groupBox.cf_contentbox);
    addGroupMenu(toolmenu,groupBox, groupBox.cf_contentbox);
    
    xmlHelper.appendNode(groupLine, "hbox", null, {width:40}, null);
      // A bit bigger filler, to move the menu further left, to make regular
      // field lines look indented

    // Helper to recursively mark all DOM nodes as collapsed or not,
    // so we know (not) to highlight collapsed alternatives
    // Also marks all lines
    function collapseMarks(xulNode, is_collapsed) {
      while(xulNode) {
        if (xulNode.cf_domNode ) {
          if ( xulNode.cf_domNode.setAttribute ) {
            if ( is_collapsed )  
              xulNode.cf_domNode.setAttribute( "cf_collapsed", "collapsed");
            else
              xulNode.cf_domNode.setAttribute( "cf_collapsed", "expanded");
          }
          xulNode.cf_collapsed = is_collapsed;
        }
        if ( xulNode.firstChild )
          collapseMarks(xulNode.firstChild, is_collapsed );
        xulNode = xulNode.nextSibling;
      }
    }// collapseMarks
    
    // collapse/expand the line.
    groupBox.collapse = function () {
      groupBox.cf_contentbox.hidden = true; 
      groupBox.cf_collapsed = true;
      groupBox.redraw();
      if ( grouptype == "ALT" ) {
        collapseMarks( groupBox.firstChild, true );
        generatePattern(true);  // regenerate, to get highlights updated
      }
    }
    groupBox.expand = function () {
      groupBox.cf_contentbox.hidden = false;
      groupBox.cf_collapsed = false;
      groupBox.redraw();
      if ( grouptype == "ALT" ) {
        collapseMarks( groupBox.firstChild, false );
        generatePattern(true);
      }
    }
    
    groupLine.addEventListener("dblclick", function(e) {
        // only groupLine, not to react in the whole box, and nested groups
        logger.debug("Dbl-click on group box " + groupBox.cf_groupType +
           " $" + groupBox.cf_varname );
        if ( groupBox.cf_collapsed )
          groupBox.expand();
        else
          groupBox.collapse();
    }, false);
    return groupBox;
  }; // drawCreateTab.addGroup

  
  ///////
  // Generating the pattern

  // A helper to take a line and pass it to a XPatternMaker
  // Recursive function, to handle nested groups etc 
  function makerLines( maker, line, groupid ) {
    while (line != null) {
      if ( line.cf_lineType == "field" ) {
        var attrs = "";
        if ( line.cf_attrvalue )
          attrs = "/" + line.cf_attrvalue + "/";
        var attrLine = line.cf_attrbox.firstChild;
        while ( attrLine ) {
          if (attrs)
            attrs += ", ";
          var disp = "@" + attrLine.cf_tagname;
          if ( attrLine.cf_attrvalue ) {
            if ( attrLine.cf_attrop == "=" )
              disp += " =\"" + attrLine.cf_attrvalue + "\"" ;
            else if ( attrLine.cf_attrop == "~" ) {
              disp += " ~/" +attrLine.cf_attrvalue + "/" ;
              // TODO - The XPattern syntax does not allow escaping slashes,
              // so we can't protect them here. This goes visibly wrong with
              // the most common attribute, url. Luckily nobody is likely to
              // do matching on that!
            }
          }
          if ( attrLine.cf_varname )
            disp += " $" + attrLine.cf_varname;
          logger.debug("maker: attr line '" + disp + "'" );
          attrs += disp;
          attrLine = attrLine.nextSibling;
        }
        logger.debug("makerLines: " + line.cf_tagname + " '" + line.cf_text + "' " +
          " (g=" + groupid + ") m=" + line.cf_modifiers + " a='" + attrs + "'" );
        maker.addNode( line.cf_domNode, line.cf_varname, line.cf_cardinality,
                       attrs, groupid, line.cf_modifiers );  
      } else if ( line.cf_lineType == "group" ) {
        var attrstr = ""; // groups don't really have attributes
        //logger.debug("makerLines: '" + line.cf_groupType +
        //   "' ingroup=" + groupid );
        var grouptype = line.cf_groupType;
        if ( grouptype == "ALT" ) {
          grouptype = "("; // make a regular group of the alternative
        }
        var newgroupno = maker.addGroup( line.cf_domNode,
              line.cf_varname, line.cf_cardinality,
              attrstr, grouptype, groupid );  // the old one is the parent
        makerLines( maker, line.cf_contentbox.firstChild, newgroupno );
        //logger.debug("makerLines: '" + grouptype + "' group " + newgroupno + " Done" );
        if ( line.cf_groupType == "ALT" ) {
          //logger.debug("ALT group, not following into next");
          return;
        }
      }
      line = line.nextSibling;
    } // line loop
  } // drawCreateTab. makerLines

  // Generate the pattern from the selections
  // autogen indicates that this is an automatic generation (and not a result
  // of the user clicking the generate button). That means that it can be
  // disabled by a config option, and error handling may be different.
  // (still within drawCreateTab)
  function generatePattern(autogen) {
      alertmsg.hidden = true;
      if ( autogen && ! context.conf["autogenerate"] ) {
        logger.debug("no autogeneration");
        waitmsg.hidden = true;
        generateButton.hidden = false;
        return;
      }
      logger.debug("About to generate pattern");
      if ( context.conf["autogenerate"] )
        generateButton.hidden = true;
      var pageDoc = context.getPageDoc();
      var hitarea = context.conf['hitarea'];
      var startnode = null;
      try {
        startnode = xmlHelper.getElementByNodeSpec(pageDoc,hitarea);
      } catch (e) {
        logger.debug("generate: Did not get the hitarea: " + e + "\n");
        if (!autogen) {
          win.alert("The xpath for the hit area is bad. \n'"+hitarea+"' \n" );
        }
        return;
      }
      if (startnode == null) {
        if (!autogen) {
          win.alert("The hit area is bad. Can not create pattern!");
        }
        return;
      }
      logger.debug("generate: Old pattern: " + context.conf['xpattern'] );
      if ( !nodebox  || !nodebox.firstChild ) {
        logger.debug("generate: no lines, clearing it");
        context.setPattern(""); // sets the history and display
        return;
      }
      
      var topline = nodebox.firstChild;
      //dumplines( "generate:" );
      if ( topline &&
          topline.cf_lineType == "group" &&
          topline.cf_groupType == "alternatives" ) {
        //logger.debug("generate: Trying to generate from alternatives");
        var altline = topline.cf_contentbox.firstChild;
        var newpatt = "";
        while ( altline ) {
          var altnum = altline.cf_altnum;
          //logger.debug("generate: alt " + altnum + ": " + altline.cf_lineType );
          if ( altline.cf_lineType == "group" &&
               altline.cf_contentbox &&  // have a real, non-empty group
               altline.cf_contentbox.firstChild ) {
            var maker = new XpatternMaker( startnode );
            makerLines(maker, altline, 0 /*in no group*/);
            var p = maker.getPattern();
            if ( p == null ) {
              logger.debug("generate: Failed to get alt-" + altnum );
              if (!autogen) {
                win.alert("Could not create a pattern for alt-" + altnum);
              return;
              }
            }
            else {
              logger.debug("generate: Got alt-" + altnum + ": " + p.dumpString(-1) );
              if ( newpatt )
                newpatt += " | \n";
              newpatt +=  p.dumpString(-1) ;
            }
          }
          altline = altline.nextSibling;
        }
        xpatt.textContent = newpatt; // set the pattern explicitly
          // so that setPattern will not thing it sees a new one, and force
          // repopulating
        context.setPattern(newpatt); // sets the history and display
      } else  {
        //logger.debug("Generating a 'simple' pattern" );
        var maker = new XpatternMaker( startnode );
        makerLines(maker, nodebox.firstChild, 0 /*in no group*/);
        var p = maker.getPattern();
        if ( p != null ) {
          var ps = p.dumpString(-1);
          logger.debug("generated pattern '" + ps + "'");
          xpatt.textContent = ps; // set the pattern explicitly
          // so that setPattern will not thing it sees a new one, and force
          // repopulating

          context.setPattern(ps); // sets the history and display
        } else {
          logger.debug("Failed to generate pattern");
          if (!autogen) {
            win.alert("Could not create a pattern!");
          }
        }
        alertmsg.hidden = true;
        waitmsg.hidden = true;
      }
      // Try to match the newly generated pattern, to get highlights on the
      // screen.
      var task = new Task(context.task.connector, "dummy");
      try { // run the step with a fake task parameter to get the outputs
        context.run(task);
      } catch(e) {
        logger.debug("Running generated pattern failed with " + e );
      }
  } // generatePattern


  ///// Populating the designer from an existing pattern
  
  // Helper to set up one field (or group) in the designer
  function designerNode(xp, besthit, depth, len, box) { 
    var prefix = "desN: " + depth + "." + len + " [" + xp.dumpString(-2)+ "] " ;
    //logger.debug(prefix + "starting xp.nodeType=" + xp.nodeType );
    if ( xp.nodeType == "(" ) { // group node
      if ( xp.firstChild ) {
        if ( xp.firstChild.nodeType == "|" ) { // or-bag
          var alt = xp.firstChild.firstChild;
          //logger.debug(prefix+"Making OR-bag " + alt.dumpString(-2) ); 
          var groupBox = addGroup ( xp, box, null, "|" );
          var altlen = 0;
          while(alt) {
            designerNode(alt, besthit, depth+1, altlen, groupBox.cf_contentbox );
            altlen++;
            alt = alt.nextAlternative;
          }
        } else { // regular group
          //logger.debug(prefix+"Making a regular group");
          var groupBox = addGroup ( xp, box, null, "(" );
          designerNode(xp.firstChild, besthit, depth+1, 0, groupBox.cf_contentbox );
        }
      }
    } else { // not a group, must be a real thing.
      // see if we have the xp node in the hits
      var hitidx = -1;
      for ( var i=0; i<besthit.hits.length; i++) {
        var h = besthit.hits[i];
        //logger.debug(prefix + "hit " + i + " " + h.name + ": " + h.value + "  " + h.pattern.dumpString(-2) );
        if ( h.pattern == xp ) {
          if ( hitidx == -1 || h.value.length > 0) {
            hitidx = i;
            //logger.debug(prefix + "found at hit " + i + ": '" + h.name + "': '" + h.value +"'");
            if ( hitidx == -1 || h.value.length > 0)
              break; // be satisfied with the first that has real text
          }
        }
      }
      //logger.debug(prefix + "Decided on hit " + hitidx  );
      if ( hitidx != -1) {
        var h = besthit.hits[hitidx];
        //logger.debug(prefix + "dom: " + h.dom.nodeName + " p=" + h.dom.parentNode.nodeName );
        var domnode = h.dom;
        if ( domnode.nodeName == "#text" )
          domnode = domnode.parentNode;
        var fieldBox = addField(domnode, h.pattern, box, "" ); // regular node
        var attrNode = h.pattern.attributes;
        while (attrNode) {
          if ( attrNode.nodeType == "/" ) {
            fieldBox.cf_attrvalue = attrNode.attrValue;
            fieldBox.cf_attrline = "/" + attrNode.attrValue + "/";
            //logger.debug(prefix + "  Attribute (regexp)" + attrNode.attrValue );
            fieldBox.redraw();
          } else {
            //logger.debug(prefix + "  Attribute " + attrNode.nodeType );
            var attrname = attrNode.nodeType.substr(1); // skip the '@'
            //logger.debug(prefix + "  Attr name '" + attrname +"'" );
            var attrdomnode = null;  // Find the attribute node
            var domattrs = fieldBox.cf_domNode.attributes ;
            for ( var a = 0 ; a < domattrs.length; a++ ) {
              if ( domattrs[a].nodeName == attrname ) {
                attrdomnode = domattrs[a];
                break;
              }
            }
            if ( attrdomnode) {
              //logger.debug(prefix + "  Attr dom @ " + attrdomnode.nodeName + "=" +
              //        attrdomnode.textContent );
              addAttrLine(attrdomnode, attrNode, fieldBox.cf_attrbox, fieldBox.cf_highlight );
            } else {
              logger.debug(prefix + "  Attr: No dom node found for " + attrname );
            }
          }
          attrNode = attrNode.nextSibling;
        }
      }
      // Recurse through the pattern
      if ( xp.firstChild )
        designerNode(xp.firstChild, besthit, depth+1, 0, box );

    } // real node
    if ( xp.nextSibling )
      designerNode(xp.nextSibling, besthit, depth, len+1, box );

  }; // drawCreateTab.designerNode

  // Helper to see if a pattern node 'needle' is contained in
  // pattern 'haystack'. Only looks at the nextSibling and firstChild chains,
  // not alternatives (except on top level). Ignores attributes, negations, etc.
  function XPatternContains ( needle, haystack, checkalternatives ) {
    while ( haystack ) {
      if ( needle == haystack )
        return true;
      if ( XPatternContains ( needle, haystack.firstChild, true ) )
        return true;
      if ( checkalternatives &&
           XPatternContains ( needle, haystack.nextAlternative, true ) )
        return true;
      haystack = haystack.nextSibling;
    }
    return false;
  }; // XPatternContains

  
  // Helper to find the best hit that is within the given (sub)pattern
    // TODO - we could measure actual complexity better, this
    // fails when one field repeats many times, instead of many different
    // fields...
  function findBestHit(xpatternHits, pattern) {
    var besthit = null ;
    
    logger.debug("findBestHit starting to look at " + xpatternHits.length +
       " hits and pattern " + pattern.dumpString(-1) );
    for ( var i=0; i< xpatternHits.length; i++) {
      if ( ! besthit ||
             xpatternHits[i].hits.length > besthit.hits.length ) {
        if ( xpatternHits[i].hits[0] ) {  // a real hit that returns something
          var hitfound = XPatternContains(xpatternHits[i].hits[0].pattern, pattern, false);
          //logger.debug("Looking at hit " + i + " found=" + hitfound );
          if ( hitfound ) {
            besthit = xpatternHits[i];
            //logger.debug("Hit " + i + " is the best so far with " +
            //  besthit.hits.length + " lines");
          }
        }
      }
    }
    return besthit;
  };// findBestHit
  
  // Set up the designer from an existing XPattern
  // autogen should be set when populating automatically. Defaults to false.
  function populateDesigner( autogen ) {
    if ( ! autogen ) // sanitize into a boolean
      autogen = false;
    alertmsg.hidden = true; 
    if ( !context.conf['xpattern'] ) {
      logger.debug("Refusing to populate from no pattern");
      waitmsg.hidden = true;
      return; 
    }
    var task = new Task(context.task.connector, "dummy");
    try { // run the step with a fake task parameter to get the outputs
      context.run(task);
    } catch (e) {
      var where = "";
      if ( e.fileName )
        where = " " + e.fileName + ":" + e.lineNumber + " ";
      if (!autogen) 
        win.alert("OOPS! " + e + "\n" + where);
      logger.error("XPattern populate: running the task failed with " +
          e + where );
      waitmsg.hidden = true;
      alertmsg.hidden = false;
      return;
    }
    // Some debug log for the decision
    // We want the main pattern to be something like (...)|(...)|(...)
    if ( task.parsedXpattern ) {
      logger.debug("Alt check: pp: " + task.parsedXpattern.dumpString(-1) );
      logger.debug("Alt check: nt: " + task.parsedXpattern.nodeType );
      if ( task.parsedXpattern.firstChild ) {
        logger.debug("Alt check: fc.nt: " + task.parsedXpattern.firstChild.nodeType );
      } else {
        logger.debug("Alt check: No firstChild");
      }
    } else {
      logger.debug("Alt check: No parsedXpattern");
    }
    
    // Check if alternative patterns
    if ( task.parsedXpattern &&
         task.parsedXpattern.nodeType == "|" &&
         task.parsedXpattern.firstChild &&
         task.parsedXpattern.firstChild.nodeType == "(" ) {
      // It is a pattern with alternatives
      logger.debug("Trying to populate a set of alternatives");
      // Create the outer group line (special one for alternativeset)
      var mainbox = addGroup ( null, nodebox, null, "alternatives" );
      nextaltnum = 1;
      var thisalt = task.parsedXpattern.firstChild;
      while ( thisalt ) {
        var altnum = nextaltnum++;
        logger.debug("Alt " + altnum + " " + thisalt.nodeType + " : " +
          thisalt.dumpString(-1) );
        var altbox = addGroup(thisalt, mainbox.cf_contentbox, null, "ALT",altnum);
        var besthit = findBestHit(task.xpatternHits, thisalt );
        if ( besthit ) {
          designerNode( thisalt.firstChild, besthit, 0,0, altbox.cf_contentbox );
        } else {
          break; // can not create, the code below will complain
        }
        thisalt = thisalt.nextAlternative;
      }
    } else { // "simple" pattern
      logger.debug("Populating from a 'simple' pattern " +
        task.parsedXpattern.dumpString(-1) );
      var besthit = findBestHit(task.xpatternHits, task.parsedXpattern );
      if ( besthit ) {
        //logger.debug("Looking at " + besthit.hits.length + " hits");
        designerNode(task.parsedXpattern, besthit, 0,0, nodebox );
      }
    }

    // Try to generate, and see if the result differs
    // In that case, display a warning, and keep the old pattern
    // Leave the designer populated with the lines anyway, in case the user
    // wants to continue from there, ignoring the warning.
    var oldpatt = context.conf.xpattern;
    generatePattern(true);
    var oldpatt2 = oldpatt.replace(/\s+/g,""); // ignore whitespace
    var newpatt2 = context.conf.xpattern.replace(/\s+/g,"");
    if ( oldpatt2.toUpperCase() != newpatt2.toUpperCase() ) {
      context.setPattern(oldpatt);
      logger.debug("Too complex. Have '" + oldpatt2 +
        "' would get '" + newpatt2 + "'" );
      alertmsg.hidden = false;
    }
    waitmsg.hidden = true;
  }; // drawCreateTab.populateDesigner

  // This gets called from setPattern 
  var updatecallback = function(patt) {
    if ( xpatt && xpatt.textContent != patt ) {
      logger.debug("Updating designer label" );
      if ( trimpat(patt) != trimpat(xpatt.textContent) ) {
        waitmsg.hidden = false; // signal we need to recalculate
        alertmsg.hidden = true; // only one msg at a time
        logger.debug("Real change in patter, invalidating the designer");
      }
      xpatt.textContent = patt;
    } else {
      logger.debug("Not updating designer label");
    }

  }; // updatecallback
  
  // drawCreateTab itself finishes here
  return updatecallback;

}; // drawCreateTab finally ends here!


///////////////
// History tab
// Always redraws the whole tab from the conf history, and from its cache
var historyhits={}; // global cache for redisplaying hit counts

ParseXpattern.prototype.drawHistoryTab = function (surface, context) {
  xmlHelper.emptyChildren(surface);
  //logger.debug("drawHistoryTab start");
  var vbox = xmlHelper.appendNode(surface, "vbox", null, null, null);
  //xulHelper.captionField(vbox,"Xpattern History");
  var historybox = xmlHelper.appendNode(vbox, "vbox", null, {flex:1}, null);
  if ( !context.conf.xpatternhistory ||
       !context.conf.xpatternhistory.length ) {
    logger.debug("No history, nothing to draw");
    return;
    }

  for ( var i = 0; i < context.conf.xpatternhistory.length; i++) {
    var patt = context.conf.xpatternhistory[i];
    var pb = xmlHelper.appendNode(historybox, "hbox" );
    xmlHelper.appendNode(pb,"vbox", null, { height:7}  );
    //logger.debug("h " + i + " hist = " + historyhits[patt] );
    if ( typeof(historyhits[patt]) == "undefined" )
      historyhits[patt] = "Try";
    var trybut = xmlHelper.appendNode(pb, "button", null,
           {label: historyhits[patt], width:50 } );
    trybut.cf_patt = patt;
    var revertbut = xmlHelper.appendNode(pb, "button", null,
           {label: "Revert", width:50, disabled:(i==0) } );
    revertbut.cf_patt = patt;
    
    var l = 0;  // find the part to highlight
    var r = patt.length-1;
    if ( i == context.conf.xpatternhistory.length -1 ) {
      // last one, no next to compare
      r = 0;
    } else { // find the difference from previous
             // actually, find length of matching beginning and end,
             // the rest must be the different part. Ignore whitespace
      var prev = context.conf.xpatternhistory[i+1];
      var pr = prev.length -1;
      var pl = 0;
      while ( l<patt.length && pl < prev.length) {
        if ( patt[l] == prev[pl] ) {
          l++;
          pl++;
        } else if ( patt[l].match(/\s/) ) {
          l++;
        } else if ( prev[pl].match(/\s/) ) {
          pl++;
        } else break; // found a true difference
      }
      while ( l > 0 && patt[l] && patt[l].match(/[a-zA-Z0-9$]/) )
        l--; // back to beginning of word
      while ( r > l && pr > pl  ) {
        if ( patt[r] == prev[pr] ) {
          r--;
          pr--;
        } else if ( patt[r].match(/\s/) ) {
          r--;
        } else if ( prev[pr].match(/\s/) ) {
          pr--
        } else break;
      }
      while ( r<patt.length && patt[r] && patt[r].match(/[a-zA-Z0-9]/) )
        r++; // skip to end of word
    }
    xmlHelper.appendNode(pb, "label", null, 
                { value: patt.substr(0,l),
                  style:"margin-right: 0px; " } );
    xmlHelper.appendNode(pb, "label", null, 
                { value: patt.substr(l, r-l+1), 
                         style:"margin-left: 0px; margin-right: 0px;font-weight:bold;"} );
    xmlHelper.appendNode(pb, "label", null, 
                         { value: patt.substr(r+1),
                         style:"margin-left: 0px; " } );
                         
    revertbut.addEventListener("click", function(e) {
      logger.debug("Revert button " + this.cf_patt );
      context.setPattern(this.cf_patt);
    }, false);
          
    trybut.addEventListener("click", function(e) {
      logger.debug("Try button " + this.cf_patt );
      var hits = "??";
      var tooltip = "";
      var task = new Task(context.task.connector, "dummy");
      var origpatt = context.conf.xpattern;
      try { // run the step with a fake task parameter to get the outputs
        context.conf.xpattern = this.cf_patt;
        context.run(task);
        hits = "" + task.xpatternHits.length + " hit";
        if ( task.xpatternHits.length != 1)
          hits += "s";
        if (task.xpatternwarnings) {
          hits += " " + task.xpatternwarnings + "w";
        } else { // count number of pieces in the hits
            var pcs = 0;
            for (var i=0; i<task.xpatternHits.length; i++) {
              var h = task.xpatternHits[i];
              for ( var j=0; j<h.hits.length; j++ ) {
                if ( h.hits[j].name && h.hits[j].value )
                  pcs++;
              }
            }
            hits += " " + pcs + "p";
            if ( task.xpatternHits.length + pcs >= 100)
              hits = hits.replace ( / hits?/, "h" );  // shorten text to fit a button
        }
      } catch (e) {
        logger.debug("Trying history pattern " + this.cf_patt + " failed " + 
            "with " + e  );
        if ( e.fileName )
          logger.debug(" at " + e.fileName + "." + e.lineNumber );
        if ( e.message == "No hits found" )
          hits = "No hits";
        else {
          hits = "Error"; // the user can revert and see it in the editor
          tooltip = e.message;
        }
      }
      this.label = hits;
      this.setAttribute("tooltiptext",tooltip);
      historyhits[this.cf_patt] = hits;
      context.conf.xpattern = origpatt;
      }, false );
    
  } // history loop


  var updatecallback = function(patt) {
    var oldhist = this.conf.xpatternhistory;
    if (!oldhist) // can happen with a brand new connector
      oldhist = []; // defensive coding, anyway.
    if ( oldhist[0] && oldhist[0] == patt ) {
      logger.debug("history update: Identical pattern, not updating");
      return; // same pattern, don't bother
    }
    var hist = [ patt ];
    for ( var i = 0; (i < oldhist.length) && (hist.length<20); i++) {
      //logger.debug("Looking at oldhist " + i + ": " + oldhist[i] );
      var good = true;
      for ( var j = 0; j <hist.length; j++ )
        if ( trimpat(hist[j]) == trimpat(oldhist[i]) ) {
          good = false;
          break;
        }
      if ( good ) {
        hist.push (oldhist[i]);
        //logger.debug("hist : " + oldhist[i] );
      }
    }
    this.conf.xpatternhistory = hist;
    this.drawHistoryTab( this.tabs[2], this);
  }; // updatecallback

  return updatecallback;
  
}; // history tab


// Highlight all hits
ParseXpattern.prototype.highlightHits = function (hitsarray) {
  // find the highlights from the dom tree
  // If we find just one dom element that is collapsed, we set the
  // collapsed flag in the pattern node, so we refuse to highlight anything
  // from that pattern. 
  for (var i=0; i<hitsarray.length; i++) {
    var h = hitsarray[i];
    for (var j=0; j<h.hits.length; j++) {
      var p = h.hits[j]["pattern"];
      p.is_collapsed = false; 
    }
  }
  for (var i=0; i<hitsarray.length; i++) {
    var h = hitsarray[i];
    for (var j=0; j<h.hits.length; j++) {
      var d = h.hits[j]["dom"];
      var p = h.hits[j]["pattern"];
      if ( d && p ) {
        if (d.nodeName == "#text" )
          d = d.parentNode;
        if ( d.nodeName != "#comment" && d.nodeName != "#cdata-section" ) {
          var hi = d.getAttribute("cf_user_highlight");
          var is_collapsed =  d.getAttribute("cf_collapsed") || "expanded";
          if (hi) {
              p.highlight = hi;
              if ( is_collapsed == "expanded") {
                xulHelper.highlightNode(d, hi, true );
              } else {
                xulHelper.unhighlightNode(d, false );
                p.is_collapsed = true;
                
              }
          }
          if ( ! p.highlight ) { // set highlight if not yet set
            var col = gethighlightcolor();
            p.highlight = col;
            if ( is_collapsed == "expanded" ) {
              xulHelper.highlightNode(d, col, true );
            } else {
              xulHelper.unhighlightNode(d, false );
              p.is_collapsed = true;
            }
          }
        }
      }
    }
  }
  // and distribute them to all hits
  for (var i=0; i<hitsarray.length; i++) {
    var h = hitsarray[i];
    for (var j=0; j<h.hits.length; j++) {
      var d = h.hits[j]["dom"];
      var p = h.hits[j]["pattern"];
      if ( d && p ) {
        if (p.highlight) {
          if (d.nodeName == "#text" ) {
            d = d.parentNode;
          }
          if ( d.nodename != "#comment" ) {
            //var is_collapsed =  d.getAttribute("cf_collapsed");
            var is_collapsed =  p.is_collapsed;
            if ( ! is_collapsed ) {
              xulHelper.highlightNode(d, p.highlight, false);
            } else {
              xulHelper.unhighlightNode(d, false);
            }
          }
        }
      }
    }
  }
  if ( this.nodebox ) {
    if ( this.nodebox.firstChild )
      this.rehighlightDesignerHits( this.nodebox.firstChild );
    else logger.debug("highlightHits: nodebox is empty, can not rehighlight!");
  }
  else
    logger.debug("highlightHits: No nodebox, can not highlight");
}; // highlightHits

// Rehighlight those hits on the page that correspond to the hits selected in the
// designer - they get only a dim highligh after generate / highlightHits, or
// none at all, if no $variable, and node not included in hits
ParseXpattern.prototype.rehighlightDesignerHits = function ( line ) {
  while (line) {
    if ( line.cf_lineType == "group" ) {
      this.rehighlightDesignerHits( line.cf_contentbox.firstChild );
    } else  if ( line.cf_domNode && ! line.cf_collapsed ) {
      xulHelper.highlightNode( line.cf_domNode, line.cf_highlight, true );
    }
    line = line.nextSibling;
  }
  // xulHelper.highlightNode: function(node, color, userhighlight )
}; // rehighlightDesignerHits

// Extract one hit into a structure of javascript objects that will
// later end up as suitable json, then xml
// Note that we need to append consequtive elements in the hits array,
// and keep track of nested groups
//  h is the hits array from XPattern match
//  j is the start index into it, in case of groups, recursion, etc
//  hitchecker contains the hitnumber to expect (and a pointer to the
//  task, in order to get to its logger...)

ParseXpattern.prototype._onehit = function (h,j, hitchecker) {
  var idx = {}; // current index for each tag (append to the fifth title)
  var res = {}; // the resulting record field->array of values
  while ( j < h.hits.length && h.hits[j]["name"] != "/" ) {
    var fullname = h.hits[j]["name"];
    var n = fullname.replace ( /\/.+$/, "" );  // remove the modifiers
    var skipthis = false; // signals to skip the hitnumber
    if ( n == "hitnumber" ) {
      if ( this.conf.removehitnumber )
        skipthis = true; // don't collect that variable at all!
      if (  this.conf.hitnumbercheck ) {
        var hn = parseInt(h.hits[j]["value"]);
        if ( hn ) { // got the actual variable, not a start marker
          logger.debug("Looking at hit number " + hn + " expecting " + hitchecker.hitnumber );
          if ( hn && hitchecker.hitnumber && hn != hitchecker.hitnumber ) {
            hitchecker.warnings++;
            if (this.conf.failhitnumber) {
              throw new StepError ("Hitnumber mismatch. " +
                  "Expected " + hitchecker.hitnumber + " got " + hn );
            } else {
              hitchecker.task.warn("Hitnumber mismatch. " +
                "Expected " + hitchecker.hitnumber + " got " + hn );
            }
          }      
          hitchecker.hitnumber = hn +1; // expect one higher next time
        }
      } // want to check 
    } // we have a hitnumber
    if ( ! skipthis ) {
      var lastchar = n.charAt(n.length-1,1);
      if ( lastchar == "/" ) {
        n = n.slice(0,-1);
      }
      var v = h.hits[j]["value"];
      //dump("parse: hit ." + j + " n='" + n + "'" + " v='" + v + "' " + "\n");
      if ( v == "" && n == "" ) { // no name, no value, must be a builder marker
        // Ignore the whole thing.
      } else if ( v == "" ) { // signals a new field
        if ( idx[n] == undefined ) { // first time we see it
          idx[n]=0;
          //dump("First time we see '" + n + "'\n");
          res[n]=[""];
        } else {
          idx[n]++;
          //dump("Incremented index '" + n + "' to " + idx[n] + "\n");
          res[n][idx[n]]="";
        }
        if ( lastchar == "/" ) { // nested group
          //dump("Recursion \n");
          var rr = this._onehit(h, j+1, hitchecker);
          //dump("Recursion done\n");
          j=rr["j"];
          res[n][idx[n]]=rr["res"];
        }
      } else { // real data
        if ( idx[n] == undefined ) {
          // May happen in groups, with unvariabled fields as in
          // ( TD { A $author : B $title } ) $item
          // <TD><A>au</A><B>ti</B>extra text</TD>
          //dump("Skipping unnamed subfield\n");
        }
        else {
          var k = idx[n];
          if (res[n][k] == undefined) {
              res[n][k] = "";
          }
          if (res[n][k] != "" && fullname.indexOf("/whitespace") == -1) {
              res[n][k] += " ";
          }
          res[n][k] += v;
          //dump("Appended to res['" + n + "']["+k+"]: '" + res[n][k] + "'\n");
        }
      }
    } // ! skipthis
    j++;
  }
  var r = {};
  r["res"] = res;
  r["j"] = j;
  return r;
} // _onehit


ParseXpattern.prototype.run = function (task) {
  var hitarea = this.conf['hitarea'];
  var pageDoc = this.getPageDoc();
  var node = null;
  try {
    node = xmlHelper.getElementByNodeSpec(pageDoc, hitarea);
  } catch (e) {
    var ns = xmlHelper.nodeSpecToString(hitarea);
    task.debug("Bad XPath for hitarea" + ns );
    task.debug(e.message);
    if ( e.fileName  )  // catch syntax errors in code, etc
          logger.debug("  in " + e.fileName + "." + e.lineNumber  );
     throw new StepError("Hit area '" + ns + "' not legal xpath");
  }
  if (node == null) {
    var ns = xmlHelper.nodeSpecToString(hitarea);
    throw new StepError("Hit area '" + ns + "' not found in the document");
  }
  //dump("hit area node: \n"); xmlHelper.dumpxml(node);
  //dump("Parsing mode: " + pageDoc.compatMode + "\n");
  var patt = this.conf['xpattern'];
  try {
      //dump("Parsing '" + patt + "'\n");
      var par = new XpatternTextParser(patt);
      var xp = par.parse();
  } catch (e) {
      dump("Parse " + e + "\n'" + patt + "'\n" );
      throw new StepError("Bad xpattern '" + patt + "'" );
      return;
  }
  //dump("Xpattern: \n" + xp.dumpString(-1) + "\n" );

  if (this.inBuilder) {
    // dump("in builder, setting hightlights on \n");
    xulHelper.unhighlightAll(node,false);
    xp.askForNodes();
  }

  var hitsarray = xp.match(node);
  //dump("match returned " + hitsarray.length + " hits\n");
  if (this.inBuilder ) {
      if ( this.conf['highlighthits'] ) {
        this.highlightHits(hitsarray);
      }
      task.xpatternHits = hitsarray; // pass to the builder
      task.parsedXpattern = xp; // remember the actual pattern tree
  }
  var resarr = [];
  var hitnumber = undefined; 
  if (this.conf.keephitnumber) {
    hitnumber = parseInt(jsonPathHelper.get(this.conf["hitnumbervar"], task.data));
  }
  logger.debug("Expecting to start with hitnumber " + hitnumber );
  var hitchecker = { hitnumber: hitnumber, warnings: 0, task: task };
  for (var i=0; i<hitsarray.length; i++) {
      var r = this._onehit(hitsarray[i],0, hitchecker);
      resarr.push(r["res"]);
  }
  if (this.conf.keephitnumber) {
    jsonPathHelper.set(this.conf["hitnumbervar"], hitchecker.hitnumber, task.data);
  }
  if (this.inBuilder ) {
    task.xpatternwarnings = hitchecker.warnings;
  }
  if (resarr.length == 0 ) {
    throw new StepError("No hits found");
  } else {
    jsonPathHelper.set(this.conf["jsonPath"], resarr, task.data);
  }
}; // run

ParseXpattern.prototype.getClassName = function () {
    return "ParseXpattern";
};

ParseXpattern.prototype.getDisplayName = function () {
  return "Parse by XPattern";
};

ParseXpattern.prototype.getDescription = function () {
  return "Parses results using XPattern.";
};

ParseXpattern.prototype.getVersion = function () {
  // 0.2 introduced the negations
  // 0.3 supports the new data model
  // 1.0 data-model related global version bump
  // 1.1 autogenerate flag in config
  // 2.0 hitnumber check
  // 3.0 xpattern modifiers
  return "3.0";
};

ParseXpattern.prototype.upgrade = function (confVer, curVer, conf) {
  // can't upgrade if the connector is newer than the step
  if (confVer > curVer)
    return false;
  if ( typeof(conf['highlighthits']) == 'undefined' ){
    conf['highlighthits'] = true;
  }
  if (confVer < 0.3) {
    conf['jsonPath'] = { path: "$.output", key:"results" };
  }
  if (confVer < 1.1) {
    logger.debug("Upgrading parse_xpattern step to 1.1" );
    conf.autogenerate = true;
  }
  if (confVer < 2.0) {
    conf.hitnumbercheck = false;
    conf.removehitnumber = true;
    conf.keephitnumber = true; 
    conf.failhitnumber = true; 
    conf.hitnumbervar = { path: "$.session", key:"hitnumber" };
  }
  return true;
};

// There is really no point in providing a renderArgs() method
// for the XPattern parser, as the XPattern itself is too long to fit
// nicely and too complex to be grokkable at a glance.
// Therefore, no renderArgs function.


// TODO - Wouldn't this be easier from the string-format XPattern?
// Just find all $something strings...
ParseXpattern.prototype.capabilityFlagDefault = function (flag) {
  if (flag.substring(0, 7) != "result-")
    return null;

  var xp_returns_field = function (level, field, xp) {
    //dump("in xp_returns_field(" + level + ", " + field + ", " + xp + ") -- xp.variable=" + xp.variable +"\n");
    if (xp.variable && xp.variable == field)
      return true;
    if (xp.attributes && xp_returns_field(level+1, field, xp.attributes) )
      return true;
    if (xp.firstChild && xp_returns_field(level+1, field, xp.firstChild) )
      return true;
    if (xp.nextAlternative && xp_returns_field(level+1, field, xp.nextAlternative) )
      return true;
    if (xp.nextSibling && xp_returns_field(level+1, field, xp.nextSibling) )
      return true;
    return false;
  }

  // TODO We should cache the parsed pattern, we get here often
  if ( this.conf.xpattern ) { // maybe we don't have any defined yet
    var par = new XpatternTextParser(this.conf['xpattern']);
    var xp = par.parse();
    if (xp_returns_field(0, flag.substring(7), xp)) {
      //dump("xp_returns_field(" + flag + ") was true!\n");
      return true;
    } else {
      //dump("xp_returns_field(" + flag + ") was false!\n");
      return null;
    }
  }
  return null;
}


