/**
 * \file connector_wrap.cpp
 * \brief Implements ConnectorWrap, ConnectorReturn, ConnectorLog
 */

#include "cf_config.h"
#include <assert.h>
#include "nsCOMPtr.h"
#include "nsServiceManagerUtils.h"
#include "connector_wrap.h"
#include <stdio.h>

ConnectorWrap::~ConnectorWrap()
{
}

nsresult ConnectorWrap::init(const char *modulePath)
{
    nsCOMPtr<nsIServiceManager> servMan;
    nsresult rv = NS_GetServiceManager(getter_AddRefs(servMan));
    if (NS_FAILED(rv))
    {
        m_logger->printf("engine", YLOG_WARN, "NS_FAILED GetServiceManager");
	return rv;
    }
    rv = servMan->GetServiceByContractID(
	"@indexdata.com/connectorruntime;1",
	NS_GET_IID(nsIConnectorRuntime),
	getter_AddRefs(m_c));

    if (NS_FAILED(rv))
    {
        m_logger->printf("engine", YLOG_WARN, "NS_FAILED GetServiceByContract");
	return rv;
    }

    m_log = new ConnectorLog(this);

    rv = m_c->Init(modulePath, m_log);
    if (NS_FAILED(rv))
    {
        m_logger->printf("engine", YLOG_WARN, "NS_FAILED Init");
        return rv;
    }
    m_return = new ConnectorReturn(this);
    return NS_OK;
}

ConnectorWrap::ConnectorWrap(const char *modulePath, CF_Logger *log_int) :
    m_c(0), m_return(0), m_resultfunc(0), m_test_func(0), m_vp(0),
    m_logger(log_int)
{
    init(modulePath);
}

bool ConnectorWrap::setBrowserWindow(nsIDOMWindow *win)
{
    if (!m_c)
        return false;
    assert(m_c);
    assert(win);
    nsresult rv = m_c->SetBrowserWindow(win);
    if (NS_FAILED(rv))
    {
        m_logger->printf("engine", YLOG_WARN, "NS_FAILED findBrowserWindow");
	return false;
    }
    return true;
}

bool ConnectorWrap::loadConf(const char *filePath, const char *sessionJSON, char **errormessage)
{
    *errormessage = 0;
    nsresult rv = m_c->LoadConf(filePath, sessionJSON, errormessage);
    m_logger->printf("engine", YLOG_LOG, "loadConf %s", filePath);
    if (NS_FAILED(rv))
    {
        m_logger->printf("engine", YLOG_WARN,
                         "NS_FAILED loadConf %s %s", filePath,
                         errormessage);
	return false;
    }
    if (*errormessage)
    {
        m_logger->printf("engine", YLOG_WARN,
                         "loadConf returns error %s", *errormessage);
        return false;
    }
    return true;
}

bool ConnectorWrap::saveConf(const char *filePath, char **errormessage)
{
    *errormessage = 0;
    nsresult rv = m_c->SaveConf(filePath, errormessage);
    m_logger->printf("engine", YLOG_LOG, "saveConf %s", filePath);
    if (NS_FAILED(rv))
    {
        m_logger->printf("engine", YLOG_WARN, "NS_FAILED saveConf %s",
                         filePath);
	return false;
    }
    if (*errormessage)
    {
        m_logger->printf("engine", YLOG_WARN,
                         "saveConf returns error %s", *errormessage);
        return false;
    }
    return true;
}

bool ConnectorWrap::runScript(const char *filePath, char **errormessage)
{
    *errormessage = 0;
    nsresult rv = m_c->RunScript(filePath, errormessage);
    m_logger->printf("engine", YLOG_LOG, "runScript %s", filePath);
    if (NS_FAILED(rv))
    {
        m_logger->printf("engine", YLOG_WARN, "NS_FAILED runScript %s",
                         filePath);
	return false;
    }
    if (*errormessage)
    {
        m_logger->printf("engine", YLOG_WARN, "runScript returns error %s",
                         *errormessage);
        return false;
    }
    return true;
}

void ConnectorWrap::runTask(const char *taskName,
                            bool optional,
			    const char *jsonStrInput,
			    void *vp,
                            resultfunc_t *resultfunc)
{
    m_resultfunc = resultfunc;
    m_vp = vp;

    m_logger->printf("engine", YLOG_LOG, "%s %s %s",
              optional ? "task_opt" : "task", taskName, jsonStrInput);
    nsresult rv = m_c->RunTask(taskName, optional, jsonStrInput, m_return);
    if (NS_FAILED(rv))
    {
        m_logger->printf("engine", YLOG_WARN, "NS_FAILED runTask %s", taskName);
        if (m_resultfunc)
            m_resultfunc(m_vp, false, "runTask failure");
    }
}

void ConnectorWrap::runTests(const char *taskOrder, void *vp,
                             resultfunc_t *resultfunc,
                             test_func_t *test_func)
{
    m_resultfunc = resultfunc;
    m_test_func = test_func;
    m_vp = vp;
    m_logger->printf("engine", YLOG_LOG, "runTests taskOrder %s", taskOrder);
    nsresult rv = m_c->RunTests(taskOrder, m_return);
    if (NS_FAILED(rv))
    {
        m_logger->printf("engine", YLOG_WARN, "NS_FAILED runTests %s",
                         taskOrder);
        if (m_resultfunc)
            m_resultfunc(m_vp, false, "runTests failed:"
                     " possibly required task(s) unavailable for test");
    }
}

bool ConnectorWrap::unitTest(const char *name)
{
    CF_BOOL retval;
    nsresult rv = m_c->UnitTest(name, &retval);
    if (NS_FAILED(rv))
        return false;
    return retval;
}

bool ConnectorWrap::DOMString(char **retval)
{
    *retval = 0;
    m_c->DOMString(retval);
    m_logger->printf("engine", YLOG_WARN,
                     "??? %s ...null string, why?\n", *retval);
    return true;
}

NS_IMPL_ISUPPORTS2(ConnectorReturn,
		   nsIConnectorReturn,
		   nsISupports)

ConnectorReturn::ConnectorReturn(ConnectorWrap *w) : m_wrap(w)
{
}

ConnectorReturn::~ConnectorReturn()
{
}

NS_IMETHODIMP ConnectorReturn::TaskDone(CF_BOOL success, const char *jsonStr)
{
    if (m_wrap->m_resultfunc)
        m_wrap->m_resultfunc(m_wrap->m_vp, success, jsonStr);
    m_wrap->m_resultfunc = 0;
    return NS_OK;
}

NS_IMETHODIMP ConnectorReturn::TestsFinished(CF_BOOL success)
{
    m_wrap->m_logger->printf("engine", YLOG_LOG,
                             "ConnectorReturn::TestsFinished %s",
                          success ? "OK" : "Failed");
    if (m_wrap->m_test_func)
        m_wrap->m_test_func(m_wrap->m_vp, success);
    m_wrap->m_test_func = 0;
    return NS_OK;
}

NS_IMPL_ISUPPORTS2(ConnectorLog,
		   nsIConnectorLog,
		   nsISupports)

ConnectorLog::ConnectorLog(ConnectorWrap *w) : m_wrap(w)
{
}

ConnectorLog::~ConnectorLog()
{
}

NS_IMETHODIMP ConnectorLog::Log(const char *level, const char *message)
{
    int ylevel = YLOG_LOG;
    if (!strcmp(level, "DEBUG"))
        ylevel = YLOG_DEBUG;
    else if (!strcmp(level, "INFO"))
        ylevel = YLOG_LOG;
    else if (!strcmp(level, "WARN"))
        ylevel = YLOG_WARN;
    else if (!strcmp(level, "ERROR"))
        ylevel = YLOG_FATAL;
    m_wrap->m_logger->printf("runtime", ylevel, "%s", message);
    return NS_OK;
}

/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */

