var EXPORTED_SYMBOLS = ["UrlManipulator", "openUrlManipulator"];
Components.utils.import('resource://indexdata/ui/app.js');
Components.utils.import('resource://indexdata/util/mozHelper.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/xulHelper.js');
Components.utils.import('resource://indexdata/util/jsonPathHelper.js');
Components.utils.import('resource://indexdata/util/logging.js');

function openUrlManipulator(url) {
  //onload registered through this will fire only once
  mozHelper.openOrFocusWindow(
      function (e) {
        var um = new UrlManipulator()
        um.init(e.target.defaultView, e.target, url);
        e.target.defaultView.controller = um;
        e.target.defaultView.redrawView = function () {
          this.controller.draw();
        }
      },
      'chrome://cfbuilder/content/url_manipulator.xul',
      'cfUrlManipulator',
      'resizable=yes,chrome=yes,titlebar=yes,centerscreen,dependent').focus();
}

//helpers
const MAX_WIDTH = 1024;
const MAX_HEIGHT = 800;

var child = xmlHelper.appendNode;
var empty = xmlHelper.emptyChildren;
var exvars = jsonPathHelper.inlineReplace;

function sizeToContentBounded(win, x, y) {
  if (win.outerWidth < x && win.outerHeight < y)
    win.sizeToContent();
}

var UrlManipulator = function () {
  this.surface = null;
  this.win = null;
  this.url = null;
  this.decode = true;
  this.encode = true;
  this.expand = false;
};

var logger = logging.getLogger();

UrlManipulator.prototype.init = function(win, doc, url) {
  this.win = win;
  this.surface = doc.getElementById('cfUrlManipulator');
  this.url = url;
};

function parseParams(qs, decode) {
  let params = [];
  if (!qs) return params;
  let pa = qs.split('&');
  for (let i = 0; i<pa.length; i++ ) {
    let kv = pa[i];
    if (kv.length < 1) continue;
    let ei =  kv.indexOf('=');
    let name = ei > -1 ? kv.substring(0, ei) : kv;
    let value = ei > -1 && ei < kv.length 
      ? kv.substring(ei+1).replace(/\+/g," ") : '';
    params.push({
      name: decode ? decodeURIComponent(name) : name, 
      value: decode ? decodeURIComponent(value) : value
    });
  }
  return params;
}

function parsePath(path, decode) {
  let comps = [];
  let parts = path.split(/\/+/);
  for (let i=0; i<parts.length; i++) {
    let p = parts[i];
    if (p)
      comps.push({value: decode ? decodeURIComponent(p) : p});
  }
  return comps;
}

function parseHash(hash, decode) {
  return decode ? decodeURIComponent(hash) : hash;
}

function parseUrl(url, decode) {
  if (typeof url != 'string') 
    return null;
  let parsed = {};
  let m = url.match(/^((?:https?\:\/\/)?[^\/?#]*)([^?#]*)\??([^#]*)#?(.*)/);
  if (m.length < 4) return null;
  parsed.host = m[1];
  parsed.path = parsePath(m[2], decode);
  parsed.ts = m[2].indexOf('/', m[2].length-1) !== -1 ?
    '/' : '';
  parsed.params = parseParams(m[3], decode);
  parsed.hash = parseHash(m[4], decode);
  return parsed;
}

function serializeParams(params, encode, expand) {
  let qs = '';
  let sep = '?';
  for (let i=0; i<params.length; i++) {
    let p = params[i];
    if (p.disabled) continue;
    let name = expand ? exvars(p.name, app.selectedTask.data) : p.name;
    name = encode ? encodeURIComponent(name) : name;
    let value = expand ? exvars(p.value, app.selectedTask.data) : p.value;
    value = encode ? encodeURIComponent(value) : value;
    qs += sep + name + "=" + value;
    sep = '&';
  }
  return qs;
}

function serializePath(parts, encode, expand) {
  let path = "";
  for (let i=0; i<parts.length; i++) {
    let p = parts[i];
    if (p.disabled) continue;
    let value = expand ? exvars(p.value, app.selectedTask.data) : p.value;
    value = encode ? encodeURIComponent(value) : value;
    path += "/" + value;;
  }
  return path;
}

function serializeHash(hash, encode, expand) {
  if (!hash) return '';
  let value = expand ? exvars(hash, app.selectedTask.data) : hash;
  value = encode ? encodeURIComponent(value) : value;
  return '#' + value;
}

function serializeUrl(url, encode, expand) {
  if (expand && !app.selectedTask) {
    let ps = Components.classes["@mozilla.org/embedcomp/prompt-service;1"]
      .getService(Components.interfaces.nsIPromptService);
    ps.alert(null, "No task selected in the CF sidebar.",
        "Variable expansion has been enabled but no task is selected in the"
        + " builder sidebar. Please select a task or disable the "
        +"'Expand variables' checkbox.");
    return;
  }
  let host = expand ? exvars(url.host, app.selectedTask.data) : url.host;
  return host + 
    serializePath(url.path, encode, expand) + url.ts + 
    serializeParams(url.params, encode, expand)+ 
    serializeHash(url.hash, encode, expand); 
}

UrlManipulator.prototype._makeRow = 
  function(rows, i, label, objs, urlBox, url, editlabel) {
  let context = this;
  let row = child(rows, "row");
  let disabled = objs[i].disabled ? true : false;
  let cb = child(row, "checkbox", null, 
      {checked: (!disabled).toString(), paramIndex: i});
  if (editlabel) {
    let txtBox = child(row, "textbox", null, 
      {value: objs[i].name, paramIndex: i});
    txtBox.addEventListener('keyup', function (e) {
      let i = parseInt(e.target.getAttribute('paramIndex'));
      objs[i].name = e.target.value; 
      urlBox.value = serializeUrl(url, context.encode, context.expand);
    });
  } else {
    child(row, "label", label);
  }
  let txtBox = child(row, "textbox", null, 
    {value: objs[i].value, paramIndex: i});
  cb.addEventListener('command', function (e) {
      let i = parseInt(e.target.getAttribute('paramIndex'));
      objs[i].disabled = !e.target.checked;
      urlBox.value = serializeUrl(url, context.encode, context.expand);
  });
  txtBox.addEventListener('keyup', function (e) {
      let i = parseInt(e.target.getAttribute('paramIndex'));
      objs[i].value = e.target.value; 
      urlBox.value = serializeUrl(url, context.encode, context.expand);
   });
}

UrlManipulator.prototype.draw = function() {
  let context = this;
  empty(this.surface);
  child(this.surface, 'separator');
  let vbox = child(this.surface, 'vbox');
  let hbox = child(vbox, "hbox");
  child(hbox, "label", "Source URL");
  let urlIn = child(hbox, 'textbox', null, {flex: 1, value: this.url, 
      size: 75});
  let genBtn = child(hbox, 'button', null, {label: "Parse"});
  let hbox2 = child(vbox, 'hbox', null, {align: "left"});
  let dCb = child(hbox2, 'checkbox', null, {label: 'Decode URL components',
      checked: context.decode});
  dCb.addEventListener('command', function (e) {
    context.decode = e.target.checked;
  });
  child(this.surface, 'separator');
  sizeToContentBounded(this.win, MAX_WIDTH, MAX_HEIGHT);
  let parseBox = child(this.surface, 'vbox');
  genBtn.addEventListener("command",
    function (e) {
      let url = '';
      try {
        url = parseUrl(urlIn.value, context.decode);
      } catch (e) {
        let ps = Components.classes["@mozilla.org/embedcomp/prompt-service;1"]
         .getService(Components.interfaces.nsIPromptService);
        ps.alert(null, "Problems parsing the URL",
          "There were problems parsing the URL (see console for details)."
          + " You may want to try turning 'Decode URL components' off.");
        //abort
        throw e;
      }
      empty(parseBox);
      let partsBox = child(parseBox, 'vbox');
      //output
      let resVBox = child(parseBox, 'vbox');
      let resHBox = child(resVBox, 'hbox');
      child(resHBox, 'label', null, {value: 'Output URL:'});
      let urlBox = child(resHBox, 'textbox', null, {readonly: 'true',
        value: serializeUrl(url, context.encode, context.expand), flex: 1});
      let openBtn = child(resHBox, 'button', null, {label: 'Open'}); 
      openBtn.addEventListener('command', function (e) {
        app.openOrReuseTab(serializeUrl(url, context.encode, context.expand));
      });
      let resHBox2 = child(resVBox, 'hbox', null, {align: "left"});
      let eCb = child(resHBox2, 'checkbox', null, {label: 'URL encode',
          checked: context.encode});
      eCb.addEventListener('command', function (e) {
        context.encode = e.target.checked;
        urlBox.value = serializeUrl(url, context.encode, context.expand);
      });
      let exCb = child(resHBox2, 'checkbox', null, {label: 'Expand variables',
          checked: context.expand});
      exCb.addEventListener('command', function (e) {
        //check if task selected
        if (!app.selectedTask) {
          let ps = Components.classes["@mozilla.org/embedcomp/prompt-service;1"]
           .getService(Components.interfaces.nsIPromptService);
          ps.alert(null, "No task selected in the CF sidebar.",
            "In order to perform variable expansion you need to select a task"
            + " in the builder sidebar");
          e.target.checked = false;
          return;
        }
        context.expand = e.target.checked;
        urlBox.value = serializeUrl(url, context.encode, context.expand);
      });
      //host
      child(partsBox, "label", null, {value: "URL base"});
      let hostB = child(partsBox, "textbox", null, {value: url.host});
      hostB.addEventListener('keyup', function (e) {
        url.host = e.target.value; 
        urlBox.value = serializeUrl(url, context.encode, context.expand);
      });
      //path
      {
        child(partsBox, "label", null, {value: "URL path"});
        let grid = child(partsBox, "grid");
        let cols = child(grid, 'columns');
        child(cols, 'column', null, {flex: 0});
        child(cols, 'column', null, {flex: 0});
        child(cols, 'column', null, {flex: 10});
        let rows = child(grid, 'rows');
        for (let i=0; i<url.path.length; i++) {
          context._makeRow(rows, i, i.toString(),
            url.path, urlBox, url);
        }
        let btnBox = child(partsBox, 'vbox', null, {align: 'left'});
        let btn = child(btnBox, 'button', null, {label: "Add"});
        btn.addEventListener('command', function (e) {
            url.path.push({value: ''});
            context._makeRow(rows, url.path.length-1, 
              (url.path.length-1).toString(),
              url.path, urlBox, url);
            sizeToContentBounded(context.win, MAX_WIDTH, MAX_HEIGHT);
        });
      }
      //params
      {
        child(partsBox, "label", null, {value: "URL parameters"});
        let grid = child(partsBox, "grid");
        let cols = child(grid, 'columns');
        child(cols, 'column', null, {flex: 0});
        child(cols, 'column', null, {flex: 2});
        child(cols, 'column', null, {flex: 10});
        let rows = child(grid, 'rows');
        for (let i=0; i<url.params.length; i++) { 
          context._makeRow(rows, i, url.params[i].name,
              url.params, urlBox, url, true);
        }
        let btnBox = child(partsBox, 'vbox', null, {align: 'left'});
        let btn = child(btnBox, 'button', null, {label: "Add"});
        btn.addEventListener('command', function (e) {
            url.params.push({value: '', name: '', disabled: false});
            context._makeRow(rows, url.params.length-1, '',
              url.params, urlBox, url, true);
            sizeToContentBounded(context.win, MAX_WIDTH, MAX_HEIGHT);
        });
      }
      //fragment
      child(partsBox, "label", null, {value: "URL fragment"});
      let hB = child(partsBox, "textbox", null, {value: url.hash});
      hB.addEventListener('keyup', function (e) {
        url.hash = e.target.value; 
        urlBox.value = serializeUrl(url, context.encode, context.expand);
      });
      //resize
      sizeToContentBounded(context.win, MAX_WIDTH, MAX_HEIGHT);
    }
  );
};
