/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Mozilla Corporation.
 * Portions created by the Initial Developer are Copyright (C) 20007
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Pelle Johnsen <pjohnsen@mozilla.com>
 *   Dave Camp <dcamp@mozilla.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "cf_config.h"
#include "xpcom-config.h"
#include "mozilla-config.h"

#include "embed.h"
#include "EmbeddingSetup.h"

// CRT headers
#include <iostream>
#include <string>
using namespace std;

// Mozilla Frozen APIs
#include "nsXULAppAPI.h"
#include "nsXPCOMGlue.h"

#include "nsCOMPtr.h"
#include "nsStringAPI.h"
#include "nsComponentManagerUtils.h"
#include "nsServiceManagerUtils.h"
#include "nsIWeakReference.h"
#include "nsIWeakReferenceUtils.h"
#include "nsIWebBrowserStream.h"
#include "nsIURI.h"
#include <nsIPrefService.h>
#include <yaz/wrbuf.h>

#include "nsIWebBrowser.h"
#include "nsIWebNavigation.h"
#include "nsEmbedCID.h"

#include "nsIWebBrowserFocus.h"
#include "nsIWidget.h"
#include "nsIWindowCreator2.h"
#include "nsIWindowWatcher.h"
#include "nsIIOService.h"
#include "nsIBaseWindow.h"
#include "nsIDocShellTreeItem.h"
#include "nsIConsoleService.h"

// our stuff
#include "WebBrowserChrome.h"
#include "ConsoleListener.h"

// globals
static nsCOMPtr<WindowCreator> sWindowCreator;

#ifndef XPCOM_GLUE
#include "nsNetUtil.h"
#endif

MozApp::MozApp(CF_Logger *logger) : m_logger(logger)
{
}

nsresult MozApp::init(const char *aAppPath, const char *aProfileDir)
{
    return InitEmbedding(aAppPath, aProfileDir, m_logger);
}

MozApp::~MozApp()
{
    TermEmbedding(m_logger);
}

nsresult MozApp::SetCharPref(const char *name, const char *value)
{
    nsresult rv;

    nsCOMPtr<nsIPrefBranch> pref (do_GetService (NS_PREFSERVICE_CONTRACTID, &rv));
    if (NS_FAILED (rv)) return rv;

    rv = pref->SetCharPref (name, value);

    return rv;
}

nsresult MozApp::SetBoolPref(const char *name, CF_BOOL value)
{
    nsresult rv;

    nsCOMPtr<nsIPrefBranch> pref (do_GetService (NS_PREFSERVICE_CONTRACTID, &rv));
    if (NS_FAILED (rv)) return rv;

    rv = pref->SetBoolPref (name, value);

    return rv;
}

nsresult MozApp::SetIntPref(const char *name, int value)
{
    nsresult rv;

    nsCOMPtr<nsIPrefBranch> pref (do_GetService (NS_PREFSERVICE_CONTRACTID, &rv));
    if (NS_FAILED (rv)) return rv;    if (NS_FAILED (rv)) return rv;

    rv = pref->SetIntPref (name, value);

    return rv;
}

nsresult MozApp::GetCharPref(const char *name, char **value)
{
    nsresult rv;

    nsCOMPtr<nsIPrefBranch> pref (do_GetService (NS_PREFSERVICE_CONTRACTID, &rv));
    if (NS_FAILED (rv)) return rv;

    rv = pref->GetCharPref(name, value);

    return rv;
}

nsresult MozApp::GetBoolPref(const char *name, CF_BOOL *value)
{
    nsresult rv;

    nsCOMPtr<nsIPrefBranch> pref (do_GetService (NS_PREFSERVICE_CONTRACTID, &rv));
    if (NS_FAILED (rv)) return rv;

    rv = pref->GetBoolPref (name, value);

    return rv;
}

nsresult MozApp::GetIntPref(const char *name, int *value)
{
    nsresult rv;

    nsCOMPtr<nsIPrefBranch> pref (do_GetService (NS_PREFSERVICE_CONTRACTID, &rv));
    if (NS_FAILED (rv)) return rv;

    rv = pref->GetIntPref (name, value);

    return rv;
}

class MozView::Private{
public:
    Private() : parentWindow(NULL), pListener(NULL),
		parentView(NULL) {}

    void* parentWindow;
    MozViewListener* pListener;
    MozView* parentView;

    nsCOMPtr<nsIWebBrowser> webBrowser;
    nsCOMPtr<nsIWebNavigation> webNavigation;
    nsCOMPtr<nsIWebBrowserChrome> chrome;
    nsCOMPtr<nsIConsoleListener> consoleListener;
    CF_Logger *logger;
};

class WindowCreator :
    public nsIWindowCreator2
{
public:
    WindowCreator() {};
    virtual ~WindowCreator() {};

    NS_DECL_ISUPPORTS
    NS_DECL_NSIWINDOWCREATOR
    NS_DECL_NSIWINDOWCREATOR2
};

NS_IMPL_ISUPPORTS2(WindowCreator, nsIWindowCreator, nsIWindowCreator2)

NS_IMETHODIMP
WindowCreator::CreateChromeWindow(nsIWebBrowserChrome *parent,
				  PRUint32 chromeFlags,
				  nsIWebBrowserChrome **_retval)
{
    NS_ENSURE_ARG_POINTER(_retval);

    // get the listener from parent
    // we assume parent is a WebBrowserChrome
    WebBrowserChrome* chrome = static_cast<WebBrowserChrome*>(parent);

    MozViewListener* pListener = chrome->GetMozView()->GetListener();
    if(!pListener)
	return NS_ERROR_FAILURE;

    MozView* mozView = pListener->OpenWindow(chromeFlags);
    if(!mozView)
	return NS_ERROR_FAILURE;

    *_retval = mozView->mPrivate->chrome;
    NS_IF_ADDREF(*_retval);
    return NS_OK;
}

NS_IMETHODIMP
WindowCreator::CreateChromeWindow2(nsIWebBrowserChrome *parent,
				   PRUint32 chromeFlags, PRUint32 contextFlags,
				   nsIURI *uri, CF_BOOL *cancel,
				   nsIWebBrowserChrome **_retval)
{
    return CreateChromeWindow(parent, chromeFlags, _retval);
}

MozView::MozView(CF_Logger *logger)
{
    mPrivate = new Private();

    mPrivate->logger = logger;

    // TODO: should probably deal with WindowCreator in InitEmbedding
    //       or maybe in mozapp
    if(sWindowCreator)
	return;

    // create an nsWindowCreator and give it to the WindowWatcher service
    sWindowCreator = new WindowCreator();
    if (!sWindowCreator)
	return;// NS_ERROR_OUT_OF_MEMORY;

    nsCOMPtr<nsIWindowWatcher> wwatch(do_GetService(NS_WINDOWWATCHER_CONTRACTID));
    if (!wwatch)
	return;// NS_ERROR_UNEXPECTED;

    wwatch->SetWindowCreator(sWindowCreator);
}

MozView::~MozView()
{
    // release browser and chrome
    nsCOMPtr<nsIBaseWindow> baseWindow;
    baseWindow = do_QueryInterface(mPrivate->webBrowser);
    if(baseWindow)
	baseWindow->Destroy();
    if(mPrivate->chrome) {
	mPrivate->chrome->SetWebBrowser(NULL);
    }

    baseWindow = NULL;

    nsresult rv;
    nsCOMPtr<nsIConsoleService> consoleService =
	do_GetService(NS_CONSOLESERVICE_CONTRACTID, &rv);

    consoleService->UnregisterListener(mPrivate->consoleListener);

    mPrivate->webBrowser = NULL;
    mPrivate->chrome = NULL;
    mPrivate->consoleListener = NULL;
    delete mPrivate;
}

nsresult MozView::CreateBrowser(void* aParentWindow, PRInt32 x, PRInt32 y,
				PRInt32 width, PRInt32 height, PRUint32 chromeFlags)
{
    mPrivate->parentWindow = aParentWindow;

    nsresult rv;

    nsCOMPtr<nsIBaseWindow> baseWindow;
    mPrivate->webBrowser = do_CreateInstance(NS_WEBBROWSER_CONTRACTID, &rv);
    if (NS_FAILED(rv)) {
        mPrivate->logger->printf("engine", YLOG_FATAL,
                                 "NS_FAILED do_CreateInstance webBrowser");
        return rv;
    }
    baseWindow = do_QueryInterface(mPrivate->webBrowser);
    rv = baseWindow->InitWindow(mPrivate->parentWindow, 0, x, y, width, height);
    if (NS_FAILED(rv)) {
        mPrivate->logger->printf("engine", YLOG_FATAL, "NS_FAILED InitWindow");
    }

    nsIWebBrowserChrome **aNewWindow = getter_AddRefs(mPrivate->chrome);
    CallQueryInterface(static_cast<nsIWebBrowserChrome*>(new WebBrowserChrome(this)), aNewWindow);

    mPrivate->webBrowser->SetContainerWindow(mPrivate->chrome);
    mPrivate->chrome->SetWebBrowser(mPrivate->webBrowser);
    mPrivate->chrome->SetChromeFlags(chromeFlags);

    if(chromeFlags & (nsIWebBrowserChrome::CHROME_OPENAS_CHROME |
		      nsIWebBrowserChrome::CHROME_OPENAS_DIALOG) ) {
	nsCOMPtr<nsIDocShellTreeItem> docShellItem(do_QueryInterface(baseWindow));
	docShellItem->SetItemType(nsIDocShellTreeItem::typeChromeWrapper);
    }

    rv = baseWindow->Create();
    if (NS_FAILED(rv)) {
        mPrivate->logger->printf("engine", YLOG_FATAL,
                                 "NS_FAILED create baseWindow");
    }
    rv =baseWindow->SetVisibility(PR_TRUE);
    if (NS_FAILED(rv)) {
        mPrivate->logger->printf("engine", YLOG_FATAL,
                                 "NS_FAILED SetVisibility");
    }

    // register the progress listener
    nsCOMPtr<nsIWebProgressListener> listener = do_QueryInterface(mPrivate->chrome);
    nsCOMPtr<nsIWeakReference> thisListener(do_GetWeakReference(listener));
    mPrivate->webBrowser->AddWebBrowserListener(thisListener, NS_GET_IID(nsIWebProgressListener));

    mPrivate->webNavigation = do_QueryInterface(mPrivate->webBrowser);

    SetFocus(true);

    // make our console listener
    mPrivate->consoleListener = new ConsoleListener(this);

    nsCOMPtr<nsIConsoleService> consoleService =
	do_GetService(NS_CONSOLESERVICE_CONTRACTID, &rv);

    consoleService->RegisterListener(mPrivate->consoleListener);
#if 1
    nsCOMPtr<nsIDOMWindow> contentWin;
    nsresult rv_contentWin =
	mPrivate->webBrowser->GetContentDOMWindow(getter_AddRefs(contentWin));
    if (NS_FAILED(rv_contentWin))
    {
	mPrivate->logger->printf("engine", YLOG_FATAL,
                                 "NS_FAILED GetContentDOMWindow");
    }
#endif

    return 0;
}

CF_Logger *MozView::GetLogger()
{
    return mPrivate->logger;
}

nsresult MozView::GetMessages()
{
    nsresult rv;
    nsCOMPtr<nsIConsoleService> consoleService =
	do_GetService(NS_CONSOLESERVICE_CONTRACTID, &rv);

    if (NS_FAILED(rv)) {
        mPrivate->logger->printf("engine", YLOG_FATAL,
                                 "do_GetService console failed");
    }
    else
    {
	nsIConsoleMessage **messages;
	PRUint32 count;
        WRBUF w = wrbuf_alloc();

	consoleService->GetMessageArray(&messages, &count);
	for (PRUint32 i = 0; i < count; i++)
	{
	    nsString msg;
	    messages[i]->GetMessageMoz(getter_Copies(msg));

	    NS_ConvertUTF16toUTF8 retvalUtf8(msg);
	    if (true)
	    {
                wrbuf_write(w, retvalUtf8.get(), retvalUtf8.Length());
                mPrivate->logger->printf("message", YLOG_LOG,
                                         "%s", wrbuf_cstr(w));
                wrbuf_rewind(w);
	    }
	    NS_IF_RELEASE(messages[i]);
	}
	NS_Free(messages);
	consoleService->Reset();
        wrbuf_destroy(w);
    }
    return 0;
}

nsresult MozView::SetPositionAndSize(PRInt32 x, PRInt32 y, PRInt32 width, PRInt32 height)
{
    nsresult rv;
    nsCOMPtr<nsIBaseWindow> baseWindow;
    baseWindow = do_QueryInterface(mPrivate->webBrowser);
    rv = baseWindow->SetPositionAndSize(x, y, width, height, PR_TRUE);
    if (NS_FAILED(rv))
	return 1;
    else
	return 0;
}

nsresult MozView::LoadURI(const char* uri)
{
    nsresult rv;
    rv = mPrivate->webNavigation->LoadURI(NS_ConvertUTF8toUTF16(uri).get(),
					  nsIWebNavigation::LOAD_FLAGS_NONE, 0, 0, 0);
    return rv;
}

nsresult MozView::LoadData(const char    *base_url,
			   const char    *content_type,
			   const PRUint8 *data,
			   PRUint32       len)
{
    nsresult rv;

    nsCOMPtr<nsIWebBrowserStream> wbStream = do_QueryInterface(mPrivate->webBrowser);
    if (!wbStream)
	return NS_ERROR_FAILURE;

    nsCOMPtr<nsIURI> uri;
#ifdef XPCOM_GLUE
    // No nsNetUtil.h in in glue mode
    nsCOMPtr<nsIIOService> ioServ =
        do_GetService("@mozilla.org/network/io-service;1");
    rv = ioServ->NewURI(nsDependentCString(base_url), nsnull, nsnull,
                        getter_AddRefs(uri));
#else
    rv = NS_NewURI(getter_AddRefs(uri), base_url);
#endif
    if (NS_FAILED(rv))
	return rv;

    rv = wbStream->OpenStream(uri, nsDependentCString(content_type));
    if (NS_FAILED(rv))
	return rv;

    rv = wbStream->AppendToStream(data, len);
    if (NS_FAILED(rv))
	return rv;

    rv = wbStream->CloseStream();
    if (NS_FAILED(rv))
	return rv;

    return NS_OK;
}

nsresult MozView::Stop()
{
    nsresult rv;
    rv = mPrivate->webNavigation->Stop(nsIWebNavigation::STOP_ALL);
    return rv;
}

nsresult MozView::Reload()
{
    nsresult rv;
    rv = mPrivate->webNavigation->Reload(nsIWebNavigation::LOAD_FLAGS_NONE);
    return rv;
}

nsresult MozView::GoBack()
{
    nsresult rv;
    rv = mPrivate->webNavigation->GoBack();
    return rv;
}

nsresult MozView::GoForward()
{
    nsresult rv;
    rv = mPrivate->webNavigation->GoForward();
    return rv;
}

PRBool MozView::CanGoBack()
{
    CF_BOOL allowBack;
    mPrivate->webNavigation->GetCanGoBack(&allowBack);
    return allowBack;
}

PRBool MozView::CanGoForward()
{
    CF_BOOL allowForward;
    mPrivate->webNavigation->GetCanGoForward(&allowForward);
    return allowForward;
}

nsresult MozView::SetFocus(PRBool focus)
{
    nsCOMPtr<nsIWebBrowserFocus> browserFocus;
    browserFocus = do_QueryInterface(mPrivate->webBrowser);
    if(focus)
	browserFocus->Activate();
    else
	browserFocus->Deactivate();
    return NS_OK;
}

void MozView::Show()
{
    nsCOMPtr<nsIBaseWindow> baseWindow;
    baseWindow = do_QueryInterface(mPrivate->webBrowser);
    baseWindow->SetVisibility(PR_TRUE);
}

void MozView::Hide()
{
    nsCOMPtr<nsIBaseWindow> baseWindow;
    baseWindow = do_QueryInterface(mPrivate->webBrowser);
    baseWindow->SetVisibility(PR_FALSE);
}

void MozView::SetListener(MozViewListener *pNewListener)
{
    mPrivate->pListener = pNewListener;
    mPrivate->pListener->SetMozView(this);
}

MozViewListener* MozView::GetListener()
{
    return mPrivate->pListener;
}

void* MozView::GetParentWindow()
{
    return mPrivate->parentWindow;
}

void* MozView::GetNativeWindow()
{
    nsCOMPtr<nsIBaseWindow> baseWindow = do_QueryInterface(mPrivate->webBrowser);
    nsCOMPtr<nsIWidget> mozWidget;

    if (NS_SUCCEEDED(baseWindow->GetMainWidget(getter_AddRefs(mozWidget)))) {
	return mozWidget->GetNativeData(NS_NATIVE_WINDOW);
    } else {
	return NULL;
    }
}

void MozView::SetParentView(MozView* parent)
{
    mPrivate->parentView = parent;
}

MozView* MozView::GetParentView()
{
    return mPrivate->parentView;
}

nsresult MozView::GetInterfaceRequestor(nsIInterfaceRequestor** aRequestor)
{
    NS_ENSURE_ARG_POINTER(aRequestor);
    return CallQueryInterface(mPrivate->webBrowser, aRequestor);
}

// ---- MozViewListener ---
MozViewListener::MozViewListener()
  : pMozView(NULL)
{
}

MozViewListener::~MozViewListener()
{
}

void MozViewListener::SetTitle(const char *newTitle)
{
}

void MozViewListener::StatusChanged(const char* newStatus, PRUint32 statusType)
{
}

void MozViewListener::OnStatusChange(const char* newStatus)
{
}

void MozViewListener::LocationChanged(const char* newLocation)
{
}

PRBool MozViewListener::OpenURI(const char* newLocation)
{
    return false;
}

void MozViewListener::DocumentLoaded(nsIDOMWindow *dom_window)
{
}

void MozViewListener::OnStateChange(nsIDOMWindow *dom_window,
				    PRUint32 aStateFlags)
{
}

void MozViewListener::SetMozView(MozView *pAMozView)
{
    pMozView = pAMozView;
}

MozView* MozViewListener::OpenWindow(PRUint32 flags)
{
    return 0;
}

void MozViewListener::SizeTo(PRUint32 width, PRUint32 height)
{
}

void MozViewListener::SetVisibility(PRBool visible)
{
}

void MozViewListener::StartModal()
{
}

void MozViewListener::ExitModal(nsresult result)
{
}

void MozViewListener::ConsoleMessage(const char *msg)
{
}

/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */
