/* This file is part of the Connector Framework
 * Copyright (C) 2008-2013 Index Data
 * See the file LICENSE for details.
 */

/** \file
 * \brief RPN to CF queries (JSON based)
 *
 * Flattens a RPN query to simple keyword-value pair(s)
 * with a bit of trickery for dates etc.
 * Also includes a routine 'parse_full' to translate the whole
 * RPN query into corresponding JSON.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <yaz/json.h>
#include <yaz/xmalloc.h>
#include <yaz/yaz-iconv.h>
#include <yaz/diagbib1.h>
#include "type7_sort.h"

#include "rpn_to_json.h"

/** Query mapping */
struct querymap {
    const char *jsonfield;
    int attrvalues[10]; /**< zero-terminated list of attr1 values */
    /* looks like the compiler wants a size here */
};

static struct querymap qmap[] = {
    /* 0 */ { "keyword", { 1035, 1016, 0 } }, /* defaults to the first */
    /* 1 */ { "year", { 30,31,0 } },  /* The very special single year thing */
    /* 2 */ { "startyear", { 30, 31, 0 } }, /* dates are special! */
    /* 3 */ { "endyear", { 30, 31, 0 } },
    /* 4 */ { "author", { 1, 1003, 0 } },
    /* 5 */ { "title", { 4, 5, 6, 0 } },
    /* 6 */ { "OCLCNum", { 1211, 0 } },
    /* 7 */ { "localID", { 12, 0 } },
    /* 8 */ { "subject", { 21, 47, 0 } }
};

const int n_qmaps = (sizeof(qmap)/sizeof(*qmap));

const int singleyearidx = 1;
const int startyearidx = 2;
const int endyearidx = 3;

struct queryvalue {
    WRBUF paramname;
    WRBUF value;
};

const int max_queryvalues=50; /* How many different indexes in a query */
/* TODO Hard-coded limit. Should be enough, since we just append if we */
/* see the same index again. Still, it is ugly */

struct queryvalues {
    struct queryvalue v[max_queryvalues];
};

// we may have up to two query results due to single year case
const int json_results_max = 2;

CF_RPN_to_JSON::CF_RPN_to_JSON()
{
    int i;
    json_results = (WRBUF *) xmalloc(sizeof(*json_results) * json_results_max);
    for (i = 0; i < json_results_max; i++)
        json_results[i] =  wrbuf_alloc();
    diag = 0;
    diag_w = wrbuf_alloc();
    full_w = wrbuf_alloc();
    capability_flags = wrbuf_alloc();
}

CF_RPN_to_JSON::~CF_RPN_to_JSON()
{
    int i;
    for (i = 0; i < json_results_max; i++)
        wrbuf_destroy(json_results[i]);
    xfree(json_results);
    wrbuf_destroy(diag_w);
    wrbuf_destroy(full_w);
    wrbuf_destroy(capability_flags);
}

void CF_RPN_to_JSON::set_capability_flags(const char *s)
{
    wrbuf_rewind(capability_flags);
    if (s)
    {
        wrbuf_puts(capability_flags, " ");
        wrbuf_puts_replace_char(capability_flags, s, ',', ' ');
        wrbuf_puts(capability_flags, " ");
    }
}

/** Find the attribute specified, and return its (numerical) value */
int CF_RPN_to_JSON::getattrvalue(const Z_AttributesPlusTerm *zapt, int attrtype)
{
    Z_AttributeList *attrlist = zapt->attributes;
    int attrval = 0;
    for (int i = 0; i < attrlist->num_attributes; i++)
    {
        if (attrlist->attributes[i]->which == Z_AttributeValue_numeric &&
            *(attrlist->attributes[i]->attributeType) == attrtype)
        {
            attrval = *(attrlist->attributes[i]->value.numeric);
            /* printf("SIMPLE: Found attr1: %d \n", attr1); */
            break;
        }
    }
    return attrval;
}

/** Find the attribute specified, and return its (string) value */
char *CF_RPN_to_JSON::getstringattr(const Z_AttributesPlusTerm *zapt,
                                    int attrtype)
{
    Z_AttributeList *attrlist = zapt->attributes;
    for (int i = 0; i < attrlist->num_attributes; i++)
    {
        if (attrlist->attributes[i]->which == Z_AttributeValue_complex &&
            *attrlist->attributes[i]->attributeType == attrtype)
        {
            struct Z_ComplexAttribute *ca =
                attrlist->attributes[i]->value.complex;
            for (int c = 0; c < ca->num_list; c++)
                if (ca->list[c]->which == Z_StringOrNumeric_string)
                    return ca->list[c]->u.string;
            break;
        }
    }
    return 0;
}

/** Find the string attribute in queryvalues.
    If not there, add it. If too many, return -1. */
int CF_RPN_to_JSON::findstringattridx(const char *strattr,
                                      struct queryvalues *qv)
{
    for (int i = 0; i < max_queryvalues; i++)
        if (qv->v[i].paramname == 0)
        { // not found, create it
            qv->v[i].paramname = wrbuf_alloc();
            wrbuf_puts(qv->v[i].paramname, strattr);
            return i;
        }
        else
        {
            if (strcmp(wrbuf_cstr(qv->v[i].paramname), strattr) == 0)
                return i; /* found it */
        }
    return -1; /* table full, since we didn't return above */
}


/** Look at the attributes and get the matching index number
    That is, the index into qmap and qvalues. Defaults to 0. */
int CF_RPN_to_JSON::getindex(const Z_AttributesPlusTerm *zapt,
                             struct queryvalues *qv)
{
    int attr1 = getattrvalue(zapt, 1);
    if (attr1 == 0)
    { /* no (numeric) attr1 found, try string */
        char *strval = getstringattr(zapt, 1);
        if (strval == 0) /* no strign found either */
            return 0; /* default to the first in qmap, keyword */
        int idx = findstringattridx(strval, qv);
        if (idx == -1)
        {
            diag = YAZ_BIB1_TOO_MANY_INDEX_TERMS_PROCESSED;
            wrbuf_printf(diag_w, "%d", attr1);
            return -1;
        }
        return idx;
    }
    /* numeric attribute found, find its idx */
    for (int idx = 0; idx < n_qmaps; idx++)
        for (int a = 0; qmap[idx].attrvalues[a] != 0; a++)
            if (qmap[idx].attrvalues[a] == attr1)
                return idx;
    /* did not find it */
    diag = YAZ_BIB1_UNSUPP_USE_ATTRIBUTE;
    wrbuf_printf(diag_w, "%d", attr1);
    return -1;
}


/** helper function to set a year in the right field */
bool CF_RPN_to_JSON::set_date(struct queryvalues *qv, int idx, int year)
{
    if (qv->v[idx].value == 0)
    {
        qv->v[idx].value = wrbuf_alloc();
    }
    else
    { /* can't have multiple start or end years */
        diag = YAZ_BIB1_UNSUPP_SEARCH;
        wrbuf_puts(diag_w, "Too many start/end dates");
        return false;
    }
    wrbuf_printf(qv->v[idx].value, "%d", year);
    return true;
}


bool CF_RPN_to_JSON::date_trickery(const Z_AttributesPlusTerm *zapt,
                                   struct queryvalues *qv,
                                   const char *term)
{
    int attr4 = getattrvalue(zapt, 4);  /* structure attribute */
    if (attr4 != 0 &&  /* not specified */
        attr4 != 1 && attr4 != 2 &&   /* word, phrase */
        attr4 != 4 && attr4 != 3 && attr4 != 109)
    { /* year, numeric string */
        diag = YAZ_BIB1_UNSUPP_STRUCTURE_ATTRIBUTE;
        wrbuf_printf(diag_w, "%d", attr4);
        return false;
    }
    int attr2 = getattrvalue(zapt, 2);  /* relation */
    if (attr2 == 0)
        attr2 = 3; /* default to equals */
    int year = atoi(term);
    if (year < 1 || year > 9999)
    {
        diag = YAZ_BIB1_UNSUPP_VALUE_FOR_USE_ATTRIBUTE;
        wrbuf_puts(diag_w,"Bad year");
        return false;
    }
    /* TODO: extract any 4-digit sequence from it, or something */
    switch (attr2)
    {
    case 1: /* less than */
        return set_date(qv, endyearidx, year-1);
    case 2: /* less than or equal */
        return set_date(qv, endyearidx, year);
    case 3: /* equal */
        return
            set_date(qv, startyearidx, year) &&
            set_date(qv, endyearidx, year);
    case 4: /* greater or equal */
        return set_date(qv, startyearidx, year);
    case 5: /* greater than */
        return set_date(qv, startyearidx, year+1);
    default:
        diag = YAZ_BIB1_UNSUPP_RELATION_ATTRIBUTE;
        wrbuf_printf(diag_w, "%d", attr2);
        return false;
    }
    return true;
}

char * CF_RPN_to_JSON::stringterm(const Z_AttributesPlusTerm *zapt)
{
    char *t = 0;
    if (zapt->term->which == Z_Term_general)
        t = (char *)(zapt->term->u.general->buf);
    else if (zapt->term->which == Z_Term_characterString)
        t = (char *)(zapt->term->u.characterString);
    return t;
}

bool CF_RPN_to_JSON::apt(const Z_AttributesPlusTerm *zapt,
                         struct queryvalues *qv)
// TODO Check for unsupported attributes, and return more errors!
{
    char *term = stringterm(zapt);
    if (!term)
    {
        diag = YAZ_BIB1_QUERY_TYPE_UNSUPP;
        return false;
    }
    int idx = getindex(zapt,qv);
    if (idx < 0) // it has set its error codes
        return false;

    // Check attributes that are always unsupported
    int attr3 = getattrvalue(zapt, 3);
    if (attr3 != 0 && attr3 != 3)
    {
        diag = YAZ_BIB1_UNSUPP_POSITION_ATTRIBUTE;
        wrbuf_printf(diag_w, "%d", attr3);
        return false;
    }

    int attr6 = getattrvalue(zapt, 6);
    if (attr6 != 0 && attr6 != 1)
    {
        diag = YAZ_BIB1_UNSUPP_COMPLETENESS_ATTRIBUTE;
        wrbuf_printf(diag_w, "%d", attr6);
        return false;
    }

    int attr5 = getattrvalue(zapt, 5);
    if (idx == singleyearidx)
    {  // It checks its own structure attrs
        if (attr5 != 0 && attr5 != 100)
        {
            diag = YAZ_BIB1_UNSUPP_TRUNCATION_ATTRIBUTE;
            wrbuf_printf(diag_w, "%d", attr5);
            return false;
        }
        return date_trickery(zapt, qv, term);
    }

    // Check remaining attributes (only for string fields)
    int attr2 = getattrvalue(zapt, 2); // relation
    if (attr2 != 0 && attr2 != 3 && attr2 != 102)
    { // unspec, equal, relev
        diag = YAZ_BIB1_UNSUPP_RELATION_ATTRIBUTE;
        wrbuf_printf(diag_w, "%d", attr2);
        return false;
    }
    if (qv->v[idx].value == 0)
        qv->v[idx].value = wrbuf_alloc();
    else
        wrbuf_puts(qv->v[idx].value, " ");
    int attr4 = getattrvalue(zapt, 4); // structure attribute
    if (attr4 == 1) // phrase - put quotes around it
    {
        wrbuf_puts(qv->v[idx].value, "\\\"");  // escaped quote
        wrbuf_puts(qv->v[idx].value, term);
        wrbuf_puts(qv->v[idx].value, "\\\"");
    }
    else if (attr4 == 0 || attr4 == 2 || attr4 == 3)  // unspec, or word
        wrbuf_puts(qv->v[idx].value, term);
    else
    {
        diag = YAZ_BIB1_UNSUPP_STRUCTURE_ATTRIBUTE;
        wrbuf_printf(diag_w, "%d", attr4);
        return false;
    }
    char trunc_str[2];
    *trunc_str = '\0';
    if (strstr(wrbuf_cstr(capability_flags), " trunc-questionmark "))
        strcpy(trunc_str, "?");
    else if (strstr(wrbuf_cstr(capability_flags), " trunc-asterisk "))
        strcpy(trunc_str, "*");

    switch (attr5)
    {
    case 0:
    case 100:
        break;
    case 1:
        if (*trunc_str &&
            strstr(wrbuf_cstr(capability_flags), " trunc-right ") )
        {
            wrbuf_puts(qv->v[idx].value, trunc_str);
        }
        else
        {
            diag = YAZ_BIB1_UNSUPP_TRUNCATION_ATTRIBUTE;
            wrbuf_printf(diag_w, "%d", attr5);
            return false;
        }
        break;
    case 2:
        if (*trunc_str &&
            strstr(wrbuf_cstr(capability_flags), " trunc-left ") )
        {
            wrbuf_insert(qv->v[idx].value, 0, trunc_str, 1);
        }
        else
        {
            diag = YAZ_BIB1_UNSUPP_TRUNCATION_ATTRIBUTE;
            wrbuf_printf(diag_w, "%d", attr5);
            return false;
        }
        break;
    case 3:
        if (*trunc_str &&
            strstr(wrbuf_cstr(capability_flags), " trunc-both ") )
        {
            wrbuf_insert(qv->v[idx].value, 0, trunc_str, 1);
            wrbuf_puts(qv->v[idx].value, trunc_str);
        }
        else
        {
            diag = YAZ_BIB1_UNSUPP_TRUNCATION_ATTRIBUTE;
            wrbuf_printf(diag_w, "%d", attr5);
            return false;
        }
        break;
    case 104: // z39.58 style
        if (index(term,'#'))
        {
            // TODO - We don't support '#' for single-char wildcard,
            // but what should we fail with ??
            diag = YAZ_BIB1_UNSUPP_SEARCH;
            wrbuf_puts(diag_w, "Single character masking not supported");
            return false;
        }
        if ( trunc_str[0] &&
             strstr(wrbuf_cstr(capability_flags), " query-wildcard ") )
        {
            for (size_t i = 0; i < wrbuf_len(qv->v[idx].value); i++)
                if (wrbuf_buf(qv->v[idx].value)[i] == '?')
                    wrbuf_buf(qv->v[idx].value)[i] = trunc_str[0];
        }
        else
        {
            diag = YAZ_BIB1_UNSUPP_TRUNCATION_ATTRIBUTE;
            wrbuf_printf(diag_w, "%d", attr5);
            return false;
        }
        break;
    default:
        diag = YAZ_BIB1_UNSUPP_TRUNCATION_ATTRIBUTE;
        wrbuf_printf(diag_w, "%d", attr5);
        return false;
    }
    return true;
}

bool CF_RPN_to_JSON::rpnstructure(const Z_RPNStructure *zs,
                                  struct queryvalues *qv)
{
    if (zs->which == Z_RPNStructure_complex)
    {
        Z_Complex *zc = zs->u.complex;
        if (!zc->roperator)
        {
            diag = YAZ_BIB1_OPERATOR_UNSUPP;
            return false;
        }
        switch (zc->roperator->which)
        {
        case Z_Operator_and:
            return rpnstructure(zc->s1, qv) && rpnstructure(zc->s2, qv);
        case Z_Operator_or:
            wrbuf_puts(diag_w, "or");
            diag = YAZ_BIB1_OPERATOR_UNSUPP;
            return false;
        case Z_Operator_and_not:
            wrbuf_puts(diag_w, "not");
            diag = YAZ_BIB1_OPERATOR_UNSUPP;
            return false;
        case Z_Operator_prox:
            wrbuf_puts(diag_w, "prox");
            diag = YAZ_BIB1_OPERATOR_UNSUPP;
            return false;
        default:
            diag = YAZ_BIB1_OPERATOR_UNSUPP;
            return false;
        }
    }
    else if (zs->which == Z_RPNStructure_simple)
    {
        if (zs->u.simple->which == Z_Operand_APT)
            return apt(zs->u.simple->u.attributesPlusTerm, qv);
        else
        {
            // no references to result sets
            diag = YAZ_BIB1_RESULT_SET_UNSUPP_AS_A_SEARCH_TERM;
            return false;
        }
    }
    diag = YAZ_BIB1_UNSUPP_SEARCH;
    return false;
}

void CF_RPN_to_JSON::database(const char *sub_database, WRBUF w)
{
    if (sub_database)
    {
        wrbuf_puts(w, "\"database\":\"");
        wrbuf_puts(w, sub_database);
        wrbuf_puts(w, "\",");
    }
}

void CF_RPN_to_JSON::build_query(struct queryvalues *qv,
                                 const char *sub_database,
                                 WRBUF w)
{
    const char *comma = "";
    wrbuf_puts(w, "{");
    database(sub_database, w);
    for (int i = 0; i < max_queryvalues; i++)
    {
        if (qv->v[i].value != 0)
        {
            wrbuf_puts(w, comma);
            wrbuf_puts(w, "\"");
            wrbuf_puts(w, wrbuf_cstr(qv->v[i].paramname) );
            wrbuf_puts(w, "\":\"");
            wrbuf_json_puts(w, wrbuf_cstr(qv->v[i].value) );
            wrbuf_puts(w, "\"");
            comma = ",";
        }
    }
    wrbuf_puts(w, "}");
}

// Write the sortkey parameter into a wrbuf
void write_sortkey(wrbuf *w, struct sort_elem *sort_list)
{
    for (; sort_list; sort_list = sort_list->next)
    {
        wrbuf_json_puts(w, sort_list->field);
        if (sort_list->flags[0])
        {
            wrbuf_puts(w, "-");
            wrbuf_json_puts(w, sort_list->flags);
        }
        if (sort_list->next)
            wrbuf_puts(w, " ");
    }
}

bool CF_RPN_to_JSON::parse(Z_RPNQuery *q, const char *sub_database,
                           struct sort_elem *sort_list)
{
    int i;
    for (i = 0; i < json_results_max; i++)
        wrbuf_rewind(json_results[i]);
    wrbuf_rewind(diag_w);

    struct queryvalues qv;
    for (i = 0; i < n_qmaps; i++)
    {
        qv.v[i].value = 0;
        qv.v[i].paramname = wrbuf_alloc();
        wrbuf_puts(qv.v[i].paramname, qmap[i].jsonfield);
    }
    for ( ; i < max_queryvalues; i++)
    {
        qv.v[i].value = 0;
        qv.v[i].paramname = 0;
    }

    // TODO - Set the first names from qmaps

    // Extract the real search terms
    if (!rpnstructure(q->RPNStructure, &qv))
        return false;

    // Add sort stuff into qv
    if (sort_list)
    {
        int idx = findstringattridx("sortkey", &qv);
        if (qv.v[idx].value == 0)
            // should always be.
            qv.v[idx].value = wrbuf_alloc();
        else
            wrbuf_puts(qv.v[idx].value, " ");
        write_sortkey( qv.v[idx].value, sort_list);
    }

    /* Date magic, part 1: If we have only one year,
     * pass it first in the single 'year' argument */
    if (( qv.v[startyearidx].value != 0) &&
        ( qv.v[endyearidx].value != 0) &&
        ( strcmp( wrbuf_cstr(qv.v[startyearidx].value),
                  wrbuf_cstr(qv.v[endyearidx].value)) == 0))
    {
        // full date range as second..
        build_query(&qv, sub_database, json_results[1]);

        // everything to singleyear..
        qv.v[singleyearidx].value = qv.v[startyearidx].value; // transfer
        qv.v[startyearidx].value = 0;

        wrbuf_destroy(qv.v[endyearidx].value);  // delete
        qv.v[endyearidx].value = 0;
    }
    // normal query including single date
    build_query(&qv, sub_database, json_results[0]);
    for (i = 0; i < max_queryvalues; i++)
    {
        wrbuf_destroy(qv.v[i].value);
        wrbuf_destroy(qv.v[i].paramname);
    }
    return true;
}

const char *CF_RPN_to_JSON::get_result(int no)
{
    if (no >= json_results_max)
        return 0;
    if (wrbuf_len(json_results[no]) == 0)
        return 0;
    return wrbuf_cstr(json_results[no]);
}

const char *CF_RPN_to_JSON::get_full()
{
    if (wrbuf_len(full_w) == 0)
        return 0;
    return wrbuf_cstr(full_w);
}

int CF_RPN_to_JSON::get_diagnostic(const char **addinfo)
{
    if (addinfo)
    {
        if (wrbuf_len(diag_w))
            *addinfo = wrbuf_cstr(diag_w);
        else
            *addinfo = 0;
    }
    return diag;
}

//////////// parse_full stuff

const int n_attrtypes = 6;

struct attrmap_t {
    int attrvalue;
    const char *string;
};

struct attrinfo_t {
    const char *attrname;
    const attrmap_t values[20]; // We won't have more than that!
};

const struct attrinfo_t attributes[] = {
    {  "", { {0,0} } }, /* unused, to get base-1 array */
    {  "field",
       { {0,0} }  /* we do not support numerical use attributes */
    },
    {  "relation",
       { {1, "lt" },
         {2, "le" },
         {3, "eq" },
         {4, "ge" },
         {5, "gt" },
         {6, "ne" },
         {100, "phonetic" },
         {101, "stem" },
         {102, "relevance" },
         {103, "alwaysmatches" },
         {0,0} }
    },
    {  "position",
       { {1, "firstinfield" },
         {2, "firstinsubfield" },
         {3, "any" },
         {0,0} }
    },
    {  "structure",
       { {1, "phrase" },
         {2, "word" },
         {3, "key" },
         {4, "year" },
         {5, "datenormalized" },
         {6, "wordlist" },
         {100, "dateunnormalized" },
         {101, "namenormalized" },
         {102, "nameunnormalized" },
         {103, "structure" },
         {104, "urx" },
         {105, "freeformtext" },
         {106, "documenttext" },
         {107, "localnumber" },
         {108, "string" },
         {109, "numericstring" },
         {0,0} }
    },
    {  "truncation",
       { {1, "right" },
         {2, "left" },
         {3, "both" },
         {100, "not" },
         {101, "masking" },
         {102, "regexp1" },
         {103, "regexp2" },
         {104, "z39.58" },
         {0,0} }
    },
    {  "completeness",
       { {1, "incompletesubfield" },
         {2, "completesubfield" },
         {3, "completefield" },
         {0,0} }
    }
};

bool CF_RPN_to_JSON::full_apt(const Z_AttributesPlusTerm *zapt)
{
    char *term = stringterm(zapt);
    if (!term)
    {
        diag = YAZ_BIB1_QUERY_TYPE_UNSUPP;
        return false;
    }
    wrbuf_puts(full_w, "{\"term\":\"");
    wrbuf_json_puts(full_w, term);
    wrbuf_puts(full_w, "\"");
    Z_AttributeList *attrlist = zapt->attributes;
    for (int i = 0; i < attrlist->num_attributes; i++)
    {
        int attrtype = *(attrlist->attributes[i]->attributeType);
        if (attrtype > n_attrtypes)
        {
            diag = YAZ_BIB1_UNSUPP_ATTRIBUTE_TYPE;
            return false;
        }
        wrbuf_printf(full_w, ",\"%s\":",
                     attributes[attrtype].attrname);
        if (attrlist->attributes[i]->which == Z_AttributeValue_numeric)
        {
            int attrval = *(attrlist->attributes[i]->value.numeric);
            const struct attrmap_t *am = attributes[attrtype].values;
            while (am->attrvalue)
            {
                if (am->attrvalue == attrval)
                {
                    wrbuf_printf(full_w, "\"%s\"", am->string);
                    break;
                }
                am++;
            }
            if (! am->attrvalue)
                wrbuf_printf(full_w, "\"%d\"", attrval);
        }
        else if (attrlist->attributes[i]->which == Z_AttributeValue_complex)
        {
            struct Z_ComplexAttribute *ca =
                attrlist->attributes[i]->value.complex;
            if (ca->list[0]->which == Z_StringOrNumeric_string)
            { // we happily use just the first string.
                wrbuf_printf(full_w, "\"%s\"", ca->list[0]->u.string);
            }
            else
            {
                diag = YAZ_BIB1_TERM_TYPE_UNSUPP;
                return false;
            }
        }
    }
    wrbuf_puts(full_w, "}");
    return true;
}

bool CF_RPN_to_JSON::full_proximity(const Z_ProximityOperator *zp)
{
    const char *boolvalues[]={"false","true"};
    const char *opnames[] = { "", "lt", "le", "eq", "ge", "gt", "ne" };
    const char *units[] = {"", "character", "word", "sentence",
                           "paragraph", "section", "chapter", "document",
                           "element", "subelement", "elementType", "byte"};
    if ( zp->exclusion )
        wrbuf_printf(full_w, "\"exclusion\":%s,",
                     boolvalues[*zp->exclusion] );
    wrbuf_printf(full_w, "\"distance\":" ODR_INT_PRINTF ",", *zp->distance );
    wrbuf_printf(full_w, "\"ordered\":%s,", boolvalues[*zp->ordered] );
    if ( *zp->relationType > 6 || *zp->relationType < 1 ) {
        diag = YAZ_BIB1_UNSUPP_PROX_RELATION;
        wrbuf_printf(diag_w, ODR_INT_PRINTF, *zp->relationType );
        return false;
    }
    wrbuf_printf(full_w, "\"relation\":\"%s\",",
                 opnames[*zp->relationType] );
    if ( zp->which != Z_ProximityOperator_known ) {
        diag = YAZ_BIB1_UNSUPP_SEARCH;
        wrbuf_puts(diag_w, "Private proximity operators not supported");
        return false;
    }
    if ( *zp->u.known < 1 || *zp->u.known > 11 ) {
        diag = YAZ_BIB1_UNSUPP_PROX_UNIT_CODE;
        wrbuf_printf(diag_w, ODR_INT_PRINTF, *zp->u.known );
        return false;
    }
    wrbuf_printf(full_w, "\"unit\":\"%s\",",
                 units[*zp->u.known] );
    return true;
}

bool CF_RPN_to_JSON::full_rpnstructure(const Z_RPNStructure *zs)
{
    if (zs->which == Z_RPNStructure_complex)
    {
        Z_Complex *zc = zs->u.complex;
        if (!zc->roperator)
        {
            diag = YAZ_BIB1_OPERATOR_UNSUPP;
            return false;
        }
        wrbuf_puts(full_w, "{\"op\":");
        switch (zc->roperator->which)
        {
        case Z_Operator_and:
            wrbuf_puts(full_w, "\"and\",");
            break;
        case Z_Operator_or:
            wrbuf_puts(full_w, "\"or\",");
            break;
        case Z_Operator_and_not:
            wrbuf_puts(full_w, "\"not\",");
            break;
        case Z_Operator_prox:
            wrbuf_puts(full_w, "\"prox\",");
            if ( ! full_proximity(zc->roperator->u.prox) )
                return false;
            break;
            diag = YAZ_BIB1_OPERATOR_UNSUPP;
            wrbuf_puts(diag_w, "prox");
            return false;
        default:
            diag = YAZ_BIB1_OPERATOR_UNSUPP;
            return false;
        }

        wrbuf_puts(full_w, "\"s1\":");
        if (! full_rpnstructure(zc->s1) )
            return false;
        wrbuf_puts(full_w, ",\"s2\":");
        if (! full_rpnstructure(zc->s2) )
            return false;
        wrbuf_puts(full_w, "}");
        return true;
    }
    else if (zs->which == Z_RPNStructure_simple)
    {
        if (zs->u.simple->which != Z_Operand_APT)
        {
            // no references to result sets
            diag = YAZ_BIB1_RESULT_SET_UNSUPP_AS_A_SEARCH_TERM;
            return false;
        }
        else
            return full_apt(zs->u.simple->u.attributesPlusTerm);
    }
    diag = YAZ_BIB1_UNSUPP_SEARCH;
    return false;
}

// A second converter that returns the whole query tree in json to be
// used in the fullquery parameter. Get the result with get_full
bool CF_RPN_to_JSON::parse_full(Z_RPNQuery *q,
                                const char *sub_database,
                                struct sort_elem *sort_list)
{
    wrbuf_rewind(diag_w);
    wrbuf_rewind(full_w);
    diag = 0;
    wrbuf_puts(full_w, "{");
    database(sub_database, full_w);
    wrbuf_puts(full_w, "\"fullquery\":");
    if (!q || !q->RPNStructure)
    {
        // Defensive coding. Can happen that there is nothing
        // left after we have removed the sort stuff
        diag = YAZ_BIB1_UNSUPP_SEARCH;
        return false;
    }
    if (!full_rpnstructure(q->RPNStructure))
        return false;
    if (sort_list)
    {
        wrbuf_puts(full_w, ",\"sortkey\":\"");
        write_sortkey(full_w, sort_list);
        wrbuf_puts(full_w, "\"");
    }
    wrbuf_puts(full_w, "}");
    return true;
}

/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */
