var EXPORTED_SYMBOLS = ["ConnectorTemplate"];
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/runtime/TaskTemplate.js');

var ConnectorTemplate = function () {
  // object properties do not guarantee ordering
  this.taskTemplates = [];
  this.taskTemplatesByName = {};
  this.requirements = {};
  this.properties = {};
  this.errorCodes = [];
  this.metaData = null;
  this.capabilityFlags = [];  // array of objects
};

ConnectorTemplate.prototype = {
  loadFromFile: function (file) {
    this.load(xmlHelper.openDoc(file.path));
  },

  load: function (ctNode) {
    this.taskTemplates = [];
    this.requirements = {};

    if (ctNode.getElementsByTagName("connectorTemplate").length != 1)
      throw new Error("file is not a proper connector template.");

    // load metadata
    this.metaData = [];
    var metaSeen = {};
    var mdContainerNodes = ctNode.getElementsByTagName("metaData");
    if (mdContainerNodes.length < 1) {
      throw new Error("No metadata prototypes found!");
    } else {
      if (mdContainerNodes.length > 1)
        dump("Multiple metadata elements found, using first...");
      var metaNodes = mdContainerNodes[0].getElementsByTagName("meta");
      for (var i=0; i<metaNodes.length; i++) {
        var attrs = {};
        for (var j=0; j<metaNodes[i].attributes.length; j++) {
          var attr = metaNodes[i].attributes[j];
          attrs[attr.name] = attr.value;
        }
        if (!metaSeen[attrs.name]) this.metaData.push(attrs);
        metaSeen[attrs.name] = true;
      }
    }

    // load capability flags
    this.capabilityFlags = [];
    var capSeen = {};
    var capContainerNodes = ctNode.getElementsByTagName("capabilityFlags");
    if (capContainerNodes.length < 1) {
        //throw new Error("No cpabilityFlags prototypes found!");
        // Don't make it a serious error quite yet! Empty set is acceptable.
        dump("No capability flags found in template");
    }
    for ( var capCont = 0; capCont < capContainerNodes.length; capCont++ ){
        var capNodes = capContainerNodes[capCont].getElementsByTagName("flag");
        for (var i=0; i<capNodes.length; i++) {
            var attrs = {};
            for (var j=0; j<capNodes[i].attributes.length; j++) {
                var attr = capNodes[i].attributes[j];
                attrs[attr.name] = attr.value;
                // dump("Loading template capability " +
                //   attr.name + " " + attr.value + "\n");
            }
            if (capSeen[attrs.name]) {
                dump("Duplicate capability flag '" + attrs.name +
                     "'. Using first\n");
            } else {
                // dump("Capability flag " + attrs.name + "\n");
                this.capabilityFlags.push(attrs);
            }
            capSeen[attrs.name] = true;
        }
    }

    // load properties
    this.properties = {};
    var propNodes = ctNode.getElementsByTagName("property");
    for (var i=0; i<propNodes.length; i++) {
      this.properties[propNodes[i].getAttribute('name')] = 
        propNodes[i].getAttribute('value');
    }

    // load tasks
    var taskNodes = ctNode.getElementsByTagName("task");
    for (var i=0; i<taskNodes.length; i++) {
      var newTaskTemplate = new TaskTemplate();
      newTaskTemplate.load(taskNodes[i]);
      newTaskTemplate.rank = i;
      this.taskTemplates.push(newTaskTemplate);
      this.taskTemplatesByName[newTaskTemplate.name] = newTaskTemplate;
    }

    // load properties
    this.errorCodes = [];
    var errCodes = ctNode.getElementsByTagName("errorCode");
    for (var i=0; i<errCodes.length; i++) {
      let name =  errCodes[i].getAttribute('name')
        || errCodes[i].getAttribute('value');
      let errCode = {
        key: name,
        value: errCodes[i].getAttribute('value')
      };
      this.errorCodes.push(errCode);
    }

    // load requirements
    var reqNodes = ctNode.getElementsByTagName("requirement");
    for (var i=0; i<reqNodes.length; i++) {
      if (reqNodes[i].hasAttribute('task')) {
        var taskName = reqNodes[i].getAttribute('task');
        if (!this.requirements[taskName])
          this.requirements[taskName] = [];
        var argNodes = reqNodes[i].getElementsByTagName("argument");
        var args = [];
        for (var j=0; j<argNodes.length; j++) {
          args.push(argNodes[j].getAttribute('name'))
        }
        this.requirements[taskName].push(args);
      }
    }
  },

  getTaskTemplates: function () {
    return this.taskTemplates;
  },

  getTaskTemplate: function (taskName) {
    return this.taskTemplatesByName[taskName];
  },

  getRequirements: function (taskName) {
    return this.requirements[taskName] || [];
  },

  getMetaData: function () {
    return this.metaData;
  },

  getCapabilityFlags: function () {
      return this.capabilityFlags;
  },

};
