var EXPORTED_SYMBOLS = ["FlatQuery"];

// Flatquery is another representation of a complex query. It is
// simpler than a real fullquery, in that it is only a linear list
// of terms, combined with simple boolean operators. No nesting,
// simple left-to-right precedence. This makes it a better fit
// for various advanced query forms, url-building, and even some
// XML queries, as well as easier to access from script steps.
// All this comes with a cost: There are queries that can simply
// not be expressed as a flatquery. It is our hope that most real-
// life queries will work.

// The Flatquery class is actually just one element in the flattened
// query; the whole query is an array of FlatQuery objects.


// TODO - Refactor with fullquery, so that we have one
// queryterm class we can use in both! and extend here with the extras

// TODO - Refactor common utilities to the queryHelper module!

Components.utils.import('resource://indexdata/util/logging.js');
Components.utils.import('resource://indexdata/runtime/StepError.js');

var logger = logging.getLogger('');
var debug = false;
// debug = true;

// List of fields to be copied from fullquery terms
// and at other places.
var fqfields = [ "term", "field", "relation", "position",
                 "structure", "truncation", "completenss" ];


// flatquery constructor
// We only need to create objects for the unittest.
// Everything else is class-level functions, like in many
// other helpers.
function FlatQuery() {
  this.term = "";
}

// Make a flatquery element
// takes a fullquery term, or another object with the right fields
// everything has a default, so nothing is really needed. Can be called
// without anything at all, although a term is usually handy.
// Likewise, the operator may be left emtpy, to be filled in later
function FlatQueryElement( fqterm, operator, defaultindex ) {
  var q = {};
  if ( operator == undefined )
    operator = "";
  q.op = operator;
  q.term=""; // will be overwritten below, mut I want to be sure that
    // a flatquery element always has both op and term, which is never
    // the case for a regular fullquery node.
  if ( fqterm == undefined )
    fqterm = {};
  for ( var fqfld in fqfields ) {
    var fld = fqfields[fqfld];
    if ( fqterm[fld] != undefined ) {
      q[fld] = fqterm[fld];
    }
  }
  if ( ! q.field )
    q.field = defaultindex;
  return q;
}; // constructor

// Flatten a fullquery into an array of flatquery elements
// throws a StepError if things get too complex.

// TODO - This is a moderately trivial flattening algorithm
// It assumes that the fullquery tree consists of and/or operators
// only, and each operator has one operand that is a plain term.
// Things to improve include:
// TODO - Handle "(a and b) and (c and d)" by reorganizing
// TODO - More advanced boolean logic. "x andnot ( a or b)"
//        can be translated into "x andnot a andnot b"
// TODO - Prox operator, and all its parameters?

FlatQuery.flattenfq = function ( fullq, maxterms, defaultindex ) {
  var flatlist = flatten(fullq, defaultindex );
  var nterms = flatlist.length;
  if ( maxterms && ( nterms > maxterms) ) {
    throw new StepError ("Query has too many terms (" + nterms +
       ", max= " + maxterms + "). ");
  }
  return flatlist;
}

function flatten( fullq, defaultindex ) {
  var flatlist = [];
  if (debug) dump("flatten: f=" + JSON.stringify(fullq) + "\n");

  // case 1: simple term
  if (fullq.term) {
    var fterm = FlatQueryElement(fullq,"",defaultindex);
    return [ fterm ];  // an array!
  }
  if ( !fullq.op ) // Should never happen
    throw new StepError ("Can not flatten query element, no term nor operator");

  // case 2: (...) op term  for and/or/not
  if ( ( fullq.op == "and" || fullq.op == "or" || fullq.op == "not" ) &&
       ( fullq.s2.term ) ) {
    var flist = flatten ( fullq.s1, defaultindex);
    var fterm = FlatQueryElement(fullq.s2, fullq.op, defaultindex);
    //fterm.foo="2";
    flist.push(fterm);
    return flist;
  }

  // case 3: term op (...)  for and/or - but not 'not', it won't work
  if ( ( fullq.op == "and" || fullq.op == "or" ) &&
    ( fullq.s1.term ) ) {
    var flist = flatten ( fullq.s2, defaultindex);
    var fterm = FlatQueryElement(fullq.s1, fullq.op, defaultindex);
    //fterm.foo="3";
    flist.push(fterm);
    return flist;
  }
  // We could not do it. Give informative error messages

  if ( fullq.op == "not" )
    throw new StepError("Query too complex to flatten: ... andnot ( ... )");
    // The easy case of ... andnot term has already been handled above

  if ( fullq.op != "and" && fullq.op != "or" && fullq.op != "not" )
    throw new StepError("Query too complex to flatten: " +
                        "can not handle " + fullq.op);
  if ( fullq.s1.op && fullq.s2.op )
    throw new StepError("Query too complex to flatten: " +
         "( ... ) " + fullq.op + " ( ... )" );

  // Fallback, in case I missed something above. Should not be needed
  throw new StepError("Query too complex to flatten: ");
}

//////////////////////////////
// Unit tests

var tests = [
  { title: "simple term",
    full: '{ "term" :"water" }',
    limit: 10,
    flat: '[{"op":"","term":"water"}]',
  },
  { title: "(a and b)",
    full: '{ "op":"and", "s1": {"term" :"a" }, "s2": {"term" :"b" } }',
    flat: '[{"op":"","term":"a"},{"op":"and","term":"b"}]',
  },
  { title: "(a or b)",
    full: '{ "op":"or", "s1": {"term" :"a" }, "s2": {"term" :"b" } }',
    flat: '[{"op":"","term":"a"},{"op":"or","term":"b"}]',
  },
  { title: "(a not b)",
    full: '{ "op":"not", "s1": {"term" :"a" }, "s2": {"term" :"b" } }',
    flat: '[{"op":"","term":"a"},{"op":"not","term":"b"}]',
  },
  { title: "(a and b) and c",
    full: '{ "op":"and", ' +
      '"s1": { "op":"and", "s1": {"term" :"a" }, "s2": {"term" :"b" } },'+
      '"s2": {"term" :"c" } }',
    flat: '[{"op":"","term":"a"},' +
      '{"op":"and","term":"b"},' +
      '{"op":"and","term":"c"}]',
  },
  { title: "(a or b) and c",
    full: '{ "op":"and", ' +
    '"s1": { "op":"or", "s1": {"term" :"a" }, "s2": {"term" :"b" } },'+
    '"s2": {"term" :"c" } }',
    flat: '[{"op":"","term":"a"},' +
    '{"op":"or","term":"b"},' +
    '{"op":"and","term":"c"}]',
  },
  { title: "(a and b) or c",
    full: '{ "op":"or", ' +
      '"s1": { "op":"and", "s1": {"term" :"a" }, "s2": {"term" :"b" } },'+
      '"s2": {"term" :"c" } }',
    flat: '[{"op":"","term":"a"},' +
      '{"op":"and","term":"b"},' +
      '{"op":"or","term":"c"}]',
  },
  { title: "(a and b) not c",
    full: '{ "op":"not", ' +
    '"s1": { "op":"and", "s1": {"term" :"a" }, "s2": {"term" :"b" } },'+
    '"s2": {"term" :"c" } }',
    flat: '[{"op":"","term":"a"},' +
    '{"op":"and","term":"b"},' +
    '{"op":"not","term":"c"}]',
  },
  { title: "a and (b and c)",
    full: '{ "op":"and", ' +
     '"s1": {"term" :"a" },'+
     '"s2": { "op":"and", "s1": {"term" :"b" }, "s2": {"term" :"c" } } }',
    flat: '[{"op":"","term":"b"},' +
      '{"op":"and","term":"c"},' +
      '{"op":"and","term":"a"}]',
  },
  { title: "a or (b and c)",
    full: '{ "op":"or", ' +
    '"s1": {"term" :"a" },'+
    '"s2": { "op":"and", "s1": {"term" :"b" }, "s2": {"term" :"c" } } }',
    flat: '[{"op":"","term":"b"},' +
    '{"op":"and","term":"c"},' +
    '{"op":"or","term":"a"}]',
  },
  { title: "a not (b and c)",
    full: '{ "op":"not", ' +
    '"s1": {"term" :"a" },'+
    '"s2": { "op":"and", "s1": {"term" :"b" }, "s2": {"term" :"c" } } }',
    flat: 'ERROR undefined: Query too complex to flatten: ... andnot ( ... )',
  },
  { title: "(a prox b)",
    full: '{ "op":"prox", "s1": {"term" :"a" }, "s2": {"term" :"b" } }',
    flat: 'ERROR undefined: Query too complex to flatten: can not handle prox',
  },

  { title: "(a and b) and (c and d)",
    full: '{ "op":"and", ' +
      '"s1": { "op":"and", "s1": {"term" :"a" }, "s2": {"term" :"b" } },' +
      '"s2": { "op":"and", "s1": {"term" :"c" }, "s2": {"term" :"d" } } }',
    flat: 'ERROR undefined: Query too complex to flatten: ( ... ) and ( ... )',
  },


];

FlatQuery.prototype.unitTest = function ( ) {
  dump("Unit test for FlatQuery starting \n");
  var termlimit = undefined;
  for ( var i in tests ) {
    if (tests[i].limit != undefined )
      termlimit = tests[i].limit;  // Keep limit from test to test
    var flat;
    var flatjs;
    try {
      var fq = JSON.parse(tests[i].full);
      flat = FlatQuery.flattenfq( fq, termlimit );
      flatjs = JSON.stringify(flat);
    } catch (e) {
      flatjs = "ERROR " + e ;
      if ( e.lineNumber && e.fileName )
        flatjs += "\n" + e.fileName + ":" + e.lineNumber ;
    }
    if ( flatjs != tests[i].flat ) {
      dump("Test " + i + ": " + tests[i].title + " FAILED \n");
      dump("fullq:    '" + tests[i].full + "'\n");
      dump("Expected: '" + tests[i].flat + "'\n");
      dump("Got:      '" + flatjs + "'\n");
      return false;
    }
    var failed = "";
    if ( flatjs.match( /^ERROR/ ) )
      failed = " failed as expected, which is";
    dump("test " + i + ": " + tests[i].title + failed + " OK\n");
    if (debug) {
      dump("fullq:    '" + tests[i].full + "'\n");
      dump("Expected: '" + tests[i].flat + "'\n");
      dump("Got:      '" + flatjs + "'\n");
    }
  }
  return true;
}

