/* This file is part of the Connector Framework
 * Copyright (C) 2008-2013 Index Data
 * See the file LICENSE for details.
 */
/**
 * \file cf_engine.h
 * \brief CF_Engine header
 */

#ifndef CF_ENGINE_H_INCLUDED
#define CF_ENGINE_H_INCLUDED
class CF_Embed;
class CF_Logger;

/**
 * Connector Engine. Encapsulates a Browser session
 *
 * The Engine may throw a CF_Engine_Error exception.
 */
class CF_Engine {
public:
    enum Flags {
        FORK = 0,
        THREAD = 1
    };
    /**
     * Constructor
     *
     * @param flags Engine flags (Bit-wise)
     * @param module_path is module path for Connector Modules.
     * @param app_path Gecko application location (must be writable);
     * may be NULL in which case exe working directory is used.
     * @param profile_path Profile area and cache;
     * may be NULL in which case exe no profile area is used.
     * @param proxy (may be NULL).
     */
    CF_Engine(enum Flags flags,
              const char *module_path,
              const char *app_path,
              const char *profile_path,
	      const char *proxy,
              const char *loglevel,
              CF_Logger *logger,
              int fd,
              int capt_stdout);

    /**
     * Destructor
     *
     * Kills the browser instance
     */
    virtual ~CF_Engine();

    /**
     * Loads connector file
     *
     * @param cf_path is the full path to connector file (.cf)
     */
    void load_cf(const char *cf_path, const char *session_json = 0);

    /**
     * Saves connector file
     *
     * @param cf_path is the full path to connector file (.cf)
     */
    void save_cf(const char *cf_path);

    /**
     * Runs Javascript from file in engine context
     *
     * @param script_path is the full path to connector file (.cf)
     */
    void run_script(const char *script_path);

    /**
     * Runs a task
     *
     * @param taskName name of task to run, say "search"
     * @param input task parameters
     * @param output result (allocated by CF_Engine)
     */
    void run_task(const char *taskName, const char *input,
		  const char **output);
    /**
     * Runs a task but don't fail if task does not exist
     *
     * @param taskName name of task to run, say "search"
     * @param input task parameters
     * @param output result (allocated by CF_Engine)
     */
    void run_task_opt(const char *taskName, const char *input,
                      const char **output);
    /**
     * Runs a task
     *
     * @param task_order order of tasks (task1,task2,..)
     * @returns true on success; false on failure
     */
    bool run_tests(const char *task_order);

    /**
     * Runs unit test
     *
     * @param name name of test to run
     * @returns true on success; false on failure
     */
    bool unit_test(const char *name);

    /** Take screen shot
     *
     * @param filename filename for output
     */
    bool screen_shot(const char *filename);

    /**
     * Return a string representation of the current DOM 
     */
    void dom_string(char **retval);
private:
    CF_Embed *m_priv;
    void init(enum Flags flags,
              int argc, char **argv, const char *module_path,
              const char *app_path, const char *mozembed_path,
              const char *proxy, const char *loglevel,
              CF_Logger *logger, int fd, int capt_stdout);
};

/**
 * Connector Engine Error
 */
class CF_Engine_Error {
protected:
    /**
     * Constructor
     *
     * @param msg Error message
     */
    CF_Engine_Error(const char *msg);
public:
    /**
     * Destructor
     */
    virtual ~CF_Engine_Error();
    /**
     * Gets Error message
     *
     * @return message
     */
    const char *what();
private:
    char *msg;
};

class CF_Engine_Error_System : public CF_Engine_Error {
public:
    CF_Engine_Error_System(const char *msg): CF_Engine_Error(msg)  { };
};

class CF_Engine_Error_Task : public CF_Engine_Error {
public:
    CF_Engine_Error_Task(const char *msg): CF_Engine_Error(msg)  { };
};


#endif
/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */

