var EXPORTED_SYMBOLS = ["Block"];

Components.utils.import('resource://indexdata/runtime/Step.js');
Components.utils.import('resource://indexdata/runtime/Comment.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/logging.js');

var logger = logging.getLogger();

var blockno = 1; // counter to number them all, to make sense of debug trace
var Block = function (task, name, label, parentStep) {
  this.task = task;
  this.steps = [];
  this.nextToRun = null;
  this.stopRunningAt = null;
  this.previousBlock = null;
  this.nextBlock = null;
  this.isError = false;
  this.isCancelled = false;
  this.name = name || null;
  this.label = label || null;
  this.parentStep = parentStep || null;
  //logger.debug("Created block " + this.debugname() );
  this.collapsed = false; //display hint
  this.blockno = blockno++;
};

Block.prototype.debugname = function() {
  return "" + this.blockno + "." + this.name + " " + this.label;
}

//statics

Block.fromConf = function (task, blockNode, parentStep) {
  let block = new Block(task, null, null, parentStep);
  //in case we deal with explicit block, read additional parameters
  if (blockNode.localName == "block") {
    block.name = blockNode.getAttribute('name');
    block.label = blockNode.getAttribute('label');
    block.collapsed = blockNode.getAttribute('collapsed') == "yes";
  }
  logger.debug("Loading block " + block.debugname() );
  for (var j=0; j<blockNode.childNodes.length; j++) {
    let node = blockNode.childNodes[j];
    switch (node.nodeType) {
      // element node
      case 1:
        switch (node.localName) {
          case "step":
            let step = Step.fromConf(task, block, node);
            block.steps.push(step);
            block.task.attachStep(step);
            logger.debug("Loaded step " + step.getClassName() + " " +
              step.renderArgs() + " into block " + block.debugname() );
            break;
        }
        break;
        // text node
      case 3:
        break;
        // comment node
      case 8:
        block.addComment(node.nodeValue);
        break;
    }
  }
  logger.debug("Loaded Block " + block.blockno + " with name "
      + (block.name ? block.name : "root-block") + " " + block.debugname() );
  return block;
};

//instance

Block.prototype.getClassName = function () {
    return "Block";
};

Block.prototype.saveConf = function (parentNode) {
  //if explicit block, save extended attrs
  if (parentNode.localName == 'block') {
    parentNode.setAttribute('name', this.name);
    parentNode.setAttribute('label', this.label);
    if (this.collapsed) parentNode.setAttribute('collapsed', 'yes');
  }
  for (var i=0; i<this.steps.length; i++) {
    let step = this.steps[i];
    if (step.getClassName() === "Comment") {
      let commentNode = parentNode.ownerDocument.createComment(step.body);
      parentNode.appendChild(commentNode);
    } else {
      let stepNode = xmlHelper.appendNode(parentNode, "step");
      this.steps[i].saveConf(stepNode);
    }
  }
};

// insert item at index, or at the end if index is
// invalid or missing. return position of inserted step.
Block.prototype.insertItem = function (item, index) {
  item.block = this;
  if (index === undefined || index < 0 || index > this.steps.length) {
    return this.steps.push(item) - 1;
  } else {
    this.steps.splice(index + 1, 0, item);
    return index + 1;
  }
};

Block.prototype.addComment = function (text, index) {
  let comment = new Comment(text, this);
  return this.insertItem(comment, index)
};

Block.prototype.getItemAt = function (index) {
  return this.steps[index];
};

Block.prototype.getItemsSize = function (index) {
  return this.steps.length;
};

Block.prototype.newStep = function (step_name, index) {
  // not sure if I like this anymore, but I might again
  /* try {
    // default conf
    let url = 'resource://indexdata/steps/' + step_name + '/' + 'defaultConf.cf';
    let doc = xmlHelper.fetchDoc(url); 
    let stepNode = doc.firstChild;
  } catch (e) { */
    // no conf
    //logger.debug('Creating new step ' + step_name + ', no defaults');
    let step = Step.fromFileName(step_name);
    if (typeof step.setDefaults === 'function') step.setDefaults();
    this.task.attachStep(step, true);
    return this.insertItem(step, index);
  //}
};

Block.prototype.itemUp = function(s) {
  if (typeof(s) != "number")
    throw new TypeError("array index is not a number");
  else if (s < this.steps.length && s > 0)
    this.steps.splice(s-1, 2, this.steps[s], this.steps[s-1]);
  else
    throw new RangeError("cannot shift array element out of bounds");
};

Block.prototype.itemDown = function(s) {
  if (typeof(s) != "number")
    throw new TypeError("array index is not a number");
  else if (s < this.steps.length-1 && s >= 0)
    this.steps.splice(s, 2, this.steps[s+1], this.steps[s]);  else
      throw new RangeError("cannot shift array element out of bounds");
};

// delete item
Block.prototype.deleteItem = function (idx) {
  return this.steps.splice(idx, 1)[0];
};

// delete all items
Block.prototype.clearItems = function () {
  this.steps = [];
};

// delete given an array of indices
Block.prototype.deleteItems = function (indices) {
  // mark for deletion
  for (var i=0; i<indices.length; i++) {
    this.steps[indices[i]].mark = true;
  }
  i = 0;
  while (i<this.steps.length) {
    if (this.steps[i].mark) {
      var step = this.steps.splice(i, 1);
    } else {
      i++;
    }
  }
};

Block.prototype.findItemIndex = function (step) {
  return this.steps.indexOf(step);
}

Block.prototype.findStepIndex = function (step) {
  for (let i=0; i<this.steps.length; i++) {
    //compare references
    if (step === this.steps[i]) return i;
  }
  return -1;
};

// runtime

Block.prototype.seekRelative = function (steps) {
  if (typeof(steps)!=="number")
    this.task._error("must provide a number of steps to seek.");
  if (steps!==parseInt(steps))
    this.task._error("seek distance must be an integer.");
  if (this.nextToRun != null && (this.nextToRun + steps) >= 0 && (this.nextToRun + steps) <= this.stopRunningAt )
    this.nextToRun+=steps;
  else
    this.task._error("attemped to seek to a step out of range.");
};


Block.prototype.seekEnd = function () {
  this.nextToRun = this.steps.length;
};

// End this iteration of the block by skipping to the end
Block.prototype.complete = function () {
  this.nextToRun = this.stopRunningAt;
};

// Complete block and mark it to not be iterated
Block.prototype.cancel = function () {
  this.complete();
  this.isCancelled = true;
};

Block.prototype.interrupt = function (delay, cb) {
  logger.debug( "interrupt on " + this.debugname()  );
  this.task.interrupt(delay, cb);
};

// Reset the run state in all steps
Block.prototype.resetState = function () {
  logger.debug("Block.resetState on " + this.debugname() );
  for ( var si = 0; si < this.steps.length; si++ ) {
    if ( this.steps[si].getBlocks ) {  // comments don't have one
      var blocks = this.steps[si].getBlocks();
      //logger.debug("Block.resetState on " + this.debugname() + " step " + 
      //  this.steps[si].toString() + " with " + blocks.length + " blocks" );
      this.steps[si].resetState();
      for ( var bi = 0; bi < blocks.length; bi++ ) {
        //logger.debug("Block.resetState recursing to step " + si + 
        //  " block " + bi + " " + blocks[bi].debugname() );
        blocks[bi].resetState();
      }
    }
  }
};
                                

Block.prototype.run = function (previousBlock, 
    /*0-indexed*/ from, /*exclusive*/to) {
  if (previousBlock) {
    this.previousBlock = previousBlock;
    previousBlock.nextBlock = this;
  } else {
    this.task._eventList.dispatchEvent("onTaskStart", true);
    this.task.info("Task started.");
  }
  from = typeof from == "number" ? parseInt(from) : 0;
  to = typeof to == "number" ? parseInt(to) : this.steps.length;
  this.isError = false;
  this.nextToRun = from;
  this.stopRunningAt = to;
  if (this.stopRunningAt == 0) {
    this.task.warn("Block: No steps to run.");
  }
  this.resume();
};

// Log error, tell engine, etc.
// Throws the same error again, unless alt step next
Block.prototype.handleError = function (e) {
  //follow the stack
  if (this.nextBlock) {
    logger.debug("Error handling passed to the next block " + this.debugname() );
    this.nextBlock.handleError();
    return;
  }
  logger.debug("Block ["+this.debugname()+"] will attempt to handle error");
  var errMsg = "Error in step '"
    + this.steps[this.nextToRun-1] +"' "
    + "#" + (this.nextToRun-1) + ": ";
  // Defensive coding, we have seen 'e is undefined' !!??
  if (e && e.message) 
    errMsg +=  e.message ;
  if (e && e.fileName ) {
    errMsg += "\nin " + e.fileName + ":" + e.lineNumber ;
  }
  // errors
  if (this.nextToRun < this.stopRunningAt
      && this.steps[this.nextToRun].isAlt
      && ! this.steps[this.nextToRun].isDisabled ) {
    //next step is an alt, mark error and let it continue
    this.task.warn(errMsg);
    this.task.warn("Step failed, trying alternatives...");
    this.isError = true;
  } else {
    //we don't handle it here but rethrow, in case we are the top
    //block we terminate
    logger.debug("Block ["+this.debugname()+"] terminates because of unhandled error");
    if (!this.previousBlock) {
      this.task._error(errMsg);
      this.task._error("Task failed.");
      this.task._eventList.dispatchEvent("onTaskFinish", false, errMsg, 
        this.nextToRun-1);
    } else {
      //pop block
      logger.debug("Block ["+this.debugname()+"] popped from stack" );
      this.previousBlock.nextBlock = null;
    }
    logger.debug("Block ["+this.debugname()+"] re-throwing the error" );
    throw e;
  }
  logger.debug("Block ["+this.debugname()+"] returning from handleError" );
};

Block.prototype.resume = function () {
  //clear interrupt
  this.task.isInterrupted = false;
  this.task.interruptTimer.cancel();
  //recreate the JS stack after interrupt
  logger.debug("Block ["+this.debugname()+"] resuming execution." );
  if (this.nextBlock) {
    try{
      this.nextBlock.resume();
    } catch(e) {
      logger.debug("Block ["+this.debugname()+"] caught an exception");
      this.handleError(e);
    }
  }
  if ( this.task.isInterrupted ) {
    logger.debug("Block " + this.debugname() + " got interrupted in the recursion");
    return;
  }
  logger.debug("Actually resuming block " + this.debugname() +
    " task.isInt=" + this.task.isInterrupted + " isError=" + this.isError );
  // run consecutive steps unless callback registered
  while (this.nextToRun < this.stopRunningAt) {
    //increase before running - in case the callback was quicker
    this.nextToRun += 1;
    let currentStep = this.steps[this.nextToRun-1];
    let nextStep = this.steps[this.nextToRun]
      // skip disabled steps
      if (currentStep.isDisabled) {
        this.task.info("Step #" + this.nextToRun+" is disabled, ignoring");
        continue;
      }
    // skip comments and other non-runnable objects
    if (!currentStep.run) {
      this.task.debug("Skipping non-runnable entry at steplist index #" 
          + this.nextToRun);
      continue;
    }
    //sweep through alternatives if no errors
    if (!this.isError && currentStep.isAlt) continue;
    // prepare for errors
    try {
      logger.debug("Trying to run step #" + this.nextToRun +
        " " + currentStep.getClassName() + " " + currentStep.renderArgs());
      currentStep.run(this.task, this);
      logger.debug("Did run step #" + this.nextToRun + " isE=" + this.isError +
        " isInt=" + this.task.isInterrupted );
      this.isError = false;
      this.task._eventList.dispatchEvent("onStepFinish",
          this.nextToRun-1, this.stopRunningAt);
    } catch (e) {
      logger.debug("Step " + this.nextToRun + " caught exception " + e +
        " in block " + this.debugname() );
      this.handleError(e);  // does not return, throws it again
    }
    // if callback return immediately, could do it once but this is cleaner
    if (this.task.isInterrupted) {
      logger.debug("Block " + this.debugname() + " was interrupted, exiting");
      return;
    };
  }
  logger.debug("Block ["+this.name+"] execution has terminated." + this.debugname() );
  //we are not interrupted and the block has finished, pop
  if (this.previousBlock)
    this.previousBlock.nextBlock = null;
  else {
    this.task.info("Task finished with no errors.");
    this.task._eventList.dispatchEvent("onTaskFinish", true, 
        this.task.output);
  }
};


//other
Block.prototype.getUsedArgs = function () {
  var usedArgs = {};
  for (var i=0; i<this.steps.length; i++) {
    var step = this.steps[i];
    if ((step.getClassName() !== "Comment")) {
      //step args
      var args = step.getUsedArgs();
      if (args) {
        for (let j=0; j<args.length; j++) {
          usedArgs[args[j]] = true;
        }
      }
      //substep args
      for (let j=0; j<step.blocks.length; j++) {
        args = step.blocks[j].getUsedArgs();
        if (args) {
          for (var k=0; k<args.length; k++) {
            usedArgs[args[j]] = true;
          }
        }
      }
    }
  }
  return Object.keys(usedArgs);
};

