var EXPORTED_SYMBOLS = ["Fullquery"];

Components.utils.import('resource://indexdata/runtime/Connector.js');
Components.utils.import('resource://indexdata/runtime/Task.js');
Components.utils.import('resource://indexdata/runtime/Step.js');
Components.utils.import('resource://indexdata/runtime/StepError.js');
Components.utils.import('resource://indexdata/util/xulHelper.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/textHelper.js');
Components.utils.import('resource://indexdata/util/queryHelper.js');
Components.utils.import('resource://indexdata/util/jsonPathHelper.js');
Components.utils.import('resource://indexdata/util/logging.js');
Components.utils.import('resource://indexdata/thirdparty/jsonPath.js');
Components.utils.import('resource://indexdata/ui/testEditor.js');

var logger = logging.getLogger();

// Enhancements

// TODO - Operators
//  - Add checkboxes for not accepting complex subexpressions
//    (child is a opnode, with a different operator. That way
//    we can have a three-way and/or, but not a combined and/or tree
//    Would be useful in google-lthings, where we 'and' translates
//    to +XX +YY, and neither XX nor YY can be an expression. Still we want to
//    allow multiple ands in one query.

// TODO
//    For EBSCO proximity, the limitation is even more severe, the sub-
//    expression can only be a simple term without ()s around it, so we
//    can not even specify a field, it seems. We could handle it by requiring
//    that the fields are the same, and then moving the field on the outside,
//    so instead of "(AU orwell) N4 (AU george)" we could produce something
//    like "AU ( orwell N4 george ). But how to make the configuration of
//    this reasonable for humans?


// TODO - Validate: No duplicate indexes (names or aliases). Highlight red

// TODO - Process a whole subtree with relational nodes on the same field
//    into one range.  (yr >= 2000 && yr = 2005 && yr <= 2009)
//    Will still fail if the terms are not in the same subtree (there is
//    some other term in between). Well, fullquerylimit is for such cases!
//    It could even be argued that it is not the job of fullquery to mess
//    with such...

// Default search setups
// These are only used when building connectors, but when running, the
// values are  always taken from the connectors conf[] array.
var defaults = { };  // Getting them from the template

const unsupported_value = "#UNSUPPORTED#";

const quotestyles = [
    [ "custom" ],   // custom must be the first in the array. No values defined!
    [ "none", "", "" ],
    [ "double quotes", '"', '"' ],
    [ "single quotes", "'", "'" ],
    [ "parentheses", "(", ")" ],
    [ "square brackets", "[", "]" ],
    [ "(Not supported)", unsupported_value,unsupported_value ]
];

var Fullquery = function () {
    this.className = "Fullquery";
    this.conf = {};
    this.conf['container'] = 'INPUT';
    this.conf['in'] = { path:'$.input', key:'fullquery'};
    this.conf['out'] = { path:'$.temp', key: 'query',
                         append: jsonPathHelper.REPLACE};
    this.conf['failnoqry'] = true;
    this.conf['defaults'] = 'none';
    this.conf['fields'] = {}; // "title" => "ti=XX"
    // this.conf['defaultfield'] = "";
      // if undset, it indicates that we have not applied defaults yet
    this.conf['operators'] = {}; // "and" => "(XX and YY)"
    this.conf['failbadfield'] = false;
    // Leave the rest unset, we take from the template
    // quote_word_L, quote_word_R
    // quote_phrase_L, quote_phrase_R
    // quote_default_L, quote_default_R
    // trunc_wildcard, trunc_left, trunc_right, trunc_both, trunc_mask
    // rangefields, range_B, range_BE, range_E
};

Fullquery.prototype = new Step();
Fullquery.prototype.constructor = Fullquery;

Fullquery.prototype.init = function() {};

///////////////
// Helpers

// Get the defaults from the template
Fullquery.prototype.getDefaults = function () {
    var templ = this.task.getTemplate();
    var defs = templ.getFullqueryDefaults();
    defaults = defs;
    defaults['none'] = { displayname: "Select a profile" };
};

// Copy the defaults from the template into conf
// Called once, when the user selects the default set.
// And never again!
Fullquery.prototype.ApplyDefaults = function () {
  var defset = this.conf['defaults'];
  var defs = defaults[defset];
  // Unfortunately the template is not quite same format as conf,
  // so we need to handle each part individually. Too many slight special
  // cases around...
  for ( var name in defs.fields ) {
    var str = defs.fields[name];
    if ( str ) { // note, we do not copy empty ones. They still go in pulldowns
      this.conf['fields'][name]=str;
    }
  }
  this.conf.defaultfield = defs.defaultfield;
  if ( this.conf.defaultfield == undefined )
    this.conf.defaultfield = "";
  this.conf.failbadfield = true;

  this.conf.op_and = unsupported_value;
  this.conf.op_or  = unsupported_value;
  this.conf.op_not = unsupported_value;
  this.conf.op_prox_ord = unsupported_value;
  this.conf.op_prox_unord = unsupported_value;
  for ( var op in defs.operators ) {
    if ( defs.operators[op] != null )
      this.conf["op_" + op] = defs.operators[op];
  }

  if ( defs.terms.quote_word ) {
    this.conf['quote_word_L'] = defs.terms.quote_word[0];
    this.conf['quote_word_R'] = defs.terms.quote_word[1];
  }
  if ( defs.terms.quote_phrase ) {
    this.conf['quote_phrase_L'] = defs.terms.quote_phrase[0];
    this.conf['quote_phrase_R'] = defs.terms.quote_phrase[1];
  }
  if ( defs.terms.quote_default ) {
    this.conf['quote_default_L'] = defs.terms.quote_default[0];
    this.conf['quote_default_R'] = defs.terms.quote_default[1];
  }

  if ( defs.terms.trunc_wildcard )
    this.conf.trunc_wildcard = defs.terms.trunc_wildcard;
  else
    this.conf.trunc_wildcard = "*";

  this.conf.trunc_left = ( defs.terms.trunc_left == "true" );
  this.conf.trunc_right = ( defs.terms.trunc_right == "true" );
  this.conf.trunc_both = ( defs.terms.trunc_both == "true" );
  this.conf.trunc_mask = ( defs.terms.trunc_mask == "true" );

  for ( var r in defs.range ) {
    if ( defs.range[r] )
      this.conf[r] = defs.range[r];
  }

  logger.debug("Copied defaults. conf now " + JSON.stringify ( this.conf) );
};

//////////////////
// Draw

Fullquery.prototype.draw = function(surface,win) {

    // Get the defaults from the template
    defaults = this.task.getTemplate().getFullqueryDefaults();
    defaults['none'] = { displayname: "Select a profile" };

    var tabs = xulHelper.tabBox( surface,
            [ "General", "Fields", "Operators", "Terms", "Ranges", "Examples" ],
            { flex: 1 } );
    //if (! this.conf['defaults'])  // make sure we have defaults
    //    this.conf['defaults']='ccl';
    this.drawGeneralTab( tabs[0], this, tabs);
    this.drawExampleTab( tabs[5], this);
    // We can't draw the other tabs yet, as they depend on having a good
    // value in conf['defaults'], which can be changed from the
    // general tab. So that's where we draw the tabs.
    // TODO - It would be nice to disable the other tabs,
    // for now they just show empty
    /*
    this.drawFieldTab( tabs[1], this);
    this.drawOperatorTab( tabs[2], this);
    this.drawTermTab( tabs[3], this);
    */
    return;
};


Fullquery.prototype.drawGeneralTab = function (surface, context, tabs) {
    var vb = xmlHelper.appendNode(surface, "vbox", null, { flex: 1 }  );
    xulHelper.jsonPathMapField(vb, context, "in", "out");
    var defnames = {};
    for ( var d in defaults ) {
        defnames[d] = defaults[d].displayname;
    }
    defattrs = { disabled: true};

    if ( context.conf["defaults"] == undefined ||
        context.conf["defaults"] == "none" )
        defattrs.disabled = false;

    var defsel = xulHelper.selectField(vb, context, "defaults",
             "Starting point", defnames, defattrs );

    xulHelper.checkbox(vb, context, "failnoqry", "Fail if no query to begin with" );

    var selchange = function () {
        if ( context.conf["defaults"] == undefined ||
            context.conf["defaults"] == "none" ) {
            defsel.disabled = false;
        } else {
            // Copy defaults over here, and nowhere else!
            // That way, it will be only done once. Bug 5323
            if ( context.conf.defaultfield == undefined ) { // first time only!
              context.ApplyDefaults();
            }
            defsel.disabled = true;
            context.drawFieldTab( tabs[1], context);
            context.drawOperatorTab( tabs[2], context);
            context.drawTermTab( tabs[3], context);
            context.drawRangeTab( tabs[4], context);

        }
    };
    selchange();
    defsel.addEventListener("command", selchange, false);
}

Fullquery.prototype.drawFieldTab = function (surface,context) {
    xmlHelper.emptyChildren(surface);

    var vb = xmlHelper.appendNode(surface, "vbox");
    var fieldbox = xmlHelper.appendNode(vb, "vbox");
    var formbox = xmlHelper.appendNode(vb, "vbox");
    var defbox = xmlHelper.appendNode(formbox, "hbox",
        null, { align: "center" });

    var defstr = xulHelper.inputField(defbox, context,
        "defaultfield", "(unspecified)", 100 );
    enattrs = {};
    if ( context.conf.defaultfield == unsupported_value ) {
        enattrs.checked = true;
        defstr.disabled = true;
    }
    var failunspec  = xulHelper.checkbox(defbox, context,
                        "", "Fail if unspecified" );
    failunspec.addEventListener("command", function(e) {
        logger.debug("uspecsupported command event checkbox: ch=" +this.checked );
        if (this.checked) {
            defstr.value = unsupported_value;
            defstr.disabled = true;
        } else {
            var defset = context.conf['defaults'];

            defstr.value = defaults[defset].defaultfield;
            defstr.disabled = false;
        }
        context.conf.defaultfield = defstr.value;
    }, false);

    var addbox = xmlHelper.appendNode(formbox, "hbox");

    var addButton = xmlHelper.appendNode(addbox, "button",
        null, {"label": "Add", "width" : 20, "height" : 30 }, null);

    var enabled = xulHelper.checkbox(addbox, context,
                            "failbadfield", "Fail if unknown field" );

    var addfield = function(name, strval) {
        //dump("Addfield '" + name + "' : '" + strval + "'\n");
        var fb = xmlHelper.appendNode(fieldbox, "hbox",
                null, { align: "center" }, null );
        //xulHelper.captionField(fb, name );
        var str = xulHelper.inputField(fb, context, "", name, 100,
                { "value":strval,
                tooltiptext:"The string to put in the final query, \n" +
                "for example 'TITLE:' or '(ti=XX)'."} );
        str.addEventListener("input", function(e) {
            context.conf['fields'][name] = this.value;
         }, false);
        var delbut = xmlHelper.appendNode(fb, "button",
              null, {"label": "Delete", "width" : 20, "height" : 30 }, null);
        delbut.addEventListener("command", function(e){
            //context.conf['fields'][name] = undefined;
            delete context.conf['fields'][name];
            xmlHelper.removeNode(fb);
        }, false );
    } // addfield of drawFieldTab

    var newField = function(e) {
        addButton.disabled = true;
        var ab = xmlHelper.appendNode(addbox, "hbox",
             null, { align: "center" });
        var fieldlist = [];
        var defset = context.conf['defaults'];
        for ( var fld in defaults[defset].fields ) {
            if ( context.conf['fields'][fld] == undefined ) {
                fieldlist.push(fld);
            }
        }
        xulHelper.captionField(ab,"Choose a field");
        var sel = xulHelper.arraySelectField(ab, context, "",fieldlist);
        sel.editable = true;
        sel.width = 200;
        var doadd =  xmlHelper.appendNode(ab, "button",
            null, {"label": "Add it", "height" : 30  }, null);
        doadd.addEventListener("command", function(e){
            var name = sel.value;
            if ( name && context.conf['fields'][name] == undefined ) {
                addButton.disabled = false;
                xmlHelper.removeNode(ab);
                var defval = defaults[defset].fields[name];
                if (!defval)
                    defval="";
                context.conf['fields'][name] = defval;
                addfield(name,defval);
            }
        }, false );
         var cancel =  xmlHelper.appendNode(ab, "button",
            null, {"label": "Cancel", "height" : 30 }, null);
            cancel.addEventListener("command", function(e){
                addButton.disabled = false;
                xmlHelper.removeNode(ab);
        }, false );
    } // newField

    for ( var fld in context.conf.fields ) {
        addfield( fld, context.conf['fields'][fld] );
    }
    addButton.addEventListener("command", newField, false );

} // drawFieldTab


Fullquery.prototype.drawOperatorTab = function (surface,context) {
    //dump("FieldTab: context: " + JSON.stringify( context.conf ) + "\n");
    xmlHelper.emptyChildren(surface);
    var hb = xmlHelper.appendNode(surface, "hbox",null, {flex:1});
    var obox = xmlHelper.appendNode(hb, "vbox");
    var defset = context.conf['defaults'];

    xmlHelper.appendNode(hb, "separator", null,
        {"class": "groove", orient:"vertical"}, null);
    xmlHelper.appendNode(hb, "spacer", {width:10} );
    var proxbox = xmlHelper.appendNode(hb, "vbox", null, {flex:1} );

    var oneoperator = function( box, opname, dispname ) {
        var opbox = xmlHelper.appendNode(box, "hbox");
        var confindex= "op_" + opname;
        logger.debug("oneoperator " + confindex + "=" + context.conf[confindex] );
        opattrs = {};
        enattrs = {};
        if ( context.conf[confindex] != unsupported_value )
            enattrs.checked = true;
        else
            opattrs.disabled = true;
        var opval = xulHelper.inputField(opbox, context, confindex, dispname,
                                         70, opattrs);
        var enabled = xulHelper.checkbox(opbox, context, "", "Supported", enattrs );
        enabled.addEventListener("command", function(e) {
            logger.debug("command on '" + opname + "' checkbox: ch=" +this.checked );
            if (!this.checked) {
                opval.value = unsupported_value;
                opval.disabled = true;
            } else {
                opval.disabled = false;
                if ( opval.value == unsupported_value )
                    if ( defaults[defset].operators[opname] == unsupported_value ||
                         defaults[defset].operators[opname] == undefined   )
                        opval.value = "";
                    else
                        opval.value = defaults[defset].operators[opname];
            }
            context.conf[confindex] = opval.value;
        }, false);
    }; // oneoperator of drawOperatorTab
    oneoperator(obox, "and", "And");
    oneoperator(obox, "or", "Or");
    oneoperator(obox, "not", "And-not");
    xulHelper.labelField(obox, "example: and", null );
    xulHelper.labelField(obox, "example: (XX and YY)", null );

    oneoperator(proxbox, "prox_ord","Proximity, ordered");
    oneoperator(proxbox, "prox_unord","Proximity, unordered");
    xulHelper.labelField(proxbox, "example: (XX N%DIST% YY)", null );
    xulHelper.labelField(proxbox,
       "the %DIST% will be replaced by the distance ", null );
} // drawOperatorTab

///// Terms

// Helper to display a line for one for of quoting

Fullquery.prototype.quoteline = function (surface,context,capt,confindex) {
    var defset = context.conf['defaults'];
    var li = confindex+"_L";
    var ri = confindex+"_R";
    /*
    if (context.conf[li] == undefined ) {
        //dump("Init: ds='" + defset + "' ci='" + confindex + "'\n");
        context.conf[li] = defaults[defset].terms[confindex][0];
        context.conf[ri] = defaults[defset].terms[confindex][1];
    }
    */
    var vb = xmlHelper.appendNode(surface, "hbox", null, { align: "center" } );
    xulHelper.captionField(vb, capt,{ width: 150 } );

    // build a pull-down for the quote styles
    var sel = xmlHelper.appendNode(vb, "menulist" );
    var menupop = xmlHelper.appendNode(sel, "menupopup");
    var found = false;
    var theSelectedItem = -1;
    for ( var i in quotestyles ) {
        var opts = { label: quotestyles[i][0], value: i };
        if ( context.conf[li] == quotestyles[i][1] &&
             context.conf[ri] == quotestyles[i][2] ) {
            opts.selected = true;
            if ( i != 0 ) {
                theSelectedItem = i;
                found = true;
            }
        }
        xmlHelper.appendNode(menupop, "menuitem", null, opts );
    }
    if (sel.selectedIndex < 0 ) {
        if ( theSelectedItem < 0)
            theSelectedItem = 0;
        sel.selectedIndex = theSelectedItem;
    }
    var attrs = { width:"40" };
    if (found)
        attrs.disabled = true;
    var lq = xulHelper.inputField(vb, context, li, " ", 10, attrs );
    var rq = xulHelper.inputField(vb, context, ri, " ", 10, attrs );

    var changesel = function () {
        var s = sel.value;
        //dump("sel changed to " + s + "\n");
        if ( s == 0 ) { // custom
            //dump("custom\n");
            lq.disabled = false;
            rq.disabled = false;
        } else {
            lq.disabled = true;
            rq.disabled = true;
            lq.value = quotestyles[s][1];
            context.conf[li] = quotestyles[s][1];
            rq.value = quotestyles[s][2];
            context.conf[ri] = quotestyles[s][2];
        }
    } // changesel of quoteline
    sel.addEventListener("command", changesel, false);

}

Fullquery.prototype.drawTermTab = function (surface,context) {
    xmlHelper.emptyChildren(surface);
    var hb = xmlHelper.appendNode(surface, "hbox",null, {flex:1});
    var qb = xmlHelper.appendNode(hb, "vbox");
    var defset = context.conf['defaults'];
    xulHelper.captionField(qb,"Quotes" );
    this.quoteline(qb, context, "word term", "quote_word");
    this.quoteline(qb, context, "phrase term", "quote_phrase");
    this.quoteline(qb, context, "default term", "quote_default");
    xmlHelper.appendNode(hb, "separator", null,
         {"class": "groove", orient:"vertical"}, null);
    xmlHelper.appendNode(hb, "spacer", {width:10} );
    var tb = xmlHelper.appendNode(hb, "vbox", null, {flex:1} );
    xulHelper.captionField(tb,"Truncation" );

    /*
    if (context.conf["trunc_wildcard"] == undefined ) {
      context.conf["trunc_wildcard"] = defaults[defset].terms["trunc_wildcard"];
    }
    */
    xulHelper.inputField(tb,context, "trunc_wildcard", "Truncation wildcard",
                         150, {width:20} );
    // TODO - Do we need this stuff ??
    // Probably not. They should never be undefined, and only one place ought
    // to set the default values.
    if (context.conf["trunc_right"] == undefined ) {
      context.conf["trunc_right"] =
        (defaults[defset].terms["trunc_right"] == "true" );
    }
    if (context.conf["trunc_left"] == undefined ) {
      context.conf["trunc_left"] =
        (defaults[defset].terms["trunc_left"] == "true" );
    }
    if (context.conf["trunc_both"] == undefined ) {
      context.conf["trunc_both"] =
        (defaults[defset].terms["trunc_both"] == "true" );
    }
    if (context.conf["trunc_mask"] == undefined ) {
      context.conf["trunc_mask"] =
        (defaults[defset].terms["trunc_mask"] == "true" );
    }
    xulHelper.checkbox(tb, context, "trunc_right",
                       "Right truncation supported (watermel*)" );
    xulHelper.checkbox(tb, context, "trunc_left",
                       "Left truncation supported (*termelon)" );
    xulHelper.checkbox(tb, context, "trunc_both",
                       "Both truncation supported (*termel*)" );
    xulHelper.checkbox(tb, context, "trunc_mask",
                       "Masking supported (wat*lon)" );

} // drawTermTab

Fullquery.prototype.drawRangeTab = function (surface,context) {
  xmlHelper.emptyChildren(surface);
  var vb = xmlHelper.appendNode(surface, "vbox",null, {flex:1});

  var r_f = xulHelper.inputField(vb, context, "rangefields",
                       "Field(s) that support ranges",  220 );
  var r_b = xulHelper.inputField(vb, context, "range_B",  "XX -   ", 120 );
  var r_r = xulHelper.inputField(vb, context, "range_BE", "XX - YY", 120 );
  var r_e = xulHelper.inputField(vb, context, "range_E",  "   - YY", 120 );
  xulHelper.labelField(vb, "Use XX and YY for the start and end values", null );
  xulHelper.labelField(vb, "You can use FF for the field, " +
                "define it in the field tab", null );
  var warninglbl = xulHelper.labelField(vb, "", { style:"color:red;"} );

  // Check if we have a nasty situation where f.ex year range is defined
  // as FF=(XX-YY), *and* year inteh field tab is defined as yr=XX.
  // This could be handled better, but at the moment we need a builder-
  // only fix, so we just display a warning.
  var checkXXwarning = function(e) {
    warninglbl.value = "";
    var allranges = "" + context.conf.range_B + context.conf.range_BE +
      context.conf.range_E;
    if ( allranges.indexOf("FF") == -1 )
      return;  // no FF used anywhere, no need to check more
    //logger.debug("XXwarning: " + context.conf.rangefields );
    var fieldlist = context.conf.rangefields.split(" ");
    for ( var fi in fieldlist ) {
      var f = fieldlist[fi];
      var fstr = context.conf.fields[f];
      //logger.debug("XXwarning looking at " + fi + ":" + f + "= " + fstr );
      if ( fstr && fstr.indexOf("XX") != -1 ) {
        warninglbl.value = "Be careful, " + f + " is defined as " + fstr;
        return;
      }
    }
  };

  r_f.addEventListener("change", checkXXwarning, false);
  r_f.addEventListener("focus", checkXXwarning, false);
  r_b.addEventListener("change", checkXXwarning, false);
  r_r.addEventListener("change", checkXXwarning, false);
  r_e.addEventListener("change", checkXXwarning, false);
  checkXXwarning();

} // drawRangeTab

Fullquery.prototype.drawExampleTab = function (surface,context) {
  Components.utils.import('resource://indexdata/ui/app.js');
  xmlHelper.emptyChildren(surface);
  const captwidth = 50;
  var vb = xmlHelper.appendNode(surface, "vbox", null, {flex:1});
  var jsonbox = xmlHelper.appendNode(vb, "hbox", null, { align: "center" });
  var jsoninput = xulHelper.inputField(jsonbox, context, "", "JSON: ",
                captwidth, {"align":"top", "flex":"1"} );
  jsoninput.flex=1; // for some reason, this works better than setting
                    // it in the attrs above, that makes it flex in both
                    // directions, this only horizontally

  var jsonEditButton = xmlHelper.appendNode(jsonbox, "button", null,
        {"label": "Edit..."});
  jsonEditButton.addEventListener("command", function (e) {
    app.newJsonEditWindow(
      JSON.parse(jsoninput.value || "{}"),
      function (obj) {
        jsoninput.value = JSON.stringify(obj);
      },
      'editData'
    );
  }, false);


  var qbox = xmlHelper.appendNode(vb, "hbox",null, {align:"top" });
  xmlHelper.appendNode(qbox, "caption", "", { width: captwidth } );
  var qry =   xmlHelper.appendNode(qbox, "label",
                "Click on the examples below",
                { flex:1, "align":"top" });
  var pastedisabled = ( context.conf.in.path != "$.input" );
  var pastebut = xmlHelper.appendNode(qbox, "button", null,
             {"label": "Paste", disabled: pastedisabled});

  function pasteExample(ex) {
    var fldname = context.conf.in.key;
    var activetest = testEditor.getActiveTest();
    if ( !activetest ) {
      logger.debug("no active test, can not paste");
      return; // Should never happen, defensive coding
    }
    activetest = activetest.getName();
    for ( var i = 0; i< context.task.tests.length; i++ ) {
      var tst = context.task.tests[i];
      logger.debug("paste: Comparing test " + tst.getName() );
      if ( tst.getName() == activetest ) {
        var oldt = tst.getArg(fldname);
        logger.debug("paste: Found it! " + fldname +"=" + oldt + " " + typeof(oldt));
        if ( typeof(oldt) != "string" ) {
          logger.debug("paste: Field " + fldname + " does not exist in test set." +
             " Will not risk creating one" );
          // this risks creating testset.yearlimit, if the step is actually
          // working on $.temp.yearlimit - and having such a test argument
          // guarantees that the repo test will always fail, without the
          // CAs being able to see what is wrong, or fix (except by editing
          // the connector file).
          return;
        }
        logger.debug("paste: Old fq = '" + oldt + "'");
        var json = jsoninput.value;
        if ( json == "" ) {
          logger.debug("paste: Nothing to paste, will not do");
          return;
        }
        tst.setArg(fldname, json);
        Components.utils.import('resource://indexdata/ui/taskPane.js');
        //taskPane.refreshData();
        taskPane.refreshArguments();
        return;
      }
    }
  }

  pastebut.addEventListener("click", function(e) {
    pasteExample(jsoninput.value);
  }, false );

  var jsonchange = function (e) {
    var value;
    try {
      value = JSON.parse(jsoninput.value);
    } catch (e) {
      qry.value = "" + e ;
      qry.setAttribute("class", "incomplete")
      return;
    }
    try {
      qry.value = context.qnode(value);
    } catch (e) {
      qry.value = "Error: " + e.message ;
      qry.setAttribute("class", "incomplete");
      return;
    }
    qry.setAttribute("class", "");
  };

  jsoninput.addEventListener("change", jsonchange, false);
  jsoninput.addEventListener("input", jsonchange, false);
  //jsoninput.addEventListener("keyup", jsonchange, false);

  var example = function ( box, title, json ) {
    var lbl = xmlHelper.appendNode(box, "label", title, {class:"text-link"} );
    lbl.addEventListener("click", function(e) {
      jsoninput.value = json;
      jsonchange();
      }, false );
    lbl.addEventListener("dblclick", function(e) {
      pasteExample(json);
      }, false );
  };

  var columns = xmlHelper.appendNode(vb, "hbox",null, {align:"top"} );
  var col = xmlHelper.appendNode(columns, "vbox", null, {width:captwidth});

  col = xmlHelper.appendNode(columns, "vbox", null, {flex:1,minwidth:captwidth} );
  example( col, "water", '{ "term":"water" }' );
  var row = xmlHelper.appendNode(col, "hbox");
  example( row, "au", '{ "term":"smith", "field":"author" }' );
  example( row, "ti", '{ "term":"yellow", "field":"title" }' );
  example( row, "kw", '{ "term":"water", "field":"keyword" }' );
  example( col, "peerreviewed",
           '{"op":"and", ' +
           ' "s1":{"term":"water"},'+
           ' "s2":{"field":"peerreviewed","term":"1"}}' );
  example( col, "fulltext",
           '{"op":"and", ' +
           ' "s1":{"term":"water"},'+
           ' "s2":{"field":"fulltext","term":"1"}}' );

  col = xmlHelper.appendNode(columns, "vbox", null, { flex:1 });
  row = xmlHelper.appendNode(col, "hbox");
  example( row, "author and title",
          '{ "op":"and", ' +
          ' "s1": { "term":"smith", "field":"author" }, ' +
          ' "s2": { "term":"yellow", "field":"title" } }' );
  example( row, "or",
          '{ "op":"or", ' +
          ' "s1": { "term":"smith", "field":"author" }, ' +
          ' "s2": { "term":"yellow", "field":"title" } }' );
  example( row, "not",
          '{ "op":"not", ' +
          ' "s1": { "term":"smith", "field":"author" }, ' +
          ' "s2": { "term":"yellow", "field":"title" } }' );
  example( col, "red and (yellow and blue)",
          '{ "op":"and", ' +
          ' "s1": { "term":"red" }, ' +
          ' "s2": { "op":"and", ' +
             ' "s1": { "term":"yellow" }, '+
             ' "s2": { "term":"blue" } } }' );
  example( col, "R and (Y and (B and G))",
          '{ "op":"and", ' +
          ' "s1": { "term":"red" }, ' +
          ' "s2": { "op":"and", ' +
             ' "s1": { "term":"yellow" }, '+
             ' "s2": { "op":"and", '+
               ' "s1": { "term":"blue" }, '+
               ' "s2": { "term":"green" } } } }' );
  example( col, "(R or Y) and (B and G)",
          '{ "op":"and", ' +
          ' "s1": { "op":"or", ' +
            ' "s1": { "term":"red" }, ' +
            ' "s2": { "term":"yellow" } }, '+
          ' "s2": { "op":"and", ' +
               ' "s1": { "term":"blue" }, '+
               ' "s2": { "term":"green" } } }' );

  col = xmlHelper.appendNode(columns, "vbox", null, { flex:1 });
  example( col, "ordered proximity",
            '{ "op":"prox", "distance":3, ' +
            ' "s1":{"term":"dylan"}, '+
            ' "s2":{"term":"zimmerman"}} ' );
  row = xmlHelper.appendNode(col, "hbox");
  example( row, "unordered",
            '{ "op":"prox", "distance":3, "ordered":false, ' +
            ' "s1":{"term":"dylan"}, '+
            ' "s2":{"term":"zimmerman"}} ' );
  example( row, "(all params)",
            '{ "op":"prox", "exclusion":false,  "distance":3, '+
            ' "ordered":true, "relation":"le",  "unit":"word", '+
            ' "s1":{"term":"dylan"}, ' +
            ' "s2":{"term":"zimmerman"} } ' );

  row = xmlHelper.appendNode(col, "hbox");
  example( row, "word", '{ "term":"water", "structure":"word" }' );
  example( row, "phrase", '{ "term":"water", "structure":"phrase" }' );

  col = xmlHelper.appendNode(columns, "vbox", null, { flex:1 });  // next column
  row = xmlHelper.appendNode(col, "hbox");
  example( row, "truncation: R",
          '{ "term":"water", "truncation":"right" }' );
  example( row, "L",
          '{ "term":"water", "truncation":"left" }' );
  row = xmlHelper.appendNode(col, "hbox");
  example( row, "L+R",
          '{ "term":"water", "truncation":"both" }' );
  example( row, "masking",
          '{ "term":"w?er", "truncation":"z39.58" }' );

  //col = xmlHelper.appendNode(columns, "vbox", null, { flex:1 });  // next column
  example( col, "year range",
           '{ "op":"and", ' +
           ' "s1": { "term":"smith", "field":"author" }, ' +
           ' "s2": { "op":"and", ' +
           ' "s1": { "term":"2000", "field":"year", "relation":"ge" },' +
           ' "s2": { "term":"2009", "field":"year", "relation":"le" } } }' );
  row = xmlHelper.appendNode(col, "hbox");
//  example( col, "au and year = 2005",
  example( row, "=",
           '{ "op":"and", ' +
           ' "s1": { "term":"smith", "field":"author" }, ' +
           ' "s2": { "term":"2005", "field":"year", "relation":"eq" } }' );
//  example( col, "au and year >= 2000",
  example( row, ">=",
           '{ "op":"and", ' +
           ' "s1": { "term":"smith", "field":"author" }, ' +
           ' "s2": { "term":"2000", "field":"year", "relation":"ge" } }' );
//  example( col, "au and year <= 2009",
  example( row, "<=",
           '{ "op":"and", ' +
           ' "s1": { "term":"smith", "field":"author" }, ' +
           ' "s2": { "term":"2009", "field":"year", "relation":"le" } }' );



} // drawExampleTab


/////////////////////
// Run

// Helper to extract an attribute value
Fullquery.prototype.attrValue = function (node, attrnum) {
    logger.error("OOPS, still calling attrValue with " + attrnum );
    // Should not happe, we pass named strings in the json, not attributes!
    if ( node.attributes )
        for ( var i in node.attributes ) {
            if ( node.attributes[i].attribute == attrnum )
                return node.attributes[i].value;
        }
    return "";
}

// Get the field and translate it to something like 'ti='
Fullquery.prototype.qfield = function (node) {
    var fld = node.field;
    if ( fld ) {
        if ( this.conf['fields'][fld] != undefined )
            return this.conf['fields'][fld];
        if ( this.conf['failbadfield'] )
          throw new StepError("Unsupported index " + fld );
    }
    if ( this.conf['defaultfield'] != undefined &&
         this.conf['defaultfield'] != unsupported_value )
        return( this.conf['defaultfield']);
    if ( this.conf['failbadfield'] || this.conf['defaultfield'] == unsupported_value )
      throw new StepError("Unspecified index not supported");
    return "";
}

// Get the term, and convert structure and trunc attributes
Fullquery.prototype.qtermstr = function (node) {
    queryHelper.validate_position(node); // Check the simple things
    queryHelper.validate_completeness(node);
    var rawterm = queryHelper.trunc_term(node, this.conf["trunc_wildcard"],
       this.conf['trunc_left'],this.conf['trunc_right'],
       this.conf['trunc_both'], this.conf['trunc_mask'] );
    var quotedterm = queryHelper.quote_string(node, rawterm,
         (this.conf['quote_phrase_L'] != unsupported_value),
         this.conf['quote_phrase_L'], this.conf['quote_phrase_R'],
         (this.conf['quote_word_L'] != unsupported_value),
         this.conf['quote_word_L'], this.conf['quote_word_R'],
         (this.conf['quote_default_L'] != unsupported_value),
         this.conf['quote_default_L'], this.conf['quote_default_R'] );
    return quotedterm;
}

Fullquery.prototype.qterm = function (node) {
//    dump("fq:qterm " + node.term + "\n");
    var fld = this.qfield(node);
    var term = this.qtermstr(node);
    if ( fld.indexOf("XX") == -1 )
      return fld + term;  // 'ti=' -> 'ti=hamlet'
    else
      return fld.replace("XX",term); // '(ti=XX)' -> '(ti=hamlet')
}

Fullquery.prototype.qop = function (node) {
    var confindex = "op_" + node.op;
    var dispname = "Operator '" + node.op + "'";
    var proxdist = "";
    if ( node.op == "prox" ) {
      if ( (node.ordered==undefined) || node.ordered ) {
        confindex = "op_prox_ord";
        dispname = "Ordered proximity";
      } else {
        confindex = "op_prox_unord";
        dispname = "Unordered proximity";
      }
      if ( node.distance == undefined ) {
        throw new StepError("Proximity without distance specified");
      }
      proxdist = "" + node.distance;
      // Check the other parameters, accept undefined, or the right
      // value, but nothing else
      if ( node.exclusion != undefined &&
           node.exclusion != false )
        throw new StepError("Proximity only supports exclusion=false (0)");
      if ( node.relation != undefined &&
          node.relation != "le" )
        throw new StepError("Proximity only supports relation=le (2)");
      if ( node.unit != undefined &&
          node.unit != "word" )
        throw new StepError("Proximity only supports unit=word (2)");
    }
    if ( this.conf[confindex] == unsupported_value ||
         !this.conf[confindex])
      throw new StepError(dispname +  " not supported");
    var operator = this.conf[ confindex ];
    operator = operator.replace("%DIST%",proxdist);

    if ( node.s1 && node.s2 ) {
      var xx = this.qnode(node.s1);
      var yy = this.qnode(node.s2);
      var result;
      if ( operator.indexOf("XX") == -1 &&
            operator.indexOf("YY") == -1 ) {
        result = xx + operator + yy;
      } else if ( operator.indexOf("XX") != -1 &&
            operator.indexOf("YY") != -1 ) {
        operator = operator.replace("XX",xx);
        operator = operator.replace("YY",yy);
        result = operator;
      } else {
        throw new StepError("Invalid operator '" +
          operator + "', need to have both XX and YY, or none of them");
      }
      return result;
    } else { // a listquerty node without child nodes
      return operator;
    }
} // qop



// Range of one or two nodes, with relations
// n2 is optional
Fullquery.prototype.range = function (n1,n2) {
  var endpoints = { start:"", end:"" };
  queryHelper.range_endpoint(n1,endpoints);
  queryHelper.range_endpoint(n2,endpoints);
  if ( (endpoints.start || endpoints.end) &&
       ( this.conf['rangefields'].indexOf(n1.field) == -1 ) ) {
      //logger.info("Unsupported range '" + endpoints.start + "' - '" + endpoints.end + "'");
      throw new StepError( "Ranges not supported for " + n1.field );
  }
  var r = "";
  if ( endpoints.start && endpoints.end ) {
    r = this.conf['range_BE'];
  } else if ( endpoints.start ) {
    r = this.conf['range_B'];
  } else if ( endpoints.end ) {
    r = this.conf['range_E'];
  }
  //logger.debug("BUG: r=" + r + " ep='" + endpoints.start + "'-'" + endpoints.end+"'" +
  //  " BE=" + this.conf['range_BE'] + " B=" + this.conf['range_B'] +
  //  " E=" + this.conf['range_E'] );
  r = r.replace("XX", endpoints.start);
  r = r.replace("YY", endpoints.end);
  //r = r.replace("FF", this.qfield(n1) );
  if ( r.indexOf("FF") != -1 ) {  // look at field def
    var ff = this.qfield(n1);
    //logger.debug("FQ: FF substitution: r='" + r + "' ff='" + ff + "'" );
    r = r.replace("FF",""); // tricky rule like yr=(XX), substitute
    if ( ff.indexOf("XX") == -1 )
      r = ff + r; // simple prefix, like 'yr=' prefixed to 2001-2009
    else
      r = ff.replace("XX",r); // as yr=(2001-2009)
  }
  return r;
}

// Special case, a range of (typically) years
// must be an AND node with term children, who have relations specified
// If not, returns an empty string.
Fullquery.prototype.range_node = function (node) {
  if (queryHelper.is_range_node(node) ) {
    return this.range ( node.s1, node.s2 );
  }
  if (queryHelper.is_relation_node(node) ) {
    return this.range ( node );
  }
  return "";
}

Fullquery.prototype.qnode = function (node) {
    //dump("fq:qnode \n");
    var range = this.range_node(node);
    if (range)
      return range;
    if ( node.term )
        return this.qterm(node);
    if (node.op)
        return this.qop(node);
    logger.error("Fullquery.node called without 'term' nor 'op': " +
      "node=" + node + " t=" + typeof(node)+ " json:" + JSON.stringify(node) );
    throw new StepError("Internal error (bad json node in query: need term or op)");
    //dump("FieldTab: context: " + JSON.stringify( context.conf ) + "\n");
}

// A listquery
// This is unnecessarily complicated, and fails with range things
// that are more than 2 nodes (yr>=2000 && yr>=2005 && yr<=2009)
// TODO - How should range stuff be handled ?? Is this enough,
// if fullquerylimit (and possibly listqueryelement) will always
// produce normalized ranges?
Fullquery.prototype.qlist = function (node) {
  //logger.info("fq: is list of " + node.length + " elements "+
  //  "is_relation: " + queryHelper.is_relation_node(node[0])  );
  var range="";
  if ( node.length == 1 && queryHelper.is_relation_node(node[0]) ) {
    //logger.info("fq:lq 1 rel = " + node[0].relation );
    range = this.range(node[0]);
  }else if ( node.length == 2 &&
       queryHelper.is_relation_node(node[0]) &&
       queryHelper.is_relation_node(node[1]) &&
       node[0].field == node[1].field )
    range = this.range(node[0],node[1]);
  if (range)
    return range;
  var s = "";
  for ( var i=0; i<node.length;i++) {
    var n = node[i];
/*    
    if ( n.op && s != "" )  // no op in beginning of a string! (bug 5191)
      s += this.qop(n);
    s += this.qterm(n);
*/
    var op = "";
    if ( s != "" ) //no op in beginning of a string! (bug 5191)
      op = this.qop(n); 
    if ( op.indexOf("XX") != -1 && op.indexOf("YY") != -1 ) { // (XX and YY)
      op = op.replace("XX",s );
      op = op.replace("YY",this.qterm(n) );
      s = op;
    } else { // regular op, just append
      s += op;
      s += this.qterm(n);
    }
  }
  return s;
}

Fullquery.prototype.run = function (task) {
    //var value = jsonPathHelper.getFirst(this.conf.in, task.data);
    var value = jsonPathHelper.get(this.conf.in, task.data);
    task.debug("fq.run: first fq:" + value + " t=" + typeof(value) +
          " json: '" + JSON.stringify(value) + "'");
    if ( Array.isArray(value) && value.length > 0 ) {
      task.debug("fq: Got an array of " + value.length + " elements, taking the [0]");
      value = value[0];
    }

    if ( Array.isArray(value) && value[0] &&
      ( typeof(value[0].op) != "string" ||
        typeof(value[0].term) != "string" )  ) {
        // Only a listquery can have both op and term!)
      task.debug("fq: Still an array of " + value.length + " elements, " +
          " that looks like a real fullquery. Taking [0] ");
      value = value[0];
    }
    if ( Array.isArray(value) && value[0] &&
       ( typeof(value[0].op) != "string" ||
         typeof(value[0].term) != "string" )  ) {
         // Only a listquery can have both op and term!)
      task.debug("fq: Still an array of " + value.length + " elements, " +
                 " that looks like a real fullquery. Taking [0] ");
      value = value[0];
    }

    task.debug("fq.run: final fq:" + value + " t=" + typeof(value) +
          " json: '" + JSON.stringify(value) + "'");
    var qstring="";
    if ( value == false || value == undefined ) {
      if ( this.conf["failnoqry"] ) {
        throw new StepError("Fullquery called without a query!");
      task.debug("fq with no query, but that's ok");
      }
    } else {
      task.debug("fq: Type of fq is " + typeof(value) + " isarray:" +
          Array.isArray(value) + " len=" +  value.length);
      if ( Array.isArray(value) ) {
        task.debug("fq.run: processing listquery" );
        qstring = this.qlist(value);
      } else {
        qstring = this.qnode(value);
      }
    }
    task.info("fullquery setting out=" +
       "'" + this.conf.out.path + "." + this.conf.out.key + "' " +
       "to '" + qstring + "'");
    jsonPathHelper.set(this.conf.out, [qstring], task.data);
};

///////////////////
// Unit test
// This is too messy to test with real connectors (although such testing
// is also needed)
//   cd .../cf/engine/src; make unittest
// After that, you can simply
//   ./cfrun -u steps/fullquery/fullquery

Fullquery.prototype.unitTest = function ( ) {
  const defaultsettings = {  // These can be overwritten in the tests
      defaults: "ccl",
      fields: { title:"ti=", author:"au=", year:"yr=", fooyear:"yr=(XX)" },
      defaultfield: "any=",
      failbadfield: false,
      op_and: " and ",   op_or: " or ",   op_not: " not ",
      op_prox_ord: "#UNSUPPORTED#",  op_prox_unord: "#UNSUPPORTED#",
      quote_word_L: "'", quote_word_R: "'",
      quote_phrase_L: '"', quote_phrase_R: '"',
      quote_default_L: "", quote_default_R: "",
      trunc_wildcard: "*",
      trunc_right: true, trunc_left: true, trunc_both: true, trunc_mask: true,
      rangefields: "year fooyear",
      range_B:"FF[XX-]", range_BE:"FF[XX-YY]", range_E:"FF[-YY]",
  }

  const tests = [
    { name: "Simple",
      override: {}, // no overrides to default settings
      capflags: [ "index-title", "index-author", "index-year",
                  "query-and", "query-or", "query-not",  "query-phrase",
                  "trunc-right", "trunc-left", "trunc-both", "query-wildcard",
                  "trunc-asterisk" ],
      notcapflags: [ "index-publisher", "query-prox", "trunc-questionmark" ],
      lines: [
        { id: "Simple term without any trickery",
          fq: '{ "term":"water" }',
          ex: 'any=water',
        },
        { // structure
          fq: '{ "term":"water", "structure":"word" }',
          ex: "any='water'",
        },
        {
          fq: '{ "term":"water", "structure":"phrase" }',
          ex: 'any="water"',
        },
        { // Simple term with a use attribute
          fq: '{ "term":"water", "field":"title" }',
          ex: 'ti=water',
        },
        {
          fq: '{ "term":"water", "field":"author" }',
          ex: 'au=water',
        },
        { // unknown field, defaults to any, since failbadfield is false
          // see below in unsupported stuff
          fq: '{ "term":"water", "field":"totallyunknownindex" }',
          ex: 'any=water',
        },
      ],
    },
    { name: "Unsupported",
      override: { failbadfield: true, defaultfield: unsupported_value,
        quote_phrase_L: unsupported_value, quote_word_L: unsupported_value,
      },
      capflags: [ "index-title" ],
      notcapflags: [ "quote-phrase" ],
      lines: [
        { // Simple term with field
          fq: '{ "term":"water", "field":"title" }',
          ex: 'ti=water',
        },
        { // Unknown field should fail
          fq: '{ "term":"water", "field":"totallyunknownindex" }',
          ex: "Exception! Unsupported index totallyunknownindex",
        },
        { // Simple term without field specified
          fq: '{ "term":"water" }',
          ex: "Exception! Unspecified index not supported",
        },
        { // Phrase, not supported
          fq: '{ "term":"water", "field":"title", "structure":"phrase" }',
          ex: "Exception! Structure 'phrase' not supported",
        },
        { // Word, not supported
          fq: '{ "term":"water", "field":"title", "structure":"word" }',
          ex: "Exception! Structure 'word' not supported",
        },
      ],
    },
    { name: "Position/Completeness",
      override: {defaultfield:""},
      lines: [
        { // Support the default
          fq: '{ "term":"water", "field":"title", ' +
               ' "completeness":"incompletesubfield" }',
          ex: 'ti=water',
        },
        { // Support the default
          fq: '{ "term":"water", "field":"title", ' +
               ' "position":"any" }',
          ex: 'ti=water',
        },
        { // And fail anything else
          fq: '{ "term":"water", "field":"title", '+
               ' "completeness":"completesubfield" }',
          ex: "Exception! Completeness not supported",
        },
        {
          fq: '{ "term":"water", "field":"title", ' +
               ' "completeness":"completefield" }',
          ex: "Exception! Completeness not supported",
        },
        {
          fq: '{ "term":"water", "field":"title", ' +
               ' "position":"firstinfield" }',
          ex: "Exception! Position not supported",
        },
        {
          fq: '{ "term":"water", "field":"title", ' +
               ' "position":"firstinsubfield" }',
          ex: "Exception! Position not supported",
        },
      ],
    },
    { name: "Listquery",
      override: {defaultfield:""},
      lines: [
        { // Simple term without any trickery
          fq: '[{ "term":"water" }]',
          ex: 'water',
        },
        { // Simple list with two terms
          fq: '[{ "term":"water" },'+
               '{ "term":"fire", "op":"and" }]',
          ex: 'water and fire',
        },
        { // Simple list with two terms. Operator also in the first
          fq: '[{ "term":"water", "op":"and" },'+
               '{ "term":"fire", "op":"and" }]',
          ex: 'water and fire', // NOT: and water and fire
        },
        { // Longer list
          fq: '[{ "term":"one" },'+
               '{ "term":"two", "op":"and" },'+
               '{ "term":"three", "op":"or" },'+
               '{ "term":"four", "op":"not" },'+
               '{ "term":"five", "op":"and" },'+
               '{ "term":"six", "op":"or" },'+
               '{ "term":"seven", "op":"not" } ]',
          ex: 'one and two or three not four and five or six not seven',
        },
      ],
    },
    { name: "Trunc",
      override: {}, // no overrides to default settings
      lines: [
        {
          fq: '{ "term":"water", "truncation":"right" }',
          ex: 'any=water*',
        },
        {
          fq: '{ "term":"water", "truncation":"left" }',
          ex: 'any=*water',
        },
        {
          fq: '{ "term":"water", "truncation":"both" }',
          ex: 'any=*water*',
        },
        {
          fq: '{ "term":"water", "truncation":"none" }',
          ex: 'any=water',
        },
        {
          fq: '{ "term":"water", "truncation":"mask" }',
          ex: "Exception! Truncation 'mask' not supported",
        },
        {
          fq: '{ "term":"water", "truncation":"z39.58" }',
          ex: 'any=water',
        },
        {
          fq: '{ "term":"wa#er", "truncation":"z39.58" }',
          ex: "Exception! Single-character masking with '#' not supported",
        },
        {
          fq: '{ "term":"wa?r", "truncation":"z39.58" }',
          ex: 'any=wa*r',
        },
        {
          fq: '{ "term":"wa?2r", "truncation":"z39.58" }',
          ex: 'any=wa*2r',
        },
        {
          fq: '{ "term":"w?t?r", "truncation":"z39.58" }',
          ex: 'any=w*t*r',
        },
      ],
    },
    { name: "Years",
      override: {}, // no overrides to default settings
      capflags: [ "index-year", "index-startyear", "index-endyear" ],
      notcapflags: [ "quote-phrase" ],
      lines: [
      { // author and year >= 2000",
        fq: '{ "op":"and", ' +
            ' "s1": { "term":"shakespeare", "field":"author" }, ' +
            ' "s2": { "term":"2000", "field":"year", "relation":"ge" } }',
        ex: "au=shakespeare and yr=[2000-]",
      },
      { // author and year <= 2000",
        fq: '{ "op":"and", ' +
            ' "s1": { "term":"shakespeare", "field":"author" }, ' +
            ' "s2": { "term":"2000", "field":"year", "relation":"le" } }',
        ex: "au=shakespeare and yr=[-2000]",
      },
      { // author and year < 2000",
        fq: '{ "op":"and", ' +
            ' "s1": { "term":"shakespeare", "field":"author" }, ' +
            ' "s2": { "term":"2000", "field":"year", "relation":"lt" } }',
        ex: "au=shakespeare and yr=[-1999]",
      },
      { // author and year < non-numerical",
        fq: '{ "op":"and", ' +
            ' "s1": { "term":"shakespeare", "field":"author" }, ' +
            ' "s2": { "term":"Y2K", "field":"year", "relation":"lt" } }',
        ex: "Exception! year must be numerical when comparing with 'lt'. ('Y2K' is not)",
      },
      { // author and year < non-numerical",
        fq: '{ "op":"and", ' +
          ' "s1": { "term":"shakespeare", "field":"author" }, ' +
          ' "s2": { "term":"Y2K", "field":"year", "relation":"le" } }',
        ex: "au=shakespeare and yr=[-Y2K]",
      },
      { // author and year with bad operator",
        fq: '{ "op":"and", ' +
          ' "s1": { "term":"shakespeare", "field":"author" }, ' +
          ' "s2": { "term":"Y2K", "field":"year", "relation":"XX" } }',
        ex: "Exception! Unsupported relation 'XX'",
      },
      { // author and date range
        fq: '{ "op":"and", ' +
                '"s1": { "term":"shakespeare", "field":"author" }, ' +
                '"s2": { "op":"and", ' +
                  '"s1": { "term":"2000", "field":"year", "relation":"ge" },' +
                  '"s2": { "term":"2009", "field":"year", "relation":"le" }}}',
        ex: "au=shakespeare and yr=[2000-2009]",
      },
      { // author and two endpoints that can be combined
        fq: '{ "op":"and", ' +
            '"s1": { "term":"shakespeare", "field":"author" }, ' +
            '"s2": { "op":"and", ' +
              '"s1": { "term":"2000", "field":"year", "relation":"lt" },' +
              '"s2": { "term":"2009", "field":"year", "relation":"le" }}}',
        //ex: "Exception! Range with two end values",
              // now we combine them, as long as valid
        ex: "au=shakespeare and yr=[-1999]",
      },
      { // author and two years that could be combined
        fq: '{ "op":"and", ' +
            '"s1": { "term":"shakespeare", "field":"author" }, ' +
            '"s2": { "op":"and", ' +
              '"s1": { "term":"2005", "field":"year", "relation":"eq" },' +
              '"s2": { "term":"2005", "field":"year", "relation":"eq" }}}',
        ex: "au=shakespeare and yr=2005 and yr=2005",
        // TODO - is this correct? technically yes, but it might make
        // sense to combine them.
      },
      { // author and two different years
        fq: '{ "op":"and", ' +
            '"s1": { "term":"shakespeare", "field":"author" }, ' +
            '"s2": { "op":"and", ' +
              '"s1": { "term":"2005", "field":"year", "relation":"eq" },' +
              '"s2": { "term":"2007", "field":"year", "relation":"eq" }}}',
        ex: "au=shakespeare and yr=2005 and yr=2007",
        // TODO - is this correct? technically yes, but it might make
        // sense to refuse the whole thing. Use fullquerylimit for this
        // kind of stuff!
      },
      { // author, year range, and individual year
        fq: '{ "op":"and", ' +
            '"s1": { "term":"shakespeare", "field":"author" }, ' +
            '"s2": { "op":"and", ' +
              '"s1": { "term":"2005", "field":"year", "relation":"eq" }, '+
              '"s2": { "op":"and", ' +
                '"s1": { "term":"2000", "field":"year", "relation":"ge" },' +
                '"s2": { "term":"2009", "field":"year", "relation":"le" }}}}',
        ex: "au=shakespeare and yr=2005 and yr=[2000-2009]",
        // This doesn't make much sense, but it is a proper transformation.
        // We could combine a bit more, but that goes against the flow in fq
        // Use a fullquerylimit instead!
      },
      { // bug 5714 (flase alarm), but a good test case. Range in funny order
        fq: '{ "op":"and", ' +
            '"s1": { "op":"and", ' +
              '"s1": { "term":"water", "field":"keyword" },' +
              '"s2": { "term":"2000", "field":"year", "relation":"ge" }},' +
            '"s2": { "term":"2009", "field":"year", "relation":"le" } }',
        ex: "any=water and yr=[2000-] and yr=[-2009]",
          // Another unlikley transormation, but not unreasonable, especially
          // if there had been parentheses around the ands, and proper
          // operators for ge/le...
      },
      { // Field that does not support ranges
        fq: '{ "op":"and", ' +
          ' "s1": { "term":"shakespeare", "field":"author" }, ' +
          ' "s2": { "term":"2000", "field":"isbn", "relation":"ge" } }',
        ex: "Exception! Ranges not supported for isbn",
      },
      { id: "list query with one element, ge",
        fq: '[ { "op":"", "term":"2000", "field":"year", "relation":"ge" } ]',
        ex: "yr=[2000-]",
      },
      { id: "list query with one element, eq",
        fq: '[ { "op":"", "term":"2000", "field":"year", "relation":"eq" } ]',
        //ex: "yr=[2000-2000]",   //used to be like this, but that was wrong!
        ex: "yr=2000",
      },
      { id: "list query with two elements, ge-le",
        fq: '[ { "op":"", "term":"2000", "field":"year", "relation":"ge" }, '+
               '{ "op":"and", "term":"2009", "field":"year", "relation":"le" } ]',
        ex: "yr=[2000-2009]",
      },
      { id: "list query with one eq-element, un-ranged field",
        fq: '[ { "op":"", "term":"2000", "field":"title", "relation":"eq" } ]',
        ex: "ti=2000",
      },
      { id: "list query with one eq-element, unsupported field, defaults to 'any'",
        fq: '[ { "op":"", "term":"2000", "field":"totallyunknownindex", "relation":"eq" } ]',
        ex: "any=2000",  // See totallyunknownindex examples above
      },
      { // author and date range, with FF in range and XX in field format
        fq: '{ "op":"and", ' +
                '"s1": { "term":"shakespeare", "field":"author" }, ' +
                '"s2": { "op":"and", ' +
                  '"s1": { "term":"2000", "field":"fooyear", "relation":"ge" },' +
                  '"s2": { "term":"2009", "field":"fooyear", "relation":"le" }}}',
        ex: "au=shakespeare and yr=([2000-2009])",
      },
      ]
    },
    { name: "Operators",
      override: {
        op_and: "(XX and YY)",
        op_or: "(XX or YY)",
        op_not: "(XX not YY)",
        defaultfield:"",
      },
      lines: [
      { // simple two-operand AND
        fq: '{ "op":"and", ' +
            ' "s1": { "term":"shakespeare", "field":"author" }, ' +
            ' "s2": { "term":"hamlet", "field":"title" } }',
        ex: "(au=shakespeare and ti=hamlet)",
      },
      { // simple two-operand OR
        fq: '{ "op":"or", ' +
            ' "s1": { "term":"shakespeare", "field":"author" }, ' +
            ' "s2": { "term":"hamlet", "field":"title" } }',
        ex: "(au=shakespeare or ti=hamlet)",
      },
      { // simple two-operand NOT
        fq: '{ "op":"or", ' +
            ' "s1": { "term":"shakespeare", "field":"author" }, ' +
            ' "s2": { "term":"hamlet", "field":"title" } }',
        ex: "(au=shakespeare or ti=hamlet)",
      },
      { // 4-way AND
        fq: '{ "op":"and", ' +
              ' "s1": { "op":"and", ' +
                ' "s1": { "term":"kernighan", "field":"author" }, ' +
                ' "s2": { "term":"rithchie", "field":"author" } }, '+
              ' "s2": { "op":"and", ' +
                ' "s1": { "term":"programming", "field":"title" }, ' +
                ' "s2": { "term":"language", "field":"title" } } } ',
        ex: "((au=kernighan and au=rithchie) and (ti=programming and ti=language))",
      },
      { // 4-way OR
        fq: '{ "op":"or", ' +
              ' "s1": { "op":"or", ' +
                ' "s1": { "term":"kernighan", "field":"author" }, ' +
                ' "s2": { "term":"rithchie", "field":"author" } }, '+
              ' "s2": { "op":"or", ' +
                ' "s1": { "term":"programming", "field":"title" }, ' +
                ' "s2": { "term":"language", "field":"title" } } } ',
        ex: "((au=kernighan or au=rithchie) or (ti=programming or ti=language))",
      },
      { // 4-way NOT
        fq: '{ "op":"not", ' +
              ' "s1": { "op":"not", ' +
                ' "s1": { "term":"kernighan", "field":"author" }, ' +
                ' "s2": { "term":"rithchie", "field":"author" } }, '+
              ' "s2": { "op":"not", ' +
                ' "s1": { "term":"programming", "field":"title" }, ' +
                ' "s2": { "term":"language", "field":"title" } } } ',
        ex: "((au=kernighan not au=rithchie) not (ti=programming not ti=language))",
      },
      { // Listquery input, XXYY operators
          fq: '[{ "term":"water", "field":"title" },'+
               '{ "term":"fire", "field":"title", "op":"and" }]',
          ex: '(ti=water and ti=fire)',
      },
      { // Listquery input, with an op in the first (bug 5191)
          fq: '[{ "term":"water", "field":"title", "op":"not" },'+
               '{ "term":"fire", "field":"title", "op":"and" }]',
          ex: '(ti=water and ti=fire)',
      },
      { // Longer list
        fq: '[{ "term":"one" },'+
              '{ "term":"two", "op":"and" },'+
              '{ "term":"three", "op":"or" },'+
              '{ "term":"four", "op":"not" },'+
              '{ "term":"five", "op":"and" },'+
              '{ "term":"six", "op":"or" },'+
              '{ "term":"seven", "op":"not" } ]',
        ex: '((((((one and two) or three) not four) and five) or six) not seven)',
      },
       
      ]
    },

    { name: "Prox",
      override: {
          op_prox_ord: "(XX prox-%DIST% YY)",
          op_prox_unord: "(XX prox-%DIST%-un YY)",
      },
      capflags: [ "query-prox" ],
      notcapflags: [ "quote-phrase" ],
      lines: [
      { // simple ordered prox
        fq: '{ "op":"prox", "distance":3, ' +
            ' "s1": { "term":"shakespeare", "field":"author" }, ' +
            ' "s2": { "term":"hamlet", "field":"title" } }',
        ex: "(au=shakespeare prox-3 ti=hamlet)",
      },
      { // simple un-ordered prox
        fq: '{ "op":"prox", "distance":3, "ordered":false, ' +
            ' "s1": { "term":"shakespeare", "field":"author" }, ' +
            ' "s2": { "term":"hamlet", "field":"title" } }',
        ex: "(au=shakespeare prox-3-un ti=hamlet)",
      },
      { // simple ordered prox (dist as string)
        fq: '{ "op":"prox", "distance":"3", "ordered":true, ' +
            ' "s1": { "term":"shakespeare", "field":"author" }, ' +
            ' "s2": { "term":"hamlet", "field":"title" } }',
        ex: "(au=shakespeare prox-3 ti=hamlet)",
      },
      { // Ordered prox, all parameters at default values
        fq: '{ "op":"prox", "exclusion":false,  "distance":3, ' +
                  ' "ordered":true, "relation":"le",  "unit":"word",  ' +
            ' "s1": { "term":"shakespeare", "field":"author" }, ' +
            ' "s2": { "term":"hamlet", "field":"title" } }',
        ex: "(au=shakespeare prox-3 ti=hamlet)",
      },
      { // bad exclusion
        fq: '{ "op":"prox", "exclusion":true,  "distance":3, ' +
                  ' "ordered":true, "relation":"le",  "unit":"word",  ' +
            ' "s1": { "term":"shakespeare", "field":"author" }, ' +
            ' "s2": { "term":"hamlet", "field":"title" } }',
        ex: "Exception! Proximity only supports exclusion=false (0)",
      },
      { // bad relation
        fq: '{ "op":"prox", "exclusion":false,  "distance":3, ' +
                  ' "ordered":true, "relation":"lt",  "unit":"word",  ' +
            ' "s1": { "term":"shakespeare", "field":"author" }, ' +
            ' "s2": { "term":"hamlet", "field":"title" } }',
        ex: "Exception! Proximity only supports relation=le (2)",
      },
      { // bad unit
        fq: '{ "op":"prox", "exclusion":false,  "distance":3, ' +
                  ' "ordered":true, "relation":"le",  "unit":"paragraph",  ' +
            ' "s1": { "term":"shakespeare", "field":"author" }, ' +
            ' "s2": { "term":"hamlet", "field":"title" } }',
        ex: "Exception! Proximity only supports unit=word (2)",
      },
      ]
    },

  ];

  logger.info("Starting unit test for Fullquery");
  // fake some runtime
  var connector = new Connector();
  var task = new Task( connector, "unittest" );
  for ( var i=0; i<tests.length; i++) {
    var t=tests[i];
    var fq = new Fullquery;
    fq.task = task;
    for ( var f in defaultsettings ) {
      fq.conf[f] = defaultsettings[f];
    }
    for ( var f in t.override ) {
      fq.conf[f] = t.override[f];
    }
    for ( var c in t.capflags ) {
      if ( ! fq.capabilityFlagDefault( t.capflags[c]) ) {
        logger.error("test " + t.name + " FAILED" );
        logger.error("Capability flag '" + t.capflags[c] + "'" +
            " should be set, but is not");
        return false;
      }
    }
    for ( var c in t.notcapflags ) {
      if ( fq.capabilityFlagDefault( t.notcapflags[c]) ) {
        logger.error("test " + t.name + " FAILED" );
        logger.error("Capability flag '" + t.notcapflags[c] + "'" +
            " should be not be set, but it is!");
        return false;
      }
    }
    for ( var lno = 0; lno<t.lines.length; lno++) {
      try {
        fullquery = JSON.parse(t.lines[lno].fq);
      } catch (e) {
        logger.error("" + e + " in test " + t.name + "." + lno  );
        logger.error(t.lines[lno].fq);
        return false;
      }
      var qstring = "";
      try {
        if ( Array.isArray(fullquery) ) {
          qstring = fq.qlist(fullquery);
        } else {
          qstring = fq.qnode(fullquery);
        }
      } catch (e) {
        qstring = "Exception! " + e.message ;
        if ( e.fileName )
          qstring += " in " + e.fileName + ":" + e.lineNumber ;
      }
      if ( qstring == t.lines[lno].ex ) {
        logger.info("test " + t.name + "." + lno + " OK");
      } else {
        var id = "";
        if ( t.lines[lno].id ) id = ' "' + t.lines[lno].id + '"';
        logger.error("test " + t.name + "." + lno + id + " FAILED");
        logger.error(t.lines[lno].fq);
        logger.error("got: " + qstring );
        logger.error("exp: " + t.lines[lno].ex);
        return false;
      }
    }
  }
  return true;
}; // unittest


//////////////////
// Misc

Fullquery.prototype.getUsedArgs = function () {
    if (this.conf.in && this.conf.in.path === "$.input")
        return [this.conf.in.key];
    else
        return [];
};

Fullquery.prototype.getClassName = function () {
    return "Fullquery";
};

Fullquery.prototype.getDisplayName = function () {
    return "Fullquery";
};

Fullquery.prototype.getDescription = function () {
  return "Translate a complex query into a search string";
};

Fullquery.prototype.getVersion = function () {
  return "2.0";
};

// Not much point in a renderArgs, way too complex to show in the builder
Fullquery.prototype.renderArgs = function () {
    if ( ! this.conf["defaults"] )
        return "";
    return this.conf["defaults"];
}

Fullquery.prototype.capabilityFlagDefault = function ( flag ) {
   var trunc_any = this.conf.trunc_left  || this.conf.trunc_right ||
                   this.conf.trunc_both || this.conf.trunc_mask;
    if (flag == "query-full") {
        return true;
    } else if (flag.substring(0, 6) == "index-") {
        var index = flag.substring(6);
        if ( typeof(this.conf.fields[index]) != "undefined")
          return true;
          // note, do not return false if not there, other steps may still
          // make use of it.
          if ( this.conf.rangefields &&
            this.conf.rangefields.indexOf(index) != -1 )
            return true; // any field mentioned in rangefields
          if ( index == "startyear" &&
               this.conf.rangefields &&
               this.conf.rangefields.indexOf("year") != -1 &&
               ( this.conf.range_B || this.conf.range_BE ) )
            return true;
          if ( index == "endyear" &&
               this.conf.rangefields &&
               this.conf.rangefields.indexOf("year") != -1 &&
               ( this.conf.range_BE || this.conf.range_E ) )
            return true;
          if ( flag == "index-keyword" &&
             /*this.conf.defaultfield && */
             this.conf.defaultfield != unsupported_value )
          return true; // unspecified index usually means a keyword.
    } else if (flag == "query-and") {
        if( this.conf.op_and != unsupported_value)
          return true;
    } else if (flag == "query-or") {
        if (this.conf.op_or != unsupported_value)
          return true;
    } else if (flag == "query-not") {
        if (this.conf.op_not != unsupported_value)
          return true;
    } else if (flag == "query-prox") {
      if (this.conf.op_prox_ord != unsupported_value ||
          this.conf.op_prox_unord != unsupported_value  )
        return true;
    } else if (flag == "query-phrase") {
        if (this.conf.quote_phrase_L != unsupported_value &&
            this.conf.quote_phrase_L != "" )
          return true;
    } else if (flag == "trunc-right") {
        if (this.conf.trunc_right )
          return true;
    } else if (flag == "trunc-left") {
        if (this.conf.trunc_left )
          return true;
    } else if (flag == "trunc-both") {
      if (this.conf.trunc_both )
        return true;
    } else if (flag == "query-wildcard") {
      if (this.conf.trunc_mask )
        return true;
    } else if (flag == "trunc-asterisk") {
        if (this.conf.trunc_wildcard == "*" && trunc_any)
          return true;
    } else if (flag == "trunc-questionmark" && trunc_any) {
        if (this.conf.trunc_wildcard == "?")
          return true;
    }
    return null;
}

Fullquery.prototype.upgrade = function (confVer, curVer, conf) {
    // can't upgrade if the connector is newer than the step
    if (confVer > curVer)
        return false;
    if (confVer < "0.2") {
        // Upgrade from old field defs
        //   Title : { name: 'title', str: 'ti=', aliases=["1","2"]}
        // Stupid javascript returns 'object' for arrays
        // so we check if it has a length method.
        if ( typeof(conf['fields']) == "object" &&
             conf['fields'].length ) {
            var o = conf['fields'];
            conf['fields'] = {};
            for (var k in o ) {
                var name = o[k].name;
                if ( name != unsupported_value &&
                     o[k].aliases.length>0 ) {
                    var a1 = o[k].aliases[0]; // just take the first
                    var str = o[k].str;
                    conf['fields'][a1]=str;
                }
                // ignore aliases
            }
        }
    }
    if (confVer < "0.3") {
        jsonPathHelper.upgradePostProc(this.conf);
    }
    if ( confVer < "1.1" ) {
      if ( conf['failnoqry'] == undefined ) {
        conf['failnoqry'] = true;
        conf['rangefields'] = "";
      }
    }
    if ( confVer < "1.2" ) {
      conf['trunc_right'] = false;
      conf['trunc_left'] = false;
      conf['trunc_both'] = false;
      conf['trunc_mask'] = false;
      if ( conf["trunc_style"] == "right" )
        conf['trunc_right'] = true;
      if ( conf["trunc_style"] == "left" )
        conf['trunc_left'] = true;
      if ( conf["trunc_style"] == "both" ) {
        conf['trunc_right'] = true;
        conf['trunc_left'] = true;
        conf['trunc_both'] = true;
      }
      delete conf['trunc_style'];
    }
    if ( confVer < "1.3" ) {
      if (conf['failbadfield'] == undefined )
        conf['failbadfield'] = false;
    }

    return true;
};

