var EXPORTED_SYMBOLS = ["Listqueryelement"];

// Extracts one or more elements from a listquery
// The result is same structure as the listquery
// Can optionally mark elements as used, and extract only unmarked elements

Components.utils.import('resource://indexdata/runtime/Connector.js');
Components.utils.import('resource://indexdata/runtime/Task.js');
Components.utils.import('resource://indexdata/runtime/Step.js');
Components.utils.import('resource://indexdata/runtime/StepError.js');

Components.utils.import('resource://indexdata/util/logging.js');
Components.utils.import('resource://indexdata/util/xulHelper.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/flatquery.js');
Components.utils.import('resource://indexdata/thirdparty/jsonPath.js');
Components.utils.import('resource://indexdata/util/jsonPathHelper.js');

var logger = logging.getLogger();
const anyindex="(any)";

// Constructor
var Listqueryelement = function () {
  this.conf = {};
  this.conf['in'] =  { path:'$.temp', key:'listquery' };
  this.conf['out'] = { path:'$.temp', key:'elements' };
  this.conf['clearused']= false;
  this.conf['markused'] = true;
  this.conf['onlyused'] = true;
  this.conf['index'] = anyindex;
  this.conf['numelements'] = "all";
  this.conf['failnothing'] = false;
  this.conf['failunused'] = false;
  this.conf['failnotanded'] = false;
  this.conf['failnotandonly'] = false;
  this.conf['failremain'] = false;
};

Listqueryelement.prototype = new Step();
Listqueryelement.prototype.constructor = Listqueryelement;
Listqueryelement.prototype.init = function() {};


///////////////////
// UI

Listqueryelement.prototype.draw = function(surface,win) {
  var vb = xmlHelper.appendNode(surface, "vbox" );
  //xulHelper.jsonPathMapField(vb, this, "in", "out");
  xulHelper.jsonPathField(vb, this, this.conf, "in", "Source " );
  xulHelper.jsonPathField(vb, this, this.conf, "out", "Output " );
  var ub = xmlHelper.appendNode(vb, "hbox",
                                null, {align: "center", flex: 1});
  xulHelper.captionField(ub,"Used elements" );
  xulHelper.checkbox(ub, this, "clearused", "Clear used flags before starting" );
  xulHelper.checkbox(ub, this, "markused", "Mark returned element(s) as used" );
  xulHelper.checkbox(ub, this, "onlyused", "Only return unused elements" );
  var ib = xmlHelper.appendNode(vb, "hbox",
                                null, {align: "center", flex: 1});
  xulHelper.captionField(ib,"Index" );
  var indexnames = [];
  var templ = this.task.getTemplate();
  logger.debug("templ=" + templ );
  var data = templ.properties.data;
  logger.debug("data=" + data );
  for ( var k in data ) {
    logger.debug(" data '" + k + "'" );
  }
  var inputs = templ.properties.data.input;
  logger.debug("inputs=" + inputs );
  for ( var inp in inputs ){
    indexnames.push( inp );
    logger.debug("found input =" + inp );
  }
  var extranames = [ anyindex ];
  xulHelper.arraySelectField(ib, this, "index", indexnames, extranames );

  xulHelper.captionField(ib,"  How many elements to return " );
  var numalternatives = { "only one": "1",
        "two": "2", "three": "3",
        "all that match": "all" };
  xulHelper.arraySelectField(ib, this, "numelements", numalternatives );

  var colbox = xmlHelper.appendNode(surface, "hbox",null, {flex:1});
  var leftside = xmlHelper.appendNode(colbox, "vbox");
  //xmlHelper.appendNode(colbox, "separator", null,
  //      {"class": "groove", orient:"vertical"}, null);
  var rightside = xmlHelper.appendNode(colbox, "vbox");
  xulHelper.checkbox(leftside, this, "failnothing",
               "Fail if nothing found" );
  xulHelper.checkbox(leftside, this, "failunused",
               "Fail if unused elements left" );
  xulHelper.checkbox(leftside, this, "failremain",
               "Fail if unused field remains in query" );
  // TODO - Add a text field for the error message(s)
  xulHelper.checkbox(rightside, this, "failnotanded",
               "Fail if list not ANDed to preceding");
  xulHelper.checkbox(rightside, this, "failnotandonly",
               "Fail if anything but ANDs" );

};


/////////////////////////
// Run

// Find the first (unused?) index, or "" if nothing found
Listqueryelement.prototype.findindex = function (lq) {
  for ( var i=0; i<lq.length; i++ ){
    if ( (!this.conf.onlyused) || (!lq[i].used)) {
      return lq[i].field;
    }
  }
  return "";
};

// clear the 'used' flags
Listqueryelement.prototype.clearused = function (lq) {
  for ( var i=0; i<lq.length; i++ ){
    lq[i].used=false;
  }
};

// simple clone function. Assumes no deeply nested structures
Listqueryelement.prototype.clonenode =function(n) {
  var c = {};
  for ( var i in n )
    c[i] = n[i];
  return c;
};

Listqueryelement.prototype.run = function (task) {
  var lq = jsonPathHelper.get(this.conf.in, task.data);
  var res = [];
  if ( Array.isArray(lq) )
      lq = lq[0];
  task.debug("lqe: " + JSON.stringify(lq) );
  var index = this.conf.index;
  if ( index == anyindex) {
    index = this.findindex(lq);
    if (this.conf.failnothing && !index)
      throw new StepError("No index found" );
  }
  if (this.conf.clearused )
    this.clearused(lq);

  if ( index ) {
    var maxelems = 999999; // something way too big
    if ( this.conf.numelements.match( /[0-9]+/ ) )
      maxelems = parseInt(this.conf.numelements);
    //task.debug("lqe: Looking for " + index + " max=" + maxelems );
    var i = 0;
    while ( i < lq.length ) {
      if ( lq[i].field == index &&
           ( (!this.conf.onlyused) || (!lq[i].used) ) &&
           maxelems>0 ) {
        res.push( this.clonenode(lq[i]) );
        task.debug("lqe: lq[" + i + "] = " + JSON.stringify(lq[i]) );
        if ( this.conf.markused )
          lq[i].used = true;
        maxelems--;
        if ( this.conf.failnotandonly &&
             lq[i].op &&
             lq[i].op != "and" )
          throw new StepError ("Only AND supported (found '" + lq[i].op + "')");
      }
      i++;
    }
  }
  //task.debug("lqe: failnothing=" + this.conf.failnothing +
  //  " res.len=" + res.length );
  if ( this.conf.failnothing && (res.length == 0)  ) {
    throw new StepError ("ListqueryElement did not find anything for " + index);
  }
  if ( this.conf.failnotanded && res.length > 0)
    if ( res[0].op != "and" && res[0].op != "" )
      throw new StepError ("Listqueryelement not ANDed to the rest of query");
  if ( this.conf.failunused ) {
    for ( var i=0; i<lq.length; i++ )
      if ( !lq[i].used ) {
        throw new StepError("ListqueryElement did not use up all of listquery. " +
          lq[i].field + ": " + lq[i].term );
      }
  }
  if ( this.conf.failremain ) { // Check that no unused field remains
    i = 0;                     // only makes sense when taking 1 or 2, not all
    while ( i < lq.length ) {
      if ( lq[i].field == index && !lq[i].used) {
        throw new StepError ("Index " + index + " remains in listquery");
      }
      i++;
    }
  }
  jsonPathHelper.set(this.conf.out, res, task.data);
};


/////////////////////////
// Unit test

Listqueryelement.prototype.unitTest = function ( ) {
  var defaultconf = {  // Can be overridden in every test
    in: { path:'$.temp', key:'listquery' },
    out:{ path:'$.temp', key:'elements' },
    clearused: false,
    markused: true,
    onlyused: true,
    index: anyindex,
    numelements: "all",
    failnothing: false,
    failunused: false,
    failnotanded: false,
    failnotandonly: false,
    failremain: false,
    };
    
  var tests = [
    { name:"simple",
      conf: { },
      lq: '[ { "term":"water", "field":"keyword" }, '+
            '{ "term":"fire", "field":"keyword", "op":"and" } ]',
      exp: '[ { "term":"water", "field":"keyword" }, '+
            '{ "term":"fire", "field":"keyword", "op":"and" } ]',
      used: "TT",
    },
    { name:"nomark",
      conf: { markused:false },
      // not specifying lq and exp here, reusing from previous
      used: "FF",
    },
    { name:"onlyone",
      conf: { numelements: "1" },
      exp: '[{"term":"water","field":"keyword"}]',
      used: "TF",
    },
    { name:"onlysecond",
      conf: { numelements: "1" },
      lq: '[ { "term":"water", "field":"keyword", "used":true }, '+
            '{ "term":"fire", "field":"keyword", "op":"and" } ]',
      exp: '[{ "term":"fire", "field":"keyword", "op":"and" } ]',
      used: "TT",
    },
    { name:"notfound",
      conf: { index: "badindex" },
      lq: '[ { "term":"water", "field":"keyword" }, '+
            '{ "term":"fire", "field":"keyword", "op":"and" } ]',
      exp: '[]',
      used: "FF",
    },
    { name:"failnotfound",
      conf: { index: "badindex", failnothing: true },
      // not specifying lq here, reusing from previous
      exp: 'Exception! ListqueryElement did not find anything for badindex',
      used: "FF",
    },
    { name:"notfailunused",
      conf: { failunused: true },
      exp: '[ { "term":"water", "field":"keyword" }, '+
            '{ "term":"fire", "field":"keyword", "op":"and" } ]',
    },
    { name:"failunused",
      conf: { numelements: "1", failunused: true },
      // not specifying lq here, reusing from previous
      exp: "Exception! ListqueryElement did not use up all of listquery. keyword: fire"
    },
    { name:"notfailandonly",
      conf: { failnotandonly: true },
      exp: '[ { "term":"water", "field":"keyword" }, '+
            '{ "term":"fire", "field":"keyword", "op":"and" } ]',
    },
    { name:"failandonly",
      lq: '[ { "term":"water", "field":"keyword" }, '+
            '{ "term":"fire", "field":"keyword", "op":"and" },' +
            '{ "term":"ice", "field":"keyword", "op":"and" },' +
            '{ "term":"air", "field":"keyword", "op":"or" } ]',
      conf: { failnotandonly: true },
      exp: "Exception! Only AND supported (found 'or')",
    },
    { name:"notfailandonly-short",  // the 'or' is not part of the thing
      lq: '[ { "term":"water", "field":"keyword" }, '+
            '{ "term":"fire", "field":"keyword", "op":"and" },' +
            '{ "term":"ice", "field":"keyword", "op":"and" },' +
            '{ "term":"air", "field":"keyword", "op":"or" } ]',
      conf: { failnotandonly: true, numelements: "2" },
      exp: '[{"term":"water","field":"keyword"},'+
            '{"term":"fire","field":"keyword","op":"and"}]'
    },
    { name:"notfailanded",  // the 'or' is not part of the thing
      lq: '[ { "term":"water", "field":"title" }, '+
            '{ "term":"fire", "field":"title", "op":"or" },' +
            '{ "term":"ice", "field":"keyword", "op":"and" },' +
            '{ "term":"air", "field":"keyword", "op":"and" } ]',
      conf: { failnotanded: true, index: "keyword" },
      exp: '[{"term":"ice","field":"keyword","op":"and"},' +
            '{"term":"air","field":"keyword","op":"and"}]'
    },
    { name:"failanded",  // the 'or' is not part of the thing
      lq: '[ { "term":"water", "field":"title" }, '+
            '{ "term":"fire", "field":"title", "op":"and" },' +
            '{ "term":"ice", "field":"keyword", "op":"or" },' +
            '{ "term":"air", "field":"keyword", "op":"and" } ]',
      conf: { failnotanded: true, index: "keyword" },
      exp: "Exception! Listqueryelement not ANDed to the rest of query",
    },
  ]; // test array

  logger.info("Starting unit test for ListqueryElement");
  // fake some runtime
  var connector = new Connector();
  var task = new Task( connector, "unittest" );
  var lq; // listquery from previous test, for reuse
  var ex; // expected result from previous test
  for ( var i=0; i<tests.length; i++) {
    var t=tests[i];
    var lqe = new Listqueryelement;
    lqe.task = task;
    for ( var f in defaultconf ) {
      lqe.conf[f] = defaultconf[f];
    }
    for ( var f in t.conf ) {
      lqe.conf[f] = t.conf[f];
    }
    if (t.lq)
      lq = JSON.parse(t.lq);  // otherwise keep the previous
    task.data.temp.listquery = JSON.parse(JSON.stringify(lq));  // make a new copy
    var res = "";
    try {
      lqe.run(task);
      res = JSON.stringify(task.data.temp.elements);
    } catch(e) {
      res = "Exception! " + e.message;
      if ( e.fileName  )  // catch syntax errors in code, etc
          res += ("  in " + e.fileName + "." + e.lineNumber  );
    }
    if ( t.exp) {  // otherwise keep the previous
      ex = t.exp;
      if ( ex.match(/[\[\{]/ ) )// looks like JSON if it has a [ or { somewhere
        ex = JSON.stringify( JSON.parse(t.exp) ); // normalize it
    }
    if ( res != ex ) {
      logger.error ("Test " + i + " " + t.name + " FAILED! ");
      logger.error ("Got: " + res );
      logger.error ("Exp: " + ex );
      return false;
    }
    if ( t.used ) {
      var used = "";  // Check the used-flags
      for ( var e=0; e<task.data.temp.listquery.length; e++) {
        if ( task.data.temp.listquery[e].used )
          used += "T";
        else
          used += "F";
      }
      if ( t.used != used ) {
        logger.error ("Test " + i + " " + t.name + " FAILED! ");
        logger.error ("Got the right result: " + res );
        logger.error ("But the used-flags differ:" );
        logger.error ("Got: " + used );
        logger.error ("Exp: " + t.used );
        return false;
      }
    }
    logger.info( i + " " + t.name + " OK");
  }
  return true;
} // unitTest

///////////////////////////
// Housekeeping

Listqueryelement.prototype.getClassName = function () {
  return "Listqueryelement";
};

Listqueryelement.prototype.getDisplayName = function () {
  return "Listquery Element";
};

Listqueryelement.prototype.getDescription = function () {
  return "Extract one or more elements from a listquery";
};

Listqueryelement.prototype.getVersion = function () {
  //return "2.0";
  return "2.1"; // 2.1 introduced the failremain flag
};

// Not much point in a renderArgs, way too complex to show in the builder
Listqueryelement.prototype.renderArgs = function () {
  var num = this.conf.numelements;
  if ( num == "all" )
    num = "(all)";
  else if ( num == "1" )
    num = "";
  else num = "(max " + num + ")";
  return num + " " + this.conf.index  ;
  return "";
};

Listqueryelement.prototype.capabilityFlagDefault = function ( flag ) {
  //dump("checking query-and, conf = " + JSON.stringify(this.conf) + "\n");
  var used = this.getUsedArgs();
  for ( var a in used ) {  // usually not the case
    if ( flag == "index-" + a )
      return true;
  }
  if ( flag == "index-" + this.conf['index'] )
    return true;
  return null;
};

Listqueryelement.prototype.getUsedArgs = function () {
  if (this.conf.in && this.conf.in.path === "$.input")
    return [this.conf.in.key];
  else
    return [];
};

Listqueryelement.prototype.upgrade = function (confVer, curVer, conf) {
  // can't upgrade if the connector is newer than the step
  if (confVer > curVer)
    return false;
  if ( confVer < "2.0" ) {
    if (conf.failnotanded == undefined)
      conf.failnotanded = false;
    if (conf.failnotandonly == undefined)
      conf.failnotandonly = false;
  }
  if ( confVer < "2.1" ) {
    if (conf.failremain == undefined)
      conf.failremain = false;
  }
  return true;
};
