var EXPORTED_SYMBOLS = ["objHelper"];
Components.utils.import("resource://indexdata/runtime/core.js");
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import("resource://indexdata/util/logging.js");
var logger = logging.getLogger('');
var objHelper = {
  objectFromXml: function (node) {
    switch(node.getAttribute("type")) {
      case "null": return null;
      case "bool": return node.textContent == "true";
      case "number": return Number(node.textContent);
      case "string": return node.textContent;
      case "array":
        var arr = [];
        for (var i=0; i<node.childNodes.length; i++) {
          var child = node.childNodes[i];
          if (child.nodeType == 1 && child.nodeName == "item")  // Node.ELEMENT_NODE == 1
            arr.push(objHelper.objectFromXml(child));
        }
        return arr;
      case "object":
        var obj = {};
        for (var i=0; i<node.childNodes.length; i++) {
          var child = node.childNodes[i];
          if (child.nodeType == 1) //Node.ELEMENT_NODE
            obj[child.nodeName] = objHelper.objectFromXml(child);
        }
        return obj;
      default: throw new SyntaxError("Cannot parse: unkown type, " +
          " at node '" + node.nodeName + "' :" +
          "'" + node.getAttribute("type") +"' " );
    }
  },
  objectToXml: function (name, obj, node) {
    switch (typeof obj) {
      case "boolean":
        return xmlHelper.appendNode(node, name, obj, {"type": "bool"});
      case "number":
        return xmlHelper.appendNode(node, name, obj, {"type": "number"});
        return;
      case "string":
        return xmlHelper.appendNode(node, name, obj, {"type": "string"});
        return;
      case "object":
        var newNode;
        //null
        if (!obj) {
          return xmlHelper.appendNode(node, name, null, {"type": "null"});
        //array
        } else if (Array.isArray(obj)) {
          var child = 
          xmlHelper.appendNode(node, name, null, {"type": "array"});
          for (var i=0; i < obj.length; i++) {
            objHelper.objectToXml("item", obj[i], child);
          }
          return child;
        // hash
        } else {
          var child = 
          xmlHelper.appendNode(node, name, null, {"type":"object"});
          for (var key in obj) {
            objHelper.objectToXml(key, obj[key], child);
          }
          return child;
        }
        return;
    }
  },

  // inspired by jQuery but unlikely as performant
  extend : function () { // parameters: target, src1, src2 ...
    let target = arguments[0];
    if (typeof target !== "object") throw new Error("objHelper.extend called without target object");
    for (let i = 1; i < arguments.length; i++) {
      if (typeof arguments[i] !== "object") throw new Error("objHelper.extend called with non-object argument " + i);
      // lazy deep copy so we don't make a rat's nest of references
      var src = JSON.parse(JSON.stringify(arguments[i]));
      for (let key in src) { target[key] = src[key] }
    }
    return target;
  },

  loadProperties: function (node) {
    var properties = {};
    for (var i=0; i<node.childNodes.length; i++) {
      var cur = node.childNodes[i];
      if (cur.nodeName.toLowerCase() === 'property' && cur.getAttribute('name')) {
        // straight string value, for backwards compatibility
        if (cur.getAttribute('value'))
          properties[cur.getAttribute('name')] = cur.getAttribute('value');
        // objhelper
        else if (cur.getAttribute('type'))
          properties[cur.getAttribute('name')] = objHelper.objectFromXml(cur);
      }
    }
    return properties;
  },
  saveProperties: function (obj, node) {
    for (var p in obj) { 
      var propNode = objHelper.objectToXml("property", obj[p], node);
      propNode.setAttribute("name", p);
    }          
  },
  loadDefaults: function (node) {
    var defaults = {};
    for (let i = 0; i < node.childNodes.length; i++) {
      let def = node.childNodes[i];
      if (def.nodeName.toLowerCase() === 'defaults') {
        for (let j = 0; j < def.childNodes.length; j++) {
          let sn = def.childNodes[j];
          if (sn.nodeName.toLowerCase() === 'step' && sn.hasAttribute('name')) {
            if (!defaults.steps) defaults.steps = {};
            defaults.steps[sn.getAttribute('name')] = objHelper.objectFromXml(sn);
          }
        }
      }
    }
    return defaults;
  },

  unitTest: function () {
    // extend
    let a = {x: {n:66}};
    let b = {y: {n:66}};
    let c = {x: {n:42}};
    let expected = '{"x":{"n":42},"y":{"n":66}}';
    let result = JSON.stringify(objHelper.extend(a, b, c));
    logger.info(result);
    if (result !== expected) { 
      logger.info("Expected: " + expected);
      return false;
    }
    return true;
  }
}

/* Version dependent helpers */
if (core.platformVersion > 17) {
  // proxy that proxies requests for __exposedProps__ to another proxy that
  // always returns 'rw'
  objHelper.exposedPropsProxy = function (obj) {
    return new Proxy(obj, {
      get: function(target, name) {
        if (name === '__exposedProps__') return objHelper.rwAlways;
        // recurse here
        else {
          let value = target[name];
          if (typeof(value) === 'object') return objHelper.exposedPropsProxy(value);
          else return value;
        }
      },
      getOwnPropertyDescriptor: function(target, name) {
        if (name === '__exposedProps__')
          return {configurable: true, enumerable: true,
                  value: objHelper.rwAlways, writable: true};
        else if (target.hasOwnProperty(name))
          return Object.getOwnPropertyDescriptor(target, name);
      },
      hasOwn: function(target, name) {
        if (name === '__exposedProps__') return true;
        else return target.hasOwnProperty(name);
      },
      has: function(target, name) {
        if (name === '__exposedProps__') return true;
        else return name in target;
      }
    });
  };
  // decorate object with a proxied __exposedProps__ property returning
  // 'rw' for every property in order to bypass the whitelisting imposed
  // by FF17 so we can let sandboxes create arbitrarily named properites
  objHelper.exposeObject = function (obj) {
    if (typeof obj === 'object' ) {
      Object.defineProperty(obj, "__exposedProps__", {
        value: objHelper.rwAlways,
        enumerable: false
      });
    }
    for (key in obj) {
      if (typeof obj[key] === 'object' ) {
        objHelper.exposeObject(obj[key]);
      }
    }
  };
  objHelper.rwAlways = new Proxy(Object.freeze({}), {
   get: function() { return "rw"; },
   getOwnPropertyDescriptor: function() {
     return {configurable: true, enumerable: true, value: "rw", writable: true }
   },
   hasOwn: function() { return true; },
   has: function() { return true; }
  });
}
