var EXPORTED_SYMBOLS = ["Connector"];
Components.utils.import("resource://indexdata/runtime/core.js");
Components.utils.import("resource://indexdata/runtime/ConnectorTemplate.js");
Components.utils.import('resource://indexdata/runtime/Task.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/objHelper.js');
Components.utils.import('resource://indexdata/util/helper.js');
Components.utils.import("resource://indexdata/util/logging.js");
var logger = logging.getLogger('');
var contentFilter = Components.classes["@indexdata.com/contentfilter;1"]
                     .getService().wrappedJSObject;
var Connector = function (template) {
  //hash of tasks indexed by name
  this.template = template || null;
  //tasks grouped by name and ordered by arg count
  this.tasks = {};
  //list of all tasks sorted by name
  this.taskList = [];
  //tasks by id
  this.tasksById = {};
  //dom of the scraped page
  this.forcedPageWindow = null;
  //implementation of nsIWebProgress
  this.taskArgumentsUpdateCb = null;
  this.stepParametersUpdateCb = null;
  this.properties = {'block_css': true,
                     'block_images':true,
                     'block_objects':true};
  this.data = {session:{}, system:{}};
  this.metaData = null;
  this.capabilityFlags = {}; // 'supports-fq' => false
    // indexed by flag name, value is true or false.
    // Only the flags explicitly set by the user, the rest come from defaults
  this.pageOverride = undefined;
  this.urlOverride = undefined;
};

Connector.prototype = {
  
  /////////////////////
  // page accessibility
  
  getPageWindow: function () {
    var pageWindow = null;
    if (this.forcedPageWindow) {
      //set page window takes precedence (e.g engine)
      pageWindow = this.forcedPageWindow;
    } else {
      // try to get the current tab
      var wm = Components.classes["@mozilla.org/appshell/window-mediator;1"]
                  .getService(Components.interfaces.nsIWindowMediator);
      var mainWindow = wm.getMostRecentWindow("navigator:browser");
      if (mainWindow) pageWindow = mainWindow.gBrowser;
    }
    if (pageWindow && (pageWindow.contentDocument || pageWindow.document))
      return pageWindow;
    else
      throw new Error('The page window is null or malformed (no content doc)');
  },
  
  setPageWindow: function (win) {
    this.forcedPageWindow = win;
  },
  
  getRealPageDoc: function () {
    var pW = this.getPageWindow();
    if (pW.contentDocument)
      return pW.contentDocument;
    else if (pW.document)
      return pW.document;
    else
      return null;
  },
  
  getPageDoc: function () {
    if (this.pageOverride) return this.pageOverride;
    return this.getRealPageDoc();
  },
  
  // Get the actual URL of the current page
  // This is tricky, there are many ways to do this, and they
  // have subtle differences in the builder and in the engine
  // depending on what we have on the screen. All start with getPageDoc().
  // Mike has reported having to use soemthing like
  //    defaultView.parent.document.URL
  // but I find it fails when displaying a XML document. Perhaps that was
  // at some sort of frameset. TODO - Check with pages that use frames!
  // In all my experiments, simple getPageDoc().location worked best.

  // B=Builder, E=Engine, with page
  //   a= about:blank
  //   b= html page
  //   c= xml page
  //   d= text plain
  //   1 .URL
  //     B: a:OK b:OK c:undef d:OK   E:  a:OK b:OK c:undef d:OK
  //   2 .location BEST SO FAR!
  //     B: a:OK b:OK c:OK d:OK   E:  a:OK b:OK c:OK d:OK
  //   3 .location.href
  //     B: a:OK b:OK c:OK d:OK   E:  a:OK b:OK c:OK d:OK
  //   4 .location.URL - undef always
  //     B: a:undef b: c: d:   E:  a: b: c: d:
  //   5 .defaultView.document.URL
  //     B: a:OK b:OK c:undef d:OK   E:  a:OK b:OK c:undef d:OK
  //   6 .defaultView.parent.document.URL
  //     B: a:OK b:OK c:undef d:OK   E:  a:file:/// b:file:/// c:file:/// d:file:///
  getRealPageUrl: function() {
    /* Debug tests
    var p = this.getPageDoc();
    try { dump("url1: " + p.URL + "\n");
    } catch(e) { dump("url1: "+e+"\n"); };
    try { dump("url2: " + p.location + "\n");
    } catch(e) { dump("url2: "+e+"\n"); };
    try { dump("url3: " + p.location.href + "\n");
    } catch(e) { dump("url3: "+e+"\n"); };
    try { dump("url4: " + p.location.URL + "\n");
    } catch(e) { dump("url4: "+e+"\n"); };
    try { dump("url5: " + p.defaultView.document.URL + "\n");
    } catch(e) { dump("url5: "+e+"\n"); };
    try { dump("url6: " + p.defaultView.parent.document.URL + "\n");
    } catch(e) { dump("url6: "+e+"\n"); };
    */
    var url="";
    try {
      // can't have a URL without a page
      var pg = this.getPageDoc();
      if (pg) url = pg.location.href;
    } catch(e) {};
    if (url) return "" + url ; // "" + casts to string
    return "";
  },
  
  getPageUrl: function() {
    if (this.urlOverride) return this.urlOverride;
    return this.getRealPageUrl();
  },
  
  // Set the page override dom tree, and its url, for future use.
  setPageOverride: function( dom, url ) {
    this.pageOverride = dom;
    if (url) this.urlOverride = url;
  },
  
  clearPageOverride: function( ) {
    this.pageOverride = undefined;
    this.urlOverride = undefined;
  },


  /////////////////////
  // Tasks and templates
  
  getTemplate: function () {
    return this.template;
  },

  findTask: function (taskName, args) {
    var taskList = this.tasks[taskName];
    if (!taskList) return null;
    var t0 = taskList[0];
    if ( ! t0.polymorphic ) {
      if ( taskList.length != 1 ) {
        t0.warn("Task '" + taskName + "' is not polymorphic, " +
          "but has " + taskList.length + " alternatives. ");
        // TODO - We could return null here to indicate that the task
        // was not found. But in the most common case, the init task,
        // it is considered optional, so not finding it will be accepted
        // later, and no error will be generated anyway. Better return
        // the first task, and run some kind of init...
      }
      return t0;
    }
    for (var t=0; t<taskList.length; t++) {
      if (typeof args == "undefined") return taskList[0];
      var hasAllArgs = true;
      var argList = taskList[t].getUsedArgs();
      for (var a=0; a<args.length; a++) {
        if (argList.indexOf(args[a])==-1) {
          hasAllArgs = false;
          break;
        }
      }
      if (hasAllArgs) return taskList[t];
    }
    // Dirty trick to get around the need of lots of dummy transforms
    // in the init task. See bug 4973.
    // If no matching task was found above, and if we are looking for
    // the init task, return the first (and probably only) one.
    if ( taskName == "init" ) return taskList[0];
    // TODO - This should be removed in some distant version, when we no
    // longer have connector files without the polymorphic flag in their
    // init tasks. Leave it it for now (version 2.9) to preserve old
    // behavior.
    return null;
  },
  
  findTasks: function (taskName) {
    return typeof this.tasks[taskName] != "undefined"
      ? this.tasks[taskName]
      : null;
  },
  
  getTaskById: function (id) {
    return this.tasksById[id];
  },
  
  getTasks: function () {
    //temporary, before the onArgUpd works
    this.taskList.sort( function (a,b) { return a.compareTo(b); } );
    return this.taskList;
  },
  
  setTaskArgumentsUpdateCb: function(cb) {
    this.taskArgumentsUpdateCb = cb;
  },
  
  onTaskArgumentsUpdate: function() {
    if (this.taskArgumentsUpdateCb) this.taskArgumentsUpdateCb();
  },
  
  setStepParametersUpdateCb: function(cb) {
    this.stepParametersUpdateCb = cb;
  },
  
  onStepParametersUpdate: function(step) {
    if (this.stepParametersUpdateCb) this.stepParametersUpdateCb(step);
  },
  
  //create new implementation of a given task by name
  createTask: function (taskName) {
    var task = new Task(this, taskName);
    //new tasks added at front since known that since empty, no params are used
    this.addTask(task);
    return task;
  },
  
  //insert and index task object
  addTask: function (task) {
    // new task group
    if (!this.tasks.hasOwnProperty(task.getName()))
      this.tasks[task.getName()] = [];
    this.tasks[task.getName()].push(task);
    //sort grouped by arg count
    this.tasks[task.getName()].sort(function (a,b) { return a.compareTo(b); });
    this.tasksById[task.id] = task;
    //push to flat list, sort by names
    this.taskList.push(task);
  },
  
  detachTask: function (task) {
    task.connector = null;
    function isAttached (element) { return !(element === task); }
    this.tasks[task.name] = this.tasks[task.name].filter(isAttached);
    if (this.tasks[task.name].length === 0) delete this.tasks[task.name];
    this.taskList = this.taskList.filter(isAttached);
    delete this.tasksById[task.id];
  },

  //////////////////////
  // Saving and loading
  
  // Slightly misnamed, returns the configuration XML, but does not actually
  // save anywhere.
  saveConf: function () {
    this.recalculateCapabilityFlags(); // in case defaults have changed
                                       // as can happen if the user edits steps
    // connector node
    var cn = xmlHelper.doc("connector");
    if (this.template && this.template.name) {
      cn.setAttribute('template', this.template.name);
    }
    if (core.version) cn.setAttribute('version', core.version);
    // meta data
    var mdn = null;
    for (var name in this.metaData) {
      if (mdn === null) mdn = xmlHelper.appendNode(cn, "metaData");
      if (this.metaData[name]) {
        xmlHelper.appendNode(mdn, "meta", null,
        {"name" : name, "content" : this.metaData[name]});
      }
    }
    // Capability flags
    var fn = null;
    for ( var flag in this.capabilityFlags ) {
      if ( this.capabilityFlags[flag] != undefined ) {
        if ( fn === null ) fn = xmlHelper.appendNode(cn, "CapabilityFlags");
        xmlHelper.appendNode(fn, "flag", null,
            {"name" : flag, "value" : this.capabilityFlags[flag]});
      }
    }
    // properties
    objHelper.saveProperties(this.properties, cn);
    for (var t in this.tasks) {
      for (var i=0; i<this.tasks[t].length; i++) {
        var tn = xmlHelper.appendNode(cn, "task", null,
            {"name": this.tasks[t][i].getName()});
        this.tasks[t][i].saveConf(tn);
      }
    }
    return cn;
  },
  
  saveConfToFile: function (file) {
    xmlHelper.saveDoc(file, this.saveConf());
  },
  
  // load from connector doc and session parameters
  loadConf: function (doc, session) {
    core.lastconnector = ""; // forget what we had before
    if (doc.getElementsByTagName("connector").length !== 1 ||
        doc.firstChild.nodeName.toLowerCase() !== "connector")
      throw new Error("file is not a proper connector")
    var cn = doc.firstChild;
    if (cn.hasAttribute('template')) {
      this.template = new ConnectorTemplate();
      this.template.loadFromUUID(cn.getAttribute('template'));
    }
    // we can't use a connector from a newer major version
    if (cn.hasAttribute('version')) {
      let loadVer = cn.getAttribute('version');
      if (!helper.checkVersion(loadVer, core.version)) {
        throw new Error('Connector generated by CF ' + loadVer
          + 'with newer major version than current ' + core.version);
      }
    }
    if (core.inBuilder) {
      // remove generated nodes
      var stripGen = function (node) {
        var children = node.childNodes;
        for (var i=0; i<children.length; i++) {
          if ((children[i].toString() === "[object Element]" &&
               children[i].hasAttribute("generator")) ||
              (children[i].textContent === "undefined")) {
            node.removeChild(children[i]);
          } else {
            if (children[i].hasChildNodes()) stripGen(children[i]);
          }
        }
      }
      stripGen(cn);
    }
    // load tasks
    this.tasks = {};
    this.taskList = [];
    this.taskById = {};
    var tns = cn.getElementsByTagName("task");
    for (var i=0; i < tns.length; i++) {
      var name = tns[i].getAttribute("name");
      var task = new Task(this, name);
      task.loadConf(tns[i]);
      this.addTask(task);
    }
    // Load capability flags
    this.capabilityFlags = {};
    //dump("Loading flags\n");
    var capgrpNodes = cn.getElementsByTagName("CapabilityFlags");
    if ( capgrpNodes.length < 1 ) {
        //dump("No capability flags found, must be an old connector\n");
    } else {
      if (capgrpNodes.length > 1)
          dump("Multiple capabilityFlags found, using first...");
      var capNodes = capgrpNodes[0].getElementsByTagName("flag");
      for (var i=0; i<capNodes.length; i++) {
        var flag = capNodes[i].getAttribute("name");
        var val = capNodes[i].getAttribute("value");
        if (val == "true" ) val = true;
        else val = false;
        this.capabilityFlags[flag]=val;
        //dump("Flag " + flag + " = " + val + "\n" );
      }
    }
    // load properties
    this.properties = objHelper.loadProperties(cn);
    this.metaData = {};
    var mdNodes = cn.getElementsByTagName("metaData");
    if (mdNodes.length < 1) {
      dump("Metadata node not found"+"\n");
    } else {
      if (mdNodes.length > 1)
        dump("Multiple metadata elements found, using first...");
      var metaNodes = mdNodes[0].getElementsByTagName("meta");
      for (var i=0; i<metaNodes.length; i++) {
        this.metaData[metaNodes[i].getAttribute("name")]
          = metaNodes[i].getAttribute("content");
      }
    }
    //missing md elems
    if (this.template) {
      for (var i=0; i<this.template.metaData.length; i++) {
        var name = this.template.metaData[i].name;
        if (typeof this.metaData[name] == "undefined")
          this.metaData[name] = "";
      }
    }
    this.updateContentFilter();
    // initialise session
    logger.log(Level.DEBUG, "Connector session arguments: "
               + JSON.stringify(session) + "\n");
    this.data.session = session || {};
  },
  
  loadConfFromFile: function(filePath, session) {
    this.loadConf(xmlHelper.openDoc2(filePath), session);
    core.lastconnector = filePath;
  },
  
  isComplete: function() {
    var taskTemplates = this.template.getTaskTemplates();
    for (var i=0; i<taskTemplates.length; i++) {
      var tt = taskTemplates[i];
      var reqs = this.template.getRequirements(tt.name);
      for (var j=0; j<reqs.length; j++) {
        if (!this.findTask(tt.name, reqs[j]))
          return false;
      }
    }
    // all tasks complete
    return true;
  },
  
  // configure content filter from connector properties
  updateContentFilter: function() {
    try {
      if ("block_css" in this.properties) {
        contentFilter.setBlockCss(this.properties["block_css"]);
      }
      if ("block_images" in this.properties) {
        contentFilter.setBlockImages(this.properties["block_images"]);
      }
      if ("block_objects" in this.properties) {
        contentFilter.setBlockObjects(this.properties["block_objects"]);
      }
      contentFilter.resetBlacklist();
      contentFilter.resetWhitelist();
      var bl = this.properties["blacklist"];
      var wl = this.properties["whitelist"];
      if (bl) {
        for (var i=0; i<bl.length; i++) contentFilter.addBlacklist(bl[i]);
      }
      if (wl) {
        for (var i=0; i<wl.length; i++) contentFilter.addWhitelist(wl[i]);
      }
    } catch(e) { dump("Exception: "+e+"\n"); }
  },

  ////////////////////
  // Capability flags
  
  // Get the whole array of (explicitly set) capabilityFlags.
  getCapabilityFlags: function() {
    return this.capabilityFlags;
  },
  
  setCapabilityFlag: function(name,val) {
    this.capabilityFlags[name]=val;
  },
  
  // Clears all flags to undefined
  clearAllCapabilityFlags: function() {
    this.capabilityFlags = [];
  },
  
  getCapabilityFlagDefault: function (name, defaultValue, alias) {
    for (var i=0; i<this.taskList.length; i++) {
      let block = this.taskList[i].block;
      let opinion = this.getCapabilityFlagDefaultBlock(block, name, alias);
      if (opinion != undefined && opinion != null) {
        return opinion;
      }
    }
    return defaultValue;
  },
  
  // get an opinion for a  default value for a capability flag within a block
  // or null when no opinion is given
  getCapabilityFlagDefaultBlock: function (block, name, alias) {
    for (let i=0; i<block.steps.length; i++) {
      let step = block.steps[i];
      if ((step.getClassName() !== "Comment") && !step.isDisabled ) {
        let opinion = step.capabilityFlagDefault(name);
        if (opinion != undefined && opinion != null) {
          logger.debug("Step "+step+" reported '"+opinion+"' on flag "+name);
          return opinion;  // return the first opinion we find
        }
        if (alias) {
          let opinion = step.capabilityFlagDefault(alias);
          if (opinion != undefined && opinion != null) {
            logger.debug("Step "+step+" reported '"+opinion+"' on flag alias "+alias);
            return opinion;
          }
        }
        //handle any blocks
        let blocks = step.blocks;
        if (blocks && blocks.length) {
          for (let j=0; j<blocks.length; j++) {
            let opinion = this.getCapabilityFlagDefaultBlock(blocks[j], 
              name, alias);
            if (opinion != undefined && opinion != null) {
              return opinion;
            }
          }
        }
      }
    }
    return null;
  },
  
  // Get the value of a capability flag, either from the array of
  // manually set flags, or by finding a default value for it from
  // the steps, or if anything else fails, using the given default-default.
  checkCapabilityFlag: function( name, defaultvalue, alias ) {
    if ( this.capabilityFlags[name] != undefined &&
         this.capabilityFlags[name] != null )
      return this.capabilityFlags[name];
    return this.getCapabilityFlagDefault(name, defaultvalue, alias );
  },
  
  // Recalculate the metadata element 'Flags' based on the flags set
  // before, and the defaults from the steps, and from the template.
  recalculateCapabilityFlags: function() {
    if ( ! this.metaData )
      this.metaData = {}; // defensive coding, should not happen
    if ( ! this.template )
      return; // should not happen. Better not touch things
    var flaglist = "";
    var templFlags = this.getTemplate().getCapabilityFlags();
    var connFlags = this.getCapabilityFlags();
    for ( var i = 0; i<templFlags.length; i++) {
      var name = templFlags[i].name;
      var alias = templFlags[i].alias;
      //var templdefault = this.getTemplate()
      //                .getCapabilityFlagDefault(name);
      var templdefault = templFlags[i].default;
      if ( templdefault == "true" )
        templdefault = true;
      else
        templdefault = false;
      //dump("template default for '" + name + "' is " + templdefault +"\n");
      if ( this.checkCapabilityFlag(name, templdefault, alias) ){
        //dump("recalc: Adding flag " + name + "\n");
        if (flaglist)
          flaglist += " ";
        flaglist += name;
      }
    }
    //dump("conn.Recalc: Old= '" + this.metaData["flags"] + "' " +
    //  "new='" + flaglist + "' :" + ( this.metaData["flags"] == flaglist) + "\n" );
    this.metaData["flags"] = flaglist;
  },
  useSessionTestValues: function () {
    if (typeof this.properties.testSessionJSON === 'string') {
      this.data.session = JSON.parse(this.properties.testSessionJSON);
    }
  },
  
};
