var EXPORTED_SYMBOLS = ["jsonPathHelper"];
Components.utils.import('resource://indexdata/thirdparty/jsonPath.js');
Components.utils.import("resource://indexdata/util/logging.js");
var logger = logging.getLogger('');

// In this container a spec is an object with two or three properties
// and is necessary to allow sparse records without cumbersome paths
//
//  path: JSONpath that yields objects
//  key: target key within those objects
//  append (optional): how to deal with the existing array on write

var jsonPathHelper = {
  // Constants
  APPEND: 1,
  CONCAT: 2,
  REPLACE: 3,

  // These two functions are hardcoded to CF style
  // $.output.result[*].field[*]

  // Perhaps one day it can read from the template and allow
  // more flexibility for when drop downs can be used.
  pathFromArray: function (arr) {
    var path = "";
    if (Array.isArray(arr) && arr.length > 0) {
      path = "$." + arr[0]
      for (let i = 1; i < arr.length; i++) {
        path += "." + arr[i] + "[*]";
      }
    }
    return path;
  },

  specFromArray: function (arr) {
    var result = {};
    result.key = arr.pop();
    result.path = jsonPathHelper.pathFromArray(arr);
    return result;
  },

  // Little helper to check a capability flag default
  //   spec - jspnPath spec to check against, typically from step.conf
  //          for example ( "$.input","keyword")
  //   flag - The flag to check agains, for example "index-keyword"
  //   path - must match the beginning of the path in the spec,
  //          f.ex "$.input" or "$.output" even for $.output[*].results
  //   flagprefix - must match the flag, f.ex. "index"
  // If all these match, returns true. Otherwise returns null
  capabilityFlagDefault: function ( spec, flag, path, flagprefix ) {
  if (!spec || !flag)
    return null;
  var flagparts = flag.split("-");
  if (flagparts[0] == flagprefix &&
      flagparts[1] == spec.key &&
      spec.path.substring(0,path.length) == path )
    return true;
  return null;
  },
  // This extends the above to apply to arrays of conditions. 
  conditionsCapable: function (conditions, flag, path, flagprefix) {
    if (Array.isArray(conditions)) {
      for (let i = 0; i < conditions.length; i++) {
        if (typeof conditions[i].operands === 'object'
          && Array.isArray(conditions[i].operands)) {
          let ops = conditions[i].operands;
          for (let j = 0; j < ops.length; j++) {
            if (typeof ops[j].jp === 'object' && jsonPathHelper.capabilityFlagDefault(
              ops[j].jp, flag, "$.input", "index")) { return true; }
          }
        }
      }
    }
  },


  // Return the first element from an array matching a JSONpath string.
  // If that would be an array, return the first element of that but do
  // not recurse further. This makes inline JSONpath, etc. expedient.
  //
  // If a path/key spec is provided and the path matches the array, the key will
  // be used to choose a property of the object (or of the first object in case
  // of an array array) and return the first value of the array contained therin
  // (going with this model of everything being an array)
  getFirst: function (spec, container) {
    if (typeof(spec) === "object") {
      var path = spec.path;
      var key = spec.key;
    } else {
      var path = spec;
    }
    // Unfortunately, neither a jsonPath of "$" nor an empty one
    // will return the top level object.
    if (path === "$") path = false;
    let result = path
      ? jsonPath(container, path)
      : [container];
    if (result === false) return null;
    let value = result[0];
    if (Array.isArray(value)) {
      if (key) {
        if (Array.isArray(value[0][key])) return value[0][key][0];
        else return value[0][key];
      }
      return value[0];
    } else {
      if (key) {
        if (Array.isArray(value[key])) return value[key][0];
        else return value[key];
      }
      return value;
    }
  },

  // Returns an array of all the spec.key of the objects returned by spec.path.
  // Lackign spec.key, simply returns the array of objects from spec.path. 
  //
  // TODO: these are coerced to arrays, presumably due to $.input ambiguity
  // can we stop doing that now that we're probably using getFirst() anywhere it'd come up?
  get: function (spec, container) {
    if (spec.path === "$") delete spec.path;
    var objs = spec.path
      ? jsonPath(container, spec.path)
      : [container];
    if (Array.isArray(objs)) {
      var results = [];
      if (spec.key) {
        for (let i = 0; i < objs.length; i++) {
          let result = objs[i][spec.key];
          // Tolerate sparsely populated keys
          if (typeof(result) !== 'undefined') {
            if (Array.isArray(result)) results.push(result);
            // Normalise singletons into arrays for now
            else results.push([result]);
          }
        }
      } else {
        results = objs;
        for (let i = 0; i < objs.length; i++) {
          if (!Array.isArray(results[i])) results[i] = [results[i]];
        }
      }
      return results;
    } else {
      return null;
    }
  },

  // Merges the array data onto obj.key according to the value of append.
  // Scalars are first coerced into an array. Replacing with undefined will
  // delete the key.
  //
  // APPEND -> push data onto array, or create new array containing data. If
  //           data is an array, merge with existing elements.
  //
  // CONCAT -> concatenate string data onto the end of every string element of
  // array. If nothing is there, create a new string property.
  // *  TODO: should this create a new element if nothing is there? It didn't
  //          before (which makes sense from a concat-to-every-element
  //          perspective) but I think it's most useful for single-string-arrays
  //          and in that context it makes sense, especially since map has been
  //          updated to completely prune empty keys.
  //
  //          Except then there is a preceding space. And what if you want to
  //          add it to the front of the string and how should this even work?
  // * TODO: currently takes only the first element of the array, should it
  //         join with spaces?
  //
  // REPLACE -> replaces the current property with data if it's an array, or
  //            an array containing data if not.
  addToArrayProp: function(data, obj, key, append) {
    if (!key) throw new Error("Key parameter not specified. Destination JSONpath must match objects and provide a key.");
    if (typeof data === 'undefined') {
      if (append === jsonPathHelper.APPEND || append === jsonPathHelper.CONCAT) {
        return;
      } else {
        delete obj[key];
        return;
      }
    }

    // scalar source handling - convert to singleton array to fit data model
    var dataArray = Array.isArray(data) ? data : [data];
    // create destination array if not present
    if (typeof obj[key] === 'undefined') obj[key] = [];
    // destination should be an array
    else if (!Array.isArray(obj[key])) obj[key] = [obj[key]];

    if (append === jsonPathHelper.APPEND) {
      obj[key] = obj[key].concat(dataArray);
    } else if (append === jsonPathHelper.CONCAT) {
      if (typeof dataArray[0] === 'string') {
        if (!obj[key]) obj[key] = [""];
        for (let j = 0; j < obj[key].length; j++) {
          if (typeof(obj[key][j]) === 'string') {
            obj[key][j] += dataArray[0];
          }
        }
      }
    } else { // REPLACE
      obj[key] = dataArray;
    }
  },
  // Takes an array and adds it to the spec.key array property of
  // every object from spec.path.
  //
  // Non-array input is coerced into an array by addToArrayProp().
  //
  // Undefined input won't create an element. Replacing with undefined will
  // delete the target key.
  //
  // spec.append determines how the array is merged (REPLACE is default):
  //
  set: function (spec, data, container) {
    if (!spec.key) throw new Error("Key parameter not specified. Destination JSONpath must match objects and provide a key.");
    if (spec.path === "$") delete spec.path;
    var objs = spec.path
      ? jsonPath(container, spec.path)
      : [container];
    for (let i = 0; i < objs.length; i++) {
      var obj = objs[i];
      jsonPathHelper.addToArrayProp(data, obj, spec.key, spec.append);
    }
  },

  // Runs func() against the src.key array property of each object returned by
  // src.path and uses jsonPathHelper.set() to add it to the dst.key property
  // of each object returned by dst.path.
  //
  // src.path and dst.path must yield the same number of objects.
  //
  // src.key is optional
  //
  // src may yield scalars which will be wrapped in arrays
  //
  // dst.key is not optional as Javascript has no scalar references
  //
  // if src.key is "*" will ignore dst.key and instead map every key in src to
  // the same key in dst, applying func() to each.
  //
  // If func() returns "##NOOP##" for an object, set() is not called and the dst
  // object is unchanged. NB: if func() returns undefined set() will delete
  // the target if it is to replace it.
  map: function (src, dst, func, container) {
    var srcObjs = src.path
      ? jsonPath(container, src.path)
      : [container];

    if (srcObjs === undefined) logger.warn("source path " + src.path + " has no match");

    if (typeof(dst) !== "object") {
      var dst = src;
    }

    if (src.path == dst.path) {
      var dstObjs = srcObjs;
    } else {
      var dstObjs = dst.path
        ? jsonPath(container, dst.path)
        : [container];
      if (dstObjs === undefined) throw new Error("destination path " + dst.path + " has no match");
      else if (srcObjs.length !== dstObjs.length) {
        throw new Error(
          "Cannot map between JSONpaths that return differing numbers of objects:\n"
          + srcObjs.length + " source objects mapping to " + dstObjs.length + " destinations"
        );
        return
      }
    }

    for (let i = 0; i < srcObjs.length; i++) {
      let dstObj = dstObjs[i];
      // build an object to map all keys from through func() 
      // and onto keys of the same name in dstObj. facilitates
      // key wildcard and other potential extensions
      let srcObj = {};
      // without src.key we're taking this for an array of scalars
      if (!src.key) srcObj[dst.key] = srcObjs[i];
      else if (src.key === '*') srcObj = srcObjs[i];
      else srcObj[dst.key] = srcObjs[i][src.key];
      for (dstKey in srcObj) {
        srcArr = srcObj[dstKey];
        if ((typeof srcArr !== 'undefined') && (!Array.isArray(srcArr))) srcArr = [srcArr];
        let transformed = func(srcArr);
        if (transformed !== "##NOOP##")
        jsonPathHelper.addToArrayProp(transformed, dstObj, dstKey, dst.append);
      }
    }
  },

  // Runs func() against each element of the array in the src.key property of
  // every object returned by src.path and collects the results in an array
  // which is returned to be stored in dst according to jsonPathHelper.map().
  // Does not support sparse arrays.
  //
  //
  // Skipping elements: 
  //
  // If func() returns undefined, nothing is pushed and mapping continues with
  // the next element. If the resulting array ends up with no elements,
  // undefined is returned to the parent jsonPathHelper.map() call.
  // Accordingly, the dst.key property will be deleted from the object.
  //
  //
  // Skipping objects:
  //
  // Should func() ever or "##NOOP##", this is not pushed to the replacement
  // array.  Instead, it is returned to the outer jsonPathHelper.map() which
  // will leave the entire dst property unchanged and execution will proceed to
  // mapping the next object.
  //
  //
  // Empty source:
  //
  // An undefined (ie. mapElements called against a non-existent key) or empty
  // source array results in undefined being passed to func() exactly once.
  // Here one might return a single value to be pushed to the results as a
  // default, undefined to create an empty result as above or "##NOOP##" to
  // move on without effect.
  //
  // If you want the step to error out instead of moving on, throw a stepError
  // in func().
  mapElements: function (src, dst, func, container) {
    jsonPathHelper.map(src, dst, function (value) {
      // Mapping from an empty source, pass undefined
      if (typeof value === 'undefined' || (Array.isArray(value) && value.length === 0)) {
        let result = func(undefined);
        if (typeof(result) === 'undefined' || result === '##NOOP##') return result;
        else return [result];
      }
      if (Array.isArray(value)) {
        if (value.length === 0) {
          if (options.emptySourceNoop) return dst;
          else return undefined;
        }
        var transformed = [];
        for (let j = 0; j < value.length; j++) {
          let result = func(value[j]);
          if (result === '##NOOP##') return result;
          if (typeof(result) !== 'undefined') transformed.push(result);
        }
        if (transformed.length === 0) return undefined;
        else return transformed;
      }
    }, container);
  },

  // Helper for step.upgrade of the old "postProc" steps
  upgradePostProc: function (conf) {
    if (conf.in !== undefined) {
      let inKeys = [];
      let outKeys = [];
      if (conf.container === 'INPUT') {
        inKeys.push('input');
        outKeys.push('input');
      } else if (conf.container !== undefined) {
        inKeys.push('output');
        outKeys.push('output');
        if (conf.container !== 'NONE') {
          if (conf.container === 'item') {
            inKeys.push('results');
            inKeys.push('item');
            outKeys.push('results');
            outKeys.push('item');
          } else {
            inKeys.push(conf.container);
            outKeys.push(conf.container);
          }
        }
      }

      inKeys.push(conf.in);
      outKeys.push(conf.out);

      conf.in = jsonPathHelper.specFromArray(inKeys);
      if (conf.out !== undefined) {
        conf.out = jsonPathHelper.specFromArray(outKeys);
      }
    }
    delete(conf.container);
  },



  // Functions for handling inline JSONpaths in braces (eg. {$.output.hits})

  // From a string with inline JSONpaths returns an object with two properties:
  // * pruned: the string with JSONpaths removed
  // * paths: an array of objects each with path and offset properties
  processInline: function (str) {
    let match = null;
    let inlineRegex = /\{(\$\..+?)\}/g;
    let offsetOffset = 0;
    let paths = [];
    while ((match = inlineRegex.exec(str)) !== null) {
      let obj = {};
      obj.path = match[1];
      obj.offset = match.index - offsetOffset;
      paths.push(obj);
      offsetOffset += obj.path.length + 2;
    }
    let pruned = str.replace(inlineRegex, "");
    return {pruned: pruned, paths:paths};
  },

  // Given the result of processInline and an object to provide the data,
  // inserts the results of the JSONpath queries into the pruned string and
  // returns
  applyInline: function (obj, container) {
    if (typeof(obj) === "object") {
      if (obj.paths.length > 0) {
        let pruned = obj.pruned;
        let lastOffset = 0;
        let components = [];
        for (let i = 0; i < obj.paths.length; i++) {
          let inline = obj.paths[i];
          components.push(pruned.slice(lastOffset, inline.offset));
          lastOffset = inline.offset;
          let match = jsonPathHelper.getFirst(inline.path, container);
          if (match !== false) {
            components.push(match);
          }
        }
        components.push(pruned.slice(lastOffset));
        return components.join("");
      } else {
        return obj.pruned;
      }
    }
  },

  // Given a string with inline JSONpaths and an object, returns the string
  // with the output of jsonPathHelper.getFirst() substituted for each path.
  inlineReplace: function (str, container) {
    let processed = jsonPathHelper.processInline(str);
    return jsonPathHelper.applyInline(processed, container);
  },

  // Replace inline JSONpaths in a nodeSpec
  inlineReplaceNodeSpec: function (nodeSpec, container) {
    // copy nodeSpec
    let newSpec = JSON.parse(JSON.stringify(nodeSpec));
    newSpec.xpath = this.inlineReplace(newSpec.xpath, container);
    if (Array.isArray(newSpec.frames)) {
      for (let i = 0; i < newSpec.frames.length; i++) {
        newSpec.frames[i] = this.inlineReplace(newSpec.frames[i], container);
      }
    }
    return newSpec;
  },


  // Little helpet to produce a short displayable string from the jsonPath
  // spec, especially for the step box
  displayString: function (spec) {
    if ( typeof(spec) != "object" )
      return "??";
    if ( spec.key )
      return spec.key;
    if ( spec.path )
      return spec.path.replace( /^[.]+\./, "");  // the last component
    return "???";
  },

  unitTest: function () {
    // strings so we can build new objects out of them
    let record = '{"results":[{"author":["billy","bob"], "title":["a book"], "item":[{"location":["a library"]}, {"location":["my house"]}]}, {"author":["jill"], "title":["a paper"], "journaltitle":"unexpected scalars"}]}';

    let result;
    let tests;

    // getFirst
    logger.info("TESTING: jsonPathHelper.getFirst()");
    tests = [
    {
      label: "record (spec)",
      spec: {path:"$", key:"results"},
      obj: JSON.parse(record),
      expected: '{"author":["billy","bob"],"title":["a book"],"item":[{"location":["a library"]},{"location":["my house"]}]}',
    },

    {
      label: "record (string)",
      spec: "$.results",
      obj: JSON.parse(record),
      expected: '{"author":["billy","bob"],"title":["a book"],"item":[{"location":["a library"]},{"location":["my house"]}]}',
    },

    {
      label: "record item",
      spec: {path:"$.results[*].item", key:"location"},
      obj: JSON.parse(record),
      expected: '"a library"',
    },

    {
      label: "undefined key",
      spec: {path:"$.results[*].item", key:"shenanigans"},
      obj: JSON.parse(record),
      expected: undefined,
    },

    {
      label: "no array",
      spec: {path:"$.results[*].item", key:"shenanigans"},
      obj: {},
      expected: null,
    },
    ];

    for (let i = 0; i < tests.length; i++) {
      let test = tests[i];
      try {
        result = jsonPathHelper.getFirst(test.spec, test.obj);
      } catch (e) {
        logger.error(e);
        return false
      }
      if (result === test.expected || JSON.stringify(result) === test.expected) {
        logger.info(" * " + test.label + " OK");
      } else {
        logger.error(" * " + test.label + " FAIL");
        logger.error("Expected: ");
        logger.error(test.expected);
        logger.error("Actual:");
        logger.error(JSON.stringify(result));
        return false;
      }
    }

    // get
    logger.info("TESTING: jsonPathHelper.get()");
    tests = [
    {
      label: "one result, no path",
      spec: {key:"results"},
      obj: JSON.parse(record),
      expected: '[[{"author":["billy","bob"],"title":["a book"],"item":[{"location":["a library"]},{"location":["my house"]}]},{"author":["jill"],"title":["a paper"],"journaltitle":"unexpected scalars"}]]',
    },

    {
      label: "two results",
      spec: {path:"$.results[*]", key:"author"},
      obj: JSON.parse(record),
      expected: '[["billy","bob"],["jill"]]',
    },

    {
      label: "sparse",
      spec: {path:"$.results[*]", key:"item"},
      obj: JSON.parse(record),
      expected: '[[{"location":["a library"]},{"location":["my house"]}]]',
    },

    {
      label: "singleton",
      spec: {path:"$.results[*]", key:"journaltitle"},
      obj: JSON.parse(record),
      expected: '[["unexpected scalars"]]',
    },

    {
      label: "non-matching key",
      spec: {path:"$.results[*]", key:"shenanigans"},
      obj: JSON.parse(record),
      expected: '[]',
    },
    ];

    for (let i = 0; i < tests.length; i++) {
      let test = tests[i];
      try {
        result = jsonPathHelper.get(test.spec, test.obj);
      } catch (e) {
        logger.error(e);
        return false
      }
      if (result === test.expected || JSON.stringify(result) === test.expected) {
        logger.info(" * " + test.label + " OK");
      } else {
        logger.error(" * " + test.label + " FAIL");
        logger.error("Expected: ");
        logger.error(test.expected);
        logger.error("Actual:");
        logger.error(JSON.stringify(result));
        dump (result);
        return false;
      }
    }


    // set
    logger.info("TESTING: jsonPathHelper.set()");
    tests = [
    {
      label: "populated, append (create)",
      spec: {append:jsonPathHelper.APPEND, path:"$.results[1]", key:"item"},
      data: [{"location":["some box"]}],
      obj: JSON.parse(record),
      expected: '{"results":[{"author":["billy","bob"],"title":["a book"],"item":[{"location":["a library"]},{"location":["my house"]}]},{"author":["jill"],"title":["a paper"],"journaltitle":"unexpected scalars","item":[{"location":["some box"]}]}]}',
    },

    {
      label: "populated, append (string)",
      spec: {append:jsonPathHelper.APPEND, path:"$.results[*]", key:"author"},
      data: 'tom',
      obj: JSON.parse(record),
      expected: '{"results":[{"author":["billy","bob","tom"],"title":["a book"],"item":[{"location":["a library"]},{"location":["my house"]}]},{"author":["jill","tom"],"title":["a paper"],"journaltitle":"unexpected scalars"}]}',
    },

    {
      label: "populated, append (array)",
      spec: {append:jsonPathHelper.APPEND, path:"$.results[*]", key:"author"},
      data: ['dick', 'harry'],
      obj: JSON.parse(record),
      expected: '{"results":[{"author":["billy","bob","dick","harry"],"title":["a book"],"item":[{"location":["a library"]},{"location":["my house"]}]},{"author":["jill","dick","harry"],"title":["a paper"],"journaltitle":"unexpected scalars"}]}',
    },

    {
      label: "populated, concat",
      spec: {append:jsonPathHelper.CONCAT, path:"$.results[*]", key:"author"},
      data: ' the amazing',
      obj: JSON.parse(record),
      expected: '{"results":[{"author":["billy the amazing","bob the amazing"],"title":["a book"],"item":[{"location":["a library"]},{"location":["my house"]}]},{"author":["jill the amazing"],"title":["a paper"],"journaltitle":"unexpected scalars"}]}',
    },

    {
      label: "populated, replace (create)",
      spec: {path:"$.results[*].item[*]", key:"availabile"},
      data: 'MISSING',
      obj: JSON.parse(record),
      expected: '{"results":[{"author":["billy","bob"],"title":["a book"],"item":[{"location":["a library"],"availabile":["MISSING"]},{"location":["my house"],"availabile":["MISSING"]}]},{"author":["jill"],"title":["a paper"],"journaltitle":"unexpected scalars"}]}',
    },

    {
      label: "populated, replace",
      spec: {path:"$.results[*].item[*]", key:"availabile"},
      data: 'MISSING',
      obj: JSON.parse(record),
      expected: '{"results":[{"author":["billy","bob"],"title":["a book"],"item":[{"location":["a library"],"availabile":["MISSING"]},{"location":["my house"],"availabile":["MISSING"]}]},{"author":["jill"],"title":["a paper"],"journaltitle":"unexpected scalars"}]}',
    },

    {
      label: "empty, replace",
      spec: {path:"$.results[*].item[*]", key:"availabile"},
      data: 'MISSING',
      obj: {},
      expected: '{}',
    },

    ];

    for (let i = 0; i < tests.length; i++) {
      let test = tests[i];
      try {
        jsonPathHelper.set(test.spec, test.data, test.obj);
      } catch (e) {
        logger.error(e);
        return false
      }
      if (result === test.expected || JSON.stringify(test.obj) === test.expected) {
        logger.info(" * " + test.label + " OK");
      } else {
        logger.error(" * " + test.label + " FAIL");
        logger.error("Expected: ");
        logger.error(test.expected);
        logger.error("Actual:");
        logger.error(JSON.stringify(test.obj));
        dump (result);
        return false;
      }
    }

    // map
    logger.info("TESTING: jsonPathHelper.map()");
    tests = [
    {
      label: "in place (no dst)",
      src: {path:"$.results[*]", key:"author"},
      func: function (el) {
        return el.toUpperCase();
      },
      obj: JSON.parse(record),
      isArrayMap: false,
      expected: '{"results":[{"author":["BILLY","BOB"],"title":["a book"],"item":[{"location":["a library"]},{"location":["my house"]}]},{"author":["JILL"],"title":["a paper"],"journaltitle":"unexpected scalars"}]}',
    },

    {
      label: "new key",
      src: {path:"$.results[*]", key:"author"},
      dst: {path:"$.results[*]", key:"AUTHOR"},
      func: function (el) {
        return el.toUpperCase();
      },
      obj: JSON.parse(record),
      isArrayMap: false,
      expected: '{"results":[{"author":["billy","bob"],"title":["a book"],"item":[{"location":["a library"]},{"location":["my house"]}],"AUTHOR":["BILLY","BOB"]},{"author":["jill"],"title":["a paper"],"journaltitle":"unexpected scalars","AUTHOR":["JILL"]}]}',
    },

    {
      label: "in place (concat)",
      src: {path:"$.results[*]", key:"author"},
      dst: {append:jsonPathHelper.CONCAT, path:"$.results[*]", key:"author"},
      func: function (el) {
        return el.toUpperCase();
      },
      obj: JSON.parse(record),
      isArrayMap: false,
      // strange but true! concat/append/replace is an option at the array level not element and uses getFirst()
      expected: '{"results":[{"author":["billyBILLY","bobBILLY"],"title":["a book"],"item":[{"location":["a library"]},{"location":["my house"]}]},{"author":["jillJILL"],"title":["a paper"],"journaltitle":"unexpected scalars"}]}',
    },

    {
      label: "in place (append)",
      src: {path:"$.results[*]", key:"author"},
      dst: {append:jsonPathHelper.APPEND, path:"$.results[*]", key:"author"},
      func: function (el) {
        return el.toUpperCase();
      },
      obj: JSON.parse(record),
      isArrayMap: false,
      expected: '{"results":[{"author":["billy","bob","BILLY","BOB"],"title":["a book"],"item":[{"location":["a library"]},{"location":["my house"]}]},{"author":["jill","JILL"],"title":["a paper"],"journaltitle":"unexpected scalars"}]}',
    },

    {
      label: "all keys (append)",
      src: {path:"$.results[*]", key:"*"},
      dst: {append:jsonPathHelper.APPEND, path:"$.results[*]", key:"ignored"},
      func: function (el) {
        if (typeof el !== 'object') return el.toUpperCase();
        else return '##NOOP##';
      },
      obj: JSON.parse(record),
      isArrayMap: false,
      expected: '{"results":[{"author":["billy","bob","BILLY","BOB"],"title":["a book","A BOOK"],"item":[{"location":["a library"]},{"location":["my house"]}]},{"author":["jill","JILL"],"title":["a paper","A PAPER"],"journaltitle":["unexpected scalars","UNEXPECTED SCALARS"]}]}',
    },

    {
      label: "in place (sparse, replace scalar)",
      src: {path:"$.results[*]", key:"journaltitle"},
      func: function (el) {
        if (typeof el !== 'undefined') return el.toUpperCase();
        else return undefined;
      },
      obj: JSON.parse(record),
      isArrayMap: false,
      expected: '{"results":[{"author":["billy","bob"],"title":["a book"],"item":[{"location":["a library"]},{"location":["my house"]}]},{"author":["jill"],"title":["a paper"],"journaltitle":["UNEXPECTED SCALARS"]}]}',
    },

    {
      label: "append array (something out of nothing)",
      src: {path:"$.results[*]", key:"item"},
      dst: {append:jsonPathHelper.APPEND, path:"$.results[*]", key:"item"},
      func: function (arr) {
        return [{location:["Amazon"]}];
      },
      obj: JSON.parse(record),
      isArrayMap: true,
      expected: '{"results":[{"author":["billy","bob"],"title":["a book"],"item":[{"location":["a library"]},{"location":["my house"]},{"location":["Amazon"]}]},{"author":["jill"],"title":["a paper"],"journaltitle":"unexpected scalars","item":[{"location":["Amazon"]}]}]}',
    },

    {
      label: "concat scalar (something out of nothing)",
      src: {path:"$.results[*].item[*]", key:"neverwas"},
      dst: {append:jsonPathHelper.CONCAT, path:"$.results[*].item[*]", key:"location"},
      func: function (el) {
        return " building";
      },
      obj: JSON.parse(record),
      isArrayMap: false,
      expected: '{"results":[{"author":["billy","bob"],"title":["a book"],"item":[{"location":["a library building"]},{"location":["my house building"]}]},{"author":["jill"],"title":["a paper"],"journaltitle":"unexpected scalars"}]}',
    },

    {
      label: "do nothing (array) on empty source",
      src: {path:"$.results[*]", key:"neverwas"},
      dst: {append:jsonPathHelper.REPLACE, path:"$.results[*]", key:"author"},
      func: function (arr) {
        return "##NOOP##";
      },
      obj: JSON.parse(record),
      isArrayMap: true,
      expected: '{"results":[{"author":["billy","bob"],"title":["a book"],"item":[{"location":["a library"]},{"location":["my house"]}]},{"author":["jill"],"title":["a paper"],"journaltitle":"unexpected scalars"}]}',
    },

    {
      label: "change no elements if one is bob",
      src: {path:"$.results[*]", key:"author"},
      dst: {append:jsonPathHelper.REPLACE, path:"$.results[*]", key:"author"},
      func: function (el) {
        if (el === 'bob') return '##NOOP##';
        return el.toUpperCase();
      },
      obj: JSON.parse(record),
      isArrayMap: false,
      expected: '{"results":[{"author":["billy","bob"],"title":["a book"],"item":[{"location":["a library"]},{"location":["my house"]}]},{"author":["JILL"],"title":["a paper"],"journaltitle":"unexpected scalars"}]}',
    },

    {
      label: "replace (nothing out of something)",
      src: {path:"$.results[*]", key:"author"},
      func: function (el) {
        return el === "bob" ? "gwen" : undefined;
      },
      obj: JSON.parse(record),
      isArrayMap: false,
      expected: '{"results":[{"author":["gwen"],"title":["a book"],"item":[{"location":["a library"]},{"location":["my house"]}]},{"title":["a paper"],"journaltitle":"unexpected scalars"}]}',
    },

    {
      label: "replace (leave well enough alone)",
      src: {path:"$.results[*]", key:"author"},
      func: function (el) {
        return el === "bob" ? "gwen" : el;
      },
      obj: JSON.parse(record),
      isArrayMap: false,
      expected: '{"results":[{"author":["billy","gwen"],"title":["a book"],"item":[{"location":["a library"]},{"location":["my house"]}]},{"author":["jill"],"title":["a paper"],"journaltitle":"unexpected scalars"}]}',
    },

    {
      label: "js filter (create)",
      src: {path:"$.results[0]", key:"title"},
      dst: {path:"$[((@.creative = []) && 'creative')][((@[0] = {}) && 0)]", key:"new"},
      func: function (el) {
        return el.toUpperCase();
      },
      obj: JSON.parse(record),
      isArrayMap: false,
      expected: '{"results":[{"author":["billy","bob"],"title":["a book"],"item":[{"location":["a library"]},{"location":["my house"]}]},{"author":["jill"],"title":["a paper"],"journaltitle":"unexpected scalars"}],"creative":[{"new":["A BOOK"]}]}',
    },
    {
      label: "map scalar source",
      src: {path:"$.results[*].title"},
      dst: {path:"$.results[*]", key:"title"},
      func: function (el) {
        return el.toUpperCase();
      },
      obj: JSON.parse(record),
      isArrayMap: false,
      expected: '{"results":[{"author":["billy","bob"],"title":["A BOOK"],"item":[{"location":["a library"]},{"location":["my house"]}]},{"author":["jill"],"title":["A PAPER"],"journaltitle":"unexpected scalars"}]}',
    },
    ];

    for (let i = 0; i < tests.length; i++) {
      let test = tests[i];
      try {
        if (test.isArrayMap) jsonPathHelper.map(test.src, test.dst, test.func, test.obj);
        else jsonPathHelper.mapElements(test.src, test.dst, test.func, test.obj);
      } catch (e) {
        logger.error(e);
        return false
      }
      if (result === test.expected || JSON.stringify(test.obj) === test.expected) {
        logger.info(" * " + test.label + " OK");
      } else {
        logger.error(" * " + test.label + " FAIL");
        logger.error("Expected: ");
        logger.error(test.expected);
        logger.error("Actual:");
        logger.error(JSON.stringify(test.obj));
        return false;
      }
    }

    return true;
  },
}
