var EXPORTED_SYMBOLS = ["queryHelper"];

// Small helper functions for query processing
// Written by Heikki

Components.utils.import('resource://indexdata/runtime/StepError.js');
Components.utils.import('resource://indexdata/util/logging.js');

var logger = logging.getLogger();

var queryHelper = {

  /////////////////////////////////////
  // Term processing

  // Truncation wildcards
  // Checks that the asked truncation is supported, and applies it
  // Returns the term with truncation, as in 'watermel*'
  // Throws StepErrors for unsupported things.
  trunc_term: function ( node, wildcard, L,R,LR,Mask ) {
    var term = node.term;
    if ( node.truncation && node.truncation != "none" &&  node.truncation != "not") {
      if ( node.truncation == "left" &&  L ) {
          term = wildcard + term;
      } else if ( node.truncation == "right" && R ) {
          term = term + wildcard;
      } else if ( node.truncation == "both" && LR ) {
          term = wildcard + term + wildcard;
      } else if ( node.truncation == "z39.58" && Mask ) {
        if ( term.indexOf("#") != -1 )
          throw new StepError("Single-character masking with '#' not supported");
        term = term.replace ( /\?/g, wildcard );
      } else {
          throw new StepError("Truncation '" + node.truncation + "'" +
              " not supported");
      }
    }
    return term;
  },

  // Put quotes around a string (presumably a truncated term),
  // depending on the structure attribute in the node, and the
  // flags. Handles three types of quoting: phrase, word, and default
  // (default is when the node does not specify. Should always be the
  // same as word, but for historical reasons, fullquery allows the user
  // to specify it). For each of these, we have a flag indicating
  // if it is supported, and strings for the left and right quote, as
  // these may be different (think [] or xml tags).
  // Almost all parameters have some kind of semi- decent defaults:
  // string from node.term, all forms supported, phrases with ", everything
  // else without quotes.
  // Throws StepErrors in case it meets somethign explicitly marked as
  // unsupported, or something weird we have decided not to support.
  // TODO - This is ugly coding. Pass the flag and strings in a structure
  // or something!
  quote_string: function ( node, string,
                           phrase, phrase_L, phrase_R,
                           word, word_L, word_R,
                           def, def_L, def_R ){
    if ( string === undefined ) string = node.term;
    if ( phrase_L === undefined ) phrase_L = '"';
    if ( phrase_R === undefined ) phrase_R = phrase_L;
    if ( word === undefined ) word = true;
    if ( word_L === undefined ) word_L = "";
    if ( word_R === undefined ) word_R = word_L;
    if ( def === undefined ) def = true;
    if ( def_L === undefined ) def_L = word_L;
    if ( def_R === undefined ) def_R = word_R;
    if ( node.structure == "phrase" && phrase ) {
      string = phrase_L + string + phrase_R;
    } else if ( node.structure == "word" && word ) {
      string = word_L + string + word_R;
    } else if ( node.structure == undefined || node.structure == "" ||
         node.structure == "key" ) { // historical reasons, key is also default
      string = def_L + string + def_R ;
    } else
      throw new StepError ("Structure '" + node.structure + "' not supported");
    logger.debug("qh: s=" + string + "  WL=" + word_L + " WR=" + word_R +
        " DL=" + def_L + " DR=" + def_R );
    return string;
  },

  ////////////////////////////
  // Validating attributes
  // TODO - Make routines to check all attributes. Throw errors on
  // attributes we know we will not support. The '_no_' functions
  // check that we don't have such attributes specified (or if we
  // have, they are the harmless defaults). All routines take an
  // optional message to use when throwing the error, so the caller
  // can be more specific.

  validate_position: function(node,msg) {
    if ( node.position && node.position != "any" )
      if (msg)
        throw new StepError(msg);
      else 
        throw new StepError("Position not supported");
  },
  
  validate_completeness: function(node,msg) {
    if ( node.completeness && node.completeness != "incompletesubfield" )
      if (msg)
        throw new StepError(msg);
      else
        throw new StepError("Completeness not supported");
  },

  // Check that the node has no truncation specified
  validate_no_trunc: function(node, msg) {
    if ( node.truncation && node.truncation != "not" ) 
      if (msg)
        throw new StepError(msg);
      else
        throw new StepError("Truncation not supported");
  },

  // Check that the node has no phrase or other structure defined
  validate_no_struct: function(node, msg) {
    if ( node.structure && node.structure != "word" &&
         node.structure != "key" ) 
      if (msg)
        throw new StepError(msg);
      else
        throw new StepError("Structure '" + node.structure + "' not supported");
  },

  // Check that the node has no phrase or other structure defined
  validate_no_relation: function(node, msg) {
    if ( node.relation && node.relation != "eq" &&
         node.relation  != "relevance" )
      if (msg)
        throw new StepError(msg);
      else
        throw new StepError("Relation not supported");
  },
  
  /////////////////////////////////////
  // year range magic


  // Check if a relation means 'equal'
  // At the moment, 'eq' and 'relevance' do, as do nodes without a relation
  is_equal_relation_node: function ( node ) {
    if ( ! node ) return false; // should not happen
    if ( ! node.relation ) return true;
    if ( node.relation == "eq" || node.relation == "relevance" ) return true;
    return false;
  },

  
  // Check if a relational node
  // That is, a node with a term and a relation (that is not 'equal')
  is_relation_node: function( node ) {
    if ( node && node.term && node.relation &&
        ! this.is_equal_relation_node(node) )
      return true;
    return false;
  },

  // Check if a range node
  // That is, an 'and' node with two term nodes, that have same field,
  // and both have relations
  is_range_node: function( node ) {
    if ( node && node.op && (node.op == "and") &&
         queryHelper.is_relation_node(node.s1) && 
         queryHelper.is_relation_node(node.s2) &&
         node.s1.field == node.s2.field ) 
      return true;
  return false;
  },

  // Create a helper structure for start/end points
  // Both arguments are optional
  new_endpoints: function ( start, end ) {
    var endpoints = { start:"", end:"" };
    if ( start )
      endpoints.start = start;
    if ( end )
      endpoints.end = end;
    return endpoints;
  },

  // extract one endpoint from a node.
  // An endpoint is a node that has a relation set up.
  // Assumes you have already decided that this is the proper field.
  // Puts it into the proper place in the endpoints structure
  // Since the endpoints are inclusive, works best with 'le' and 'ge',
  // but if called with 'lt' or 'gt', will make a suitable endpoint by
  // adding/subtracting one. This can go wrong for non-numerical values,
  // in which case throws a StepError
  // If the relation is 'eq', attempts to set both end points. You may
  // want to check for this and do something else.
  // Tries its best to combine the new years with the ones in endpoints,
  // ( y>=2000 && y>2002 => y>=2002). If such combining results in an
  // impossible range, throws an error.
  range_endpoint: function ( node, endpoints ) {
  var start = "";
  var end = "";
  if (!node)
    return;
  if ( node.op && node.op != "and" )
    return; // can't make ranges from OR nodes
  var rel = node.relation;
  if ( queryHelper.is_equal_relation_node(node) )
    rel = "eq";
  if ( rel == "lt" ) {
    var y = parseInt(node.term);
    if ( isNaN(y) )
      throw new StepError( node.field + " must be numerical when " +
        "comparing with 'lt'. ('" + node.term + "' is not)" );
    end = "" + (y-1) ;
  } else if ( rel == "le" ) {
    end = node.term;
  } else if ( rel == "ge" ) {
    start = node.term;
  } else if ( rel == "gt" ) {
    var y = parseInt(node.term);
    if ( isNaN(y) )
      throw new StepError( node.field + " must be numerical when " +
        "comparing with 'gt'. ('" + node.term + "' is not)" );
    start = "" + (y+1);
  } else if ( rel == "eq" ) {
    start = node.term;
    end = node.term;
  } else {
    throw new StepError( "Unsupported relation '" + rel + "'" );
  }

  // Combine new and old end points
  if ( start ) {
    if ( !endpoints.start || start > endpoints.start )
      endpoints.start = start;
  }
  if ( end ) {
    if ( !endpoints.end || end < endpoints.end )
      endpoints.end = end;
  }
  // Check if we ended up with an impossible range
  // These can come from many weird queries, not only the
  // obvious y>2009 and y<2008, but also y=2000 and y=2002, etc
  if ( endpoints.start && endpoints.end &&
        endpoints.start > endpoints.end )
    throw new StepError("Impossible range " +
        endpoints.start + "-" + endpoints.end );
}, // range_endpoint


  
} // queryHelper