#!/usr/bin/python

import argparse;
import numpy;
import re;
import pprint;

# FYI this generates a --help
parser = argparse.ArgumentParser(description='Compare timings from CFWS stdout.')
parser.add_argument('--task', help='task', required=True)
parser.add_argument('--occurrence', metavar='N', help='nth occurrence in file (from 1)', type=int, required=True)
parser.add_argument('--context', metavar='N', help='top/bottom of ordered lists', type=int, default=3)
parser.add_argument('--before', nargs='+', help='baseline files', required=True)
parser.add_argument('--after', nargs='+', help='same number of test files', required=True)
args = vars(parser.parse_args())

searchstring = 'timing ' + args['task']

filecount = len(args['before'])
if filecount != len(args['after']):
  raise Exception("Not the same number of files before and after")

def timing(file, targetocc):
  occcount = 0
  occurrence = False
  for line in open(file):
    if line.startswith(searchstring):
      time = float(re.search('real=([0-9.]+) ', line).group(1))
      if time:
        occurrence = time
        occcount += 1
    if occcount == targetocc:
      occurrence = time
      break
  return occurrence

results = [];
for i in range(filecount):
  result = {};
  beforefile = args['before'][i]
  afterfile = args['after'][i]
  before = timing(beforefile, args['occurrence'])
  after = timing(afterfile, args['occurrence'])
  if before and after:
    result = {
      'beforefile': beforefile,
      'before': before,
      'afterfile': afterfile,
      'after': after,
      'delta': before - after,
      'deltapercent': 100 * ((before - after)/before),
    }
    results.append(result)

def print_stats(resultkey):
  context = args['context']
  results_bykey = sorted(results, key=lambda k: k[resultkey])
  just_values = [occurrence[resultkey] for occurrence in results_bykey]
  print ("Mean: ".rjust(22) + str(numpy.mean(just_values)))
  print ("Median: ".rjust(22) + str(numpy.median(just_values)))
  print ("Standard deviation: ".rjust(22) + str(numpy.std(just_values)))
  print ("Top %d: ".rjust(22)) % context
  for result in results_bykey[context*-1:]:
    print '                      ' + str(result[resultkey]) + ' (%(before)f -> %(after)f, %(beforefile)s)' % result
  print ("Bottom %d: ".rjust(22)) % context
  for result in results_bykey[:context]:
    print '                      ' + str(result[resultkey]) + ' (%(before)f -> %(after)f, %(beforefile)s)' % result

print("Comparing timings for two runs of task " + searchstring)
print("Time before")
print_stats('before')
print("Time after")
print_stats('after')
print("Difference between before and after (seconds):")
print_stats('delta')
print
print("Difference between before and after (percent):")
print_stats('deltapercent')
