/* This file is part of the Connector Framework
 * Copyright (C) 2008-2013 Index Data
 * See the file LICENSE for details.
 */
/**
 * \file cf_engine.cpp
 * \brief Implementation of CF_Embed / CF_Engine.
 */
#include "mozilla-config.h"
#include "cf_config.h"
#include <stdlib.h>
#include <errno.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <assert.h>
#include <unistd.h>
#include <signal.h>
#include <poll.h>
#include <yaz/xmalloc.h>
#include <yaz/snprintf.h>
#include <yaz/log.h>
#include "nsStringAPI.h"

#include "embed.h"
#include "moz_web_view.h"
#include "cf_engine.h"
#include "cf_embed.h"
#include "connector_wrap.h"
#include <gtk/gtk.h>

GtkWidget *g_notebook;
GtkWidget *g_entry;
GtkWidget *g_statusbar;

CF_Embed *global_cf_embed = 0;

static void SetProxyHTTP(MozApp *mozApp, CF_Logger *logger,
                         const char *host, int port)
{
    const int type = 1;
    nsresult rv;

    // https://developer.mozilla.org/en/Mozilla_Networking_Preferences

    rv = mozApp->SetCharPref("network.proxy.http", host);
    if (NS_FAILED(rv))
        logger->printf("engine", YLOG_WARN, "network.proxy.http failed "
                       "host=%s rv=%08x", host, rv);

    rv = mozApp->SetIntPref("network.proxy.http_port", port);
    if (NS_FAILED(rv))
        logger->printf("engine", YLOG_WARN, 
                       "network.proxy.http_port failed port=%d rv=%08x",
                       port, rv);

    rv = mozApp->SetCharPref("network.proxy.ssl", host);
    if (NS_FAILED(rv))
        logger->printf("engine", YLOG_WARN,
                       "network.proxy.ssl failed host=%s rv=%08x", host, rv);

    rv = mozApp->SetIntPref("network.proxy.ssl_port", port);
    if (NS_FAILED(rv))
        logger->printf("engine", YLOG_WARN,
                       "network.proxy.ssl_port failed port=%d rv=%08x",
                       port, rv);

    rv = mozApp->SetIntPref("network.proxy.type", type);
    if (NS_FAILED(rv))
        logger->printf("engine", YLOG_WARN,
                       "network.proxy.type 1 failed type=%d "
                       " rv=%08x", type, rv);
}

void
title_cb(MozWebView *view, const char *title, gpointer user_data)
{
    gtk_label_set_text(GTK_LABEL(user_data), title);
}

void
location_cb(MozWebView *view, const char *uri, gpointer user_data)
{
    gtk_entry_set_text (GTK_ENTRY(g_entry), uri);
}

void
status_cb(MozWebView *view, const char *status, gpointer user_data)
{
    gtk_statusbar_push(GTK_STATUSBAR(g_statusbar), 0, status);
}

void
activate_cb(GtkWidget *widget, gpointer user_data)
{
    const char *uri = gtk_entry_get_text (GTK_ENTRY (widget));

    int id = gtk_notebook_get_current_page (GTK_NOTEBOOK (g_notebook));
    GtkWidget *view = gtk_notebook_get_nth_page (GTK_NOTEBOOK (g_notebook), id);
    moz_web_view_load_uri(MOZ_WEB_VIEW(view), uri);
}

void
close_cb(GtkWidget *button, GtkWidget *view)
{
    int page = gtk_notebook_page_num(GTK_NOTEBOOK(g_notebook), view);
    gtk_notebook_remove_page(GTK_NOTEBOOK(g_notebook), page);
}

GtkWidget *
add_page (GtkNotebook *notebook)
{
    GtkWidget *view = moz_web_view_new();
    GtkWidget *box = gtk_hbox_new(FALSE, 5);
    GtkWidget *tab_label = gtk_label_new("Loading...");
    GtkWidget *close_button = gtk_button_new();
    gtk_button_set_relief(GTK_BUTTON(close_button), GTK_RELIEF_NONE);

    GtkWidget *close = gtk_image_new_from_stock(GTK_STOCK_CLOSE,
						GTK_ICON_SIZE_MENU);
    gtk_container_add(GTK_CONTAINER(close_button), close);

    gtk_box_pack_start(GTK_BOX(box), tab_label, TRUE, TRUE, 0);
    gtk_box_pack_end(GTK_BOX(box), close_button, FALSE, FALSE, 0);

    gtk_widget_show_all(box);
    gtk_widget_show(view);

    g_signal_connect(view, "title-changed", G_CALLBACK(title_cb), tab_label);
    g_signal_connect(view, "location-changed", G_CALLBACK(location_cb), NULL);
    g_signal_connect(view, "status-changed", G_CALLBACK(status_cb), NULL);
    g_signal_connect(close_button, "clicked", G_CALLBACK(close_cb), view);

    moz_web_view_load_uri(MOZ_WEB_VIEW(view), "chrome://cfengine/content/engine.xul");
    int page = gtk_notebook_append_page(notebook, view, box);
    gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), page);

    return view;
}

void
new_tab_cb(GtkToolItem *item, void *user_data)
{
    add_page(GTK_NOTEBOOK(g_notebook));
}

gboolean child_input_handler(GIOChannel *source,
			     GIOCondition condition,
			     gpointer data)
{
    if (condition == G_IO_IN)
        get_CF_Embed()->input_handler(source);
    else if (condition == G_IO_HUP)
        gtk_main_quit();
    else
    {
        get_CF_Embed()->logger->printf("engine", YLOG_WARN,
                                       "unexpected event=%d",
                                       (int) condition);
        gtk_main_quit();
    }
    return TRUE;

}

void CF_Embed::main()
{
    const char *str = display.set_display();
    if (str)
        logger->printf("engine", YLOG_LOG, "DISPLAY=%s", str);

    global_cf_embed = this;

    task_timing = yaz_timing_create();

    gtk_init(&argc, &argv); // must be called before XRE_NotifyProfile

    mozApp = new MozApp(logger);

    if (NS_FAILED(mozApp->init(app_path, profile_path)))
    {
        write_response("FAIL\n", 5);
        // we not listening to pipe input by exiting here
        close(response_fd[1]);
        response_fd[1] = -1;
        return ;
    }

    GtkWidget *window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    g_signal_connect(window, "delete_event", gtk_main_quit, NULL);

    GtkWidget *box = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(window), box);

    GtkWidget *toolbar = gtk_toolbar_new();
    gtk_box_pack_start(GTK_BOX(box), toolbar, FALSE, FALSE, 0);

    g_notebook = gtk_notebook_new();

    g_entry = gtk_entry_new();
    g_signal_connect(g_entry, "activate", G_CALLBACK(activate_cb), g_notebook);

    GtkToolItem *toolitem = gtk_tool_item_new();
    gtk_container_add(GTK_CONTAINER(toolitem), g_entry);
    gtk_tool_item_set_expand(toolitem, TRUE);
    gtk_toolbar_insert(GTK_TOOLBAR(toolbar), toolitem, 0);

    toolitem = gtk_tool_button_new_from_stock(GTK_STOCK_ADD);
    gtk_tool_button_set_label(GTK_TOOL_BUTTON(toolitem), "New Tab");
    gtk_toolbar_insert(GTK_TOOLBAR(toolbar), toolitem, 1);
    g_signal_connect(toolitem, "clicked", G_CALLBACK(new_tab_cb), g_notebook);

    gtk_box_pack_start(GTK_BOX(box), g_notebook, TRUE, TRUE, 0);

    g_statusbar = gtk_statusbar_new();
    gtk_box_pack_start(GTK_BOX(box), g_statusbar, FALSE, FALSE, 0);

    gtk_window_set_default_size(GTK_WINDOW(window), 1024, 768);

    add_page(GTK_NOTEBOOK(g_notebook));

    gtk_widget_show_all (window);

    GdkWindow *gdkwindow = window->window;
    main_window_xid = GDK_DRAWABLE_XID(gdkwindow);

    GIOChannel *channel_input = g_io_channel_unix_new(request_fd[0]);
    g_io_add_watch(channel_input, (GIOCondition) (G_IO_IN|G_IO_HUP),
                   child_input_handler, 0);

    yaz_timing_stop(task_timing);
    logger->printf("timing", YLOG_LOG, "boot user=%f real=%f sys=%f",
            yaz_timing_get_user(task_timing),
            yaz_timing_get_real(task_timing),
            yaz_timing_get_sys(task_timing));
    if (proxy_host)
        SetProxyHTTP(mozApp, logger, proxy_host, proxy_port);
    gtk_main();
}

void task_result_handler(void *vp, bool success, const char *str)
{
    get_CF_Embed()->result_handler(success, str);
}

void test_finished(void *vp, bool success)
{
    get_CF_Embed()->result_handler(success, 0);
}

void test_task_done(void *vp, bool success, const char *str)
{
    if (!success)
    {
        // test failed
        get_CF_Embed()->result_handler(false, str);
    }
}

void CF_Embed::result_handler(bool success, const char *str)
{
    WRBUF w = wrbuf_alloc();
    if (!success)
        wrbuf_puts(w, "FAIL");
    else
        wrbuf_printf(w, "OK");
    if (str)
    {
        wrbuf_puts(w, " ");
        wrbuf_puts_replace_char(w, str, '\n', ' ');
    }
    wrbuf_puts(w, "\n");
    write_response(wrbuf_buf(w), wrbuf_len(w));
    wrbuf_destroy(w);
    yaz_timing_stop(task_timing);
    logger->printf("timing", YLOG_LOG, "%s user=%f real=%f sys=%f",
                   wrbuf_cstr(current_task),
                   yaz_timing_get_user(task_timing),
                   yaz_timing_get_real(task_timing),
                   yaz_timing_get_sys(task_timing));
    fflush(stdout);
}

int CF_Embed::write_request(WRBUF w)
{
    ssize_t r = 0;
    size_t off = 0;

    signal(SIGPIPE, SIG_IGN);
    while (off < wrbuf_len(w))
    {
        r = write(request_fd[1], wrbuf_buf(w) + off, wrbuf_len(w) - off);
        if (r == -1)
        {
            if (errno == EINTR)
                continue;
            return -1;
        }
        off += r;
    }
    return 0;
}

void CF_Embed::write_response(const void *buf, size_t count)
{
    size_t r = write(response_fd[1], buf, count);
    if (r != count)
    {
        logger->printf("engine", YLOG_WARN,
                       "write_response returned %lld, expected %lld",
                       (long long) r, (long long) count);
    }
}

void CF_Embed::input_handler(GIOChannel *source)
{
    gchar *str_return;
    gsize length;
    GError *gerror = 0;
    for (;;)
    {
        GIOStatus status =
            g_io_channel_read_line(source, &str_return, &length, 0, &gerror);
        if (status == G_IO_STATUS_ERROR)
        {
            logger->printf("engine", YLOG_WARN,
                           "g_io_channel_read_line failed: %s",
                           gerror->message);
            WRBUF w = wrbuf_alloc();
            wrbuf_printf(w, "FAIL g_io_channel_read_line failed: %s\n",
                         gerror->message);
            write_response(wrbuf_buf(w), wrbuf_len(w));
            wrbuf_destroy(w);
            gtk_main_quit();
        }
        if (status != G_IO_STATUS_AGAIN)
            break;
    }
    if (length)
    {
	WRBUF w = wrbuf_alloc(); // zero-length means no response (yet)
        char *cp = str_return;
        const char *cmd = cp;
        const char *first_arg = "";
        const char *extra_args = "";

        if (str_return[length-1] == '\n')
            str_return[length-1] = '\0';

        // split into 3 tokens, cmd, first_arg, extra_args - separted by blanks
        // Perform in-place null'ing..
        for (; *cp && *cp != ' '; cp++)
            ;
        if (*cp)
        {
            *cp++ = '\0'; // terminate cmd
            for (first_arg = cp; *cp && *cp != ' '; cp++)
                ;
            if (*cp)
            {
                *cp++ = '\0'; // terminate first_arg
                while (*cp == ' ')
                    cp++;
                extra_args = cp;
            }
        }
        if (!strcmp(cmd, "load_cf"))
        {
            char *errormessage = 0;
            if (connector->loadConf(first_arg, extra_args, &errormessage))
                wrbuf_printf(w, "OK\n");
            else
            {
                wrbuf_printf(w, "FAIL load_cf");
                if (errormessage)
                    wrbuf_printf(w, ": %s", errormessage);
                wrbuf_puts(w, "\n");
            }
        }
        else if (!strcmp(cmd, "save_cf"))
        {
            char *errormessage = 0;
            if (connector->saveConf(first_arg, &errormessage))
                wrbuf_printf(w, "OK\n");
            else
            {
                wrbuf_printf(w, "FAIL save_cf");
                if (errormessage)
                    wrbuf_printf(w, ": %s", errormessage);
                wrbuf_puts(w, "\n");
            }
        }
        else if (!strcmp(cmd, "run_script"))
        {
            char *errormessage = 0;
            if (connector->runScript(first_arg, &errormessage))
                wrbuf_printf(w, "OK\n");
            else
            {
                wrbuf_printf(w, "FAIL run_script");
                if (errormessage)
                    wrbuf_printf(w, ": %s", errormessage);
                wrbuf_puts(w, "\n");
            }
        }
        else if (!strcmp(cmd, "run_task"))
        {
            yaz_timing_start(task_timing);
            wrbuf_rewind(current_task);
            wrbuf_puts(current_task, first_arg);
            connector->runTask(first_arg, false,
                               extra_args, 0, task_result_handler);
        }
        else if (!strcmp(cmd, "run_task_opt"))
        {
            yaz_timing_start(task_timing);
            wrbuf_rewind(current_task);
            wrbuf_puts(current_task, first_arg);
            connector->runTask(first_arg, true,
                               extra_args, 0, task_result_handler);
        }
        else if (!strcmp(cmd, "run_tests"))
        {
            yaz_timing_start(task_timing);
            wrbuf_rewind(current_task);
            wrbuf_puts(current_task, first_arg);
            connector->runTests(first_arg, 0, test_task_done,
                                test_finished);
        }
        else if (!strcmp(cmd, "unit_test"))
        {
            yaz_timing_start(task_timing);
            wrbuf_rewind(current_task);
            wrbuf_puts(current_task, extra_args);
            if (connector->unitTest(extra_args))
                wrbuf_printf(w, "OK\n");
            else
                wrbuf_printf(w, "FAIL\n");
        }
        else if (!strcmp(cmd, "dom_string"))
        {
            logger->printf("engine", YLOG_DEBUG,
                           "dom_string command handler.\n");
            char *dom_str = 0;
            if (connector->DOMString(&dom_str))
            {
                get_CF_Embed()->result_handler(true, dom_str);
            }
            else
                wrbuf_printf(w, "FAIL\n");
        }
        else if (!strcmp(cmd, "quit"))
        {
            wrbuf_printf(w, "OK\n");
            delete mozApp;
            mozApp = 0;
            gtk_main_quit();
        }
        else
        {
            wrbuf_printf(w, "FAIL %s\n", cmd);
        }
        fflush(stdout);
        if (wrbuf_len(w))
            write_response(wrbuf_buf(w), wrbuf_len(w));
	wrbuf_destroy(w);
    }
    g_free(str_return);
}

CF_Embed *get_CF_Embed()
{
    return global_cf_embed;
}

CF_Engine::CF_Engine(enum Flags flags,
                     const char *builder_path,
                     const char *app_path,
                     const char *profile_path, const char *proxy,
                     const char *loglevel,
                     CF_Logger *logger, int fd, int capt_stdout)
{
    m_priv = 0;
    init(flags, 0, 0, builder_path, app_path, profile_path, proxy,
         loglevel, logger, fd, capt_stdout);
}

bool CF_Embed::wait_init_handshake(std::string &error_msg)
{
    yaz_timing_t myt = yaz_timing_create();
    yaz_timing_start(myt);

    const char *output = 0;
    bool sys_err = false;
    bool res = wait_result_parent(&output, 20000, &sys_err);
    yaz_timing_stop(myt);

    logger->printf("timing", YLOG_LOG,
                   "handshake user=%f real=%f sys=%f",
                   yaz_timing_get_user(myt),
                   yaz_timing_get_real(myt),
                   yaz_timing_get_sys(myt));
    yaz_timing_destroy(&myt);

    if (!res)
    {
        if (output && *output)
        {
            logger->printf("engine", YLOG_LOG,
                           "engine create fail: %s", output);
            error_msg = output;
        }
        else
        {
            error_msg = "engine creation failure";
        }
        return false;
    }
    if (output)
        main_window_xid = atol(output);
    return true;
}

bool CF_Embed::spawn_fork(std::string &error_msg, int fd, int capt_stdout)
{
    int i, lfd = -1;
    FILE *yaz_log_FILE = yaz_log_file();
    if (yaz_log_FILE)
        lfd = fileno(yaz_log_FILE);

    pid = ::fork();
    switch (pid)
    {
    case 0: /* child */
        // close all files except log files and pipe for communication */
        for (i = getdtablesize(); --i > 2; )
        {
            if (i != lfd && i != fd &&
                i != response_fd[1] && i != request_fd[0])
                close(i);
        }
        response_fd[0] = -1;
        request_fd[1] = -1;
        signal(SIGTERM, SIG_DFL);
        if (fd != -1)
        {
            close(0);
            if (capt_stdout)
                dup(fd);
            close(1);
            if (capt_stdout)
                dup(fd);
            close(2);
            if (capt_stdout)
                dup(fd);
        }
        main();
        exit(0);
    case -1: /* error */
        throw CF_Engine_Error_System("fork system call failed");
        error_msg = "fork system call failed";
        return false;
    default:
        /* parent */
        close(response_fd[1]);
        response_fd[1] = -1;
        close(request_fd[0]);
        request_fd[0] = -1;

        logger->printf("engine", YLOG_LOG,
                       "CF fork: parentpid=%lld childpid=%lld",
                       (long long) getpid(), (long long) pid);

        return wait_init_handshake(error_msg);
    }
    return true;
}

static void *thread_handler(void *p)
{
    CF_Embed *em = (CF_Embed *) p;
    em->main();
    return 0;
}

bool CF_Embed::spawn_thread(std::string &err_msg)
{
    pthread_create(&child_thread, 0, thread_handler, this);
    using_threads = true;
    return wait_init_handshake(err_msg);
}

void CF_Engine::init(enum Flags flags,
                     int argc, char **argv, const char *builder_path,
                     const char *app_path, const char *profile_path,
                     const char *proxy, const char *loglevel,
                     CF_Logger *logger, int fd, int capt_stdout)
{
    if (!builder_path)
	throw CF_Engine_Error_System("builder_path NULL");
    CF_Embed *embed = new CF_Embed(argc, argv,
                                   builder_path, app_path, profile_path,
                                   proxy, loglevel, logger,
                                   fd);

    embed->logger->printf("engine", YLOG_LOG,
                          "init: flags=%d, builder_path=%s app_path=%s "
                          "profile_path=%s proxy=%s", flags,
                          builder_path ? builder_path : "null",
                          app_path ? app_path : "null",
                          profile_path ? profile_path : "null",
                          proxy ? proxy : "null");
#if FORCE_DISPLAY
    // hack to spread display to several, already running, Xvfb's
    char display_str[40];

    // :2.0,:3.0,  ..., :150.0
    int no = ((getpid()) % 149U) + 2;

    sprintf(display_str, ":%d.0", no);
    setenv("DISPLAY", display_str, 1);

    embed->logger->printf("engine", "display=%s", display_str);
#endif

    std::string msg;
    bool ret = false;
    if ((flags & 1) == FORK)
    {
        ret = embed->spawn_fork(msg, fd, capt_stdout);
    }
    else
    {
        ret = embed->spawn_thread(msg);
    }
    if (!ret)
    {
        delete embed;
        throw CF_Engine_Error_System(msg.c_str());
    }
    m_priv = embed;
}

const char *CF_Embed::get_module_path()
{
    return module_path;
}

const char *CF_Embed::get_app_path()
{
    return app_path;
}

CF_Engine::~CF_Engine()
{
    delete m_priv;
}

void CF_Embed::printf(const char *module, int level, const char *fmt, ...)
{
    va_list ap;
    char buf[4096];

    va_start(ap, fmt);
    yaz_vsnprintf(buf, sizeof(buf)-1, fmt, ap);
    ::printf("%s %s\n", module, buf);
    va_end(ap);
}


void CF_Embed::copy_initial_js_files(const char *a_profile_path,
                                     const char *a_app_path,
                                     const char *loglevel)
{
    WRBUF src_fname = wrbuf_alloc();
    WRBUF dst_fname = wrbuf_alloc();
    const char *fnames[] = {"prefs.js", "user.js", 0};
    int i;
    assert(a_profile_path);

    if (!a_app_path)
        a_app_path = ".";

    for (i = 0; fnames[i]; i++)
    {
        wrbuf_rewind(src_fname);
        wrbuf_rewind(dst_fname);
        wrbuf_printf(src_fname, "%s/defaults/preferences/%s",
                     a_app_path, fnames[i]);
        wrbuf_printf(dst_fname, "%s/%s", a_profile_path, fnames[i]);
        FILE *src_file = fopen(wrbuf_cstr(src_fname), "r");
        if (src_file)
        {
            FILE *dst_file = fopen(wrbuf_cstr(dst_fname), "w");
            if (dst_file)
            {
                while (1)
                {
                    char buf[256];
                    size_t rd = fread(buf, 1, sizeof(buf), src_file);
                    if (rd > 0)
                        fwrite(buf, 1, rd, dst_file);
                    if (rd < sizeof(buf))
                        break;
                }
                if (loglevel && *loglevel)
                    fprintf(dst_file,
                            "user_pref(\"indexdata.cf.logging.root\", \"%s\");\n",
                            loglevel);
                fclose(dst_file);
            }
            fclose(src_file);
        }
    }
    wrbuf_destroy(src_fname);
    wrbuf_destroy(dst_fname);
}

CF_Embed::CF_Embed(int a_argc, char **a_argv,
                   const char *a_module_path,
                   const char *a_app_path,
                   const char *a_profile_path,
		   const char *a_proxy,
                   const char *a_loglevel,
                   CF_Logger *a_logger,
                   int fd)
    : logger(a_logger),
      profile_temp(0),
      pid(0), connector(0),
      argc(a_argc),
      argv(a_argv),
      using_threads(false),
      mozApp(0), /* zero up here.. */
      task_timing(0),
      main_window_xid(1),
      display(a_logger)
{
    if (!logger)
        logger = this;

    if (!display.lock(fd))
    {
        const char *e = display.get_error();

        logger->printf("engine", YLOG_FATAL, "display lock: %s", e);

        throw CF_Engine_Error_System(e);
    }

    if (!a_proxy)
        proxy_host = 0;
    else
    {
        const char *cp = strrchr(a_proxy, ':');
        if (!cp || cp == a_proxy)
            throw CF_Engine_Error_System("proxy: missing port");
        proxy_port = atoi(cp + 1);
        if (proxy_port <= 0)
        {
            throw CF_Engine_Error_System("proxy: bad port");
        }
        proxy_host = xstrndup(a_proxy, cp - a_proxy);
    }

    if (pipe(request_fd) < 0)
    {
        xfree(proxy_host);
        throw CF_Engine_Error_System("pipe system call failed");
    }
    if (pipe(response_fd) < 0)
    {
        xfree(proxy_host);
        close(request_fd[0]);
        close(request_fd[1]);
        throw CF_Engine_Error_System("pipe system call failed");
    }
    child_thread = 0;
    result_wrbuf = wrbuf_alloc();
    module_path = xstrdup(a_module_path);
    app_path = 0;
    if (a_app_path)
        app_path = xstrdup(a_app_path);

    if (!a_profile_path)
        a_profile_path = "/tmp/cf_profile_XXXXXX";

    profile_path = xstrdup(a_profile_path);

    if (strstr(profile_path, "XXXXXX"))
    {
        mkdtemp(profile_path);
        profile_temp = 1;
        copy_initial_js_files(profile_path, a_app_path, a_loglevel);
    }
    current_task = wrbuf_alloc();
}

CF_Embed::~CF_Embed()
{
    if (using_threads)
    {
        const char *output = 0;
        WRBUF w = wrbuf_alloc();
        wrbuf_puts(w, "quit\n");
        write_request(w);
        wrbuf_destroy(w);

        bool sys_err = false;
        wait_result_parent(&output, 20000, &sys_err);

        logger->printf("engine", YLOG_LOG, "waiting for threads to join");
        pthread_join(child_thread, 0); // and wait for it to stop
    }
    if (request_fd[0] != -1)
        close(request_fd[0]);
    if (request_fd[1] != -1)
        close(request_fd[1]);
    if (response_fd[0] != -1)
        close(response_fd[0]);
    if (response_fd[1] != -1)
        close(response_fd[1]);
    wrbuf_destroy(result_wrbuf);
    delete mozApp; // has no effect when removed for QUIT for threaded mode
    mozApp = 0;
    if (pid)
    {
        logger->printf("engine", YLOG_LOG, "kill %lld", (long long) pid);
        int r = kill(pid, SIGTERM);
        if (r == -1)
            logger->printf("engine", YLOG_WARN,
                           "kill %lld failed", (long long) pid);
        else
        {
            int status;
            waitpid(pid, &status, 0);
        }
    }
    if (profile_temp && profile_path)
    {
        struct stat buf;
        if (stat(profile_path, &buf) == 0)
        {
            WRBUF w = wrbuf_alloc();
            wrbuf_printf(w, "/bin/rm -r %s", profile_path);
            system(wrbuf_cstr(w));
            logger->printf("engine", YLOG_LOG, "remove %s", profile_path);
            wrbuf_destroy(w);
        }
    }
    if (task_timing)
        yaz_timing_destroy(&task_timing);
    wrbuf_destroy(current_task);
    xfree(module_path);
    xfree(app_path);
    xfree(profile_path);
    xfree(proxy_host);
}

bool CF_Embed::wait_result_parent(const char **output, int timeout_ms, bool *sys_err)
{
    *sys_err = false;
    wrbuf_rewind(result_wrbuf);
    int round = 1;
    int step_ms = 10000; // 10 secs
    while (1)
    {
	char result_line[1024];
        int r = 0;
        struct pollfd fds;
        fds.fd = response_fd[0];
        fds.events = POLLIN;
        fds.revents = 0;

        r = poll(&fds, 1, timeout_ms > step_ms ? step_ms : timeout_ms);
        if (r == 0)
        {
            timeout_ms -= step_ms;
            if (timeout_ms > 0)
            {
                logger->printf("engine", YLOG_LOG,
                               "poll waited for 10 secs. pid=%lld round=%d",
                               (long long) pid, round);
                round++;
                continue;
            }
            wrbuf_puts(result_wrbuf, "FAIL timeout");
            if (wrbuf_len(current_task) > 0)
            {
                wrbuf_puts(result_wrbuf, " - Task: ");
                wrbuf_puts(result_wrbuf, wrbuf_cstr(current_task));
            }
            else
                wrbuf_puts(result_wrbuf, " - no task");
            *sys_err = true;
            break;
        }
        else if (r == -1)
        {
            int x = errno;
            logger->printf("engine", YLOG_LOG, "poll: %s", strerror(x));
            if (x == EINTR)
                continue;
            wrbuf_printf(result_wrbuf, "FAIL poll: %s", strerror(x));
            *sys_err = true;
            break;
        }
	r = read(response_fd[0], result_line, sizeof(result_line)-1);
        if (r == -1)
        {
            int x = errno;
            wrbuf_printf(result_wrbuf, "FAIL read: %s", strerror(x));
            *sys_err = true;
            break;
        }
	if (r < 1)
	    break;
	result_line[r] = '\0';
	char *nl = strchr(result_line, '\n');
	if (nl)
	{
	    *nl = '\0';
	    wrbuf_puts(result_wrbuf, result_line);
	    break;
	}
	wrbuf_puts(result_wrbuf, result_line);
    }
    const char *cp = wrbuf_cstr(result_wrbuf);
    if (output)
    {
	const char *cp1 = strchr(cp, ' ');
	*output = cp1 ? cp1+1 : "";
    }
    if (!memcmp(cp, "OK", 2))
    {
	return true;
    }
    return false;
}

void CF_Embed::get_result_parent(const char *default_msg, const char **output)
{
    bool sys_err = false;
    assert(output);
    // wait up to two minutes for ANY operation
    if (!wait_result_parent(output, 120000, &sys_err))
    {
        if (*output && **output)
        {
            if (sys_err)
            {
    	        logger->printf("engine", YLOG_WARN, 
                               "sys err for %s", default_msg);
                throw CF_Engine_Error_System(*output);
            }
	    else
                throw CF_Engine_Error_Task(*output);
        }
        else
            throw CF_Engine_Error_Task(default_msg);
    }
}

void CF_Embed::load_cf_parent(const char *cf_file, const char *session_json)
{
    const char *output = 0;
    if (request_fd[1] == -1)
    {
	throw CF_Engine_Error_System("load_cf but no browser");
    }
    WRBUF w = wrbuf_alloc();
    if (session_json)
    {
      wrbuf_printf(w, "load_cf %s %s\n", cf_file, session_json);
    }
    else
    {
      wrbuf_printf(w, "load_cf %s\n", cf_file);
    }
    write_request(w);
    wrbuf_destroy(w);

    get_result_parent("load_cf", &output);
}

void CF_Engine::load_cf(const char *cf_file, const char *session_json)
{
    m_priv->load_cf_parent(cf_file, session_json);
}

void CF_Embed::save_cf_parent(const char *cf_file)
{
    const char *output = 0;
    if (request_fd[1] == -1)
    {
	throw CF_Engine_Error_System("save_cf but no browser");
    }
    WRBUF w = wrbuf_alloc();
    wrbuf_printf(w, "save_cf %s\n", cf_file);

    write_request(w);
    wrbuf_destroy(w);

    get_result_parent("save_cf", &output);
}

void CF_Engine::save_cf(const char *cf_file)
{
    m_priv->save_cf_parent(cf_file);
}

void CF_Embed::run_script_parent(const char *script_file)
{
    const char *output = 0;
    if (request_fd[1] == -1)
    {
	throw CF_Engine_Error_System("run_script but no browser");
    }
    WRBUF w = wrbuf_alloc();
    wrbuf_printf(w, "run_script %s\n", script_file);

    write_request(w);
    wrbuf_destroy(w);

    get_result_parent("run_script", &output);
}

void CF_Engine::run_script(const char *script_file)
{
    m_priv->run_script_parent(script_file);
}

/* function to ensure our input parameters are one line only (JSON) */
static void add_task_input(WRBUF w, const char *input)
{
    for (; *input; input++)
        if (strchr("\r\n", *input))
            wrbuf_putc(w, ' ');
        else
            wrbuf_putc(w, *input);
}

void CF_Embed::run_task_parent(const char *taskName, bool optional,
                               const char *input,
                               const char **output)
{
    if (request_fd[1] == -1)
    {
	throw CF_Engine_Error_System("run_task but no browser");
    }
    wrbuf_rewind(current_task);
    wrbuf_puts(current_task, taskName);
    WRBUF w = wrbuf_alloc();
    wrbuf_printf(w, "%s %s ", optional ? "run_task_opt" : "run_task", taskName);
    add_task_input(w, input);
    wrbuf_puts(w, "\n");
    write_request(w);
    wrbuf_destroy(w);

    get_result_parent("run_task", output);
}

void CF_Engine::run_task(const char *taskName, const char *input,
			 const char **output)
{
    m_priv->run_task_parent(taskName, false, input, output);
}

void CF_Engine::run_task_opt(const char *taskName, const char *input,
                             const char **output)
{
    m_priv->run_task_parent(taskName, true, input, output);
}

bool CF_Embed::run_tests_parent(const char *task_order)
{
    const char *output = 0;
    if (request_fd[1] == -1)
    {
	throw CF_Engine_Error_System("run_task but no browser");
    }
    WRBUF w = wrbuf_alloc();
    wrbuf_puts(w, "run_tests ");
    add_task_input(w, task_order);
    wrbuf_puts(w, "\n");
    write_request(w);
    wrbuf_destroy(w);

    get_result_parent("run_tests", &output);
    return true;
}

bool CF_Engine::run_tests(const char *task_order)
{
    return m_priv->run_tests_parent(task_order);
}

bool CF_Embed::unit_test_parent(const char *name)
{
    const char *output = 0;
    if (request_fd[1] == -1)
    {
	throw CF_Engine_Error_System("run_task but no browser");
    }
    WRBUF w = wrbuf_alloc();
    wrbuf_puts(w, "unit_test - "); // task name is '-'
    add_task_input(w, name);
    wrbuf_puts(w, "\n");
    write_request(w);
    wrbuf_destroy(w);

    get_result_parent("unit_test", &output);
    return true;
}

bool CF_Engine::unit_test(const char *name)
{
    return m_priv->unit_test_parent(name);
}

bool CF_Embed::screen_shot_parent(const char *filename)
{
    const char *display_name = display.get_display();
    WRBUF w = wrbuf_alloc();

    wrbuf_printf(w, "xwd -id %lld ", (long long) main_window_xid);
    if (display_name)
        wrbuf_printf(w, "-display %s ", display_name);

    wrbuf_printf(w, "| xwdtopnm | pnmtopng >%s", filename);
    int ret = system(wrbuf_cstr(w));

    wrbuf_destroy(w);
    if (ret)
        return false;
    return true;
}

bool CF_Engine::screen_shot(const char *filename)
{
    return m_priv->screen_shot_parent(filename);
}

void CF_Embed::dom_string_parent(char **retval)
{
    WRBUF w = wrbuf_alloc();
    wrbuf_printf(w, "dom_string\n");
    write_request(w);
    wrbuf_destroy(w);
    get_result_parent("dom_string", (const char **) retval);
}

void CF_Engine::dom_string(char **retval)
{
    m_priv->dom_string_parent(retval);
}

void CF_Embed::browser_ready(ConnectorWrap *connector)
{
    this->connector = connector;

    char response_str[60];
    sprintf(response_str, "OK %lu\n", (unsigned long) main_window_xid);

    write_response(response_str, strlen(response_str));
}

void CF_Embed::browser_failure(const char *msg)
{
    gtk_main_quit();
    write_response("FAIL\n", 5);
}

CF_Engine_Error::CF_Engine_Error(const char *msg)
{
    this->msg = xstrdup(msg);
}

CF_Engine_Error::~CF_Engine_Error()
{
    xfree(msg);
}

const char *CF_Engine_Error::what()
{
    return msg;
}

const char *CF_Logger::logstr(int level)
{
    if (level & YLOG_FATAL)
        return "fatal";
    if (level & YLOG_DEBUG)
        return "debug";
    if (level & YLOG_WARN)
        return "warn";
    if (level & YLOG_LOG)
        return "log";
    return "-";
}

/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */

