var EXPORTED_SYMBOLS = ["StepTree"];
Components.utils.import('resource://indexdata/util/textHelper.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/objHelper.js');
Components.utils.import('resource://indexdata/util/logging.js');
Components.utils.import("resource://indexdata/ui/app.js");

var logger = logging.getLogger();

//save some typing
var child = xmlHelper.appendNode;

var StepTree = function () {};

// callbacks can include: onChange, onSelect, onDeselect
StepTree.prototype.init = function (tree, callbacks) {
  this.tree = tree;
  this.cb = {}; 
  this.cb.onChange = callbacks.onChange || function () {};
  this.cb.onSelect = callbacks.onSelect || function () {};
  this.cb.onDeselect = callbacks.onDeselect || function () {};
  this.selectedStep = null;
  this.selectedIndex = null;
  this.ignoreSelectEvents = false;
};

StepTree.prototype.setTask = function (task) {
  this.task = task;
};

StepTree.prototype.getValidSelection = function () {
  let items = [];
  let indices = [];
  let start = {};
  let end = {};
  let treeView = this.tree.view;
  let numRanges = treeView.selection.getRangeCount();
  //copy
  for (let i=0; i<numRanges; i++) {
    treeView.selection.getRangeAt(i, start, end);
    for (let j=start.value; j<=end.value; j++) {
      //blocks are drawn as tree items and may be selected but we skip them
      let item = treeView.getItemAtIndex(j).itemModel;
      if (item.getClassName() != 'Block' ) {
        //it's not a block, check if the parent is selected
        let p = item.block ? item.block.parentStep : null;
        if (p == null || (items.indexOf(p) == -1)) {
          items.push(item);
          indices.push(j);
        }
      }
    }
  }
  return [indices, items];
};

StepTree.prototype.getCurrentBlockAndIndex = function () {
  //default to root block
  let block = this.task.block;
  let index = block.getItemsSize()-1;
  let treeIndex = this.tree.currentIndex;
  if (treeIndex != -1) {
    let item = this.tree.view.getItemAtIndex(treeIndex).itemModel;
    if (item.getClassName() == 'Block' ) {
      block = item;
      index = block.getItemsSize()-1;
    } else {
      block = item.block;
      index = block.findItemIndex(item);
    }
  }
  return [block, index];
};

StepTree.prototype.refresh = function () {
  logger.debug("About to refresh step tree");
  let task = this.task;
  if (task == null) {
    logger.debug("Task is null, ignore refreshing steps");
    return;
  }
  logger.debug("selected task is "+task.name);
  let treechildren = this.tree.ownerDocument.getElementById("cfStepTreeContents");
  //make sure our callbacks ignore any select events triggered when
  //the tree is refreshed
  this.ignoreSelectEvents = true;
  xmlHelper.emptyChildren(treechildren);
  if (task.block.getItemsSize() < 1) {
    this.cb.onDeselect();
    return;
  }
  let block = task.block;
  //initialize indent array
  let indent = [];
  for (let i=0; i<block.getItemsSize(); i++) {
    indent[i] = 0;
  }
  //task size
  let treeIndex = {value: 0, firstIndexInSelection: -1, numSelected: 0};
  for (let i=0; i<block.getItemsSize(); i++) {
    let step = block.getItemAt(i);
    initIndentFrom(step, indent, i);
    if (step.getClassName() === "Comment") {
      logger.debug("will draw Comment at "+treeIndex.value);
      this.drawCommentItem(treechildren, step, i, treeIndex, 
          indent[i], false);
    } else {
      logger.debug("will draw "+step.getDisplayName()+" at "+treeIndex.value);
      this.drawStepItem(treechildren, step, i,  treeIndex, indent[i], false);
    }
  }
  this.ignoreSelectEvents = false;
  //scroll to first item in selection
  if (treeIndex.firstIndexInSelection !== -1) {
    this.tree.treeBoxObject.ensureRowIsVisible
      (treeIndex.firstIndexInSelection);
  }
  //toggle step pane for selected item (if only one, eg after adding
  //or hide it entriely (pasting multiselections, moving etc)
  if (treeIndex.numSelected === 1) {
    if (treeIndex.firstItemInSelection !== this.selectedStep)
      this.onItemSelected();
  } else if (treeIndex.numSelected > 1) {
    this.cb.onDeselect();
  } else {
    let rc = this.tree.view.rowCount;
    // hide step pane if no items to select
    if (rc < 0) this.cb.onDeselect();
    else {
      // we may have removed rows since last time, need to stay in bounds
      if (this.selectedIndex < rc) {
        this.tree.view.selection.select(this.selectedIndex);
      } else if (rc > 0) {
        this.tree.view.selection.select(rc - 1);
      }
      this.onItemSelected(); 
      this.tree.treeBoxObject.ensureRowIsVisible(this.selectedIndex);
    }  
  }
};

StepTree.prototype.drawCommentItem = function (container, step, blockIndex, 
                         treeIndex, indent, collapsed) {
  let commentText = "// " + step.body;
  commentText = commentText.replace( /\s+/g, " " );  // clean spaces
  let treeitem = child(container, "treeitem");
  treeitem.itemModel = step;
  let treerow = child(treeitem, "treerow");
  child(treerow, "treecell", null, {label: blockIndex});
  child(treerow, "treecell", null,
    {tooltiptext: step.body, label: commentText});
  restoreSelection(step, this.tree, treeIndex);
  if (!collapsed) treeIndex.value++;
};

StepTree.prototype.drawStepItem = function (container, step, blockIndex, treeIndex,
                      indent, collapsed) {
  //treeitem
  let item = child(container, "treeitem");
  //put references to the step so we can retrieve it through selection
  item.itemModel = step;
  //treerow
  let row = child(item, "treerow");
  let disabled = step.isDisabled == true;
  let tooltip = step.getComments().join("; ");
  if (step.isDeprecated()) {
    tooltip = "DEPRECATED - " + step.getDeprecationInfo();
  }
  let label = step.getDisplayName();
  let args = step.renderArgs();
  if (args) {
    label += ": " + args;
  }
  //ordinal
  let numcell = child(row, "treecell", null, 
      {label: blockIndex, 
      style: "min-width: 2em; max-width: 2em; padding: 0; margin: 0;",
      properties: step.isDisabled ? "inactive" : ""});
  //space
  //child(labelcell, "span", "\u00A0", null, HTMLNS);
  //indent dots
  for (let i=0; i<indent; i++) {
     label = "\u00B7 " + label;
  }
  //name
  let properties = [];
  if (step.isAlt) properties.push("alt");
  if (step.isDisabled) properties.push("inactive");
  let labelcell = child(row, "treecell", null,
    {label: label, tooltiptext: tooltip, properties: properties.join(' ')});
  restoreSelection(step, this.tree, treeIndex);
  if (!collapsed) treeIndex.value++;
  //if contain blocks
  if (step.blocks.length) {
    item.setAttribute('container', 'true');
    let open = !collapsed && !step.collapsed;
    item.setAttribute('open', open);
    item.setAttribute('empty', 'false');
    let children = child(item, 'treechildren');
    for (let i=0; i<step.blocks.length; i++) {
      let block = step.blocks[i];
      logger.debug("will draw block " + block.name);
      let open = !collapsed && !step.collapsed && !block.collapsed;
      let blockitem = child(children, 'treeitem', null,
          {container: true, open: open});
      blockitem.itemModel = block;
      if (!collapsed && !step.collapsed) treeIndex.value++;
      let blockrow = child(blockitem, 'treerow');
      //dummy cell to fill # column
      child(blockrow, 'treecell', null, {label: " "});
      let blockcell = child(blockrow, 'treecell', null, 
          {label: block.label+" "+step.renderBlockArgs(i), 
            properties: "block" + (step.isDisabled ? " inactive" : "")});
      let blockchildren = child(blockitem, 'treechildren');
      let indentArr = [];
      for (let j=0; j<block.getItemsSize(); j++) {
        indentArr[j] = 0;
      }
      for (let j=0; j<block.steps.length; j++) {
        let s = block.steps[j];
        initIndentFrom(s, indentArr, j);
        if (s.getClassName() === "Comment") {
          logger.debug("will draw Comment at "+treeIndex.value);
          this.drawCommentItem(blockchildren, s, j, treeIndex, 
              indentArr[j], (collapsed || step.collapsed || block.collapsed));
        } else {
          logger.debug("will draw "+step.getDisplayName()+" at "
              +treeIndex.value);
          this.drawStepItem(blockchildren, s, j,  treeIndex, 
              indentArr[j], (collapsed || step.collapsed || block.collapsed));
        }
      }
    }
  }
};

StepTree.prototype.cutSteps = function () {
  app.stepClip = [];
  //[indices,items], wish it was ruby
  let items = this.getValidSelection()[1];
  //we remove items starting from the end, otherwise the indices will mess up
  for (var i=items.length-1; i>=0; i--) {
    let b = items[i].block;
    let index = b.findItemIndex(items[i]);
    app.stepClip.unshift(b.deleteItem(index));
  }
  this.refresh();
  this.cb.onChange();
};

StepTree.prototype.copySteps = function () {
  app.stepClip = [];
  let items = this.getValidSelection()[1];
  for (var i=0; i<items.length; i++) {
    let b = items[i].block;
    let index = b.findItemIndex(items[i]);
    app.stepClip.push(b.getItemAt(index).clone());
  }
};

StepTree.prototype.pasteSteps = function () {
  if (!Array.isArray(app.stepClip) || app.stepClip.length === 0) {
    logger.debug("Cannot paste, buffer empty");
    return;
  }
  let blockAndIndex = this.getCurrentBlockAndIndex();
  let block = blockAndIndex[0];
  let index = blockAndIndex[1];
  for (var i=0; i<app.stepClip.length; i++) {
    var clone = app.stepClip[i].clone();
    block.insertItem(clone, index);
    if (clone.getClassName() !== "Comment") 
      block.task.attachStep(clone, true);
    index++;
    clone.markItemForNextSelection = true;
  }
  this.refresh();
  this.cb.onChange();
};

StepTree.prototype.toggleItemFlag = function (flag) {
  let items = this.getValidSelection()[1];
  for (let i = 0; i<items.length; i++) {
    let item = items[i];
    if (i === 0) var newState = !item[flag]  
    if (item.getClassName() === "Comment") continue;
    item[flag] = newState;
  }
  this.refresh();
  this.cb.onChange();
};

StepTree.prototype.newComment = function () { 
  let bAI = this.getCurrentBlockAndIndex();
  let block = bAI[0];
  let index = bAI[1];
  let i = block.addComment("", index);
  block.getItemAt(i).markItemForNextSelection = true; 
  this.refresh();
  this.cb.onChange();
};

StepTree.prototype.newStep = function (step_name) {
  let bAI = this.getCurrentBlockAndIndex();
  let block = bAI[0];
  let index = bAI[1];
  logger.debug("about to add new step at " + index);
  let i = block.newStep(step_name, index);
  logger.debug("added step at " + i);
  var step = block.getItemAt(i);
  if (!step.defaults) step.defaults = {};
  [step.task.connector.template, step.task.template].forEach(function (t) {
    if (t && t.defaults && t.defaults.steps && t.defaults.steps[step_name]) {
      objHelper.extend(step.defaults, t.defaults.steps[step_name]);
    } 
  });
  if (step.defaults.conf) objHelper.extend(step.conf, step.defaults.conf);
  step.markItemForNextSelection = true;
  this.refresh();
  this.cb.onChange();
};

StepTree.prototype.deleteSteps = function () {
  var items = this.getValidSelection()[1];
  //we remove items starting from the end, otherwise the indices will mess up
  for (var i=items.length-1; i>=0; i--) {
    let b = items[i].block;
    let index = b.findItemIndex(items[i]);
    b.deleteItem(index);
  }
  this.refresh();
  this.cb.onChange();
};

StepTree.prototype.stepMove = function(up /*up true*/) {
  let ii = this.getValidSelection();
  let items = ii[1];
  //organize selected items by blocks
  let entries = [];
  for (let i=0; i<items.length; i++) {
    let b = items[i].block;
    if (b.selectionIndex) {
      //block seen
      let entry = entries[b.selectionIndex-1];
      entry.items.push(items[i]);
    } else {
      b.selectionIndex = 
        entries.push({block: b, items: [items[i]]});
    }
  }
  for (let i=0; i<entries.length; i++) {
    let entry = entries[i];
    let b = entry.block;
    for (let j=0; j<entry.items.length; j++) {
      //we iterate backwards when moving down
      let indexInSelection = up ? j : entry.items.length-j-1;
      let item = entry.items[indexInSelection];
      let indexInBlock = b.findItemIndex(item);
      let selectionSize = entry.items.length;
      logger.debug(" moving selIdx " 
          + indexInSelection + " selSize " + selectionSize 
          + " blckIdx " + indexInBlock + " blockSize " + b.getItemsSize()); 
      if (up && indexInSelection < indexInBlock) {
        b.itemUp(indexInBlock);
      //when moving down
      } else if (!up && 
        (selectionSize-indexInSelection) < (b.getItemsSize()-indexInBlock)) {
        logger.debug("step is being moved down");
        b.itemDown(indexInBlock);
      }
      item.markItemForNextSelection = true;
    }
    b.selectionIndex = 0;
  }
  this.refresh();
  this.cb.onChange();
};

StepTree.prototype.selectAtOffset = function (offset) {
  let sel = this.tree.view.selection;
  let rows = this.tree.view.rowcount;
  if (rows === 0) return;
  let cur = this.selectedIndex;
  if (!cur) cur = 0; 
  let dst = offset + this.selectedIndex;
  if (dst < 0) dst = 0;
  if (dst >= this.tree.view.rowcount) dst = this.tree.view.rowcount - 1;
  if (dst === sel) return;
  sel.select(dst);
};

StepTree.prototype.onStepTreeClick = function (ev) {
  var tree =  this.tree;
  var treeBox = tree.treeBoxObject;
  var row = {}, col = {}, cell = {};
  // row.value has the row index
  // col.value has the column object
  treeBox.getCellAt(ev.clientX, ev.clientY, row, col, cell);
  // row out of bound
  if (row.value == -1) return;
  let treeitem = tree.view.getItemAtIndex(row.value);
  //remember collapsed
  if (cell.value == 'twisty') {
    let collapsed = treeitem.getAttribute('open') == 'false';
    treeitem.itemModel.collapsed = collapsed;
    this.cb.onChange();
  } else if (this.selectedIndex === row.value
    && new Date().getTime() - this.selectTime/1000 > 500) {
    // Need to redisplay if clicking current item, but want to avoid 
    // calling onSelect again if it was just called because of select.
    // Select has to happen first and click still has to be called. 
    logger.debug("Select time: " + this.selectTime + " Now: " + new Date().getTime() + " Diff: " + (new Date().getTime() - this.selectTime/1000));
    let item = treeitem.itemModel;
    this.cb.onSelect(item, row.value); 
  }
};

StepTree.prototype.onItemSelected = function () {
  let index = this.tree.currentIndex;
  logger.debug("step selected at "+index+", " 
      + (this.ignoreSelectEvents ? "ignoring" : "handling"));
  if (this.ignoreSelectEvents) return;
  let item = this.tree.view.getItemAtIndex(index).itemModel;
  //selected again?
  if (item === this.selectedStep) return;
  this.selectedStep = item;
  this.selectedIndex = index;
  this.cb.onSelect(item, index);
};

var initIndentFrom = function (step, indent, i) {
  if (step.getIndentRange) {
    let ir = step.getIndentRange();
    if (ir[1] < 0) ir[1] = step.block.getItemsSize();
    let level = indent[i]+1;
    for (var j = ir[0]; j<ir[1]; j++) {
      indent[i+j] += level;
    }
  }
};

var restoreSelection = function (item, tree, treeIndex) {
  if (item.markItemForNextSelection) {
    if (treeIndex.firstIndexInSelection == -1) {
      treeIndex.firstIndexInSelection = treeIndex.value;
      treeIndex.firstItemInSelection = item;
    }
    item.markItemForNextSelection = false;
    treeIndex.numSelected++;
    tree.view.selection.toggleSelect(treeIndex.value);
  }
};
