EXPORTED_SYMBOLS = ["taskPane"];
const DEBUG = true;
const HTMLNS = "http://www.w3.org/1999/xhtml";
const NS = "http://www.mozilla.org/keymaster/gatekeeper/there.is.only.xul";
Components.utils.import("resource://indexdata/runtime/core.js");
Components.utils.import("resource://indexdata/ui/app.js");
Components.utils.import("resource://indexdata/util/logging.js");
Components.utils.import("resource://indexdata/ui/testEditor.js");
Components.utils.import("resource://indexdata/ui/StepTree.js");
Components.utils.import("resource://indexdata/ui/DataBrowser.js");
Components.utils.import("resource://indexdata/runtime/ConnectorTemplate.js");
Components.utils.import("resource://indexdata/runtime/Connector.js");
Components.utils.import("resource://indexdata/runtime/Task.js");
Components.utils.import("resource://indexdata/runtime/Tester.js");
Components.utils.import("resource://indexdata/util/mozHelper.js");
Components.utils.import("resource://indexdata/util/xmlHelper.js");
Components.utils.import("resource://indexdata/util/xulHelper.js");
Components.utils.import("resource://indexdata/util/textHelper.js");
Components.utils.import("resource://indexdata/util/io.js");
// abbreviations
var child = xmlHelper.appendNode;
var empty = xmlHelper.emptyChildren;
var prompts = Components.classes["@mozilla.org/embedcomp/prompt-service;1"]
.getService(Components.interfaces.nsIPromptService);
// redirect 'test' logging to runMeter, careful since at times run meter
// may not exist, also filter DEBUG messages
logging.getLogger('test_execution').addAppender({
  append: function (prefix, level, msg) {
    if (level === 'DEBUG') return;
    if (taskPane.runMeter)
      taskPane.updateMeterLog(level, prefix + level + msg);
  },
  toString: function() { return "RMAppender"; }
});
//retrieve the default logger
var logger = logging.getLogger();
var callbacks = {
  canSave: function (canSave) {
    if (canSave) {
      logger.debug("taskPane: enabling save button");
      taskPane.enableSave();
    } else {
      logger.debug("taskPane: disabling save button");
      taskPane.disableSave();
    }
  },
  lastOpened: function (lastOpened) {
    logger.debug("taskPane: lastOpened file name changed");
    // only applies if task pane was open, otherwise wait until onTaskPaneLoad
    if (app.isSidebarOn) taskPane.setSidebarTitle();
  },
  testNameChanged: function () {
    logger.debug("Test name changed notified");
    // only applies if task pane was open, otherwise wait until onTaskPaneLoad
    if (app.isSidebarOn) taskPane.refreshTaskTests();
  },
  fullqueryEdited: function (editedValue) {
    logger.debug("Fullquery has been edited through the editor");
    testEditor.getActiveTest().setArg('fullquery', editedValue);
    taskPane.refreshArguments();
  },
  // this is THE ONLY callback that calls the method to refresh the pane
  // TODO this needs to have more granularity -
  // seperate events for task arg updates, step list and so on
  // for now just refresh the whole pane aytime anything changes
  selectedTask: function (selectedTask) {
    //what follows assumes the sidebar (taskPane) is loaaded!
    //DO NOT force 'toggleSidebar' in here, it's async and won't help
    //instead we might consider moving all "refresh" to a sidebar calback
    logger.debug("taskPane: selectedTask has changed, refreshing contents..");
    if (app.selectedTask) {
      //if no tests, disable
      if (app.selectedTask.getTests().length != 0) {
        taskPane.switchActiveTest(app.selectedTask.getTests()[0]);
        taskPane.setDisabledTestControls(false);
      } else {
        taskPane.setDisabledTestControls(true);
      }
      app.selectedTask.addHandler("onTaskFinish",
        function (outcome, results, errpos) {
          if (!outcome) taskPane.highlightError(errpos);
          taskPane.dataBrowser.refresh();
          taskPane.refreshArguments();
        }
      );
    }
    if (app.isSidebarOn) {
      taskPane.stepTree.setTask(app.selectedTask);
      taskPane.dataBrowser.setTask(app.selectedTask);
      taskPane.refreshTaskPane();
    }
  },
  connector: function (connector) {
    logger.debug("taskPane: connector has changed, force refresh...");
    connector
      .setTaskArgumentsUpdateCb(taskPane.onTaskArgumentsUpdate);
    connector
      .setStepParametersUpdateCb(taskPane.onStepParametersUpdate);
    //new connector, old selected task is invalid!{
    var tasks = app.connector.findTasks(
      app.connector.template.properties["defaultTaskSelection"]);
    if (!tasks && app.isSidebarOn) {
      app.popupError("Default task not found, you may have the wrong template. Current template is: " + app.template.name);
      logger.debug('Cannot find default task to select, try refresh with null');
      app.set("selectedTask", null);
    } else {
      logger.debug('Selected task changed to ' + tasks[0].name);
      app.set("selectedTask", tasks[0]);
    }
  },
  template: function () {
    logger.debug("changed template: modifying sidebar title");
    taskPane.setSidebarTitle();
  }
};
// data structure for the builder interface.
var taskPane = {
  // wait for the main window
  tpDoc: null,
  stepTree: null,
  stepBox: null,
  stepSplit: null,
  taskMenu: null,
  stepDescs: {},
  mainWindow: null,
  runMeter: null,
  selectedStep: null,
  stepClip: [],
  subscribe: function () {
    for (let cb in callbacks) app.subscribe(cb, callbacks[cb]);
  },
  unsubscribe: function () {
    for (let cb in callbacks) app.unsubscribe(cb, callbacks[cb]);
  },
  setDisabledTestControls: function (disable) {
    taskPane.tpDoc.getElementById('cfTaskTests').disabled = disable;
    //taskPane.tpDoc.getElementById('cfTestAdd').disabled = disable;
    taskPane.tpDoc.getElementById('cfTestRemove').disabled = disable;
    taskPane.tpDoc.getElementById('cfTestEdit').disabled = disable;
  },
  onTaskArgumentsUpdate: function() {
    taskPane.refreshTaskMenu();
    taskPane.refreshArguments();
  },
  onStepParametersUpdate: function(step) {
    logger.debug("step parameters updated for " + step );
    app.set("canSave", true);
    taskPane.stepTree.refresh();
  },
  onNewTaskLoad: function (e) {
    if (!app.connector.template) throw new Error("no template loaded");
    var doc = e.target;
    var list = doc.getElementById("cfNewTaskList");
    var taskTemplates = app.connector.template.getTaskTemplates();
    for (var i=0; i<taskTemplates.length; i++) {
      var tt = taskTemplates[i];
      var item = xmlHelper.appendNode(list, "listitem", null, {"label":tt.name, "value":tt.name})
      // xbl:inherits does this for us, ff 3.6 gets pushy about it
      // xmlHelper.appendNode(item, "listcell", null, {"label":tt.name})
      xmlHelper.appendNode(item, "listcell", null,
          {"label":tt.getArgNames().join(", ")})
    }
  },
  addNewTask: function (taskName) {
    var task = app.connector.createTask(taskName)
    task.createDefaultTests();
    app.set("selectedTask", task);
    app.set("canSave", true);
  },
  onNewTaskUnload: function(e) {
    e.target.defaultView.removeEventListener('load',
        taskPane.onNewTaskLoad, false);
    e.target.defaultView.removeEventListener('unload',
        taskPane.onNewTaskUnload, false);
  },
  onTaskPaneLoad: function (e) {
    logger.debug("onTaskPaneLoad: window ready, preparing task pane...");
    taskPane.subscribe();
    //not be fully safe (see next comment), should we check the originalTarget?
    app.set("isSidebarOn", true);
    //IMPORTANT! ------------------------
    //child windows (step browser, meta editor etc) opened via window.open
    //WITHOUT the 'chrome' feature, TRIGGER parent's 'onload' event and
    //subsequently cause this method to be re-executed, however the new window
    //reference seem not to be useable (drawable) eventthough it contains all
    //components, needs more investigation

    // get xul doc/win of the view
    taskPane.tpDoc = e.target;
    app.set("mainWindow", e.currentTarget.top);
    taskPane.mainWindow = e.currentTarget.top;
    
    // and some commonly referenced elements
    taskPane.stepBox = taskPane.mainWindow.document.getElementById("cfStepBox");
    taskPane.stepSplit = taskPane.mainWindow.document.getElementById("cfContentSplitter");
    taskPane.taskMenu = taskPane.tpDoc.getElementById("cfTaskMenu");

    // set up the step tree
    let stepTreeElement = taskPane.tpDoc.getElementById("cfStepTree")
    taskPane.stepTree = new StepTree();
    taskPane.stepTree.init(stepTreeElement, {
      onChange: function () { app.set("canSave", true); }, 
      onSelect: function (item, index) { 
        empty(taskPane.stepBox);
        if (item.getClassName() != "Block") {
          item.draw(taskPane.stepBox, taskPane.mainWindow);
          taskPane.stepSplit.setAttribute("collapsed", "false");
          taskPane.stepBox.setAttribute("collapsed", "false");
        } else {
          taskPane.stepSplit.setAttribute("collapsed", "true");
          taskPane.stepBox.setAttribute("collapsed", "true");
        }
        taskPane.selectedStep = item;
        app.selectedTask.selectedStepIndex = index;
        app.notify("selectedStep");
      },
      onDeselect: function () {
        // No steps at the moment, hide step box.
        taskPane.stepBox.setAttribute("collapsed", "true");
        taskPane.stepSplit.setAttribute("collapsed", "true");
        return;
      }
    });
    // set up the data browser
    let dbel = taskPane.tpDoc.getElementById("cfDataBrowser")
    let dbcontext = taskPane.tpDoc.getElementById("dataBrowserContextMenu");
    taskPane.dataBrowser = new DataBrowser();
    taskPane.dataBrowser.init(dbel, dbcontext);

    //check licence and disallow the task pane if not accepted.
    //if so disable the builder
    var isLicenceAccepted = function () {
      try {
        if (core.idprefs.getIntPref("licenceaccepted") > 0)
          return true;
        else
          return false;
      } catch (err) {
        return false;
      }
    }
    if (!isLicenceAccepted()) {
      taskPane.mainWindow.openDialog('chrome://cfbuilder/content/licence.xul',
          null, 'modal').focus();
      if (!isLicenceAccepted()) {
        taskPane.mainWindow.toggleSidebar('viewCfBuilderSidebar');
        app.disableApp();
        taskPane.popupError("CF Builder will be disabled when Firefox is "
            +"restarted. You can enable it in the Tools > Add-ons menu.");
        return;
      }
    }

    taskPane.stepBox.addEventListener("input", function(e) {
      app.connector.onStepParametersUpdate(taskPane.selectedStep);
    }, false);
    taskPane.stepBox.addEventListener("change", function(e) {
      app.connector.onStepParametersUpdate(taskPane.selectedStep);
    }, false);
    taskPane.stepBox.addEventListener("command", function(e) {
      app.connector.onStepParametersUpdate(taskPane.selectedStep);
    }, false);

    //THIS IS SOO HACKY
    //otherwsie the task pane won't be redrawn if it was re-opened
    if (app.connector) app.notify("connector");

    // The sidebar title needs setting, because it is cached from
    // the previous run and displays the name of a no-longer-loaded
    // connector if we leave it.  And if the task pane is closed and
    // re-opened, the title gets re-set to the value specified back in
    // the overlay.
    taskPane.setSidebarTitle();
  },
  onTaskPaneUnload: function (e) {
    logger.debug("onTaskPaneUnload: window destroyed");
    taskPane.unsubscribe();
    taskPane.mainWindow = null;
    app.set("isSidebarOn", false);
    // This will be selected via Task.selectedStepIndex
    taskPane.selectedStep = null;
    //this method is called (also) with each sidebar on/off toggle
    if (taskPane.stepBox) {
      taskPane.stepBox.setAttribute("collapsed", "true");
      taskPane.stepSplit.setAttribute("collapsed", "true");
    }
    //we could null the tpDoc at this point, but why bother?
    //also all unregistered listeners will get register back again in the
    //chrome/xul when sidebar is opened again
  },
  onTaskSelect: function (t) {
    app.selectedTask = app.connector.getTaskById(t);
    if (typeof (app.selectedTask.selectedStepIndex) === 'undefined') 
      app.selectedTask.selectedStepIndex = 0;
    taskPane.selectedStep = null;
    app.notify("selectedTask");
  },
  // populate args from test, check task findability, set up data refresh
  prepTask: function(task) {
    let test = testEditor.getActiveTest();
    let testArgs = Object.keys(test.getArgs());
    task.data.input = test.generateTaskInput();
    if (taskPane.runMeter) {
      let found = task.connector.findTask(task.getName(), testArgs);
      if (task !==  found) {
        let msg = 'task matched request for "' + task.getName() +
          '" with arguments: ' + testArgs.join(', ');
        if (found) {
          taskPane.updateMeterLog("WARN", "A different " + msg);
        } else {
          taskPane.updateMeterLog("WARN", "No " + msg);
        }
      }
    }
  },
  onRunAll: function() {
    let task = app.selectedTask;
    taskPane.bindToRunMeter();
    //FIXME this does not handle blocks
    for (let i=0; i < task.block.getItemsSize(); i++) {
      if (typeof task.block.getItemAt(i).processConf === "function")
        task.block.getItemAt(i).processConf();
    }
    task.clearData();
    taskPane.prepTask(task);
    task.run();
  },
  highlightError: function (pos) {
    //TODO not implemented
  },
  bindToRunMeter: function () {
    taskPane.stepSplit.setAttribute("collapsed", "false");
    taskPane.stepBox.setAttribute("collapsed", "false");
    taskPane.runMeter = taskPane.drawRunMeter(taskPane.stepBox);
    taskPane.selectedStep = null;
    app.selectedTask.addHandler("onStepFinish", taskPane.updateMeterProgress);
    app.selectedTask.addHandler("onMessageLog", taskPane.updateMeterLog);
    app.selectedTask.addHandler("onTaskFinish", taskPane.updateMeterOutcome);
  },
  drawRunMeter: function (surface) {
    xmlHelper.emptyChildren(surface);
    var vbox = xmlHelper.appendNode(surface, "vbox",null,
                     { "flex":1,"align":"stretch"} );
    xmlHelper.appendNode(vbox, "progressmeter", null,
        {"id" : "cfRunProgress", "value": "0", "mode": "determined"});
    var lBox = xmlHelper.appendNode(vbox, "listbox", null,
        {"id" : "cfLogConsole", "flex":1,"align":"stretch"} );
    var lCols = xmlHelper.appendNode(lBox, "listcols", null,
                        { "flex":1,"align":"stretch"} );
    xmlHelper.appendNode(lCols, "listcol", null, {"flex":"1"});
    var lHead = xmlHelper.appendNode(lBox, "listhead");
    xmlHelper.appendNode(lHead, "listheader", null,
        {"label":"Log message"});
    return vbox.ownerDocument;
  },
  onRunSelected: function () {
    let items = taskPane.stepTree.getValidSelection()[1];
    let from = items.length > 0 ? items[0].block.findItemIndex(items[0]) : 0;
    for (let i=0; i<items.length; i++) {
      let nextItem = i+1 < items.length ? items[i+1] : null;
      let item = items[i];
      let block = item.block;
      let index = block.findItemIndex(item);
      //what is this process conf?
      if (typeof item.processConf === "function") item.processConf();
      if (nextItem == null //eos
        || nextItem.block != item.block //we got to the new block
        || index - to > 1) { //or next selection
        let step = block.getItemAt(from);
        let switched = false;
        if (step.isAlt === true) { step.isAlt = false; switched = true };
        logger.info("About to run steps from "+from+" to "+index);
        taskPane.prepTask(app.selectedTask);
        block.run(null, from, index+1);
        if (switched) step.isAlt = true;
        //move cursor
        from = index;
      }
    }
  },
  popupError: function (errorMsg) {
    taskPane.mainWindow.alert(errorMsg);
  },
  removeCurrentTask: function () {
    if (app.selectedTask) {
      if (prompts.confirm(null, "Please confirm",
          "Are you sure you want to delete the current task?")) {
        app.selectedTask.detach();
        app.set("canSave", true);
        app.set("selectedTask", app.connector.taskList[0]||null);
      }
    } else {
      prompts.alert(null, "Warning", "No selected task to delete.");
    }
  },
  openTaskWindow: function (mainWin) {
    mozHelper.openWindow(taskPane.onTaskPaneLoad,
        "chrome://cfbuilder/content/task_window.xul");
  },
  openStepWindow: function (mainWin) {
    mozHelper.openWindow(function (e) {
        taskPane.stepBox = e.target.getElementById("cfStepBox");
        } , "chrome://cfbuilder/content/step_window.xul");
  },
  openRunMeter: function (mainWin) {
    mozHelper.openWindow(function (e) {
        taskPane.runMeter = e.target;
        if (app.selectedTask) {
          app.selectedTask.addHandler("onStepFinish",
            taskPane.updateMeterProgress);
          app.selectedTask.addHandler("onMessageLog",
            taskPane.updateMeterLog);
        }
      },
      "chrome://cfbuilder/content/run_meter.xul");
  },
  updateMeterProgress: function (stepIdx, total) {
    var pmBar = taskPane.runMeter.getElementById("cfRunProgress");
    if (pmBar == null) return;
    pmBar.value = (stepIdx+1)/total*100;
  },
  updateMeterLog: function (level, logMessage) {
    var lbConsole = taskPane.runMeter.getElementById("cfLogConsole");
    if (lbConsole == null) return;
    var logItem = xmlHelper.appendNode(lbConsole, "listitem", null,
      {"label":logMessage} );
    switch (level) {
      case "WARN" : logItem.setAttribute("style", "color: brown;"); break;
      case "ERROR" : logItem.setAttribute("style", "color: red;"); break;
    }
  },
  updateMeterOutcome: function (outcome, result) {
    //success
    if (outcome) {
      var rm = taskPane.runMeter;
      if (rm == null) return;
      var pmBar = rm.getElementById("cfRunProgress");
      if (pmBar == null) return;
      pmBar.value = 100;
    //failure
    } else {
      taskPane.popupError(result);
    }
  },
  resetMeter: function () {
    var lbConsole = taskPane.runMeter.getElementById("cfLogConsole");
    xmlHelper.emptyChildren(lbConsole, "listitem");
    var pmBar = taskPane.runMeter.getElementById("cfRunProgress");
    pmBar.value = 0;
  },
  setSidebarTitle: function () {
    var string = "";
    if (app.lastOpened && app.lastOpened.path) {
      string += app.lastOpened.path.replace(/.*[\/\\]/, "")
        + " (local)";
    }
    if (app.connector && app.connector.metaData && app.connector.metaData.origin) {
      if (string) string += ", ";
      string += app.connector.metaData.origin.replace(/.*[\/\\]/, "")
        + " (repo)";
    }
    if (!string) string = "Unnamed Connector";
    if (app.connector && app.connector.template && app.connector.template.name) {
      string += " [" + app.connector.template.name + "]";
    }
    var node = taskPane.mainWindow.document.getElementById("sidebar-title");
    node.value = string;
    node.setAttribute("style", "font-weight: bold");
    node.setAttribute("tooltiptext", string);
  },
  
  // Tests
  runCurrentTest: function () {
    if (testEditor.getActiveTest() == null)
      logger.debug("Current test is null");
    else {
      taskPane.bindToRunMeter();
      testEditor.getActiveTest().run();
    }
  },
  onTestSelect: function (idx) {
    if (app.selectedTask == null) return;
    logger.debug("Switching test to " + idx );
    idx = parseInt(idx);
    taskPane.switchActiveTest(app.selectedTask.getTests()[idx]);
    //FIXME this should be done using publish/subscribe model (via app)
    taskPane.refreshArguments();
  },
  onTestRemove: function (idx) {
    logger.debug("Removing test " + idx);
    idx = parseInt(idx);
    app.selectedTask.removeTestByIdx(idx);
    if (app.selectedTask.getTests().length > 0) {
      var selidx = idx > 0 ? idx - 1 : 0;
      taskPane.switchActiveTest(app.selectedTask.getTests()[selidx]);
    } else {
      taskPane.setDisabledTestControls(true);
    }
    //FIXME this should be done using publish/subscribe model (via app)
    taskPane.refreshTaskTests();
    taskPane.refreshArguments();
  },
  addTest: function (testName) {
    var test = app.selectedTask.createTest(testName);
    taskPane.switchActiveTest(test);
    //FIXME this should be done using publish/subscribe model (via app)
    taskPane.refreshTaskTests();
    taskPane.setDisabledTestControls(false);
  },
  switchActiveTest: function (test) {
    logger.debug("Switched to " + test.getName());
    testEditor.setActiveTest(test);
  },
  runTester: function() {
    var input = {value:
      app.connector.template.properties["defaultTaskTestsOrder"]};
    var check = {value: false}; // default the checkbox to false
    var result = prompts.prompt(null, "Run tasks' tests in sequence",
        "Specify task tests execution order using  "
      + "a comma-seperated list of task names, "
      + "trailing '?' indicates an optional task that may be missing. "
      + "Multiple executions may be performed from one order string, "
      + "and for each execution, tests are selected based on their names.",
        input, null, check);
    if (!result) return;
    var suite =  input.value;
    var tester = new Tester();
    tester.onTestSuccess = function () {
      taskPane.popupError("Test suite ("+suite+") successfull.");
    };
    tester.onTestFailure = function () {
      taskPane.popupError("Test suite ("+suite
          +") failed!, check STDOUT for details.");
    };
    if (!tester.runInOrder(app.connector, suite)) {
      taskPane.popupError("Cannot execute test suite ("+suite
          +") - missing required tasks!");
    }
  },

  /*
   * All refresh* methods are supposed to be simple, redrawing methods called
   * at the very last rendering phase after all business logic has been
   * performed.
   * Especially, avoid any explicit method calls that may cause event cascades,
   * e.g DO NO call any UI callbacks explicitly (like onTaskSelect, etc).
   * Those methods are full of unnessary asserts (connector != null,
   * selectedTask != null) handled higher in the chain.
   */
  refreshTaskPane: function () {
    if (app.connector == null) {
      logger.debug("refreshTaskPane: FATAL, connector is null");
      return;
    }
    taskPane.refreshTaskTests();
    taskPane.refreshTaskMenu();
    taskPane.refreshArguments();
    taskPane.dataBrowser.refresh();
    taskPane.stepTree.refresh();
  },
  refreshTaskMenu: function () {
    var menu = taskPane.taskMenu;
    var popup = menu.getElementsByTagName("menupopup")[0];
    xmlHelper.emptyChildren(popup);
    var theSelectedItem = null;
    var defaultItem = null;
    var tasks = app.connector.getTasks();
    if (!tasks) {
      logger.debug("Connector contains no tasks, menu will not be refreshed");
      return;
    }
    for (var i=0; i<tasks.length; i++) {
      var argstring = ''
      if (tasks[i].getUsedArgs().length > 0)
        argstring = ' (' + tasks[i].getUsedArgs().join(', ') + ')'
      var item = xmlHelper.appendNode(popup, "menuitem", null,
          {"label": tasks[i].name + argstring, "value": tasks[i].id});
      if (tasks[i].name
        == app.connector.template.properties["defaultTaskSelection"])
        defaultItem = item;
      if (app.selectedTask && (tasks[i].id == app.selectedTask.id))
        theSelectedItem = item;
    }
    if (theSelectedItem) {
      menu.selectedItem = theSelectedItem;
    } else if (defaultItem) {
      menu.selectedItem = defaultItem;
    } else if (menu.itemCount > 0) {
      menu.selectedIndex = 0;
    } else {
      menu.selectedIndex = -1;
    }
  },
  refreshArguments: function () {
    var list = taskPane.tpDoc.getElementById("cfArgumentList");
    let test = testEditor.getActiveTest();
    xulHelper.clearList(list);
    if (app.selectedTask == null) {
      logger.debug("selectedTask is null, ignore refreshing args");
    } else {
      var argNames = app.selectedTask.getArgNames();
      for (var i=0; i<argNames.length; i++) {
        var arg = argNames[i];
        let value = test.getArg(arg) || "";
        var item = xmlHelper.appendNode(list, "listitem", null,
          {"value": arg, allowevents: true}, null, false);
        var label = xmlHelper.appendNode(item, "listcell", null,
            {"label": arg, "context":"argListContextMenu"});
        label.setAttribute("class",
                           (app.selectedTask.isArgUsed(arg) ? "" : "un") +
                           "usedTaskArgumentName");
        var cell = xmlHelper.appendNode(item, "listcell");
        var box = xmlHelper.appendNode(cell, "textbox", null,
          { "flex":1, "value": value,
            "class": "tight", "id": (arg+"Textbox") });
        box.addEventListener("input", function(e) {
          let pp = this.parentNode.parentNode;
          if (this.value) {
            testEditor.getActiveTest().setArg(pp.value, this.value);
          } else {
            // drop empty strings
            testEditor.getActiveTest().delArg(pp.value);
          }
          app.set("canSave", true);
         }, false);
        //put magic cql button
        if (arg == 'fullquery') {
          var cqledit = xmlHelper.appendNode(box, "image", null,
              {src: 'chrome://cfbuilder/content/icons/cql.png'});
          cqledit.addEventListener("click", function (e) {
              app.mainWindow.open('chrome://cfbuilder/content/cql_editor.xul', 
                'cqlEditor'
                , 'resizable=yes,chrome=yes,titlebar=yes,centerscreen,dependent').focus();
              });
        }
        list.appendChild(item);
      }
    }
  },
  refreshTaskTests: function () {
    var menuList = taskPane.tpDoc.getElementById("cfTaskTests");
    var popup = menuList.getElementsByTagName("menupopup")[0];
    xmlHelper.emptyChildren(popup);
    menuList.selectedIndex = -1;
    if (app.selectedTask == null) return;
    var tests = app.selectedTask.getTests();
    for (var i=0; i<tests.length; i++) {
        xmlHelper.appendNode(popup, "menuitem", null,
            {"label": tests[i].getName(), "value": i});
        if (tests[i] == testEditor.getActiveTest()) {
          menuList.selectedIndex = i;
        }
      }
  },
  // To be called every time anything is changed
  enableSave: function () {
    taskPane.tpDoc.getElementById("cfSaveButton").disabled = false;
  },
  disableSave: function () {
    taskPane.tpDoc.getElementById("cfSaveButton").disabled = true;
  }
};
