var EXPORTED_SYMBOLS = ["xmlHelper"];
var Cc = Components.classes
var Ci = Components.interfaces
Components.utils.import('resource://indexdata/util/logging.js');

var logger = logging.getLogger('xmlHelper');

var xmlHelper = {

    // Fetch the xml doc
    // url can be a http:// url, or a name of a local file
    // The isLocalFile flag causes 'file:///' to be prepended to the name
    // when opening it.
    // The doXIncludes flag causes XIncludes to be expanded
    // The httpMethod is optional, defaults to GET
    // the postContent is only useful if the httpMethod is POST
    // it can be a dom tree, or a string.
    // TODO Pass an optional extra-header array for passing extra
    // headers to the request
    // TODO Now POST always implies content type application/soap+xml.
    // It might be better to allow a SOAP type, and leave others free
    // to pass what ever content types they way
    // extraHeaders is a hash of additional headers to put in the request.
    // It may contain a content-type.
    fetchDoc: function (url,
                        isLocalFile, doXIncludes,
                        httpMethod, postContent, extraHeaders ) {
      if ( !httpMethod )
        httpMethod = "GET";
      if ( !postContent )
        postContent = null;
      if ( !extraHeaders )
        extraHeaders = {};
      var req = Cc["@mozilla.org/xmlextras/xmlhttprequest;1"]
         .createInstance(Ci.nsIXMLHttpRequest);
      var urlpath = url;
      if (isLocalFile)
          urlpath = "file:///" + url ;
      var msgurl = url.replace( /[\?\&].*$/,""); // remove params.
        // Makes shorter log messages, and does not expose passwords
        // as these can propagate all the way to real users.
      req.open(httpMethod, urlpath, false);  // false = synchronous
      req.overrideMimeType('text/xml');
      for ( var h in extraHeaders ) {
        logger.debug("Setting request header '" + h + "' to '"
            + extraHeaders[h] + "'");
        req.setRequestHeader( h, extraHeaders[h] );
      }
      if (httpMethod == "POST" && ! extraHeaders['Content-Type'] ) {
        req.setRequestHeader("Content-Type",
                             "application/soap+xml; charset=utf-8");
        logger.debug("Defaulting 'Content-Type' to '"+
                             "application/soap+xml; charset=utf-8'");
      }
      try {
        logger.debug("just about to " + httpMethod );
        req.send(postContent);
      } catch (e) {
        logger.info(e.message + "\n");
        throw new Error("Error fetching XML from " + msgurl );
      }
      logger.debug("status: " + req.status );
      /*
      for ( var k in req ) {
        var v = req[k];
        if (typeof(v) == "function" ) v = "function";
        logger.info("req." + k + ": " + v );
      }
      */
      if ( req.status != 0 && req.status != 200 ) {
        logger.debug("Failure status " + req.status + ": ");
        logger.debug("Headers: " + req.getAllResponseHeaders() );
        if ( req.responseText )
          logger.debug("Response: " + req.responseText );
        var err = new Error ("Error " + req.status + " fetching " + msgurl );
        // Very nonstandard way of passing additional error info!
        if ( req.responseText )
          err.responseText = req.responseText;
        if ( req.responseXML ) {
          err.responseXML = req.responseXML;
        }
        throw err;
      }
      logger.info("Headers: " + req.getAllResponseHeaders() );
      var doc = req.responseXML;
      if (doc.documentElement.nodeName == "parsererror") {
          logger.warn("Malformed XML from " + msgurl );
          logger.debug(doc.documentElement.textContent );
          xmlHelper.dumpxml(doc);
          throw new Error("malformed xml");
      }
      if (doXIncludes) {
        var lastIndex = url.lastIndexOf('/');
        if (lastIndex < 0) lastIndex = url.lastIndexOf('\\');
        var baseDir = lastIndex < 0 ? "" : url.substr(0, lastIndex);
        logger.log(Level.DEBUG, "found baseDir '" + baseDir + "' " +
            "for path '" + urlpath + "'");
        doc = xmlHelper.resolveXIncludes(doc, "file:" + baseDir);
      }
      return doc;
    },

    // open and parse xml doc given a path (local file). resolve XIncludes.
    openDoc: function (path) {
        return this.fetchDoc( path, true, true );
    },


    openDoc2: function (path) {
      var file = Cc["@mozilla.org/file/local;1"].
                   createInstance(Ci.nsILocalFile);
      file.initWithPath(path);
      var fstream = Cc["@mozilla.org/network/file-input-stream;1"].
        createInstance(Ci.nsIFileInputStream);
      var cstream = Cc["@mozilla.org/intl/converter-input-stream;1"].
        createInstance(Ci.nsIConverterInputStream);
      fstream.init(file, 0x01, 0444, 0);
      cstream.init(fstream, "UTF-8", 0, 0);
      var str = {};
      var data = "";
      while (cstream.readString(0xffffffff, str)) {
        data += str.value;
      }
      cstream.close();
      var parser = Cc["@mozilla.org/xmlextras/domparser;1"]
                     .createInstance(Ci.nsIDOMParser);
      var doc = parser.parseFromString(data, "text/xml");
      if (doc.documentElement.nodeName == "parsererror") {
        throw new Error(doc.documentElement.firstChild.nodeValue
            || "malformed XML");
      }
      return doc;
    },

    // save xml doc given a path or a file handle
    saveDoc: function (file, doc) {
      // file is a path, open handle
      if (typeof file == "string") {
        var path = file;
        file = Cc["@mozilla.org/file/local;1"].createInstance(Ci.nsILocalFile)
        file.initWithPath(path);
        if (file.exists() === false) {
          try {
            file.create(Ci.nsIFile.NORMAL_FILE_TYPE, 420);
          } catch (e) {
            dump(e.message + "\n");
            throw new Error("cannot create file (possibly access denied)");
          }
        }
      }

      //create output stream on the file handle
      var fos = Cc["@mozilla.org/network/file-output-stream;1"]
        .createInstance(Ci.nsIFileOutputStream);
      try {
        fos.init(file, 0x04 | 0x08 | 0x20, 420, 0);
      } catch (e) {
        dump(e.message + "\n");
        throw new Error("write error (possibly access denied)");
      }

      //create serializer
      var xs = Cc["@mozilla.org/xmlextras/xmlserializer;1"]
        .createInstance(Ci.nsIDOMSerializer);
      xs.serializeToStream(doc, fos, "UTF-8");
      fos.close();
    },

    // create new xml document given the root node
    // return the reference to the root node
    doc: function (root) {
      if (typeof root != "string") throw new Error("Must specify a root node.");
      var parser = Cc["@mozilla.org/xmlextras/domparser;1"]
        .createInstance(Ci.nsIDOMParser);
      var doc = parser.parseFromString("<" + root + "/>", "text/xml");
      return doc.documentElement;
    },

    // Create a whole new xml document from a string
    // contentType defaults to "text/xml" but can also be "text/html", or 
    // anything else Mozillas parser handles.
    docFromString: function (str, contentType) {
      if (!contentType)
        contentType = "text/xml";
      var parser = Cc["@mozilla.org/xmlextras/domparser;1"]
        .createInstance(Ci.nsIDOMParser);
      var doc = parser.parseFromString(str, contentType );
      return doc.documentElement;
    },

    // append new node to a parent node,
    // join controls if the node should be
    // appended or only created
    // ALL arguments after tagName are optional.
    appendNode: function (parent, tagName, textContent, attrs, ns, join) {
        if (!parent)
            throw new Error("Cannot append node, parent is null "+
                "(when appending '"+tagName+"')");
        var doc = parent.ownerDocument;
        var node;
        if (ns != undefined)
          node = doc.createElementNS(ns, tagName);
        else
          try {
            node = doc.createElement(tagName);
          } catch (e) {
            dump("Creating node '"+tagName+"' failed.\n");
            throw e;
          }
        if (join != undefined && join === false) {
        } else {
          parent.appendChild(node);
        }
        if (textContent != undefined && textContent != null) {
            var textNode = doc.createTextNode(textContent);
            node.appendChild(textNode);
        }
        if (attrs != undefined && typeof attrs == 'object') {
            for (var attName in attrs) {
                node.setAttribute(attName, attrs[attName]);
            }
        }
        return node;
    },
    
    removeNode: function (node) {
        if (node.parentNode != null)
          node.parentNode.removeChild(node);
    },

    emptyChildren: function (elem, nodeName) {
        if (nodeName === undefined) {
         while (elem.firstChild) elem.removeChild(elem.firstChild);
        } else {
          var items = elem.getElementsByTagName(nodeName);
          while(items.length > 0)
            elem.removeChild(items[items.length - 1]);
        }
    },

    // Checks if a node is somehow descendening from the (grand)parent
    isAncestor: function (node, parent) {
        while(node) {
            if ( node === parent )
                return true;
            node = node.parentNode;
        }
        return false;
    },

    getElementXpath: function (target, truncate, nsList) {
      var doc = target.ownerDocument;
      /*
      dump("XML: getElementXpath: doc=" + doc + " targ=" + target +
           " n=" + target.nodeName +
           " l=" + target.localName +
           " type=" + target.nodeType +
           " t.prefix=" +  target.prefix +
           " t.ns=" + target.namespaceURI +
           " nsL=" + nsList + "\n");
      */
      var paths = [];
      var siblingIndices = [];
      var idString = "";
      var foundPath = false;
      var nsResolver;
      if ( nsList ) {
        nsResolver = xmlHelper.makeNsResolverFromList(nsList);
      } else {
        nsResolver = doc.createNSResolver( doc );
      }

      var buildPath = function(id, components) {
        var path = id + (components.length ? "/" + components.join("/") : "");
        return path;
      }
      // tests that we get the right node and it's the only node we get
      var testPath = function(path) {
        /*
        var count = xmlHelper.getMatchCountByXpath(doc, path, nsResolver);
        dump("getElementXpath:testPath: '" + path + " " +
          "count=" + count + " " +
          "targ=" + target + "\n");
        if ( count == 1 ) {
          var found = xmlHelper.getElementByXpath(doc, path, nsResolver);
          dump("getElementXpath:testPath: Found " + path + " " +
               "t=" + target.nodeName +
               " f=" + found.nodeName + " r=" + ( target == found ) + "\n");
        }
        */
        return (xmlHelper.getMatchCountByXpath(doc, path, nsResolver)==1)
              && (xmlHelper.getElementByXpath(doc, path, nsResolver)==target);
      }

      // build array of paths and parallel array of sibling indices
      // iterating through nodes from leaf to root or node with id
      var node = target;
      for (; node && node.nodeType == 1; node = node.parentNode) {
        /*
        dump("XML: getElementXpath: loop: node=" + node + "=" + node.nodeName +
            " type=" + node.nodeType + " id=" + node.id +
            " pref=" + node.prefix + "  nsuri=" + node.namespaceURI +
            "\n"); */
        // non-unique ids are a surprisingly common occurence
        if (node.id &&
           (xmlHelper.getMatchCountByXpath(doc,
                 'id("' + node.id + '")', nsResolver)==1)) {
          // we've found a unique parent and can stop
          idString = 'id("' + node.id + '")';
          break;
        } else if (node.name && node.type &&
                   node.type !== "checkbox" && node.type !== "radio") {
          // assume name attribute is unique among siblings
          // and that we care the name is what it was when we stored it
          paths.splice(0, 0, node.nodeName +
              '[@name="' + node.name + '"]');
          siblingIndices.splice(0, 0, null);
             /* was localName.toLowerCase() */
        } else if (node.className) {
          // this counting thing is the worst part, lets at least give it
          // some class
          // className is just the contents of the class attribute, it was
          // just renamed. Nothing special is done here so "foo bar" is
          // considered separate from "bar foo" despite those being
          // technically identical.

          // div[contains(concat(' ',normalize-space(@class),' '),' foo ')]
          // is too crazy and i don't think buys enough less brittleness
          // to justify the likely cost in the engine, but thanks anyhow to:
          // http://pivotallabs.com/users/alex/blog/articles/427-xpath-css-class-matching
          var index = 0;
          for (var sibling = node.previousSibling; sibling;
              sibling = sibling.previousSibling) {
            if (sibling.nodeName == node.nodeName &&  // was localName
                sibling.className == node.className) ++index;
          }
          var classSpec = '[@class="' + node.className + '"]';
          //paths.splice(0, 0, node.localName.toLowerCase() + classSpec);
          paths.splice(0, 0, node.nodeName + classSpec);
          siblingIndices.splice(0, 0, index);
        } else {
          var index = 0;
          for (var sibling = node.previousSibling; sibling;
              sibling = sibling.previousSibling) {
            //if (sibling.localName == node.localName) ++index;
            if (sibling.nodeName == node.nodeName) ++index;
          }
          var nn = node.nodeName;
          // special case, find a prefix from the nsList
          if ( node.namespaceURI && ! node.prefix && nsList ) {
            for ( var pref in nsList ) {
              if ( nsList[pref] == node.namespaceURI ) {
                nn = pref + ":" + nn;
                // dump("XML: getElementXpath: Found prefix '" + pref + "' " +
                // " for " + node.namespaceURI + "\n");
                break;
              }
            }
          }
          paths.splice(0, 0, nn );
          //dump("XML: getElementXpath: loop: Paths: " + paths + "\n");
          siblingIndices.splice(0, 0, index);
        }
      }

      // test if we need the indices at all or this path is unique
      if (testPath(buildPath(idString, paths)))
        foundPath = true;

      // originally was going to generate/test all combinations
      // in a nice way but got caught up in too much reading about
      // Gray codes and things and will just test combinations of one
      // or two offsets as that will usually do the trick and prevents
      // the off chance of things going out to lunch for a while in
      // the builder if someone selects a deeply nested node as the
      // number of combinations goes up fast.

      // first lets see if we can get away with only one of them
      // prefearably at the end so we can truncate more
      if (!foundPath) {
        for (var i=paths.length-1; i>=0; i--) {
          if (siblingIndices[i]||siblingIndices[i]===0) {
            pathsCopy = paths.slice(0);
            pathsCopy[i] = pathsCopy[i] + "[" + (siblingIndices[i]+1) + "]";
            var newXpath = buildPath(idString, pathsCopy);
            if (testPath(newXpath)) {
              paths = pathsCopy;
              foundPath = true;
              break;
            }
          }
        }
      }

      // if one doesn't work, two might
      if (!foundPath) {
        for (var i=paths.length-2; i>=0; i--) {
          for (var j=1; j<paths.length-i; j++) {
            if ((siblingIndices[i]||siblingIndices[i]===0) &&
                 (siblingIndices[j]||siblingIndices[j]===0)) {
              pathsCopy = paths.slice(0);
              pathsCopy[i] = pathsCopy[i] + "[" + (siblingIndices[i]+1) + "]";
              pathsCopy[j] = pathsCopy[j] + "[" + (siblingIndices[j]+1) + "]";
              var newXpath = buildPath(idString, pathsCopy);
              if (testPath(newXpath)) {
                paths = pathsCopy;
                foundPath = true;
                break;
              }
            }
          }
          if (foundPath)
            break;
        }
      }

      // fall back on adding indices depth first until it yields a unique node
      if (!foundPath) {
        for (var i=0; i<paths.length; i++) {
          pathsCopy = paths.slice(0);
          for (var j=paths.length-i; j<paths.length; j++) {
            if (siblingIndices[j] || siblingIndices[j] === 0)
              pathsCopy[j] = paths[j] + "[" + (siblingIndices[j]+1) + "]";
          }
          var newXpath = buildPath(idString, pathsCopy);
          if (testPath(newXpath)) {
            paths = pathsCopy;
            foundPath = true;
            break;
          }
        }
      }

      // some really weird invalid markup produces a DOM that will not yield a
      // valid XPath, such as <ALIGN="left">
      if (!foundPath) {
        //throw new Error("Cannot generate XPath for target node. " +
        //   "Possibly due to bad markup.");
        return false;
        // TODO - Should return false, but need to catch that
        // everywhere getElementXpath is called
      }

      if (paths.length > 4 && truncate) {
        var i = paths.length - 3;
        while (i > 0) {
          var thepath = idString + "//" + paths.slice(i).join("/");
          var r;
          try { // Defensive coding, evaluate may fail occasionally
            r = doc.evaluate(thepath, doc, nsResolver, 4, null);
            r.iterateNext();
            if (!r.iterateNext())
              break;
          } catch(e) {
            break;
          }
          i--;
        }
        //dump("XML: getElementXpath: clean: Returning " +
        //  "'" + thepath +"' i=" + i + "\n");
        return thepath;
      }
      var finalpath = buildPath(idString, paths);
      //dump("XML: getElementXpath: final: Returning " +
      //"'" + finalpath + "'\n");
      return finalpath;
    },

    getElementByXpath: function (elem, xpath, nsresolver) {
      if (!nsresolver)
        nsresolver = null;
      var doc = elem.ownerDocument || elem.documentElement.ownerDocument;
      var result = doc.evaluate(xpath, elem, nsresolver,
                     9,  /* FIRST_ORDERED_NODE_TYPE */
                     null);
      return result.singleNodeValue;
      // Throws errors in case of nonexisting namespace prefixes!
      // Check with getMatchCountByXpath first!
    },

    getMatchCountByXpath: function (elem, xpath, nsresolver) {
      if (!nsresolver)
          nsresolver = null;
      var doc = elem.documentElement.ownerDocument;
      try {
        return doc.evaluate(xpath, doc, nsresolver, 6, null).snapshotLength;
      } catch(err) {
        return 0;
      }
    },

    getElementsByClass: function (searchClass, node, tag) {
        var classElements = new Array();
        if (node == null)
            node = document;
        if (tag == null)
            tag = '*';
        var els = node.getElementsByTagName(tag);
        var elsLen = els.length;
        var pattern = new RegExp( "\\b" + searchClass + "\\b" );
        for (let i = 0, j = 0; i < elsLen; i++) {
            if ( pattern.test(els[i].className) ) {
                classElements[j] = els[i];
                j++;
            }
        }
        return classElements;
    },

    getPathsToFrame: function (node) {
      var paths = [];
      if ( node.ownerDocument.defaultView ) {
        // XML docs don't have defaultViews!
        var frame = node.ownerDocument.defaultView.frameElement;
        while (frame) {
          paths.push(xmlHelper.getElementXpath(frame, true));
          frame = frame.ownerDocument.defaultView.frameElement;
        }
      }
      return paths;
    },

    getDocFromFramePaths: function (elem, frames) {
      var doc = elem.documentElement.ownerDocument;
      for (var i = frames.length - 1; i >= 0; i--) {
        var frameDoc = xmlHelper.getElementByXpath(doc, frames[i]);
        doc = frameDoc.contentDocument;
      }
      return doc;
    },

    checkNodeSpec: function (nodeSpec) {
      if (typeof nodeSpec != "object" || !nodeSpec['xpath'])
        throw new Error("Invalid nodeSpec: " + nodeSpec + "\n");
    },

    getNodeSpec: function (node) {
      var nodeSpec = {};
      nodeSpec['frames'] = xmlHelper.getPathsToFrame(node);
      var xp = xmlHelper.getElementXpath(node, true);
      if ( xp ){ // got one without namespace trickery
        nodeSpec['xpath'] = xp;
        //dump("getNodeSpec succeeds directly \n");
        return nodeSpec;
      }
      var doc = node.ownerDocument;
      var nsList = xmlHelper.getNsList(doc,node);
      xp = xmlHelper.getElementXpath(node, true, nsList );
      if ( xp ) {
        nodeSpec['xpath'] = xp;
        nodeSpec['namespaces'] = nsList;
        //dump("getNodeSpec succeeds with nslist " + nsList + "\n");
        return nodeSpec;
      }
      //dump("getNodeSpec FAILS\n");
      return false;

    },

    nodeSpecToString: function (nodeSpec) {
      // [frame %%] [frame %%] xpath [@@ ns1="..."] [@@ ns2="..."]
      if (typeof(nodeSpec) == "string")
        return nodeSpec; // legacy stuff
      var s = "";
      if (nodeSpec['frames'])
        for (var i = nodeSpec['frames'].length - 1; i >= 0; i--) {
          s = s + nodeSpec['frames'][i] + " %% ";
        }
      s = s + nodeSpec['xpath'];
      if (nodeSpec['namespaces']) {
        for (var ns in nodeSpec['namespaces'] )
          s = s + " @@ " + ns + "=\"" + nodeSpec['namespaces'][ns]+"\"";
      }
      return s;
    },

    nodeSpecFromString: function (str) {
      // [frame %%] [frame %%] xpath [@@ ns1="..."] [@@ ns2="..."]
      var nodeSpec = {};
      var namespaces = str.split("@@");
      nodeSpec['namespaces'] = {};
      for (var i = 1; i < namespaces.length; i++ ) {
          var m =  namespaces[i].match(/^\s*(\w+)="([^ "]+)"\s*/);
          if ( m ) {
            //dump("Found ns def '" + m[1] +"' = ' " + m[2] + "'\n");
            nodeSpec['namespaces'][m[1]] = m[2];
          } else
              throw new Error("Bad namespace def in nodeSpec: '" + namespaces[i] + "'\n");
      }
      //var parts = str.split("%%");
      var parts = namespaces[0].split("%%");
      nodeSpec['frames']=[];
      if (parts.length > 1) {
        for ( var i = parts.length - 2; i >= 0; i--) {
          nodeSpec['frames'].push(parts[i].replace(/^\s+|\s+$/g, ''));
             // that regexp just trims leading and trailing space
        }
      }
      nodeSpec['xpath'] = parts[parts.length - 1];
      return nodeSpec;
    },

    // Helper to make a nsresolver from a map
    // map['id'] = 'http://www.indexdata.com/namespaces/foo/2010';
    // Useful with nodeSpecs, but also elsewhere
    makeNsResolverFromList : function ( map ) {
      var nsResolver = function (prefix) {
        var url = map[prefix];
        if (!url)
          url = null;
        return url;
      };
      return nsResolver;
    },

    // Helper to make a namespaceResolver, either from the nodespec,
    // or directly from the doc.
    makeNsResolver : function (doc, nodeSpec) {
        var nsdefined = false;
        for ( var k in nodeSpec['namespaces'] ) {
            // messy way to test if we have actual definitions, not just {}
            nsdefined = true;
        }
        if ( ! nsdefined ) {
            var nsResolver = doc.createNSResolver( doc );
            for ( var k in nsResolver ) {
                //dump("Default nsR: '" + k + "' = '" + nsResolver[k] + "'\n");
            }
            return nsResolver;
        }
        return xmlHelper.makeNsResolverFromList( nodeSpec['namespaces'] );
    },

    // Helper to extract the namespaces needed to refer to the node
    getNsList: function(doc, node) {
      var nspaces = {};
      var urimap = {};
      var counter = 1; // to create ns prefixes ns1, ns2...
      for( ; node ; node = node.parentNode ) {
        if (  node.namespaceURI &&
             ! urimap[node.namespaceURI] ) {
          var p = node.prefix;
          if ( ! p ) {
            p = "ns" + counter;
            counter++;
          }
          urimap[node.namespaceURI] = p;
          nspaces[p] = node.namespaceURI;
          //dump("getNsList: '" + p + "' = '" + nspaces[p] + "'\n");
        }
      }
      return nspaces;
    },

    // Unwrap node spec and evalute an xpath against it
    //
    // type is the constant result type requested. These constants are not
    // included by default in the extension's context so we'll just comment.
    evaluateNodeSpec: function (elem, nodeSpec, type) {
      xmlHelper.checkNodeSpec(nodeSpec);
      var doc = elem.documentElement.ownerDocument;
      if (nodeSpec['frames'])
        doc = xmlHelper.getDocFromFramePaths(elem, nodeSpec['frames']);
      var nsresolver = xmlHelper.makeNsResolver(doc, nodeSpec);
      if ( !nsresolver)
        nsresolver = null;

      return doc.evaluate(nodeSpec.xpath, doc, nsresolver, type, null);
    },

    getElementByNodeSpec: function (elem, nodeSpec) {
      // 9 -> FIRST_ORDERED_NODE_TYPE
      return xmlHelper.evaluateNodeSpec(elem, nodeSpec, 9).singleNodeValue;
    },

    getMatchCountByNodeSpec: function (elem, nodeSpec) {
      try {
        return xmlHelper.evaluateNodeSpec(elem, nodeSpec, 6).snapshotLength;
      } catch(err) {
        return 0;
      }
    },

    dumpxml: function (xml) {
      var str = xmlHelper.serializexml(xml)+ "\n";
      dump(str);
      return str;
    },

    serializexml: function(xml) {
      var s = Cc["@mozilla.org/xmlextras/xmlserializer;1"].createInstance(Ci.nsIDOMSerializer);
      var xmlStr = s.serializeToString(xml);
      return xmlStr;
    },

    // Return a string that contains a node name and bit of the text content
    // suitably cleaned for debug logs. 
    nodeToString: function(node, maxlen){
      if (!maxlen)
        maxlen = 20;
      if (!node)
        return "(null)";
      var s = node.localName || node.nodeName || "(??)";
      if (node.textContent) {
        var c = " " + node.textContent;
        c = c.replace( /\s+/g, " ");
        s += c.substr(0,maxlen);
      }
      return s;
    },

    // The resolveXIncludes() function was originally lifted from
    //  https://developer.mozilla.org/en/XInclude
    // but has needed some modifications to work in the CF's FireFox environment.
    //
    // Most importantly, there is either a nasty bug or a horrible
    // policy decision in nsIXMLHttpRequest which prevents it from
    // opening a local file using a relative HREF such as
    // "initTask.cff".  Attempts to do this yield the uninformative error:
    //  Component returned failure code: 0x80520012
    //  (NS_ERROR_FILE_NOT_FOUND) [nsIXMLHttpRequest.send]
    //
    // The most frustrating part is that system-call tracing shows
    // that the underlying code is not only correctly resolving such
    // relative HREFs, it is opening and reading the entire contents
    // of the correct file, before then discarding them and throwing
    // the error.  Very dumb.
    //
    // The workaround is to pass an explicit baseDir into the XInclude
    // resolver, and use this as a prefix to upgrade relative HREFs,
    // which are recognised using a simple regular expression.  The
    // baseDir is often a partial file: URL, but there is no reason it
    // couldn't be an http: or other URL.

    resolveXIncludes: function (docu, baseDir) {
        // http://www.w3.org/TR/xinclude/#xml-included-items
        var xincludes = docu.getElementsByTagNameNS('http://www.w3.org/2001/XInclude', 'include');
        if (xincludes) {
            logger.log(Level.DEBUG, "found " + xincludes.length + " XInclude nodes");

            for (let i = 0; i < xincludes.length; i++) {
                logger.log(Level.DEBUG, "handling XInclude node #" + (i+1) + " of " + xincludes.length);
                var xinclude = xincludes[i];
                var href = xinclude.getAttribute('href');
                if (!href.match(/^[a-z]+:/)) {
                    // It's not an absolute HREF
                    var old = href;
                    href = baseDir + "/" + href;
                    //logger.log(Level.INFO, "upgraded relative href '" + old + "' to " + href);
                }
                var parse = xinclude.getAttribute('parse');
                var xpointer = xinclude.getAttribute('xpointer');
                var encoding = xinclude.getAttribute('encoding'); // e.g., UTF-8 // "text/xml or application/xml or matches text/*+xml or application/*+xml" before encoding (then UTF-8)
                var accept = xinclude.getAttribute('accept'); // header "Accept: "+x
                var acceptLanguage = xinclude.getAttribute('accept-language'); // "Accept-Language: "+x
                var xiFallback = xinclude.getElementsByTagNameNS('http://www.w3.org/2001/XInclude', 'fallback')[0]; // Only one such child is allowed
                if (href === '' || href === null) { // Points to same document if empty (null is equivalent to empty string)
                    href = null; // Set for uniformity in testing below
                    if (parse === 'xml' && xpointer === null) {
                        alert('There must be an XPointer attribute present if "href" is empty an parse is "xml"');
                        return false;
                    }
                }
                else if (href.match(/#$/, '') || href.match(/^#/, '')) {
                    alert('Fragment identifiers are disallowed in an XInclude "href" attribute');
                    return false;
                }
                var j;
                var xincludeParent = xinclude.parentNode;
                try {
                //netscape.security.PrivilegeManager.enablePrivilege('UniversalXPConnect UniversalBrowserRead'); // Necessary with file:///-located files trying to reach external sites
                    if (href !== null) {
                        var response, responseType;
                        var request = Cc["@mozilla.org/xmlextras/xmlhttprequest;1"]
                          .createInstance(Ci.nsIXMLHttpRequest);
                        request.open('GET', href, false);
                        request.setRequestHeader('If-Modified-Since', 'Thu, 1 Jan 1970 00:00:00 GMT');
                        request.setRequestHeader('Cache-Control', 'no-cache');
                        if (accept) {
                            request.setRequestHeader('Accept', accept);
                        }
                        if (acceptLanguage) {
                            request.setRequestHeader('Accept-Language', acceptLanguage);
                        }
                        switch (parse) {
                            case 'text':
                                // Priority should be on media type:

                                var contentType = request.getResponseHeader('Content-Type');

                                //text/xml; charset="utf-8" // Send to get headers first?
                                // Fix: We test for file extensions as well in case file:// doesn't return content type (as seems to be the case); can some other tool be uesd in FF (or IE) to detect encoding of local file? Probably just need BOM test since other encodings must be specified
                                var patternXML = /\.(svg|xml|xul|rdf|xhtml)$/;
                                if ((contentType && contentType.match(/[text|application]\/(.*)\+?xml/)) || (href.indexOf('file://') === 0 && href.match(patternXML))) {
                                    /* Grab the response as text (see below for that routine) and then find encoding within*/
                                   var encName = '([A-Za-z][A-Za-z0-9._-]*)';
                                   var pattern = new RegExp('^<\\?xml\\s+.*encoding\\s*=\\s*([\'"])'+encName+'\\1.*\\?>'); // Check document if not?
                                   // Let the request be processed below
                                }
                                else {
                                    if (encoding === '' || encoding === null) { // Encoding has no effect on XML
                                        encoding = 'utf-8';
                                    }
                                    request.overrideMimeType('text/plain; charset='+encoding); //'x-user-defined'
                                }
                                responseType = 'responseText';
                                break;
                            case null:
                            case 'xml':
                                responseType = 'responseXML';
                                break;
                            default:
                                alert('XInclude element contains an invalid "parse" attribute value');
                                return false;
                                break;
                        }
                        request.send(null);
                        if((request.status === 200 || request.status === 0) && request[responseType] !== null) {
                            response = request[responseType];
                             if (responseType === 'responseXML') {
                                // apply xpointer (only xpath1() subset is supported)
                                var responseNodes;
                                if (xpointer) {
                                    // XPathResult.ORDERED_NODE_SNAPSHOT_TYPE was
                                    // undefined so we use a literal 7
                                    var xpathResult = response.evaluate(
                                                                     xpointer,
                                                                     response,
                                                                     null,
                                                                     7,
                                                                     null
                                                                  );
                                    var a = [];
                                    for(var k = 0; k < xpathResult.snapshotLength; k++) {
                                    a[k] = xpathResult.snapshotItem(k);
                                    }
                                    responseNodes = a;
                                }
                                else { // Otherwise, the response must be a single well-formed document response
                                    responseNodes = [response.documentElement]; // Put in array so can be treated the same way as the above
                                }
                                // PREPEND ANY NODE(S) (AS XML) -- REMOVE XINCLUDE LATER
                                for (let j=0; j < responseNodes.length ; j++) {
                                    xincludeParent.insertBefore(responseNodes[j], xinclude);
                                }
                             }
                             else if (responseType === 'responseText') {
                                 if (encName) {
                                      var encodingType = response.match(pattern);
                                      if (encodingType) {
                                          encodingType = encodingType[2];
                                      }
                                      else {
                                          encodingType = 'utf-8';
                                      }
                                      // Need to make a whole new request apparently since cannot convert the encoding after receiving it (to know what the encoding was)
                                      var request2 = new XMLHttpRequest();
                                      request2.overrideMimeType('text/plain; charset='+encodingType);
                                      request2.open('GET', href, false);
                                      request2.setRequestHeader('If-Modified-Since', 'Thu, 1 Jan 1970 00:00:00 GMT');
                                      request2.setRequestHeader('Cache-Control', 'no-cache');
                                      request2.send(null);
                                      response = request2[responseType]; // Update the response for processing
                                 }

                                 // REPLACE XINCLUDE WITH THE RESPONSE AS TEXT
                                 var textNode = docu.createTextNode(response);                             xincludeParent.replaceChild(textNode, xinclude);
                             }

                            // replace xinclude in doc with response now (as plain text or XML)
                        }
                    }
                }
                catch (e) { // Use xi:fallback if XInclude retrieval above failed
                    if (!xiFallback) throw new Error("XInclude can't include '" + href + "': " + e);
                    var xiFallbackChildren = xiFallback.childNodes;
                    // PREPEND ANY NODE(S) -- REMOVE XINCLUDE LATER
                    for (let j = 0; j < xiFallbackChildren.length ; j++) {
                        xincludeParent.insertBefore(xiFallbackChildren[j], xinclude);
                    }
                }
                logger.log(Level.DEBUG, "finished XInclude node #" + (i+1) + " of " + xincludes.length);
            }
            // Delete in reverse order so shrinking .length does not prematurely end the loop
            for (let i = xincludes.length-1; i >= 0; i--) {
                logger.log(Level.DEBUG, "deleting XInclude node #" + (i+1) + " of " + xincludes.length);
                xincludes[i].parentNode.removeChild(xincludes[i]);
            }
        }
        return docu;
    }

} // xmlHelper
