/* This file is part of the Connector Framework
 * Copyright (C) 2008-2013 Index Data
 * See the file LICENSE for details.
 */
/**
 * \file cf_display.cpp
 * \brief Pick DISPLAY for engine
 */

#include <sys/wait.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include "cf_display.h"
#include <yaz/nmem.h>
#include <pthread.h>
#include <X11/Xlib.h>

#define MAX_DISPLAY 1000
static pthread_mutex_t static_display_mutex;
static pthread_once_t static_display_once = PTHREAD_ONCE_INIT;
int static_display_list[MAX_DISPLAY];

static void init_handler(void)
{
    int i;
    for (i = 0; i < MAX_DISPLAY; i++)
        static_display_list[i] = 0;
    pthread_mutex_init(&static_display_mutex, 0);
}

static int unixLock(int fd, int type, int cmd)
{
    struct flock area;
    area.l_type = type;
    area.l_whence = SEEK_SET;
    area.l_len = area.l_start = 0L;

    return fcntl(fd, cmd, &area);
}

CF_Display::CF_Display(CF_Logger *logger)
{
    m_logger = logger;
    m_display = wrbuf_alloc();
    cf_lck_fd = -1;
    fname_lck_file = wrbuf_alloc();
    fname_log_file = wrbuf_alloc();
    unlink_fname = false;
    wrbuf_error = wrbuf_alloc();
    xvfb_pid = 0;
    dno = 0;

    (void) pthread_once(&static_display_once, init_handler);
}

const char *CF_Display::get_error()
{
    return wrbuf_cstr(wrbuf_error);
}

const char *CF_Display::get_display()
{
    if (wrbuf_len(m_display))
        return wrbuf_cstr(m_display);
    return 0;
}

const char *CF_Display::set_display()
{
    const char *disp = get_display();
    if (disp)
        setenv("DISPLAY", disp, 1);
    return disp;
}

bool CF_Display::lock(int log_fd)
{
    const char *display_lock = getenv("CF_DISPLAY_LOCK");

    if (display_lock)
    {
        const char *display_cmd = getenv("CF_DISPLAY_CMD");
        const char *display_start_str = getenv("CF_DISPLAY_START");
        int display_start = 400;
        WRBUF lockdir = wrbuf_alloc();

        wrbuf_puts(lockdir, display_lock);

        if (display_start_str && *display_start_str)
            display_start = atoi(display_start_str);

        if (!display_cmd)
            display_cmd = "Xvfb -ac -screen 0 1024x768x24";

        bool ret = lock_via_xvfb_cmd(wrbuf_cstr(lockdir),
                                     display_cmd, display_start, log_fd);
        wrbuf_destroy(lockdir);
        return ret;
    }
    else
        return true;
}

bool CF_Display::lock_via_xvfb_cmd(const char *lockdir, const char *xvfb_cmd,
                                   int display_base, int log_fd)
{
    const char *sep = strchr(xvfb_cmd, ' ');
    wrbuf_rewind(wrbuf_error);
    if (!sep || strlen(xvfb_cmd) < 4)
    {
        wrbuf_printf(wrbuf_error, "Invalid format of CF_DISPLAY_CMD: %s",
                     xvfb_cmd);
        return false;
    }

    if (mkdir(lockdir, 0775) == -1 && errno != EEXIST)
    {
        if (m_logger)
            m_logger->printf("engine", YLOG_WARN, "display: mkdir %s: %s",
                             lockdir, strerror(errno));
    }
    while (1)
    {
        // consider .pid file
        wrbuf_rewind(fname_lck_file);
        wrbuf_printf(fname_lck_file, "%s/Xvfb.%d.LCK",
                     lockdir, display_base + dno);

        pthread_mutex_lock(&static_display_mutex);
        if (static_display_list[dno] == 0)
        {
            cf_lck_fd = ::open(wrbuf_cstr(fname_lck_file),
                               O_RDWR|O_CREAT, 0666);
            if (cf_lck_fd == -1)
            {
                int x = errno;
                pthread_mutex_unlock(&static_display_mutex);
                if (m_logger)
                    m_logger->printf("engine", YLOG_WARN, "display open %s: %s",
                                     wrbuf_cstr(fname_lck_file), strerror(x));
                wrbuf_printf(wrbuf_error, "Could not open %s: %s",
                             wrbuf_cstr(fname_lck_file), strerror(x));
                return false;
            }
            if (unixLock(cf_lck_fd, F_WRLCK, F_SETLK) != -1)
            {
                static_display_list[dno] = 1;
                pthread_mutex_unlock(&static_display_mutex);
                break;
            }
            ::close(cf_lck_fd);
            cf_lck_fd = -1;
        }
        pthread_mutex_unlock(&static_display_mutex);
        dno++;
        if (dno == MAX_DISPLAY)
        {
            if (m_logger)
                m_logger->printf("engine", YLOG_WARN,
                                 "display dno=%d giving up", dno);
            wrbuf_printf(wrbuf_error, "display dno=%d giving up", dno);
            return false;
        }
    }
    if (m_logger)
        m_logger->printf("engine", YLOG_LOG, "display: lock %s",
                         wrbuf_cstr(fname_lck_file));

    pid_t pid = 0;
    WRBUF x11_lock_fname = wrbuf_alloc();
    wrbuf_printf(x11_lock_fname, "/tmp/.X%d-lock", display_base + dno);
    int x11_lock_fd = open(wrbuf_cstr(x11_lock_fname), O_RDONLY);
    if (x11_lock_fd == -1 && errno != ENOENT)
    {
        int x = errno;
        if (m_logger)
            m_logger->printf("engine", YLOG_WARN, "display: open %s: %s",
                             wrbuf_cstr(x11_lock_fname), strerror(x));
        wrbuf_printf(wrbuf_error, "display: open %s: %s",
                     wrbuf_cstr(x11_lock_fname), strerror(x));
        wrbuf_destroy(x11_lock_fname);
        close();
        return false;
    }
    if (x11_lock_fd != -1)
    {
        char buf[40];
        ssize_t no_read = read(x11_lock_fd, buf, sizeof(buf)-1);
        int x = errno;
        ::close(x11_lock_fd);

        if (no_read == -1)
        {
            if (m_logger)
                m_logger->printf("engine", YLOG_WARN, "display: read %s: %s",
                                 wrbuf_cstr(x11_lock_fname), strerror(x));
            wrbuf_printf(wrbuf_error, "display: read %s: %s",
                         wrbuf_cstr(x11_lock_fname), strerror(x));
            wrbuf_destroy(x11_lock_fname);
            close();
            return false;
        }
        buf[no_read] = '\0';
        pid = atol(buf);
    }
    wrbuf_destroy(x11_lock_fname);

    if (pid > 0)
    {
        while (1)
        {
            int r = kill(pid, SIGTERM);
            int x = errno;
            if (m_logger)
            {
                if (r == -1)
                    m_logger->printf("engine", YLOG_WARN,
                                     "display: kill stale Xvfb PID=%ld: %s",
                                     (long) pid, strerror(x));
                else
                    m_logger->printf("engine", YLOG_WARN,
                                     "display: kill stale Xvfb PID=%ld",
                                     (long) pid);
            }
            if (r == -1)
            {
                if (x == ESRCH)
                    break; /* stale process is gone */
                else
                {
                    wrbuf_printf(wrbuf_error, "display: unable to kill stale Xvfb "
                                 "PID=%ld: %s", (long) pid, strerror(x));
                    close();
                    return false;
                }
            }
            struct timespec req;
            req.tv_sec = 0;
            req.tv_nsec = 50000000; /* 1/20 sec */
            ::nanosleep(&req, 0);
        }
    }
    wrbuf_printf(fname_log_file, "%s/Xvfb.%d.log", lockdir, display_base + dno);

    unlink_fname = true;

    xvfb_pid = ::fork();
    if (xvfb_pid == -1)
    {
        wrbuf_printf(wrbuf_error, "fork: %s", strerror(errno));
        close();
        return false;
    }
    else if (xvfb_pid == 0)
    {
        int i;
        for (i = getdtablesize(); --i >= 0; )
            ::close(i);

        // stdin
        int log_file_fd = open(wrbuf_cstr(fname_log_file),
                               O_CREAT|O_WRONLY|O_TRUNC, 0666);
        if (log_file_fd != 0)
            exit(1);

        dup(log_file_fd); // stdout
        dup(log_file_fd); // stderr

        char display_spec[50];

        sprintf(display_spec, ":%d", display_base + dno);

        NMEM n = nmem_create();

        char **args1;
        int num;
        nmem_strsplit(n, " ", xvfb_cmd, &args1, &num);

        char **args2 = (char **) nmem_malloc(n, sizeof(*args2) * (num + 2));
        args2[0] = args1[0];
        args2[1] = display_spec;
        for (i = 1; i < num; i++)
            args2[i + 1] = args1[i];
        args2[num + 1] = 0;

        if (strchr(*args2, '/'))
            execv(*args2, args2);
        else
            execvp(*args2, args2);

        exit(0);
        return false;
    }

    ftruncate(cf_lck_fd, 0);
    char pidstr[60];
    sprintf(pidstr, "%ld", (long) xvfb_pid);
    if ((size_t) write(cf_lck_fd, pidstr, strlen(pidstr)) != strlen(pidstr))
    {
        int x = errno;
        if (m_logger)
            m_logger->printf("engine", YLOG_WARN, "write %s: %s",
                             wrbuf_cstr(fname_lck_file), strerror(x));
        wrbuf_printf(wrbuf_error, "write %s: %s",
                     wrbuf_cstr(fname_lck_file), strerror(x));
        close();
        return false;
    }

    wrbuf_rewind(m_display);
    wrbuf_printf(m_display, ":%d.0", display_base + dno);

    if (m_logger)
        m_logger->printf("engine", YLOG_LOG, "display: spawn DISPLAY=%s PID=%s",
             wrbuf_cstr(m_display), pidstr);

    int i;
    for (i = 0; i < 40; i++)
    {
        Display *x11_display = XOpenDisplay(wrbuf_cstr(m_display));
        if (x11_display)
        {
            if (m_logger)
                m_logger->printf("engine", YLOG_LOG,
                                 "display: XOpenDisplay OK for %s i=%d",
                                 wrbuf_cstr(m_display), i);
            XCloseDisplay(x11_display);
            return true;
        }

        if (m_logger && i > 10)
            m_logger->printf("engine", YLOG_WARN,
                             "display: XOpenDisplay FAIL for %s i=%d",
                             wrbuf_cstr(m_display), i);

        int status;
        if (waitpid(xvfb_pid, &status, WNOHANG) == xvfb_pid)
        {
            if (WIFEXITED(status))
            {
                if (m_logger)
                    m_logger->printf(
                        "engine", YLOG_WARN,
                        "display: Xvfb process %ld exited with status %d",
                        (long) xvfb_pid, WEXITSTATUS(status));
                break;
            }
            else if (WIFSIGNALED(status))
            {
                if (m_logger)
                    m_logger->printf(
                        "engine", YLOG_WARN,
                        "display: Xvfb process %ld killed by signal %d",
                        (long) xvfb_pid, WTERMSIG(status));
                break;
            }
        }
        struct timespec req;
        req.tv_sec = 0;
        req.tv_nsec = 50000000; /* 1/20 sec */

        if (::nanosleep(&req, 0))
        {
            int x = errno;
            if (m_logger)
                m_logger->printf("engine", YLOG_WARN,
                                 "display: nanosleep errno=%d", x);
            if (x != EINTR)
                break;
        }
    }

    if (m_logger)
        m_logger->printf("engine", YLOG_WARN,
                         "display: Giving up for DISPLAY=%s",
                         wrbuf_cstr(m_display));
    wrbuf_rewind(wrbuf_error);
    wrbuf_printf(wrbuf_error, "Xvfb start failure. DISPLAY=%s",
                 wrbuf_cstr(m_display));
    close();
    return false;
}

void CF_Display::close()
{
    if (xvfb_pid)
    {
        int r = kill(xvfb_pid, SIGTERM);
        int x = errno;

        if (m_logger)
        {
            if (r == 0)
                m_logger->printf("engine", YLOG_DEBUG,
                                 "display: kill Xvfb DISPLAY=%s PID=%ld",
                                 wrbuf_cstr(m_display), (long) xvfb_pid);
            else
                m_logger->printf("engine", YLOG_WARN,
                                 "display: kill Xvfb DISPLAY=%s PID=%ld: %s",
                                 wrbuf_cstr(m_display), (long) xvfb_pid,
                                 strerror(x));
        }
        if (r == 0)
        {
            int status;
            waitpid(xvfb_pid, &status, 0);
        }
    }
    if (unlink_fname && wrbuf_len(fname_lck_file))
        ::unlink(wrbuf_cstr(fname_lck_file));

    xvfb_pid = 0;
    if (cf_lck_fd != -1)
    {
        ::close(cf_lck_fd);
        cf_lck_fd = -1;
    }
    pthread_mutex_lock(&static_display_mutex);
    static_display_list[dno] = 0;
    pthread_mutex_unlock(&static_display_mutex);
}

CF_Display::~CF_Display()
{
    close();
    wrbuf_destroy(wrbuf_error);
    wrbuf_destroy(m_display);
    wrbuf_destroy(fname_lck_file);
    wrbuf_destroy(fname_log_file);
}


/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */

