/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/* vim: set ts=4 et sw=4 tw=80: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Nokia.
 *
 * The Initial Developer of the Original Code is Nokia Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2012
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Oleg Romashin <romaxa@gmail.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "nsAppDirectoryServiceDefs.h"
#include "nsProfileDirServiceProvider.h"
#include "nsXPCOMGlue.h"
#include "nsXULAppAPI.h"
#include <iostream>
#include <string>
#include <unistd.h>

#define MAX_PATH 2048

using namespace std;

// ------------------------------------------------------------------------

nsIDirectoryServiceProvider *sAppFileLocProvider = 0;
nsCOMPtr<nsIFile> sProfileDir = 0;
nsISupports * sProfileLock = 0;

XRE_InitEmbedding2Type XRE_InitEmbedding2 = 0;
XRE_TermEmbeddingType XRE_TermEmbedding = 0;
XRE_NotifyProfileType XRE_NotifyProfile = 0;
XRE_GetBinaryPathType XRE_GetBinaryPath = 0;
XRE_LockProfileDirectoryType XRE_LockProfileDirectory = 0;

class MozEmbedDirectoryProvider : public nsIDirectoryServiceProvider2
{
public:
    NS_DECL_ISUPPORTS_INHERITED
    NS_DECL_NSIDIRECTORYSERVICEPROVIDER
    NS_DECL_NSIDIRECTORYSERVICEPROVIDER2
};

static MozEmbedDirectoryProvider kDirectoryProvider;

NS_IMPL_QUERY_INTERFACE2(MozEmbedDirectoryProvider,
                         nsIDirectoryServiceProvider,
                         nsIDirectoryServiceProvider2)

NS_IMETHODIMP_(nsrefcnt)
MozEmbedDirectoryProvider::AddRef()
{
    return 1;
}

NS_IMETHODIMP_(nsrefcnt)
MozEmbedDirectoryProvider::Release()
{
    return 1;
}

NS_IMETHODIMP
MozEmbedDirectoryProvider::GetFile(const char *aKey, bool *aPersist,
                                   nsIFile* *aResult)
{
    if (sAppFileLocProvider) {
        nsresult rv = sAppFileLocProvider->GetFile(aKey, aPersist, aResult);
        if (NS_SUCCEEDED(rv))
            return rv;
    }

    if (sProfileDir && !strcmp(aKey, NS_APP_USER_PROFILE_50_DIR)) {
        *aPersist = PR_TRUE;
        return sProfileDir->Clone(aResult);
    }

    if (sProfileDir && !strcmp(aKey, NS_APP_PROFILE_DIR_STARTUP)) {
        *aPersist = PR_TRUE;
        return sProfileDir->Clone(aResult);
    }

    if (sProfileDir && !strcmp(aKey, NS_APP_CACHE_PARENT_DIR)) {
        *aPersist = PR_TRUE;
        return sProfileDir->Clone(aResult);
    }

    return NS_ERROR_FAILURE;
}

NS_IMETHODIMP
MozEmbedDirectoryProvider::GetFiles(const char *aKey,
                                    nsISimpleEnumerator* *aResult)
{
    nsCOMPtr<nsIDirectoryServiceProvider2>
        dp2(do_QueryInterface(sAppFileLocProvider));

    if (!dp2)
        return NS_ERROR_FAILURE;

    return dp2->GetFiles(aKey, aResult);
}

static nsresult
InitEmbedding(const char* aProfilePath, const char* aGREHome)
{
  nsresult rv;

  // get rid of the bogus TLS warnings
  NS_LogInit();

  string xpcomPath = aGREHome;
  xpcomPath += "/libxpcom.so";
  cout << "Loading XPCOM from:" << xpcomPath << endl;

  // start the glue, i.e. load and link against xpcom shared lib
  rv = XPCOMGlueStartup(xpcomPath.c_str());
  if (NS_FAILED(rv)) {
    cerr << "Could not start XPCOM glue." << endl;
    return 2;
  }

  // load XUL functions
  nsDynamicFunctionLoad nsFuncs[] = {
      {"XRE_InitEmbedding2", (NSFuncPtr*)&XRE_InitEmbedding2},
      {"XRE_TermEmbedding", (NSFuncPtr*)&XRE_TermEmbedding},
      {"XRE_NotifyProfile", (NSFuncPtr*)&XRE_NotifyProfile},
      {"XRE_GetBinaryPath", (NSFuncPtr*)&XRE_GetBinaryPath},
      {"XRE_LockProfileDirectory", (NSFuncPtr*)&XRE_LockProfileDirectory},
      {0, 0}
  };
  rv = XPCOMGlueLoadXULFunctions(nsFuncs);

  if (NS_FAILED(rv)) {
    cerr << "Could not load XUL functions." << endl;
    return 4;
  }

  // strip the filename from xpcom so we have the dir instead
  size_t lastslash = xpcomPath.find_last_of("/\\");
  if (lastslash == string::npos) {
    cerr << "Invalid path to xpcom:" << xpcomPath << "." << endl;
    return 3;
  }
  string xpcomDir = xpcomPath.substr(0, lastslash);
  if (!getenv("MOZILLA_FIVE_HOME")) {
    setenv("MOZILLA_FIVE_HOME", xpcomDir.c_str(), 1);
  }

  // create nsIFile pointing to xpcomDir
  nsCOMPtr<nsIFile> xuldir;
  rv = XRE_GetBinaryPath(xpcomDir.c_str(), getter_AddRefs(xuldir));
  if (NS_FAILED(rv)) {
    cerr << "Unable to create nsIFile for xuldir " << xpcomDir << endl;
    return 6;
  }

  // create nsIFile pointing to appdir
  char self[MAX_PATH];
  // TODO: works on linux, need solution for unices which do not support this
  ssize_t len;
  if ((len = readlink("/proc/self/exe", self, sizeof(self)-1)) != -1)
    self[len] = '\0';
  string selfPath(self);
  size_t lastslash_t = selfPath.find_last_of("/\\");
  if (lastslash_t == string::npos) {
    cerr << "Invalid module filename: " << self << "." << endl;
    return 7;
  }

  selfPath = selfPath.substr(0, lastslash_t);

  nsCOMPtr<nsIFile> appdir;
  rv = XRE_GetBinaryPath(selfPath.c_str(), getter_AddRefs(appdir));
  if (NS_FAILED(rv)) {
    cerr << "Unable to create nsIFile for appdir." << endl;
    return 8;
  }

  // setup profile dir
  nsCString pr(aProfilePath);
  if (!pr.IsEmpty()) {
    if (pr.First() != '/') {
      pr.Assign(getenv("HOME"));
      pr.AppendLiteral("/.mozilla/");
      pr.Append(aProfilePath);
    }
    printf("Creating profile in:%s\n", pr.get());
    rv = NS_NewNativeLocalFile(pr, PR_FALSE,
                               getter_AddRefs(sProfileDir));
    if (NS_FAILED(rv)) {
      cerr << "NS_NewNativeLocalFile failed." << endl;
      return 10;
    }
  } else {
    // for now use a subdir under appdir
    nsCOMPtr<nsIFile> profFile;
    rv = appdir->Clone(getter_AddRefs(profFile));
    if (NS_FAILED(rv)) {
      cerr << "Unable to clone nsIFile." << endl;
      return 11;
    }

    sProfileDir = do_QueryInterface(profFile);
    sProfileDir->AppendNative(NS_LITERAL_CSTRING("mozembed"));
  }

  // create dir if needed
  bool dirExists = true;
  rv = sProfileDir->Exists(&dirExists);
  if (!dirExists) {
    sProfileDir->Create(nsIFile::DIRECTORY_TYPE, 0700);
  }

  // Lock profile directory
  if (sProfileDir && !sProfileLock) {
    rv = XRE_LockProfileDirectory(sProfileDir, &sProfileLock);
    if (NS_FAILED(rv)) {
      cerr << "Unable to lock profile directory." << endl;
      return 12;
    }
  }

  printf(">>>>>>Func:%s::%d Init Embedding right now\n", __PRETTY_FUNCTION__, __LINE__);
  // init embedding
  rv = XRE_InitEmbedding2(xuldir, appdir,
                          const_cast<MozEmbedDirectoryProvider*>(&kDirectoryProvider));
  printf(">>>>>>Func:%s::%d Init Embedding right finished rv:%x\n", __PRETTY_FUNCTION__, __LINE__, rv);
  if (NS_FAILED(rv)) {
    cerr << "XRE_InitEmbedding2 failed." << endl;
    return 9;
  }

  // initialize profile:
  XRE_NotifyProfile();

  NS_LogTerm();

  return NS_OK;
}

nsresult TermEmbedding()
{
  // get rid of the bogus TLS warnings
  NS_LogInit();

  XRE_TermEmbedding();

  // make sure this is freed before shutting down xpcom
  NS_IF_RELEASE(sProfileLock);
  sProfileDir = 0;

  NS_LogTerm();

  return NS_OK;
}

// Taken from
// https://bug736564.bugzilla.mozilla.org/attachment.cgi?id=606849
int main(int argc, char** argv)
{
    int ret = 0;
    printf(">>>>>>Func:%s::%d Init Embedding start\n", __PRETTY_FUNCTION__, __LINE__);
    InitEmbedding("mozembed", argc == 2 ? argv[1] : DEFAULT_GRE_HOME);
    printf(">>>>>>Func:%s::%d Embedding Initialized\n", __PRETTY_FUNCTION__, __LINE__);
    TermEmbedding();
    printf(">>>>>>Func:%s::%d Embedding exited\n", __PRETTY_FUNCTION__, __LINE__);
    return ret;
}
