/* This file is part of the Connector Framework
 * Copyright (C) 2008-2013 Index Data
 * See the file LICENSE for details.
 */

/** \file
 * \brief JSON to Facet response conversion
 */

#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <yaz/xmalloc.h>
#include <yaz/yaz-iconv.h>
#include <yaz/proto.h>

#include "json.h"

struct json_sax {
    struct json_node **stack;
    int max_level;
    int level;
};

static struct json_sax *json_sax_init(int max_level, struct json_node *n)
{
    struct json_sax *p = (struct json_sax *) xmalloc(sizeof(*p));
    p->max_level = max_level;
    p->level = 0;
    p->stack = (struct json_node **) xmalloc((1+max_level) * sizeof(*p->stack));
    p->stack[0] = n;
    return p;
}

static void json_sax_destroy(struct json_sax *p)
{
    xfree(p->stack);
    xfree(p);
}

static struct json_node *json_sax_get_node(struct json_sax *p)
{
    struct json_node *n = p->stack[p->level];
    return n;
}

static int json_sax_move(struct json_sax *p)
{
    struct json_node *n = p->stack[p->level];

    if (n && (n->type == json_node_object || n->type == json_node_array
              || n->type == json_node_list || n->type == json_node_pair))
    {
        p->stack[p->level] = n->u.link[1];
        n = n->u.link[0];
        if (p->level < p->max_level)
            p->stack[++p->level] = n;
    }
    else
        n = 0;
    while (!n && p->level > 0)
        n = p->stack[--p->level];
    return p->level;
}

void facet_to_json(const Z_FacetList *facet_list, WRBUF w)
{
    int i, first = 1;

    wrbuf_puts(w, "{");
    for (i = 0; i < facet_list->num; i++)
    {
        Z_FacetField *ff = facet_list->elements[i];
        struct yaz_facet_attr attr;
        int j;

        yaz_facet_attr_init(&attr);
        attr.sortorder = -1;
        attr.limit = -1;
        attr.start = -1;
        yaz_facet_attr_get_z_attributes(ff->attributes, &attr);
        if (attr.errcode || attr.useattr == 0)
            continue;

        if (first)
            first = 0;
        else
            wrbuf_puts(w, ",");

        wrbuf_puts(w, "\"");
        wrbuf_json_puts(w, attr.useattr);
        wrbuf_puts(w, "\":{");
        if (attr.sortorder >= 0)
            wrbuf_printf(w, "\"sortorder\":%d,", attr.sortorder);
        if (attr.limit >= 0)
            wrbuf_printf(w, "\"limit\":%d,", attr.limit);
        if (attr.start >= 0)
            wrbuf_printf(w, "\"start\":%d,", attr.start);
        wrbuf_puts(w, "\"terms\":[");
        for (j = 0; j < ff->num_terms; j++)
        {
            Z_Term *term = ff->terms[j]->term;
            Odr_int *count = ff->terms[j]->count;

            if (j)
                wrbuf_puts(w, ",");
            wrbuf_puts(w, "{\"term\":");
            if (count && term->which == Z_Term_general)
            {
                wrbuf_puts(w, "\"");
                wrbuf_json_write(w, term->u.general->buf, term->u.general->len);
                wrbuf_puts(w, "\"");
            }
            else
                wrbuf_puts(w, "null");
            wrbuf_puts(w, ",");
            if (count)
            {
                wrbuf_puts(w, "\"count\":");
                wrbuf_printf(w, ODR_INT_PRINTF, *count);
            }
            wrbuf_puts(w, "}");
        }
        wrbuf_puts(w, "]}");
    }
    wrbuf_puts(w, "}");
}

Z_FacetList *json_to_facet(struct json_node *res, ODR odr)
{
    Z_FacetList *facet_list = 0;
    struct json_node *facets_node = json_get_object(res, "facets");

    if (facets_node)
    {
        WRBUF w = wrbuf_alloc();
        json_write_wrbuf(facets_node, w);
        yaz_log(YLOG_LOG, "facet json: %s", wrbuf_cstr(w));
        wrbuf_destroy(w);
    }
    if (facets_node)
    {
        struct json_sax *sax = json_sax_init(20, facets_node);
        int level = 0;
        int flevel = 0;
        bool count_value = false;
        bool term_value = false;
        int facet_size = 10; /* how many facets at most */
        int facet_idx = 0;
        int term_idx = 0;
        int term_size = 40; /* how many terms at most for each facet */

        facet_list = facet_list_create(odr, facet_size);
        while (1)
        {
            struct json_node *n = json_sax_get_node(sax);
            if (!n)
                break;
            if (n->type == json_node_string && flevel == 0)
                flevel = level;
            if (n->type == json_node_string && level == flevel)
            {
                if (facet_idx < facet_size)
                {
                    Z_AttributeList *al =
                        zget_AttributeList_use_string(odr, n->u.string);
                    facet_list->elements[facet_idx] =
                        facet_field_create(odr, al, term_size);
                    term_idx = 0;
                    facet_list->elements[facet_idx]->num_terms = 0;
                    facet_list->num = ++facet_idx;
                }
            }
            if (n->type == json_node_string && level > flevel)
            {
                if (term_value)
                {
                    if (facet_idx > 0 && facet_idx <= facet_size
                        && term_idx < term_size)
                    {
                        Z_FacetTerm *facet_term =
                            facet_term_create_cstr(odr, n->u.string, 0);
                        facet_list->elements[facet_idx - 1]->terms[term_idx] =
                            facet_term;
                        facet_list->elements[facet_idx - 1]->num_terms =
                            ++term_idx;
                    }
                    term_value = false;
                }
                else if (count_value)
                {
                    if (facet_idx > 0 && facet_idx <= facet_size
                        && term_idx > 0 && term_idx <= term_size)
                    {
                        *facet_list->elements[facet_idx - 1]->
                            terms[term_idx - 1]->count =
                            odr_atoi(n->u.string);
                    }
                    count_value = false;
                }
                else
                {
                    if (!strcmp(n->u.string, "count"))
                        count_value = true;
                    else if (!strcmp(n->u.string, "term"))
                        term_value = true;
                }
            }
            if (n->type == json_node_number && level > flevel)
            {
                if (count_value)
                {
                    if (facet_idx > 0 && facet_idx <= facet_size
                        && term_idx > 0 && term_idx <= term_size)
                    {
                        *facet_list->elements[facet_idx - 1]->
                            terms[term_idx - 1]->count = n->u.number;
                    }
                    count_value = false;
                }
            }
            level = json_sax_move(sax);
        }
        json_sax_destroy(sax);
    }
    return facet_list;
}

/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */
