var EXPORTED_SYMBOLS = ["Anyallquery"];

// Takes a listquery and tries to extract some terms that
// can got into typical fields some forms, like
//  - all words
//  - any of the words
//  - none of the words
//  - exact phrase
// Returns these lists (in listquery format) in separate variables,
// typically in $.temp.

Components.utils.import('resource://indexdata/runtime/Connector.js');
Components.utils.import('resource://indexdata/runtime/Task.js');
Components.utils.import('resource://indexdata/runtime/Step.js');
Components.utils.import('resource://indexdata/runtime/StepError.js');

Components.utils.import('resource://indexdata/util/logging.js');
Components.utils.import('resource://indexdata/util/xulHelper.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/flatquery.js');
Components.utils.import('resource://indexdata/thirdparty/jsonPath.js');
Components.utils.import('resource://indexdata/util/jsonPathHelper.js');

var logger = logging.getLogger();
const anyindex="(any)";

// Constructor
var Anyallquery = function () {
  this.conf = {};
  this.conf['in'] =  { path:'$.temp', key:'listquery' };
  this.conf['anylist'] = { path:'$.temp', key:'any' };
  this.conf['anysupported'] = true;
  this.conf['anyrequired'] = false;
  this.conf['alllist'] = { path:'$.temp', key:'all' };
  this.conf['allsupported'] = true;
  this.conf['allrequired'] = false;
  this.conf['nonelist'] = { path:'$.temp', key:'none' };
  this.conf['nonesupported'] = true;
  this.conf['nonerequired'] = false;
  this.conf['exactlist'] = { path:'$.temp', key:'exact' };
  this.conf['exactsupported'] = true;
  this.conf['exactrequired'] = false;
  this.conf['singlesupported'] = false;
  this.conf['singletype'] = { path:'$.temp', key:'combtype' };
  this.conf['singlelist'] = { path:'$.temp', key:'combined' };
};


Anyallquery.prototype = new Step();
Anyallquery.prototype.constructor = Anyallquery;
Anyallquery.prototype.init = function() {};


///////////////////
// UI

function oneline(context, box, name, caption) {
  var cb = xmlHelper.appendNode(box, "hbox",  null, { align: "center" });
  xulHelper.captionField(cb, caption, { width: 80 });
  xulHelper.checkbox(cb, context, name+"supported", "Supported" );
  xulHelper.checkbox(cb, context, name+"required", "Required" );
  xulHelper.jsonPathField(cb, context, context.conf, name + "list", "" );
}

Anyallquery.prototype.draw = function(surface,win) {
  var context = this;
  var vb = xmlHelper.appendNode(surface, "vbox" );
  //xulHelper.jsonPathMapField(vb, this, "in", "out");
  xulHelper.jsonPathField(vb, context, context.conf, "in", "Source " );
  oneline(context, vb, "all",  "All");
  oneline(context, vb, "any",  "Any");
  oneline(context, vb, "none", "None");
  oneline(context, vb, "exact","Exact");
  var cb = xmlHelper.appendNode(vb, "hbox",  null, { align: "center" });
  xulHelper.checkbox(cb, context, "singlesupported",
                   "Make a single combined list into" );
  cb = xmlHelper.appendNode(vb, "hbox",  null, { align: "center" });
  xulHelper.captionField(cb, "List", { width: 80 });
  xulHelper.jsonPathField(cb, context, context.conf, "singlelist", "" );
  xulHelper.labelField(vb,"And put 'any'/'all'/'none'/'exact' into");
  cb = xmlHelper.appendNode(vb, "hbox",  null, { align: "center" });
  xulHelper.captionField(cb, "Type", { width: 80 });
  xulHelper.jsonPathField(cb, context, context.conf, "singletype", "" );

};


/////////////////////////
// Run


Anyallquery.prototype.run = function (task) {
  var lq = jsonPathHelper.get(this.conf.in, task.data);
  task.debug("anyall: starting with " + JSON.stringify(lq) );
  if ( !lq || !lq.length || lq.length != 1 ) {
    task.debug("anyall: Got a bad in: " + JSON.stringify(lq) )
    return; // Should not happen. TODO - Throw an error, optionally(?)
  }
  lq = lq[0];  // get returns an array of all matches, of which there is one
  var any=[];
  var all=[];
  var none=[];
  var exact=[];

  if ( !lq || lq.length == 0) {
    task.debug("anyall: nothing to do");
  }
  else if ( lq.length == 1 ) {
    if ( lq[0].structure == "phrase" ) {
      task.debug("anyall: single phrase" );
      if ( !this.conf.exactsupported )
        throw new StepError("Phrases not supported");
      if ( lq[0].op == "" || lq[0].op == "and" )
        exact.push(lq[0]);
      else
        throw new StepError("Phrases only supported with AND");
    } else {
      if ( lq[0].op == "not" ) {
        task.debug("anyall: single not-term" );
        none.push(lq[0]);
      } else {
        task.debug("anyall: single term" );
        all.push(lq[0]);
      }
    }
  }
  else {  // longer list
    var op = lq[0].op;
    if ( op == "" || op == undefined ) {
      op = lq[1].op; // can be in the all first element of whole listquery
    }
    if ( op == "not" )
      op = "and";
    task.debug("anyall: listcase: op=" + op );

    for ( var i=0; i<lq.length; i++ ) {
      task.debug("anyall: list[" + i + "]: " +
        "op=" + lq[i].op + " term='" +  lq[i].term + "'");
      var thisop = lq[i].op;
      if ( thisop == "" || thisop == undefined )
        thisop = op;
      if ( lq[i].structure == "phrase" && !this.conf.exactsupported)
        throw new StepError("Phrases not supported");

      if ( thisop == "not" )
        none.push(lq[i]);
      else if ( thisop == "and" && lq[i].structure == "phrase" ) {
        if ( exact.length == 0 )
          exact.push(lq[i]);
        else
          throw new StepError("Query too complex, multiple phrases");
      }
      else if ( lq[i].structure == "phrase" )
        throw new StepError("Query too complex, phrases can only be ANDed");
      else if ( thisop == op && op == "and" )
        all.push(lq[i]);
      else if ( thisop == op && op == "or" )
        any.push(lq[i]);
      else throw new StepError("Query too complex, can not mix AND/OR");
        // TODO - This is overly simple. We can handle queries like
        // (foo or bar) and gryf
        //   all: gryf
        //   any: foo bar
        // but not
        // ( foo and bar ) or gryf
    }
  }

  // Put the results into task.data
  // implemented as a loop through the types array, not to repeat too much code
  var types = { all: all, any: any, none: none, exact:exact };
  var single = [];  // collect the single line stuff too, while looping
  var stype = "";
  for ( var typ in types ) {
    task.debug("anyall: looping through " + typ + " with " + types[typ].length);
    if ( this.conf[ typ + "supported" ] )
      jsonPathHelper.set(this.conf[typ+"list"], types[typ], task.data);
    else if ( types[typ].length > 0 )
      throw new StepError("'" + typ + "' type lists not supported");
    if ( this.conf[ typ + "required" ] && types[typ].length == 0)
      throw new StepError("Did not get anything " +
        " in '" + typ + "' list, which is required");
    if ( this.conf.singlesupported && types[typ].length > 0 ) {
      if ( stype )
        throw new StepError ("Query too complex, can not make '" +
          stype + "' and '" + typ + "' into same list");
      single = types[typ];
      stype = typ;
      task.debug("Found single " + typ + " with " + single.length + " elements");
    }
  }
  if ( this.conf.singlesupported ) {
    task.debug("Setting the singleton: " +
      JSON.stringify(this.conf.singletype) + "=" + stype + " " +
      JSON.stringify(this.conf.singlelist) + "=" + JSON.stringify(single) );
      
    jsonPathHelper.set(this.conf.singletype, stype, task.data);
    jsonPathHelper.set(this.conf.singlelist, single, task.data);
  }
};

/////////////////////////
// Unit test

// Little helper to check a result line
function checkresult ( name, step, test, task ) {
  var expected = test[name];
  if ( expected == undefined )
    expected = "";
  var gindex = name + "list";
  var got = jsonPathHelper.get(step.conf[gindex], task.data);
  //logger.info("check: " + test.name + "." + name + ": " +
  //            "e='" + JSON.stringify(expected) + "' "+
  //            "g='" + JSON.stringify(got) + "'");
  if ( got && Array.isArray(got) )
    got = got[0];
  got = JSON.stringify(got);
  if ( got == "[]")
    got = "";
  //logger.info("check: " + name + " e='" + expected + "' g='" + got + "'");
  var res = true;
  if (!got && !expected)
    res = true;
  else if ( got && !expected ) {
    logger.error("Test '" + test.name + "' FAILED ");
    logger.error("Expected no output in " + name );
    logger.error("But got " + got );
    res = false;
  }
  else if ( !got && expected ) {
    logger.error("Test '" + test.name + "' FAILED ");
    logger.error("Got no output in " + name );
    logger.error("Expected " + expected );
    res = false;
  }
  else if ( got != expected ) {
    logger.error("Test '" + test.name + "' FAILED ");
    logger.error("Different output in " + name );
    logger.error("Expected " + expected );
    logger.error("but got  " + got );
    res = false;
  }
  if ( name == "single" ) { // check the single type as well
    var styp = jsonPathHelper.get(step.conf.singletype, task.data)
    var sexp = test.singletype;
    if ( sexp && styp != sexp ) {
      logger.error("Test '" + test.name + "' FAILED ");
      logger.error("Different output singletype " );
      logger.error("Expected " + sexp );
      logger.error("but got  " + styp );
      res = false;
    }
  }
  return res;
}

Anyallquery.prototype.unitTest = function ( ) {
  const tests = [
    { name: 'single term',
      //'override': {}, // run with the defaults from constructor
      query: '[ { "op":"", "term":"foo" } ]' ,
      // error: "expected error message",
      all: '[{"op":"","term":"foo"}]',
      //any: '',
      //none: '',
      //exact: '',
      //single: '' ,
      //singletype: '',
    },
    { name: 'combined',
      override: { singlesupported: true},
      query: '[ { "op":"", "term":"foo" } ]' ,
      all: '[{"op":"","term":"foo"}]',
      single: '[{"op":"","term":"foo"}]' ,
      singletype: 'all',
    },
    { name: 'and',
      override: { singlesupported: true},
      query: '[ { "op":"", "term":"foo" },' +
               '{ "op":"and", "term":"bar" } ]' ,
      all : '[{"op":"","term":"foo"},{"op":"and","term":"bar"}]',
      single : '[{"op":"","term":"foo"},{"op":"and","term":"bar"}]',
      singletype: 'all',
    },
    { name:  'or',
      query: '[ { "op":"", "term":"foo" },' +
               '{ "op":"or", "term":"bar" } ]' ,
      any :  '[{"op":"","term":"foo"},{"op":"or","term":"bar"}]',
    },
    { name: 'or-unsupp',
      override: { anysupported: false },
      query: '[ { "op":"", "term":"foo" },' +
               '{ "op":"or", "term":"bar" } ]' ,
      error: "'any' type lists not supported",
    },
    { name: 'not',
      query: '[ { "op":"", "term":"foo" },' +
               '{ "op":"not", "term":"bar" } ]' ,
      all : '[{"op":"","term":"foo"}]',
      none: '[{"op":"not","term":"bar"}]',
    },
    { name: 'not-singlefail',
      override: { singlesupported: true},
      query: '[ { "op":"", "term":"foo" },' +
               '{ "op":"not", "term":"bar" } ]' ,
      error: "Query too complex, can not make 'all' and 'none' into same list",
    },
    { name: 'single-not',
      override: { singlesupported: true},
      query: '[ { "op":"not", "term":"bar" } ]' ,
      none: '[{"op":"not","term":"bar"}]',
      single: '[{"op":"not","term":"bar"}]',
      singletype: 'none',
    },
    { name: 'andor',
      query: '[ { "op":"", "term":"foo" },' +
                  '{ "op":"and", "term":"bar" }, '+
                  '{ "op":"or", "term":"gryf" } ]' ,
      error: "Query too complex, can not mix AND/OR",
    },
    { name: 'phrase',
      override: { singlesupported: true},
      query: '[ { "op":"", "term":"foo", "structure":"phrase" } ]' ,
      exact: '[{"op":"","term":"foo","structure":"phrase"}]',
      single: '[{"op":"","term":"foo","structure":"phrase"}]',
      singletype: 'exact',
    },
    { name: 'phrase-unsupp',
      override: { exactsupported: false },
      query: '[ { "op":"", "term":"foo", "structure":"phrase" } ]' ,
      error: "Phrases not supported",
    },
    { name: 'and-phrase',
      query: '[ { "op":"", "term":"foo" }, '+
               '{ "op":"and", "term":"bar", "structure":"phrase" }]' ,
      exact: '[{"op":"and","term":"bar","structure":"phrase"}]',
      all: '[{"op":"","term":"foo"}]',
    },
    { name: 'or-phrase',
      query: '[ { "op":"", "term":"foo" }, '+
               '{ "op":"or", "term":"bar", "structure":"phrase" }]' ,
      error: "Query too complex, phrases can only be ANDed",
    },
    { name: 'single-or-phrase',
      query: '[ { "op":"or", "term":"foo", "structure":"phrase" } ]' ,
      error: "Phrases only supported with AND",
    },
    { name: 'all-required',
      override: { allrequired: true },
      query: '[ { "op":"", "term":"foo" },' +
                  '{ "op":"and", "term":"bar" } ]' ,
      all : '[{"op":"","term":"foo"},{"op":"and","term":"bar"}]',
    },
    { name: 'all-required-fail',
      override: { allrequired: true },
      query: '[ { "op":"", "term":"foo" },' +
               '{ "op":"or", "term":"bar" } ]' ,
      error: "Did not get anything  in 'all' list, which is required",
    },
    { name: 'any-required-fail',
      override: { anyrequired: true },
      query: '[ { "op":"", "term":"foo" },' +
               '{ "op":"and", "term":"bar" } ]' ,
      error: "Did not get anything  in 'any' list, which is required",
    },
    { name: 'none-required-fail',
      override: { nonerequired: true },
      query: '[ { "op":"", "term":"foo" },' +
               '{ "op":"and", "term":"bar" } ]' ,
      error: "Did not get anything  in 'none' list, which is required",
    },
    { name: 'exact-required-fail',
      override: { exactrequired: true },
      query: '[ { "op":"", "term":"foo" },' +
               '{ "op":"and", "term":"bar" } ]' ,
      error: "Did not get anything  in 'exact' list, which is required",
    },
  ];
  logger.info("Starting unit test for Anyall");
  // fake some runtime
  var connector = new Connector();
  for ( var i=0; i<tests.length; i++) {
    var t=tests[i];
    var task = new Task( connector, "unittest" );
    //logger.debug("Test '" + t.name + "' : '" + t.query + "'");
    // for some reasun unittest won't show debug output ###
    var aa = new Anyallquery;
    aa.task = task;
    if (t.override)
      for ( var f in t.override ) {
        aa.conf[f] = t.override[f];
      }
    //task.info("Config after override: " + JSON.stringify(aa.conf) );
    try {
        qry = JSON.parse(t.query);
        jsonPathHelper.set(aa.conf.in, qry, task.data);
    } catch (e) {
      logger.error("" + e.message + " in '" + t.name +"'" );
      if ( e.fileName )
        logger.error( " in " + e.fileName + ":" + e.lineNumber );
      logger.error(t.query);
      return false;  // no need to continue further
    }

    var runok = false;
    try {
      aa.run( task );
      runok = true;
    } catch (e) {
      if ( t.error && t.error == e.message ) {
        logger.debug("test " + t.name + " failed as expected");
      } else {
        logger.error( t.name + " FAILED" );
        logger.error( "Exception! " + e.message );
        if ( t.error )
          logger.error( "Expected   " + t.error );
        if ( e.fileName )
          logger.error( " in " + e.fileName + ":" + e.lineNumber);
        return false;
      }
    }
    if ( runok ) {
      if ( ! checkresult( "all", aa, t, task ) )
        return false;
      if ( ! checkresult( "any", aa, t, task ) )
        return false;
      if ( ! checkresult( "none", aa, t, task ) )
        return false;
      if ( ! checkresult( "exact", aa, t, task ) )
        return false;
      if ( ! checkresult( "single", aa, t, task ) )
        return false;
    }
    logger.info( "Test '" + t.name + "' OK");
  } // test loop
  return true;
};

///////////////////////////
// Housekeeping

Anyallquery.prototype.getClassName = function () {
  return "Anyallquery";
};

Anyallquery.prototype.getDisplayName = function () {
  return "Any/All/None query";
};

Anyallquery.prototype.getDescription = function () {
  return "Split a listquery into any/all/none lists";
};

Anyallquery.prototype.getVersion = function () {
  return "2.0"; // 2.0 adds the required flags and one-line stuff
};

Anyallquery.prototype.renderArgs = function () {
  // TODO - make something that looks nice
  return "";
};

Anyallquery.prototype.capabilityFlagDefault = function ( flag ) {
  // We can say something about and/or/not, but that's all 
  // Can not say anything about indexes etc.
  if ( flag == "query-and" && this.conf.allsupported )
    return true;
  if ( flag == "query-or" && this.conf.anysupported )
    return true;
  if ( flag == "query-not" && this.conf.nonesupported )
    return true;
  if ( flag == "query-phrase" && this.conf.exactsupported )
    return true;
  return null;
};

Anyallquery.prototype.getUsedArgs = function () {
  if (this.conf.in && this.conf.in.path === "$.input")
    return [this.conf.in.key];
  else
    return [];
};

Anyallquery.prototype.upgrade = function (confVer, curVer, conf) {
  // can't upgrade if the connector is newer than the step
  if (confVer > curVer)
    return false;
  if ( confVer < 2.0 ) {
    this.conf.anyrequired = false;
    this.conf.allrequired = false;
    this.conf.nonerequired = false;
    this.conf.exactrequired = false;
  }
  return true;
};
