// ProxyUrl step
// Does two things:
//  1) writes a session file, storing auth, cookies, etc
//  2) rewrites all URLs in the results to point to cf-proxy, which will
//     then read the session file, re-establish the session, and proxy
//     traffic to the site, as if it was the same user who made the search.
//
// TODO - Add the connector name (and version?) to the session file too,
// as a comment. May come in handy when debugging bad session files.
//
// TODO - Once we have data types in the template, get rid of the hard-coded
// field names, and replace with an array of all template fields that are of
// type url, or have a proxy flag, or something. Keep a list of selected fields
// in the connector, instead of the hard-coded booleans.
// Also, make it configurable where the fields are to be found, do not
// hard-code $.output.resulst[*] - other templates may use different places.


var EXPORTED_SYMBOLS = ["ProxyUrl"];
Components.utils.import('resource://indexdata/runtime/Step.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/xulHelper.js');
Components.utils.import('resource://indexdata/util/configfile.js');
Components.utils.import('resource://indexdata/util/io.js');
Components.utils.import('resource://indexdata/runtime/StepError.js');
Components.utils.import('resource://indexdata/runtime/core.js');
Components.utils.import('resource://indexdata/util/jsonPathHelper.js');
Components.utils.import('resource://indexdata/util/logging.js');

var logger = logging.getLogger();

var ProxyUrl = function () {
  this.conf = {
    in: {path:"$.output.results[*]", key:"url"},
    out: {path:"$.output.results[*]", key:"url"},
    do_url: true,
    do_thumburl: true,
    do_fulltexturl: true,
    do_custom: false,
    // no permalink, it should never be proxified
    sessionvar: {path:"$.session", key:"cproxysession"},
    write_everytime: true,
    new_everytime: false,
    omit_username: false,
    omit_password: false,
    omit_proxy: false,
    omit_baseurl: false,
    omit_referer: false,
    cookiedomainexpansions: [], // array of lines like ".id.com www auth find"
    omitcookies: [], // array of lines like "ajax.googleapis.com:" or
                     // "POST somesite.com: +some[cC]ookie .*"
    skiplines: [], // array of lines like "indexdata.com .*\.google.com"
    replacements: [], // array of custom replacements: pattern replacement options
    basicauthwith : [], // array of regexps for deciding when to use HTTP basic auth
    extras: [], // array of extra lines
  };
};
ProxyUrl.prototype = new Step();
ProxyUrl.prototype.constructor = ProxyUrl;

ProxyUrl.prototype.init = function() {};

ProxyUrl.prototype.draw = function(surface) {
  this.tabs = xulHelper.tabBox( surface,
    [ "Fields", "Session", "Auth",
      "Cookie wildcards", "Omit cookies", "Skip Links",
      "Replacements",  "BasicAuth", "Extra", "Session file" ],
    { flex: 1 });
  var i = 0;
  this.drawUrlTab( this.tabs[i++] );
  this.drawSessionTab( this.tabs[i++] );
  this.drawAuthTab( this.tabs[i++] );
  this.drawWildcardTab( this.tabs[i++] );
  this.drawOmitTab( this.tabs[i++] );
  this.drawSkipTab( this.tabs[i++] );
  this.drawReplaceTab( this.tabs[i++] );
  this.drawBasicAuthTab( this.tabs[i++] );
  this.drawExtraTab( this.tabs[i++] ); 
  this.drawSesfileTab( this.tabs[i++] );
} // draw

ProxyUrl.prototype.drawUrlTab = function(surface) {
  var vb = xmlHelper.appendNode(surface, "vbox", null,
                { flex: 1, pack: "top", align: "left" } );

  var cbox = xmlHelper.appendNode(vb, "hbox", null,
                { flex: 1, pack: "baseline", align: "left" });
  xulHelper.captionField(cbox, "Proxify these", null );
  xulHelper.labelField(cbox, "in $.output.results[*]: ", null );
  xulHelper.checkbox(cbox, this, "do_url", "url" );
  xulHelper.checkbox(cbox, this, "do_thumburl", "thumburl" );
  xulHelper.checkbox(cbox, this, "do_fulltexturl", "fulltexturl" );

  xulHelper.checkbox(vb, this, "do_custom", "Proxify custom field:" );

  xulHelper.jsonPathMapField(vb, this, "in", "out",
    {path:"$.output.results[*]", key:"url"},
    {path:"$.output.results[*]", key:"url"});
}; // drawUrlTab

ProxyUrl.prototype.drawSessionTab = function(surface) {
  var vb = xmlHelper.appendNode(surface, "vbox", null,
                { flex: 1, pack: "top", align: "left" } );
  xulHelper.checkbox(vb, this, "write_everytime",
                  "Rewrite session file every time step is run" );
  xulHelper.checkbox(vb, this, "new_everytime",
                  "Create a new session number every time step is run" );
  xulHelper.jsonPathField(vb, this,  this.conf,
                  "sessionvar",  "Store session number in",
                  {path:"$.session", key:"cproxysession"} );
}; // drawSessionTab

ProxyUrl.prototype.drawAuthTab = function(surface) {
  var vb = xmlHelper.appendNode(surface, "vbox", null,
                { flex: 1, pack: "top", align: "left" } );
  xulHelper.captionField(vb, "Omit the following from the session file");
  var hb = xmlHelper.appendNode(vb, "hbox", null,
                { flex: 1, pack: "top", align: "left" } );
  xulHelper.checkbox(hb, this, "omit_username",
                  "Username" );
  xulHelper.checkbox(hb, this, "omit_password",
                  "Password" );
  xulHelper.checkbox(hb, this, "omit_proxy",
                  "Proxyip" );
  xulHelper.checkbox(hb, this, "omit_baseurl",
                  "Base URL" );
  xulHelper.checkbox(hb, this, "omit_referer",
                  "Referer" );
  var txt =
    "Normally cproxy passes the username and password in HTTP auth. " +
    "Occasionally this goes wrong, if we already have an established " +
    "session (and the cookies to prove it), and if the site gets " +
    "confused when it sees both session and HTTP auth. Here you can " +
    "disable the HTTP auth (or part of it), and/or the proxy setting." +
    "The baseUrl is used to determine when to send Basic HTTP authentication." +
    "Omitting it makes the cproxy to send authentication on every request, " +
    "which probably is not what you want.";
  var helptext = xmlHelper.appendNode(vb, "description", txt );
}; // drawAuthTab

ProxyUrl.prototype.drawWildcardTab = function(surface) {
  xmlHelper.emptyChildren(surface);
  var context = this;
  logger.debug("Wildcardtab: Starting to draw");
  var vb = xmlHelper.appendNode(surface, "vbox", null,
                { flex: 1, pack: "top", align: "left" } );
  var tbox = xmlHelper.appendNode(vb, "hbox");
  xulHelper.captionField(tbox, "Domain", {width:220} );
  xulHelper.captionField(tbox, "Wildcard expansion" );
  if ( ! this.conf.cookiedomainexpansions ||
         this.conf.cookiedomainexpansions.length == 0  ) {
    this.conf.cookiedomainexpansions = [ ];
  }

  // Helper to draw one line
  function drawline( line, index ) {
    var lbox = xmlHelper.appendNode(vb, "hbox", null );
    var pos = line.indexOf(" ");
    if ( ! line )
      pos = 0;
    var domain = line.substr(0,pos);
    var list = line.substr(pos+1); // the rest
    xulHelper.captionField(lbox, domain, {width:220});
    xulHelper.labelField(lbox, list, { flex:1, width:"600" });
    var delBut= xmlHelper.appendNode(lbox, "button", null,
        {"label": "Delete"});
    delBut.addEventListener("command", function (e) {
        logger.debug("Wildcardtab: delbut on " + index );
        context.conf.cookiedomainexpansions.splice(index,1);
        context.drawWildcardTab(surface);
      }, false);
  } // drawline of drawWildcardTab
  
  for ( var i = 0; i < this.conf.cookiedomainexpansions.length; i++ ) {
    drawline ( this.conf.cookiedomainexpansions[i], i );
  }
  var inpbox = xmlHelper.appendNode(vb, "hbox", null );
  var domainInput = xulHelper.inputFieldOnly(inpbox, this, "", {width:220}) 
  var listInput = xulHelper.inputFieldOnly(inpbox, this, "", {flex:1,size:"100"})
  var addBut = xmlHelper.appendNode(inpbox, "button", null,
        {"label": "Add"});
  addBut.addEventListener("command", function (e) {
      if ( domainInput.value == "" )
        return; // will not add empties
      if ( listInput.value == "" )
        return;
      var s = domainInput.value + " " + listInput.value;
      if ( s.substr(0,1) != "." )
        s = "." + s;
      context.conf.cookiedomainexpansions.push( s );
      context.drawWildcardTab(surface);
    }, false);
  
  var txt =
    "Directs the cproxy to expand wildcard cookies to given subdomains. " +
    "For example, cookies set to .indexdata.com could be expanded to " +
    "www.indexdata.com, auth.indexdata.com, and search.indexdata.com. " +
    "This is done by setting the domain to \".indexdata.com\", and the " +
    "expansions to \"www auth search\". Special expansions include \"-\" " +
    "to disable built-in expansions, \"*\" to expand to the whole session, " +
    "and \".\" to expand to the current page. In most cases the cproxy " +
    "has good enough defaults, and all this can be left empty. ";
  var helptext = xmlHelper.appendNode(vb, "description", txt );
  logger.debug("Wildcardtab: All done");
}; // drawWildcardTab


ProxyUrl.prototype.drawOmitTab = function(surface) {
  xmlHelper.emptyChildren(surface);
  var context = this;
  logger.debug("OmitTab: Starting to draw");
  var vb = xmlHelper.appendNode(surface, "vbox", null,
                { flex: 1, pack: "top", align: "left" } );
  var tbox = xmlHelper.appendNode(vb, "hbox");
  xulHelper.captionField(tbox, "Method", {width:120} );
  xulHelper.captionField(tbox, "Domain", {width:220} );
  xulHelper.captionField(tbox, "Cookie names" );
  if ( ! this.conf.omitcookies ||
         this.conf.omitcookies.length == 0  ) {
    this.conf.omitcookies = [ ];
  }

  // Helper to draw one line
  function drawline( line, index ) {
    var lbox = xmlHelper.appendNode(vb, "hbox", null );
    var pos = line.indexOf(" ");
    if ( ! line )
      pos = 0;
    var method = "";
    var domain = line.substr(0,pos);
    var list = line.substr(pos+1);
    pos = domain.indexOf("\\s");
    if ( pos > -1 ) {
      method = domain.substr(0,pos);
      domain = domain.substr(pos+2);
    }
    logger.debug("OmitTab: Split omit line '" + line + "' at " + pos +
      " into '" + method + "' and '" + domain + "' and '" + list + "'" );
    xulHelper.captionField(lbox, method, {width:120});
    xulHelper.captionField(lbox, domain, {width:220});
    xulHelper.labelField(lbox, list, { flex:1, width:"600" });
    var delBut= xmlHelper.appendNode(lbox, "button", null,
        {"label": "Delete"});
    delBut.addEventListener("command", function (e) {
        logger.debug("OmitTab: delbut on " + index );
        context.conf.omitcookies.splice(index,1);
        context.drawOmitTab(surface);
      }, false);
  } // drawline
  
  for ( var i = 0; i < this.conf.omitcookies.length; i++ )
    drawline ( this.conf.omitcookies[i], i );
  var inpbox = xmlHelper.appendNode(vb, "hbox", null );
  var methodInput = xulHelper.inputFieldOnly(inpbox, this, "", {width:120})
  var domainInput = xulHelper.inputFieldOnly(inpbox, this, "", {width:220})
  var listInput = xulHelper.inputFieldOnly(inpbox, this, "", {flex:1,size:"100"})
  var addBut = xmlHelper.appendNode(inpbox, "button", null,
        {"label": "Add"});
  addBut.addEventListener("command", function (e) {
      logger.debug("OmitTab: Add button");
      if ( domainInput.value == "" ) 
        return; // will not add empties
      var s = methodInput.value ;
      if (s)
        s += "\\s";
      s += domainInput.value + " " + listInput.value;
      logger.debug("OmitTab: Add: '" + s + "'");
      context.conf.omitcookies.push( s );
      context.drawOmitTab(surface);
    }, false);
  var txt =
    "Directs the cproxy to omit (some) cookies to given domains, " +
    "optionally only for GET or POST requests. The domain name must be a " +
    "regular expression, and the list of cookie names is optional, defaulting " +
    "to all of them. Each name is a regular expression. " +
    "Normally none of this should not be necessary.";
  var helptext = xmlHelper.appendNode(vb, "description", txt );
  logger.debug("OmitTab: All done");
}; // drawOmitTab

ProxyUrl.prototype.drawSkipTab = function(surface) {
  xmlHelper.emptyChildren(surface);
  var context = this;
  logger.debug("SkipTab: Starting to draw");
  var vb = xmlHelper.appendNode(surface, "vbox", null,
                { flex: 1, pack: "top", align: "left" } );
  var tbox = xmlHelper.appendNode(vb, "hbox");
  xulHelper.captionField(tbox, "Site", {width:220} );
  xulHelper.captionField(tbox, "Link" );
  if ( ! this.conf.skiplines ||
         this.conf.skiplines.length == 0  ) {
    this.conf.skiplines = [ ];
  }

  // Helper to draw one line
  function drawline( line, index ) {

    var lbox = xmlHelper.appendNode(vb, "hbox", null );
    var pos = line.indexOf(" ");
    if ( ! line )
      pos = 0;
    var site = line.substr(0,pos);
    var link = line.substr(pos+1);
    logger.debug("SkipTab: Split skip line '" + line + "' at " + pos +
      " into '" + site + "' and '" + link + "'" );
    xulHelper.captionField(lbox, site, {width:220});
    xulHelper.labelField(lbox, link, { flex:1, width:"600" });
    var delBut= xmlHelper.appendNode(lbox, "button", null,
        {"label": "Delete"});
    delBut.addEventListener("command", function (e) {
        logger.debug("SkipTab: delbut on " + index );
        context.conf.skiplines.splice(index,1);
        context.drawSkipTab(surface);
      }, false);
  } // drawline

  for ( var i = 0; i < this.conf.skiplines.length; i++ ) 
    drawline ( this.conf.skiplines[i], i );
  var inpbox = xmlHelper.appendNode(vb, "hbox", null );
  var siteInput = xulHelper.inputFieldOnly(inpbox, this, "", {width:220})
  var linkInput = xulHelper.inputFieldOnly(inpbox, this, "", {flex:1,size:"100"})
  var addBut = xmlHelper.appendNode(inpbox, "button", null,
        {"label": "Add"});
  addBut.addEventListener("command", function (e) {
      logger.debug("SkipTab: Add button");
      if ( siteInput.value == "" && linkInput.value == "")
        return; // will not add empties
      if ( siteInput.value == "" )
        siteInput.value = ".*";  // regexp to match everything
      if ( linkInput.value == "" )
        linkInput.value = ".*";  // regexp to match everything
      var s = siteInput.value + " " + linkInput.value;
      logger.debug("SkipTab: Add: '" + s + "'");
      context.conf.skiplines.push( s );
      context.drawSkipTab(surface);
    }, false);
  var txt =
    "Directs the cproxy not to proxify some links. " +
    "The site is a regular expression to define on which pages these rules " +
    "apply. The link is also a regular expression specifying which links " +
    "not to proxify. Either may be empty defaulting to all sites or all " +
    "links, but obviously one should be there. This is a special operation " +
    "that should almost never be used";
  var helptext = xmlHelper.appendNode(vb, "description", txt );
  logger.debug("SkipTab: All done");
} // drawSkipTab


// Note that the display fields are in a different order than
// the elements in the line, because the replacement may contain
// spaces, it must be the last one.
ProxyUrl.prototype.drawReplaceTab = function(surface) {
  xmlHelper.emptyChildren(surface);
  var context = this;
  logger.debug("ReplaceTab: Starting to draw");
  var vb = xmlHelper.appendNode(surface, "vbox", null,
                                   { flex: 1, pack: "top", align: "left" } );
  var tbox = xmlHelper.appendNode(vb, "hbox");
  xulHelper.captionField(tbox, "Pattern", {width:600, flex:1} );
  xulHelper.captionField(tbox, "Replacement", {width:600, flex:1} );
  xulHelper.captionField(tbox, "Options", {width:600, flex:1 } );
  xulHelper.captionField(tbox, "Content-type", {width:600, flex:1} );
  var dummybut = xmlHelper.appendNode(tbox, "button", null, // dummy filler
                                      {"label": "X", // to keep them flex in sync
                              "style": "visibility: hidden;"});
  if ( ! this.conf.replacements ||
    this.conf.replacements.length == 0  ) {
    this.conf.replacements = [ ];
    }
    
  // Helper to draw one line
  function drawline( line, index ) {
    var lbox = xmlHelper.appendNode(vb, "hbox", null );
    var parts = line.split(" ");
    var patt = parts.shift();
    var opt = parts.shift();
    var cont = parts.shift();
    var repl = parts.join(" ");
    logger.debug("ReplaceTab: Split repl line '" + line + "' " +
      "into patt='" + patt + "' opt='" + opt + "' cont='" + cont + "' "+
      "repl='" + repl + "'" );
    //xulHelper.labelField(lbox, patt, {width:220, flex:1});
    //xulHelper.labelField(lbox, repl, {width:220, flex:1});
    xulHelper.labelField(lbox, patt, {width:600, flex:1});
    xulHelper.labelField(lbox, repl, {width:600, flex:1});
    xulHelper.captionField(lbox, opt, {width:600, flex:1});
    xulHelper.labelField(lbox, cont, {width:600, flex:1});
    var delBut= xmlHelper.appendNode(lbox, "button", null,
                                      {"label": "Delete"});
    delBut.addEventListener("command", function (e) {
      logger.debug("ReplaceTab: delbut on " + index );
      context.conf.replacements.splice(index,1);
      context.drawReplaceTab(surface);
    }, false);
  } // drawline
    
  for ( var i = 0; i < this.conf.replacements.length; i++ ) 
    drawline ( this.conf.replacements[i], i );
  var inpbox = xmlHelper.appendNode(vb, "hbox", null );
  var pattInput = xulHelper.inputFieldOnly(inpbox, this, "", {size:100, flex:1})
  var replInput = xulHelper.inputFieldOnly(inpbox, this, "", {size:100, flex:1})
  var optInput = xulHelper.inputFieldOnly(inpbox, this, "", {size:100, flex:1} );
  var contInput = xulHelper.inputFieldOnly(inpbox, this, "", {size:100, flex:1} );
  var addBut = xmlHelper.appendNode(inpbox, "button", null,
                                    {"label": "Add"});
  addBut.addEventListener("command", function (e) {
    logger.debug("ReplaceTab: Add button");
    if ( pattInput.value == "" )
      return; // will not add empties
    if ( optInput.value == "" )
      optInput.value = ".";
    if ( contInput.value == "" )
      contInput.value = ".*";  // regexp to match everything
    var s = pattInput.value + " " + optInput.value + " " +
      contInput.value + " " + replInput.value;
    logger.debug("ReplaceTab: Add: '" + s + "'");
    context.conf.replacements.push( s );
    context.drawReplaceTab(surface);
  }, false);
  var txt =
    "Directs the cproxy to replace any string pattern in the content " +
    "of a page. This is a tool of last resort, and should be used very " +
    "sparingly!. You can write any regular expression, use ()'s to catch " +
    "items, and use $1 etc in the replacement string. The content-type can " +
    "limit this to only pages of given, type, for example text/html. The " +
    "options currently only support 'i' for ignoring the case, and a dot " +
    "to indicate no options."
  var helptext = xmlHelper.appendNode(vb, "description", txt );
  logger.debug("ReplaceTab: All done");
} // drawReplaceTab
    
ProxyUrl.prototype.drawBasicAuthTab = function(surface) {
  xmlHelper.emptyChildren(surface);
  var context = this;
  logger.debug("drawBasicAuthTab: Starting to draw");
  var vb = xmlHelper.appendNode(surface, "vbox", null,
                                   { flex: 1, pack: "top", align: "left" } );
  var tbox = xmlHelper.appendNode(vb, "hbox");
  xulHelper.captionField(tbox, "Site", {width:600, flex:1} );
  var dummybut = xmlHelper.appendNode(tbox, "button", null, // dummy filler
                                      {"label": "X", // to keep them flex in sync
                              "style": "visibility: hidden;"});
  if ( ! this.conf.basicauthwith ||
    this.conf.basicauthwith.length == 0  ) {
    this.conf.basicauthwith = [ ];
    }

  // Helper to draw one line
  function drawline( line, index ) {
    var lbox = xmlHelper.appendNode(vb, "hbox", null );
    xulHelper.labelField(lbox, line, {width:600, flex:1});
    var delBut= xmlHelper.appendNode(lbox, "button", null,
                                      {"label": "Delete"});
    delBut.addEventListener("command", function (e) {
      logger.debug("BasicAuthTab: delbut on " + index );
      context.conf.basicauthwith.splice(index,1);
      context.drawBasicAuthTab(surface);
    }, false);
  } // drawline

  for ( var i = 0; i < this.conf.basicauthwith.length; i++ )
    drawline ( this.conf.basicauthwith[i], i );
  var inpbox = xmlHelper.appendNode(vb, "hbox", null );
  var lineInput = xulHelper.inputFieldOnly(inpbox, this, "", {size:100, flex:1})
  var addBut = xmlHelper.appendNode(inpbox, "button", null,
                                    {"label": "Add"});
  addBut.addEventListener("command", function (e) {
    logger.debug("BasicAuthTab: Add button");
    if ( lineInput.value == "" )
      return; // will not add empties
    var s = lineInput.value;
    logger.debug("BasicAuthTab: Add: '" + s + "'");
    context.conf.basicauthwith.push( s );
    context.drawBasicAuthTab(surface);
  }, false);
  var txt =
    "Regular expressions that will be matched against the hostname of a proxified "+
    "link. If any match, (and we have a username), then the HTTP Authentication "+
    "is added to the request. If nothing specified, the decision is based on the "+
    "BaseUrl line instead, which is what you want in most cases.";
  var helptext = xmlHelper.appendNode(vb, "description", txt );
  logger.debug("BasicAuthTab: All done");
} // drawBasicAuthTab

ProxyUrl.prototype.drawExtraTab = function(surface) {
  xmlHelper.emptyChildren(surface);
  var context = this;
  logger.debug("ExtraTab: Starting to draw");
  var vb = xmlHelper.appendNode(surface, "vbox", null,
                { flex: 1, pack: "top", align: "left" } );
  var tbox = xmlHelper.appendNode(vb, "hbox");
  xulHelper.captionField(tbox, "Tag", {width:220} );
  xulHelper.captionField(tbox, "Line" );
  if ( ! this.conf.extras ||
         this.conf.extras.length == 0  ) {
    this.conf.extras = [ ];
  }

  // Helper to draw one line
  function drawline( line, index ) {
    var lbox = xmlHelper.appendNode(vb, "hbox", null );
    var pos = line.indexOf(" ");
    if ( ! line )
      pos = 0;
    var tag = line.substr(0,pos);
    var rest = line.substr(pos+1); // the rest
    xulHelper.captionField(lbox, tag, {width:220});
    xulHelper.labelField(lbox, rest, { flex:1, width:"600" });
    var delBut= xmlHelper.appendNode(lbox, "button", null,
        {"label": "Delete"});
    delBut.addEventListener("command", function (e) {
        logger.debug("ExtraTab: delbut on " + index );
        context.conf.extras.splice(index,1);
        context.drawExtraTab(surface);
      }, false);
  } // drawline
  
  for ( var i = 0; i < this.conf.extras.length; i++ ) 
    drawline ( this.conf.extras[i], i );
  var inpbox = xmlHelper.appendNode(vb, "hbox", null );
  var tagInput = xulHelper.inputFieldOnly(inpbox, this, "", {width:220})
  var restInput = xulHelper.inputFieldOnly(inpbox, this, "", {flex:1,size:"100"})
  var addBut = xmlHelper.appendNode(inpbox, "button", null,
        {"label": "Add"});
  addBut.addEventListener("command", function (e) {
      if ( tagInput.value == "" )
        return; // will not add empties
      if ( restInput.value == "" )
        return;
      var s = tagInput.value + " " + restInput.value;
      context.conf.extras.push( s );
      context.drawExtraTab(surface);
    }, false);
  var txt =
    "Here you can add extra lines into the session file. Normally you " +
    "would not want to do anything like that, but in some rare cases, " +
    "when you know there will be a new cproxy running on a site, you can " +
    "add configuration options here that are not (yet) known in the builder. " +
    "Let me repeat, you should never use this, unless you really know what " +
    "you are doing. ";
  var helptext = xmlHelper.appendNode(vb, "description", txt );
  logger.debug("ExtraTab: All done");
}; // drawExtraTab

ProxyUrl.prototype.drawSesfileTab = function(surface) {
  xmlHelper.emptyChildren(surface);
  var context = this;
  logger.debug("SesfileTab: Starting to draw");
  var initsession = ""; // session from init task args, if any

  // This init-task trickery is the same as in the run method below.
  // dirty, but seems to work... Less relevant in the builder, anyway.
  var init = this.task.connector.findTask("init");
  if ( init ) {
    initsession = init.getArgValue("cproxysession");
  }
  var storedsession = jsonPathHelper.getFirst( this.conf.sessionvar, this.task.data);
  var session = this.getsessionnumber(this.task, initsession, storedsession);
  var sf = this.sessioncontent(this.task, session, init);
  var vb = xmlHelper.appendNode(surface, "vbox", null,
                         { flex: 1, pack: "top", align: "left" } );
  var lines = sf.split("\n");
  for ( var i = 0; i< lines.length; i++ )
    xulHelper.labelField(vb, lines[i], {} );
  logger.debug("SesfileTab: Done drawing");
} // drawSesfileTab


////////////////
// Runtime stuff

// Dump all cookies into a string, to be saved in the session file
function dumpcookies(){
    var cookieMgr = Components.classes["@mozilla.org/cookiemanager;1"]
             .getService(Components.interfaces.nsICookieManager);
    var c="";
    for (var e = cookieMgr.enumerator; e.hasMoreElements();) {
        var cookie = e.getNext().QueryInterface(Components.interfaces.nsICookie);
        // Build a (new-format) cookie line, used in cf-proxy 3.0 and onwards
        var scl = "Set-Cookie "; // headers use a ':', session files don't
        scl += cookie.name + "=" + cookie.value + ";";
          // name=value is always the first
        if ( cookie.host ) {
            scl += "Domain=" + cookie.host + ";";
        }
        if ( cookie.path ) {
            scl += "Path=" + cookie.path + ";";
        }
        if ( cookie.expires ) {
            var ex = new Date( cookie.expires * 1000 );
              // cookie time in seconds, date takes milliseconds since 01-01-1970
            scl += "Expires=" + ex.toString() + ";";
        }
        if ( cookie.isSecure ) {
            scl += "Secure;" ;
        }
        //logger.debug(scl);
        c += scl + "\n";
        // note that the line always ends in a semicolon. Makes parsing easier
        // see http://developer.mozilla.org/en/docs/nsICookie for a list
        // of attributes on cookie
        // See also http://tools.ietf.org/html/rfc6265 for detailed instructions
        // on parsing and generating cookies.
    }
    return c;
} // dumpcookies

// Helper to dump various versions and other debug info into a string
// and from there to the session file. These are not really needed by
// the cproxy (yet?), but will be helpful when debugging future problems.
ProxyUrl.prototype.versions = function() {
  var c = "";
  c += "CfVersion " + core.version + "\n";
  c += "CfConnector " + core.lastconnector + "\n";   
  c += "CfStep " + this.getClassName() + " " + this.getVersion() +
       " " + this.task.name + "\n";
  return c;
} // versions


// Helper to dump all lines of a given type into a string
function seslines( tag, lines ) {
  if ( ! lines )
    return "";
  var c = "";
  for ( var i = 0; i < lines.length; i++ ) {
    var cl="";
    if ( tag != "" )
      cl = tag + " ";
    cl += lines[i];
    logger.debug(cl);
    c += cl + "\n";
  }
  return c;
} // seslines


// Decide on the session number
//  - If we have written a session before
//     - if new_everytime, create a number
//     - else use the existing
//        - this is the only case when we may not need to write the session file
//  - Else (we have not done a session before)
//     - take from init params if possible
//     - else create a random one

ProxyUrl.prototype.getsessionnumber = function(task, init, stored) {
  if ( stored ) {  // we have a stored session, we have written a session file
    if ( ! this.conf.new_everytime ) { // and may use it
      task.debug("Using stored session number '" + stored + "'" );
      return stored;
    }
  } else { // No stored session, we are running for the first time
    if (init) { // Init arg gives the session. Store for future use
      task.debug("Using session number from init args '" + init + "'" );
      jsonPathHelper.set( this.conf.sessionvar, init, task.data);
      return init;
    }
  }
  // If we get this far, we need to create a random session number
  var newsession = Math.floor(Math.random()*1000000);
  jsonPathHelper.set( this.conf.sessionvar, newsession, task.data);
  task.debug("Created a new session number '" + newsession + "'" );
  return newsession;
}; // getsessionnumber

// Generate the content for the session file
ProxyUrl.prototype.sessioncontent = function (task, session, init) {
  var doc = this.getPageDoc();
  var c = "";
  c += "CfSession " + session + "\n";
  c += this.versions();
  if ( !this.conf.omit_referer )
    c += "Referer " + doc.location + "\n";
  if ( !this.conf.omit_baseurl )
    c += "Baseurl " + doc.location + "\n";
  if ( init ) {
    if ( !this.conf.omit_username )
      c += "Username " + init.getArgValue("username") + "\n";
    if ( !this.conf.omit_password )
      c += "Password " + init.getArgValue("password") + "\n";
    if ( !this.conf.omit_proxy )
      c += "Proxyip " + init.getArgValue("proxyip") + "\n";
  }
  c += seslines("CookieDomainExpansion", this.conf.cookiedomainexpansions );
  c += seslines("OmitCookie", this.conf.omitcookies );
  c += seslines("SkipLink", this.conf.skiplines );  
  c += seslines("Custom-Replacement", this.conf.replacements );
  c += seslines("Basicauthwith", this.conf.basicauthwith );
  c += seslines("", this.conf.extras );
  c += dumpcookies();
  return c;
} // sessioncontent

ProxyUrl.prototype.writesessionfile= function (task, session, content) {
  var sesdir = ConfigFile.getconfigvalue(task, "sessiondir");
  if ( !sesdir ) {
    // Do not throw an error, breaks the builder if no access to /etc
    // or no cfg file
    task.warn("Can not store session file " + session + ". " +
        "No 'sessiondir' found in config file. " +
        "Install cf-proxy?" );
    return; 
  }
  try {
  var sesfilename = sesdir + "/cf." + session ;
  var sesfile = Components.classes["@mozilla.org/file/local;1"].
      createInstance(Components.interfaces.nsILocalFile);
      sesfile.initWithPath(sesfilename);
  var fos = io.fos(sesfile, "" );  // file output stream
  io.write(fos,content);  // writes the string and closes fos!
  // If these throw exceptions, we are configured badly, and en
  // exception is all right
  task.info("ProxyURL session file stored at "+sesfile.path);
  } catch (e) {
    logger.error("Storing session file " + sesfilename + " failed! " +
      e.name + " " + e.message );
    if ( e.fileName && e.lineNumber )
      logger.error("  in " + e.fileName + ":" +  e.lineNumber );
    e.message = "Storing session file " + sesfilename + " failed: " +
        e.message;
    throw(e);
  }
}; // writesessionfile


ProxyUrl.prototype.proxifyUrls = function (task, session) {
  var proxyhost = ConfigFile.getconfigvalue(task, "proxyhostname");
  if (!proxyhost) {
      // Can not throw an error - breaks builders
      // if they don't have a config file
      proxyhost="SOME.UNKNOWN.PROXY.HOST";
  }

  // Callback helper to rewrite a single URL
  var makeproxyurl = function (val) {
    if (val === undefined) return undefined;
    if ( val.indexOf(proxyhost) ==-1) { // avoid duplicate proxying
        // duplicates may happen in the builder, if rerunning the step
        var prefix =  "http://" + proxyhost + "/" + session + "/"; 
        val = val.replace( /^http:\/\//, prefix );
    }
    return val;
  }; // makeproxyurl callback

  if (this.conf.do_url)
    jsonPathHelper.mapElements(
      {path:"$.output.results[*]", key:"url"},
      {path:"$.output.results[*]", key:"url"},
      makeproxyurl, task.data);
  if (this.conf.do_thumburl)
    jsonPathHelper.mapElements(
      {path:"$.output.results[*]", key:"thumburl"},
      {path:"$.output.results[*]", key:"thumburl"},
      makeproxyurl, task.data);
  if (this.conf.do_fulltexturl)
    jsonPathHelper.mapElements(
      {path:"$.output.results[*]", key:"fulltexturl"},
      {path:"$.output.results[*]", key:"fulltexturl"},
      makeproxyurl, task.data);
  if (this.conf.do_custom)
    jsonPathHelper.mapElements(
      this.conf.in, this.conf.out,
      makeproxyurl, task.data);
    
}; // proxifyUrls


ProxyUrl.prototype.run = function (task) {
  var initsession = ""; // session from init task args, if any
  // TODO - Check if we have the arguments on the connector level,
  // (in $.session) and in that case, don't bother with the init task
  // But careful with this, there are connectors that clean up
  // username/password in init args to disable http auth in cproxy

  // Get the arguments for the init task
  // Admittedly dirty, but we need to know them...
  // Seems not to work in the builder
  var init = task.connector.findTask("init");
  // TODO - Ought to pass the arguments we want
  // but in real life, we have at most one init task...
  if ( init ) {
    var nocproxy =  init.getArgValue("nocproxy");
    task.debug("got nocproxy='" + nocproxy + "' t=" + typeof(nocproxy) );
    if ( nocproxy ) {
      task.info("nocproxy specified, skipping proxying" );
      return;
    }
    initsession = init.getArgValue("cproxysession");
  }
  var storedsession = jsonPathHelper.getFirst( this.conf.sessionvar, task.data);
  var session = this.getsessionnumber(task, initsession, storedsession);
  var needtowrite = this.conf.write_everytime ||  // request to write
     ( session != storedsession ); // or a new file, need to write that too
  if ( needtowrite ) {
    var c = this.sessioncontent(task, session, init);
    this.writesessionfile(task, session, c );
  } else {
    task.info("session already written, no need to write again");
  }

  this.proxifyUrls(task, session);

};

// This is a bit dirty - it lists the arguments it takes from an init task.
// If there is a proxyUrl step in the init task, the init task will match when
// invoked with the usual arguments. But if there is no proxyUrl step in the
// init task, and there is a init task, things may be a bit strange. Still,
// this should only increase the chances that a task matches, freeing us of
// the need of (at least some of the) null transform steps...
ProxyUrl.prototype.getUsedArgs = function () {
  return [ "cproxysession", "username", "password", "proxyip", "nocproxy" ];
};

ProxyUrl.prototype.getClassName = function () {
  return "ProxyUrl";
};

ProxyUrl.prototype.getDisplayName = function () {
  return "Proxy URL";
};

ProxyUrl.prototype.getDescription = function () {
  return "Changes the URL into one that will be proxied by the CfProxy";
};

ProxyUrl.prototype.getVersion = function () {
  //return "1.1";  // 1.1 adds multiple URL-like fields
  //return "1.2";  // 1.2 adds $sessio.cproxysession and checkboxes for rewriting
                 // also omit_auth checkboxes
  //return "1.3"; // Adds cookiedomainexpansions, etc
  return "1.4"; // Adds baseurl and basicauthwith
};

ProxyUrl.prototype.renderArgs = function () {
  var s="";
  if (this.conf.do_url)
    s += "url, ";
  if (this.conf.do_thumburl)
    s += "thumburl, ";
  if (this.conf.do_fulltexturl)
    s += "fulltexturl, ";
  if ( this.conf.do_custom &&
       this.conf.in && this.conf.in.key &&
       this.conf.out && this.conf.out.key ) {
    s += this.conf.in.key
    if ( this.conf.in.key != this.conf.out.key )
      s += "->" + this.conf.out.key
  }
  s = s.replace( /,\s*$/, ""); // remove last comma
  return s;
};

ProxyUrl.prototype.upgrade = function (confVer, curVer, conf) {
  // can't upgrade if the connector is newer than the step
  if (confVer > curVer)
    return false;

  if (confVer < 0.2) {
    jsonPathHelper.upgradePostProc(this.conf);
  }
  if (confVer < 1.1 ) {
    if ( typeof(conf.in) == "undefined" ||
         typeof(conf.in.path) == "undefined" ||
         ( conf.in.path == "$.output.results[*]" &&
           conf.in.key == "url" ) ) {  // the usual case
      conf.do_url = true;
      conf.do_thumburl = false;
      conf.do_fulltexturl = false;
      conf.do_custom = false;
      logger.debug("Upgraded proxy_url from " + confVer + ", the usual case");
    } else { // custom field configured
      conf.do_url = false;
      conf.do_thumburl = false;
      conf.do_fulltexturl = false;
      conf.do_custom = true;
      logger.debug("Upgraded proxy_url from " + confVer + ", special conf");
    }
  }
  if ( confVer < 1.2 ) {
    conf.sessionvar = {path:"$.session", key:"cproxysession"};
    conf.write_everytime = true;
    conf.new_everytime = false;
    logger.debug("Upgraded proxy_url from " + confVer + " default cproxysession var");
    // the various omit_ flags default to false, no need to set them explicitly
  }
  if ( confVer < 1.3 ) {
    conf.cookiedomainexpansions = [];
    conf.omitcookies = [];
    conf.skiplines = [];
    conf.replacements = [];
    conf.extras = [];
    logger.debug("Upgraded extra parameters from confVer to empty defaults");
  }
  if ( confVer < 1.4 ) {
    conf.basicauthwith = [];
    conf.omit_baseurl = true; // default to the old behavior, even though it
                             // doesn't make that much sense.
    conf.omit_referer = false;
  }
  return true;
};
