package Masterkey::Admin::Config;

use strict;
use warnings;
use Scalar::Util;
use IO::File;
use Errno qw(ENOENT);

use Masterkey::Admin::Profile;
use Masterkey::Admin::LogLevel;
use IO::String;


# If $tab is undefined, then this is the configuration for the whole
# site rather than one of the tab configurations: use the name "SITE"
# and do not try to compile a profile.
#
sub new {
    my $class = shift();
    my($admin, $tab, @path) = @_;

    my $this = bless {
	admin => $admin,
	path => \@path,
	values => {},
	profile => undef,
    }, $class;
    Scalar::Util::weaken($this->{admin});

    my %sym = (
	thisHost => $ENV{SERVER_NAME},
    );
    $this->_parse((defined $tab ? $tab : "SITE") . ".config", 0, \%sym);
    if (defined $tab) {
	my $type = $this->value("type");
	if (defined $type) {
	    $this->{profile} = new Masterkey::Admin::Profile($this, $type);
	}
    }

    return $this;
}

sub _parse {
    my $this = shift();
    my($name, $level, $sym) = @_;

    (my $fh, $level) = $this->_open_file($name, $level);
    my $continuation = 0;
    my $key = undef;
    my $value = "";
    while (my $line = $fh->getline()) {
	chomp($line);
	$line =~ s/#.*//;
	$line =~ s/\s+$//;
	$line =~ s/^\s+//;
	next if $line eq "";
	if ($continuation) {
	    $value .= $line;
	} elsif ($line =~ /^%set\s+(.*?)=(.*)/) {
	    my($name, $v2) = ($1, $2); # Save old values of $1 and $2
	    $sym->{$name} = $this->_substitute($v2, $sym);
	} elsif ($line =~ s/^%include\s+//) {
	    $this->_parse($this->_substitute($line, $sym), $level, $sym);
	} else {
	    $this->_set_value($key, $value, $sym);
	    ($key, $value) = ($line =~ /(.*?):\s*(.*)/);
	}
	if (defined $value && $value =~ s/\\$//) {
	    $continuation = 1;
	} else {
	    $continuation = 0;
	}
    }

    $this->_set_value($key, $value, $sym);
    $fh->close();
    return $this;
}


# Opens the named file, searching in the path starting at the
# specified level, returning both the opened filehandle and the level
# at which the file was found.  Dies if the file is absent from all
# directories in the path, or of another error occurs (e.g. a file is
# found but unreadable) -- so the result need not be checked.
#
sub _open_file {
    my $this = shift();
    my($name, $level, $isValueFile) = @_;
#    use Carp; confess "_open_file($name)" if $name =~ /^\//;

    if ($name =~ /^http:\/\//) {
	return $this->_open_http($name, $level, $isValueFile);
    }

    my $i = 0;
    if ($name =~ s/^<(.*)>$/$1/) {
	# Start looking for the file one level below current
	$i = $level+1;
    }

    my $file;
    while ($i < @{ $this->{path} }) {
	my $dir = $this->{path}->[$i];
	$file = "$dir/$name";
	my $fh = new IO::File("<$file");
	if (defined $fh) {
	    $this->{admin}->log(Masterkey::Admin::LogLevel::CONFIG,
				"opened '$name' at level $i ($dir)");
	    return ($fh, $i);
	}
	last if $! != ENOENT;
	$this->{admin}->log(Masterkey::Admin::LogLevel::CONFIG,
			    "no file '$name' at level $i ($dir)");
	$i++;
    }

    die("can't open ", ($isValueFile ? "value" : "configuration"),
	" file '$file' $!");
}


sub _set_value {
    my $this = shift();
    my($key, $value, $sym) = @_;

    if (!defined $key) {
	return;
    } elsif ($value =~ s/^%include\s+//) {
	(my $fh) = $this->_open_file($value, 0, 1);
	my $tmp = "";
	while (my $t2 = $fh->getline()) {
	    $tmp .= $t2;
	}
	$fh->close();
	$this->{values}->{$key} = $tmp;
    } else {
	$value = $this->_substitute($value, $sym);
	undef $value if $value eq "";
	#warn "$key=$value";
	$this->{values}->{$key} = $value;
    }
}

sub _substitute {
    my $this = shift();
    my($tail, $sym) = @_;

    my $res = "";
    while ($tail =~ s/(.*?)\$\{(.*?)\}// ||
	   $tail =~ s/(.*?)\$\((.*?)\)//) {
	my($prefix, $name) = ($1, $2);
	if ($prefix =~ /\$$/) {
	    # Sequence was double-dollar quoted, as in $${foo}: do not expand
	    $res .= $prefix . "{$name}";
	} else {
	    my $val = $sym->{$name};
	    $val = "" if !defined $val;
	    $res .= $prefix . $val;
	}
    }

    return $res . $tail;
}

sub configKeys {
    my $this = shift();
    my($prefix) = @_;

    my $len = length($prefix);
    my @keys;
    foreach my $key (sort keys %{ $this->{values} }) {
	push @keys, $key
	    if substr($key, 0, $len) eq $prefix;
    }
    return @keys;
}

sub value {
    my $this = shift();
    my($key) = @_;

    return $this->{values}->{$key};
}

sub profile { shift()->{profile} }


# This is inefficient: we should considing caching the result
sub sectionHasVisibleFields {
    my $this = shift();
    my($firstFieldInSection) = @_;

    my @fields = $this->profile()->fields();
    my $i = 0;
    #warn "\nsearching for visible fields starting from '$firstFieldInSection'";

    # First, find the nominated field within the profile
    while ($i < @fields) {
	my $field = $fields[$i]->[0];
	#warn "searching for start at field '$field'";
	last if $field eq $firstFieldInSection;
	$i++;
    }
    die "sectionHasVisibleFields() couldn't find field '$firstFieldInSection'" if $i == @fields;
    #warn "found start field '$firstFieldInSection' at $i";

    my $oldI = $i;
    while ($i < @fields) {
	my $field = $fields[$i]->[0];
	if ($i > $oldI && $this->value("field.$field.startSection")) {
	    #warn "found new section stating with '$field'";
	    return 0;
	}
	if ($this->value("field.$field.visible")) {
	    #warn "found visible field '$field'";
	    return 1;
	}
	$i++;
    }

    #warn "fell off end of field-list";
    return 0;
}


use LWP;

sub _open_http {
    my $this = shift();
    my($url, $level, $isValueFile) = @_;

    my $admin = $this->{admin};
    my $ua = $this->{ua};
    if (!defined $ua) {
	$ua = new LWP::UserAgent();
	$this->{ua} = $ua;
    }

    ### Could cache previously loaded files, as HTTP is relatively expensive

    my $res = $ua->get($url);
    die("can't open ", ($isValueFile ? "value" : "configuration"),
	" URL '$url' ", $res->status_line())
	if !$res->is_success();

    my $text = $res->content();

    # We need to return an object that behaves like an IO::File.
    # It has to support getline() and close()
    # For value files, we just return the content as-is.
    if ($isValueFile) {
	return new IO::String($text);
    }

    # For key-value configuration in the Torus, we have to rearrange
    # into the format of a configuration file.

    # We can rely on Torus.pm having imported this method already.
    # Makes life simpler since we could get either XML::Simple version.
    my $xml = Masterkey::Admin::Torus::XMLin($text, ForceArray => 1,
					     KeyAttr => [ 'name' ]);
    my $layer;
    if (defined $xml->{layer}) {
	# It's a single record
	$layer = $xml->{layer}->{final};
    } else {
	# Assume it's a record list, such as a search result
	$layer = $xml->{record}->[0]->{layer}->{final};
    }

    if (!defined $layer) {
	warn "no 'final' layer for '$url': maybe no records found?";
	return new IO::String("");
    }

    my $config = "";
    foreach my $key (keys %$layer) {
	next if $key =~ /:/; # Namespace, etc.
	next if grep { $key eq $_ } qw (id realm creationDate
				        lastModified configId displayName);
	my $data = $layer->{$key};
	if (!ref $data) {
	    # I think this is how XML::LibXML::Simple returns it
	    # Nothing to be done here.
	} elsif (ref $data eq "ARRAY") {
	    # I think this is how XML::Simple returns it
	    $data = $data->[0];
	} else {
	    die "non-array config value '$data' for '$key'"
	}

	$key =~ s/_/./g;
	$data =~ s/\n/\\\n/gm;
	$config .= "$key: $data\n";
    }

    #warn $config;
    return new IO::String($config);
}


1;
