var EXPORTED_SYMBOLS = ["Conditional"];

Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/jsonPathHelper.js');
Components.utils.import('resource://indexdata/runtime/StepError.js');
Components.utils.import('resource://indexdata/util/logging.js');

var logger = logging.getLogger();


/* The constructor takes a reference to an array of conditions which will be
 * updated in the draw() method and used to determine the output of
 * eval(). Each have the following properties: 
 *
 * operands: [ {path: key: useconstant: constantvalue, ... }, ...]
 * operator: key from the operators object defined below
 * negate: if true, negate the result (default: false) 
 * connective: "AND" or "OR", (default: "AND") 
 *
 * The operands used to be like this:
 *   operands: [{usejp: <boolean>, jp: {path: key:}, value: <constant>}, ...]
 * There is an upgrade function to convert these to the new jp format.
 * 
 * The second parameter is an object for JSONpaths to run against.
 *
 *
 * Possible future features:
 * - test if ANY or ALL of a jp evaluate to true rather than just FIRST
 */
var Conditional = function (conditions, jptarget) {
  this.conditions = conditions;
  this.jptarget = jptarget
};

// static
Conditional.operators = {
  always: {
    label: 'always',
    short: 'always',
    compare: function() {
      return true;
    },
    arity: 0, // takes no arguments
  },
  empty: {
    label: 'is empty',
    short: 'empty',
    compare: function(v) {
      if (typeof(v) === 'undefined' || typeof(v) === 'null' || v === '') return true;
      else return false;
    },
    arity: 1
  },
  equals: {
    label: 'equals',
    short: '==',
    compare: function(l, r) {
      return l == r;
    },
    arity: 2
  },
  match: {
    label: 'matches regex',
    short: '=~', 
    compare: function(l, r) {
      // TODO: these should probably throw execeptions to be handled appropriately by steps
      // everything matches nothing
      if (typeof(r) === 'undefined' || typeof(r) === 'null' || r === '') return true;
      // regex only works on strings
      if (typeof(l) != "string" ) l = "";
      let re = new RegExp(String(r));
      return re.test(String(l));
    },
    arity: 2
  },
  contains: {
    label: 'contains string',
    short: 'contains', 
    compare: function(l, r) {
      // everything matches contains nothing
      if (typeof(r) === 'undefined' || typeof(r) === 'null' || r === '') return true;
      return String(l).indexOf(String(r)) !== -1;
    },
    arity: 2
  },
  gt: {
    label: 'greater than',
    short: '>', 
    compare: function(l, r) {
      logger.info("  gt: l='" + l + "' " + typeof(l) + " r='" + r + "' " + typeof(r) ); // ###
      return Number(l) > Number(r);
    },
    arity: 2
  },
  gteq: {
    label: 'greater than or equal',
    short: '>=', 
    compare: function(l, r) {
      return Number(l) >= Number(r);
    },
    arity: 2
  },
  lt: {
    label: 'less than',
    short: '<', 
    compare: function(l, r) {
      return Number(l) < Number(r);
    },
    arity: 2
  },
  lteq: {
    label: 'less than or equal',
    short: '<=', 
    compare: function(l, r) {
      return Number(l) <= Number(r);
    },
    arity: 2
  }
};

Conditional.oneToString = function(cond) { 
  let op = this.operators[cond.operator];
  let str = '';
  if (cond.negate) str += '!';
  str += '(';
  if (op.arity > 0) {
    str += jsonPathHelper.displayString(cond.operands[0], true);
  }
  str += ' ' + op.short + ' ';
  if (op.arity > 1) {
    str += jsonPathHelper.displayString(cond.operands[1], true);
  }
  str += ')';
  return str;
}
  
// because closures
Conditional.makeRemoveListener = function (context, step, surface, i) {
  return function (e) {
    if (context.conditions.length > 1) {
      context.conditions.splice(i, 1);
      context.draw(surface, step);
    }
  };
};

// instance
Conditional.prototype.draw = function (vbox, step, caption) {
  Components.utils.import('resource://indexdata/util/xulHelper.js');
  logger.debug("Conditional.draw: this " + JSON.stringify(this) );

  var context = this;
  var defaultCondition = {
    operands: [{useconstant: false, path:'', key:''}, {useconstant: true, constantvalue:''}],
    operator: 'contains'
  };
  if (this.conditions.length < 1) {
    this.conditions.push(defaultCondition);
    logger.debug("Cond.draw: using default condition " + JSON.stringify(this.conditions) );
  }
  xmlHelper.emptyChildren(vbox);

  // conditions
  let condition;
  let hbox;
  for (let i = 0; i < this.conditions.length; i++) {
    condition = this.conditions[i];
    let currentBox = xmlHelper.appendNode(vbox, 'vbox');

    // first row
    hbox = xmlHelper.appendNode(vbox, "hbox", null, {align: "center"}); 
    if (caption && i === 0) xmlHelper.appendNode(hbox, "caption", caption);
    if (i > 0) {
      let boolSelect = xmlHelper.appendNode(hbox, "menulist");
      let boolMenu = xmlHelper.appendNode(boolSelect, "menupopup");
      let boolItemAnd = xmlHelper.appendNode(boolMenu, "menuitem", null,
        {value: 'AND', label: 'AND'});
      let boolItemOr = xmlHelper.appendNode(boolMenu, "menuitem", null,
        {value: 'OR', label: 'OR'});
      xulHelper.selectMenulistItemWithValue(boolSelect, condition.connective || 'AND');
      boolSelect.addEventListener('select',
        xulHelper.makeValueListener(condition, 'connective'), false);
    }
    xulHelper.jsonPathField(hbox, step,  condition.operands, 0, null, null,
                  { singlevalue: true, constantallowed: true, rwmode: "r" }  );
      // surface, step, confObj, confKey, caption, initial, options
      // note that confKey can also be a numerical thing, if confObj is an array,
      // as it is here.
    
    // second row
    hbox = xmlHelper.appendNode(vbox, "hbox", null, {align: "center"}); 
    let negateButton = xmlHelper.appendNode(hbox, "checkbox", null,
      {'label': 'NOT'}, null);
    if (condition.negate === true) negateButton.checked = true;
    negateButton.addEventListener("command",
      xulHelper.makeToggleListener(condition, 'negate'), false);
    let opSelect = xmlHelper.appendNode(hbox, "menulist");
    let opMenu = xmlHelper.appendNode(opSelect, "menupopup");
    let opItems = {};
    for (let key in Conditional.operators) {
      let op = Conditional.operators[key];
      opItems[key] = xmlHelper.appendNode(opMenu, "menuitem", null,
        {value: key, label: op.label});
    }
    xulHelper.selectMenulistItemWithValue(opSelect, condition.operator);
    opSelect.addEventListener("select",
                    xulHelper.makeValueListener(condition, 'operator'), false);
    xulHelper.jsonPathField(hbox, step,  condition.operands, 1, null, null,
             { singlevalue: true, constantallowed: true, rwmode: "r" }  );
    let rm = xmlHelper.appendNode(hbox, "image", null,
      {src: "chrome://cfbuilder/content/icons/window-close.png"}, null);
    rm.addEventListener("click",
                 Conditional.makeRemoveListener(context, step, vbox, i), false);
  }
  let add = xmlHelper.appendNode(hbox, "image", null,
                  {src: "chrome://cfbuilder/content/icons/list-add.png"}, null);
  add.addEventListener("click", function (e) {
    context.conditions.push(defaultCondition);
    context.draw(vbox, step);
  }, false);
}; // draw

Conditional.prototype.eval = function () {
  let combined = this._evalOne(this.conditions[0]);
  for (let i = 1; i < this.conditions.length; i++) {
    let current = this._evalOne(this.conditions[i]);
    if (this.conditions[i].connective === 'OR')
      combined = combined || current;
    else
      combined = combined && current;
  }
  return combined;
};

Conditional.prototype.toString = function () {
  if (!Array.isArray(this.conditions) || this.conditions.length < 1) return "";
  let str = Conditional.oneToString(this.conditions[0]);
  for (let i=1; i<this.conditions.length; i++) {
    let cur = this.conditions[i];
    str += cur.connective === 'OR' ? ' || ' : ' && ';  
    str += Conditional.oneToString(cur);
  }
  return str; 
};

Conditional.prototype.getUsedArgs = function () {
  let result = [];
  //logger.info("Conditional.getUsedArgs (" + this.conditions.length + " conditions)");
  for (let i = 0; i < this.conditions.length; i++) {
    let operands = this.conditions[i].operands;
    //logger.info(" condition " + (i+1) + ": checking " + operands.length + " operands");
    for (let j = 0; j < operands.length; j++) {
      let op = operands[j];
      //logger.info("  operand " + (j+1) + ": op=" + JSON.stringify(op));
      if (!op.useconstant && op.path === "$.input")
        result.push(op.key);
    }
  }
  //logger.info("returning list of " + result.length + " items: " + JSON.stringify(result));
  return result;
}

// Helper to upgrade conditions to take advantage of the new varselector, with
// useconstant flag and constantvalue inside the jp
Conditional.prototype.upgradeConstants = function () {
  for (let i = 0; i < this.conditions.length; i++) {
    let operands = this.conditions[i].operands;
    logger.debug(" Upgrading condition " + i + ": checking " + operands.length + " operands");
    for (let j = 0; j < operands.length; j++) {
      let op = operands[j];
      logger.debug("    operand  " + j + ": op=" + JSON.stringify(op));
      if ( typeof(op.useconstant) == "undefined" ) {
        if ( op.usejp ) {
          let oldjp = op.jp;
          op.jp = undefined;
          op.useconstant = false;
          //op.foo="var";
          for( let k in oldjp )
            op[k] = oldjp[k];
        } else {
          op.useconstant = true;
          op.constantvalue = op.value;
        }
        op.value = undefined;
        op.usejp = undefined;
        logger.debug("    upgraded " + j + ": op=" + JSON.stringify(op));
      } else {
        logger.debug("    already upgraded, did not touch");
      }
    }
  }
} // upgradeConstants

// private
Conditional.prototype._getValue = function (cond) {
  if ( typeof(cond.usejp) != "undefined" )
    throw new StepError("Trying to use an old-type Conditional: " +
      JSON.stringify(cond) );
  return jsonPathHelper.getFirst(cond, this.jptarget);
};

Conditional.prototype._evalOne = function (condition) {
  let op = Conditional.operators[condition.operator];
  let left;
  let right;
  if (op.arity > 0)
    left = this._getValue(condition.operands[0]);
  if (op.arity > 1)
    right = this._getValue(condition.operands[1]); 
  if (condition.negate)
    return !(op.compare(left, right));
  else return op.compare(left, right);
};

// Would be nice to have the option of a class rather than instance unit test
Conditional.prototype.unitTest = function () {
  return Conditional.unitTest();
};

Conditional.unitTest = function () {
  let record = {
    "results":[
      {"author":["billy","bob"], "title":["a book"],
          "item":[{"location":["a library"]}, {"location":["my house"]}]},
      {"author":["jill"], "title":["a paper"], "journaltitle":"unexpected scalars"}],
    "regex":"^a.*r$"};
  let tests = [
    {
      oldconditions: [ // these are testing the upgrade routine. 
        {operands: [{usejp: false, value: '3'},
                    {usejp: false, value: 2}], operator: 'gt'},
        {operands: [{usejp: false, value: ''},
                    {usejp: false, value: ''}], operator: 'always', connective: 'AND'}
      ],
      conditions: [
        {operands: [{useconstant: true, constantvalue: '3'},
                    {useconstant: true, constantvalue: 2 }], operator: 'gt'},
        {operands: [{useconstant: true, constantvalue: ''},
                    {useconstant: true, constantvalue: ''}], operator: 'always', connective: 'AND'}
      ],
      expected: true
    },
    {
      oldconditions: [{
        operands: [
          {usejp: true, jp: {path: '$.results[0]', key: 'title'}},
          {usejp: false, value: 'book'}
        ],
         operator: 'contains',
         negate: true
       }],
      conditions: [{
        operands: [{useconstant: false, path: '$.results[0]', key: 'title'},
                   {useconstant: true, constantvalue: 'book'} ],
        operator: 'contains',
        negate: true
      }],
      expected: false
    },
    {
      oldconditions: [{  // test that we detect already upgraded conditions, and 
        operands: [ // don't try to upgrade again
          {useconstant: false, path: '$.results[1]', key: 'title'},
          {useconstant: false, path: '$', key: 'regex'} ],
        operator: 'match',
      }],
      conditions: [{
        operands: [
          {useconstant: false, path: '$.results[1]', key: 'title'},
          {useconstant: false, path: '$', key: 'regex'} ],
        operator: 'match',
      }],
      expected:true
    },
    {
      conditions: [
        {
          operands: [{useconstant: false, path: '$', key: 'nothere'}],
          operator: 'empty',
          negate: true
        },
        {
          operands: [{useconstant: false, path: '$.results[1]', key: 'title'}],
          operator: 'empty',
          connective: 'OR',
        }
      ],
      expected:false
    },
    {
      conditions: [
        {
          operands: [{useconstant: true, constantvalue:''}],
          operator: 'empty',
        },
        {
          operands: [{useconstant: true, constantvalue:42}],
          operator: 'empty',
          negate: true
        }
      ],
      expected:true
    }
  ];
  // TODO - Move the upgrade tests into their own test set
  // this is too messy.
  let result = false;
  for (let i = 0; i < tests.length; i++) {
    let test = tests[i];
    let instance = new Conditional(test.conditions, record);
    if ( test.oldconditions ) {  // test the upgrade function
      let oldinst = new Conditional(test.oldconditions, record);
      oldinst.upgradeConstants();
      let upgraded = JSON.stringify(oldinst.conditions);
      let news = JSON.stringify(test.conditions);
      if ( news != upgraded ) {
        logger.error("Upgrading conditions failed ");
        logger.error("  Expected: " + JSON.stringify(test.conditions) )
        logger.error("  Got:      " + upgraded );
        return false;
      }
    }
    try {
      result = instance.eval();
    } catch (e) {
      logger.error(e);
      return false
    }
    if (result === test.expected) {
      logger.info(" * " + String(instance) + " is " + test.expected + ",  OK");
    } else {
      logger.error(" * " + String(instance) + " isn't " + test.expected  + ", FAIL");
      return false;
    }
  }
  return true;
};

