var EXPORTED_SYMBOLS = ["ConnectorTemplate"];
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/objHelper.js');
Components.utils.import('resource://indexdata/runtime/TaskTemplate.js');
Components.utils.import('resource://indexdata/runtime/core.js');
Components.utils.import('resource://indexdata/util/logging.js');
var logger = logging.getLogger();
var ConnectorTemplate = function () {
  // object properties do not guarantee ordering
  this.taskTemplates = [];
  this.taskTemplatesByName = {};
  this.requirements = {};
  this.properties = {};
  this.errorCodes = [];
  this.metaData = null;
  this.name = null;
  this.capabilityFlags = [];  // array of objects
};
ConnectorTemplate.prototype = {
  // UUID is filename
  loadFromUUID: function (name) {
    // if we've loaded this before, we still have the DOM lying around
    if (core.templates[name]) { 
      this.load(core.templates[name]);
      return;
    }
    let file = false;
    let env = Components.classes["@mozilla.org/process/environment;1"].
              getService(Components.interfaces.nsIEnvironment);
    // environment variable takes precedence
    if (env.exists('CF_TEMPLATE_PATH')) {
      file = this.fileInPaths(env.get('CF_TEMPLATE_PATH').split(':'), name);  
    }
    // then preferences
    if (file === false) {
      // can't test if a pref exists, just need to catch the exception
      try { 
        let pref = core.idprefs.getCharPref('templatePath');
        if (pref) file = this.fileInPaths(pref.split(':'), name);
      } catch (e) {};
    }
    // finally, local templates
    if (file === false && core.cfRootFile !== null) {
      file = Components.classes["@mozilla.org/file/local;1"]
          .createInstance(Components.interfaces.nsILocalFile);
      file.initWithPath(core.cfRootFile.path + '/templates/' + name);
    } 
    if (file && file.exists()) this.loadFromFile(file);
    else throw new Error("Template " + name + " not available");
  },
  fileInPaths: function (paths, filename) {
    for (let i = 0; i < paths.length; i++) {
      let file = Components.classes["@mozilla.org/file/local;1"]
          .createInstance(Components.interfaces.nsILocalFile);
      file.initWithPath(paths[i] + '/' + filename);
      if (file.exists() !== false) return file;
    }
    return false;
  }, 
  loadFromFile: function (file) {
    let doc = xmlHelper.openDoc(file.path);
    let ctElements = doc.getElementsByTagName("connectorTemplate");
    if (!ctElements || ctElements.length !== 1)
      throw new Error("Invalid connector template: does not contain exactly one connectorTemplate element.");
    let ctNode = ctElements[0];
    if (ctNode.hasAttribute('name')) {
      let name = ctNode.getAttribute('name');
      if (name !== file.leafName)
        throw new Error("Connector Template name " + name + " does not match name of loaded file " + file.leafName);
    }
    this.load(ctElements[0]);
  },
  load: function (ctNode) {
    if (ctNode.hasAttribute('name')) {
      this.name = ctNode.getAttribute('name');
      core.templates[this.name] = ctNode;
    }
    this.properties = objHelper.loadProperties(ctNode);
    this.defaults = objHelper.loadDefaults(ctNode);
    this.taskTemplates = [];
    this.errorCodes = [];
    this.requirements = {};
    var mdContainerNodes = [];
    for (let i = 0; i < ctNode.childNodes.length; i++) {
      let nd = ctNode.childNodes[i];
      switch (nd.localName) {
      case 'metaData':
        mdContainerNodes.push(nd);
        break;
      case 'property':
        this.properties[nd.getAttribute('name')] = nd.getAttribute('value');
        break;
      case 'task':
        let newTaskTemplate = new TaskTemplate();
        newTaskTemplate.load(nd);
        newTaskTemplate.rank = i;
        this.taskTemplates.push(newTaskTemplate);
        this.taskTemplatesByName[newTaskTemplate.name] = newTaskTemplate;
        break;
      case 'errorCode':
        this.errorCodes.push({
          key: nd.getAttribute('name') || nd.getAttribute('value'),
          value: nd.getAttribute('value')
        });
        break;
      case 'requirement':
        if (nd.hasAttribute('task')) {
          var taskName = nd.getAttribute('task');
          if (!this.requirements[taskName]) this.requirements[taskName] = [];
          var argNodes = nd.getElementsByTagName("argument");
          var args = [];
          for (var j=0; j<argNodes.length; j++) {
            args.push(argNodes[j].getAttribute('name'))
          }
          this.requirements[taskName].push(args);
        }
        break;
      }
    }
    // load metadata
    this.metaData = [];
    var metaSeen = {};
    if (mdContainerNodes.length < 1) {
      throw new Error("No metadata prototypes found!");
    } else {
      if (mdContainerNodes.length > 1)
        logger.warn("Multiple metadata elements found, using first...");
      var metaNodes = mdContainerNodes[0].getElementsByTagName("meta");
      for (var i=0; i<metaNodes.length; i++) {
        var attrs = {};
        for (var j=0; j<metaNodes[i].attributes.length; j++) {
          var attr = metaNodes[i].attributes[j];
          attrs[attr.name] = attr.value;
        }
        if (!metaSeen[attrs.name]) this.metaData.push(attrs);
        metaSeen[attrs.name] = true;
      }
    }
    // load capability flags
    var capContainerNodes = ctNode.getElementsByTagName("capabilityFlags");
    this.capabilityFlags = [];
    var capSeen = {};
    if (capContainerNodes.length < 1) {
      //throw new Error("No cpabilityFlags prototypes found!");
      // Don't make it a serious error quite yet! Empty set is acceptable.
      logger.warn("No capability flags found in template");
    }
    for ( var capCont = 0; capCont < capContainerNodes.length; capCont++ ){
      var capNodes = capContainerNodes[capCont].getElementsByTagName("flag");
      for (var i=0; i<capNodes.length; i++) {
        var attrs = {};
        for (var j=0; j<capNodes[i].attributes.length; j++) {
          var attr = capNodes[i].attributes[j];
          attrs[attr.name] = attr.value;
          // logger.warn("Loading template capability " +
          //   attr.name + " " + attr.value + "\n");
        }
        if (capSeen[attrs.name]) {
          logger.warn("Duplicate capability flag '" + attrs.name +
               "'. Using first\n");
        } else {
          // logger.warn("Capability flag " + attrs.name + "\n");
          this.capabilityFlags.push(attrs);
        }
        capSeen[attrs.name] = true;
      }
    }
  },

  getTaskTemplates: function () { return this.taskTemplates; },
  getTaskTemplate: function (taskName) {
    return this.taskTemplatesByName[taskName];
  },
  getRequirements: function (taskName) {
    return this.requirements[taskName] || [];
  },
  getMetaData: function () { return this.metaData; },
  getCapabilityFlags: function () { return this.capabilityFlags; },
};
