/*
 * Step superclass - all steps need extend this class to conform to the Step
 * interface and implement common functionality (e.g configuration parsing)
 * Most interface methods are "virtual" and must be redefined while common
 * functionality methods may be overriddden to change the default bahaviour.
 */
var EXPORTED_SYMBOLS = ["Step"];
Components.utils.import('resource://indexdata/runtime/Block.js');
Components.utils.import('resource://indexdata/util/helper.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/objHelper.js');
Components.utils.import('resource://indexdata/util/textHelper.js');
Components.utils.import('resource://indexdata/util/jsonPathHelper.js');
Components.utils.import('resource://indexdata/util/logging.js');
var logger = logging.getLogger();
var Step = function () {
  // JS does not support real class inheritance, so we fake it by using the
  // prototype of the super class as the base for the child class. Unfortunately
  // the protype for the child is instantiated only once when the child class
  // is defined. That means that all the members and methods are shared across
  // instances of the same child class (pseudo-static with access to all child
  // memmbers) - carefull with that.
  this.conf = null;  //configuration
  this.task = null;  //parent task
  this.comments = null; //never initialize arrays here!!!! they will be shared
  this.isAlt = false;
  this.block = null; //conaining block
  this.blocks = null; //child blocks, if any
  this.collapsed = false; //display hint
};
// any step instance variable initialization needs to end up here
Step.prototype.initInstance = function () {
  this.blocks = [];
  this.comments = [];
};

// interface and base methods (required implementation)
Step.prototype.constructor = function (task) {
  throw new Error("Constructor: method not implemened.");
};
Step.prototype.init = function (task) {
  throw new Error("Init: method not implemened.");
};
Step.prototype.draw = function (surface) {
  throw new Error("Draw: method not implemened.");
};
Step.prototype.run = function (task) {
  throw new Error("Run: method not implemened.");
};
Step.prototype.getComments = function () { return this.comments; };

// read-only attributes
Step.prototype.getClassName = function () { return "Step"; };
Step.prototype.getDisplayName = function () { return "No name specified."; };
Step.prototype.getDescription = function () { return "No description specified."; };
Step.prototype.getVersion = function () { return "0.0"; };
Step.prototype.getBlocks = function () { return this.blocks; };

// methods with default implementation, (overriding optional)
Step.prototype.getIndentRange = function () {
  //[from, to] as indexed from the current position
  //altness is intented on top of it
  return [0,0];
};
Step.prototype.renderArgs = function () {
  // Trivial default implementation: individual step classes should
  // override this to provide a brief string representing the step
  // arguments.
  return "";
};
Step.prototype.renderBlockArgs = function (index) {
  //steps might want to render some detailed info about their blocks
  return "";
};
Step.prototype.toString = function () {
  var name = this.getDisplayName();
  var args = this.renderArgs();
  if (args) name += " (" + args + ")";
  return name;
}
// Used arguments do not need to be unique as they are aggregated in the Task
// TODO - Get from conf, to make it more automatic
Step.prototype.getUsedArgs = function () {
  return jsonPathHelper.usedArgs(this);
};

// reset run state
Step.prototype.resetState = function () { }; 
// return true if the step has been deprecated and should no longer be used
// for new connectors
Step.prototype.isDeprecated = function () { return false; };
// reason for deprecation and/or what steps should be used instead
Step.prototype.getDeprecationInfo = function () { return ""; };
// upgrade: return true if conf has successfully been modified for
// compatibility with current version
Step.prototype.upgrade = function (confVer, curVer, conf) {
  // can't upgrade if the connector is newer than the step
  if (confVer > curVer)
    return false;
  // this is something you can keep adding to if you do tests as ranges ie.:

  // if (confVer < 2)
  //   ...do updates to 3
  // if (confVer < 3)
  //   ...update to 4

  // when 5 comes out you can just just add another if for that
  // (possibly just returning true if you've done a major version
  // increment that doesn't break backwards compatibility) and
  // you'll have upgrades all the way from version 2.
  return true;
}
// Check if the capability flag should default to something.
// Return true, false, or null for don't-care
// Goes through the conf, locates what looks like jsonPaths, and
// guesses decent index- and result- defaults. It tries to be clever, and
// avoid flagging input variables if the conf setting is named 'out', and
// output flags for 'in'. This is totally unreliable, but happens to match
// the way things are done in transform, split, join, and other steps that
// use both input and output.
// if this logic is not suitable for a step, it should override the function
Step.prototype.capabilityFlagDefault = function ( flag ) {
  if ( !this.conf ) return null;
  for ( var c in this.conf ) {
    if ( this.conf[c] &&
         this.conf[c].path != undefined && // looks like a jsonPath
         this.conf[c].key != undefined ) {
      if(0) logger.debug ("Capdef: " +   // ###
        "f='" + flag + "' " +
        "p='" + this.conf[c].path +"' " +
        "k='" + this.conf[c].key +"' " );
      if ( c != 'out'  &&
           jsonPathHelper.capabilityFlagDefault(
             this.conf[c], flag, "$.input", "index") )
        return true;
      if ( c != 'in' &&
           jsonPathHelper.capabilityFlagDefault(
            this.conf[c], flag, "$.output.results", "result") )
        return true;
      if ( c != 'in' &&
           jsonPathHelper.capabilityFlagDefault(
            this.conf[c], flag, "$.output.facets", "facet") )
        return true;
      // TODO - It is not nice to mention output.results and facets here,
      // they should come from the template directly. See also CP-3739
    }
    // Many steps store conditions. These test input but don't generate results
    if (c === 'conditions' && jsonPathHelper.conditionsCapable(
        this.conf[c], flag, "$.input", "index")) { return true; }
  }
  return null;
};

// saving/loading to/from DOM and files
// you can optionally define processConf() for pre-processing
Step.prototype.writeConf = function (confObj, stepNode) {
  objHelper.objectToXml("stepConf", confObj, stepNode);
};
Step.prototype.readConf = function (confObj, confNode) {
  var parsedConf = objHelper.objectFromXml(confNode);
  for (var key in parsedConf) confObj[key] = parsedConf[key];
};
Step.prototype.saveConf = function (stepNode) {
  stepNode.setAttribute("name", textHelper.ccToUS(this.getClassName()));
  stepNode.setAttribute("version", this.getVersion());
  if (this.isAlt) stepNode.setAttribute("alt", "yes");
  if (this.isDisabled) stepNode.setAttribute("disabled", "yes");
  if (this.collapsed) stepNode.setAttribute("collapsed", "yes");
  if (typeof this.processConf === "function") this.processConf();
  if (this.conf) {
    this.writeConf(this.conf, stepNode);
  }
  //blocks
  for (let i=0; i<this.blocks.length; i++) {
    let block = this.blocks[i];
    let blockNode = xmlHelper.appendNode(stepNode, 'block');
    block.saveConf(blockNode);
  }
};
Step.prototype.reqVer = function () { return this.conf.reqVer; }
Step.prototype.loadConf = function (task, block, stepNode) {
  var reqVer = stepNode.getAttribute("version");
  var curVer = this.getVersion();
  this.block = block;
  this.conf = {};
  this.blocks = [];
  this.conf["reqVer"] = reqVer;
  //flags
  if (stepNode.getAttribute("alt") === "yes") this.isAlt = true;
  if (stepNode.getAttribute("disabled") === "yes") this.isDisabled = true;
  if (stepNode.getAttribute("collapsed") === "yes") this.collapsed = true;
  // load conf and blocks
  for (var j=0; j<stepNode.childNodes.length; j++) {
    let node = stepNode.childNodes[j];
    switch (node.nodeType) {
      // element node
      case 1:
        switch (node.localName) {
          case "stepConf":
            this.readConf(this.conf, node);
            break;
          case "block":
            this.blocks.push(Block.fromConf(task, node, this));
            break;
        }
        break;
    }
  }
  // attempt to promote conf to current version
  if (reqVer !== curVer &&
    !this.upgrade(Number(reqVer), Number(curVer), this.conf)) {
    // major.minor version checking
    // major updates break compatibility, minor are only backward compatible
    if (!helper.checkVersion(reqVer, curVer)) {
      var name = stepNode.getAttribute("name");
      throw new Error("This connector requires step '" + name
          + "' version " + reqVer
          + " while the available version is " + curVer);
    }
  }  
};

// misc. instance methods you won't likely override
Step.prototype.getPageWindow = function () {
  return this.task.connector.getPageWindow();
};
Step.prototype.getPageDoc = function () {
  return this.task.connector.getPageDoc();
};
//step specific cloning
Step.prototype.clone = function () {
  var clone = new this.constructor();
  clone.initInstance();
  var conf = JSON.parse(JSON.stringify(this.conf));
  clone.conf = conf;
  //be picky abot what is being cloned -- we don't want references or internal
  //step state, only 'persistent' attributes should survive cloning
  clone.isAlt = this.isAlt;
  clone.isDisabled = this.isDisabled;
  //blocks
  for (let i=0; i<this.blocks.length; i++) {
    let b = this.blocks[i];
    //we set the task to null initially, attachStep will fix it
    let bclone = new Block(null, b.name, b.label, clone);
    clone.blocks.push(bclone);
    for (let j=0; j<b.steps.length; j++) {
      let s = b.steps[j];
      let c = s.clone();
      bclone.steps.push(c);
      c.block = bclone;
    }
  }
  return clone;
};

//static functions
Step.fromConf = function (task, block, node) {
  let step = Step.fromFileName(node.getAttribute("name"));
  // I have no idea why on earth we need it here
  Components.utils.import('resource://indexdata/runtime/Block.js');
  step.loadConf(task, block, node);
  return step;
};
// loads step instance given the file name
Step.fromFileName = function (step_name) {
  var step;
  try {
    Components.utils.import('resource://indexdata/steps/'
        + step_name + '/' + step_name + '.js');
    step = eval("new " + textHelper.usToCC(step_name) + "()");
    step.initInstance();
  } catch (e) {
    if ( e.name == "NS_ERROR_FILE_NOT_FOUND" )
      throw new Error ( "Required step " +
          "'" + step_name + "' missing." );
    // Could be something like SyntaxError etc
    throw new Error ( "Problem loading step " +
        "'" + step_name + "':  " +
        e.name + ": " + e.message +" " +
        e.fileName + ":" + e.lineNumber );
  }
  return step;
};

