var EXPORTED_SYMBOLS = ["Assert"];
Components.utils.import('resource://indexdata/runtime/Conditional.js');
Components.utils.import('resource://indexdata/runtime/Step.js');
Components.utils.import('resource://indexdata/runtime/StepError.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/xulHelper.js');
Components.utils.import('resource://indexdata/util/jsonPathHelper.js');
Components.utils.import('resource://indexdata/util/logging.js');

var logger = logging.getLogger();

var Assert = function () {
  this.conf = {};
  this.conf.conditions = []; // data for this.conditional, instantiated on init
  this.conf['error'] = '';
  this.conf.succeed = false;
};
Assert.prototype = new Step();
Assert.prototype.constructor = Assert;

Assert.prototype.init = function() {
  this.conditional = new Conditional(this.conf.conditions, this.task.data);
};

Assert.prototype.draw = function(surface) {
  xmlHelper.emptyChildren(surface);
  var context = this;
  var captionWidth = 140;

  xmlHelper.appendNode(surface, "caption", "Ensure that this is true:");
  var condBox = xmlHelper.appendNode(surface, "vbox");
  this.conditional.draw(condBox, this);

  xmlHelper.appendNode(surface, "separator");
  xmlHelper.appendNode(surface, "caption", "Error when condition is not met:");
  
  var hbox2 = xmlHelper.appendNode(surface, "hbox", null, {align:"center"});
  var hboxCode = xmlHelper.appendNode(hbox2, "hbox", null, {align:"center"});
  xulHelper.captionField(hboxCode, "Code:", {width: 50});
  xulHelper.arraySelectField(hbox2, this, 'errorCode',
      this.task.connector.template.errorCodes,
      {});
  xulHelper.inputField(hbox2, this, "error", "Message:",
      80, { flex:1 });

  xulHelper.checkbox(surface, this, 'succeed', 'end task with success');
};

Assert.prototype.run = function (task) {
  if (this.conditional.eval()) {
    task.info("Assertion " + String(this.conditional) + " passed", this);
    return;
  } else {
    let msg = jsonPathHelper.inlineReplace(this.conf['error'], task.data);
    if (this.conf.succeed) {
      task.warn("Assertion " + String(this.conditional) +" failed, task complete; exiting.", this);
      task.warn(msg);
      this.task.complete();
      return;
    } else {
      if (this.conf['errorCode']) msg = this.conf['errorCode'] + ":" + msg;
      //only warn since error might be handled with an ALT step
      task.warn("Assertion " + String(this.conditional) 
          + " failed, specified error will be thrown..", this);
      throw new StepError(msg);
    }
  }
  return;
};

Assert.prototype.getClassName = function () {
  return "Assert";
};

Assert.prototype.getDisplayName = function () {
  return "Assert";
};

Assert.prototype.getDescription = function () {
  return "Asserts a condition against a stored value.";
};

Assert.prototype.getVersion = function () {
  // return "1.1";  added the mode setting
  // return "2.0";  // error optional (successful exit), error text optional
  // return "3.0";  // use Conditional
  return "4.0"; // new form of conditional, with constants built into the jp
};

Assert.prototype.renderArgs = function () {
  if (this.conditional)
    return String(this.conditional);
};

Assert.prototype.getUsedArgs = function () {
  if (this.conditional)
    return this.conditional.getUsedArgs();
  return [];
};

Assert.prototype.upgrade = function (confVer, curVer, conf) {
  // can't upgrade if the connector is newer than the step
  logger.debug("assert upgrade from " + confVer + " to " + curVer );
  if (confVer > curVer)
    return false;

  if (confVer < 0.2) {
    let spec = {key: conf.target};
    let container = conf.container;
    if (container === "INPUT") {
      spec.path = "$.input";
    } else if (container !== undefined) {
      spec.path = "$.output";
      if (container !== "NONE") {
        if (container === "item") {
          spec.path += ".results[*].item[*]";
        } else {
          spec.path += "." + container + '[*]';
        }
      }
    }
    delete conf.container;
    conf.target = spec;
  }
  if (confVer < 1.1) {
    if ( typeof(conf.mode) == "undefined" )
      conf.mode = "miss";
  }
  if (confVer < 3.0) {
    // second element is negation
    let modeMap = {
      "always":["always", true],
      "match":["match", true],
      "miss":["match", false],
      "empty":["empty", true], 
      "value":["empty", false]
    };
    conf.conditions = [{
      operands: [
        {usejp: true, jp:{key: conf.target.key, path: conf.target.path}}, 
        {usejp: false, value: conf.constantoperand}
      ],
      operator: modeMap[conf.mode][0],
      negate: modeMap[conf.mode][1]
    }];
    delete conf.mode; 
    delete conf.constantoperand;
    delete conf.target;
  }
  if (confVer < 4.0) {
    var c = new Conditional(conf.conditions, null);
    c.upgradeConstants();
    conf.conditions = c.conditions;
    logger.debug("assert: upgraded conditions to " + JSON.stringify( conf.conditions) );
  }

  return true;
};
