var EXPORTED_SYMBOLS = ["Fullquerylimit"];

// Extracts a query limite from a fullquery. That is, any clause
// that is purely ANDed to the rest of the query, and uses a given
// index. Often 'year', but can be anything else.
// Handles three separate cases
//  - A single value
//  - A (numerical) range, with one or two values (can be open in one end)
//  - A OR-bag of individual values (like source types, etc)
// Returns a listquery segment, and (optionally?) removes the elements from
// the fullquery.

// TODO Or-bags

Components.utils.import('resource://indexdata/runtime/Connector.js');
Components.utils.import('resource://indexdata/runtime/Task.js');
Components.utils.import('resource://indexdata/runtime/Step.js');
Components.utils.import('resource://indexdata/runtime/StepError.js');
Components.utils.import('resource://indexdata/util/xulHelper.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/textHelper.js');
Components.utils.import('resource://indexdata/util/jsonPathHelper.js');
Components.utils.import('resource://indexdata/util/queryHelper.js');
Components.utils.import('resource://indexdata/util/logging.js');
Components.utils.import('resource://indexdata/thirdparty/jsonPath.js');


var logger = logging.getLogger();

var Fullquerylimit = function () {
    this.className = "Fullquerylimit";
    this.conf = {};
    this.conf['in'] = { path:'$.input', key:'fullquery'};
    this.conf['out'] = { path:'$.temp', key: 'year' };
    this.conf['field'] = "year";
    this.conf['ranges'] = true;
    this.conf['makerange'] = false;
    this.conf['makeendpoints'] = true;
    this.conf['limitendpoints'] = true;
    this.conf['startdef'] = "";
    this.conf['enddef'] = "";
    this.conf['orvalues'] = false;
    this.conf['mergeorlist'] = false;
    this.conf['remove'] = true;
    this.conf['failremain'] = true;
    this.conf['failnomatch'] = false;
    this.conf['makedefault'] = false;
    this.conf['defaultvalue'] = "";

};

Fullquerylimit.prototype = new Step();
Fullquerylimit.prototype.constructor = Fullquerylimit;

Fullquerylimit.prototype.init = function() {};


/////////////////////
// UI

Fullquerylimit.prototype.draw = function(surface,win) {
  var context = this;
  var vb = xmlHelper.appendNode(surface, "vbox" );
  xulHelper.jsonPathMapField(vb, context, "in", "out");
  xulHelper.inputField(vb, context, "field", "Index field" );
    // TODO - get a pulldown from search template inputs
  var rb = xmlHelper.appendNode(vb, "hbox" );
  xulHelper.checkbox(rb, context, "ranges",
                     "Support ranges (2001-2003)" );
  xulHelper.checkbox(rb, context, "makerange",
                     "Make a range even of a single value " +
                       "(Deprecated, use endpoints!)" );
  
  var eb = xmlHelper.appendNode(vb, "hbox", null, {align:"baseline"} );
  var eplabel = "Make endpoints " +
    this.conf.out.path + "." + "start" + this.conf.out.key + " and " +
    this.conf.out.path + "." + "end" + this.conf.out.key +".";
    // TODO - Make the label change when the out changes
  xulHelper.checkbox(eb, context, "makeendpoints", eplabel );
  var eb2 = xmlHelper.appendNode(vb, "hbox", null, {align:"baseline"} );
  xulHelper.labelField(eb2,"      Default to" );
  xulHelper.inputField(eb2,context,"startdef","", 10, {width:45} );
  xulHelper.inputField(eb2,context,"enddef","-", 8, {width:45} );
  xulHelper.labelField(eb2,"(YYYY for current)" );
  xulHelper.checkbox(eb2, context, "limitendpoints", 
                     "Limit endpoints to fit default range" );
  
  
  var rb = xmlHelper.appendNode(vb, "hbox" );
  xulHelper.checkbox(rb, context, "remove",
                     "Remove the limits from the fullquery" );
  xulHelper.checkbox(rb, context, "failremain",
                     "Fail if the field remains anywhere in the query" );
  var nb = xmlHelper.appendNode(vb, "hbox" );
  xulHelper.checkbox(nb, context, "failnomatch",
                     "Fail if no limit found" );
  xulHelper.checkbox(nb, context, "makedefault",
                     "If not found, fake it with value" );
  xulHelper.inputFieldOnly(nb, context, "defaultvalue");

  var ob = xmlHelper.appendNode(vb, "hbox" );
  xulHelper.checkbox(ob, context, "orvalues",
                     "Support OR-list (type = book or article or ...)" );
  var mlabel = "Merge all OR-values into a single string in " +
    this.conf.out.path + "." +  this.conf.out.key + "values";
  xulHelper.checkbox(ob, context, "mergeorlist", mlabel );

  /* TODO
  // Update the label for end points every time the out variable changes
  var out
  var updateendpointlabel = function () {

  }
  */

};

///////////////////
// Run

Fullquerylimit.prototype.node = function ( n ) {
  this.task.debug("Looking at node " + JSON.stringify(n) );
  if ( ! n.op ) return;   // term node, uninteresting
  if ( n.op != "and" ) return; // or-node, can't be a limit  // TODO - or-bags
  var y1 = islimitnode(n.s1, this.conf.field, undefined, this.conf.orvalues );
  var y2 = islimitnode(n.s2, this.conf.field, undefined, this.conf.orvalues);
  if ( y1 && y2 ) { // should never happen
    // Well, can happen with unlikely queries like
    // ( a or b ) and ( c or d ). The node itself is not a limit node,
    // because of the mixed operators. Both children are valid limits.
    // but what we have is not a limit, and there is no need to dig deeper.
    this.task.debug("Oops, descended into a limit node, both children are " +
               "limit nodes!. This may go wrong");
  } else if (y1) {
    //this.task.debug("y1: " + JSON.stringify(y1) );
    if ( this.limitnodes.length>0 && ! y1[0].op )
      y1[0].op = n.op;
    this.limitnodes = this.limitnodes.concat(y1);
    if ( this.conf.remove ) {
      replace(n, n.s2);
      this.node(n);
    } else {
      this.node(n.s2);
    }
  } else if (y2) {
    //this.task.debug("y2: " + JSON.stringify(y2) );
    if ( this.limitnodes.length>0 && ! y2[0].op )
      y2[0].op = n.op;
    this.limitnodes = this.limitnodes.concat(y2);
    if ( this.conf.remove ) {
      replace(n, n.s1);
      this.node(n);
    } else {
      this.node(n.s1);
    }
  } else {
    this.node(n.s1);
    this.node(n.s2);
  }
};



// Checks if the node is (a subtree of only) nodes with the right field,
// and the same operator all the way down (and/or)
// (OR only accepted if orvalues==true)
function islimitnode(n, field, op, orvalues) {
  logger.debug("islimit: f=" + field + " op=" + op + " orv=" + orvalues +
      " " + JSON.stringify(n));
  if ( !n ) { // should never happen
    return false; // but can if test data misses s1 or s2 in a op node!
  }
  if ( n.field == field ) { // term node
    if ( op == "or" && !queryHelper.is_equal_relation_node(n) )
      return false; // or-bags only work with equal nodes
    var n2 = {};
    replace(n2,n);
    n2.op = op;
    return [ n2 ];
  }
  if ( !op && ( (n.op == "and") || (n.op == "or") ) )
    op = n.op;
  if ( op == "or" && !orvalues )
    return false;
  if ( op && op == n.op ) {
       var y1 = islimitnode(n.s1,field,op,orvalues);
       var y2 = islimitnode(n.s2,field,op,orvalues);
       if ( y1 && y2 )
         return y1.concat(y2);
  }
  return false;
};

// Replace all fields of n with those in r
// used for removing a node in the fullquery tree
// We know that the node we work on will be an op-node
// The stuff replace into it is typically one of its children
function replace(n, r) {
  delete n.op; // clear known fields, it is always a op-node
  delete n.s1;
  delete n.s2;
  for ( var k in r )
    n[k] = r[k];
};

// Check if
function remains(node,field) {
  if ( node.field == field )
    return true;
  if ( node.s1 && remains(node.s1,field) )
    return true;
  if ( node.s2 && remains(node.s2,field) )
    return true;
  return false;
};

Fullquerylimit.prototype.rootnode = function ( n ) {
  this.limitnodes = [];
  this.node(n);
  this.task.debug("Fullquerylimit: Got limit nodes " +
      JSON.stringify(this.limitnodes) );
  if ( this.limitnodes.length == 0 ) { // nothing found
    if ( this.conf.failnomatch )
      throw new StepError ("No querylimit '" + this.conf.field + "' found");
    if ( this.conf.makedefault ) {
        var defval = jsonPathHelper.inlineReplace(this.conf.defaultvalue,
                                                  this.task.data);
        this.limitnodes = [
           { field: this.conf.field,
             term:  defval,
             op: "" }] ;
    }
  }
  
  if (this.limitnodes.length >1 && !this.conf.ranges ) {
    throw new StepError ("Ranges not supported for " + this.conf.field );
  }
  
  if (this.conf.makerange && this.limitnodes.length == 1 &&
      queryHelper.is_equal_relation_node(this.limitnodes[0]) ){
    var ln = this.limitnodes[0];
    this.limitnodes = [
        { field: ln.field,
          term:  ln.term,
          op: "",
          relation: "ge" },
        { field: ln.field,
          term:  ln.term,
          op: "and",
          relation: "le" },
        ] ;
  }

  if ( this.conf.failremain &&  remains(n,this.conf.field) )
    throw new StepError ("Limit field '" + this.conf.field + "' " +
       "still remains after removing queryfilter");

  this.ep = queryHelper.new_endpoints();
  if ( this.conf.makeendpoints ) {
    for ( var i = 0; i<this.limitnodes.length; i++) {
      queryHelper.range_endpoint( this.limitnodes[i], this.ep);
    }
    logger.debug("Endpoints before defaults:" + JSON.stringify(this.ep) );
    var dt = new Date;
    var startdef = "";
    if (this.conf.startdef)
      startdef = jsonPathHelper.inlineReplace(this.conf.startdef, this.task.data);
    var enddef = "";
    if ( this.conf.enddef) 
      enddef = jsonPathHelper.inlineReplace(this.conf.enddef, this.task.data);
    // TODO - Expand values in start/enddef!
    if ( ! this.ep.start && startdef ) {
      var val = startdef;
      val = val. replace( /YYYY/i ,dt.getFullYear() );
      val = jsonPathHelper.inlineReplace(val, this.task.data);
      this.ep.start = val;
    }
    if ( ! this.ep.end && enddef ) {
      var val = enddef;
      val = val. replace( /YYYY/i ,dt.getFullYear() );
      val = jsonPathHelper.inlineReplace(val, this.task.data);
      this.ep.end = val;
    }
    logger.debug("Endpoints after defaults:" + JSON.stringify(this.ep) );
    if ( this.conf.limitendpoints ) {
      if ( this.ep.start && startdef && 
           this.ep.start < startdef) {
        this.ep.start = startdef;
      }
      if ( this.ep.end && enddef && 
          this.ep.end > enddef) {
        this.ep.end = enddef;
        }
      logger.debug("Endpoints after limits:" + JSON.stringify(this.ep) );
      if ( this.ep.start && this.ep.end && 
            this.ep.start > this.ep.end )
        throw new StepError ("Bad value for " + this.conf.field + 
            ". Must be in " + startdef + "-" + enddef );
    }
  }

  if ( this.conf.mergeorlist ) {
    this.orvalues = "";
    for ( var i = 0; i<this.limitnodes.length; i++) {
      if ( this.limitnodes[i].op == "or" ) {
        if ( this.orvalues )
          this.orvalues += " ";
        this.orvalues += this.limitnodes[i].term;
      }
    }
    logger.debug("orvalues '" + this.orvalues + "'");
  }
};

Fullquerylimit.prototype.run = function (task) {
  var fq = jsonPathHelper.getFirst(this.conf.in, task.data);
  if ( Array.isArray(fq) )
      fq = fq[0];
  if (!fq)
    throw new StepError ("Fullquerylimit called without a fullquery");
  this.rootnode(fq);
  jsonPathHelper.set(this.conf.out, this.limitnodes , task.data);

  if ( this.conf.makeendpoints ) {
    var k = { path: this.conf.out.path, key: "start" + this.conf.out.key };
    jsonPathHelper.set(k, this.ep.start , task.data);
    k.key = "end" + this.conf.out.key;
    jsonPathHelper.set(k, this.ep.end , task.data);
  }
  if ( this.conf.mergeorlist ) {
    var k = { path: this.conf.out.path, key: this.conf.out.key + "values" };
    jsonPathHelper.set(k, this.orvalues , task.data);

  }
};


///////////////////
// Unit test
// This is too messy to test with real connectors (although such testing
// is also needed)

Fullquerylimit.prototype.unitTest = function ( ) {
  const defaultsettings = {  // These can be overwritten in the tests
      field:"year",
      ranges: true,
      makeendpoints: true,
      orvalues: false,
      mergeorlist: false,
      remove: true,
      failnomatch: false,
      failremain: false,
  };

  const tests = [
    { name: "Simple",
      override: {}, // no overrides to default settings
      lines: [
        // nothing found
        { qry: '{"term":"water"}', // original query
          exp: '[]' ,              // list of limit fields expected
          fin: '{"term":"water"}', // final query (defaults to original)
          start: '',  // expected endpoints
          end: '',
        },
        // One limit term in simple and
        { qry: '{"op":"and", ' +
            '"s1": { "term":"water" }, ' +
            '"s2": { "term":"2012", "field":"year" } }',
          exp: '[{"term":"2012","field":"year"}]',
          fin: '{"term":"water"}',
          start: '2012',
          end: '2012',
        },
        { qry: '{"op":"and", ' +
           '"s1": { "term":"2012", "field":"year" }, ' +
           '"s2": { "term":"water" } }',
          exp: '[{"term":"2012","field":"year"}]',
          fin: '{"term":"water"}',
          start: '2012',
          end: '2012',
        },
        // or-search, not a limit
        { qry: '{"op":"or", ' +
            '"s1": { "term":"water" }, ' +
            '"s2": { "term":"2012", "field":"year" } }',
          exp: '[]',
          fin: '{"op":"or","s1":{"term":"water"},'+
                          '"s2":{"term":"2012","field":"year"}}',
          start: '',
          end: '',
        },
        { qry: '{"op":"or", ' +
           '"s1": { "term":"2012", "field":"year" }, ' +
           '"s2": { "term":"water" } }',
          exp: '[]',
          fin: '{"op":"or","s1":{"term":"2012","field":"year"},'+
                           '"s2":{"term":"water"}}',
          start: '',
          end: '',
        },
      ]
    },
    { name: "Noremove",
      override: { remove:false },
      lines: [
        // nothing found
        { qry: '{"term":"water"}', // original query
          exp: '[]' ,              // list of limit fields expected
             // in this set, we always expect the unmodified query back
        },
        // One limit term in simple and
        { qry: '{"op":"and", ' +
            '"s1": { "term":"water" }, ' +
            '"s2": { "term":"2012", "field":"year" } }',
          exp: '[{"term":"2012","field":"year"}]',
          start: '2012',
          end: '2012',
        },
        { qry: '{"op":"and", ' +
           '"s1": { "term":"2012", "field":"year" }, ' +
           '"s2": { "term":"water" } }',
          exp: '[{"term":"2012","field":"year"}]',
        },
        // or-search, not a limit
        { qry: '{"op":"or", ' +
            '"s1": { "term":"water" }, ' +
            '"s2": { "term":"2012", "field":"year" } }',
          exp: '[]',
        },
        { qry: '{"op":"or", ' +
           '"s1": { "term":"2012", "field":"year" }, ' +
           '"s2": { "term":"water" } }',
          exp: '[]',
        },
      ]
    },
    { name: "Range",
      override: {}, // no overrides to default settings
      lines: [
        // two limit terms in simple and
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"and", ' +
                   '"s1": { "term":"2001", "field":"year", "relation":"ge" }, '+
                   '"s2": { "term":"2009", "field":"year", "relation":"le" } }}',
          exp: '[{"term":"2001","field":"year","relation":"ge","op":"and"},'+
                '{"term":"2009","field":"year","relation":"le","op":"and"}]',
          fin: '{"term":"water"}',
          start: '2001',
          end: '2009',
        },
        // Limit terms around a real term
        { qry: '{"op":"and", ' +
           '"s1": { "term":"2001", "field":"year", "relation":"ge" }, ' +
           '"s2": { "op":"and", ' +
                   '"s1": { "term":"water" }, '+
                   '"s2": { "term":"2009", "field":"year", "relation":"le" } }}',
          exp: '[{"term":"2001","field":"year","relation":"ge"},'+
                '{"term":"2009","field":"year","relation":"le","op":"and"}]',
          fin: '{"term":"water"}',
          start: '2001',
          end: '2009',
        },
        // Limit terms around a simple term
        { qry: '{"op":"and", ' +
           '"s1": { "term":"2001", "field":"year", "relation":"ge" }, ' +
           '"s2": { "op":"and", ' +
                   '"s1": { "term":"2009", "field":"year", "relation":"le" }, '+
                   '"s2": { "term":"water" } }}',
          exp: '[{"term":"2001","field":"year","relation":"ge"},'+
                '{"term":"2009","field":"year","relation":"le","op":"and"}]',
          fin: '{"term":"water"}',
          start: '2001',
          end: '2009',
        },
        // limit with gt lt
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"and", ' +
                   '"s1": { "term":"2000", "field":"year", "relation":"gt" }, '+
                   '"s2": { "term":"2010", "field":"year", "relation":"lt" } }}',
          exp: '[{"term":"2000","field":"year","relation":"gt","op":"and"},'+
                '{"term":"2010","field":"year","relation":"lt","op":"and"}]',
          fin: '{"term":"water"}',
          start: '2001',
          end: '2009',
        },
      ],
    },
    { name: "NoRange",
      override: { 'ranges': false, remove:false }, 
      lines: [
        // two limit terms in simple and
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"and", ' +
                   '"s1": { "term":"2001", "field":"year", "relation":"ge" }, '+
                   '"s2": { "term":"2009", "field":"year", "relation":"le" } }}',
          exp: "Exception! Ranges not supported for year",
        },
        { qry: '{"op":"and", ' +
            '"s1": { "term":"water" }, ' +
            '"s2": { "term":"2012", "field":"year" } }',
          exp: '[{"term":"2012","field":"year"}]',
          start: '2012',
          end: '2012',
        },
      ],
    },
    { name: "DefaultValue",
      override: { makedefault: true, defaultvalue:"2000"},
      lines: [
        // nothing found
        { qry: '{"term":"water"}',
          exp: '[{"field":"year","term":"2000","op":""}]' ,
          start: '2000',
          end: '2000',
        },
        // One limit term in simple and - make no use of default
        { qry: '{"op":"and", ' +
            '"s1": { "term":"water" }, ' +
            '"s2": { "term":"2012", "field":"year" } }',
          exp: '[{"term":"2012","field":"year"}]',
          fin: '{"term":"water"}',
          start: '2012',
          end: '2012',
        },
        // or-search, not a limit. So make a default limit
        { qry: '{"op":"or", ' +
            '"s1": { "term":"water" }, ' +
            '"s2": { "term":"2012", "field":"year" } }',
          exp: '[{"field":"year","term":"2000","op":""}]',
          fin: '{"op":"or","s1":{"term":"water"},'+
                          '"s2":{"term":"2012","field":"year"}}',
          start: '2000',
          end: '2000',
        },
      ]
    },

    { name: "MakeRange",
      override: { makerange:true,makedefault: true, defaultvalue:"2000"  },
      lines: [
        // One limit term, force it into a range
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "term":"2001", "field":"year" } } ',
          exp: '[{"field":"year","term":"2001","op":"","relation":"ge"},'+
                '{"field":"year","term":"2001","op":"and","relation":"le"}]',
          fin: '{"term":"water"}',
          start: '2001',
          end: '2001',
        },
        // no limit, use default, make range
        { qry: '{"term":"water"}',
          exp: '[{"field":"year","term":"2000","op":"","relation":"ge"},'+
                '{"field":"year","term":"2000","op":"and","relation":"le"}]' ,
          start: '2000',
          end: '2000',
        },
        // Only one term with relation, no range
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "term":"2001", "field":"year", "relation":"le" } } ',
          exp:  '[{"term":"2001","field":"year","relation":"le"}]' ,
          fin: '{"term":"water"}',
          start: '',
          end: '2001',
        },
      ],
    },


    { name: "Fail",
      override: { failnomatch:true, failremain: true },
      lines: [
        // nothing found, should fail
        { qry: '{"term":"water"}', // original query
          exp: "Exception! No querylimit 'year' found" ,
          fin: '{"term":"water"}'  // final query
        },
        // finds a limit, will not fail
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "term":"2012", "field":"year" } }',
          exp: '[{"term":"2012","field":"year"}]',
          fin: '{"term":"water"}'
        },
        // Or-query, not a limit, so it fails
        { qry: '{"op":"or", ' +
          '"s1": { "term":"water" }, ' +
          '"s2": { "term":"2012", "field":"year" } }',
          exp: "Exception! No querylimit 'year' found",
          fin: '{"op":"or","s1":{"term":"water"},"s2":{"term":"2012","field":"year"}}'
        },
        { // limit, but same field inside an OR. Fails with failremain
          qry: '{"op":"and", ' +
             '"s1": { "term":"2001", "field":"year" }, ' +
             '"s2": { "op":"or", ' +
               '"s1": { "term":"2009", "field":"year", "relation":"le" }, '+
               '"s2": { "term":"water" } }}',
          //  '[{"term":"2001","field":"year"}]',
          exp: "Exception! Limit field 'year' still remains after removing queryfilter",
          fin: '{"op":"or","s1":{"term":"2009","field":"year","relation":"le"},'+
                          '"s2":{"term":"water"}}',
        },
      ]
    },

    { name: "FailNoRemove",
      override: { remove:false, failnomatch:true, failremain: true },
      lines: [
        // nothing found, should fail
        { qry: '{"term":"water"}', // original query
          exp: "Exception! No querylimit 'year' found" ,
        },
        // finds a limit, but does not remove it, so it remains, causing failure
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "term":"2012", "field":"year" } }',
          exp: "Exception! Limit field 'year' still remains after removing queryfilter",
        },
        // Or-query, not a limit, so it fails
        { qry: '{"op":"or", ' +
          '"s1": { "term":"water" }, ' +
          '"s2": { "term":"2012", "field":"year" } }',
          exp: "Exception! No querylimit 'year' found",
        },
        { // limit, but same field inside an OR.
          // Fails with failremain,
          qry: '{"op":"and", ' +
             '"s1": { "term":"2001", "field":"year" }, ' +
             '"s2": { "op":"or", ' +
               '"s1": { "term":"2009", "field":"year", "relation":"le" }, '+
               '"s2": { "term":"water" } }}',
          //  '[{"term":"2001","field":"year"}]',
          exp: "Exception! Limit field 'year' still remains after removing queryfilter",
        },
      ]
    },

    { name: "Endpoints",
      override: {}, // no overrides to default settings
      lines: [
        // two limit terms in simple and
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"and", ' +
                   '"s1": { "term":"2001", "field":"year", "relation":"ge" }, '+
                   '"s2": { "term":"2009", "field":"year", "relation":"le" } }}',
          exp: '[{"term":"2001","field":"year","relation":"ge","op":"and"},'+
                '{"term":"2009","field":"year","relation":"le","op":"and"}]',
          fin: '{"term":"water"}',
          start: '2001',
          end: '2009',
        },
        // two limit terms, both for start
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"and", ' +
                   '"s1": { "term":"2001", "field":"year", "relation":"ge" }, '+
                   '"s2": { "term":"2005", "field":"year", "relation":"ge" } }}',
          exp: '[{"term":"2001","field":"year","relation":"ge","op":"and"},'+
                '{"term":"2005","field":"year","relation":"ge","op":"and"}]',
          fin: '{"term":"water"}',
          start: '2005',
          end: '',
        },
        // two limits terms, both for end
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"and", ' +
                   '"s1": { "term":"2001", "field":"year", "relation":"le" }, '+
                   '"s2": { "term":"2005", "field":"year", "relation":"le" } }}',
          exp: '[{"term":"2001","field":"year","relation":"le","op":"and"},'+
                '{"term":"2005","field":"year","relation":"le","op":"and"}]',
          fin: '{"term":"water"}',
          start: '',
          end: '2001',
        },
        // two limits terms, le and eq
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"and", ' +
                   '"s1": { "term":"2005", "field":"year", "relation":"le" }, '+
                   '"s2": { "term":"2001", "field":"year", "relation":"eq" } }}',
          exp: '[{"term":"2005","field":"year","relation":"le","op":"and"},'+
                '{"term":"2001","field":"year","relation":"eq","op":"and"}]',
          fin: '{"term":"water"}',
          start: '2001',
          end: '2001',
        },
        // two limits terms, eq and eq, same value
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"and", ' +
                   '"s1": { "term":"2005", "field":"year", "relation":"eq" }, '+
                   '"s2": { "term":"2005", "field":"year", "relation":"eq" } }}',
          exp: '[{"term":"2005","field":"year","relation":"eq","op":"and"},'+
                '{"term":"2005","field":"year","relation":"eq","op":"and"}]',
          fin: '{"term":"water"}',
          start: '2005',
          end: '2005',
        },
        // two limits terms, eq and eq, different value
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"and", ' +
                   '"s1": { "term":"2001", "field":"year", "relation":"eq" }, '+
                   '"s2": { "term":"2005", "field":"year", "relation":"eq" } }}',
          exp: "Exception! Impossible range 2005-2001",
          fin: '{"term":"water"}',
          start: '2005',  // Note, these will not be set in the
          end: '2001', // task variables, but are visible to the test
        },
        // three limits terms, eq value and le-ge range
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"and", ' +
             '"s1": { "term":"2005", "field":"year", "relation":"eq" }, ' +
             '"s2": { "op":"and", ' +
                   '"s1": { "term":"2001", "field":"year", "relation":"ge" }, '+
                   '"s2": { "term":"2009", "field":"year", "relation":"le" } }}}',
          exp: '[{"term":"2005","field":"year","relation":"eq","op":"and"},'+
                '{"term":"2001","field":"year","relation":"ge","op":"and"},'+
                '{"term":"2009","field":"year","relation":"le","op":"and"}]',
          fin: '{"term":"water"}',
          start: '2005',
          end: '2005',
        },
        // three limits terms, le-ge range and eq value
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"and", ' +
             '"s1": { "op":"and", ' +
                   '"s1": { "term":"2001", "field":"year", "relation":"ge" }, '+
                   '"s2": { "term":"2009", "field":"year", "relation":"le" }}, '+
             '"s2": { "term":"2005", "field":"year", "relation":"eq" }  }}',
          exp: '[{"term":"2001","field":"year","relation":"ge","op":"and"},'+
                '{"term":"2009","field":"year","relation":"le","op":"and"},'+
                '{"term":"2005","field":"year","relation":"eq","op":"and"}]',
          fin: '{"term":"water"}',
          start: '2005',
          end: '2005',
        },
        // Two overlapping ranges
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"and", ' +
             '"s1": { "op":"and", ' +
                   '"s1": { "term":"2001", "field":"year", "relation":"ge" }, '+
                   '"s2": { "term":"2007", "field":"year", "relation":"le" }}, '+
             '"s2": { "op":"and", ' +
                   '"s1": { "term":"2003", "field":"year", "relation":"ge" }, '+
                   '"s2": { "term":"2009", "field":"year", "relation":"le" }} }}',
          exp: '[{"term":"2001","field":"year","relation":"ge","op":"and"},'+
                '{"term":"2007","field":"year","relation":"le","op":"and"},'+
                '{"term":"2003","field":"year","relation":"ge","op":"and"},'+
                '{"term":"2009","field":"year","relation":"le","op":"and"}]',
          fin: '{"term":"water"}',
          start: '2003',
          end: '2007',
        },
        // Two ranges, not overlapping
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"and", ' +
             '"s1": { "op":"and", ' +
                   '"s1": { "term":"2001", "field":"year", "relation":"ge" }, '+
                   '"s2": { "term":"2003", "field":"year", "relation":"le" }}, '+
             '"s2": { "op":"and", ' +
                   '"s1": { "term":"2007", "field":"year", "relation":"ge" }, '+
                   '"s2": { "term":"2009", "field":"year", "relation":"le" }} }}',
          exp: "Exception! Impossible range 2007-2003",
          fin: '{"term":"water"}',
          start: '2007', // Note! These will not be set into task.data
          end: '2003', // because of the exception, but are visible to the test
        },
        // Simple or-bag, which is not supported in this test set.
        // So, this is not a limit at all!
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"or", ' +
                   '"s1": { "term":"2001", "field":"year" }, '+
                   '"s2": { "term":"2009", "field":"year" } }}',
          exp: '[]',
          // no fin:, expect the unmodified query
          start: '',
          end: '',
        },
      ],
    },

    { name: "Endpointdefault",
      override: { makeendpoints:true, startdef:"1900", enddef:"2999" },
      //override: { makeendpoints:true, startdef:"YYYY", enddef:"YYYY" },
      // This can be used for a manual test, the first will fail, but
      // you can see that endpoints end up with current year
      lines: [
        // no year
        { qry:' { "term":"water" }',
          exp: '[]',
          fin: '{"term":"water"}',
          start: '1900',
          end: '2999',
        },
        // two limit terms in simple and
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"and", ' +
                   '"s1": { "term":"2001", "field":"year", "relation":"ge" }, '+
                   '"s2": { "term":"2009", "field":"year", "relation":"le" } }}',
          exp: '[{"term":"2001","field":"year","relation":"ge","op":"and"},'+
                '{"term":"2009","field":"year","relation":"le","op":"and"}]',
          fin: '{"term":"water"}',
          start: '2001',
          end: '2009',
        },
        // ge
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "term":"2001", "field":"year", "relation":"ge" } }',
          exp: '[{"term":"2001","field":"year","relation":"ge"}]',
          fin: '{"term":"water"}',
          start: '2001',
          end: '2999',
        },
        // le
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "term":"2009", "field":"year", "relation":"le" } }',
          exp: '[{"term":"2009","field":"year","relation":"le"}]',
          fin: '{"term":"water"}',
          start: '1900',
          end: '2009',
        },
        // eq
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "term":"2005", "field":"year", "relation":"eq" } }',
          exp: '[{"term":"2005","field":"year","relation":"eq"}]',
          fin: '{"term":"water"}',
          start: '2005',
          end: '2005',
        },
      ],
    }, // Endpointdefault
    
    { name: "LimitEndpoint",
      override: { makeendpoints:true,  limitendpoints: true,
                  startdef:"1993", enddef:"2003" },
      lines: [
        // no year
        { qry:' { "term":"water" }',
          exp: '[]',
          fin: '{"term":"water"}',
          start: '1993',
          end: '2003',
        },
        // two limit terms in simple and
        { qry: '{"op":"and", ' +
            '"s1": { "term":"water" }, ' +
            '"s2": { "op":"and", ' +
              '"s1": { "term":"1900", "field":"year", "relation":"ge" }, '+
              '"s2": { "term":"2999", "field":"year", "relation":"le" } }}',
          exp: '[{"term":"1900","field":"year","relation":"ge","op":"and"},'+
            '{"term":"2999","field":"year","relation":"le","op":"and"}]',
          fin: '{"term":"water"}',
          start: '1993',
          end: '2003',
        },
        // ge
        { qry: '{"op":"and", ' +
          '"s1": { "term":"water" }, ' +
          '"s2": { "term":"1900", "field":"year", "relation":"ge" } }',
          exp: '[{"term":"1900","field":"year","relation":"ge"}]',
          fin: '{"term":"water"}',
          start: '1993',
          end: '2003',
        },
        // le
        { qry: '{"op":"and", ' +
            '"s1": { "term":"water" }, ' +
            '"s2": { "term":"2999", "field":"year", "relation":"le" } }',
          exp: '[{"term":"2999","field":"year","relation":"le"}]',
          fin: '{"term":"water"}',
          start: '1993',
          end: '2003',
        },
        // eq
        { qry: '{"op":"and", ' +
            '"s1": { "term":"water" }, ' +
            '"s2": { "term":"2001", "field":"year", "relation":"eq" } }',
          exp: '[{"term":"2001","field":"year","relation":"eq"}]',
          fin: '{"term":"water"}',
          start: '2001',
          end: '2001',
        },
        // Range too early
        { qry: '{"op":"and", ' +
            '"s1": { "term":"water" }, ' +
            '"s2": { "op":"and", ' +
              '"s1": { "term":"1900", "field":"year", "relation":"ge" }, '+
              '"s2": { "term":"1910", "field":"year", "relation":"le" } }}',
          exp: "Exception! Bad value for year. Must be in 1993-2003",
          fin: '{"term":"water"}',
        },
        // Range too late
        { qry: '{"op":"and", ' +
          '"s1": { "term":"water" }, ' +
          '"s2": { "op":"and", ' +
            '"s1": { "term":"2050", "field":"year", "relation":"ge" }, '+
            '"s2": { "term":"2060", "field":"year", "relation":"le" } }}',
          exp: "Exception! Bad value for year. Must be in 1993-2003",
          fin: '{"term":"water"}',
    },
        ],
    }, // LimitEndpoint
    
    { name: "OR-bags",
      override: { orvalues: true, mergeorlist: true },
      lines: [
        // Simple or-bag with two values
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"or", ' +
                   '"s1": { "term":"2001", "field":"year" }, '+
                   '"s2": { "term":"2009", "field":"year" } }}',
          exp: '[{"term":"2001","field":"year","op":"or"},'+
                '{"term":"2009","field":"year","op":"or"}]',
          fin: '{"term":"water"}',
          start: '', // Not a range!
          end: '',
          orvalues: '2001 2009',
        },
        // non-equal relation, is not a proper or-bag
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"or", ' +
                   '"s1": { "term":"2001", "field":"year", "relation":"ge" }, '+
                   '"s2": { "term":"2009", "field":"year" } }}',
          exp: '[]',
          start: '',
          end: '',
          orvalues: '',
        },
        // ( a or b ) or ( c or d )
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"or", ' +
             '"s1": { "op":"or", ' +
                   '"s1": { "term":"2001", "field":"year" }, '+
                   '"s2": { "term":"2003", "field":"year" }}, '+
             '"s2": { "op":"or", ' +
                   '"s1": { "term":"2007", "field":"year" }, '+
                   '"s2": { "term":"2009", "field":"year" }} }}',
          exp: '[{"term":"2001","field":"year","op":"or"},' +
                '{"term":"2003","field":"year","op":"or"},' +
                '{"term":"2007","field":"year","op":"or"},' +
                '{"term":"2009","field":"year","op":"or"}]',
          orvalues: '2001 2003 2007 2009',
          fin: '{"term":"water"}',
          start: '',
          end: '',
        },
        // ( a or ( b or ( c or d ) ) )
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"or", ' +
             '"s1": { "term":"2001", "field":"year" }, '+
             '"s2": { "op":"or", ' +
                   '"s1": { "term":"2003", "field":"year" }, '+
                   '"s2": { "op":"or", ' +
                     '"s1": { "term":"2007", "field":"year" }, '+
                     '"s2": { "term":"2009", "field":"year" }} }}}',
          exp: '[{"term":"2001","field":"year","op":"or"},' +
                '{"term":"2003","field":"year","op":"or"},' +
                '{"term":"2007","field":"year","op":"or"},' +
                '{"term":"2009","field":"year","op":"or"}]',
          orvalues: '2001 2003 2007 2009',
          fin: '{"term":"water"}',
          start: '',
          end: '',
        },
        // ( a or ( b or ( c or d ) ) )
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"or", ' +
             '"s1": { "op":"or", ' +
                   '"s1": { "op":"or", ' +
                     '"s1": { "term":"2001", "field":"year" }, '+
                     '"s2": { "term":"2003", "field":"year" }},' +
                   '"s2": { "term":"2007", "field":"year" } },' +
             '"s2": { "term":"2009", "field":"year" } }}',
          exp: '[{"term":"2001","field":"year","op":"or"},' +
                '{"term":"2003","field":"year","op":"or"},' +
                '{"term":"2007","field":"year","op":"or"},' +
                '{"term":"2009","field":"year","op":"or"}]',
          orvalues: '2001 2003 2007 2009',
          fin: '{"term":"water"}',
          start: '',
          end: '',
        },
        // ( a or b ) or ( c AND d )  - not a limit at all!
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"or", ' +
             '"s1": { "op":"or", ' +
                   '"s1": { "term":"2001", "field":"year" }, '+
                   '"s2": { "term":"2003", "field":"year" }}, '+
             '"s2": { "op":"and", ' +
                   '"s1": { "term":"2007", "field":"year" }, '+
                   '"s2": { "term":"2009", "field":"year" }} }}',
          exp: '[]',
          orvalues: '',
          start: '',
          end: '',
        },
        // ( a or b ) AND ( c or d )  - not a limit at all!
        { qry: '{"op":"and", ' +
           '"s1": { "term":"water" }, ' +
           '"s2": { "op":"and", ' +
             '"s1": { "op":"or", ' +
                   '"s1": { "term":"2001", "field":"year" }, '+
                   '"s2": { "term":"2003", "field":"year" }}, '+
             '"s2": { "op":"or", ' +
                   '"s1": { "term":"2007", "field":"year" }, '+
                   '"s2": { "term":"2009", "field":"year" }} }}',
          exp: '[]',
          start: '',
          end: '',
          orvalues: '',
        },
      ],
    },

    // TODO - A test set with OR queries, with limiters here and there
  ];

  logger.info("Starting unit test for Fullquerylimit");
  // fake some runtime
  var connector = new Connector();
  var task = new Task( connector, "unittest" );
  for ( var i=0; i<tests.length; i++) {
    var t=tests[i];
    var fql = new Fullquerylimit;
    fql.task = task;
    for ( var f in defaultsettings ) {
      fql.conf[f] = defaultsettings[f];
    }
    for ( var f in t.override ) {
      fql.conf[f] = t.override[f];
    }
    for ( var lno = 0; lno<t.lines.length; lno++) {
      var fq = t.lines[lno].qry;
      var expect = t.lines[lno].exp;
      var fullquery;
      var finalquery = t.lines[lno].fin;
      try {
        fullquery = JSON.parse(fq);
        fq = JSON.stringify(fullquery); // normalize it
      } catch(e) {
        logger.error("Invalid json in unit test '" + t.name + "'." + lno  );
        logger.error(fq);
        logger.error(e);
        if ( e.fileName  )  // catch syntax errors in code, etc
          logger.error("  in " + e.fileName + "." + e.lineNumber  );
        return false;
      }
      fql.limitnodes = [];
      var result = null;
      try {
        fql.rootnode(fullquery);
        result = JSON.stringify(fql.limitnodes);
      } catch (e) {
        result = "Exception! " + e.message ;
        if ( e.fileName )
          result += " in " + e.fileName + ":" + e.lineNumber;
      }
      if ( result != expect ) {
        logger.error("Test " + t.name + "." + lno + " FAILED");
        logger.error("  result   " + result );
        logger.error("  Expexted " + expect );
        return false;
      }
      var finalexp;
      if ( t.lines[lno].fin )
        finalexp = t.lines[lno].fin; // expect query with stuff removed
      else
        finalexp = fq; // expect original query back
      if ( finalexp != JSON.stringify(fullquery) ) {
        logger.error("Test " + t.name + "." + lno + " FAILED");
        logger.error("  result ok: " + result );
        logger.error("  Final query " + JSON.stringify(fullquery) );
        logger.error("  Expexted    " + finalexp);
        return false;
      }
      if ( ( typeof(t.lines[lno].start) != "undefined" &&
              t.lines[lno].start != fql.ep.start ) ||
            ( typeof(t.lines[lno].end) != "undefined" &&
              t.lines[lno].end != fql.ep.end ) ) {
        logger.error("Test " + t.name + "." + lno + " FAILED");
        logger.error("  Got endpoints '" + fql.ep.start +
                                   "' - '" + fql.ep.end +"'");
        logger.error("  Expected      '" + t.lines[lno].start +
                                   "' - '" + t.lines[lno].end +"'");
        return false;
      }
      if ( typeof(t.lines[lno].orvalues ) != "undefined" &&
              t.lines[lno].orvalues != fql.orvalues ) {
        logger.error("Test " + t.name + "." + lno + " FAILED");
        logger.error("  Got orvalues '" + fql.orvalues  + "'");
        logger.error("  Expected     '" + t.lines[lno].orvalues + "'");
        return false;
      }
      logger.info("Test " + t.name + "." + lno +  " OK");
    }
  }
  return true;
}

///////////////////
// Housekeeping

Fullquerylimit.prototype.getUsedArgs = function () {
    if (this.conf.in && this.conf.in.path === "$.input")
        return [this.conf.in.key];
    else
        return [];
};

Fullquerylimit.prototype.getClassName = function () {
    return "Fullquerylimit";
};

Fullquerylimit.prototype.getDisplayName = function () {
    return "Fullquerylimit";
};

Fullquerylimit.prototype.getDescription = function () {
  return "Translate a complex query into a search string";
};

Fullquerylimit.prototype.getVersion = function () {
  //return "1.2";  // 1.2 adds start/end defaults
  //return "1.3";  // 1.3 actually honors the "support ranges" box, when off.
  return "2.0"; // 2.0 adds the "limit range to defaults check"
};

// Not much point in a renderArgs, way too complex to show in the builder
Fullquerylimit.prototype.renderArgs = function () {
    return this.conf["field"];
}

Fullquerylimit.prototype.capabilityFlagDefault = function ( flag ) {
    if (flag.substring(0, 6) == "index-") {
        var index = flag.substring(6);
        if ( this.conf['field'] == index )
          return true;
        if ( this.conf['field'] == "year" &&
             this.conf['ranges'] &&
             ( index == "startyear" || index == "endyear" ) )
          return true;
    }
    return null;
}

Fullquerylimit.prototype.upgrade = function (confVer, curVer, conf) {
    // can't upgrade if the connector is newer than the step
    if (confVer > curVer)
        return false;
    if (conf.makeendpoints === undefined )
      conf.makeendpoints = false;
    if (conf.limitendpoints === undefined )
      conf.limitendpoints = false;    
    return true;
};

