var EXPORTED_SYMBOLS = ["Http"];

// A step that can be a HTTP client for various web services
// Especially a SOAP client to Elsevier's search service, and
// hopefully others as well.

// This is the second version of the http client, one that always just
// fetches a string from the HTTP server. Parsing it into DOM or elsewhere
// is left for another step, for example renderXML.

// TODO  Checkbox "Wait for load"

Components.utils.import('resource://indexdata/runtime/Task.js');
Components.utils.import('resource://indexdata/runtime/Step.js');
Components.utils.import('resource://indexdata/runtime/StepError.js');
Components.utils.import('resource://indexdata/util/xulHelper.js');
Components.utils.import('resource://indexdata/util/xmlHelper.js');
Components.utils.import('resource://indexdata/util/textHelper.js');
Components.utils.import('resource://indexdata/util/jsonPathHelper.js');
Components.utils.import('resource://indexdata/util/htmlHelper.js');
Components.utils.import('resource://indexdata/util/logging.js');
Components.utils.import('resource://indexdata/thirdparty/jsonPath.js');
Components.utils.import('resource://indexdata/runtime/core.js');
var Cc = Components.classes
var Ci = Components.interfaces

var logger = logging.getLogger();


var Http = function () {
  this.className = "Http";
  this.conf = {};
  this.conf['type'] = "GET";
  this.conf['url'] = "";  // constant url
  this.conf['urljp'] = { path:'$.temp', key: 'url' };
  this.conf['useconst'] = false; // this defines which url to use, constant or $var
  this.conf['failonerror'] = true; 
  
  this.conf['reqjp'] = ""; // variable to take the request from
  this.conf['hdrjp'] = ""; // extra headers jp
  
  this.conf['out'] = { path:'$.temp', key: 'response',
                         append: jsonPathHelper.REPLACE };
  this.conf['respheaders'] = false;
  this.conf['resphdrjp'] = { path:'$.temp', key: 'responseheaders',
                          append: jsonPathHelper.REPLACE };
};

Http.prototype = new Step();
Http.prototype.constructor = Http;
Http.prototype.init = function() {};

const types = { "POST": "POST",
                "GET": "GET",
                "PUT": "PUT" };
// More to be added in some later version


//////////////////
// Draw
Http.prototype.draw = function(surface,win) {

  var tabs = xulHelper.tabBox( surface,
          [ "General", "Request", "Response" ],  
          { flex: 1 } );
  this.drawGeneralTab( tabs[0], this, tabs);
  this.drawRequestTab( tabs[1], this, tabs);
  this.drawResponseTab( tabs[2], this, tabs);

};

Http.prototype.drawGeneralTab = function (surface, context, tabs) {
  var context = this;

  
  var vb = xmlHelper.appendNode(surface, "vbox", null,
           {"flex":1,"align":"stretch"} );

  var typesel = xulHelper.selectField(vb, context, "type",
            "Service type", types, {} );

  var urlJpField = xulHelper.jsonPathField(vb, this, this.conf,
           "urljp", "URL", null,
              { rwmode:"r", singlevalue: true,
                constantallowed: true } );
  
  var hdrCheck = xulHelper.checkbox(vb, this, "failonerror",
                                    "Fail on error" );
}; // drawGeneralTab


Http.prototype.drawRequestTab = function (surface, context, tabs) {
  var vb = xmlHelper.appendNode(surface, "vbox", null,
                                {"flex":1,"align":"stretch"} );
  var urlJpField = xulHelper.jsonPathField(vb, this, this.conf,
             "reqjp", "POST data", null,
              { rwmode:"r", singlevalue: true,
                nulltext: "(No data)", nullmenutext: "Nothing" } );             
  var hdrJpField = xulHelper.jsonPathField(vb, this, this.conf, "hdrjp", 
              "Add custom headers", null,
              { rwmode:"r", singlevalue: true,
                nulltext: "Do not add any", nullmenutext: "Nothing" } );
};

Http.prototype.drawResponseTab = function (surface, context, tabs) {
  var vb = xmlHelper.appendNode(surface, "vbox", null,
                                {"flex":1,"align":"stretch"} );
  var outField = xulHelper.jsonPathField(vb, this, this.conf,
               "out", "Put the response in", null,
              { rwmode: "w", singlevalue: true,
                nulltext: "Do not save", nullmenutext: "Nowhere" } );
  var outField = xulHelper.jsonPathField(vb, this, this.conf, "resphdrjp",
          "Save response headers",null,
          { rwmode: "w", singlevalue: true,
            nulltext: "Do not save", nullmenutext: "Nowhere" } );
};

//////////////////
// Run


Http.prototype.getUrl = function (task) {
  logger.debug("http.getUrl: " + JSON.stringify(this.conf['urljp']) );
  var url = jsonPathHelper.getFirst(this.conf['urljp'], task.data);
  if (!url) {
    var disp = jsonPathHelper.displayString(this.conf['urljp']);
    throw new StepError("URL " + disp + " not specified (value was: " + url+")");
  }
  return url;
};

Http.prototype.getHeaders = function (task) {
  var headers = {};
  var hstr = jsonPathHelper.getFirst(this.conf['hdrjp'], task.data);
    // can return null if no hdrjp in conf, or if it has no path
  logger.debug("http: hdrjp='" + JSON.stringify(this.conf['hdrjp'])+"'" );
  logger.debug("http: hstr='" + hstr + "' " + JSON.stringify(hstr) );
  if ( !hstr )
    return {}; // no headers to add
  var harr = hstr.split("\n");
  for ( var i in harr ) {
    var hline = harr[i];
    hline = hline.replace( /^#.*$/, "" ); // skip comment lines
    if ( hline ) {
      var m = hline.match( /^([^:]+)\s*:\s*(.*)$/ );
      if ( m ) {
        var k = m[1];
        var v = m[2];
        headers[k]=v;
      } else {
        throw new StepError ("Invalid header line '" + hline + "'");
      }
    }
  }
  return headers;
};


Http.prototype.run = function (task) {
  var url=this.getUrl(task);
  var msgurl = url.replace( /[\?\&].*$/,""); // remove params.
    // Makes shorter log messages, and does not expose passwords
    // as these can propagate all the way to real users.
  var reqtype = this.conf['type'] ;

  var extraHeaders = this.getHeaders(task);

  var postContent = null; // default to no data, as in GET
  
  if ( reqtype == "POST" || reqtype == "PUT" ) {
    var reqjp = this.conf['reqjp'];
    if ( ! reqjp )
      throw new StepError("No request specified");
    postContent = jsonPathHelper.getFirst(reqjp, task.data);
    if ( typeof(postContent) != "string" ) {
      logger.debug("Bad POST content. Type= " + typeof(postContent) + 
        ": '" + postContent + "'" );
      throw new StepError("POST content not a string");
    }
    if ( ! postContent ) {
      logger.debug("POST with empty content??");
      postContent = "";
    }
    if ( ! extraHeaders['Content-Type'] ) {
      extraHeaders['Content-Type'] = "application/soap+xml; charset=utf-8";
      logger.debug("Defaulting 'Content-Type' to '"+
                            "application/soap+xml; charset=utf-8'");
    }
  } // post

  // We have a fetchXML in XmlHelper, but it is so full of special
  // cases that it is easier to do things directly here
  var httpreq = Cc["@mozilla.org/xmlextras/xmlhttprequest;1"]
        .createInstance(Ci.nsIXMLHttpRequest);
  httpreq.overrideMimeType('text/plain');

  httpreq.open(reqtype, url, false);  // false = synchronous
  
  for ( var h in extraHeaders ) {
    logger.debug("Setting request header '" + h + "' to '"
        + extraHeaders[h] + "'");
    httpreq.setRequestHeader( h, extraHeaders[h] );
  }

  try {
    logger.debug("just about to " + reqtype );
    httpreq.send(postContent);
  } catch (e) {
    var msg = "Failed to " + reqtype + " " + msgurl + " : " + e.message;
    logger.info(msg);
    if ( this.conf.failonerror )
      throw new Error( msg );
    if ( this.conf.respheaders ) { // fake response headers
      var hdrs = {
        "HTTP_STATUS" : "503", // Service Unavailable. Maybe use 518 instead?
        "HTTP_STATUS_TEXT": msg  };
      jsonPathHelper.set(this.conf["resphdrjp"], hdrs, task.data);
    }
    return; 
  } // catch
  
  logger.debug("status: " + httpreq.status );

  if ( httpreq.status != 0 && httpreq.status != 200 && this.conf.failonerror) {
    logger.debug("Failure status " + httpreq.status + ": ");
    logger.debug("Headers: " + httpreq.getAllResponseHeaders() );
    if ( httpreq.responseText )
      logger.debug("Response: " + httpreq.responseText );
    throw new Error ("Error " + httpreq.status + " " +
       reqtype + " " + msgurl );
  }
  var txt = httpreq.responseText;
  logger.debug("http: got response: " + txt.substring(0,60) );
  jsonPathHelper.set(this.conf.out, [txt], task.data);
  
  if ( this.conf.respheaders ) {
    var h = { "HTTP_STATUS" : httpreq.status,
              "HTTP_STATUS_TEXT" : httpreq.statusText };
    var allheaders =  httpreq.getAllResponseHeaders();
    var hdrarr = allheaders.split("\n");
    for ( var i = 0; i < hdrarr.length; i++ ) {
      logger.debug("looking at header " + i + " '" + hdrarr[i] + "'" );
      // hdrarr[i]= hdrarr[i].replace( /Con/, "Con: ");  // test to force multiple 
      // headers with the same tag. 
      var m = hdrarr[i].match( /^([^:]+)(.*)/ );
      if (m) {
        var k = m[1];
        logger.debug(" k=" + k + " m=" + m );
        if ( ! h[ k ] )
          h[ k ] = [];
        h[k].push(m[2]);        
      }
    }
    jsonPathHelper.set(this.conf["resphdrjp"], h, task.data);
  } // headers
  return;
};


//////////////////
// Housekeeping

Http.prototype.getClassName = function () {
  return "Http";
};

Http.prototype.getDisplayName = function () {
  return "Http";
};

Http.prototype.getDescription = function () {
  return "Make a HTTP request";
};

Http.prototype.getVersion = function () {
    // 1.0 was the initial version
  return "1.1";  // uses constant in JsonPathField, not separate checkbox
  
};

// Not much point in a renderArgs, way too complex to show in the builder
// Show the type (GET/POST), just to have something
Http.prototype.renderArgs = function () {
  logger.debug("render http: t=" + this.conf.type + " d=" +
  jsonPathHelper.displayString( this.conf['urljp']) );
  var typ =  this.conf['type'] || "";
  return typ + " " + jsonPathHelper.displayString(this.conf['urljp'],true);
    
};

Http.prototype.capabilityFlagDefault = function ( flag ) {
  //dump("checking query-and, conf = " + JSON.stringify(this.conf) + "\n");
  if (flag == "webservice-connector")
    return true;
  var used = this.getUsedArgs();
  for ( var a in used ) {
    if ( flag == "index-" + a )
      return true;
  }
  return null;
};

Http.prototype.getUsedArgs = function () {
  // Gather used arguments from inline JSONpaths
  let args = [];
  let usedArgRegex = /\{\$\.input\.(.+?)\}/g;
  let match;
  while ((match = usedArgRegex.exec(this.conf['reqjp'])) !== null) args.push([match[1]]);
  while ((match = usedArgRegex.exec(this.conf['urljp'])) !== null) args.push([match[1]]);
  return args;
}


Http.prototype.upgrade = function (confVer, curVer, conf) {
  // can't upgrade if the connector is newer than the step
  if (confVer > curVer)
    return false;
  if ( confVer < 1.1 ) {
      conf.urljp.useconstant = conf.useconst;
      conf.urljp.constantvalue = conf.url;
      conf.url = undefined;
      conf.useconst = undefined;
  }
  return true;
};

